#include "etpan-abook-ldap.h"

#include <stdlib.h>
#include <string.h>
#include <libetpan/libetpan.h>

#include "etpan-abook.h"
#include "etpan-error.h"
#include "etpan-log.h"
#include "etpan-connection-types.h"
#include "etpan-nls.h"
#include "etpan-ldap.h"

#define DEFAULT_LDAP_PORT 389
#define DEFAULT_LDAP_VERSION 2

struct abook_data {
  struct etpan_ldap_config * config;
  struct etpan_ldap_session * ldap;
  
  char * hostname;
  int port;
  char * base;
  char * username;
  char * password;
  int connection_type;
  int auth_type;

  char * threaded_hostname;
  int threaded_port;
  char * threaded_base;
  char * threaded_username;
  char * threaded_password;
  int threaded_connection_type;
  int threaded_auth_type;
  
  int connected;
};

static struct etpan_error * setup(struct etpan_abook * abook);
static void free_data(struct etpan_abook * abook);

static struct etpan_error * connect(struct etpan_abook * abook);
static void disconnect(struct etpan_abook * abook);

static struct etpan_error * lookup(struct etpan_abook * abook,
    const char * key, carray * result);

static struct etpan_abook_driver ldap_driver = {
  .name = "ldap",
  .network = 1,
  .setup = setup,
  .free_data = free_data,
  .connect = connect,
  .disconnect = disconnect,
  .lookup = lookup,
};

static struct etpan_error * connect(struct etpan_abook * abook)
{
  struct abook_data * data;
  int r;
  struct etpan_ldap_config * config;
  struct etpan_ldap_session * ldap;
  struct etpan_error * error;
  
  data = etpan_abook_get_data(abook);
  
  if (data->connected) {
    return NULL;
  }
  
  config = malloc(sizeof(* config));
  if (config == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  config->hostname = data->threaded_hostname;
  config->port = data->threaded_port;
  if (data->threaded_connection_type == ETPAN_CONNECTION_TLS)
    config->tls = 1;
  else
    config->tls = 0;
  config->version = DEFAULT_LDAP_VERSION;
  config->base = data->threaded_base;
  config->binddn = data->threaded_username;
  config->bindpw = data->threaded_password;
  config->filter = "(|(cn=%*)(mail=%*)(givenname=%*)(sn=%*))";
  config->attrs.mail = NULL;
  config->attrs.cn = NULL;
  config->attrs.givenname = NULL;
  config->attrs.sn = NULL;
  config->attrs.fullname = NULL;
  config->sizelimit = 200;
  config->timeout = 60;
  
  ldap = etpan_ldap_session_new(config);
  if (ldap == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  r = etpan_ldap_connect(ldap);
  if (r == ETPAN_LDAP_ERROR_AUTH) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_AUTH);
    etpan_error_set_short_description(error, _("Authentication failed"));
    etpan_error_strf_long_description(error,
        _("Connection to LDAP account %s failed."
            "Check your login, password and authentication type."),
        etpan_abook_get_id(abook));
    goto free_ldap;
  }
  else if (r != ETPAN_LDAP_NO_ERROR) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CONNECT);
    etpan_error_set_short_description(error, _("Unexpected error"));
    etpan_error_strf_long_description(error,
        _("An unexpected error (%i) occurred while connecting to LDAP account %s."),
        r, etpan_abook_get_id(abook));
    goto free_ldap;
  }
  
  data->config = config;
  data->ldap = ldap;
  data->connected = 1;
  
  return NULL;
  
 free_ldap:
  etpan_ldap_session_free(ldap);
  free(config);
  return error;
}

static void disconnect(struct etpan_abook * abook)
{
  struct abook_data * data;
  
  data = etpan_abook_get_data(abook);
  
  if (!data->connected)
    return;
  
  etpan_ldap_session_free(data->ldap);
  data->ldap = NULL;
  free(data->config);
  data->config = NULL;
  data->connected = 0;
}

struct etpan_abook * etpan_abook_ldap_new(void)
{
  struct etpan_abook * abook;
  struct abook_data * data;
  
  abook = etpan_abook_new();
  
  data = malloc(sizeof(* data));
  if (data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  data->config = NULL;
  data->ldap = NULL;
  
  data->hostname = NULL;
  data->port = DEFAULT_LDAP_PORT;
  data->base = NULL;
  data->username = NULL;
  data->password = NULL;
  data->connection_type = ETPAN_CONNECTION_PLAIN;
  data->auth_type = ETPAN_AUTH_NONE;

  data->threaded_hostname = NULL;
  data->threaded_port = DEFAULT_LDAP_PORT;
  data->threaded_base = NULL;
  data->threaded_username = NULL;
  data->threaded_password = NULL;
  data->threaded_connection_type = ETPAN_CONNECTION_PLAIN;
  data->threaded_auth_type = ETPAN_AUTH_NONE;
  
  data->connected = 0;
  
  etpan_abook_set_data(abook, data);
  etpan_abook_set_driver(abook, &ldap_driver);
  
  return abook;
}

void etpan_abook_ldap_set_hostname(struct etpan_abook * abook, char * hostname)
{
  struct abook_data * data;
  
  data = etpan_abook_get_data(abook);
  if (hostname != data->hostname) {
    free(data->hostname);
    if (hostname != NULL) {
      data->hostname = strdup(hostname);
      if (data->hostname == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      data->hostname = NULL;
  }
}

char * etpan_abook_ldap_get_hostname(struct etpan_abook * abook)
{
  struct abook_data * data;
  
  data = etpan_abook_get_data(abook);
  
  return data->hostname;
}

void etpan_abook_ldap_set_username(struct etpan_abook * abook, char * username)
{
  struct abook_data * data;
  
  data = etpan_abook_get_data(abook);
  if (username != data->username) {
    free(data->username);
    if (username != NULL) {
      data->username = strdup(username);
      if (data->username == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      data->username = NULL;
  }
}

char * etpan_abook_ldap_get_username(struct etpan_abook * abook)
{
  struct abook_data * data;
  
  data = etpan_abook_get_data(abook);
  
  return data->username;
}

void etpan_abook_ldap_set_password(struct etpan_abook * abook, char * password)
{
  struct abook_data * data;
  
  data = etpan_abook_get_data(abook);
  if (password != data->password) {
    free(data->password);
    if (password != NULL) {
      data->password = strdup(password);
      if (data->password == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      data->password = NULL;
  }
}

char * etpan_abook_ldap_get_password(struct etpan_abook * abook)
{
  struct abook_data * data;
  
  data = etpan_abook_get_data(abook);
  
  return data->password;
}

void etpan_abook_ldap_set_base(struct etpan_abook * abook, char * base)
{
  struct abook_data * data;
  
  data = etpan_abook_get_data(abook);
  if (base != data->base) {
    free(data->base);
    if (base != NULL) {
      data->base = strdup(base);
      if (data->base == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      data->base = NULL;
  }
}

char * etpan_abook_ldap_get_base(struct etpan_abook * abook)
{
  struct abook_data * data;
  
  data = etpan_abook_get_data(abook);
  
  return data->base;
}

void etpan_abook_ldap_set_port(struct etpan_abook * abook, int port)
{
  struct abook_data * data;
  
  data = etpan_abook_get_data(abook);
  data->port = port;
}

int etpan_abook_ldap_get_port(struct etpan_abook * abook)
{
  struct abook_data * data;
  
  data = etpan_abook_get_data(abook);
  
  return data->port;
}

void etpan_abook_ldap_set_connection_type(struct etpan_abook * abook,
    int connection_type)
{
  struct abook_data * data;
  
  data = etpan_abook_get_data(abook);
  data->connection_type = connection_type;
}

int etpan_abook_ldap_get_connection_type(struct etpan_abook * abook)
{
  struct abook_data * data;
  
  data = etpan_abook_get_data(abook);
  
  return data->connection_type;
}

void etpan_abook_ldap_set_auth_type(struct etpan_abook * abook,
    int auth_type)
{
  struct abook_data * data;
  
  data = etpan_abook_get_data(abook);
  data->auth_type = auth_type;
}

int etpan_abook_ldap_get_auth_type(struct etpan_abook * abook)
{
  struct abook_data * data;
  
  data = etpan_abook_get_data(abook);
  
  return data->auth_type;
}

static struct etpan_error * setup(struct etpan_abook * abook)
{
  struct abook_data * data;
  
  data = etpan_abook_get_data(abook);
  
  free(data->threaded_hostname);
  if (data->hostname != NULL) {
    data->threaded_hostname = strdup(data->hostname);
    if (data->threaded_hostname == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  else {
    data->threaded_hostname = NULL;
  }

  free(data->threaded_base);
  if (data->base != NULL) {
    data->threaded_base = strdup(data->base);
    if (data->threaded_base == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  else {
    data->threaded_base = NULL;
  }

  free(data->threaded_username);
  if (data->username != NULL) {
    data->threaded_username = strdup(data->username);
    if (data->threaded_username == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  else {
    data->threaded_username = NULL;
  }

  free(data->threaded_password);
  if (data->password != NULL) {
    data->threaded_password = strdup(data->password);
    if (data->threaded_password == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  else {
    data->threaded_password = NULL;
  }
  
  data->threaded_port = data->port;
  data->threaded_connection_type = data->connection_type;
  data->threaded_auth_type = data->auth_type;
  
  return NULL;
}

static void free_data(struct etpan_abook * abook)
{
  struct abook_data * data;
  
  disconnect(abook);

  data = etpan_abook_get_data(abook);
  
  free(data->hostname);
  free(data->base);
  free(data->username);
  free(data->password);

  free(data->threaded_hostname);
  free(data->threaded_base);
  free(data->threaded_username);
  free(data->threaded_password);
  
  free(data);
}

static struct etpan_abook_entry *
get_entry_from_ldap_entry(struct etpan_ldap_abook_entry * ldap_entry)
{
  struct etpan_abook_entry * entry;
  
  entry = etpan_abook_entry_new();
  etpan_abook_entry_set_firstname(entry, ldap_entry->firstname);
  etpan_abook_entry_set_lastname(entry, ldap_entry->lastname);
  etpan_abook_entry_set_fullname(entry, ldap_entry->fullname);
  etpan_abook_entry_set_address(entry, ldap_entry->address);
  
  return entry;
}

static struct etpan_error * lookup(struct etpan_abook * abook,
    const char * key, carray * result)
{
  struct abook_data * data;
  unsigned int i;
  int r;
  carray * ldap_list;
  struct etpan_error * error;
  
  data = etpan_abook_get_data(abook);
  
  r = etpan_ldap_search(data->ldap, key, &ldap_list);
  if (r == ETPAN_LDAP_ERROR_PARTIAL) {
    /* ignore error */
  }
  if (r != ETPAN_LDAP_NO_ERROR) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CONNECT);
    etpan_error_set_short_description(error, _("Unexpected error"));
    etpan_error_strf_long_description(error,
        _("An unexpected error (%i) occurred while connecting to LDAP account %s."),
        r, etpan_abook_get_id(abook));
    goto err;
  }
  
  ETPAN_LOG("ldap search %i", carray_count(ldap_list));
  
  for(i = 0 ; i < carray_count(ldap_list) ; i ++) {
    struct etpan_ldap_abook_entry * ldap_entry;
    struct etpan_abook_entry * entry;
    
    ldap_entry = carray_get(ldap_list, i);
    entry = get_entry_from_ldap_entry(ldap_entry);
    r = carray_add(result, entry, NULL);
    if (r < 0) {
      etpan_abook_entry_free(entry);
      ETPAN_LOG_MEMORY_ERROR;
    }
  }
  
  error = NULL;
  
 err:
  return error;
}
