#include "etpan-abook-manager.h"

#include <stdlib.h>
#include <string.h>

#include "etpan-abook.h"
#include "etpan-error.h"
#include "etpan-signal.h"
#include "etpan-log.h"
#include "etpan-thread-manager-app.h"

struct etpan_abook_manager * etpan_abook_manager_new(void)
{
  struct etpan_abook_manager * manager;
  
  manager = malloc(sizeof(* manager));
  if (manager == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  manager->abook_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (manager->abook_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  manager->ordered_list = carray_new(4);
  if (manager->ordered_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;

  manager->stop_remaining = 0;
  
  return manager;
}

void etpan_abook_manager_free(struct etpan_abook_manager * manager)
{
  chashiter * iter;
  
  for(iter = chash_begin(manager->abook_hash) ; iter != NULL ;
      iter = chash_next(manager->abook_hash, iter)) {
    chashdatum value;
    struct etpan_abook * abook;
    
    chash_value(iter, &value);
    abook = value.data;
    
    etpan_abook_free(abook);
  }
  
  carray_free(manager->ordered_list);
  chash_free(manager->abook_hash);
  free(manager);
}

chash * etpan_abook_manager_get_abook_hash(struct etpan_abook_manager *
    manager)
{
  return manager->abook_hash;
}

struct etpan_abook *
etpan_abook_manager_get_abook(struct etpan_abook_manager * manager,
    char * name)
{
  chashdatum key;
  chashdatum value;
  int r;
  
  key.data = name;
  key.len = strlen(name) + 1;
  
  r = chash_get(manager->abook_hash, &key, &value);
  if (r < 0)
    return NULL;
  
  return value.data;
}

void etpan_abook_manager_add_abook(struct etpan_abook_manager * manager,
    struct etpan_abook * abook)
{
  chashdatum key;
  chashdatum value;
  int r;
  char * name;
  
  name = etpan_abook_get_id(abook);
  key.data = name;
  key.len = strlen(name) + 1;
  value.data = abook;
  value.len = 0;
  
  r = chash_set(manager->abook_hash, &key, &value, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  r = carray_add(manager->ordered_list, abook, NULL);
  if (r < 0) {
    chash_delete(manager->abook_hash, &key, NULL);
    ETPAN_LOG_MEMORY_ERROR;
  }
}

void etpan_abook_manager_remove_abook(struct etpan_abook_manager *
    manager, struct etpan_abook * abook)
{
  chashdatum key;
  int r;
  char * name;
  unsigned int i;
  
  name = etpan_abook_get_id(abook);
  key.data = name;
  key.len = strlen(name) + 1;
  
  r = chash_delete(manager->abook_hash, &key, NULL);
  /* ignore error */
  
  for(i = 0 ; i < carray_count(manager->ordered_list) ; i ++) {
    struct etpan_abook * current_abook;
    
    current_abook = carray_get(manager->ordered_list, i);
    if (current_abook == abook) {
      carray_delete_slow(manager->ordered_list, i);
      break;
    }
  }
}

static struct etpan_abook_manager * default_manager = NULL;

void etpan_abook_manager_set_default(struct etpan_abook_manager * manager)
{
  default_manager = manager;
}

struct etpan_abook_manager * etpan_abook_manager_get_default(void)
{
  return default_manager;
}

carray * etpan_abook_manager_get_ordered_list(struct etpan_abook_manager * manager)
{
  return manager->ordered_list;
}

void etpan_abook_manager_notify_modification(struct etpan_abook_manager * manager)
{
  etpan_signal_send(etpan_signal_manager_get_default(),
      ETPAN_ABOOK_MANAGER_MODIFICATION_SIGNAL,
      manager, NULL);
}

static void abook_stop_callback(int cancelled, void * dummy, void * cb_data)
{
  struct etpan_abook_manager * manager;
  (void) dummy;
  (void) cancelled;
  
  manager = cb_data;

  manager->stop_remaining --;
  if (manager->stop_remaining > 0)
    return;
  
  if (manager->stop_callback != NULL)
    manager->stop_callback(manager->stop_cb_data);
  manager->stop_callback = NULL;
  manager->stop_cb_data = NULL;
}

void etpan_abook_manager_stop(struct etpan_abook_manager * manager,
    void (* callback)(void *), void * cb_data)
{
  chashiter * iter;
  chash * abook_hash;
  
  if (manager->stop_remaining) {
    ETPAN_LOG("abook manager stop called twice");
    etpan_crash();
  }
  
  manager->stop_cb_data = cb_data;
  manager->stop_callback = callback;
  manager->stop_remaining = 0;
  
  abook_hash = etpan_abook_manager_get_abook_hash(manager);
  if (chash_count(abook_hash) == 0) {
    manager->stop_remaining = 1;
    abook_stop_callback(0, NULL, manager);
    return;
  }
  
  for(iter = chash_begin(abook_hash) ; iter != NULL ;
      iter = chash_next(abook_hash, iter)) {
    chashdatum value;
    struct etpan_abook * abook;
    
    chash_value(iter, &value);
    abook = value.data;
    
    manager->stop_remaining ++;
    etpan_abook_disconnect(etpan_thread_manager_app_get_default(),
        abook, abook_stop_callback, manager);
  }
}

void etpan_abook_manager_setup(struct etpan_abook_manager * manager)
{
  (void) manager;
}

void etpan_abook_manager_unsetup(struct etpan_abook_manager * manager)
{
  chashiter * iter;
  chash * abook_hash;
  
  abook_hash = etpan_abook_manager_get_abook_hash(manager);
  
  for(iter = chash_begin(abook_hash) ; iter != NULL ;
      iter = chash_next(abook_hash, iter)) {
    chashdatum value;
    struct etpan_abook * abook;
    
    chash_value(iter, &value);
    abook = value.data;
    
    etpan_abook_unsetup(abook);
  }
}

