#include "etpan-abook-request.h"

#include <stdlib.h>
#include <string.h>

#include "etpan-error.h"
#include "etpan-log.h"
#include "etpan-abook.h"
#include "etpan-abook-manager.h"
#include "etpan-thread-manager.h"
#include "etpan-thread-manager-app.h"
#include "etpan-signal.h"
#include "etpan-nls.h"

struct etpan_abook_request *
etpan_abook_request_new(struct etpan_abook_manager * manager)
{
  struct etpan_abook_request * request;
  
  request = malloc(sizeof(* request));
  if (request == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  request->remaining = 0;
  request->manager = manager;
  request->result = carray_new(16);
  if (request->result == NULL)
    ETPAN_LOG_MEMORY_ERROR;

  request->remaining_op = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (request->remaining_op == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  request->cancelled = 0;
  request->flags = 0;
  request->error_list = NULL;
  request->error = NULL;
  
  return request;
}

static void clear_result(struct etpan_abook_request * request)
{
  unsigned int i;
  int r;
  
  for(i = 0 ; i < carray_count(request->result) ; i ++) {
    struct etpan_abook_entry * entry;
    
    entry = carray_get(request->result, i);
    etpan_abook_entry_free(entry);
  }
  r = carray_set_size(request->result, 0);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  ETPAN_ERROR_FREE(request->error);
  request->error = NULL;
}

void etpan_abook_request_free(struct etpan_abook_request * request)
{
  clear_result(request);
  chash_free(request->remaining_op);
  carray_free(request->result);
  free(request);
}

void etpan_abook_request_cancel(struct etpan_abook_request * request)
{
  chashiter * iter;
  
  request->cancelled = 1;
  for(iter = chash_begin(request->remaining_op) ; 
      iter != NULL ;
      iter = chash_next(request->remaining_op, iter)) {
    struct etpan_thread_op * op;
    chashdatum value;
    
    chash_value(iter, &value);
    op = value.data;
    etpan_thread_op_cancel(op);
  }
  chash_clear(request->remaining_op);
}

static void lookup(struct etpan_abook_request * request);
static void lookup_callback(int cancelled,
    struct etpan_abook_lookup_result * result, void * cb_data);
static void notify_updated(struct etpan_abook_request * request);
static void notify_finished(struct etpan_abook_request * request);
static void sort(struct etpan_abook_request * request);
static int compare_entry(const void * a, const void * b);

void etpan_abook_request_lookup(struct etpan_abook_request * request,
    char * key)
{
  request->cancelled = 0;
  
  clear_result(request);
  
  request->error_list = carray_new(4);
  if (request->error_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  if (request->remaining != 0) {
    ETPAN_LOG("error, request was not finished");
    etpan_crash();
  }
  
  request->key = strdup(key);
  if (request->key == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  lookup(request);
}


struct thread_data {
  struct etpan_abook_request * request;
  struct etpan_abook * abook;
};

static void lookup(struct etpan_abook_request * request)
{
  carray * abook_list;
  unsigned int i;
  int r;
  
  abook_list = etpan_abook_manager_get_ordered_list(request->manager);
  
  for(i = 0 ; i < carray_count(abook_list) ; i ++) {
    struct etpan_abook * abook;
    struct etpan_thread_op * op;
    struct thread_data * data;
    chashdatum key;
    chashdatum value;
    
    abook = carray_get(abook_list, i);
    
    if (request->flags & ETPAN_ABOOK_REQUEST_FLAGS_DISABLE_NETWORK) {
      if (abook->driver->network)
        continue;
    }
    
    data = malloc(sizeof(* data));
    if (data == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    
    data->abook = abook;
    data->request = request;
    
    ETPAN_LOG("launch op %p", abook);
    op = etpan_abook_lookup(etpan_thread_manager_app_get_default(),
        abook, request->key, lookup_callback, data);
      
    request->remaining ++;
      
    key.data = &abook;
    key.len = sizeof(abook);
    value.data = op;
    value.len = 0;
    r = chash_set(request->remaining_op, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  ETPAN_LOG("remaining finished ? %i", request->remaining);
  if (request->remaining == 0) {
    notify_finished(request);
  }
}

static void lookup_callback(int cancelled,
    struct etpan_abook_lookup_result * result, void * cb_data)
{
  struct etpan_abook_request * request;
  carray * list;
  unsigned int i;
  int r;
  struct thread_data * thread_data;
  struct etpan_abook * abook;
  chashdatum key;
  struct etpan_error * error;
  
  thread_data = cb_data;
  request = thread_data->request;
  abook = thread_data->abook;
  free(thread_data);
  
  request->remaining --;
  
  key.data = &abook;
  key.len = sizeof(abook);
  chash_delete(request->remaining_op, &key, NULL);
  
  if (cancelled)
    return;
    
  if (result->error != NULL) {
    error = result->error;
    result->error = NULL;
    
    r = carray_add(request->error_list, error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    result->error = NULL;
  }
  else {
    list = result->entry_list;
    result->entry_list = NULL;
    for(i = 0 ; i < carray_count(list) ; i ++) {
      struct etpan_abook_entry * entry;
      
      entry = carray_get(list, i);
      r = carray_add(request->result, entry, NULL);
      if (r < 0) {
        etpan_abook_entry_free(entry);
        ETPAN_LOG_MEMORY_ERROR;
      }
    }
    carray_free(list);
      
    sort(request);
    
    notify_updated(request);
    
    ETPAN_LOG("entries %i", carray_count(request->result));
  }
  ETPAN_LOG("remaining finished ? %i", request->remaining);
  
  if (request->remaining == 0) {
    notify_finished(request);
  }
}

static void sort(struct etpan_abook_request * request)
{
  qsort(carray_data(request->result),
      carray_count(request->result), sizeof(void *),
      compare_entry);
}

static int compare_entry(const void * a, const void * b)
{
  struct etpan_abook_entry * entry_a;
  struct etpan_abook_entry * entry_b;
  char * value_a;
  char * value_b;
  
  entry_a = * (struct etpan_abook_entry **) a;
  entry_b = * (struct etpan_abook_entry **) b;
  
  switch (entry_a->match_type) {
  case ETPAN_ABOOK_MATCH_ADDRESS:
    value_a = entry_a->address;
    break;
        
  case ETPAN_ABOOK_MATCH_FIRSTNAME:
    value_a = entry_a->firstname;
    break;
      
  case ETPAN_ABOOK_MATCH_LASTNAME:
    value_a = entry_a->lastname;
    break;
        
  case ETPAN_ABOOK_MATCH_NICKNAME:
    value_a = entry_a->nickname;
    break;
        
  default:
  case ETPAN_ABOOK_MATCH_NONE:
  case ETPAN_ABOOK_MATCH_FULLNAME:
    value_a = entry_a->fullname;
    if (value_a == NULL)
      value_a = entry_a->address;
    break;
  }
  
  switch (entry_b->match_type) {
  case ETPAN_ABOOK_MATCH_ADDRESS:
    value_b = entry_b->address;
    break;
        
  case ETPAN_ABOOK_MATCH_FIRSTNAME:
    value_b = entry_b->firstname;
    break;
      
  case ETPAN_ABOOK_MATCH_LASTNAME:
    value_b = entry_b->lastname;
    break;
        
  case ETPAN_ABOOK_MATCH_NICKNAME:
    value_b = entry_b->nickname;
    break;
        
  default:
  case ETPAN_ABOOK_MATCH_NONE:
  case ETPAN_ABOOK_MATCH_FULLNAME:
    value_b = entry_b->fullname;
    if (value_b == NULL)
      value_b = entry_b->address;
    break;
  }
  
  return strcasecmp(value_a, value_b);
}



static void notify_finished(struct etpan_abook_request * request)
{
  unsigned int i;
  struct etpan_error * error;
  int r;
  
  if (request->cancelled) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CANCELLED);
    etpan_error_set_short_description(error,
        _("Lookup in address books cancelled"));
    etpan_error_strf_long_description(error,
        _("Lookup in address books has been cancelled."));
    
    r = carray_add(request->error_list, error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  if (carray_count(request->error_list) == 0) {
    error = NULL;
  }
  else if (carray_count(request->error_list) == 1) {
    error = carray_get(request->error_list, 0);
  }
  else {
    error = etpan_error_multiple();
    for(i = 0 ; i < carray_count(request->error_list) ; i ++) {
      struct etpan_error * suberror;
      
      suberror = carray_get(request->error_list, i);
      etpan_error_add_child(error, suberror);
    }
  }
  carray_free(request->error_list);
  request->error_list = NULL;
  request->error = error;
  
  etpan_signal_send(etpan_signal_manager_get_default(),
    ETPAN_ABOOK_REQUEST_FINISHED, request, NULL);
}

static void notify_updated(struct etpan_abook_request * request)
{
  if (request->cancelled)
    return;
  
  etpan_signal_send(etpan_signal_manager_get_default(),
    ETPAN_ABOOK_REQUEST_UPDATED_RESULT, request, NULL);
}

struct etpan_error *
etpan_abook_request_get_error(struct etpan_abook_request * request)
{
  return request->error;
}

carray * etpan_abook_request_get_result(struct etpan_abook_request * request)
{
  return request->result;
}

void etpan_abook_request_set_flags(struct etpan_abook_request * request,
    int flags)
{
  request->flags = flags;
}

int etpan_abook_request_get_flags(struct etpan_abook_request * request)
{
  return request->flags;
}

