#include "etpan-abook-vcard.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <string.h>
#include <libetpan/libetpan.h>

#include "etpan-abook.h"
#include "etpan-error.h"
#include "vobject.h"
#include "vcc.h"
#include "etpan-log.h"
#include "etpan-nls.h"

/* TODO : performance of vcard abook : do a log(n) lookup or use lucene */

static pthread_mutex_t lock = PTHREAD_MUTEX_INITIALIZER;

struct abook_data {
  char * filename;
  
  char * threaded_filename;
  time_t file_timestamp;
  off_t file_size;
  
  carray * entry_list;
  
  int connected;
};

static struct etpan_error * setup(struct etpan_abook * abook);
static void free_data(struct etpan_abook * abook);

static struct etpan_error * connect(struct etpan_abook * abook);
static void disconnect(struct etpan_abook * abook);

static struct etpan_error * lookup(struct etpan_abook * abook,
    const char * key, carray * result);

static struct etpan_abook_driver vcard_driver = {
  .name = "vcard",
  .network = 0,
  .setup = setup,
  .free_data = free_data,
  .connect = connect,
  .disconnect = disconnect,
  .lookup = lookup,
};

static void get_entry_from_vcard(carray * list, VObject * v);

static int parse_vcards_file(char * filename, VObject ** result);

static void empty_entry_list(struct etpan_abook * abook);

static void entry_list_from_vcard(struct etpan_abook * abook,
    VObject * vcard_list);

static struct etpan_error * vcard_read_dir(struct etpan_abook * abook,
    struct stat * stat_buf);
static struct etpan_error * vcard_read_file(struct etpan_abook * abook,
    struct stat * stat_buf);

static struct etpan_error * connect(struct etpan_abook * abook)
{
  struct abook_data * data;
  int r;
  struct stat stat_buf;
  struct etpan_error * error;
  
  data = etpan_abook_get_data(abook);
  
  r = stat(data->threaded_filename, &stat_buf);
  if (r < 0) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FILE);
    etpan_error_set_short_description(error, _("File not found"));
    etpan_error_strf_long_description(error, _("The V-Card file was not found (%s)"), data->threaded_filename);
    goto err;
  }
  
  if (data->connected) {
    if ((stat_buf.st_size == data->file_size) ||
        (stat_buf.st_mtime == data->file_timestamp)) {
      return NULL;
    }
  }
  
  if ((stat_buf.st_mode & S_IFDIR) != 0) {
    return vcard_read_dir(abook, &stat_buf);
  }
  else {
    return vcard_read_file(abook, &stat_buf);
  }
  
 err:
  return error;
}

static struct etpan_error * vcard_read_dir(struct etpan_abook * abook,
    struct stat * stat_buf)
{
  struct abook_data * data;
  int r;
  VObject * vcard_list;
  VObject * v;
  struct etpan_error * error;
  DIR * dir;
  
  data = etpan_abook_get_data(abook);
  
  empty_entry_list(abook);
  
  dir = opendir(data->threaded_filename);
  if (dir == NULL) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, _("Folder not valid"));
    etpan_error_strf_long_description(error, _("Could not open V-Card folder (%s)"), data->threaded_filename);
    goto err;
  }
  
  while (1) {
    struct dirent * ent;
    char filename[PATH_MAX];
    
    ent = readdir(dir);
    if (ent == NULL)
      break;
    
    if (ent->d_name[0] == '.')
      continue;
    
    snprintf(filename, sizeof(filename), "%s/%s", data->threaded_filename,
        ent->d_name);
    
    ETPAN_LOG("%s", data->threaded_filename);
    ETPAN_LOG("%s", ent->d_name);
    ETPAN_LOG("%s", filename);
    r = parse_vcards_file(filename, &vcard_list);
    if (r < 0) {
      /* skip */
      continue;
    }
    
    entry_list_from_vcard(abook, vcard_list);
    
    v = vcard_list;
    while (v) {
      VObject * t;
      
      t = v;
      v = nextVObjectInList(v);
      cleanVObject(t);
    }
  }
  
  data->connected = 1;
  data->file_size = stat_buf->st_size;
  data->file_timestamp = stat_buf->st_mtime;
  
  return NULL;
  
 err:
  return error;
}

static struct etpan_error * vcard_read_file(struct etpan_abook * abook,
    struct stat * stat_buf)
{
  struct abook_data * data;
  int r;
  VObject * vcard_list;
  VObject * v;
  struct etpan_error * error;
  
  data = etpan_abook_get_data(abook);
  
  empty_entry_list(abook);
  
  r = parse_vcards_file(data->threaded_filename, &vcard_list);
  if (r < 0) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, _("Parse error"));
    etpan_error_strf_long_description(error, _("The content of the V-Card file is not valid (%s)"), data->threaded_filename);
    goto err;
  }
  
  data->connected = 1;
  data->file_size = stat_buf->st_size;
  data->file_timestamp = stat_buf->st_mtime;
  
  entry_list_from_vcard(abook, vcard_list);
  
  v = vcard_list;
  while (v) {
    VObject * t;
    
    t = v;
    v = nextVObjectInList(v);
    cleanVObject(t);
  }
  
  return NULL;
  
 err:
  return error;
}

static void disconnect(struct etpan_abook * abook)
{
  struct abook_data * data;
  
  data = etpan_abook_get_data(abook);
  
  if (!data->connected)
    return;
  
  empty_entry_list(abook);
  data->connected = 0;
}

struct etpan_abook * etpan_abook_vcard_new(void)
{
  struct etpan_abook * abook;
  struct abook_data * data;
  
  abook = etpan_abook_new();
  data = malloc(sizeof(* data));
  if (data == NULL)
    ETPAN_LOG_MEMORY_ERROR;

  data->filename = NULL;
  data->threaded_filename = NULL;
  data->file_timestamp = (time_t) -1;
  data->file_size = 0;
  data->connected = 0;
  
  data->entry_list = carray_new(16);
  if (data->entry_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  etpan_abook_set_data(abook, data);
  etpan_abook_set_driver(abook, &vcard_driver);
  
  return abook;
}

static void etpan_MimeErrorHandler(char * s)
{
  ETPAN_LOG("%s", s);
}

typedef void (* MimeErrorHandler)(char *);
extern void registerMimeErrorHandler(MimeErrorHandler me);
extern VObject * Parse_MIME_FromFileName(char * fname);

static int parse_vcards_file(char * filename, VObject ** result)
{
  VObject * v;
  
  registerMimeErrorHandler(etpan_MimeErrorHandler);
  
  pthread_mutex_lock(&lock);
  v = Parse_MIME_FromFileName(filename);
  pthread_mutex_unlock(&lock);
  if (v == NULL)
    return -1;
  
  * result = v;
  
  return 0;
}

void etpan_abook_vcard_set_filename(struct etpan_abook * abook, char * filename)
{
  struct abook_data * data;
  
  data = etpan_abook_get_data(abook);
  if (filename != data->filename) {
    free(data->filename);
    if (filename != NULL) {
      data->filename = strdup(filename);
      if (data->filename == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      data->filename = NULL;
  }
}

char * etpan_abook_vcard_get_filename(struct etpan_abook * abook)
{
  struct abook_data * data;
  
  data = etpan_abook_get_data(abook);
  
  return data->filename;
}

static struct etpan_error * setup(struct etpan_abook * abook)
{
  struct abook_data * data;
  
  data = etpan_abook_get_data(abook);
  
  free(data->threaded_filename);
  if (data->filename != NULL) {
    data->threaded_filename = strdup(data->filename);
    if (data->threaded_filename == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  else {
    data->threaded_filename = NULL;
  }
  
  return NULL;
}

static void free_data(struct etpan_abook * abook)
{
  struct abook_data * data;
  
  disconnect(abook);
  
  data = etpan_abook_get_data(abook);
  
  carray_free(data->entry_list);
  free(data->threaded_filename);
  free(data->filename);
  free(data);
}

static char * get_value(VObject * eachProp)
{
  int type;
  char * value;
  
  value = NULL;
  type = vObjectValueType(eachProp);
  switch (type) {
  case VCVT_STRINGZ:
    value = strdup(vObjectStringZValue(eachProp));
    if (value == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    break;
    
  case VCVT_USTRINGZ:
    value = fakeCString(vObjectUStringZValue(eachProp));
    if (value == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    break;
  }
  
  return value;
}

#define NICKNAME "NICKNAME"

static void get_entry_from_vcard(carray * list, VObject * v)
{
  VObjectIterator t;
  char * fullname;
  char * firstname;
  char * lastname;
  char * nickname;
  int r;
  
  fullname = NULL;
  firstname = NULL;
  lastname = NULL;
  nickname = NULL;
  
  initPropIterator(&t, v);
  
  while (moreIteration(&t)) {
    VObject * eachProp;
    const char * n;
    
    eachProp = nextVObject(&t);
    
    n = vObjectName(eachProp);
    if (strcmp(n, VCFullNameProp) == 0) {
      free(fullname);
      fullname = get_value(eachProp);
    }
    else if (strcmp(n, NICKNAME) == 0) {
      free(nickname);
      nickname = get_value(eachProp);
    }
    else if (strcmp(n, VCNameProp) == 0) {
      VObjectIterator nameIter;
      
      initPropIterator(&nameIter, eachProp);
      while (moreIteration(&nameIter)) {
        VObject * eachProp;
        char * name;
        
        eachProp = nextVObject(&nameIter);
        n = vObjectName(eachProp);
        name = get_value(eachProp);
        if (name == NULL)
          continue;
        
        if (strcmp(n, VCFamilyNameProp) == 0) {
          free(lastname);
          lastname = name;
        }
        
        if (strcmp(n, VCGivenNameProp) == 0) {
          free(firstname);
          firstname = name;
        }
      }
    }
  }

  initPropIterator(&t, v);
  
  while (moreIteration(&t)) {
    VObject * eachProp;
    const char * n;
    
    eachProp = nextVObject(&t);
    
    n = vObjectName(eachProp);
    if (strcmp(n, VCEmailAddressProp) == 0) {
      char * address;
      struct etpan_abook_entry * entry;
      
      address = get_value(eachProp);
      
      entry = etpan_abook_entry_new();
      if (entry == NULL)
        continue;
      
      etpan_abook_entry_set_firstname(entry, firstname);
      etpan_abook_entry_set_lastname(entry, lastname);
      etpan_abook_entry_set_fullname(entry, fullname);
      etpan_abook_entry_set_address(entry, address);
      etpan_abook_entry_set_nickname(entry, nickname); 
      
      free(address);
      
      r = carray_add(list, entry, NULL);
      if (r < 0) {
        etpan_abook_entry_free(entry);
        ETPAN_LOG_MEMORY_ERROR;
      }
    }
  }
  
  free(firstname);
  free(lastname);
  free(fullname);
  free(nickname);
}

static void empty_entry_list(struct etpan_abook * abook)
{
  struct abook_data * data;
  unsigned int i;

  data = etpan_abook_get_data(abook);
  
  for(i = 0 ; i < carray_count(data->entry_list) ; i ++) {
    struct etpan_abook_entry * entry;
    
    entry = carray_get(data->entry_list, i);
    etpan_abook_entry_free(entry);
  }
  carray_set_size(data->entry_list, 0);
}

static void entry_list_from_vcard(struct etpan_abook * abook,
    VObject * vcard_list)
{
  struct abook_data * data;
  VObject * v;
  
  data = etpan_abook_get_data(abook);
  
  v = vcard_list;
  while (v) {
    VObjectIterator t;
    
    initPropIterator(&t, v);
    
    get_entry_from_vcard(data->entry_list, v);
    
    v = nextVObjectInList(v);
  }
}

static struct etpan_error * lookup(struct etpan_abook * abook,
    const char * key, carray * result)
{
  struct abook_data * data;
  unsigned int i;
  int r;
  
  data = etpan_abook_get_data(abook);
  
  for(i = 0 ; i < carray_count(data->entry_list) ; i ++) {
    struct etpan_abook_entry * entry;
    
    entry = carray_get(data->entry_list, i);
    if (etpan_abook_entry_match(entry, key)) {
      struct etpan_abook_entry * dup_entry;
      
      dup_entry =  etpan_abook_entry_dup(entry);
      r = carray_add(result, dup_entry, NULL);
      if (r < 0) {
        etpan_abook_entry_free(dup_entry);
        ETPAN_LOG_MEMORY_ERROR;
      }
    }
  }
  
  return NULL;
}
