#include "etpan-abook.h"

#include <stdlib.h>
#include <string.h>

#include "etpan-error.h"
#include "etpan-thread-manager.h"
#include "etpan-thread-manager-app.h"
#include "etpan-log.h"
#include "etpan-nls.h"

static struct etpan_error * abook_connect(struct etpan_abook * abook);
static void abook_disconnect(struct etpan_abook * abook);

struct etpan_abook * etpan_abook_new(void)
{
  struct etpan_abook * abook;
  
  abook = malloc(sizeof(* abook));
  if (abook == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  abook->id = NULL;
  abook->data = NULL;
  abook->driver = NULL;
  
  return abook;
}

void etpan_abook_free(struct etpan_abook * abook)
{
  if (abook->driver != NULL) {
    if (abook->driver->free_data != NULL)
      abook->driver->free_data(abook);
  }
  
  free(abook->id);
  free(abook);
}

void etpan_abook_set_id(struct etpan_abook * abook, char * id)
{
  if (id != abook->id) {
    free(abook->id);
    if (id != NULL) {
      abook->id = strdup(id);
      if (abook->id == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      abook->id = NULL;
  }
}

char * etpan_abook_get_id(struct etpan_abook * abook)
{
  return abook->id;
}

struct etpan_error * etpan_abook_setup(struct etpan_abook * abook)
{
  struct etpan_error * error;
  
  if (abook->driver->setup != NULL) {
    error = abook->driver->setup(abook);
    if (error != NULL)
      return error;
  }
  
  return NULL;
}

void * etpan_abook_get_data(struct etpan_abook * abook)
{
  return abook->data;
}

void etpan_abook_set_data(struct etpan_abook * abook, void * data)
{
  abook->data = data;
}

void etpan_abook_set_driver(struct etpan_abook * abook,
    struct etpan_abook_driver * driver)
{
  abook->driver = driver;
}

struct etpan_abook_driver *
etpan_abook_get_driver(struct etpan_abook * abook)
{
  return abook->driver;
}


struct etpan_abook_entry * etpan_abook_entry_new(void)
{
  struct etpan_abook_entry * entry;
  
  entry = malloc(sizeof(* entry));
  if (entry == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  entry->firstname = NULL;
  entry->lastname = NULL;
  entry->fullname = NULL;
  entry->address = NULL;
  entry->nickname = NULL;
  entry->match_type = ETPAN_ABOOK_MATCH_NONE;
  
  return entry;
}

void etpan_abook_entry_free(struct etpan_abook_entry * entry)
{
  free(entry->firstname);
  free(entry->lastname);
  free(entry->fullname);
  free(entry->address);
  free(entry->nickname);
  free(entry);
}

void etpan_abook_entry_set_firstname(struct etpan_abook_entry * entry,
    char * firstname)
{
  if (firstname != entry->firstname) {
    free(entry->firstname);
    if (firstname != NULL) {
      entry->firstname = strdup(firstname);
      if (entry->firstname == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      entry->firstname = NULL;
  }
}

void etpan_abook_entry_set_lastname(struct etpan_abook_entry * entry,
    char * lastname)
{
  if (lastname != entry->lastname) {
    free(entry->lastname);
    if (lastname != NULL) {
      entry->lastname = strdup(lastname);
      if (entry->lastname == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      entry->lastname = NULL;
  }
}

void etpan_abook_entry_set_fullname(struct etpan_abook_entry * entry,
    char * fullname)
{
  if (fullname != entry->fullname) {
    free(entry->fullname);
    if (fullname != NULL) {
      entry->fullname = strdup(fullname);
      if (entry->fullname == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      entry->fullname = NULL;
  }
}

void etpan_abook_entry_set_address(struct etpan_abook_entry * entry,
    char * address)
{
  if (address != entry->address) {
    free(entry->address);
    if (address != NULL) {
      entry->address = strdup(address);
      if (entry->address == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      entry->address = NULL;
  }
}

void etpan_abook_entry_set_nickname(struct etpan_abook_entry * entry,
    char * nickname)
{
  if (nickname != entry->nickname) {
    free(entry->nickname);
    if (nickname != NULL) {
      entry->nickname = strdup(nickname);
      if (entry->nickname == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      entry->nickname = NULL;
  }
}

char * etpan_abook_entry_get_firstname(struct etpan_abook_entry * entry)
{
  return entry->firstname;
}

char * etpan_abook_entry_get_lastname(struct etpan_abook_entry * entry)
{
  return entry->lastname;
}

char * etpan_abook_entry_get_fullname(struct etpan_abook_entry * entry)
{
  return entry->fullname;
}

char * etpan_abook_entry_get_address(struct etpan_abook_entry * entry)
{
  return entry->address;
}

char * etpan_abook_entry_get_nickname(struct etpan_abook_entry * entry)
{
  return entry->nickname;
}

int etpan_abook_entry_match(struct etpan_abook_entry * entry,
    const char * key)
{
  size_t len;
  
  len = strlen(key);
  
  if (entry->fullname != NULL) {
    if (strncasecmp(key, entry->fullname, len) == 0)
      return ETPAN_ABOOK_MATCH_FULLNAME;
  }
  if (entry->address != NULL) {
    if (strncasecmp(key, entry->address, len) == 0)
      return ETPAN_ABOOK_MATCH_ADDRESS;
  }
  if (entry->firstname != NULL) {
    if (strncasecmp(key, entry->firstname, len) == 0)
      return ETPAN_ABOOK_MATCH_FIRSTNAME;
  }
  if (entry->lastname != NULL) {
    if (strncasecmp(key, entry->lastname, len) == 0)
      return ETPAN_ABOOK_MATCH_LASTNAME;
  }
  if (entry->nickname != NULL) {
    if (strncasecmp(key, entry->nickname, len) == 0)
      return ETPAN_ABOOK_MATCH_NICKNAME;
  }
  
  return 0;
}

struct etpan_abook_entry *
etpan_abook_entry_dup(struct etpan_abook_entry * entry)
{
  struct etpan_abook_entry * dup_entry;
  
  dup_entry = etpan_abook_entry_new();
  if (dup_entry == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  etpan_abook_entry_set_firstname(dup_entry,
      etpan_abook_entry_get_firstname(entry));
  
  etpan_abook_entry_set_lastname(dup_entry,
      etpan_abook_entry_get_lastname(entry));
  
  etpan_abook_entry_set_fullname(dup_entry,
      etpan_abook_entry_get_fullname(entry));
  
  etpan_abook_entry_set_address(dup_entry,
      etpan_abook_entry_get_address(entry));
  
  etpan_abook_entry_set_nickname(dup_entry,
      etpan_abook_entry_get_nickname(entry));
  
  return dup_entry;
}

/* lookup */

struct etpan_abook_lookup_param {
  char * key;
  struct etpan_abook * abook;
};

static void
lookup_cleanup(struct etpan_thread_op * op)
{
  struct etpan_abook_lookup_param * param;
  struct etpan_abook_lookup_result * result;
  unsigned int i;
  
  param = op->param;
  free(param->key);
  free(param);
  op->param = NULL;
  
  result = op->result;
  ETPAN_ERROR_FREE(result->error);
  if (result->entry_list != NULL) {
    for(i = 0 ; i < carray_count(result->entry_list) ; i ++) {
      struct etpan_abook_entry * entry;
      
      entry = carray_get(result->entry_list, i);
      etpan_abook_entry_free(entry);
    }
    carray_free(result->entry_list);
  }
  free(result);
  op->result = NULL;
}

static void
lookup_run(struct etpan_thread_op * op)
{
  struct etpan_abook_lookup_param * param;
  struct etpan_abook_lookup_result * result;
  struct etpan_error * error;
  
  param = op->param;
  result = op->result;
  
  error = abook_connect(param->abook);
  if (error == NULL) {
    if (param->abook->driver->lookup != NULL)
      error = param->abook->driver->lookup(param->abook, param->key,
          result->entry_list);
    else {
      char long_description[1024];
      
      snprintf(long_description, sizeof(long_description),
          _("lookup is not implemented for %s"),
          param->abook->driver->name);
      
      error = etpan_error_internal(long_description);
    }
  }
  
  if (error == NULL) {
    unsigned int i;
    
    for(i = 0 ; i < carray_count(result->entry_list) ; i ++) {
      struct etpan_abook_entry * entry;
      
      entry = carray_get(result->entry_list, i);
      entry->match_type = etpan_abook_entry_match(entry, param->key);
    }
  }
  
  result->error = error;
}

struct etpan_thread_op *
etpan_abook_lookup(struct etpan_thread_manager_app * manager,
    struct etpan_abook * abook, char * key,
    void (* callback)(int, struct etpan_abook_lookup_result *, void *),
    void * cb_data)
{
  struct etpan_thread_op * op;
  struct etpan_abook_lookup_param * param;
  struct etpan_abook_lookup_result * result;
  
  param = malloc(sizeof(* param));
  if (param == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  param->abook = abook;
  param->key = strdup(key);
  if (param->key == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  result = malloc(sizeof(* result));
  if (result == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  result->error = NULL;
  result->entry_list = carray_new(16);
  if (result->entry_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  op = etpan_thread_op_new();
  op->param = param;
  op->result = result;
  
  op->cancellable = 1;
  op->run = lookup_run;
  op->callback = (void (*)(int, void *, void *)) callback;
  op->callback_data = cb_data;
  op->cleanup = lookup_cleanup;
  
  etpan_thread_manager_app_abook_schedule(manager, abook, op);
  
  return op;
}



/* disconnect */

struct etpan_abook_disconnect_param {
  struct etpan_abook * abook;
};

static void
disconnect_cleanup(struct etpan_thread_op * op)
{
  struct etpan_abook_disconnect_param * param;
  
  param = op->param;
  free(param);
  op->param = NULL;
  
  free(op->result);
  op->result = NULL;
}

static void
disconnect_run(struct etpan_thread_op * op)
{
  struct etpan_abook_disconnect_param * param;
  
  param = op->param;

  abook_disconnect(param->abook);
}

struct etpan_thread_op *
etpan_abook_disconnect(struct etpan_thread_manager_app * manager,
    struct etpan_abook * abook,
    void (* callback)(int, void * /* dummy argument */,
        void *),
    void * cb_data)
{
  struct etpan_thread_op * op;
  struct etpan_abook_disconnect_param * param;
  
  param = malloc(sizeof(* param));
  if (param == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  param->abook = abook;
  
  op = etpan_thread_op_new();
  op->param = param;
  op->result = NULL;
  
  op->cancellable = 0;
  op->run = disconnect_run;
  op->callback = (void (*)(int, void *, void *)) callback;
  op->callback_data = cb_data;
  op->cleanup = disconnect_cleanup;
  
  etpan_thread_manager_app_abook_schedule(manager, abook, op);
  
  return op;
}


static struct etpan_error * abook_connect(struct etpan_abook * abook)
{
  struct etpan_error * error;
  
  /* try to connect */
  if (abook->driver->connect != NULL)
    error = abook->driver->connect(abook);
  else
    error = NULL;
  
  /* disconnect everything if it failed */
  if (error != NULL) {
    ETPAN_LOG("reconnection of %s", etpan_abook_get_id(abook));
    
    abook_disconnect(abook);
    
    /* retry connection */
    if (abook->driver->connect != NULL)
      error = abook->driver->connect(abook);
    else
      error = NULL;
  }
  
  return error;
}

static void abook_disconnect(struct etpan_abook * abook)
{
  if (abook->driver->disconnect != NULL)
    abook->driver->disconnect(abook);
}

void etpan_abook_entry_str(char * str, size_t len,
    struct etpan_abook_entry * entry)
{
  if (entry->fullname != NULL) {
    if (entry->address != NULL) {
      snprintf(str, len, "%s <%s>", entry->fullname, entry->address);
    }
    else {
      snprintf(str, len, "%s <>", entry->fullname);
    }
  }
  else {
    if (entry->address != NULL) {
      snprintf(str, len, "<%s>", entry->address);
    }
    else {
      snprintf(str, len, "<>");
    }
  }
}

void etpan_abook_unsetup(struct etpan_abook * abook)
{
  etpan_thread_manager_app_unbind_abook_thread(etpan_thread_manager_app_get_default(), abook);
}
