/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-ldap.c,v 1.6 2004/08/24 11:33:53 g_roualland Exp $
 * Contributed by Gal Roualland <gael.roualland@dial.oleane.com>
 */


#include "etpan-ldap.h"

#ifdef HAVE_CONFIG_H
#define CONFIG_H

#include <config.h>
#endif

#ifdef USE_LDAP

#ifdef HAVE_LBER_H
#include <lber.h>
#endif
#include <ldap.h>
#ifndef LDAP_OPT_SUCCESS
#define LDAP_OPT_SUCCESS LDAP_SUCCESS
#endif
#include <errno.h>
#include <sys/types.h>
#include <sys/time.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <time.h>
#include <libetpan/carray.h>
#include <libetpan/mmapstring.h>
#include "etpan-ldap-types.h"
#include "etpan-log.h"

struct etpan_ldap_session {
  struct etpan_ldap_config * config;
  LDAP * ld;
  MMAPString * filterbuf;
  int ld_errno;
  int sys_errno;
};

struct etpan_ldap_session *
etpan_ldap_session_new(struct etpan_ldap_config * cfg)
{
  struct etpan_ldap_session * session;

  session = malloc(sizeof(struct etpan_ldap_session));
  if (session == NULL)
    return NULL;

  session->filterbuf = mmap_string_new(NULL);
  if (session->filterbuf == NULL) {
    free(session);
    return NULL;
  }

  session->config = cfg;
  session->ld = NULL;
  session->ld_errno = LDAP_SUCCESS;
  session->sys_errno = 0;

  return session; 
}

/* maps LDAP errors to generic error returns */
static int etpan_ldap_ret(struct etpan_ldap_session * session) {
  switch (session->ld_errno) {
  case LDAP_SUCCESS:
    if (errno != 0)
      return ETPAN_LDAP_ERROR_SYSTEM;
    return ETPAN_LDAP_ERROR_INTERNAL;
    
  case LDAP_TIMELIMIT_EXCEEDED:
  case LDAP_SIZELIMIT_EXCEEDED:
  case LDAP_PARTIAL_RESULTS:
    return ETPAN_LDAP_ERROR_PARTIAL;

  case LDAP_INAPPROPRIATE_AUTH:
  case LDAP_INVALID_CREDENTIALS:
  case LDAP_INSUFFICIENT_ACCESS:
  case LDAP_AUTH_UNKNOWN:
    return ETPAN_LDAP_ERROR_AUTH;

  default:
    return ETPAN_LDAP_ERROR_LDAP;
  }
}

int etpan_ldap_connect(struct etpan_ldap_session * session)   
{
  int ret = ETPAN_LDAP_NO_ERROR;
  struct timeval tv = { session->config->timeout, 0 };
  int tls_value;
  
  session->ld = ldap_init(session->config->hostname, session->config->port);
  if (session->ld == NULL) {
    session->ld_errno = LDAP_SUCCESS;
    session->sys_errno = errno;
    ret = ETPAN_LDAP_ERROR_SYSTEM;
    goto fail;
  }

  if (session->config->sizelimit > 0) {
#ifdef LDAP_OPT_SIZELIMIT
    if (ldap_set_option(session->ld, LDAP_OPT_SIZELIMIT,
			&(session->config->sizelimit)) != LDAP_OPT_SUCCESS) {
      ret = ETPAN_LDAP_ERROR_INTERNAL;
      goto fail;
    }
#else
    session->ld->ld_sizelimit = session->config->sizelimit;
#endif
  }
    
  if (session->config->timeout > 0) {
#ifdef LDAP_OPT_TIMELIMIT
    if (ldap_set_option(session->ld, LDAP_OPT_TIMELIMIT,
			&(session->config->timeout)) != LDAP_OPT_SUCCESS) {
      ret = ETPAN_LDAP_ERROR_INTERNAL;
      goto fail;
    }
#else
    session->ld->ld_timelimit = session->config->timeout;
#endif

  /* OpenLDAP 2.x specific */
#ifdef LDAP_OPT_TIMEOUT
    ldap_set_option(session->ld, LDAP_OPT_TIMEOUT, &tv);
#endif
#ifdef LDAP_OPT_NETWORK_TIMEOUT    
    ldap_set_option(session->ld, LDAP_OPT_NETWORK_TIMEOUT, &tv);
#endif    
  }


#ifdef LDAP_OPT_PROTOCOL_VERSION
  if (ldap_set_option(session->ld, LDAP_OPT_PROTOCOL_VERSION,
		      &(session->config->version)) != LDAP_OPT_SUCCESS) {
    ret = ETPAN_LDAP_ERROR_INTERNAL;
    goto fail;
  }
#endif

#ifdef LDAP_OPT_X_TLS
  if (session->config->tls)
    tls_value = LDAP_OPT_X_TLS_HARD;
  else
    tls_value = LDAP_OPT_X_TLS_NEVER;
  if (ldap_set_option(session->ld, LDAP_OPT_X_TLS,
          &tls_value) != LDAP_OPT_SUCCESS) {
    ret = ETPAN_LDAP_ERROR_INTERNAL;
    goto fail;
  }
#endif
  
  /* XXX use asynchronous version for timeout */
  if (session->config->binddn != NULL &&
      ((session->ld_errno = ldap_simple_bind_s(session->ld, session->config->binddn,
					       session->config->bindpw))
       != LDAP_SUCCESS))
    goto fail;
  
  return ETPAN_LDAP_NO_ERROR;

 fail:
  if (session->ld != NULL) {
    ldap_unbind(session->ld);
    session->ld = NULL;
  }
  return (ret != ETPAN_LDAP_NO_ERROR) ? ret : etpan_ldap_ret(session);
}

void etpan_ldap_session_free(struct etpan_ldap_session * session)
{
  if (session->filterbuf)
    mmap_string_free(session->filterbuf);
  if (session->ld != NULL) 
    ldap_unbind(session->ld);
  free(session);
}


/* builds filter by expanding filter in filterbuf, replacing
   occurences of % per escaped value of key */
static int etpan_ldap_build_filter(struct etpan_ldap_session * session,
				   const char * key)
{
  const char * s, * p, * k, * e;
  char escape[4];

  mmap_string_truncate(session->filterbuf, 0);
  
  s = session->config->filter;
  while (s != NULL) {
    p = strchr(s, '%');
    /* insert len up to % or end */
    if (mmap_string_append_len(session->filterbuf,
            s, (p != NULL ?
                (size_t) (p - s) : (size_t) strlen(s))) == NULL) 
      goto fail;
    
    if (p == NULL)
      break;

    /* insert key with filter chars escaped */
    k = key;
    while (*k != '\0') {
      for (e = k; *e != '\0'; e++)
	if (*e == '\\' || *e == '*' || *e == '|' ||
	    *e == '=' || *e == '?' || *e == '!' ||
	    *e == '&' || *e == ')' || *e == '(' ||
	    *e == ',')
	  break;
      
      if (mmap_string_append_len(session->filterbuf, k, e - k) == NULL)
	goto fail;

      if (*e == '\0')
	break;

      sprintf(escape, "\\%x", *e);
      if (mmap_string_append_len(session->filterbuf, escape, 3) == NULL)
	goto fail;

      k = e + 1;
    }
    s = p + 1;
  }
  return ETPAN_LDAP_NO_ERROR;

 fail:
  session->sys_errno = errno;
  return ETPAN_LDAP_ERROR_SYSTEM;
}

static int etpan_ldap_dosearch(struct etpan_ldap_session * session,
			       char * base, int scope, char * filter,
			       char ** attrs, int retry, LDAPMessage **res) {
  int ret;
  struct timeval tv;

  while (retry >= 0) {
    /* connect if needed */
    if (session->ld == NULL &&
	(ret = etpan_ldap_connect(session)) != ETPAN_LDAP_NO_ERROR)
      return ret;
    
    if (session->config->timeout > 0) {
      tv.tv_sec = session->config->timeout;
      tv.tv_usec = 0;
      session->ld_errno = ldap_search_st(session->ld, base, scope, filter,
					 attrs, 0, &tv, res);
    } else
      session->ld_errno = ldap_search_s(session->ld, base, scope, filter,
					attrs, 0, res);
    
    if (session->ld_errno == LDAP_PROTOCOL_ERROR ||
	session->ld_errno == LDAP_TIMEOUT ||
	session->ld_errno == LDAP_SERVER_DOWN) {
      ldap_unbind(session->ld);
      session->ld = NULL;
      retry--;
      continue;
    }
    break;
  }
  return (session->ld_errno == LDAP_SUCCESS ||
	  session->ld_errno == LDAP_NO_SUCH_OBJECT) ?
    ETPAN_LDAP_NO_ERROR : etpan_ldap_ret(session);
}

int etpan_ldap_search(struct etpan_ldap_session * session,
		      const char * key, carray ** results)
{
  int res;
  LDAPMessage * msg = NULL;
  LDAPMessage * entry;
  char * attrs[6];
  unsigned int i;
  
  session->ld_errno = LDAP_SUCCESS;
  session->sys_errno = 0;
  *results = NULL;
  
  /* first build filter from filter and key */
  if ((res = etpan_ldap_build_filter(session, key)) != ETPAN_LDAP_NO_ERROR)
    return res;

  /* build list of wanted attributes */
  attrs[0] = session->config->attrs.mail != NULL ?
    session->config->attrs.mail : "mail";
  attrs[1] = session->config->attrs.cn != NULL ?
    session->config->attrs.cn : "cn";
  attrs[2] = session->config->attrs.sn != NULL ?
    session->config->attrs.sn : "sn";
  attrs[3] = session->config->attrs.givenname != NULL ?
    session->config->attrs.givenname : "givenName";
  attrs[4] = session->config->attrs.fullname != NULL ?
    session->config->attrs.givenname : "fullName";
  attrs[5] = NULL;
  
  ETPAN_LOG("ldap search %s %s", session->config->base, session->filterbuf->str);
  
  msg = NULL;
  res = etpan_ldap_dosearch(session, 
      session->config->base != NULL ? session->config->base : "",
      LDAP_SCOPE_SUBTREE,
      session->filterbuf->str, attrs, 1, &msg);
  if (res != ETPAN_LDAP_NO_ERROR && (res != ETPAN_LDAP_ERROR_PARTIAL || msg == NULL)) {
    if (msg != NULL)
      ldap_msgfree(msg);
    return (res != ETPAN_LDAP_ERROR_PARTIAL ? res : ETPAN_LDAP_ERROR_LDAP);
  }

  if (msg == NULL) /* no entries */
    return ETPAN_LDAP_NO_ERROR;

  if ((*results = carray_new(16)) == NULL) {
    ldap_msgfree(msg);
    session->sys_errno = errno;
    return ETPAN_LDAP_ERROR_SYSTEM;
  }
  
  for (entry = ldap_first_entry(session->ld, msg);
       entry != NULL;
       entry = ldap_next_entry(session->ld, entry)) {
    char ** mail_list;
    char ** cn_list;
    char ** sn_list;
    char ** givenname_list;
    char ** fullname_list;
    char * fullname;
    char * givenname;
    char * sn;
    
    mail_list = NULL;
    cn_list = NULL;
    sn_list = NULL;
    givenname_list = NULL;
    fullname_list = NULL;
    fullname = NULL;
    givenname = NULL;
    sn = NULL;
    
    /* first check the mail attribute : if we don't have any, skip this entry */
    if ((mail_list = ldap_get_values(session->ld, entry, attrs[0])) == NULL ||
	mail_list[0] == NULL) {
      continue;
    }
    
    /* then check for a full name if available */
    if (fullname == NULL) {
      fullname_list = ldap_get_values(session->ld, entry, attrs[4]);
      if (fullname_list != NULL) {
        if (fullname_list[0] != NULL) {
          fullname = strdup(fullname_list[0]);
        }
      }
      ldap_value_free(fullname_list);
      fullname_list = NULL;
    }
    if (fullname == NULL) {
      cn_list = ldap_get_values(session->ld, entry, attrs[1]);
      if (cn_list != NULL) {
        if (cn_list[0] != NULL) {
          fullname = strdup(cn_list[0]);
        }
      }
      ldap_value_free(cn_list);
      cn_list = NULL;
    }
    
    if (givenname == NULL) {
      givenname_list = ldap_get_values(session->ld, entry, attrs[3]);
      if (givenname_list != NULL) {
        if (givenname_list[0] != NULL) {
          givenname = strdup(givenname_list[0]);
        }
      }
      ldap_value_free(givenname_list);
      givenname_list = NULL;
    }
    
    if (sn == NULL) {
      sn_list = ldap_get_values(session->ld, entry, attrs[2]);
      if (sn_list != NULL) {
        if (sn_list[0] != NULL) {
          sn = strdup(sn_list[0]);
        }
      }
      ldap_value_free(sn_list);
      sn_list = NULL;
    }
    
    /* then loop around each mail address and add a abook entry for each */
    for(i = 0; mail_list[i] != NULL; i++) {
      struct etpan_ldap_abook_entry * entry;
      
      entry = calloc(1, sizeof(struct etpan_ldap_abook_entry));
      if (entry == NULL) {
	goto free_common;
      }
      
      if (fullname != NULL)
        entry->fullname = strdup(fullname);
      else
        entry->fullname = NULL;
      if (givenname != NULL)
        entry->firstname = strdup(givenname);
      else
        entry->firstname = NULL;
      if (entry->lastname != NULL)
        entry->lastname = strdup(sn);
      else
        entry->lastname = NULL;
      entry->address = strdup(mail_list[i]);
      if (entry->address == NULL)
        goto free_entry;
      
      if (carray_add(*results, entry, NULL) < 0) {
	goto free_entry;
      }
      
      continue;
      
    free_entry:
      free(entry->address);
      free(entry->firstname);
      free(entry->lastname);
      free(entry->fullname);
      free(entry);
    free_common:
      free(fullname);
      free(givenname);
      free(sn);
      goto fail;
    }
    free(fullname);
    free(givenname);
    free(sn);
    ldap_value_free(mail_list);
  }
  
  ldap_msgfree(msg);
  return etpan_ldap_ret(session) != ETPAN_LDAP_ERROR_PARTIAL ?
    ETPAN_LDAP_NO_ERROR : ETPAN_LDAP_ERROR_PARTIAL;
  /* XXX handle errors while fetching entries */

 fail:
  if (*results != NULL) {
    for (i = 0; i < carray_count(*results); i++) {
      struct etpan_ldap_abook_entry * entry;
      
      entry = carray_get(* results, i);
      free(entry->address);
      free(entry->firstname);
      free(entry->lastname);
      free(entry->fullname);
      free(entry);
    }
    carray_free(*results);
    *results = NULL;
  }
  ldap_msgfree(msg);

  return ETPAN_LDAP_ERROR_SYSTEM;
}

char * etpan_ldap_error(struct etpan_ldap_session * session)
{
  if (session->ld_errno != LDAP_SUCCESS)
    return ldap_err2string(session->ld_errno);
  if (session->sys_errno != 0)
    return strerror(session->sys_errno);
  return "Unknown LDAP Error";
}

#endif
