#include "etpan-filter-action.h"

#ifndef ETPAN_MODULE_LOG_NAME
#define ETPAN_MODULE_LOG_NAME "FILTER"
#endif

#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "etpan-log.h"
#include "etpan-message-copy.h"
#include "etpan-message-delete.h"
#include "etpan-message-archive.h"
#include "etpan-signal.h"
#include "etpan-message.h"
#include "etpan-account-manager.h"
#include "etpan-account.h"
#include "etpan-storage.h"
#include "etpan-folder.h"
#include "etpan-error.h"
#include "etpan-serialize.h"
#include "etpan-thread-manager.h"
#include "etpan-thread-manager-app.h"
#include "etpan-message-header.h"
#include "etpan-message-color.h"
#include "etpan-mail-manager.h"
#include "etpan-message-fetcher.h"
#include "etpan-utils.h"
#include "etpan-message-composer.h"
#include "etpan-text-wrapper.h"
#include "etpan-address.h"

static struct etpan_filter_action_description * get_description(char * name);

struct etpan_filter_action * etpan_filter_action_new(char * name,
    carray * param_list)
{
  struct etpan_filter_action * action;
  struct etpan_filter_action_description * description;
  
  description = get_description(name);
  if (description == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  ETPAN_LOCAL_LOG("create action: %p", param_list);
  action = malloc(sizeof(* action));
  if (action == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  action->description = description;
  action->param_list = param_list;
  
  return action;
}

void etpan_filter_action_free(struct etpan_filter_action * action)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(action->param_list) ; i ++) {
    struct etpan_serialize_data * param;
    
    param = carray_get(action->param_list, i);
    etpan_serialize_data_free(param);
  }
  carray_free(action->param_list);
  free(action);
}

void etpan_filter_action_run(struct etpan_filter_state * filter_state,
    struct etpan_filter_action * action,
    struct etpan_message * msg,
    void (* callback)(int stop, void * cb_data), void * cb_data)
{
  action->description->run(action, filter_state, msg, callback, cb_data);
}

struct etpan_filter_action * etpan_filter_action_dup(struct etpan_filter_action * action)
{
  unsigned int i;
  struct etpan_filter_action * action_dup;
  carray * param_list;
  int r;
  
  param_list = carray_new(4);
  if (param_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  ETPAN_LOCAL_LOG("action: %s %p", action->description->name,
      action->param_list);
  for(i = 0 ; i < carray_count(action->param_list) ; i ++) {
    struct etpan_serialize_data * param;
    struct etpan_serialize_data * param_dup;
    
    param = carray_get(action->param_list, i);
    param_dup = etpan_serialize_data_dup(param);
    
    r = carray_add(param_list, param_dup, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  action_dup = etpan_filter_action_new(action->description->name, param_list);
  
  return action_dup;
}

struct run_action_list_data {
  struct etpan_filter_state * filter_state;
  carray * action_list;
  struct etpan_message * msg;
  void (* callback)(void * cb_data);
  void * cb_data;
  unsigned int current_action;
  int stop;
};

static void filter_action_list_run_next_action(struct run_action_list_data * data);

void etpan_filter_action_list_run(struct etpan_filter_state * filter_state,
    carray * action_list,
    struct etpan_message * msg,
    void (* callback)(void * cb_data), void * cb_data)
{
  struct run_action_list_data * data;
  
  data = malloc(sizeof(* data));
  if (data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  data->filter_state = filter_state;
  data->action_list = action_list;
  data->msg = msg;
  data->callback = callback;
  data->cb_data = cb_data;
  data->current_action = 0;
  data->stop = 0;
  
  filter_action_list_run_next_action(data);
}

static void filter_action_list_run_callback(struct run_action_list_data * data);
static void filter_action_list_run_action_callback(int stop, void * data);

static void filter_action_list_run_next_action(struct run_action_list_data * data)
{
  struct etpan_filter_action * action;
  
  if ((data->current_action >= carray_count(data->action_list)) || data->stop) {
    filter_action_list_run_callback(data);
    return;
  }
  
  action = carray_get(data->action_list, data->current_action);
  etpan_filter_action_run(data->filter_state, action,
      data->msg,
      filter_action_list_run_action_callback, data);
}

static void filter_action_list_run_action_callback(int stop, void * data)
{
  struct run_action_list_data * action_list_data;
  
  action_list_data = data;
  action_list_data->stop = stop;
  action_list_data->current_action ++;
  
  filter_action_list_run_next_action(action_list_data);
}

static void filter_action_list_run_callback(struct run_action_list_data * data)
{
  data->callback(data->cb_data);
  free(data);
}

static chash * action_description_hash = NULL;

static struct etpan_filter_action_description * get_description(char * name)
{
  chashdatum key;
  chashdatum value;
  int r;
  
  key.data = name;
  key.len = strlen(name);
  r = chash_get(action_description_hash, &key, &value);
  if (r < 0)
    ETPAN_CRASH("action description name not valid %s", name);
  
  return value.data;
}

static void unregister_all(void)
{
  chashiter * iter;
  carray * table;
  unsigned int i;
  int r;
  
  table = carray_new(16);
  if (table == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  for(iter = chash_begin(action_description_hash) ; iter != NULL ;
      iter = chash_next(action_description_hash, iter)) {
    chashdatum value;
    struct etpan_filter_action_description * description;
    
    chash_value(iter, &value);
    description = value.data;
    r = carray_add(table, description->name, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  for(i = 0 ; i < carray_count(table) ; i ++) {
    char * name;
    
    name = carray_get(table, i);
    etpan_filter_action_unregister_description(name);
  }
  
  carray_free(table);
}

static void register_default_action(void);

void etpan_filter_action_register_default(void)
{
  register_default_action();
}

void etpan_filter_action_unregister_all(void)
{
  unregister_all();
}

void etpan_filter_action_register_description(struct etpan_filter_action_description * description)
{
  chashdatum key;
  chashdatum value;
  int r;
  
  if (action_description_hash == NULL) {
    action_description_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
    if (action_description_hash == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  key.data = description->name;
  key.len = strlen(description->name);
  value.data = description;
  value.len = 0;
  r = chash_set(action_description_hash, &key, &value, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

void etpan_filter_action_unregister_description(char * name)
{
  chashdatum key;
  
  key.data = name;
  key.len = strlen(name);
  chash_delete(action_description_hash, &key, NULL);
}

/* action implementation */

static void action_stop_run(struct etpan_filter_action * action,
    struct etpan_filter_state * filter_state,
    struct etpan_message * msg,
    void (* callback)(int stop, void * cb_data), void * cb_data);

static void action_move_run(struct etpan_filter_action * action,
    struct etpan_filter_state * filter_state,
    struct etpan_message * msg,
    void (* callback)(int stop, void * cb_data), void * cb_data);

static void action_copy_run(struct etpan_filter_action * action,
    struct etpan_filter_state * filter_state,
    struct etpan_message * msg,
    void (* callback)(int stop, void * cb_data), void * cb_data);

static void action_show_message_run(struct etpan_filter_action * action,
    struct etpan_filter_state * filter_state,
    struct etpan_message * msg,
    void (* callback)(int stop, void * cb_data), void * cb_data);

static void action_color_run(struct etpan_filter_action * action,
    struct etpan_filter_state * filter_state,
    struct etpan_message * msg,
    void (* callback)(int stop, void * cb_data), void * cb_data);

static void action_delete_run(struct etpan_filter_action * action,
    struct etpan_filter_state * filter_state,
    struct etpan_message * msg,
    void (* callback)(int stop, void * cb_data), void * cb_data);

static void action_archive_run(struct etpan_filter_action * action,
    struct etpan_filter_state * filter_state,
    struct etpan_message * msg,
    void (* callback)(int stop, void * cb_data), void * cb_data);

static void action_mark_as_seen_run(struct etpan_filter_action * action,
    struct etpan_filter_state * filter_state,
    struct etpan_message * msg,
    void (* callback)(int stop, void * cb_data), void * cb_data);

static void action_mark_as_flagged_run(struct etpan_filter_action * action,
    struct etpan_filter_state * filter_state,
    struct etpan_message * msg,
    void (* callback)(int stop, void * cb_data), void * cb_data);

static void action_execute_run(struct etpan_filter_action * action,
    struct etpan_filter_state * filter_state,
    struct etpan_message * msg,
    void (* callback)(int stop, void * cb_data), void * cb_data);

static void action_reply_run(struct etpan_filter_action * action,
    struct etpan_filter_state * filter_state,
    struct etpan_message * msg,
    void (* callback)(int stop, void * cb_data), void * cb_data);

static void action_forward_run(struct etpan_filter_action * action,
    struct etpan_filter_state * filter_state,
    struct etpan_message * msg,
    void (* callback)(int stop, void * cb_data), void * cb_data);

static struct etpan_filter_action_description
action_description_list[] = {
  {"stop", action_stop_run},
  {"move", action_move_run},
  {"copy", action_copy_run},
  {"show", action_show_message_run},
  {"color", action_color_run},
  {"delete", action_delete_run},
  {"archive", action_archive_run},
  {"mark_as_seen", action_mark_as_seen_run},
  {"mark_as_flagged", action_mark_as_flagged_run},
  {"execute", action_execute_run},
  {"reply", action_reply_run},
  {"forward", action_forward_run},
};

static void register_default_action(void)
{
  unsigned int i;
  
  for(i = 0 ; i < sizeof(action_description_list) / sizeof(action_description_list[0]) ; i ++) {
    etpan_filter_action_register_description(&action_description_list[i]);
  }
}

/* ************* */
/* actions */

static char * get_str_from_param(carray * param_list, unsigned int i)
{
  char * str;
  struct etpan_serialize_data * param;
  
  param = carray_get(param_list, i);
  str = etpan_serialize_data_get_str(param);
  
  return str;
}

static int64_t get_int_from_param(carray * param_list, unsigned int i)
{
  int64_t value;
  struct etpan_serialize_data * param;
  
  param = carray_get(param_list, i);
  value = etpan_serialize_data_get_int64(param);
  
  return value;
}

/* stop */

static void action_stop_run(struct etpan_filter_action * action,
    struct etpan_filter_state * filter_state,
    struct etpan_message * msg,
    void (* callback)(int stop, void * cb_data), void * cb_data)
{
  callback(1, cb_data);
}

/* move/copy */

struct move_data {
  int do_delete;
  struct etpan_message_copy * copy;
  void (* callback)(int result, void * cb_data);
  void * cb_data;
};

static void copy_with_delete_done_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);

static void action_copy_with_delete_run(struct etpan_filter_action * action,
    struct etpan_filter_state * filter_state,
    struct etpan_message * msg, int do_delete,
    void (* callback)(int result, void * cb_data), void * cb_data)
{
  struct etpan_message_copy * copy;
  char * uid;
  chash * msglist;
  struct etpan_folder * folder;
  struct etpan_storage * storage;
  struct etpan_account * account;
  chashdatum key;
  chashdatum value;
  int r;
  char * account_name;
  char * folder_name;
  struct etpan_serialize_data * param_value;
  struct move_data * move_data;
  
  ETPAN_ASSERT(carray_count(action->param_list) == 2, "should have 2 parameters");
  
  param_value = carray_get(action->param_list, 0);
  account_name = etpan_serialize_data_get_str(param_value);
  param_value = carray_get(action->param_list, 1);
  folder_name = etpan_serialize_data_get_str(param_value);
  
  move_data = malloc(sizeof(* move_data));
  if (move_data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  move_data->callback = callback;
  move_data->cb_data = cb_data;
  move_data->do_delete = do_delete;
  
  copy = etpan_message_copy_new();
  move_data->copy = copy;
  etpan_message_copy_set_delete(copy, do_delete);
  msglist = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (msglist == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  uid = etpan_message_get_uid(msg);
  key.data = uid;
  key.len = strlen(uid) + 1;
  value.data = msg;
  value.len = 0;
  r = chash_set(msglist, &key, &value, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  etpan_message_copy_set_msglist(copy, msglist);
  chash_free(msglist);
  
  account = etpan_account_manager_get_account(etpan_account_manager_get_default(),
      account_name);
  ETPAN_ASSERT(account != NULL, "account should not be NULL");
  storage = etpan_account_get_storage(account);
  ETPAN_ASSERT(storage != NULL, "storage should not be NULL");
  folder = etpan_storage_get_folder(storage, folder_name);
  ETPAN_ASSERT(folder != NULL, "folder should not be NULL");
  
  etpan_message_copy_set_destination(copy, folder);
  etpan_message_copy_setup(copy);
  
  ETPAN_SIGNAL_ADD_HANDLER(copy, ETPAN_MESSAGE_COPY_FINISHED_SIGNAL,
      copy_with_delete_done_handler, move_data);
  
  etpan_message_copy_run(copy);
}

static void copy_with_delete_done_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct move_data * move_data;
  struct etpan_message_copy * copy;
  struct etpan_error * error;
  
  move_data = user_data;
  copy = move_data->copy;
  
  ETPAN_SIGNAL_REMOVE_HANDLER(copy, ETPAN_MESSAGE_COPY_FINISHED_SIGNAL,
      copy_with_delete_done_handler, move_data);
  
  error = etpan_message_copy_get_error(copy);
  if (error != NULL) {
    etpan_error_log(error);
  }
  
  etpan_message_copy_unref(copy);
  
  if (move_data->do_delete) {
    move_data->callback(1, move_data->cb_data);
  }
  else {
    move_data->callback(0, move_data->cb_data);
  }
  
  free(move_data);
}

static void action_move_run(struct etpan_filter_action * action,
    struct etpan_filter_state * filter_state,
    struct etpan_message * msg,
    void (* callback)(int result, void * cb_data), void * cb_data)
{
  action_copy_with_delete_run(action,
      filter_state,
      msg, 1,
      callback, cb_data);
}

static void action_copy_run(struct etpan_filter_action * action,
    struct etpan_filter_state * filter_state,
    struct etpan_message * msg,
    void (* callback)(int result, void * cb_data), void * cb_data)
{
  action_copy_with_delete_run(action,
      filter_state,
      msg, 0,
      callback, cb_data);
}

/* show message */

static void action_show_message_run(struct etpan_filter_action * action,
    struct etpan_filter_state * filter_state,
    struct etpan_message * msg,
    void (* callback)(int stop, void * cb_data), void * cb_data)
{
  ETPAN_ASSERT(carray_count(action->param_list) == 0, "should have 0 parameters");
  
  ETPAN_LOCAL_LOG("%s", etpan_message_get_description(msg));
  callback(0, cb_data);
}

/* color */

static void action_color_run(struct etpan_filter_action * action,
    struct etpan_filter_state * filter_state,
    struct etpan_message * msg,
    void (* callback)(int stop, void * cb_data), void * cb_data)
{
  struct etpan_message_header * header;
  char * msgid;
  char * color;
  
  ETPAN_ASSERT(carray_count(action->param_list) == 1, "should have 1 parameter");
  
  color = get_str_from_param(action->param_list, 0);
  
  header = etpan_message_get_header(msg);
  msgid = etpan_message_header_get_msgid(header);
  if (msgid == NULL) {
    callback(0, cb_data);
    return;
  }
  
  etpan_message_color_set_color(etpan_message_color_get_default(),
      msgid, color);
  
  callback(0, cb_data);
}

/* delete */

struct delete_data {
  struct etpan_message_delete * delete;
  void (* callback)(int stop, void * cb_data);
  void * cb_data;
};

static void delete_done_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);

static void action_delete_run(struct etpan_filter_action * action,
    struct etpan_filter_state * filter_state,
    struct etpan_message * msg,
    void (* callback)(int stop, void * cb_data), void * cb_data)
{
  struct etpan_message_delete * delete;
  char * uid;
  chash * msglist;
  struct etpan_folder * folder;
  struct etpan_storage * storage;
  struct etpan_account * account;
  chashdatum key;
  chashdatum value;
  int r;
  struct delete_data * delete_data;
  
  ETPAN_ASSERT(carray_count(action->param_list) == 0, "should have 0 parameters");
  
  delete_data = malloc(sizeof(* delete_data));
  if (delete_data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  delete_data->callback = callback;
  delete_data->cb_data = cb_data;
  
  delete = etpan_message_delete_new();
  delete_data->delete = delete;
  
  msglist = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (msglist == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  uid = etpan_message_get_uid(msg);
  key.data = uid;
  key.len = strlen(uid) + 1;
  value.data = msg;
  value.len = 0;
  r = chash_set(msglist, &key, &value, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  etpan_message_delete_set_msglist(delete, msglist);
  chash_free(msglist);
  
  etpan_message_delete_setup(delete);
  
  ETPAN_SIGNAL_ADD_HANDLER(delete, ETPAN_MESSAGE_DELETE_FINISHED_SIGNAL,
      delete_done_handler, delete_data);
  
  etpan_message_delete_run(delete);
}

static void delete_done_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct delete_data * delete_data;
  struct etpan_message_delete * delete;
  struct etpan_error * error;
  
  delete_data = user_data;
  delete = delete_data->delete;
  
  ETPAN_SIGNAL_REMOVE_HANDLER(delete, ETPAN_MESSAGE_DELETE_FINISHED_SIGNAL,
      delete_done_handler, delete_data);
  
  error = etpan_message_delete_get_error(delete);
  if (error != NULL) {
    etpan_error_log(error);
  }
  
  etpan_message_delete_unref(delete);
  
  delete_data->callback(1, delete_data->cb_data);
  
  free(delete_data);
}

/* archive */

struct archive_data {
  struct etpan_message_archive * archive;
  void (* callback)(int stop, void * cb_data);
  void * cb_data;
};

static void archive_done_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);

static void action_archive_run(struct etpan_filter_action * action,
    struct etpan_filter_state * filter_state,
    struct etpan_message * msg,
    void (* callback)(int stop, void * cb_data), void * cb_data)
{
  struct etpan_message_archive * archive;
  char * uid;
  chash * msglist;
  struct etpan_folder * folder;
  struct etpan_storage * storage;
  struct etpan_account * account;
  chashdatum key;
  chashdatum value;
  int r;
  struct archive_data * archive_data;
  
  ETPAN_ASSERT(carray_count(action->param_list) == 0, "should have 0 parameters");
  
  archive_data = malloc(sizeof(* archive_data));
  if (archive_data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  archive_data->callback = callback;
  archive_data->cb_data = cb_data;
  
  archive = etpan_message_archive_new();
  archive_data->archive = archive;
  
  msglist = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (msglist == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  uid = etpan_message_get_uid(msg);
  key.data = uid;
  key.len = strlen(uid) + 1;
  value.data = msg;
  value.len = 0;
  r = chash_set(msglist, &key, &value, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  etpan_message_archive_set_msglist(archive, msglist);
  chash_free(msglist);
  
  etpan_message_archive_setup(archive);
  
  ETPAN_SIGNAL_ADD_HANDLER(archive, ETPAN_MESSAGE_ARCHIVE_FINISHED_SIGNAL,
      archive_done_handler, archive_data);
  
  etpan_message_archive_run(archive);
}

static void archive_done_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct archive_data * archive_data;
  struct etpan_message_archive * archive;
  struct etpan_error * error;
  
  archive_data = user_data;
  archive = archive_data->archive;
  
  ETPAN_SIGNAL_REMOVE_HANDLER(archive, ETPAN_MESSAGE_ARCHIVE_FINISHED_SIGNAL,
      archive_done_handler, archive_data);
  
  error = etpan_message_archive_get_error(archive);
  if (error != NULL) {
    etpan_error_log(error);
  }
  
  etpan_message_archive_unref(archive);
  
  archive_data->callback(1, archive_data->cb_data);
  
  free(archive_data);
}

/* mark as seen */

static void action_mark_as_seen_run(struct etpan_filter_action * action,
    struct etpan_filter_state * filter_state,
    struct etpan_message * msg,
    void (* callback)(int stop, void * cb_data), void * cb_data)
{
  struct etpan_message_flags * flags;
  int value;
  struct etpan_folder * folder;
  
  flags = etpan_message_get_flags(msg);
  value = etpan_message_flags_get_value(flags);
  value |= ETPAN_FLAGS_SEEN;
  etpan_message_flags_set_value(flags, value);
  etpan_message_set_flags(msg, flags);
  
  folder = etpan_message_get_folder(msg);
  etpan_mail_manager_folder_check(etpan_mail_manager_get_default(), folder);
  
  callback(0, cb_data);
}

/* mark as flagged */

static void action_mark_as_flagged_run(struct etpan_filter_action * action,
    struct etpan_filter_state * filter_state,
    struct etpan_message * msg,
    void (* callback)(int stop, void * cb_data), void * cb_data)
{
  struct etpan_message_flags * flags;
  int value;
  struct etpan_folder * folder;
  
  flags = etpan_message_get_flags(msg);
  value = etpan_message_flags_get_value(flags);
  value |= ETPAN_FLAGS_FLAGGED;
  etpan_message_flags_set_value(flags, value);
  etpan_message_set_flags(msg, flags);
  
  folder = etpan_message_get_folder(msg);
  etpan_mail_manager_folder_check(etpan_mail_manager_get_default(), folder);
  
  callback(0, cb_data);
}

/* execute */

struct execute_data {
  int result;
  char * command;
  struct etpan_message * msg;
  void (* callback)(int result, void * cb_data);
  void * data;
  struct etpan_message_fetcher * fetcher;
};

static void action_execute_content_fetched(char * signal_name, void * sender,
    void * signal_data, void * user_data);

static void action_execute_run(struct etpan_filter_action * action,
    struct etpan_filter_state * filter_state,
    struct etpan_message * msg,
    void (* callback)(int stop, void * cb_data), void * cb_data)
{
  struct execute_data * execute_data;
  struct etpan_message_fetcher * fetcher;
  char * command;
  
  ETPAN_ASSERT(carray_count(action->param_list) == 1, "should have 1 parameters");
  
  command = get_str_from_param(action->param_list, 0);
  
  execute_data = malloc(sizeof(* execute_data));
  if (execute_data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  execute_data->result = 0;
  execute_data->msg = msg;
  execute_data->command = strdup(command);
  if (execute_data->command == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  execute_data->callback = callback;
  execute_data->data = cb_data;
  execute_data->fetcher = NULL;
  
  fetcher = etpan_message_fetcher_new();
  etpan_message_fetcher_set_flags(fetcher,
      ETPAN_MESSAGE_FETCHER_FLAGS_SOURCE);
  etpan_message_fetcher_set_message(fetcher, msg);
  etpan_message_fetcher_setup(fetcher);
  execute_data->fetcher = fetcher;
  
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_MESSAGE_FETCHER_FINISHED_SIGNAL, fetcher, execute_data,
      action_execute_content_fetched);
  
  etpan_message_fetcher_run(fetcher);
}

static void action_execute_op_run(struct etpan_thread_op * op);
static void action_execute_callback(int cancelled, void * result,
    void * data);

static void action_execute_content_fetched(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_message_fetcher * fetcher;
  struct execute_data * execute_data;
  struct etpan_thread_op * op;
  
  fetcher = sender;
  execute_data = user_data;
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_MESSAGE_FETCHER_FINISHED_SIGNAL, fetcher, execute_data,
      action_execute_content_fetched);
  
  if (etpan_message_fetcher_get_error(fetcher) != NULL) {
    action_execute_callback(0, NULL, execute_data);
    return;
  }
  
  op = etpan_thread_op_new();
  op->param = execute_data;
  op->result = NULL;
  
  op->cancellable = 0;
  op->run = action_execute_op_run;
  op->callback = (void (*)(int, void *, void *)) action_execute_callback;
  op->callback_data = execute_data;
  op->cleanup = NULL;
  
  etpan_thread_manager_app_misc_schedule(etpan_thread_manager_app_get_default(),
      op);
}

static void action_execute_op_run(struct etpan_thread_op * op)
{
  struct execute_data * execute_data;
  char command[PATH_MAX];
  carray * list;
  struct etpan_part_fetch_info * part_info;
  char tmp_path[PATH_MAX];
  char quoted_tmp_path[PATH_MAX];
  char quoted_command[PATH_MAX];
  char * content;
  size_t content_length;
  int r;
  
  execute_data = op->param;
  
  list = etpan_message_fetcher_get_part_list(execute_data->fetcher);
  if (carray_count(list) == 0)
    goto err;
  
  part_info = carray_get(list, 0);
  etpan_part_fetch_info_get_content(part_info, &content, &content_length);
  r = etpan_get_tmp_path(tmp_path, sizeof(tmp_path));
  if (r < 0) {
    goto err;
  }
  r = etpan_write_file(tmp_path, content, content_length);
  if (r < 0) {
    goto err;
  }
  
  r = etpan_quote_filename(quoted_tmp_path, sizeof(quoted_tmp_path), tmp_path);
  if (r < 0) {
    goto err;
  }
  r = etpan_quote_filename(quoted_command, sizeof(quoted_command),
      execute_data->command);
  if (r < 0) {
    goto err;
  }
  
  snprintf(command, sizeof(command), "%s %s",
      quoted_command, quoted_tmp_path);
  
  system(command);
  
  unlink(tmp_path);
  
  return;
  
 err:
  execute_data->result = 1;
  etpan_message_fetcher_unref(execute_data->fetcher);
  execute_data->fetcher = NULL;
}

static void action_execute_callback(int cancelled, void * result,
    void * data)
{
  struct execute_data * execute_data;
  
  execute_data = data;
  
  execute_data->callback(execute_data->result, execute_data->data);
  
  free(execute_data->command);
  free(execute_data);
}

struct reply_data {
  struct etpan_message * msg;
  char * text;
  char * recipient;
  size_t size;
  void (* callback)(int stop, void * cb_data);
  void * cb_data;
  struct etpan_message_composer * composer;
};

#define WRAP_COUNT 72

static void reply_callback(char * signal_name, void * sender,
    void * signal_data, void * user_data);
static void reply_sent(struct etpan_error * error, void * cb_data);

static void action_reply_run(struct etpan_filter_action * action,
    struct etpan_filter_state * filter_state,
    struct etpan_message * msg,
    void (* callback)(int stop, void * cb_data), void * cb_data)
{
  struct etpan_message_composer * composer;
  struct etpan_folder * folder;
  struct etpan_storage *storage;
  struct etpan_account * account;
  struct reply_data * data;
  char * text;
  
  ETPAN_ASSERT(carray_count(action->param_list) == 1, "should have 1 parameters");
  
  text = get_str_from_param(action->param_list, 0);
  
  composer = etpan_message_composer_new();
  folder = etpan_message_get_folder(msg);
  storage = etpan_folder_get_storage(folder);
  account = etpan_storage_get_account(storage);
  etpan_message_composer_set_account(composer, account);
  
  data = malloc(sizeof(* data));
  if (data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  data->msg = msg;
  data->text = strdup(text);
  if (data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  data->recipient = NULL;
  data->size = strlen(text);
  data->callback = callback;
  data->cb_data = cb_data;
  data->composer = composer;
  
  ETPAN_SIGNAL_ADD_HANDLER(composer,
      ETPAN_MESSAGE_COMPOSER_READY_SIGNAL, reply_callback,
      data);
  
  etpan_message_composer_reply(composer, msg, ETPAN_REPLY_TYPE_NORMAL);
}

static void reply_callback(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct reply_data * data;
  struct etpan_message_composer * composer;
  struct etpan_error * error;
  char * wrapped;
  size_t wrapped_size;
  
  composer = sender;
  data = user_data;
  
  ETPAN_SIGNAL_REMOVE_HANDLER(composer,
      ETPAN_MESSAGE_COMPOSER_READY_SIGNAL, reply_callback,
      data);
  
  error = etpan_message_composer_reply_get_error(composer);
  if (error != NULL) {
    data->callback(1, data->cb_data);
    etpan_message_composer_free(composer);
    free(data->text);
    free(data);
    return;
  }
  
  etpan_wrap_text(data->text, data->size, &wrapped, &wrapped_size, WRAP_COUNT);
  etpan_message_composer_preprend_text(composer, wrapped, wrapped_size);
  free(wrapped);
  
  etpan_message_composer_send(composer, reply_sent, data);
}

static void reply_sent(struct etpan_error * error, void * cb_data)
{
  struct reply_data * data;
  
  data = cb_data;
  
  etpan_message_composer_free(data->composer);
  free(data->text);
  free(data);
}

static void forward_callback(char * signal_name, void * sender,
    void * signal_data, void * user_data);
static void forward_sent(struct etpan_error * error, void * cb_data);

static void action_forward_run(struct etpan_filter_action * action,
    struct etpan_filter_state * filter_state,
    struct etpan_message * msg,
    void (* callback)(int stop, void * cb_data), void * cb_data)
{
  struct etpan_message_composer * composer;
  struct etpan_folder * folder;
  struct etpan_storage *storage;
  struct etpan_account * account;
  struct reply_data * data;
  char * text;
  char * recipient;
  
  ETPAN_ASSERT(carray_count(action->param_list) == 2, "should have 2 parameters");
  
  text = get_str_from_param(action->param_list, 0);
  recipient = get_str_from_param(action->param_list, 1);
  
  composer = etpan_message_composer_new();
  folder = etpan_message_get_folder(msg);
  storage = etpan_folder_get_storage(folder);
  account = etpan_storage_get_account(storage);
  etpan_message_composer_set_account(composer, account);
  
  data = malloc(sizeof(* data));
  if (data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  data->msg = msg;
  data->text = strdup(text);
  if (data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  data->recipient = strdup(recipient);
  if (recipient == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  data->size = strlen(text);
  data->callback = callback;
  data->cb_data = cb_data;
  data->composer = composer;
  
  ETPAN_SIGNAL_ADD_HANDLER(composer,
      ETPAN_MESSAGE_COMPOSER_READY_SIGNAL, forward_callback,
      data);
  
  etpan_message_composer_forward(composer, msg, ETPAN_FORWARD_TYPE_NORMAL);
}

static void forward_callback(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct reply_data * data;
  struct etpan_message_composer * composer;
  struct etpan_error * error;
  char * wrapped;
  size_t wrapped_size;
  struct etpan_message_header * header;
  carray * address_list;
  struct etpan_address * address;
  
  composer = sender;
  data = user_data;
  
  ETPAN_SIGNAL_REMOVE_HANDLER(composer,
      ETPAN_MESSAGE_COMPOSER_READY_SIGNAL, reply_callback,
      data);
  
  error = etpan_message_composer_reply_get_error(composer);
  if (error != NULL) {
    data->callback(1, data->cb_data);
    etpan_message_composer_free(data->composer);
    free(data->recipient);
    free(data->text);
    free(data);
    return;
  }
  
  header = etpan_message_composer_get_header(composer);
  address_list = etpan_address_list_new();
  address = etpan_address_new();
  etpan_address_set_address(address, data->recipient);
  etpan_address_list_add_address(address_list, address);
  etpan_address_list_free(address_list);
  
  etpan_wrap_text(data->text, data->size, &wrapped, &wrapped_size, WRAP_COUNT);
  etpan_message_composer_preprend_text(composer, wrapped, wrapped_size);
  free(wrapped);
  
  etpan_message_composer_send(composer, forward_sent, data);
}

static void forward_sent(struct etpan_error * error, void * cb_data)
{
  struct reply_data * data;
  
  data = cb_data;
  
  etpan_message_composer_free(data->composer);
  free(data->text);
  free(data);
}

char * etpan_filter_action_get_description_name(struct etpan_filter_action * action)
{
  return action->description->name;
}

carray * etpan_filter_action_get_param_list(struct etpan_filter_action * action)
{
  return action->param_list;
}
