#include "etpan-filter-condition.h"

#ifndef ETPAN_MODULE_LOG_NAME
#define ETPAN_MODULE_LOG_NAME "FILTER"
#endif

#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>

#include "etpan-log.h"
#include "etpan-address.h"
#include "etpan-message.h"
#include "etpan-message-header.h"
#include "etpan-message-fetcher.h"
#include "etpan-part-header.h"
#include "etpan-signal.h"
#include "etpan-error.h"
#include "etpan-part.h"
#include "etpan-serialize.h"
#include "etpan-folder.h"
#include "etpan-storage.h"
#include "etpan-account.h"
#include "etpan-abook.h"
#include "etpan-abook-request.h"
#include "etpan-abook-manager.h"
#include "etpan-thread-manager-app.h"
#include "etpan-utils.h"

enum {
  ETPAN_MATCH_TYPE_CONTAINS,
  ETPAN_MATCH_TYPE_DOES_NOT_CONTAIN,
  ETPAN_MATCH_TYPE_BEGIN,
  ETPAN_MATCH_TYPE_END,
  ETPAN_MATCH_TYPE_EQUAL,
  ETPAN_MATCH_TYPE_NOT_EQUAL,
};

static struct etpan_filter_condition_description * get_description(char * name);

struct etpan_filter_condition * etpan_filter_condition_new(char * name,
    carray * param_list)
{
  struct etpan_filter_condition * condition;
  struct etpan_filter_condition_description * description;
  
  description = get_description(name);
  if (description == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  condition = malloc(sizeof(* condition));
  if (condition == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  condition->description = description;
  condition->param_list = param_list;
  condition->children = NULL;
  
  return condition;
}

void etpan_filter_condition_free(struct etpan_filter_condition * condition)
{
  unsigned int i;
  
  if (condition->children != NULL) {
    for(i = 0 ; i < carray_count(condition->children) ; i ++) {
      struct etpan_filter_condition * child;
      
      child = carray_get(condition->children, i);
      etpan_filter_condition_free(child);
    }
    carray_free(condition->children);
  }
  for(i = 0 ; i < carray_count(condition->param_list) ; i ++) {
    struct etpan_serialize_data * param;
    
    param = carray_get(condition->param_list, i);
    etpan_serialize_data_free(param);
  }
  carray_free(condition->param_list);
  free(condition);
}

struct etpan_filter_condition * etpan_filter_condition_dup(struct etpan_filter_condition * condition)
{
  unsigned int i;
  struct etpan_filter_condition * condition_dup;
  carray * param_list;
  int r;
  
  param_list = carray_new(4);
  if (param_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  for(i = 0 ; i < carray_count(condition->param_list) ; i ++) {
    struct etpan_serialize_data * param;
    struct etpan_serialize_data * param_dup;
    
    param = carray_get(condition->param_list, i);
    param_dup = etpan_serialize_data_dup(param);
    
    r = carray_add(param_list, param_dup, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  condition_dup = etpan_filter_condition_new(condition->description->name, param_list);
  
  if (condition->children != NULL) {
    for(i = 0 ; i < carray_count(condition->children) ; i ++) {
      struct etpan_filter_condition * child;
      struct etpan_filter_condition * child_dup;
      
      child = carray_get(condition->children, i);
      child_dup = etpan_filter_condition_dup(child);
      etpan_filter_condition_add(condition_dup, child_dup);
    }
  }
  
  return condition_dup;
}

void etpan_filter_condition_add(struct etpan_filter_condition * parent,
    struct etpan_filter_condition * child)
{
  int r;
  
  if (parent->children == NULL) {
    parent->children = carray_new(4);
    if (parent->children == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  r = carray_add(parent->children, child, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

void etpan_filter_condition_evaluate(struct etpan_filter_condition * condition,
    struct etpan_message * msg,
    void (* callback)(int result, void * data),
    void * data)
{
  condition->description->evaluate(condition, msg, callback, data);
}

static chash * condition_description_hash = NULL;

static struct etpan_filter_condition_description * get_description(char * name)
{
  chashdatum key;
  chashdatum value;
  int r;
  
  key.data = name;
  key.len = strlen(name);
  r = chash_get(condition_description_hash, &key, &value);
  if (r < 0)
    ETPAN_CRASH("condition description name not valid %s", name);
  
  return value.data;
}

static void unregister_all(void)
{
  chashiter * iter;
  carray * table;
  unsigned int i;
  int r;
  
  table = carray_new(16);
  if (table == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  for(iter = chash_begin(condition_description_hash) ; iter != NULL ;
      iter = chash_next(condition_description_hash, iter)) {
    chashdatum value;
    struct etpan_filter_condition_description * description;
    
    chash_value(iter, &value);
    description = value.data;
    r = carray_add(table, description->name, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  for(i = 0 ; i < carray_count(table) ; i ++) {
    char * name;
    
    name = carray_get(table, i);
    etpan_filter_condition_unregister_description(name);
  }
  
  carray_free(table);
}

static void register_default_condition(void);

void etpan_filter_condition_register_default(void)
{
  register_default_condition();
}

void etpan_filter_condition_unregister_all(void)
{
  unregister_all();
}

void etpan_filter_condition_register_description(struct etpan_filter_condition_description * description)
{
  chashdatum key;
  chashdatum value;
  int r;
  
  if (condition_description_hash == NULL) {
    condition_description_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
    if (condition_description_hash == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  key.data = description->name;
  key.len = strlen(description->name);
  value.data = description;
  value.len = 0;
  r = chash_set(condition_description_hash, &key, &value, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

void etpan_filter_condition_unregister_description(char * name)
{
  chashdatum key;
  
  key.data = name;
  key.len = strlen(name);
  chash_delete(condition_description_hash, &key, NULL);
}


/* ************************************ */
/* implementations */

static void condition_from_evaluate(struct etpan_filter_condition * condition,
    struct etpan_message * msg,
    void (* callback)(int result, void * cb_data),
    void * data);

static void condition_recipient_evaluate(struct etpan_filter_condition * condition,
    struct etpan_message * msg,
    void (* callback)(int result, void * cb_data),
    void * data);

static void condition_content_evaluate(struct etpan_filter_condition * condition,
    struct etpan_message * msg,
    void (* callback)(int result, void * cb_data),
    void * data);

static void condition_and_evaluate(struct etpan_filter_condition * condition,
    struct etpan_message * msg,
    void (* callback)(int result, void * cb_data),
    void * data);

static void condition_or_evaluate(struct etpan_filter_condition * condition,
    struct etpan_message * msg,
    void (* callback)(int result, void * cb_data),
    void * data);

static void condition_header_evaluate(struct etpan_filter_condition * condition,
    struct etpan_message * msg,
    void (* callback)(int result, void * cb_data),
    void * data);

static void condition_mailing_list_evaluate(struct etpan_filter_condition * condition,
    struct etpan_message * msg,
    void (* callback)(int result, void * cb_data),
    void * data);

static void condition_true_evaluate(struct etpan_filter_condition * condition,
    struct etpan_message * msg,
    void (* callback)(int result, void * cb_data),
    void * data);

static void condition_false_evaluate(struct etpan_filter_condition * condition,
    struct etpan_message * msg,
    void (* callback)(int result, void * cb_data),
    void * data);

static void condition_subject_evaluate(struct etpan_filter_condition * condition,
    struct etpan_message * msg,
    void (* callback)(int result, void * cb_data),
    void * data);

static void condition_not_evaluate(struct etpan_filter_condition * condition,
    struct etpan_message * msg,
    void (* callback)(int result, void * cb_data),
    void * data);

static void condition_seen_evaluate(struct etpan_filter_condition * condition,
    struct etpan_message * msg,
    void (* callback)(int result, void * cb_data),
    void * data);

static void condition_flagged_evaluate(struct etpan_filter_condition * condition,
    struct etpan_message * msg,
    void (* callback)(int result, void * cb_data),
    void * data);

static void condition_attachment_name_evaluate(struct etpan_filter_condition * condition,
    struct etpan_message * msg,
    void (* callback)(int result, void * cb_data),
    void * data);

static void condition_has_attachment_evaluate(struct etpan_filter_condition * condition,
    struct etpan_message * msg,
    void (* callback)(int result, void * cb_data),
    void * data);

static void condition_account_evaluate(struct etpan_filter_condition * condition,
    struct etpan_message * msg,
    void (* callback)(int result, void * cb_data),
    void * data);

static void condition_age_range_evaluate(struct etpan_filter_condition * condition,
    struct etpan_message * msg,
    void (* callback)(int result, void * cb_data),
    void * data);

static void condition_date_range_evaluate(struct etpan_filter_condition * condition,
    struct etpan_message * msg,
    void (* callback)(int result, void * cb_data),
    void * data);

static void condition_abook_evaluate(struct etpan_filter_condition * condition,
    struct etpan_message * msg,
    void (* callback)(int result, void * cb_data),
    void * data);


static void condition_execute_evaluate(struct etpan_filter_condition * condition,
    struct etpan_message * msg,
    void (* callback)(int result, void * cb_data),
    void * data);

static struct etpan_filter_condition_description
condition_description_list[] = {
  /*
  {"signed", evaluate_signed},
  {"encrypted", evaluate_encrypted},
  {"junk", evaluate_junk},
  */
  {"from", condition_from_evaluate}, /* const */
  {"content", condition_content_evaluate}, /* const */
  {"and", condition_and_evaluate},
  {"recipient", condition_recipient_evaluate}, /* const */
  {"or", condition_or_evaluate},
  {"header", condition_header_evaluate}, /* const */
  {"mailing_list", condition_mailing_list_evaluate}, /* const */
  {"true", condition_true_evaluate}, /* const */
  {"false", condition_false_evaluate}, /* const */
  {"subject", condition_subject_evaluate}, /* const */
  {"not", condition_not_evaluate},
  {"seen", condition_seen_evaluate},
  {"flagged", condition_flagged_evaluate},
  {"attachment_name", condition_attachment_name_evaluate}, /* const */
  {"has_attachment", condition_has_attachment_evaluate}, /* const */
  {"account", condition_account_evaluate}, /* const */
  {"age_range", condition_age_range_evaluate},
  {"date_range", condition_date_range_evaluate}, /* const */
  {"abook", condition_abook_evaluate},
  {"execute", condition_execute_evaluate},
};

static void register_default_condition(void)
{
  unsigned int i;
  
  for(i = 0 ; i < sizeof(condition_description_list) / sizeof(condition_description_list[0]) ; i ++) {
    etpan_filter_condition_register_description(&condition_description_list[i]);
  }
}

static int strcontains(char * big, char * little, int casematch)
{
  int result;
  
  big = strdup(big);
  if (big == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  little = strdup(little);
  if (little == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  if (!casematch) {
    char * p;
    
    for(p = big ; * p != 0 ; p ++)
      * p = (char) toupper((unsigned char) * p);
    for(p = little ; * p != 0 ; p ++)
      * p = (char) toupper((unsigned char) * p);
  }
  
  result = (strstr(big, little) != NULL);
  
  free(little);
  free(big);
  
  return result;
}

static int str_match(char * content, int match_type, char * str)
{
  if (str == NULL)
    return 0;
  if (content == NULL)
    return 0;
  
  switch (match_type) {
  case ETPAN_MATCH_TYPE_CONTAINS:
    {
      if (strcontains(content, str, 0))
        return 1;
      return 0;
    }
    break;
  case ETPAN_MATCH_TYPE_DOES_NOT_CONTAIN:
    {
      if (!strcontains(content, str, 0))
        return 1;
      return 0;
    }
    break;
  case ETPAN_MATCH_TYPE_BEGIN:
    if (strncasecmp(content, str, strlen(str)) == 0)
      return 1;
    return 0;
    break;
  case ETPAN_MATCH_TYPE_END:
    {
      size_t len_content;
      size_t len_str;
      
      len_content = strlen(content);
      len_str = strlen(str);
      if (strncasecmp(content + len_content - len_str, str, len_str) == 0)
        return 1;
    }
    break;
  case ETPAN_MATCH_TYPE_EQUAL:
    if (strcasecmp(content, str) == 0)
      return 1;
    return 0;
    break;
  case ETPAN_MATCH_TYPE_NOT_EQUAL:
    if (strcasecmp(content, str) != 0)
      return 1;
    return 0;
    break;
  }
  
  ETPAN_CRASH("should not reach");
  return 0;
}

static int address_match(struct etpan_address * addr, int match_type,
                         char * str)
{
  if (str_match(etpan_address_get_display_name(addr), match_type, str))
    return 1;
  if (str_match(etpan_address_get_address(addr), match_type, str))
    return 1;
  
  return 0;
}

static int address_list_match(carray * addr_list, int match_type,
    char * str)
{
  unsigned int i;
  
  if (addr_list == NULL)
    return 0;
  
  for(i = 0 ; i < carray_count(addr_list) ; i ++) {
    struct etpan_address * addr;
    
    addr = carray_get(addr_list, i);
    if (address_match(addr, match_type, str))
      return 1;
  }
  
  return 0;
}

static int get_match_type_from_str(char * str)
{
  if (strcasecmp(str, "contains") == 0)
    return ETPAN_MATCH_TYPE_CONTAINS;
  else if (strcasecmp(str, "doesnotcontain") == 0)
    return ETPAN_MATCH_TYPE_DOES_NOT_CONTAIN;
  else if (strcasecmp(str, "begin") == 0)
    return ETPAN_MATCH_TYPE_BEGIN;
  else if (strcasecmp(str, "end") == 0)
    return ETPAN_MATCH_TYPE_END;
  else if (strcasecmp(str, "equals") == 0)
    return ETPAN_MATCH_TYPE_EQUAL;
  else if (strcasecmp(str, "isnotequalto") == 0)
    return ETPAN_MATCH_TYPE_NOT_EQUAL;
  else
    return ETPAN_MATCH_TYPE_CONTAINS;
}

static char * get_str_from_param(carray * param_list, unsigned int i)
{
  char * str;
  struct etpan_serialize_data * param;
  
  param = carray_get(param_list, i);
  str = etpan_serialize_data_get_str(param);
  
  return str;
}

static int64_t get_int_from_param(carray * param_list, unsigned int i)
{
  int64_t value;
  struct etpan_serialize_data * param;
  
  param = carray_get(param_list, i);
  value = etpan_serialize_data_get_int64(param);
  
  return value;
}

static int get_match_type_from_param(carray * param_list, unsigned int i)
{
  char * str;
  
  str = get_str_from_param(param_list, i);
  return get_match_type_from_str(str);
}

/* from */

static void condition_from_evaluate(struct etpan_filter_condition * condition,
    struct etpan_message * msg,
    void (* callback)(int result, void * cb_data),
    void * data)
{
  struct etpan_message_header * header;
  carray * addr_list;
  int result;
  int match_type;
  char * str;
  
  ETPAN_ASSERT(carray_count(condition->param_list) == 2, "should have 2 parameters");
  
  match_type = get_match_type_from_param(condition->param_list, 0);
  
  header = etpan_message_get_header(msg);
  addr_list = etpan_message_header_get_from(header);
  
  str = get_str_from_param(condition->param_list, 1);
  
  result = address_list_match(addr_list, match_type, str);
  
  callback(result, data);
}

/* recipient */

static void condition_recipient_evaluate(struct etpan_filter_condition * condition,
    struct etpan_message * msg,
    void (* callback)(int result, void * cb_data),
    void * data)
{
  struct etpan_message_header * header;
  carray * addr_list;
  int result;
  int match_type;
  char * str;
  int b;
  
  ETPAN_ASSERT(carray_count(condition->param_list) == 2, "should have 2 parameters");
  
  match_type = get_match_type_from_param(condition->param_list, 0);
  
  header = etpan_message_get_header(msg);
  addr_list = etpan_message_header_get_to(header);
  
  str = get_str_from_param(condition->param_list, 1);
  
  result = address_list_match(addr_list, match_type, str);
  
  if (result) {
    callback(result, data);
    return;
  }
  
  str = get_str_from_param(condition->param_list, 1);
  
  result = 0;
  addr_list = etpan_message_header_get_to(header);
  b = address_list_match(addr_list, match_type, str);
  result = result || b;
  addr_list = etpan_message_header_get_cc(header);
  b = address_list_match(addr_list, match_type, str);
  result = result || b;
  
  if (result) {
    callback(result, data);
    return;
  }
  
  callback(0, data);
}

/* content */

struct content_fetched_data {
  void * data;
  carray * param_list;
  void (* callback)(int result, void * cb_data);
};

static void condition_content_fetched(char * signal_name, void * sender,
    void * signal_data, void * user_data);

static void condition_content_evaluate(struct etpan_filter_condition * condition,
    struct etpan_message * msg,
    void (* callback)(int result, void * cb_data),
    void * data)
{
  struct etpan_message_fetcher * fetcher;
  struct content_fetched_data * content_fetched_data;
  
  ETPAN_ASSERT(carray_count(condition->param_list) == 2, "should have 2 parameters");
  
  fetcher = etpan_message_fetcher_new();
  etpan_message_fetcher_set_flags(fetcher,
      ETPAN_MESSAGE_FETCHER_FLAGS_DECODE_ATTACHMENT_CHARSET);
  etpan_message_fetcher_set_message(fetcher, msg);
  etpan_message_fetcher_setup(fetcher);
  
  content_fetched_data = malloc(sizeof(* content_fetched_data));
  if (content_fetched_data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  content_fetched_data->data = data;
  content_fetched_data->param_list = condition->param_list;
  content_fetched_data->callback = callback;
  
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_MESSAGE_FETCHER_FINISHED_SIGNAL, fetcher, content_fetched_data,
      condition_content_fetched);
  
  etpan_message_fetcher_run(fetcher);
}

static void condition_content_fetched(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_message_fetcher * fetcher;
  void * data;
  carray * param_list;
  struct content_fetched_data * content_fetched_data;
  carray * list;
  void (* callback)(int result, void * cb_data);
  struct etpan_error * error;
  unsigned int i;
  int match_type;
  char * str;
  (void) signal_name;
  (void) signal_data;
  
  fetcher = sender;
  content_fetched_data = user_data;
  data = content_fetched_data->data;
  param_list = content_fetched_data->param_list;
  callback = content_fetched_data->callback;
  free(content_fetched_data);
  
  match_type = get_match_type_from_param(param_list, 0);
  str = get_str_from_param(param_list, 1);
  
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_MESSAGE_FETCHER_FINISHED_SIGNAL, fetcher, content_fetched_data,
      condition_content_fetched);
  
  error = etpan_message_fetcher_get_error(fetcher);
  if (error != NULL) {
    /* log error */
    etpan_error_log(error);
    etpan_message_fetcher_unref(fetcher);
    callback(-1, data);
    return;
  }
  
  list = etpan_message_fetcher_get_part_list(fetcher);
  for(i = 0 ; i < carray_count(list) ; i ++) {
    struct etpan_part_fetch_info * part_info;
    int type;
    struct etpan_part * part;
    int part_type;
    
    part_info = carray_get(list, i);
    type = etpan_part_fetch_info_get_type(part_info);
    part = etpan_part_fetch_info_get_part(part_info);
    part_type = etpan_part_get_type(part);
    
    if ((type == ETPAN_PART_FETCH_INFO_TYPE_TEXT) &&
        (part_type == ETPAN_PART_TYPE_SINGLE)) {
      char * content;
      size_t content_length;
      int result;
      
      etpan_part_fetch_info_get_content(part_info,
          &content, &content_length);
      
      result = str_match(content, match_type, str);
      if (result) {
        etpan_message_fetcher_unref(fetcher);
        callback(1, data);
        return;
      }
    }
  }
  
  etpan_message_fetcher_unref(fetcher);
  callback(0, data);
}

/* and */

struct and_data {
  carray * list;
  unsigned int current_index;
  struct etpan_message * msg;
  void (* callback)(int result, void * cb_data);
  void * cb_data;
  int result;
};

static void condition_and_run_next(struct and_data * data);

static void condition_and_evaluate(struct etpan_filter_condition * condition,
    struct etpan_message * msg,
    void (* callback)(int result, void * cb_data),
    void * cb_data)
{
  struct and_data * data;
  
  ETPAN_ASSERT(carray_count(condition->param_list) == 0, "should have 0 parameters");
  
  data = malloc(sizeof(* data));
  if (data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  data->current_index = 0;
  data->list = condition->children;
  data->msg = msg;
  data->callback = callback;
  data->cb_data = cb_data;
  data->result = 1;
  
  condition_and_run_next(data);
}

static void condition_and_run_done(struct and_data * data);
static void condition_and_run_next_callback(int result, void * cb_data);

static void condition_and_run_next(struct and_data * data)
{
  struct etpan_filter_condition * condition;
  
  if (data->current_index >= carray_count(data->list)) {
    condition_and_run_done(data);
    return;
  }
  
  condition = carray_get(data->list, data->current_index);
  
  etpan_filter_condition_evaluate(condition, data->msg,
      condition_and_run_next_callback, data);
}

static void condition_and_run_next_callback(int result, void * cb_data)
{
  struct and_data * data;
  
  data = cb_data;
  
  if (!result) {
    data->result = 0;
    condition_and_run_done(data);
    return;
  }
  
  data->current_index ++;
  
  condition_and_run_next(data);
}

static void condition_and_run_done(struct and_data * data)
{
  data->callback(data->result, data->cb_data);
  free(data);
}



/* or */

struct or_data {
  carray * list;
  unsigned int current_index;
  struct etpan_message * msg;
  void (* callback)(int result, void * cb_data);
  void * cb_data;
  int result;
};

static void condition_or_run_next(struct or_data * data);

static void condition_or_evaluate(struct etpan_filter_condition * condition,
    struct etpan_message * msg,
    void (* callback)(int result, void * cb_data),
    void * cb_data)
{
  struct or_data * data;
  
  ETPAN_ASSERT(carray_count(condition->param_list) == 0, "should have 0 parameters");
  
  data = malloc(sizeof(* data));
  if (data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  data->current_index = 0;
  data->list = condition->children;
  data->msg = msg;
  data->callback = callback;
  data->cb_data = cb_data;
  data->result = 0;
  
  condition_or_run_next(data);
}

static void condition_or_run_done(struct or_data * data);
static void condition_or_run_next_callback(int result, void * cb_data);

static void condition_or_run_next(struct or_data * data)
{
  struct etpan_filter_condition * condition;
  
  if (data->current_index >= carray_count(data->list)) {
    condition_or_run_done(data);
    return;
  }
  
  condition = carray_get(data->list, data->current_index);
  
  etpan_filter_condition_evaluate(condition, data->msg,
      condition_or_run_next_callback, data);
}

static void condition_or_run_next_callback(int result, void * cb_data)
{
  struct or_data * data;
  
  data = cb_data;
  
  if (result) {
    data->result = 1;
    condition_or_run_done(data);
    return;
  }
  
  data->current_index ++;
  
  condition_or_run_next(data);
}

static void condition_or_run_done(struct or_data * data)
{
  data->callback(data->result, data->cb_data);
  free(data);
}

/* header */

struct header_fetched_data {
  void * data;
  carray * param_list;
  void (* callback)(int result, void * cb_data);
};

static void condition_header_fetched(char * signal_name, void * sender,
    void * signal_data, void * user_data);

static void condition_header_evaluate(struct etpan_filter_condition * condition,
    struct etpan_message * msg,
    void (* callback)(int result, void * cb_data),
    void * data)
{
  struct etpan_message_fetcher * fetcher;
  struct header_fetched_data * header_fetched_data;
  
  ETPAN_ASSERT(carray_count(condition->param_list) == 3, "should have 3 parameters");
  
  fetcher = etpan_message_fetcher_new();
  etpan_message_fetcher_set_flags(fetcher,
      ETPAN_MESSAGE_FETCHER_FLAGS_DECODE_ATTACHMENT_CHARSET);
  etpan_message_fetcher_set_message(fetcher, msg);
  etpan_message_fetcher_setup(fetcher);
  
  header_fetched_data = malloc(sizeof(* header_fetched_data));
  if (header_fetched_data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  header_fetched_data->data = data;
  header_fetched_data->param_list = condition->param_list;
  header_fetched_data->callback = callback;
  
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_MESSAGE_FETCHER_FINISHED_SIGNAL, fetcher, header_fetched_data,
      condition_header_fetched);
  
  etpan_message_fetcher_run(fetcher);
}

static void condition_header_fetched(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_message_fetcher * fetcher;
  void * data;
  carray * param_list;
  struct header_fetched_data * header_fetched_data;
  carray * list;
  void (* callback)(int result, void * cb_data);
  struct etpan_error * error;
  unsigned int i;
  int match_type;
  char * header_name;
  char * str;
  (void) signal_name;
  (void) signal_data;
  
  fetcher = sender;
  header_fetched_data = user_data;
  data = header_fetched_data->data;
  param_list = header_fetched_data->param_list;
  callback = header_fetched_data->callback;
  free(header_fetched_data);
  
  match_type = get_match_type_from_param(param_list, 0);
  header_name = get_str_from_param(param_list, 1);
  str = get_str_from_param(param_list, 2);
  
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_MESSAGE_FETCHER_FINISHED_SIGNAL, fetcher, header_fetched_data,
      condition_header_fetched);
  
  error = etpan_message_fetcher_get_error(fetcher);
  if (error != NULL) {
    /* log error */
    etpan_error_log(error);
    etpan_message_fetcher_unref(fetcher);
    callback(-1, data);
    return;
  }
  
  list = etpan_message_fetcher_get_part_list(fetcher);
  if (carray_count(list) > 0) {
    struct etpan_part_fetch_info * part_info;
    int type;
    struct etpan_part * part;
    int part_type;
    
    part_info = carray_get(list, 0);
    type = etpan_part_fetch_info_get_type(part_info);
    part = etpan_part_fetch_info_get_part(part_info);
    part_type = etpan_part_get_type(part);
    
    if ((type == ETPAN_PART_FETCH_INFO_TYPE_TEXT) &&
        (part_type == ETPAN_PART_TYPE_MESSAGE)) {
      char * content;
      size_t content_length;
      int result;
      char * header_content;
      carray * msg_header;
      
      etpan_part_fetch_info_get_content(part_info,
          &content, &content_length);
      
      msg_header = etpan_unparsed_header_parse(content, content_length);
      header_content = NULL;
      for(i = 0 ; i < carray_count(msg_header) ; i ++) {
        struct etpan_unparsed_header_item * header_item;
        char * name;
        
        header_item = carray_get(msg_header, i);
        name = etpan_unparsed_header_item_get_name(header_item);
        if (strcasecmp(name, header_name) == 0) {
          header_content = etpan_unparsed_header_item_get_value(header_item);
        }
      }
      if (header_content != NULL) {
        result = str_match(header_content, match_type, str);
      }
      else {
        result = 0;
      }
      etpan_unparsed_header_free(msg_header);
      
      if (result) {
        etpan_message_fetcher_unref(fetcher);
        callback(1, data);
        return;
      }
    }
  }
  
  etpan_message_fetcher_unref(fetcher);
  callback(0, data);
}

/* mailing-list - List-Post */

static void condition_mailing_list_evaluate(struct etpan_filter_condition * condition,
    struct etpan_message * msg,
    void (* callback)(int result, void * cb_data),
    void * data)
{
  struct etpan_filter_condition * condition_header;
  struct etpan_serialize_data * type;
  struct etpan_serialize_data * header_name;
  struct etpan_serialize_data * str;
  carray * param_list;
  int r;
  
  ETPAN_ASSERT(carray_count(condition->param_list) == 2, "should have 2 parameters");
  
  param_list = carray_new(3);
  if (param_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  type = etpan_serialize_data_dup(carray_get(param_list, 0));
  str = etpan_serialize_data_dup(carray_get(param_list, 1));
  r = carray_add(param_list, type, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  header_name = etpan_serialize_data_new_str("List-Post");
  r = carray_add(param_list, header_name, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  r = carray_add(param_list, str, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  condition_header = etpan_filter_condition_new("header", param_list);
  
  condition_header_evaluate(condition_header,
      msg, callback, data);
  
  etpan_filter_condition_free(condition_header);
}

/* true */

static void condition_true_evaluate(struct etpan_filter_condition * condition,
    struct etpan_message * msg,
    void (* callback)(int result, void * cb_data),
    void * data)
{
  ETPAN_ASSERT(carray_count(condition->param_list) == 0, "should have 0 parameters");
  callback(1, data);
}

/* false */

static void condition_false_evaluate(struct etpan_filter_condition * condition,
    struct etpan_message * msg,
    void (* callback)(int result, void * cb_data),
    void * data)
{
  ETPAN_ASSERT(carray_count(condition->param_list) == 0, "should have 0 parameters");
  callback(0, data);
}

/* subject */

static void condition_subject_evaluate(struct etpan_filter_condition * condition,
    struct etpan_message * msg,
    void (* callback)(int result, void * cb_data),
    void * data)
{
  struct etpan_message_header * header;
  char * subject;
  int result;
  int match_type;
  char * str;
  
  ETPAN_ASSERT(carray_count(condition->param_list) == 2, "should have 2 parameters");
  
  match_type = get_match_type_from_param(condition->param_list, 0);
  
  header = etpan_message_get_header(msg);
  subject = etpan_message_header_get_subject(header);
  
  str = get_str_from_param(condition->param_list, 1);
  
  if (subject != NULL) {
    result = str_match(subject, match_type, str);
  }
  else {
    result = 0;
  }
  
  callback(result, data);
}

/* not */

struct not_data {
  void * data;
  void (* callback)(int result, void * cb_data);
};

static void condition_not_callback(int result, void * cb_data);

static void condition_not_evaluate(struct etpan_filter_condition * condition,
    struct etpan_message * msg,
    void (* callback)(int result, void * cb_data),
    void * data)
{
  struct etpan_filter_condition * condition_item;
  struct not_data * not_data;
  
  ETPAN_ASSERT(carray_count(condition->param_list) == 0, "should have 0 parameters");
  
  condition_item = carray_get(condition->children, 0);
  
  not_data = malloc(sizeof(* not_data));
  if (not_data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  not_data->data = data;
  not_data->callback = callback;
  
  etpan_filter_condition_evaluate(condition_item, msg,
      condition_not_callback, not_data);
}

static void condition_not_callback(int result, void * cb_data)
{
  struct not_data * not_data;
  
  not_data = cb_data;
  not_data->callback(!result, not_data->data);
  free(not_data);
}

/* seen */

static void condition_seen_evaluate(struct etpan_filter_condition * condition,
    struct etpan_message * msg,
    void (* callback)(int result, void * cb_data),
    void * data)
{
  struct etpan_message_flags * flags;
  int result;
  int value;
  
  ETPAN_ASSERT(carray_count(condition->param_list) == 0, "should have 0 parameters");
  
  flags = etpan_message_get_flags(msg);
  value = etpan_message_flags_get_value(flags);
  if ((value & ETPAN_FLAGS_SEEN) != 0)
    result = 1;
  else
    result = 0;
  
  callback(result, data);
}

/* flagged */

static void condition_flagged_evaluate(struct etpan_filter_condition * condition,
    struct etpan_message * msg,
    void (* callback)(int result, void * cb_data),
    void * data)
{
  struct etpan_message_flags * flags;
  int result;
  int value;
  
  ETPAN_ASSERT(carray_count(condition->param_list) == 0, "should have 0 parameters");
  
  flags = etpan_message_get_flags(msg);
  value = etpan_message_flags_get_value(flags);
  if ((value & ETPAN_FLAGS_FLAGGED) != 0)
    result = 1;
  else
    result = 0;
  
  callback(result, data);
}

/* attachment_name */

struct attachment_name_data {
  void * data;
  carray * param_list;
  void (* callback)(int result, void * cb_data);
};

static void condition_attachment_name_fetched(char * signal_name, void * sender,
    void * signal_data, void * user_data);

static void condition_attachment_name_evaluate(struct etpan_filter_condition * condition,
    struct etpan_message * msg,
    void (* callback)(int result, void * cb_data),
    void * data)
{
  struct etpan_message_fetcher * fetcher;
  struct attachment_name_data * attachment_name_data;
  
  ETPAN_ASSERT(carray_count(condition->param_list) == 2, "should have 2 parameters");
  
  fetcher = etpan_message_fetcher_new();
  etpan_message_fetcher_set_flags(fetcher,
      ETPAN_MESSAGE_FETCHER_FLAGS_DECODE_ATTACHMENT_CHARSET);
  etpan_message_fetcher_set_message(fetcher, msg);
  etpan_message_fetcher_setup(fetcher);
  
  attachment_name_data = malloc(sizeof(* attachment_name_data));
  if (attachment_name_data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  attachment_name_data->data = data;
  attachment_name_data->param_list = condition->param_list;
  attachment_name_data->callback = callback;
  
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_MESSAGE_FETCHER_FINISHED_SIGNAL, fetcher, attachment_name_data,
      condition_attachment_name_fetched);
  
  etpan_message_fetcher_run(fetcher);
}

static void condition_attachment_name_fetched(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_message_fetcher * fetcher;
  void * data;
  carray * param_list;
  struct attachment_name_data * attachment_name_data;
  carray * list;
  void (* callback)(int result, void * cb_data);
  struct etpan_error * error;
  unsigned int i;
  int match_type;
  char * str;
  (void) signal_name;
  (void) signal_data;
  
  fetcher = sender;
  attachment_name_data = user_data;
  data = attachment_name_data->data;
  param_list = attachment_name_data->param_list;
  callback = attachment_name_data->callback;
  free(attachment_name_data);
  
  match_type = get_match_type_from_param(param_list, 0);
  str = get_str_from_param(param_list, 1);
  
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_MESSAGE_FETCHER_FINISHED_SIGNAL, fetcher, attachment_name_data,
      condition_attachment_name_fetched);
  
  error = etpan_message_fetcher_get_error(fetcher);
  if (error != NULL) {
    /* log error */
    etpan_error_log(error);
    etpan_message_fetcher_unref(fetcher);
    callback(-1, data);
    return;
  }
  
  list = etpan_message_fetcher_get_part_list(fetcher);
  for(i = 0 ; i < carray_count(list) ; i ++) {
    struct etpan_part_fetch_info * part_info;
    int type;
    struct etpan_part * part;
    int part_type;
    struct etpan_part_header * part_header;
    
    part_info = carray_get(list, i);
    type = etpan_part_fetch_info_get_type(part_info);
    part = etpan_part_fetch_info_get_part(part_info);
    part_type = etpan_part_get_type(part);
    part_header = etpan_part_get_header(part);
    
    if (part_type == ETPAN_PART_TYPE_SINGLE) {
      char * content;
      int result;
      
      content = etpan_part_header_get_filename(part_header);
      
      result = str_match(content, match_type, str);
      if (result) {
        etpan_message_fetcher_unref(fetcher);
        callback(1, data);
        return;
      }
    }
  }
  
  etpan_message_fetcher_unref(fetcher);
  callback(0, data);
}

/* has_attachment */

struct has_attachment_data {
  void * data;
  carray * param_list;
  void (* callback)(int result, void * cb_data);
};

static void condition_has_attachment_fetched(char * signal_name, void * sender,
    void * signal_data, void * user_data);

static void condition_has_attachment_evaluate(struct etpan_filter_condition * condition,
    struct etpan_message * msg,
    void (* callback)(int result, void * cb_data),
    void * data)
{
  struct etpan_message_fetcher * fetcher;
  struct has_attachment_data * has_attachment_data;
  
  ETPAN_ASSERT(carray_count(condition->param_list) == 0, "should have 0 parameters");
  
  fetcher = etpan_message_fetcher_new();
  etpan_message_fetcher_set_flags(fetcher,
      ETPAN_MESSAGE_FETCHER_FLAGS_DECODE_ATTACHMENT_CHARSET);
  etpan_message_fetcher_set_message(fetcher, msg);
  etpan_message_fetcher_setup(fetcher);
  
  has_attachment_data = malloc(sizeof(* has_attachment_data));
  if (has_attachment_data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  has_attachment_data->data = data;
  has_attachment_data->param_list = condition->param_list;
  has_attachment_data->callback = callback;
  
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_MESSAGE_FETCHER_FINISHED_SIGNAL, fetcher, has_attachment_data,
      condition_has_attachment_fetched);
  
  etpan_message_fetcher_run(fetcher);
}

static void condition_has_attachment_fetched(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_message_fetcher * fetcher;
  void * data;
  carray * param_list;
  struct has_attachment_data * has_attachment_data;
  carray * list;
  void (* callback)(int result, void * cb_data);
  struct etpan_error * error;
  unsigned int i;
  int match_type;
  char * str;
  (void) signal_name;
  (void) signal_data;
  
  fetcher = sender;
  has_attachment_data = user_data;
  data = has_attachment_data->data;
  param_list = has_attachment_data->param_list;
  callback = has_attachment_data->callback;
  free(has_attachment_data);
  
  match_type = get_match_type_from_param(param_list, 0);
  str = get_str_from_param(param_list, 1);
  
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_MESSAGE_FETCHER_FINISHED_SIGNAL, fetcher, has_attachment_data,
      condition_has_attachment_fetched);
  
  error = etpan_message_fetcher_get_error(fetcher);
  if (error != NULL) {
    /* log error */
    etpan_error_log(error);
    etpan_message_fetcher_unref(fetcher);
    callback(-1, data);
    return;
  }
  
  list = etpan_message_fetcher_get_part_list(fetcher);
  for(i = 0 ; i < carray_count(list) ; i ++) {
    struct etpan_part_fetch_info * part_info;
    int type;
    struct etpan_part * part;
    int part_type;
    struct etpan_part_header * part_header;
    
    part_info = carray_get(list, i);
    type = etpan_part_fetch_info_get_type(part_info);
    part = etpan_part_fetch_info_get_part(part_info);
    part_type = etpan_part_get_type(part);
    part_header = etpan_part_get_header(part);
    
    if (part_type == ETPAN_PART_TYPE_SINGLE) {
      char * content;
      
      content = etpan_part_header_get_filename(part_header);
      
      if (content != NULL) {
        etpan_message_fetcher_unref(fetcher);
        callback(1, data);
        return;
      }
    }
  }
  
  etpan_message_fetcher_unref(fetcher);
  callback(0, data);
}

/* account */

static void condition_account_evaluate(struct etpan_filter_condition * condition,
    struct etpan_message * msg,
    void (* callback)(int result, void * cb_data),
    void * data)
{
  struct etpan_folder * folder;
  struct etpan_storage * storage;
  struct etpan_account * account;
  char * str;
  int result;
  
  ETPAN_ASSERT(carray_count(condition->param_list) == 1, "should have 1 parameters");
  
  folder = etpan_message_get_folder(msg);
  storage = etpan_folder_get_storage(folder);
  account = etpan_storage_get_account(storage);
  
  str = get_str_from_param(condition->param_list, 0);
  
  result = 0;
  if (strcmp(str, etpan_account_get_id(account)) == 0)
    result = 1;
  
  callback(result, data);
}

/* age_range */

static void condition_age_range_evaluate(struct etpan_filter_condition * condition,
    struct etpan_message * msg,
    void (* callback)(int result, void * cb_data),
    void * data)
{
  struct etpan_message_header * header;
  time_t timestamp;
  time_t today;
  int64_t left;
  int64_t right;
  int64_t age;
  int result;
  
  ETPAN_ASSERT(carray_count(condition->param_list) == 2, "should have 2 parameters");
  
  left = get_int_from_param(condition->param_list, 0);
  right = get_int_from_param(condition->param_list, 1);
  
  header = etpan_message_get_header(msg);
  timestamp = etpan_message_header_get_date(header);
  today = time(NULL);
  
  age = timestamp - today;
  result = 1;
  if (left != -1) {
    if (age < left)
      result = 0;
  }
  if (right != -1) {
    if (age > right)
      result = 0;
  }
  
  callback(result, data);
}

/* date_range */

static void condition_date_range_evaluate(struct etpan_filter_condition * condition,
    struct etpan_message * msg,
    void (* callback)(int result, void * cb_data),
    void * data)
{
  struct etpan_message_header * header;
  time_t timestamp;
  time_t today;
  int64_t left;
  int64_t right;
  int64_t date;
  int result;
  
  ETPAN_ASSERT(carray_count(condition->param_list) == 2, "should have 2 parameters");
  
  left = get_int_from_param(condition->param_list, 0);
  right = get_int_from_param(condition->param_list, 1);
  
  header = etpan_message_get_header(msg);
  timestamp = etpan_message_header_get_date(header);
  today = time(NULL);
  
  date = timestamp;
  result = 1;
  if (left != -1) {
    if (date < left)
      result = 0;
  }
  if (right != -1) {
    if (date > right)
      result = 0;
  }
  
  callback(result, data);
}

/* abook */

struct match_data_abook {
  struct etpan_message * msg;
  unsigned int current_index;
  void (* callback)(int, void *);
  void * cb_data;
  struct etpan_abook_request * request;
  carray * from;
  int enable_network;
};

static void abook_request_next(struct match_data_abook * data);
static void abook_lookup_done_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);

static void condition_abook_evaluate(struct etpan_filter_condition * condition,
    struct etpan_message * msg,
    void (* callback)(int result, void * cb_data),
    void * data)
{
  struct match_data_abook * abook_data;
  struct etpan_message_header * header;
  carray * from;
  int enable_network;
  
  ETPAN_ASSERT(carray_count(condition->param_list) == 1, "should have 1 parameters");
  
  header = etpan_message_get_header(msg);
  from = etpan_message_header_get_from(header);
  
  abook_data = malloc(sizeof(* data));
  if (abook_data == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  enable_network = get_int_from_param(condition->param_list, 0);
  
  abook_data->current_index = 0;
  abook_data->msg = msg;
  abook_data->callback = callback;
  abook_data->cb_data = data;
  abook_data->request = NULL;
  abook_data->from = from;
  abook_data->enable_network = enable_network;
  
  abook_request_next(abook_data);
}

static void abook_request_next(struct match_data_abook * data)
{
  struct etpan_address * address;
  char * address_str;
  struct etpan_abook_request * request;
  
  if (data->current_index >= carray_count(data->from)) {
    data->callback(0, data->cb_data);
    free(data);
    return;
  }
  
  address = carray_get(data->from, data->current_index);
  address_str = etpan_address_get_address(address);
  
  request = etpan_abook_request_new(etpan_abook_manager_get_default());
  etpan_abook_request_set_flags(request,
      ETPAN_ABOOK_REQUEST_FLAGS_DISABLE_NETWORK);
  
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_ABOOK_REQUEST_FINISHED, request, data,
      abook_lookup_done_handler);

  etpan_abook_request_lookup(request, address_str);
  
  data->request = request;
}

static void abook_lookup_done_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct match_data_abook * data;
  carray * abook_list;
  unsigned int i;
  
  data = user_data;
  
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_ABOOK_REQUEST_FINISHED, data->request, user_data,
      abook_lookup_done_handler);
  
  abook_list = etpan_abook_request_get_result(data->request);
  for(i = 0 ; i < carray_count(abook_list) ; i ++) {
    struct etpan_abook_entry * entry;
    char * address_entry;
    struct etpan_address * address;
    char * address_str;
    
    entry = carray_get(abook_list, i);
    address_entry = etpan_abook_entry_get_address(entry);
    
    address = carray_get(data->from, data->current_index);
    address_str = etpan_address_get_address(address);
    
    if (strcasecmp(address_entry, address_str) == 0) {
      data->callback(1, data->cb_data);
      free(data);
      return;
    }
  }
  
  etpan_abook_request_free(data->request);
  
  data->current_index ++;
  
  abook_request_next(data);
}

/* execute */

struct execute_data {
  int result;
  char * command;
  struct etpan_message * msg;
  void (* callback)(int result, void * cb_data);
  void * data;
  struct etpan_message_fetcher * fetcher;
};

static void condition_execute_content_fetched(char * signal_name, void * sender,
    void * signal_data, void * user_data);

static void condition_execute_evaluate(struct etpan_filter_condition * condition,
    struct etpan_message * msg,
    void (* callback)(int result, void * cb_data),
    void * data)
{
  struct execute_data * execute_data;
  struct etpan_message_fetcher * fetcher;
  char * command;
  
  ETPAN_ASSERT(carray_count(condition->param_list) == 1, "should have 1 parameters");
  
  command = get_str_from_param(condition->param_list, 0);
  
  execute_data = malloc(sizeof(* execute_data));
  if (execute_data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  execute_data->result = 0;
  execute_data->msg = msg;
  execute_data->command = strdup(command);
  if (execute_data->command == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  execute_data->callback = callback;
  execute_data->data = data;
  execute_data->fetcher = NULL;
  
  fetcher = etpan_message_fetcher_new();
  etpan_message_fetcher_set_flags(fetcher,
      ETPAN_MESSAGE_FETCHER_FLAGS_SOURCE);
  etpan_message_fetcher_set_message(fetcher, msg);
  etpan_message_fetcher_setup(fetcher);
  execute_data->fetcher = fetcher;
  
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_MESSAGE_FETCHER_FINISHED_SIGNAL, fetcher, execute_data,
      condition_execute_content_fetched);
  
  etpan_message_fetcher_run(fetcher);
}

static void condition_execute_run(struct etpan_thread_op * op);
static void condition_execute_callback(int cancelled, void * result,
    void * data);

static void condition_execute_content_fetched(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_message_fetcher * fetcher;
  struct execute_data * execute_data;
  struct etpan_thread_op * op;
  
  fetcher = sender;
  execute_data = user_data;
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_MESSAGE_FETCHER_FINISHED_SIGNAL, fetcher, execute_data,
      condition_execute_content_fetched);
  
  if (etpan_message_fetcher_get_error(fetcher) != NULL) {
    execute_data->result = -1;
    condition_execute_callback(0, NULL, execute_data);
    return;
  }
  
  op = etpan_thread_op_new();
  op->param = execute_data;
  op->result = NULL;
  
  op->cancellable = 0;
  op->run = condition_execute_run;
  op->callback = (void (*)(int, void *, void *)) condition_execute_callback;
  op->callback_data = execute_data;
  op->cleanup = NULL;
  
  etpan_thread_manager_app_misc_schedule(etpan_thread_manager_app_get_default(),
      op);
}

static void condition_execute_run(struct etpan_thread_op * op)
{
  struct execute_data * execute_data;
  char command[PATH_MAX];
  carray * list;
  struct etpan_part_fetch_info * part_info;
  char tmp_path[PATH_MAX];
  char quoted_tmp_path[PATH_MAX];
  char quoted_command[PATH_MAX];
  char * content;
  size_t content_length;
  int r;
  
  execute_data = op->param;
  
  list = etpan_message_fetcher_get_part_list(execute_data->fetcher);
  if (carray_count(list) == 0)
    goto err;
  
  part_info = carray_get(list, 0);
  etpan_part_fetch_info_get_content(part_info, &content, &content_length);
  r = etpan_get_tmp_path(tmp_path, sizeof(tmp_path));
  if (r < 0) {
    goto err;
  }
  r = etpan_write_file(tmp_path, content, content_length);
  if (r < 0) {
    goto err;
  }
  
  r = etpan_quote_filename(quoted_tmp_path, sizeof(quoted_tmp_path), tmp_path);
  if (r < 0) {
    unlink(tmp_path);
    goto err;
  }
  r = etpan_quote_filename(quoted_command, sizeof(quoted_command),
      execute_data->command);
  if (r < 0) {
    unlink(tmp_path);
    goto err;
  }
  
  snprintf(command, sizeof(command), "%s %s",
      quoted_command, quoted_tmp_path);
  
  r = system(command);
  if (!WIFEXITED(r)) {
    unlink(tmp_path);
    goto err;
  }
  
  if (WEXITSTATUS(r) != 0) {
    execute_data->result = 0;
  }
  else {
    execute_data->result = 1;
  }
  
  unlink(tmp_path);
  etpan_message_fetcher_unref(execute_data->fetcher);
  execute_data->fetcher = NULL;
  
  return;
  
 err:
  etpan_message_fetcher_unref(execute_data->fetcher);
  execute_data->fetcher = NULL;
  execute_data->result = -1;
}

static void condition_execute_callback(int cancelled, void * result,
    void * data)
{
  struct execute_data * execute_data;
  
  execute_data = data;
  
  execute_data->callback(execute_data->result, execute_data->data);
  
  free(execute_data->command);
  free(execute_data);
}

char * etpan_filter_condition_get_description_name(struct etpan_filter_condition * condition)
{
  return condition->description->name;
}

carray * etpan_filter_condition_get_param_list(struct etpan_filter_condition * condition)
{
  return condition->param_list;
}

carray * etpan_filter_condition_get_children(struct etpan_filter_condition * condition)
{
  return condition->children;
}
