#include "etpan-filter-config.h"

#include <stdlib.h>
#include <string.h>

#include "etpan-log.h"
#include "etpan-error.h"
#include "etpan-utils.h"
#include "etpan-filter.h"
#include "etpan-filter-condition.h"
#include "etpan-filter-action.h"
#include "etpan-config-types.h"
#include "etpan-nls.h"
#include "etpan-filter-parser-glue.h"
#include "etpan-tc.h"
#include "etpan-serialize.h"

struct etpan_filter_config * etpan_filter_config_new(void)
{
  struct etpan_filter_config * config;
  
  config = malloc(sizeof(* config));
  if (config == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  config->filter_list = carray_new(16);
  if (config->filter_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return config;
}

void etpan_filter_config_free(struct etpan_filter_config * config)
{
  etpan_filter_config_reset(config);
  carray_free(config->filter_list);
  free(config);
}

struct etpan_error * etpan_filter_config_read(struct etpan_filter_config * config, char * filename)
{
  int r;
  
  if (!etpan_file_exists(filename)) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, _("Configuration file not found"));
    etpan_error_strf_long_description(error, _("The filter configuration file %s has not been found"), filename);
    
    return error;
  }
  
  r = etpan_filter_parser_parse_file(config, filename);
  if (r != 0) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, "Invalid configuration file");
    etpan_error_strf_long_description(error, _("The filter configuration file %s contains invalid data"), filename);
    
    return error;
  }
  
  return NULL;
}

struct etpan_error * etpan_filter_config_read_default(void)
{
  struct etpan_filter_config * config;
  char * home;
  char filename[PATH_MAX];
  
  config = etpan_filter_config_get_default();
  
  home = getenv("HOME");
  ETPAN_ASSERT(home != NULL, "Home directory is not defined. It should be defined in environment variable HOME.");
  
  snprintf(filename, sizeof(filename), "%s/%s", home, ETPAN_FILTER_CONFIG_FILE);
  
  return etpan_filter_config_read(config, filename);
}

static struct etpan_filter_config * default_config = NULL;

struct etpan_filter_config * etpan_filter_config_get_default(void)
{
  return default_config;
}

void etpan_filter_config_set_default(struct etpan_filter_config * config)
{
  default_config = config;
}

carray * etpan_filter_config_get_rule_list(struct etpan_filter_config * config)
{
  return config->filter_list;
}

void etpan_filter_config_add_rule(struct etpan_filter_config * config,
    struct etpan_filter * rule)
{
  int r;
  
  r = carray_add(config->filter_list, rule, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

struct etpan_filter_config * etpan_filter_config_dup(struct etpan_filter_config * config)
{
  struct etpan_filter_config * dup_config;
  carray * dup_list;
  carray * list;
  unsigned int i;
  int r;
  
  dup_config = etpan_filter_config_new();
  dup_list = etpan_filter_config_get_rule_list(dup_config);
  list = etpan_filter_config_get_rule_list(config);
  for(i = 0 ; i < carray_count(list) ; i ++) {
    struct etpan_filter * filter;
    struct etpan_filter * dup_filter;
    
    filter = carray_get(list, i);
    dup_filter = etpan_filter_dup(filter);
    r = carray_add(dup_list, dup_filter, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  return dup_config;
}

static void run_next_filter(struct etpan_filter_state * state);

void etpan_filter_config_apply_message(struct etpan_filter_state * state,
    struct etpan_filter_config * config,
    struct etpan_message * msg,
    void (* filter_callback)(void *), void * cb_data)
{
  etpan_message_ref(msg);
  state->config = config;
  state->current_index = 0;
  state->filter_callback = filter_callback;
  state->cb_data = cb_data;
  state->msg = msg;
  
  run_next_filter(state);
}

static void run_next_filter_done(struct etpan_filter_state * state);
static void run_next_filter_callback(int filtered, void * data);

static void run_next_filter(struct etpan_filter_state * state)
{
  struct etpan_filter * filter;
  carray * list;
  
  list = etpan_filter_config_get_rule_list(state->config);
  
  if (state->current_index >= carray_count(list)) {
    run_next_filter_done(state);
    return;
  }
  
  filter = carray_get(list, state->current_index);
  etpan_filter_message(state, filter, state->msg,
      run_next_filter_callback, state);
}

static void run_next_filter_callback(int filtered, void * data)
{
  struct etpan_filter_state * state;
  
  if (filtered) {
    ETPAN_LOG("this condition matched");
  }
  
  state = data;
  state->current_index ++;
  run_next_filter(state);
}

static void run_next_filter_done(struct etpan_filter_state * state)
{
  state->filter_callback(state->cb_data);
  etpan_message_unref(state->msg);
  state->msg = NULL;
}

void etpan_filter_config_reset(struct etpan_filter_config * config)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(config->filter_list) ; i ++) {
    struct etpan_filter * filter;
    
    filter = carray_get(config->filter_list, i);
    etpan_filter_free(filter);
  }
  
  carray_set_size(config->filter_list, 0);
}

struct etpan_error * etpan_filter_config_write_default(void)
{
  struct etpan_filter_config * config;
  char * home;
  char filename[PATH_MAX];
  
  config = etpan_filter_config_get_default();
  
  home = getenv("HOME");
  ETPAN_ASSERT(home != NULL, "Home directory is not defined. It should be defined in environment variable HOME.");
  
  snprintf(filename, sizeof(filename), "%s/%s", home, ETPAN_FILTER_CONFIG_FILE);
  
  return etpan_filter_config_write(config, filename);
}

static struct etpan_error * condition_write(FILE * f, struct etpan_filter_condition * cond);
static struct etpan_error * action_write(FILE * f, struct etpan_filter_action * action);

struct etpan_error * etpan_filter_config_write(struct etpan_filter_config * config, char * filename)
{
  carray * list;
  unsigned int i;
  FILE * f;
  struct etpan_error * error;
  
  f = fopen(filename, "wb");
  
  list = etpan_filter_config_get_rule_list(config);
  for(i = 0 ; i < carray_count(list) ; i ++) {
    struct etpan_filter * filter;
    struct etpan_filter_condition * cond;
    carray * action_list;
    char * name;
    char * quoted_name;
    unsigned int k;
    
    filter = carray_get(list, i);
    
    name = etpan_filter_get_name(filter);
    quoted_name = etpan_quote_string(name);
    fprintf(f, "\"%s\" :\n", quoted_name);
    free(quoted_name);
    
    cond = etpan_filter_get_condition(filter);
    error = condition_write(f, cond);
    if (error != NULL) {
      return error;
    }
    
    fprintf(f, "\n");
    fprintf(f, "{\n");
    action_list = etpan_filter_get_action_list(filter);
    for(k = 0 ; k < carray_count(action_list) ; k ++) {
      struct etpan_filter_action * action;
      
      action = carray_get(action_list, k);
      error = action_write(f, action);
      if (error != NULL) {
        return error;
      }
    }
    fprintf(f, "}\n");
    fprintf(f, "\n");
  }
  
  fclose(f);
  
  return NULL;
}

static struct etpan_error * condition_write(FILE * f, struct etpan_filter_condition * cond)
{
  char * description_name;
  carray * children;
  
  description_name = etpan_filter_condition_get_description_name(cond);
  children = etpan_filter_condition_get_children(cond);
  if (strcasecmp(description_name, "and") == 0) {
    unsigned int i;
    
    for(i = 0 ; i < carray_count(children) ; i ++) {
      struct etpan_filter_condition * child;
      
      child = carray_get(children, i);
      condition_write(f, child);
      if (i != carray_count(children) - 1) {
        fprintf(f, " & ");
      }
    }
  }
  else if (strcasecmp(description_name, "or") == 0) {
    unsigned int i;
    
    for(i = 0 ; i < carray_count(children) ; i ++) {
      struct etpan_filter_condition * child;
      
      child = carray_get(children, i);
      condition_write(f, child);
      if (i != carray_count(children) - 1) {
        fprintf(f, " | ");
      }
    }
  }
  else if (strcasecmp(description_name, "not") == 0) {
    struct etpan_filter_condition * child;
    
    child = carray_get(children, 0);
    
    fprintf(f, "!(");
    condition_write(f, cond);
    fprintf(f, ")");
  }
  else {
    unsigned int i;
    carray * param_list;
    
    fprintf(f, "%s(", description_name);
    param_list = etpan_filter_condition_get_param_list(cond);
    for(i = 0 ; i < carray_count(param_list) ; i ++) {
      int type;
      struct etpan_serialize_data * data;
      
      data = carray_get(param_list, i);
      type = etpan_serialize_data_get_type(data);
      if (type == ETPAN_SERIALIZE_INT64) {
        int64_t value;
        
        value = etpan_serialize_data_get_int64(data);
        fprintf(f, "%i", (unsigned int) value);
      }
      else if (type == ETPAN_SERIALIZE_STRING) {
        char * str;
        char * quoted_name;
        
        str = etpan_serialize_data_get_str(data);
        quoted_name = etpan_quote_string(str);
        fprintf(f, "\"%s\"", quoted_name);
        free(quoted_name);
      }
      else {
        ETPAN_CRASH("invalid parameter type");
      }
      
      if (i != carray_count(param_list) - 1) {
        fprintf(f, ", ");
      }
    }
    fprintf(f, ")");
  }
  
  return NULL;
}

static struct etpan_error * action_write(FILE * f, struct etpan_filter_action * action)
{
  unsigned int i;
  carray * param_list;
  char * description_name;
  
  description_name = etpan_filter_action_get_description_name(action);
  fprintf(f, "    %s(", description_name);
  param_list = etpan_filter_action_get_param_list(action);
  for(i = 0 ; i < carray_count(param_list) ; i ++) {
    int type;
    struct etpan_serialize_data * data;
      
    data = carray_get(param_list, i);
    type = etpan_serialize_data_get_type(data);
    if (type == ETPAN_SERIALIZE_INT64) {
      int64_t value;
        
      value = etpan_serialize_data_get_int64(data);
      fprintf(f, "%i", (unsigned int) value);
    }
    else if (type == ETPAN_SERIALIZE_STRING) {
      char * str;
      char * quoted_name;
        
      str = etpan_serialize_data_get_str(data);
      quoted_name = etpan_quote_string(str);
      fprintf(f, "\"%s\"", quoted_name);
      free(quoted_name);
    }
    else {
      ETPAN_CRASH("invalid parameter type");
    }
      
    if (i != carray_count(param_list) - 1) {
      fprintf(f, ", ");
    }
  }
  fprintf(f, ");\n");
  
  return NULL;
}

