%{

#ifndef ETPAN_MODULE_LOG_NAME
#define ETPAN_MODULE_LOG_NAME "FILTER"
#endif

#include <string.h>
#include <inttypes.h>
#include <stdio.h>
#include "etpan-filter-parser-internal.h"
#include "etpan-filter-utils.h"
#include "etpan-serialize.h"
#include "etpan-log.h"
#include "etpan-error.h"
#include "etpan-filter-condition.h"
#include "etpan-filter-action.h"
#include "etpan-filter.h"
#include "etpan-filter-config.h"

#define yylex etpan_filter_lexer_lex

 static void etpan_filter_lexer_error(char *);
 static void add_parameter(struct etpan_serialize_data * param);
 static void cleanup_parameter_list(void);
 static void cleanup_function_call(void);
 static void cleanup_condition_stack(void);
 static void create_function_call();
 static void set_function_name(char * str);
 static void set_rule_name(char * str);
 static void create_condition_from_function_call(void);
 static void create_and_condition(void);
 static void create_or_condition(void);
 static void create_not_condition(void);
 static void create_constant_condition(int value);
 static void create_condition_from_function_call(void);
 static void add_action(struct etpan_filter_action * action);
 static void cleanup_action_list(void);
 static void create_filter(void);
 static void setup_parameter_list(void);
 static void create_action_from_function_call(void);

%}

%union {
  int64_t i;
  char *s;
}

%token <i> INT
%token <s> STRING
%token <s> DSTRING
%token <s> IDENT
%token FALSE_KEYWORD
%token TRUE_KEYWORD

 /*
%type <s> STRING DSTRING IDENT
%type <i> INT
 */

%%

/*
from('equals', str),
from('contains', str)
recipient('equals', str)
recipient('contains', str)
subject('contains', str)
header('name', str)
mailing_list('equals', str)
mailing_list('contains', str)
account(str)
*/

/*
copy(folder_name)
move(folder_name)
set_color(color)
execute(file_with_message_content)
delete()
add_flag('seen')
add_flag('flagged')
forward(message_content)
reply(message_content)
stop()
*/

 /*
   from('equals', 'dinh.viet.hoa@free.fr'),
 */

filter_list:
filter filter_list
| filter;

filter:
rule_name ':' condition '{' optional_action_list '}' {
  create_filter();
};

rule_name:
STRING {
  ETPAN_LOCAL_LOG("string %s", $1);
  set_rule_name($1);
}
| DSTRING {
  char * str;
  
  ETPAN_LOCAL_LOG("string %s", $1);
  str = etpan_filter_unquote_string($1);
  set_rule_name(str);
  free(str);
};

condition:
expression;

expression:
expression_or;

expression_or:
expression_and
| expression '|' expression {
  create_or_condition();
};

expression_and:
expression_unary
| expression '&' expression {
  create_and_condition();
};

expression_unary:
'(' expression ')' {
  /* do nothing, keep expression on stack */
}
| function_call {
  /* push expression on stack */
  create_condition_from_function_call();
  }
| '!' expression {
  create_not_condition();
  }
| TRUE_KEYWORD {
  create_constant_condition(1);
  }
| FALSE_KEYWORD {
  create_constant_condition(0);
  };

function_call:
function_name '('
{
  setup_parameter_list();
}
optional_parameter_list ')'
{
  create_function_call();
};

function_name:
IDENT {
  set_function_name($1);
};

optional_parameter_list:
parameter_list
|;

parameter_list:
parameter ',' parameter_list
| parameter;

parameter:
STRING {
  char * str;
  struct etpan_serialize_data * param;
  
  str = $1;
  param = etpan_serialize_data_new_str(str);
  add_parameter(param);
}
| DSTRING {
  char * str;
  struct etpan_serialize_data * param;
  
  str = etpan_filter_unquote_string($1);
  param = etpan_serialize_data_new_str(str);
  free(str);
  add_parameter(param);
}
| INT {
  struct etpan_serialize_data * param;
  
  param = etpan_serialize_data_new_int64($1);
  add_parameter(param);
  };

optional_action_list:
action_list
|;

action_list:
action action_list
| action;

action:
function_call ';' {
  create_action_from_function_call();
};

%%

static void
etpan_filter_lexer_error(char *msg)
{
  fprintf(stderr, "%u (%u:%u): %s\n", etpan_filter_parser_param.offset,
      etpan_filter_parser_param.line, etpan_filter_parser_param.column,
      msg);
}

void etpan_filter_parser_cleanup(void)
{
  /* cleanup param */  
  set_rule_name(NULL);
  set_function_name(NULL);
  cleanup_parameter_list();
  cleanup_function_call();
  cleanup_condition_stack();
  cleanup_action_list();
}

static void setup_parameter_list(void)
{
  etpan_filter_parser_param.current_param_list = carray_new(16);
  if (etpan_filter_parser_param.current_param_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
}

void free_param_list(carray * param_list)
{
  unsigned int i;

  ETPAN_LOCAL_LOG("FREE PARAM LIST");
  for(i = 0 ; i < carray_count(param_list) ; i ++) {
    struct etpan_serialize_data * param;
    
    param = carray_get(param_list, i);
    etpan_serialize_data_free(param);
  }
  carray_free(param_list);
}

static void cleanup_parameter_list(void)
{
  if (etpan_filter_parser_param.current_param_list == NULL)
    return;
  
  free_param_list(etpan_filter_parser_param.current_param_list);
  etpan_filter_parser_param.current_param_list = NULL;
}

static void add_parameter(struct etpan_serialize_data * param)
{
  int r;
  
  r = carray_add(etpan_filter_parser_param.current_param_list, param, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

static void create_function_call(void)
{
  struct function_call * call;
  char * function_name;
  
  function_name = etpan_filter_parser_param.current_function_name;
  ETPAN_LOCAL_LOG("function call %s %i", function_name, carray_count(etpan_filter_parser_param.current_param_list));
  
  call = malloc(sizeof(* call));
  if (call == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  call->function_name = function_name;
  call->param_list = etpan_filter_parser_param.current_param_list;
  etpan_filter_parser_param.current_param_list = NULL;
  etpan_filter_parser_param.current_function_name = NULL;
  
  etpan_filter_parser_param.current_function_call = call;
}

static void cleanup_function_call(void)
{
  struct function_call * call;
  
  call = etpan_filter_parser_param.current_function_call;
  if (call == NULL)
    return;
  
  free_param_list(call->param_list);
  free(call->function_name);
  free(call);
  etpan_filter_parser_param.current_function_call = NULL;
}

static void set_function_name(char * str)
{
  free(etpan_filter_parser_param.current_function_name);
  if (str == NULL) {
    etpan_filter_parser_param.current_function_name = NULL;
  }
  else {
    etpan_filter_parser_param.current_function_name = strdup(str);
    if (etpan_filter_parser_param.current_function_name == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
}

static void set_rule_name(char * str)
{
  free(etpan_filter_parser_param.current_rule_name);
  if (str == NULL) {
    etpan_filter_parser_param.current_rule_name = NULL;
  }
  else {
    etpan_filter_parser_param.current_rule_name = strdup(str);
    if (etpan_filter_parser_param.current_rule_name == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
}

static void cleanup_condition_stack(void)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(etpan_filter_parser_param.condition_stack) ; i ++) {
    struct etpan_filter_condition * condition;
    
    condition = carray_get(etpan_filter_parser_param.condition_stack, i);
    etpan_filter_condition_free(condition);
  }
  
  carray_free(etpan_filter_parser_param.condition_stack);
  etpan_filter_parser_param.condition_stack = NULL;
}

static void push_condition(struct etpan_filter_condition * condition)
{
  int r;
  
  if (etpan_filter_parser_param.condition_stack == NULL) {
    etpan_filter_parser_param.condition_stack = carray_new(4);
    if (etpan_filter_parser_param.condition_stack == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  r = carray_add(etpan_filter_parser_param.condition_stack, condition, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

static struct etpan_filter_condition * pop_condition(void)
{
  unsigned int count;
  struct etpan_filter_condition * value;
  
  if (etpan_filter_parser_param.condition_stack == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  count = carray_count(etpan_filter_parser_param.condition_stack);
  if (count == 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  value = carray_get(etpan_filter_parser_param.condition_stack, count - 1);
  
  carray_set_size(etpan_filter_parser_param.condition_stack, count - 1);
  
  return value;
}

static void create_condition_from_function_call(void)
{
  struct etpan_filter_condition * condition;
  struct function_call * call;
  
  call = etpan_filter_parser_param.current_function_call;
  condition = etpan_filter_condition_new(call->function_name,
      call->param_list);
  free(call->function_name);
  free(call);
  push_condition(condition);
  
  etpan_filter_parser_param.current_function_call = NULL;
}

static void create_and_condition(void)
{
  struct etpan_filter_condition * condition;
  struct etpan_filter_condition * condition1;
  struct etpan_filter_condition * condition2;
  carray * param_list;
  
  condition2 = pop_condition();
  condition1 = pop_condition();
  
  param_list = carray_new(4);
  if (param_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  condition = etpan_filter_condition_new("and", param_list);
  etpan_filter_condition_add(condition, condition1);
  etpan_filter_condition_add(condition, condition2);
  
  push_condition(condition);
}

static void create_or_condition(void)
{
  struct etpan_filter_condition * condition;
  struct etpan_filter_condition * condition1;
  struct etpan_filter_condition * condition2;
  carray * param_list;
  
  condition2 = pop_condition();
  condition1 = pop_condition();
  
  param_list = carray_new(4);
  if (param_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  condition = etpan_filter_condition_new("or", param_list);
  etpan_filter_condition_add(condition, condition1);
  etpan_filter_condition_add(condition, condition2);
  
  push_condition(condition);
}

static void create_not_condition(void)
{
  struct etpan_filter_condition * condition;
  struct etpan_filter_condition * condition1;
  carray * param_list;
  
  condition1 = pop_condition();
  
  param_list = carray_new(4);
  if (param_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  condition = etpan_filter_condition_new("not", param_list);
  etpan_filter_condition_add(condition, condition1);
  
  push_condition(condition);
}

static void create_constant_condition(int value)
{
  carray * param_list;
  struct etpan_filter_condition * condition;
  
  param_list = carray_new(4);
  if (param_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  if (value)
    condition = etpan_filter_condition_new("true", param_list);
  else
    condition = etpan_filter_condition_new("false", param_list);
  
  push_condition(condition);
}

static void create_action_from_function_call(void)
{
  struct etpan_filter_action * action;
  struct function_call * call;
  
  call = etpan_filter_parser_param.current_function_call;
  ETPAN_LOCAL_LOG("create action: %s %p", call->function_name, call->param_list);
  action = etpan_filter_action_new(call->function_name,
      call->param_list);
  free(call->function_name);
  free(call);
  add_action(action);
  
  etpan_filter_parser_param.current_function_call = NULL;
}

static void cleanup_action_list(void)
{
  unsigned int i;
  
  if (etpan_filter_parser_param.action_list == NULL)
    return;
  
  for(i = 0 ; i < carray_count(etpan_filter_parser_param.action_list) ; i ++) {
    struct etpan_filter_action * action;
    
    action = carray_get(etpan_filter_parser_param.action_list, i);
    etpan_filter_action_free(action);
  }
  carray_free(etpan_filter_parser_param.action_list);
  
  etpan_filter_parser_param.action_list = NULL;
}

static void add_action(struct etpan_filter_action * action)
{
  int r;
  
  if (etpan_filter_parser_param.action_list == NULL) {
    etpan_filter_parser_param.action_list = carray_new(4);
    if (etpan_filter_parser_param.action_list == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  r = carray_add(etpan_filter_parser_param.action_list, action, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

static void create_filter(void)
{
  struct etpan_filter * filter;
  struct etpan_filter_condition * condition;
  unsigned int i;
  
  filter = etpan_filter_new();
  ETPAN_LOCAL_LOG("create file [%s]", etpan_filter_parser_param.current_rule_name);
  etpan_filter_set_name(filter, etpan_filter_parser_param.current_rule_name);
  condition = pop_condition();
  etpan_filter_set_condition(filter, condition);
  for(i = 0 ; i < carray_count(etpan_filter_parser_param.action_list) ; i ++) {
    struct etpan_filter_action * action;
    
    action = carray_get(etpan_filter_parser_param.action_list, i);
    etpan_filter_add_action(filter, action);
  }
  
  carray_free(etpan_filter_parser_param.action_list);
  etpan_filter_parser_param.action_list = NULL;
  
  etpan_filter_config_add_rule(etpan_filter_parser_param.config, filter);
  ETPAN_LOCAL_LOG("added rule");
}
