#include "etpan-filter.h"

#include <string.h>
#include <stdlib.h>

#include "etpan-log.h"
#include "etpan-address.h"
#include "etpan-message.h"
#include "etpan-message-header.h"
#include "etpan-message-fetcher.h"
#include "etpan-signal.h"
#include "etpan-error.h"
#include "etpan-part.h"
#include "etpan-filter-condition.h"
#include "etpan-filter-action.h"

void etpan_filter_init(void)
{
  etpan_filter_condition_register_default();
  etpan_filter_action_register_default();
}

void etpan_filter_done(void)
{
  etpan_filter_action_unregister_all();
  etpan_filter_condition_unregister_all();
}

struct etpan_filter * etpan_filter_dup(struct etpan_filter * filter)
{
  struct etpan_filter * dup_filter;
  unsigned int i;
  int r;
  
  dup_filter = malloc(sizeof(* dup_filter));
  if (dup_filter == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  if (filter->name != NULL) {
    dup_filter->name = strdup(filter->name);
    if (dup_filter->name == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  else {
    dup_filter->name = NULL;
  }
  dup_filter->condition = etpan_filter_condition_dup(filter->condition);
  
  dup_filter->action_list = carray_new(carray_count(filter->action_list));
  for(i = 0 ; i < carray_count(filter->action_list) ; i ++) {
    struct etpan_filter_action * action;
    struct etpan_filter_action * dup_action;
    
    action = carray_get(filter->action_list, i);
    dup_action = etpan_filter_action_dup(action);
    r = carray_add(dup_filter->action_list, dup_action, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  return dup_filter;
}

struct etpan_filter * etpan_filter_new(void)
{
  struct etpan_filter * filter;
  
  filter = malloc(sizeof(* filter));
  filter->name = NULL;
  filter->condition = NULL;
  filter->action_list = carray_new(4);
  if (filter->action_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return filter;
}

void etpan_filter_free(struct etpan_filter * filter)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(filter->action_list) ; i ++) {
    struct etpan_filter_action * action;
    
    action = carray_get(filter->action_list, i);
    etpan_filter_action_free(action);
  }
  carray_free(filter->action_list);
  etpan_filter_condition_free(filter->condition);
  free(filter->name);
  free(filter);
}

void etpan_filter_set_condition(struct etpan_filter * filter,
    struct etpan_filter_condition * condition)
{
  filter->condition = condition;
}

struct etpan_filter_condition * etpan_filter_get_condition(struct etpan_filter * filter)
{
  return filter->condition;
}

void etpan_filter_add_action(struct etpan_filter * filter,
    struct etpan_filter_action * action)
{
  int r;
  
  r = carray_add(filter->action_list, action, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

carray * etpan_filter_get_action_list(struct etpan_filter * filter)
{
  return filter->action_list;
}

void etpan_filter_set_name(struct etpan_filter * filter, char * name)
{
  if (filter->name != name) {
    free(filter->name);
    if (name != NULL) {
      filter->name = strdup(name);
      if (filter->name == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else {
      filter->name = NULL;
    }
  }
}

char * etpan_filter_get_name(struct etpan_filter * filter)
{
  return filter->name;
}

struct filter_data {
  void (* callback)(int filtered, void *);
  void * cb_data;
  struct etpan_filter * filter;
  struct etpan_message * msg;
  struct etpan_filter_state * filter_state;
};

static void condition_callback(int result, void * data);

void etpan_filter_message(struct etpan_filter_state * filter_state,
    struct etpan_filter * filter,
    struct etpan_message * msg,
    void (* callback)(int filtered, void *), void * cb_data)
{
  struct filter_data * data;
  
  data = malloc(sizeof(* data));
  if (data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  data->callback = callback;
  data->cb_data = cb_data;
  data->msg = msg;
  data->filter = filter;
  data->filter_state = filter_state;
  
  etpan_filter_condition_evaluate(filter->condition, msg,
      condition_callback, data);
}

static void filter_action_callback(void * data);
static void filter_callback(int result, struct filter_data * data);

static void condition_callback(int result, void * data)
{
  struct filter_data * filter_data;
  struct etpan_filter * filter;
  
  filter_data = data;
  filter = filter_data->filter;
  
  if (result) {
    etpan_filter_action_list_run(filter_data->filter_state, filter->action_list,
        filter_data->msg, filter_action_callback, filter_data);
  }
  else {
    filter_callback(result, filter_data);
  }
}

static void filter_action_callback(void * data)
{
  struct filter_data * filter_data;
  
  filter_data = data;
  filter_callback(1, filter_data);
}

static void filter_callback(int result, struct filter_data * data)
{
  data->callback(result, data->cb_data);
  free(data);
}


struct etpan_filter_state * etpan_filter_state_new(void)
{
  struct etpan_filter_state * filter_state;
  
  filter_state = malloc(sizeof(* filter_state));
  if (filter_state == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  filter_state->config = NULL;
  filter_state->current_index = 0;
  filter_state->filter_callback = NULL;
  filter_state->cb_data = NULL;
  filter_state->msg = NULL;
  
  filter_state->update_folder_hash = chash_new(CHASH_DEFAULTSIZE,
      CHASH_COPYKEY);
  if (filter_state->update_folder_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  filter_state->checkpoint_folder_hash = chash_new(CHASH_DEFAULTSIZE,
      CHASH_COPYKEY);
  if (filter_state->checkpoint_folder_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return filter_state;
}

void etpan_filter_state_free(struct etpan_filter_state * filter_state)
{
  chash_free(filter_state->checkpoint_folder_hash);
  chash_free(filter_state->update_folder_hash);
  free(filter_state);
}
