#include "etpan-folder-filter.h"

#include <stdlib.h>
#include <string.h>
#include <libgen.h>

#include "etpan-account.h"
#include "etpan-storage.h"
#include "etpan-log.h"
#include "etpan-msg-list-iterator.h"
#include "etpan-folder.h"
#include "etpan-filter.h"
#include "etpan-message.h"
#include "etpan-serialize.h"
#include "etpan-sqldb.h"
#include "etpan-utils.h"
#include "etpan-account-manager.h"
#include "etpan-filter-config.h"
#include "etpan-thread-manager-app.h"
#include "etpan-signal.h"

static void add_pending_filtered_db(struct etpan_folder_filter * folder_filter,
    char * uid);
static void flush_pending_filtered_db(struct etpan_folder_filter * folder_filter);

static void setup_update(struct etpan_folder_filter * folder_filter);
static void unsetup_update(struct etpan_folder_filter * folder_filter);

struct etpan_folder_filter * etpan_folder_filter_new(void)
{
  struct etpan_folder_filter * folder_filter;
  
  folder_filter = malloc(sizeof(* folder_filter));
  if (folder_filter == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  folder_filter->ref_count = 1;
  folder_filter->folder = NULL;
  folder_filter->filter_config = NULL;
  folder_filter->filter_config_copy = NULL;
  folder_filter->iterator = etpan_msg_list_iterator_new();
  folder_filter->filter_one_callback = NULL;
  folder_filter->filter_one_cb_data = NULL;
  folder_filter->filter_state = etpan_filter_state_new();
  folder_filter->cancelled = 0;
  folder_filter->filtered_db = NULL;
  folder_filter->msg = NULL;
  folder_filter->step_callback_wrapped = NULL;
  folder_filter->callback_wrapped = NULL;
  folder_filter->cb_data_wrapped = NULL;
  folder_filter->running = 0;
  folder_filter->path = NULL;
  folder_filter->pending_filtered_db = chash_new(CHASH_DEFAULTSIZE,
      CHASH_COPYKEY);
  
  return folder_filter;
}

static void etpan_folder_filter_free(struct etpan_folder_filter * folder_filter)
{
  unsetup_update(folder_filter);
  
  chash_free(folder_filter->pending_filtered_db);
  
  free(folder_filter->path);
  etpan_msg_list_iterator_free(folder_filter->iterator);
  if (folder_filter->folder != NULL) {
    if (folder_filter->filtered_db != NULL) {
      etpan_sqldb_close(folder_filter->filtered_db);
      etpan_sqldb_free(folder_filter->filtered_db);
      folder_filter->filtered_db = NULL;
    }
    etpan_folder_unref(folder_filter->folder);
    folder_filter->folder = NULL;
  }
  free(folder_filter);
}

void etpan_folder_filter_ref(struct etpan_folder_filter * folder_filter)
{
  folder_filter->ref_count ++;
}

void etpan_folder_filter_unref(struct etpan_folder_filter * folder_filter)
{
  folder_filter->ref_count --;
  if (folder_filter->ref_count != 0)
    return;
  
  etpan_folder_filter_free(folder_filter);
}

void etpan_folder_filter_set_folder(struct etpan_folder_filter * folder_filter,
    struct etpan_folder * folder)
{
  folder_filter->folder = folder;
  etpan_folder_ref(folder);
}

struct etpan_folder * etpan_folder_filter_get_folder(struct etpan_folder_filter * folder_filter)
{
  return folder_filter->folder;
}

void etpan_folder_filter_set_filter(struct etpan_folder_filter * folder_filter,
    struct etpan_filter_config * filter_config)
{
  folder_filter->filter_config = filter_config;
}

static void filter_msg(struct etpan_message * msg, void * data, void (* callback)(int iterate_continue, void *), void * cb_data);

void etpan_folder_filter_setup(struct etpan_folder_filter * folder_filter)
{
  int r;
  carray * columns;
  char * tmp_path;
  char * basedir;
  
  ETPAN_ASSERT(folder_filter->folder != NULL, "folder should not be NULL");
  
  tmp_path = strdup(folder_filter->path);
  basedir = dirname(tmp_path);
  etpan_make_dir(basedir);
  free(tmp_path);
  
  columns = carray_new(16);
  if (columns == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  r = carray_add(columns, "filtered", NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  folder_filter->filtered_db = etpan_sqldb_new(folder_filter->path, columns);
  carray_free(columns);
  
  r = etpan_sqldb_open(folder_filter->filtered_db);
  if (r < 0) {
    ETPAN_CRASH("could not open filtered database");
  }
  
  setup_update(folder_filter);
}

static void run_step_callback(void * data);
static void run_callback(void * data);

static void etpan_folder_filter_run(struct etpan_folder_filter * folder_filter,
    void (* step_callback)(void *),
    void (* callback)(void *),
    void * cb_data)
{
  chash * msg_hash;
  chash * non_filtered_msg_hash;
  chashiter * iter;
  
  msg_hash = etpan_folder_get_message_hash(folder_filter->folder);
  ETPAN_LOG("filter msg: %s %i",
      etpan_folder_get_location(folder_filter->folder),
      chash_count(msg_hash));
  non_filtered_msg_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (non_filtered_msg_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  for(iter = chash_begin(msg_hash) ; iter != NULL ;
      iter = chash_next(msg_hash, iter)) {
    chashdatum key;
    chashdatum value;
    struct etpan_message * msg;
    
    chash_key(iter, &key);
    chash_value(iter, &value);
    msg = value.data;
    if (!etpan_message_is_filtered(msg)) {
      chash_set(non_filtered_msg_hash, &key, &value, NULL);
    }
  }
  ETPAN_LOG("non-filtered msg: %i", chash_count(non_filtered_msg_hash));
  
  etpan_msg_list_iterator_set_msg_list(folder_filter->iterator,
      non_filtered_msg_hash);
  chash_free(non_filtered_msg_hash);
  
  etpan_msg_list_iterator_set_function(folder_filter->iterator, filter_msg,
      folder_filter);
  
  folder_filter->step_callback_wrapped = step_callback;
  folder_filter->callback_wrapped = callback;
  folder_filter->cb_data_wrapped = cb_data;
  ETPAN_LOG("filter_config: %s %p %p",
      etpan_storage_get_id(etpan_folder_get_storage(folder_filter->folder)),
      folder_filter, folder_filter->filter_config);
  folder_filter->filter_config_copy =
    etpan_filter_config_dup(folder_filter->filter_config);
  
  folder_filter->running = 1;
  etpan_msg_list_iterator_run(folder_filter->iterator,
      run_step_callback, run_callback, folder_filter);
}

static void run_step_callback(void * data)
{
  struct etpan_folder_filter * folder_filter;
  
  folder_filter = data;
  folder_filter->step_callback_wrapped(folder_filter->cb_data_wrapped);
}

static void run_callback(void * data)
{
  struct etpan_folder_filter * folder_filter;
  
  folder_filter = data;
  etpan_filter_config_free(folder_filter->filter_config_copy);
  folder_filter->filter_config_copy = NULL;
  folder_filter->running = 0;
  folder_filter->callback_wrapped(folder_filter->cb_data_wrapped);
}

static void encode_uint32(uint32_t value, void ** p_data, size_t * p_length)
{
  struct etpan_serialize_data * sdata;
  
  sdata = etpan_serialize_data_new_uint32(value);
  etpan_serialize_encode(sdata, p_data, p_length);
  etpan_serialize_data_free(sdata);
}

static uint32_t decode_uint32(void * data, size_t length)
{
  struct etpan_serialize_data * sdata;
  uint32_t value;
  
  sdata = etpan_serialize_decode(data, length);
  value = etpan_serialize_data_get_uint32(sdata);
  etpan_serialize_data_free(sdata);
  
  return value;
}

static void filter_callback(void * data);

static void filter_msg(struct etpan_message * msg, void * data, void (* callback)(int iterate_continue, void *), void * cb_data)
{
  struct etpan_folder_filter * folder_filter;
  void * value;
  size_t len;
  int r;
  char * uid;
  
  folder_filter = data;
  
  folder_filter->filter_one_callback = callback;
  folder_filter->filter_one_cb_data = cb_data;
  
  uid = etpan_message_get_uid(msg);
  folder_filter->msg = msg;
  
  if (etpan_message_is_filtered(msg)) {
    folder_filter->filter_one_callback(!folder_filter->cancelled,
        folder_filter->filter_one_cb_data);
    return;
  }
  
  ETPAN_LOG("filtering %s", uid);
  etpan_message_ref(msg);
  etpan_filter_config_apply_message(folder_filter->filter_state,
      folder_filter->filter_config_copy, msg, filter_callback, folder_filter);
}

static void filter_callback(void * data)
{
  struct etpan_folder_filter * folder_filter;
  void * value;
  size_t len;
  int r;
  struct etpan_message * msg;
  char * uid;
  
  folder_filter = data;
  
  msg = folder_filter->msg;
  uid = etpan_message_get_uid(msg);
  ETPAN_LOG("filtering %s done", uid);
  etpan_message_set_filtered(msg, 1);
  add_pending_filtered_db(folder_filter, uid);
  
  etpan_message_unref(msg);
  folder_filter->msg = NULL;
  
  folder_filter->filter_one_callback(!folder_filter->cancelled,
      folder_filter->filter_one_cb_data);
}

void etpan_folder_filter_cancel(struct etpan_folder_filter * folder_filter)
{
  folder_filter->cancelled = 1;
}

int etpan_folder_filter_is_running(struct etpan_folder_filter * folder_filter)
{
  return folder_filter->running;
}

char * etpan_folder_filter_get_path_for_folder(struct etpan_folder_filter * folder_filter)
{
  struct etpan_storage * storage;
  struct etpan_account * account;
  char * location;
  char * filter_dir;
  char result[PATH_MAX];
  char * qmb;
  char * value;
  
  location = etpan_folder_get_location(folder_filter->folder);
  
  storage = etpan_folder_get_storage(folder_filter->folder);
  account = etpan_storage_get_account(storage);
  
  filter_dir = etpan_get_filter_dir(etpan_account_manager_get_default(),
      account);
  qmb = etpan_quote_mailbox(location);
  
  snprintf(result, sizeof(result), "%s/%s.db", filter_dir, qmb);
  
  free(qmb);
  free(filter_dir);
  
  value = strdup(result);
  if (value == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return value;
}

void etpan_folder_filter_set_path(struct etpan_folder_filter * folder_filter,
    char * path)
{
  if (path != folder_filter->path) {
    free(folder_filter->path);
    if (path != NULL) {
      folder_filter->path = strdup(path);
      if (folder_filter->path == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      folder_filter->path = NULL;
  }
}

char * etpan_folder_filter_get_path(struct etpan_folder_filter * folder_filter)
{
  return folder_filter->path;
}

static chash * get_filtered_messages(struct etpan_folder_filter * folder_filter)
{
  unsigned int i;
  carray * keys;
  chash * result;
  int r;
  
  result = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (result == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  keys = etpan_sqldb_get_keys(folder_filter->filtered_db);
  for(i = 0 ; i < carray_count(keys) ; i ++) {
    char * uid;
    chashdatum key;
    chashdatum value;
    void * serialized_value;
    size_t serialized_len;
    int filtered;
    
    uid = carray_get(keys, i);
    
    r = etpan_sqldb_get(folder_filter->filtered_db,
        uid, "filtered", &serialized_value,  &serialized_len);
    if (r == 0) {
      /* already filtered */
      filtered = decode_uint32(serialized_value, serialized_len);
      free(serialized_value);
    }
    else {
      filtered = 0;
    }
    
    if (filtered) {
      key.data = uid;
      key.len = strlen(uid) + 1;
      value.data = NULL;
      value.len = 0;
      r = chash_set(result, &key, &value, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
    }
  }
  etpan_sqldb_keys_free(keys);
  
  return result;
}

static void
fetch_filtered_msg_list_run(struct etpan_thread_op * op);

static void
fetch_filtered_msg_list_cleanup(struct etpan_thread_op * op);

void etpan_folder_filter_fetch_filtered_message_list(struct etpan_thread_manager_app * manager,
    struct etpan_folder_filter * folder_filter,
    void (* callback)(int, struct etpan_folder_filter_filtered_msg_list_result *,
        void *),
    void * data)
{
  struct etpan_thread_op * op;
  struct etpan_folder_filter_filtered_msg_list_result * result;
  
  result = malloc(sizeof(* result));
  if (result == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  result->msg_hash = NULL;
  
  op = etpan_thread_op_new();
  op->param = folder_filter;
  op->result = result;
  
  op->cancellable = 1;
  op->run = fetch_filtered_msg_list_run;
  op->callback = (void (*)(int, void *, void *)) callback;
  op->callback_data = data;
  op->cleanup = fetch_filtered_msg_list_cleanup;
  
  etpan_thread_manager_app_storage_schedule(manager,
      folder_filter->folder->storage, op);
}

static void
fetch_filtered_msg_list_run(struct etpan_thread_op * op)
{
  struct etpan_folder_filter_filtered_msg_list_result * result;
  struct etpan_folder_filter * folder_filter;
  
  folder_filter = op->param;
  result = op->result;
  result->msg_hash = get_filtered_messages(folder_filter);
}

static void
fetch_filtered_msg_list_cleanup(struct etpan_thread_op * op)
{
  struct etpan_folder_filter_filtered_msg_list_result * result;
  
  result = op->result;
  if (result != NULL) {
    chash_free(result->msg_hash);
    result->msg_hash = NULL;
  }
}

#if 0
static void folder_content_changed_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);
#endif

static void setup_update(struct etpan_folder_filter * folder_filter)
{
#if 0
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_FOLDER_MSGLIST_UPDATED,
      folder_filter->folder, folder_filter,
      folder_content_changed_handler);
  
  folder_content_changed_handler(ETPAN_FOLDER_MSGLIST_UPDATED,
      folder_filter->folder,
      NULL, folder_filter);
#endif
}

static void unsetup_update(struct etpan_folder_filter * folder_filter)
{
#if 0
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_FOLDER_MSGLIST_UPDATED,
      folder_filter->folder, folder_filter,
      folder_content_changed_handler);
#endif
}

static void folder_filter_step_callback(void * data);
static void folder_filter_callback(void * data);

#if 0
static void folder_content_changed_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_folder_filter * folder_filter;
  
  folder_filter = user_data;
  
  ETPAN_LOG("content changed");
  etpan_folder_filter_run(folder_filter,
      folder_filter_step_callback, folder_filter_callback, folder_filter);
}
#endif

static void folder_filter_step_callback(void * data)
{
  struct etpan_folder_filter * folder_filter;
  
  folder_filter = data;
  if (chash_count(folder_filter->pending_filtered_db) >= 100) {
    flush_pending_filtered_db(folder_filter);
  }
}

static void folder_filter_callback(void * data)
{
  struct etpan_folder_filter * folder_filter;
  
  ETPAN_LOG("filtered all");
  
  folder_filter = data;
  flush_pending_filtered_db(folder_filter);
  
  if (folder_filter->pending) {
    etpan_folder_filter_force_run(folder_filter);
  }
}

static void flush_pending_filtered_db(struct etpan_folder_filter * folder_filter)
{
  chashiter * iter;
  int r;
  
  if (chash_count(folder_filter->pending_filtered_db) == 0)
    return;
  
  etpan_sqldb_begin_transaction(folder_filter->filtered_db);
  for(iter = chash_begin(folder_filter->pending_filtered_db) ; iter != NULL ;
      iter = chash_next(folder_filter->pending_filtered_db, iter)) {
    void * value;
    size_t len;
    chashdatum key;
    char * uid;
    
    chash_key(iter, &key);
    uid = key.data;
    encode_uint32(1, &value, &len);
    r = etpan_sqldb_set(folder_filter->filtered_db,
        uid, "filtered", value,  len);
    /* ignore error */
    free(value);
  }
  etpan_sqldb_end_transaction(folder_filter->filtered_db);
  chash_clear(folder_filter->pending_filtered_db);
}

static void add_pending_filtered_db(struct etpan_folder_filter * folder_filter,
    char * uid)
{
  chashdatum key;
  chashdatum value;
  int r;
  
  key.data = uid;
  key.len = strlen(uid) + 1;
  value.data = NULL;
  value.len = 0;
  r = chash_set(folder_filter->pending_filtered_db, &key, &value, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

void etpan_folder_filter_force_run(struct etpan_folder_filter * folder_filter)
{
  if (etpan_folder_filter_is_running(folder_filter)) {
    folder_filter->pending = 1;
    return;
  }
  
  folder_filter->pending = 0;
  ETPAN_LOG("force run");
  etpan_folder_filter_run(folder_filter,
      folder_filter_step_callback, folder_filter_callback, folder_filter);
}
