#include "etpan-folder-imap-sync.h"
#include "etpan-folder-imap-sync-private.h"

#include <stdlib.h>
#include <string.h>

#define ETPAN_MODULE_LOG_NAME "IMAPSYNC"

#include "etpan-folder.h"
#include "etpan-error.h"
#include "etpan-log.h"
#include "etpan-imap-sync.h"
#include "etpan-imap-sync-private.h"
#include "etpan-imap-folder-sync.h"
#include "etpan-imap-folder-sync-private.h"
#include "etpan-storage-imap-sync.h"
#include "etpan-storage-imap-sync-private.h"
#include "etpan-signal.h"

static struct etpan_error * setup(struct etpan_folder * folder);

static struct etpan_error * fetch_msg_list(struct etpan_folder * folder,
    chash * current_msg_list);

static struct etpan_error * append_msg(struct etpan_folder * folder,
    char * message, size_t size,
    struct etpan_message_flags * flags);

static struct etpan_error * check(struct etpan_folder * folder);

static struct etpan_error * status(struct etpan_folder * folder,
    unsigned int * p_count, unsigned int * p_unseen, unsigned int * p_recent);

static struct etpan_error * expunge(struct etpan_folder * folder);

static void free_data(struct etpan_folder * folder);

static struct etpan_error * connect(struct etpan_folder * folder);
static void disconnect(struct etpan_folder * folder);

static int is_sub_folder(struct etpan_folder * parent,
    struct etpan_folder * child);

static struct etpan_folder_driver imap_sync_driver = {
  .name = "imap-sync",
  
  .setup = setup,
  .fetch_msg_list = fetch_msg_list,
  .append_msg = append_msg,
  .check = check,
  .status = status,
  .expunge = expunge,
  .free_data = free_data,
  .connect = connect,
  .disconnect = disconnect,
  .is_sub_folder = is_sub_folder,
};

struct folder_data {
  struct etpan_imap_folder_sync * folder_sync;
};

struct etpan_folder * etpan_folder_imap_sync_new(void)
{
  struct etpan_folder * folder;
  struct folder_data * data;
  
  folder = etpan_folder_new();
  data = malloc(sizeof(* data));
  if (data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  data->folder_sync = NULL;
  
  etpan_folder_set_data(folder, data);
  etpan_folder_set_driver(folder, &imap_sync_driver);
  
  return folder;
}

static void content_updated_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);

void etpan_folder_imap_sync_set_folder_sync(struct etpan_folder * folder,
    struct etpan_imap_folder_sync * folder_sync)
{
  struct folder_data * data;
  
  data = etpan_folder_get_data(folder);
  data->folder_sync = folder_sync;
  etpan_imap_folder_sync_ref(folder_sync);
  ETPAN_SIGNAL_ADD_HANDLER(folder_sync,
      ETPAN_IMAP_FOLDER_SYNC_CONTENT_UPDATED_SIGNAL,
      content_updated_handler, folder);
}

static void content_updated_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_folder * folder;
  
  folder = user_data;
  
  ETPAN_SIGNAL_SEND(folder,
      ETPAN_FOLDER_IMAP_SYNC_CONTENT_UPDATED_ENDED_SIGNAL);
}

static struct etpan_error * setup(struct etpan_folder * folder)
{
  return NULL;
}

static struct etpan_error * fetch_msg_list(struct etpan_folder * folder,
    chash * current_msg_list)
{
  struct folder_data * data;
  
  data = etpan_folder_get_data(folder);
  
  return etpan_imap_folder_sync_fetch_msg_list(folder, data->folder_sync,
      current_msg_list);
}

static struct etpan_error * append_msg(struct etpan_folder * folder,
    char * message, size_t size,
    struct etpan_message_flags * flags)
{
  struct folder_data * data;
  struct etpan_error * error;
  
  data = etpan_folder_get_data(folder);
  
  error = etpan_imap_folder_sync_append(data->folder_sync,
      message, size, flags);
  
  return error;
}

static struct etpan_error * check(struct etpan_folder * folder)
{
  struct etpan_imap_folder_sync * folder_sync;
  struct folder_data * data;
  struct etpan_error * error;
  
  data = etpan_folder_get_data(folder);
  folder_sync = data->folder_sync;
  
  error = etpan_imap_folder_sync_check(folder_sync);
  
  return error;
}

static struct etpan_error * status(struct etpan_folder * folder,
    unsigned int * p_count, unsigned int * p_unseen, unsigned int * p_recent)
{
  struct etpan_imap_folder_sync * folder_sync;
  struct folder_data * data;
  struct etpan_error * error;
  
  data = etpan_folder_get_data(folder);
  folder_sync = data->folder_sync;
  
  error = etpan_imap_folder_sync_status(folder_sync,
      p_count, p_unseen, p_recent);
  
  return error;
}

static struct etpan_error * expunge(struct etpan_folder * folder)
{
  return NULL;
}

static void free_data(struct etpan_folder * folder)
{
  struct folder_data * data;
  
  data = etpan_folder_get_data(folder);
  ETPAN_SIGNAL_REMOVE_HANDLER(data->folder_sync,
      ETPAN_IMAP_FOLDER_SYNC_CONTENT_UPDATED_SIGNAL,
      content_updated_handler, folder);
  etpan_imap_folder_sync_unref(data->folder_sync);
  free(data);
}

static struct etpan_error * connect(struct etpan_folder * folder)
{
  return NULL;
}

static void disconnect(struct etpan_folder * folder)
{
  /* do nothing */
}

static int is_sub_folder(struct etpan_folder * parent,
    struct etpan_folder * child)
{
  char * location;
  char * child_location;
  char * prefix;
  size_t size;
  struct folder_data * data;
  struct etpan_imap_folder_sync * folder_sync;
  char separator;
  
  data = etpan_folder_get_data(parent);
  
  location = etpan_folder_get_location(parent);
  child_location = etpan_folder_get_location(child);
  size = strlen(location) + 2;
  prefix = malloc(size);
  if (prefix == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  folder_sync = data->folder_sync;
  separator = etpan_imap_folder_sync_get_imap_separator(folder_sync);
  snprintf(prefix, size, "%s%c", location, separator);
  if (strncmp(prefix, child_location, size - 1) == 0)
    return 1;
  
  return 0;
}

void etpan_folder_imap_sync_check_msg(struct etpan_folder * folder,
    struct etpan_message * msg,
    struct etpan_message_flags * flags)
{
  struct etpan_imap_folder_sync * folder_sync;
  struct folder_data * data;
  
  data = etpan_folder_get_data(folder);
  folder_sync = data->folder_sync;
  
  etpan_imap_folder_sync_check_msg(folder_sync, msg, flags);
}

struct etpan_error *
etpan_folder_imap_sync_get_body(struct etpan_folder * folder,
    char * uid, struct mailimap_body ** result)
{
  struct etpan_imap_folder_sync * folder_sync;
  struct folder_data * data;
  
  data = etpan_folder_get_data(folder);
  folder_sync = data->folder_sync;
  
  return etpan_imap_folder_sync_get_body(folder_sync, uid, result);
}

struct etpan_error *
etpan_folder_imap_sync_get_part(struct etpan_folder * folder,
    char * uid, char * section, int part_type,
    void ** p_data, size_t * p_length)
{
  struct etpan_imap_folder_sync * folder_sync;
  struct folder_data * data;
  
  data = etpan_folder_get_data(folder);
  folder_sync = data->folder_sync;
  
  return etpan_imap_folder_sync_get_part(folder_sync,
    uid, section, part_type, p_data, p_length);
}

struct etpan_imap_folder_sync *
etpan_folder_imap_sync_get_folder_sync(struct etpan_folder * folder)
{
  struct folder_data * data;
  
  data = etpan_folder_get_data(folder);
  return data->folder_sync;
}
