#include "etpan-imap-folder-sync.h"

#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#define ETPAN_MODULE_LOG_NAME "IMAPSYNC"

#include "etpan-log.h"
#include "etpan-error.h"
#include "etpan-nls.h"
#include "etpan-imap-sync.h"
#include "etpan-sqldb.h"
#include "etpan-account-manager.h"
#include "etpan-utils.h"
#include "etpan-lep.h"
#include "etpan-message.h"
#include "etpan-message-header.h"
#include "etpan-serialize.h"
#include "etpan-imap-serialize.h"
#include "etpan-thread-manager.h"
#include "etpan-thread-manager-app.h"
#include "etpan-imap-sync.h"
#include "etpan-imap-sync-private.h"
#include "etpan-account.h"
#include "etpan-time.h"
#include "etpan-signal.h"
#include "etpan-message-imap-sync.h"
#include "etpan-imap.h"
#include "etpan-uuid.h"
#include "etpan-mail-manager.h"

#define SYNC_INTERVAL (5 * 60)
#define FLAG_SYNC_INTERVAL (15 * 60)

/* TODO : sync along with body (for INBOX only,
   needed to filter messages to other mailboxes
   without they have a chance to appear in INBOX,
   maybe filter during sync) */

enum {
  STATE_DISCONNECTED,
  STATE_CONNECTED,
  STATE_STREAM_CLOSED,
  STATE_HAS_UIDLIST,
  STATE_HAS_ENVELOPE,
  STATE_HAS_FLAGS,
  STATE_HAS_BODY,
};

enum {
  EP_SECTION_MESSAGE, /* whole part */
  EP_SECTION_HEADER, /* header */
  EP_SECTION_MIME, /* mime headers */
  EP_SECTION_BODY, /* body only */
};

struct flags_pair {
  uint32_t uid;
  struct etpan_message_flags * flags;
};

enum {
  ERROR_DOMAIN_SYNC,
  ERROR_DOMAIN_FETCH_MSG_LIST,
  ERROR_DOMAIN_FETCH,
};

static struct etpan_error *
rewrite_error(struct etpan_imap_folder_sync * folder_sync, uint32_t uid,
    int domain,
    struct etpan_error * error);

#if 0
static mailimap * get_imap(struct etpan_imap_folder_sync * folder_sync)
{
  mailimap * imap;
  struct imap_session_state_data * data;
  struct mailfolder * ep_folder;
  mailsession * session;
  
  ep_folder = folder_sync->ep_folder;
  session = ep_folder->fld_session;
  if (strcasecmp(session->sess_driver->sess_name, "imap-cached") == 0) {
    struct imap_cached_session_state_data * cached_data;
    
    cached_data = session->sess_data;
    session = cached_data->imap_ancestor;
  }
  
  data = session->sess_data;
  imap = data->imap_session;
  
  return imap;
}

static mailsession * get_session(struct etpan_imap_folder_sync * folder_sync)
{
  struct mailfolder * ep_folder;
  mailsession * session;
  
  ep_folder = folder_sync->ep_folder;
  session = ep_folder->fld_session;
  
  return session;
}
#endif

static mailsession * get_session(struct etpan_imap_folder_sync * folder_sync)
{
  struct mailstorage * ep_storage;
  mailsession * session;
  
  ep_storage = etpan_imap_sync_get_storage(folder_sync->imap_sync);
  session = ep_storage->sto_session;
  
  return session;
}

static mailimap * get_imap(struct etpan_imap_folder_sync * folder_sync)
{
  mailimap * imap;
  struct imap_session_state_data * data;
  mailsession * session;
  
  session = get_session(folder_sync);
  if (strcasecmp(session->sess_driver->sess_name, "imap-cached") == 0) {
    struct imap_cached_session_state_data * cached_data;
    
    cached_data = session->sess_data;
    session = cached_data->imap_ancestor;
  }
  
  data = session->sess_data;
  imap = data->imap_session;
  
  return imap;
}

static struct etpan_error * select_folder(struct etpan_imap_folder_sync * folder_sync)
{
  struct etpan_error * error;
  struct etpan_imap_sync * imap_sync;
  
  imap_sync = folder_sync->imap_sync;
  
  if (imap_sync->threaded_current_location != NULL) {
    if (strcmp(imap_sync->threaded_current_location, folder_sync->location) == 0)
      return NULL;
  }
  
  error = etpan_imap_select(get_session(folder_sync), folder_sync->location);
  if (error == NULL) {
    free(imap_sync->threaded_current_location);
    imap_sync->threaded_current_location = strdup(folder_sync->location);
    if (imap_sync->threaded_current_location == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  return error;
}

static struct etpan_error *
fetch_uid_list(struct etpan_imap_folder_sync * folder_sync,
    uint32_t lastuid, uint32_t * p_lastuid);
static struct etpan_error *
fetch_env_list(struct etpan_imap_folder_sync * folder_sync,
    struct mailimap_set * set, uint32_t * p_envlastuid);
static struct etpan_error *
fetch_flags_list(struct etpan_imap_folder_sync * folder_sync,
    struct mailimap_set * set);

static void get_uid_list(struct etpan_sqldb * db, uint32_t ** p_uid_list,
    unsigned int * p_count);

#define MSG_ENV_PER_GROUP 100
#define MSG_FLAG_PER_GROUP 200

static void flush_pending_flags(struct etpan_imap_folder_sync * folder_sync);

static struct etpan_error * header_db_open(struct etpan_imap_folder_sync * folder_sync) WARN_UNUSED_RESULT;
static struct etpan_error * body_db_open(struct etpan_imap_folder_sync * folder_sync) WARN_UNUSED_RESULT;
static struct etpan_error * append_db_open(struct etpan_imap_folder_sync * folder_sync) WARN_UNUSED_RESULT;

static struct etpan_error * header_db_open(struct etpan_imap_folder_sync * folder_sync)
{
  char filename[PATH_MAX];
  char * quoted_location;
  char * path;
  carray * columns;
  struct etpan_imap_sync * imap_sync;
  int r;
  
  pthread_mutex_lock(&folder_sync->header_lock);
  
  imap_sync = folder_sync->imap_sync;
  
  path = etpan_get_imap_cache_dir(etpan_account_manager_get_default(),
      imap_sync);
  quoted_location = etpan_quote_mailbox(folder_sync->location);
  
  snprintf(filename, sizeof(filename), "%s/folders", path);
  etpan_make_dir(filename);
  
  snprintf(filename, sizeof(filename), "%s/folders/%s.db", path, quoted_location);
  free(quoted_location);
  free(path);
  
  columns = carray_new(4);
  if (columns == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  r = carray_add(columns, "headers", NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  r = carray_add(columns, "flags", NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  r = carray_add(columns, "presence", NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  folder_sync->header_db = etpan_sqldb_new(filename, columns);
  carray_free(columns);
  
  r = etpan_sqldb_open(folder_sync->header_db);
  if (r < 0) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_DATABASE);
    etpan_error_set_short_description(error, _("Cache could not be written"));
    etpan_error_strf_long_description(error,
        _("Cache of mailbox %s could not be written properly on disk."),
        folder_sync->location);
  }
  
  return NULL;
}

static void header_db_close(struct etpan_imap_folder_sync * folder_sync)
{
  etpan_sqldb_close(folder_sync->header_db);
  etpan_sqldb_free(folder_sync->header_db);
  folder_sync->header_db = NULL;
  pthread_mutex_unlock(&folder_sync->header_lock);
}

static struct etpan_error * body_db_open(struct etpan_imap_folder_sync * folder_sync)
{
  carray * columns;
  char * path;
  char * quoted_location;
  char filename[PATH_MAX];
  struct etpan_imap_sync * imap_sync;
  int r;
  
  pthread_mutex_lock(&folder_sync->body_lock);
  
  imap_sync = folder_sync->imap_sync;
  
  path = etpan_get_imap_cache_dir(etpan_account_manager_get_default(),
      imap_sync);
  quoted_location = etpan_quote_mailbox(folder_sync->location);
  
  snprintf(filename, sizeof(filename), "%s/folders", path);
  etpan_make_dir(filename);
  
  snprintf(filename, sizeof(filename), "%s/folders/%s-body.db", path, quoted_location);
  free(quoted_location);
  free(path);
  
  columns = carray_new(4);
  if (columns == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  r = carray_add(columns, "uid", NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  r = carray_add(columns, "content", NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  folder_sync->body_db = etpan_sqldb_new(filename, columns);
  carray_free(columns);
  
  r = etpan_sqldb_open(folder_sync->body_db);
  if (r < 0) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_DATABASE);
    etpan_error_set_short_description(error, _("Cache could not be written"));
    etpan_error_strf_long_description(error,
        _("Cache of mailbox %s could not be written properly on disk."),
        folder_sync->location);
  }
  
  return NULL;
}

static void body_db_close(struct etpan_imap_folder_sync * folder_sync)
{
  etpan_sqldb_close(folder_sync->body_db);
  etpan_sqldb_free(folder_sync->body_db);
  folder_sync->body_db = NULL;
  pthread_mutex_unlock(&folder_sync->body_lock);
}

static struct etpan_error * append_db_open(struct etpan_imap_folder_sync * folder_sync)
{
  carray * columns;
  char * path;
  char * quoted_location;
  char filename[PATH_MAX];
  struct etpan_imap_sync * imap_sync;
  int r;
  
  imap_sync = folder_sync->imap_sync;
  
  path = etpan_get_imap_cache_dir(etpan_account_manager_get_default(),
      imap_sync);
  quoted_location = etpan_quote_mailbox(folder_sync->location);
  
  snprintf(filename, sizeof(filename), "%s/folders", path);
  etpan_make_dir(filename);
  
  snprintf(filename, sizeof(filename), "%s/folders/%s-append.db", path, quoted_location);
  free(quoted_location);
  free(path);
  
  columns = carray_new(4);
  if (columns == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  r = carray_add(columns, "content", NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  r = carray_add(columns, "flags", NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  folder_sync->append_db = etpan_sqldb_new(filename, columns);
  carray_free(columns);
  
  r = etpan_sqldb_open(folder_sync->append_db);
  if (r < 0) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_DATABASE);
    etpan_error_set_short_description(error, _("Cache could not be written"));
    etpan_error_strf_long_description(error,
        _("Cache of mailbox %s could not be written properly on disk."),
        folder_sync->location);
  }
  
  return NULL;
}

static void append_db_close(struct etpan_imap_folder_sync * folder_sync)
{
  etpan_sqldb_close(folder_sync->append_db);
  etpan_sqldb_free(folder_sync->append_db);
  folder_sync->append_db = NULL;
}

static void encode_uint32(uint32_t value, void ** p_data, size_t * p_length)
{
  struct etpan_serialize_data * sdata;
  
  sdata = etpan_serialize_data_new_uint32(value);
  etpan_serialize_encode(sdata, p_data, p_length);
  etpan_serialize_data_free(sdata);
}

static uint32_t decode_uint32(void * data, size_t length)
{
  struct etpan_serialize_data * sdata;
  uint32_t value;
  
  sdata = etpan_serialize_decode(data, length);
  value = etpan_serialize_data_get_uint32(sdata);
  etpan_serialize_data_free(sdata);
  
  return value;
}

static void encode_str(char * str, size_t length,
    void ** p_data, size_t * p_length)
{
  struct etpan_serialize_data * sdata;
  
  sdata = etpan_serialize_data_new_sized_str(str, length);
  etpan_serialize_encode(sdata, p_data, p_length);
  etpan_serialize_data_free(sdata);
}

static void decode_str(void * data, size_t length,
    char ** p_value, size_t * p_length)
{
  struct etpan_serialize_data * sdata;
  char * value;
  size_t value_length;
  
  sdata = etpan_serialize_decode(data, length);
  etpan_serialize_data_get_sized_str(sdata, &value, &value_length);
  if (p_value != NULL) {
    char * dup_value;
    
    dup_value = malloc(value_length + 1);
    if (dup_value == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    dup_value[value_length] = '\0';
    memcpy(dup_value, value, value_length);
    
    * p_value = dup_value;
  }
  if (p_length != NULL) {
    * p_length = value_length;
  }
  etpan_serialize_data_free(sdata);
}

static int compare_uid(const void * a, const void * b)
{
  const uint32_t * puid1;
  const uint32_t * puid2;
  
  puid1 = a;
  puid2 = b;
  
  return * puid1 - * puid2;
}

static void get_uid_list(struct etpan_sqldb * db, uint32_t ** p_uid_list,
    unsigned int * p_count)
{
  carray * keys;
  unsigned int i;
  uint32_t * uid_list;
  unsigned int count;
  
  keys = etpan_sqldb_get_keys(db);
  count = carray_count(keys);
  
  uid_list = malloc(count * sizeof(* uid_list));
  if (uid_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  for(i = 0 ; i < count ; i ++) {
    uint32_t uid;
    char * key;
    
    key = carray_get(keys, i);
    uid = strtoul(key, NULL, 10);
    uid_list[i] = uid;
  }
  qsort(uid_list, count, sizeof(* uid_list), compare_uid);
  
  etpan_sqldb_keys_free(keys);
  
  * p_count = count;
  * p_uid_list = uid_list;
}

static void find_elt(uint32_t * list, unsigned int count, uint32_t value,
                    unsigned int first_elt, unsigned int last_elt,
                    unsigned int * p_left_index, unsigned int * p_right_index)
{
  unsigned int middle;
  unsigned int left;
  unsigned int right;
    
  left = first_elt;
  right = last_elt;
    
  while (1) {
    middle = (left + right) / 2;
    if (list[middle] <= value) {
      left = middle;
    }
    if (list[middle] == value) {
      break;
    }
    if (list[middle] > value) {
      right = middle;
    }
        
    if (left == right)
      break;
    if (left + 1 == right)
      break;
  }
    
  * p_left_index = left;
  * p_right_index = right;
}

static carray * get_groups(uint32_t * list, unsigned int count,
    uint32_t first, uint32_t last, unsigned int count_per_group)
{
  carray * table;
  unsigned int first_elt;
  unsigned int last_elt;
  unsigned int current_elt;
  unsigned int remaining;
  int r;
  
  ETPAN_LOCAL_LOG("table : %u %u", first, last);
  
  table = carray_new(4);
  if (table == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  if (count == 0) {
    ETPAN_LOCAL_LOG("table empty");
    return table;
  }
  
  first_elt = 0;
  last_elt = count - 1;
  if (first != 0) {
    unsigned int left;
    unsigned int right;
      
    find_elt(list, count, first, first_elt, last_elt, &left, &right);
      
    if (list[right] < first)
      first_elt = right;
    else
      first_elt = left;
      
    if (list[first_elt] < first)
      first_elt ++;
      
    if (first_elt > last_elt) {
      ETPAN_LOCAL_LOG("table empty");
      return table;
    }
  }
  if (last != 0) {
    unsigned int left;
    unsigned int right;
      
    find_elt(list, count, last, first_elt, last_elt, &left, &right);
    if (list[left] > last)
      last_elt = left;
    else
      last_elt = right;
      
    if (list[last_elt] > last) {
      if (last_elt > 0) {
        last_elt --;
              
        if (first_elt > last_elt) {
          ETPAN_LOCAL_LOG("table empty");
          return table;
        }
      }
      else {
        ETPAN_LOCAL_LOG("table empty");
        return table;
      }
    }
  }
  
  ETPAN_LOCAL_LOG("=> table : %u %u", list[first_elt], list[last_elt]);
  
  remaining = last_elt - first_elt + 1;
  current_elt = first_elt;
  while (remaining >= count_per_group) {
    unsigned int upper;
    struct mailimap_set * set;
    
    upper = current_elt + count_per_group - 1;
    set = mailimap_set_new_interval(list[current_elt], list[upper]);
    if (set == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    
    r = carray_add(table, set, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    current_elt = upper + 1;
    remaining -= count_per_group;
  }
  
  if (remaining > 0) {
    unsigned int upper;
    struct mailimap_set * set;
    
    upper = last_elt;
    set = mailimap_set_new_interval(list[current_elt], list[upper]);
    if (set == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    
    r = carray_add(table, set, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  return table;
}

static void imap_add_envelope_fetch_att(struct mailimap_fetch_type * fetch_type)
{
  struct mailimap_fetch_att * fetch_att;
  int r;
  char * header;
  clist * hdrlist;
  struct mailimap_header_list * imap_hdrlist;
  struct mailimap_section * section;

  fetch_att = mailimap_fetch_att_new_envelope();
  if (fetch_att == NULL)
    ETPAN_LOG_MEMORY_ERROR;

  r = mailimap_fetch_type_new_fetch_att_list_add(fetch_type, fetch_att);
  if (r != MAILIMAP_NO_ERROR)
    ETPAN_LOG_MEMORY_ERROR;
  
  header = strdup("References");
  if (header == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  hdrlist = clist_new();
  if (hdrlist == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  r = clist_append(hdrlist, header);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  imap_hdrlist = mailimap_header_list_new(hdrlist);
  if (imap_hdrlist == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  section = mailimap_section_new_header_fields(imap_hdrlist);
  if (section == NULL)
    ETPAN_LOG_MEMORY_ERROR;

  fetch_att = mailimap_fetch_att_new_body_peek_section(section);
  if (fetch_att == NULL)
    ETPAN_LOG_MEMORY_ERROR;

  r = mailimap_fetch_type_new_fetch_att_list_add(fetch_type, fetch_att);
  if (r != MAILIMAP_NO_ERROR)
    ETPAN_LOG_MEMORY_ERROR;
}

static void imap_get_msg_att_info(struct mailimap_msg_att * msg_att,
    uint32_t * puid,
    struct mailimap_envelope ** pimap_envelope,
    char ** preferences,
    size_t * pref_size,
    struct mailimap_msg_att_dynamic ** patt_dyn,
    struct mailimap_body ** pimap_body)
{
  clistiter * item_cur;
  uint32_t uid;
  struct mailimap_envelope * imap_envelope;
  char * references;
  size_t ref_size;
  struct mailimap_msg_att_dynamic * att_dyn;
  struct mailimap_body * imap_body;

  uid = 0;
  imap_envelope = NULL;
  references = NULL;
  ref_size = 0;
  att_dyn = NULL;
  imap_body = NULL;

  for(item_cur = clist_begin(msg_att->att_list) ; item_cur != NULL ;
      item_cur = clist_next(item_cur)) {
    struct mailimap_msg_att_item * item;

    item = clist_content(item_cur);
      
    switch (item->att_type) {
    case MAILIMAP_MSG_ATT_ITEM_STATIC:
      switch (item->att_data.att_static->att_type) {
      case MAILIMAP_MSG_ATT_BODYSTRUCTURE:
	if (imap_body == NULL)
	  imap_body = item->att_data.att_static->att_data.att_bodystructure;
	break;

      case MAILIMAP_MSG_ATT_ENVELOPE:
	if (imap_envelope == NULL) {
	  imap_envelope = item->att_data.att_static->att_data.att_env;
	}
	break;
	  
      case MAILIMAP_MSG_ATT_UID:
	uid = item->att_data.att_static->att_data.att_uid;
	break;

      case MAILIMAP_MSG_ATT_BODY_SECTION:
	if (references == NULL) {
	  references = item->att_data.att_static->att_data.att_body_section->sec_body_part;
	  ref_size = item->att_data.att_static->att_data.att_body_section->sec_length;
	}
	break;
      }
      break;

    case MAILIMAP_MSG_ATT_ITEM_DYNAMIC:
      if (att_dyn == NULL) {
	att_dyn = item->att_data.att_dyn;
      }
      break;
    }
  }

  if (puid != NULL)
    * puid = uid;
  if (pimap_envelope != NULL)
    * pimap_envelope = imap_envelope;
  if (preferences != NULL)
    * preferences = references;
  if (pref_size != NULL)
    * pref_size = ref_size;
  if (patt_dyn != NULL)
    * patt_dyn = att_dyn;
  if (pimap_body != NULL)
    * pimap_body = imap_body;
}

static struct etpan_error *
fetch_result_to_db(struct etpan_imap_folder_sync * folder_sync,
    struct etpan_sqldb * db, clist * fetch_result,
    uint32_t * p_lastuid);

static struct etpan_error *
fetch_uid_list(struct etpan_imap_folder_sync * folder_sync,
    uint32_t lastuid, uint32_t * p_lastuid)
{
  struct mailimap_fetch_att * fetch_att;
  struct mailimap_fetch_type * fetch_type;
  struct mailimap_set * set;
  int r;
  struct etpan_error * error;
  clist * fetch_result;
  int had_error;
  
  error = select_folder(folder_sync);
  if (error != NULL)
    goto exit;
  
  fetch_type = mailimap_fetch_type_new_fetch_att_list_empty();
  if (fetch_type == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }

  fetch_att = mailimap_fetch_att_new_uid();
  if (fetch_att == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  r = mailimap_fetch_type_new_fetch_att_list_add(fetch_type, fetch_att);
  if (r != MAILIMAP_NO_ERROR)
    ETPAN_LOG_MEMORY_ERROR;
  
  set = mailimap_set_new_interval(lastuid + 1, 0);
  if (set == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  r = mailimap_uid_fetch(get_imap(folder_sync), set,
      fetch_type, &fetch_result);
  mailimap_set_free(set);
  mailimap_fetch_type_free(fetch_type);
  
  if (r == MAILIMAP_ERROR_MEMORY)
    ETPAN_LOG_MEMORY_ERROR;
  
  if (r == MAILIMAP_ERROR_STREAM) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_STREAM);
    etpan_error_set_short_description(error, _("Connection closed"));
    etpan_error_strf_long_description(error,
        _("An error occurred while synchronizing account %s.\n"
            "Connection was closed while getting message list of mailbox %s"),
         etpan_imap_sync_get_id(folder_sync->imap_sync),
        folder_sync->location);
    goto exit;
  }
  else if (r == MAILIMAP_ERROR_PARSE) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, _("Parse error"));
    etpan_error_strf_long_description(error,
        _("An error occurred while synchronizing account %s.\n"
            "The application did not understand what the server sent while getting message list of mailbox %s."), etpan_imap_sync_get_id(folder_sync->imap_sync),
        folder_sync->location);
    goto exit;
  }
  else if (r != MAILIMAP_NO_ERROR) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FETCH);
    etpan_error_set_short_description(error, _("Could not get information on the mailbox"));
    etpan_error_strf_long_description(error,
        _("An error occurred while synchronizing account %s.\n"
            "Message list of mailbox %s could not be retrieved from the server."), etpan_imap_sync_get_id(folder_sync->imap_sync),
        folder_sync->location);
    goto exit;
  }
  
  error = header_db_open(folder_sync);
  if (error != NULL) {
    goto free_fetch_result;
  }
  had_error = 0;
  r = etpan_sqldb_begin_transaction(folder_sync->header_db);
  if (r < 0) {
    had_error = 1;
  }
  error = fetch_result_to_db(folder_sync, folder_sync->header_db, fetch_result, p_lastuid);
  r = etpan_sqldb_end_transaction(folder_sync->header_db);
  if (r < 0) {
    had_error = 1;
  }
  header_db_close(folder_sync);
  if (error == NULL) {
    if (had_error) {
      error = etpan_error_new();
      etpan_error_set_code(error, ERROR_DATABASE);
      etpan_error_set_short_description(error, _("Cache could not be written"));
      etpan_error_strf_long_description(error,
          _("An error occurred while synchronizing account %s.\n"
          "Cache of mailbox %s could not be written properly on disk."),
          etpan_imap_sync_get_id(folder_sync->imap_sync),
          folder_sync->location);
    }
  }
  else {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(folder_sync, 0, ERROR_DOMAIN_SYNC,
        error);
    ETPAN_ERROR_FREE(error);
    error = new_error;
  }
  
  if (error != NULL) {
    goto free_fetch_result;
  }
  
  error = NULL;
  
 free_fetch_result:
  mailimap_fetch_list_free(fetch_result);
 exit:
  return error;
}

static struct etpan_error *
fetch_env_list(struct etpan_imap_folder_sync * folder_sync,
    struct mailimap_set * set, uint32_t * p_envlastuid)
{
  struct mailimap_fetch_att * fetch_att;
  struct mailimap_fetch_type * fetch_type;
  int r;
  struct etpan_error * error;
  clist * fetch_result;
  int had_error;
  
  error = select_folder(folder_sync);
  if (error != NULL)
    goto exit;
  
  fetch_type = mailimap_fetch_type_new_fetch_att_list_empty();
  if (fetch_type == NULL)
    ETPAN_LOG_MEMORY_ERROR;

  fetch_att = mailimap_fetch_att_new_uid();
  if (fetch_att == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  r = mailimap_fetch_type_new_fetch_att_list_add(fetch_type, fetch_att);
  if (r != MAILIMAP_NO_ERROR)
    ETPAN_LOG_MEMORY_ERROR;

  fetch_att = mailimap_fetch_att_new_flags();
  if (fetch_att == NULL)
    ETPAN_LOG_MEMORY_ERROR;

  r = mailimap_fetch_type_new_fetch_att_list_add(fetch_type, fetch_att);
  if (r != MAILIMAP_NO_ERROR)
    ETPAN_LOG_MEMORY_ERROR;
  
  imap_add_envelope_fetch_att(fetch_type);
  
  r = mailimap_uid_fetch(get_imap(folder_sync), set,
      fetch_type, &fetch_result);
  mailimap_fetch_type_free(fetch_type);

  if (r == MAILIMAP_ERROR_MEMORY)
    ETPAN_LOG_MEMORY_ERROR;
  
  if (r == MAILIMAP_ERROR_STREAM) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_STREAM);
    etpan_error_set_short_description(error, _("Connection closed"));
    etpan_error_strf_long_description(error,
        _("An error occurred while synchronizing account %s.\n"
            "Connection was closed while getting message list of mailbox %s"),
         etpan_imap_sync_get_id(folder_sync->imap_sync),
        folder_sync->location);
    goto exit;
  }
  else if (r == MAILIMAP_ERROR_PARSE) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, _("Parse error"));
    etpan_error_strf_long_description(error,
        _("An error occurred while synchronizing account %s.\n"
            "The application did not understand what the server sent while getting message list of mailbox %s."), etpan_imap_sync_get_id(folder_sync->imap_sync),
        folder_sync->location);
    goto exit;
  }
  else if (r != MAILIMAP_NO_ERROR) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FETCH);
    etpan_error_set_short_description(error, _("Could not get information on the mailbox"));
    etpan_error_strf_long_description(error,
        _("An error occurred while synchronizing account %s.\n"
            "Message list of mailbox %s could not be retrieved from the server."), etpan_imap_sync_get_id(folder_sync->imap_sync),
        folder_sync->location);
    goto exit;
  }

  error = header_db_open(folder_sync);
  if (error != NULL) {
    goto free_fetch_result;
  }
  had_error = 0;
  r = etpan_sqldb_begin_transaction(folder_sync->header_db);
  if (r < 0) {
    had_error = 1;
  }
  error = fetch_result_to_db(folder_sync, folder_sync->header_db, fetch_result, p_envlastuid);
  r = etpan_sqldb_end_transaction(folder_sync->header_db);
  if (r < 0) {
    had_error = 1;
  }
  header_db_close(folder_sync);
  if (error == NULL) {
    if (had_error) {
      error = etpan_error_new();
      etpan_error_set_code(error, ERROR_DATABASE);
      etpan_error_set_short_description(error, _("Cache could not be written"));
      etpan_error_strf_long_description(error,
          _("An error occurred while synchronizing account %s.\n"
          "Cache of mailbox %s could not be written properly on disk."),
          etpan_imap_sync_get_id(folder_sync->imap_sync),
          folder_sync->location);
    }
  }
  else {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(folder_sync, 0, ERROR_DOMAIN_SYNC,
        error);
    ETPAN_ERROR_FREE(error);
    error = new_error;
  }
  
  if (error != NULL) {
    goto free_fetch_result;
  }
  
  error = NULL;
  
 free_fetch_result:
  mailimap_fetch_list_free(fetch_result);
 exit:
  return error;
}

static struct etpan_error *
fetch_flags_list(struct etpan_imap_folder_sync * folder_sync,
    struct mailimap_set * set)
{
  struct mailimap_fetch_att * fetch_att;
  struct mailimap_fetch_type * fetch_type;
  int r;
  struct etpan_error * error;
  clist * fetch_result;
  int had_error;
  
  error = select_folder(folder_sync);
  if (error != NULL)
    goto exit;
  
  fetch_type = mailimap_fetch_type_new_fetch_att_list_empty();
  if (fetch_type == NULL)
    ETPAN_LOG_MEMORY_ERROR;

  fetch_att = mailimap_fetch_att_new_uid();
  if (fetch_att == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  r = mailimap_fetch_type_new_fetch_att_list_add(fetch_type, fetch_att);
  if (r != MAILIMAP_NO_ERROR)
    ETPAN_LOG_MEMORY_ERROR;

  fetch_att = mailimap_fetch_att_new_flags();
  if (fetch_att == NULL)
    ETPAN_LOG_MEMORY_ERROR;

  r = mailimap_fetch_type_new_fetch_att_list_add(fetch_type, fetch_att);
  if (r != MAILIMAP_NO_ERROR)
    ETPAN_LOG_MEMORY_ERROR;
  
  r = mailimap_uid_fetch(get_imap(folder_sync), set,
      fetch_type, &fetch_result);
  mailimap_fetch_type_free(fetch_type);

  if (r == MAILIMAP_ERROR_MEMORY)
    ETPAN_LOG_MEMORY_ERROR;
  
  if (r == MAILIMAP_ERROR_STREAM) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_STREAM);
    etpan_error_set_short_description(error, _("Connection closed"));
    etpan_error_strf_long_description(error,
        _("An error occurred while synchronizing account %s.\n"
            "Connection was closed while getting message list of mailbox %s"),
         etpan_imap_sync_get_id(folder_sync->imap_sync),
        folder_sync->location);
    goto exit;
  }
  else if (r == MAILIMAP_ERROR_PARSE) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, _("Parse error"));
    etpan_error_strf_long_description(error,
        _("An error occurred while synchronizing account %s.\n"
            "The application did not understand what the server sent while getting message list of mailbox %s."), etpan_imap_sync_get_id(folder_sync->imap_sync),
        folder_sync->location);
    goto exit;
  }
  else if (r != MAILIMAP_NO_ERROR) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FETCH);
    etpan_error_set_short_description(error, _("Could not get information on the mailbox"));
    etpan_error_strf_long_description(error,
        _("An error occurred while synchronizing account %s.\n"
            "Message list of mailbox %s could not be retrieved from the server."), etpan_imap_sync_get_id(folder_sync->imap_sync),
        folder_sync->location);
    goto exit;
  }

  error = header_db_open(folder_sync);
  if (error != NULL) {
    goto free_fetch_result;
  }
  had_error = 0;
  r = etpan_sqldb_begin_transaction(folder_sync->header_db);
  if (r < 0) {
    had_error = 1;
  }
  error = fetch_result_to_db(folder_sync, folder_sync->header_db, fetch_result, NULL);
  r = etpan_sqldb_end_transaction(folder_sync->header_db);
  if (r < 0) {
    had_error = 1;
  }
  header_db_close(folder_sync);
  if (error == NULL) {
    if (had_error) {
      error = etpan_error_new();
      etpan_error_set_code(error, ERROR_DATABASE);
      etpan_error_set_short_description(error, _("Cache could not be written"));
      etpan_error_strf_long_description(error,
          _("An error occurred while synchronizing account %s.\n"
          "Cache of mailbox %s could not be written properly on disk."),
          etpan_imap_sync_get_id(folder_sync->imap_sync),
          folder_sync->location);
    }
  }
  else {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(folder_sync, 0, ERROR_DOMAIN_SYNC,
        error);
    ETPAN_ERROR_FREE(error);
    error = new_error;
  }
  
  if (error != NULL) {
    goto free_fetch_result;
  }
  
  error = NULL;
  
 free_fetch_result:
  mailimap_fetch_list_free(fetch_result);
 exit:
  return error;
}

static void
update_count_with_flags(struct etpan_imap_folder_sync * folder_sync,
    struct etpan_message_flags * flags, int revert)
{
  int value;
  
  value = etpan_message_flags_get_value(flags);
  if ((value & ETPAN_FLAGS_DELETED) == 0) {
    if (revert)
      folder_sync->msg_count --;
    else
      folder_sync->msg_count ++;
    if ((value & ETPAN_FLAGS_SEEN) == 0) {
      if (revert)
        folder_sync->unseen_count --;
      else
        folder_sync->unseen_count ++;
      if ((value & ETPAN_FLAGS_NEW) != 0) {
        if (revert)
          folder_sync->recent_count --;
        else
          folder_sync->recent_count ++;
      }
    }
  }
}

static struct etpan_error *
fetch_result_to_db(struct etpan_imap_folder_sync * folder_sync,
    struct etpan_sqldb * db, clist * fetch_result,
    uint32_t * p_lastuid)
{
  clistiter * cur;
  int r;
  uint32_t lastuid;
  int had_error;
  
  had_error = 0;
  lastuid = 0;
  for(cur = clist_begin(fetch_result) ; cur != NULL ;
      cur = clist_next(cur)) {
    struct mailimap_msg_att * msg_att;
    uint32_t uid;
    struct mailimap_envelope * imap_envelope;
    struct mailimap_msg_att_dynamic * att_dyn;
    char * references;
    size_t ref_size;
    char uid_str[20];
    uint32_t presence;
    void * data;
    size_t length;
    
    msg_att = clist_content(cur);
    
    uid = 0;
    imap_get_msg_att_info(msg_att, &uid, &imap_envelope,
        &references, &ref_size, &att_dyn, NULL);
    if (uid == 0)
      continue;
    
    snprintf(uid_str, sizeof(uid_str), "%lu", (unsigned long) uid);

    if (uid > lastuid)
      lastuid = uid;
    
    presence = 1;
    encode_uint32(presence, &data, &length);
    r = etpan_sqldb_set(db, uid_str, "presence", data, length);
    free(data);
    if (r < 0) {
      had_error = 1;
    }
    
    if (imap_envelope != NULL) {
      void * data;
      size_t length;
      struct mailimf_fields * ep_fields;
      struct etpan_message_header * headers;
      
      ep_fields = etpan_imap_env_to_lep(imap_envelope,
          references, ref_size);
      
      headers = etpan_header_from_lep_header(ep_fields);
      mailimf_fields_free(ep_fields);
      
      etpan_message_header_serialize(headers, &data, &length);
      r = etpan_sqldb_set(db, uid_str, "headers", data, length);
      free(data);
      etpan_message_header_free(headers);
      
      if (r < 0) {
        had_error = 1;
      }
    }

    if (att_dyn != NULL) {
      void * data;
      size_t length;
      struct mail_flags * ep_flags;
      struct etpan_message_flags * flags;
      chashdatum key;
      chashdatum value;
      
      ep_flags = etpan_imap_flags_to_lep(att_dyn);
      flags = etpan_lep_flags_from_lep(ep_flags);
      mail_flags_free(ep_flags);
      
      etpan_message_flags_serialize(flags, &data, &length);
      r = etpan_sqldb_set(db, uid_str, "flags", data, length);
      free(data);
      
      update_count_with_flags(folder_sync, flags, 0);
      
      etpan_message_flags_free(flags);
      
      if (r < 0) {
        had_error = 1;
      }
      
      key.data = uid_str;
      key.len = strlen(uid_str) + 1;
      value.data = NULL;
      value.len = 0;
      r = chash_set(folder_sync->flags_updated_hash, &key, &value, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
    }
  }
  
  if (p_lastuid != NULL)
    * p_lastuid = lastuid;
  
  if (had_error) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_DATABASE);
    etpan_error_set_short_description(error, _("Cache could not be written"));
    etpan_error_strf_long_description(error,
        _("Cache of mailbox %s could not be written properly on disk."),
        folder_sync->location);
    
    return error;
  }
  
  return NULL;
}

struct etpan_imap_folder_sync *
etpan_imap_folder_sync_new(struct etpan_imap_sync * imap_sync,
    char * location, int noinferiors, char separator)
{
  struct etpan_imap_folder_sync * folder_sync;
#if 0
  struct mailstorage * ep_storage;
  struct mailfolder * ep_folder;
#endif
  int r;
  
  folder_sync = malloc(sizeof(* folder_sync));
  if (folder_sync == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  folder_sync->imap_sync = imap_sync;
  
  folder_sync->location = strdup(location);
  if (folder_sync->location == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
#if 0
  ep_storage = etpan_imap_sync_get_storage(imap_sync);
  ep_folder = mailfolder_new(ep_storage, location, NULL);
  if (ep_folder == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  folder_sync->ep_folder = ep_folder;
#endif
  folder_sync->header_db = NULL;
  folder_sync->body_db = NULL;
  pthread_mutex_init(&folder_sync->header_lock, NULL);
  pthread_mutex_init(&folder_sync->body_lock, NULL);
  folder_sync->noinferiors = noinferiors;
  folder_sync->separator = separator;
  folder_sync->state = STATE_DISCONNECTED;
  folder_sync->last_flags_sync_date = 0;
  folder_sync->env_lastuid = 0;
  folder_sync->last_sync_date = 0;
  folder_sync->fetch_groups = NULL;
  folder_sync->flag_groups = NULL;
  folder_sync->body_uid_list = NULL;
  folder_sync->body_uid_count = 0;
  folder_sync->uid_list = NULL;
  folder_sync->uid_count = 0;
  folder_sync->current_fetch_group_index = 0;
  folder_sync->current_flag_group_index = 0;
  folder_sync->current_body_index = 0;
  folder_sync->op = NULL;
  folder_sync->opened = 1;
  folder_sync->pending_flags = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (folder_sync->pending_flags == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  folder_sync->msg_count = 0;
  folder_sync->unseen_count = 0;
  folder_sync->recent_count = 0;
  r = pthread_mutex_init(&folder_sync->pending_flags_lock, NULL);
  if (r != 0) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  folder_sync->check_requested = 0;
  r = pthread_mutex_init(&folder_sync->check_requested_lock, NULL);
  if (r != 0) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  folder_sync->append_db = NULL;
  folder_sync->append_has_items = 1;
  folder_sync->append_queue = NULL;
  folder_sync->current_append_index = 0;
  r = pthread_mutex_init(&folder_sync->append_lock, NULL);
  if (r != 0) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  folder_sync->flags_updated_hash = NULL;
  folder_sync->skip_flags = 0;
  folder_sync->connected = 0;
  folder_sync->error = NULL;
  folder_sync->ref_count = 1;
  r = pthread_mutex_init(&folder_sync->ref_lock, NULL);
  if (r != 0) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  return folder_sync;
}

static void etpan_imap_folder_sync_free(struct etpan_imap_folder_sync * folder_sync)
{
  pthread_mutex_destroy(&folder_sync->ref_lock);
  pthread_mutex_destroy(&folder_sync->append_lock);
  pthread_mutex_destroy(&folder_sync->check_requested_lock);
  pthread_mutex_destroy(&folder_sync->pending_flags_lock);
  flush_pending_flags(folder_sync);
  chash_free(folder_sync->pending_flags);
  
  pthread_mutex_destroy(&folder_sync->body_lock);
  pthread_mutex_destroy(&folder_sync->header_lock);
#if 0
  if (folder_sync->ep_folder != NULL) {
    mailfolder_free(folder_sync->ep_folder);
  }
#endif
  free(folder_sync->location);
  free(folder_sync);
}

void etpan_imap_folder_sync_ref(struct etpan_imap_folder_sync * folder_sync)
{
  pthread_mutex_lock(&folder_sync->ref_lock);
  folder_sync->ref_count ++;
  pthread_mutex_unlock(&folder_sync->ref_lock);
}

void etpan_imap_folder_sync_unref(struct etpan_imap_folder_sync * folder_sync)
{
  int ref_count;
  
  pthread_mutex_lock(&folder_sync->ref_lock);
  folder_sync->ref_count --;
  ref_count = folder_sync->ref_count;
  pthread_mutex_unlock(&folder_sync->ref_lock);
  if (ref_count == 0) {
    etpan_imap_folder_sync_free(folder_sync);
  }
}

static struct etpan_error *
fetch_parts(struct etpan_imap_folder_sync * folder_sync,
    uint32_t uid);

static void get_body(clist * fetch_result,
    struct mailimap_body ** result, struct mailimap_envelope ** p_envelope);

static void recursive_mark(carray * parts_list,
    struct mailmime * mime);

static struct etpan_error *
imap_fetch_section(struct etpan_imap_folder_sync * folder_sync,
    uint32_t msg_uid,
    struct mailimap_section * section,
    char ** result, size_t * result_len);

static char * get_section_id_str(struct mailmime * mime);

static struct mailimap_section *
get_imap_section_from_mime(struct mailmime * mime,
    int section_type);

static char *
get_description(struct etpan_imap_folder_sync * folder_sync,
    uint32_t uid)
{
  struct etpan_error * error;
  int r;
  void * data;
  size_t length;
  struct etpan_message_header * header;
  char uid_str[40];
  char * description;
  
  error = header_db_open(folder_sync);
  if (error != NULL) {
    ETPAN_ERROR_FREE(error);
    goto err;
  }
  
  snprintf(uid_str, sizeof(uid_str), "%lu", (unsigned long) uid);
  r = etpan_sqldb_get(folder_sync->header_db, uid_str,
      "headers", &data, &length);
  if (r < 0) {
    goto close_db;
  }
  header = etpan_message_header_unserialize(data, length);
  free(data);
  header_db_close(folder_sync);
  
  description = etpan_message_compute_description_string(header);
  etpan_message_header_free(header);
  if (description == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return description;
  
 close_db:
  header_db_close(folder_sync);
 err:
  description = strdup(_("Unknown message"));
  return description;
}

static struct etpan_error *
fetch_parts(struct etpan_imap_folder_sync * folder_sync,
    uint32_t uid)
{
  mailimap * imap;
  struct mailimap_set * set;
  struct mailimap_body * body;
  struct mailimap_fetch_att * fetch_att;
  struct mailimap_fetch_type * fetch_type;
  int r;
  clist * fetch_result;
  void * data;
  size_t length;
  char part_id[256];
  struct mailmime * mime;
  unsigned int i;
  struct etpan_error * error;
  carray * parts_list;
  struct mailimap_envelope * envelope;
  char uid_str[20];
  struct mailimf_fields * fields;
  struct mailmime_content * content_message;
  struct mailmime * new_body;
  int had_error;
  
  snprintf(part_id, sizeof(part_id), "%lu-body", (unsigned long) uid);
  
  error = body_db_open(folder_sync);
  if (error != NULL) {
    goto err;
  }
  r = etpan_sqldb_get(folder_sync->body_db, part_id, "content", &data, &length);
  body_db_close(folder_sync);
  if (r == 0) {
    free(data);
    ETPAN_LOCAL_LOG("already fetched %lu", (unsigned long) uid);
    /* already fetched */
    error = NULL;
    goto err;
  }
  
  error = select_folder(folder_sync);
  if (error != NULL) {
    goto err;
  }
  
  snprintf(uid_str, sizeof(uid_str), "%lu", (unsigned long) uid);
  
  ETPAN_LOCAL_LOG("fetch body %lu", (unsigned long) uid);
  
  folder_sync->fetching_msg = 1;
  folder_sync->fetching_msg_uid = uid;
  
  imap = get_imap(folder_sync);
  
  set = mailimap_set_new_single(uid);
  if (set == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  fetch_type = mailimap_fetch_type_new_fetch_att_list_empty();
  if (fetch_type == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  fetch_att = mailimap_fetch_att_new_uid();
  if (fetch_att == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  r = mailimap_fetch_type_new_fetch_att_list_add(fetch_type, fetch_att);
  if (r != MAILIMAP_NO_ERROR)
    ETPAN_LOG_MEMORY_ERROR;
  
  fetch_att = mailimap_fetch_att_new_bodystructure();
  if (fetch_att == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  r = mailimap_fetch_type_new_fetch_att_list_add(fetch_type, fetch_att);
  if (r != MAILIMAP_NO_ERROR)
    ETPAN_LOG_MEMORY_ERROR;
  
  fetch_att = mailimap_fetch_att_new_envelope();
  if (fetch_att == NULL)
    ETPAN_LOG_MEMORY_ERROR;

  r = mailimap_fetch_type_new_fetch_att_list_add(fetch_type, fetch_att);
  if (r != MAILIMAP_NO_ERROR)
    ETPAN_LOG_MEMORY_ERROR;
  
  r = mailimap_uid_fetch(imap, set, fetch_type, &fetch_result);
  mailimap_fetch_type_free(fetch_type);
  mailimap_set_free(set);
  
  if (r == MAILIMAP_ERROR_MEMORY)
    ETPAN_LOG_MEMORY_ERROR;
  
  if (r == MAILIMAP_ERROR_STREAM) {
    char * description;
    
    description = get_description(folder_sync, uid);
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_STREAM);
    etpan_error_set_short_description(error, _("Connection closed"));
    etpan_error_strf_long_description(error,
        _("An error occurred while getting content of the following message:\n"
            "%s"
            "Connection was closed."),
        description);
    free(description);
    goto err;
  }
  else if (r == MAILIMAP_ERROR_PARSE) {
    char * description;
    
    description = get_description(folder_sync, uid);
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, _("Parse error"));
    etpan_error_strf_long_description(error,
        _("An error occurred while getting content of the following message:\n"
            "%s"
            "The application did not understand what the server sent."),
        description);
    free(description);
    goto err;
  }
  else if (r != MAILIMAP_NO_ERROR) {
    char * description;
    
    description = get_description(folder_sync, uid);
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FETCH);
    etpan_error_set_short_description(error, _("Could not get the content of a message"));
    etpan_error_strf_long_description(error,
        _("An error occurred while getting content of the following message:\n"
            "%s"
            "Message content could not be retrieved from the server."),
        description);
    free(description);
    goto err;
  }
  
  body = NULL;
  envelope = NULL;
  get_body(fetch_result, &body, &envelope);
  if ((body == NULL) || (envelope == NULL)) {
    char * description;
    
    if (body != NULL) {
      mailimap_body_free(body);
    }
    if (envelope != NULL) {
      mailimap_envelope_free(envelope);
    }
    
    description = get_description(folder_sync, uid);
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FETCH);
    etpan_error_set_short_description(error, _("Could not get the content of a message"));
    etpan_error_strf_long_description(error,
        _("An error occurred while getting content of the following message:\n"
            "%s"
            "Message content could not be retrieved from the server."),
        description);
    free(description);
    goto free_fetch_result;
  }
  
  /* write to DB */
  error = body_db_open(folder_sync);
  if (error != NULL) {
    mailimap_body_free(body);
    mailimap_envelope_free(envelope);
    goto free_fetch_result;
  }
  r = etpan_sqldb_begin_transaction(folder_sync->body_db);
  if (r < 0)
    had_error = 1;
  etpan_imap_body_serialize(body, &data, &length);
  snprintf(part_id, sizeof(part_id), "%lu-body", (unsigned long) uid);
  had_error = 0;
  r = etpan_sqldb_set(folder_sync->body_db, part_id, "uid",
      uid_str, strlen(uid_str));
  if (r < 0)
    had_error = 1;
  r = etpan_sqldb_set(folder_sync->body_db, part_id, "content", data, length);
  if (r < 0)
    had_error = 1;
  free(data);
  
  etpan_imap_envelope_serialize(envelope, &data, &length);
  snprintf(part_id, sizeof(part_id), "%lu-envelope", (unsigned long) uid);
  r = etpan_sqldb_set(folder_sync->body_db, part_id, "uid",
      uid_str, strlen(uid_str));
  if (r < 0)
    had_error = 1;
  r = etpan_sqldb_set(folder_sync->body_db, part_id, "content", data, length);
  if (r < 0)
    had_error = 1;
  free(data);
  r = etpan_sqldb_end_transaction(folder_sync->body_db);
  if (r < 0)
    had_error = 1;
  body_db_close(folder_sync);
  /* write to DB - done */
  
  /* add message part */
  mime = etpan_imap_body_to_lep(body);
  fields = etpan_imap_env_to_lep(envelope, NULL, 0);
  content_message = mailmime_get_content_message();
  if (content_message == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  new_body = mailmime_new(MAILMIME_MESSAGE, NULL,
      0, NULL, content_message,
      NULL, NULL, NULL, NULL, fields, mime);
  mime = new_body;
  
  parts_list = carray_new(16);
  if (parts_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  recursive_mark(parts_list, mime);
  for(i = 0 ; i < carray_count(parts_list) ; i ++) {
    struct mailmime * mime_to_fetch;
    char * part_content;
    size_t part_content_len;
    char * section_id_str;
    struct mailimap_section * section;
    char * suffix;
    
    part_content = NULL;
    part_content_len = 0;
    section_id_str = NULL;
    * part_id = '\0';
    mime_to_fetch = carray_get(parts_list, i);
    section = NULL;
    suffix = NULL;
    if (mime_to_fetch->mm_type == MAILMIME_MESSAGE) {
      section = get_imap_section_from_mime(mime_to_fetch, EP_SECTION_HEADER);
      suffix = "-header";
    }
    else if (mime_to_fetch->mm_type == MAILMIME_SINGLE) {
      section = get_imap_section_from_mime(mime_to_fetch, EP_SECTION_MESSAGE);
      suffix = "";
    }
    else {
      ETPAN_CRASH("trying to cache multipart");
    }
    error = imap_fetch_section(folder_sync, uid, section,
        &part_content, &part_content_len);
    if (error != NULL) {
      goto free_parts_list;
    }
    
    /* write to DB */
    error = body_db_open(folder_sync);
    if (error != NULL) {
      mailimap_body_free(body);
      mailimap_envelope_free(envelope);
      goto free_parts_list;
    }
    r = etpan_sqldb_begin_transaction(folder_sync->body_db);
    if (r < 0)
      had_error = 1;
    
    section_id_str = get_section_id_str(mime_to_fetch);
    
    snprintf(part_id, sizeof(part_id), "%lu-%s%s",
        (unsigned long) uid, section_id_str, suffix);
    
    ETPAN_LOCAL_LOG("caching %s", part_id);
    r = etpan_sqldb_set(folder_sync->body_db, part_id, "uid",
        uid_str, strlen(uid_str));
    if (r < 0)
      had_error = 1;
    encode_str(part_content, part_content_len, &data, &length);
    r = etpan_sqldb_set(folder_sync->body_db, part_id, "content",
        data, length);
    free(data);
    free(part_content);
    if (r < 0)
      had_error = 1;
    
    r = etpan_sqldb_end_transaction(folder_sync->body_db);
    if (r < 0)
      had_error = 1;
    body_db_close(folder_sync);
    /* write to DB - done */
    
    free(section_id_str);
  }
  
  if (had_error) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_DATABASE);
    etpan_error_set_short_description(error, _("Cache could not be written"));
    etpan_error_strf_long_description(error,
        _("Cache of mailbox %s could not be written properly on disk."),
        folder_sync->location);
    goto free_parts_list;
  }
  
  error = NULL;
  
 free_parts_list:
  carray_free(parts_list);
  mailmime_free(mime);
 free_fetch_result:
  mailimap_fetch_list_free(fetch_result);
 err:
  return error;
}

static char * get_section_id_str(struct mailmime * mime)
{
  int r;
  struct mailmime_section * part;
  char * str;
  char part_id_str[256];
  char * p;
  clistiter * cur;
  size_t remaining;
  int first;
  
  r = mailmime_get_section_id(mime, &part);
  if (r != MAILIMF_NO_ERROR)
    ETPAN_LOG_MEMORY_ERROR;
  
  first = 1;
  p = part_id_str;
  * p = '\0';
  remaining = sizeof(part_id_str);
  for(cur = clist_begin(part->sec_list) ; cur != NULL ;
      cur = clist_next(cur)) {
    uint32_t * p_num;
    int count;
    
    p_num = clist_content(cur);    
    if (first) {
      count = snprintf(p, remaining, "%u", * p_num);
      first = 0;
    }
    else {
      count = snprintf(p, remaining, ".%u", * p_num);
    }
    
    if ((unsigned int) count > remaining)
      ETPAN_LOG_MEMORY_ERROR;
    
    remaining -= count;
    first = 0;
    
    p += count;
  }
  
  str = strdup(part_id_str);
  if (str == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  mailmime_section_free(part);
  
  return str;
}

static void get_body(clist * fetch_result,
    struct mailimap_body ** result, struct mailimap_envelope ** p_envelope)
{
  clistiter * cur;
  struct mailimap_body * body;
  struct mailimap_envelope * envelope;
  
  body = NULL;
  envelope = NULL;
  
  for(cur = clist_begin(fetch_result) ; cur != NULL ;
      cur = clist_next(cur)) {
    struct mailimap_msg_att * msg_att;
    uint32_t uid;
    struct mailimap_body * current_body;
    struct mailimap_envelope * current_envelope;
    
    msg_att = clist_content(cur);
    
    uid = 0;
    imap_get_msg_att_info(msg_att, &uid, &current_envelope,
        NULL, NULL, NULL, &current_body);
    if (uid == 0)
      continue;
    
    if (body == NULL)
      body = current_body;
    
    if (envelope == NULL)
      envelope = current_envelope;
  }
  
  * result = body;
  * p_envelope = envelope;
}

static void recursive_mark_single_part(carray * parts_list,
    struct mailmime * mime);
static void recursive_mark_multipart(carray * parts_list,
    struct mailmime * mime);
static void recursive_mark_message(carray * parts_list,
    struct mailmime * mime);

static void recursive_mark(carray * parts_list,
    struct mailmime * mime)
{
  switch (mime->mm_type) {
  case MAILMIME_SINGLE:
    recursive_mark_single_part(parts_list, mime);
    break;
  case MAILMIME_MULTIPLE:
    recursive_mark_multipart(parts_list, mime);
    break;
  case MAILMIME_MESSAGE:
    recursive_mark_message(parts_list, mime);
    break;
  }
}

static void recursive_mark_single_part(carray * parts_list,
    struct mailmime * mime)
{
  char * content_type;
  int fetch;
  int r;
  
  fetch = 0;
  content_type = etpan_get_content_type_str(mime->mm_content_type);
  
  if ((strcasecmp(content_type, "text/plain") == 0) ||
      (strcasecmp(content_type, "text/html") == 0) ||
      (strcasecmp(content_type, "text/enriched") == 0))
    fetch = 1;
  
  free(content_type);
  
  if (!fetch)
    return;
  
  r = carray_add(parts_list, mime, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

static void recursive_mark_multipart(carray * parts_list,
    struct mailmime * mime)
{
  clistiter * iter;
  
  for(iter = clist_begin(mime->mm_data.mm_multipart.mm_mp_list) ; iter != NULL ;
      iter = clist_next(iter)) {
    struct mailmime * submime;
    
    submime = clist_content(iter);
    recursive_mark(parts_list, submime);
  }
}

static void recursive_mark_message(carray * parts_list,
    struct mailmime * mime)
{
  int r;
  
  r = carray_add(parts_list, mime, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  recursive_mark(parts_list, mime->mm_data.mm_message.mm_msg_mime);
}




static struct mailimap_section *
section_to_imap_section(struct mailmime_section * section, int type)
{
  struct mailimap_section_part * section_part;
  struct mailimap_section * imap_section;
  clist * list;
  clistiter * cur;
  int r;
  
  list = clist_new();
  if (list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  for(cur = clist_begin(section->sec_list) ; cur != NULL ;
      cur = clist_next(cur)) {
    uint32_t value;
    uint32_t * id;

    value = * (uint32_t *) clist_content(cur);
    id = malloc(sizeof(* id));
    if (id == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    
    * id = value;
    r  = clist_append(list, id);
    if (r != 0)
      ETPAN_LOG_MEMORY_ERROR;
  }

  section_part = mailimap_section_part_new(list);
  if (section_part == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  imap_section = NULL;

  switch (type) {
  case EP_SECTION_MESSAGE:
    imap_section = mailimap_section_new_part(section_part);
    break;
  case EP_SECTION_HEADER:
    imap_section = mailimap_section_new_part_header(section_part);
    break;
  case EP_SECTION_MIME:
    imap_section = mailimap_section_new_part_mime(section_part);
    break;
  case EP_SECTION_BODY:
    imap_section = mailimap_section_new_part_text(section_part);
    break;
  }

  if (imap_section == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  return imap_section;
}

static struct etpan_error *
fetch_imap(struct etpan_imap_folder_sync * folder_sync,
    mailimap * imap, uint32_t msg_uid,
    struct mailimap_fetch_type * fetch_type,
    char ** result, size_t * result_len)
{
  int r;
  struct mailimap_msg_att * msg_att;
  struct mailimap_msg_att_item * msg_att_item;
  clist * fetch_result;
  struct mailimap_set * set;
  char * text;
  size_t text_length;
  clistiter * cur;
  struct etpan_error * error;

  set = mailimap_set_new_single(msg_uid);
  if (set == NULL)
    ETPAN_LOG_MEMORY_ERROR;

  r = mailimap_uid_fetch(imap, set,
      fetch_type, &fetch_result);
  
  mailimap_set_free(set);

  if (r == MAILIMAP_ERROR_MEMORY)
    ETPAN_LOG_MEMORY_ERROR;
  
  if (r == MAILIMAP_ERROR_STREAM) {
    char * description;
    
    description = get_description(folder_sync, msg_uid);
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_STREAM);
    etpan_error_set_short_description(error, _("Connection closed"));
    etpan_error_strf_long_description(error,
        _("An error occurred while getting content of the following message:\n"
            "%s"
            "Connection was closed."),
        description);
    free(description);
    goto err;
  }
  else if (r == MAILIMAP_ERROR_PARSE) {
    char * description;
    
    description = get_description(folder_sync, msg_uid);
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, _("Parse error"));
    etpan_error_strf_long_description(error,
        _("An error occurred while getting content of the following message:\n"
            "%s"
            "The application did not understand what the server sent."),
        description);
    free(description);
    goto err;
  }
  else if (r != MAILIMAP_NO_ERROR) {
    char * description;
    
    description = get_description(folder_sync, msg_uid);
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FETCH);
    etpan_error_set_short_description(error, _("Could not get the content of a message"));
    etpan_error_strf_long_description(error,
        _("An error occurred while getting content of the following message:\n"
            "%s"
            "Message content could not be retrieved from the server."),
        description);
    free(description);
    goto err;
  }

  if (clist_begin(fetch_result) == NULL) {
    char * description;
    
    description = get_description(folder_sync, msg_uid);
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FETCH);
    etpan_error_set_short_description(error, _("Could not get the content of a message"));
    etpan_error_strf_long_description(error,
        _("An error occurred while getting content of the following message:\n"
            "%s"
            "Message content could not be retrieved from the server."),
        description);
    free(description);
    goto free_fetch_result;
  }

  msg_att = clist_begin(fetch_result)->data;

  text = NULL;
  text_length = 0;

  for(cur = clist_begin(msg_att->att_list) ; cur != NULL ;
      cur = clist_next(cur)) {
    msg_att_item = clist_content(cur);

    if (msg_att_item->att_type == MAILIMAP_MSG_ATT_ITEM_STATIC) {
      if (msg_att_item->att_data.att_static->att_type ==
	  MAILIMAP_MSG_ATT_BODY_SECTION) {
        if (msg_att_item->att_data.att_static->att_data.att_body_section->sec_body_part != NULL) {
          text = strdup(msg_att_item->att_data.att_static->att_data.att_body_section->sec_body_part);
          if (text == NULL)
            ETPAN_LOG_MEMORY_ERROR;
          text_length =
            msg_att_item->att_data.att_static->att_data.att_body_section->sec_length;
          break;
        }
      }
    }
  }

  if (text == NULL) {
    char * description;
    
    description = get_description(folder_sync, msg_uid);
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FETCH);
    etpan_error_set_short_description(error, _("Could not get the content of a message"));
    etpan_error_strf_long_description(error,
        _("An error occurred while getting content of the following message:\n"
            "%s"
            "Message content could not be retrieved from the server."),
        description);
    free(description);
    goto free_fetch_result;
  }
  
  * result = text;
  * result_len = text_length;
  
  error = NULL;
  
 free_fetch_result:
  mailimap_fetch_list_free(fetch_result);
 err:
  return error;
}

static struct mailimap_section *
get_imap_section_from_mime(struct mailmime * mime,
    int section_type)
{
  struct mailimap_section * section;
  struct mailmime_section * part;
  int r;
  
  if (mime->mm_parent == NULL) {
    section = NULL;
    switch (section_type) {
    case EP_SECTION_MESSAGE:
      section = mailimap_section_new(NULL);
      break;
    case EP_SECTION_HEADER:
      section = mailimap_section_new_header();
      break;
    case EP_SECTION_MIME:
      ETPAN_CRASH("main part has no MIME headers");
      break;
    case EP_SECTION_BODY:
      section = mailimap_section_new_text();
      break;
    }
    
    if (section == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  else {
    if (section_type == EP_SECTION_MIME) {
      if (mime->mm_parent->mm_parent == NULL) {
        ETPAN_CRASH("main part has no MIME headers");
      }
    }
    
    r = mailmime_get_section_id(mime, &part);
    if (r != MAILIMF_NO_ERROR)
      ETPAN_LOG_MEMORY_ERROR;
    
    section = section_to_imap_section(part, section_type);
    mailmime_section_free(part);
  }
  
  return section;
}

static struct mailimap_section *
get_imap_section_from_string(char * section_str, int part_type)
{
  struct mailimap_section * section;
  char * p;
  char * current_section;
  struct mailimap_section_part * section_part;
  clist * list;
  int r;
  char * dup_section_str;
  
  list = clist_new();
  if (list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  dup_section_str = strdup(section_str);
  if (dup_section_str == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  current_section = dup_section_str;
  if (* current_section != '\0') {
    while (current_section != NULL) {
      uint32_t value;
      uint32_t * p_value;
    
      p = strchr(current_section, '.');
      if (p != NULL)
        * p = '\0';
      value = strtoul(current_section, NULL, 10);
      p_value = malloc(sizeof(* p_value));
      if (p_value == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      * p_value = value;
      r = clist_append(list, p_value);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
    
      if (p != NULL)
        current_section = p + 1;
      else
        current_section = NULL;
    }
  }
  free(dup_section_str);
  
  section_part = mailimap_section_part_new(list);
  if (section_part == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  switch (part_type) {
  case EP_SECTION_MESSAGE:
    section = mailimap_section_new_part(section_part);
    break;
  case EP_SECTION_HEADER:
    section = mailimap_section_new_part_header(section_part);
    break;
  case EP_SECTION_MIME:
    section = mailimap_section_new_part_mime(section_part);
    break;
  case EP_SECTION_BODY:
    section = mailimap_section_new_part_text(section_part);
    break;
  default:
    section = NULL;
    ETPAN_CRASH("invalid part type");
    break;
  }
  if (section == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return section;
}

static struct etpan_error *
imap_fetch_section(struct etpan_imap_folder_sync * folder_sync,
    uint32_t msg_uid,
    struct mailimap_section * section,
    char ** result, size_t * result_len)
{
  struct mailimap_fetch_att * fetch_att;
  struct mailimap_fetch_type * fetch_type;
  char * text;
  size_t text_length;
  struct etpan_error * error;
  
  error = select_folder(folder_sync);
  if (error != NULL) {
    return error;
  }
  
  fetch_att = mailimap_fetch_att_new_body_peek_section(section);
  if (fetch_att == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  fetch_type = mailimap_fetch_type_new_fetch_att(fetch_att);
  if (fetch_type == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  text_length = 0;
  text = NULL;
  error = fetch_imap(folder_sync,
      get_imap(folder_sync), msg_uid, fetch_type, &text, &text_length);
  ETPAN_LOCAL_LOG("fetch imap : %p %u -> %p", text, (unsigned int) text_length, error);
  
  mailimap_fetch_type_free(fetch_type);
  
  if (error != NULL)
    return error;
  
  * result = text;
  * result_len = text_length;
  
  return NULL;
}

#if 0
static struct etpan_error *
connect(struct etpan_imap_folder_sync * folder_sync)
{
  int r;
  struct etpan_imap_sync * imap_sync;
  struct etpan_error * error;
  struct mailstorage * ep_storage;
  struct mailfolder * ep_folder;
  static mailimap * imap;
  uint32_t uidvalidity;
  uint32_t current_uidvalidity;
  carray * keys;
  unsigned int i;
  
  if (folder_sync->connected) {
    return NULL;
  }
  
  imap_sync = folder_sync->imap_sync;
  if (!imap_sync->connected) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CONNECT);
    etpan_error_set_short_description(error, _("Not connected"));
    etpan_error_strf_long_description(error, "TODO");
    
    return error;
  }
  
  ep_storage = etpan_imap_sync_get_storage(imap_sync);
  if (ep_storage == NULL)
    ETPAN_CRASH("imap sync not connected yet");
  
  ep_folder = mailfolder_new(ep_storage, folder_sync->location, NULL);
  if (ep_folder == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  folder_sync->ep_folder = ep_folder;
  
  r = mailfolder_connect(ep_folder);
  if (r == MAIL_ERROR_MEMORY)
    ETPAN_LOG_MEMORY_ERROR;
  
  if (r == MAIL_ERROR_CONNECT) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CONNECT);
    etpan_error_set_short_description(error, _("Could not connect"));
    etpan_error_strf_long_description(error,
        _("Connection to account %s failed. "
            "The connection to IMAP server %s, port %i could not be established.\n"
            "Check the name of server and port."),
        etpan_imap_sync_get_id(imap_sync),
        etpan_imap_sync_get_hostname(imap_sync),
        etpan_imap_sync_get_port(imap_sync));
    goto free;
  }
  else if (r == MAIL_ERROR_LOGIN) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_AUTH);
    etpan_error_set_short_description(error, _("Authentication failed"));
    etpan_error_strf_long_description(error,
        _("Connection to account %s failed. "
            "Check your login, password and authentication type."),
        etpan_imap_sync_get_id(imap_sync));
    goto free;
  }
  else if (r == MAIL_ERROR_PARSE) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, _("Parse error"));
    etpan_error_strf_long_description(error, _("Connection to account %s failed. The application did not understand what the server sent."),
        etpan_imap_sync_get_id(imap_sync));
    goto free;
  }
  else if (r == MAIL_ERROR_STREAM) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_STREAM);
    etpan_error_set_short_description(error, _("Connection closed"));
    etpan_error_strf_long_description(error,
        _("Connection to account %s failed. "
            "The connection closed unexpectedly."),
        etpan_imap_sync_get_id(imap_sync));
    goto free;
  }
  else if (r != MAIL_NO_ERROR) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CONNECT);
    etpan_error_set_short_description(error, _("Unexpected error"));
    etpan_error_strf_long_description(error,
        _("Detailed error|Connection to account %s failed. "
            "An unexpected error (libetpan: %i) occurred."),
        etpan_imap_sync_get_id(imap_sync), r);
    goto free;
  }
  
  imap = get_imap(folder_sync);
  uidvalidity = imap->imap_selection_info->sel_uidvalidity;
  current_uidvalidity = etpan_imap_sync_get_uidvalidity(folder_sync->imap_sync,
      folder_sync->location);
  if (current_uidvalidity == 0) {
    error = etpan_imap_sync_set_uidvalidity(folder_sync->imap_sync,
        folder_sync->location, uidvalidity);
    if (error != NULL)
      goto disconnect;
  }
  else if (current_uidvalidity != uidvalidity) {
    int had_error;
    
    ETPAN_LOG("reset cache");
    had_error = 0;
    
    error = header_db_open(folder_sync);
    if (error != NULL) {
      goto disconnect;
    }
    r = etpan_sqldb_reset(folder_sync->header_db);
    if (r < 0)
      had_error = 1;
    header_db_close(folder_sync);
    
    error = body_db_open(folder_sync);
    if (error != NULL) {
      goto disconnect;
    }
    r = etpan_sqldb_reset(folder_sync->body_db);
    if (r < 0)
      had_error = 1;
    body_db_close(folder_sync);
    
    if (had_error) {
      struct etpan_error * error;
      
      error = etpan_error_new();
      etpan_error_set_code(error, ERROR_DATABASE);
      etpan_error_set_short_description(error, _("Cache could not be written"));
      etpan_error_strf_long_description(error,
          _("Cache of mailbox %s could not be written properly on disk."),
          folder_sync->location);
      goto disconnect;
    }
    
    error = etpan_imap_sync_set_uidvalidity(folder_sync->imap_sync,
        folder_sync->location, uidvalidity);
    if (error != NULL)
      goto disconnect;
  }
  error = append_db_open(folder_sync);
  if (error != NULL) {
    goto disconnect;
  }
  keys = etpan_sqldb_get_keys(folder_sync->append_db);
  if (carray_count(keys) == 0) {
    pthread_mutex_lock(&folder_sync->append_lock);
    folder_sync->append_has_items = 0;
    pthread_mutex_unlock(&folder_sync->append_lock);
  }
  etpan_sqldb_keys_free(keys);
  append_db_close(folder_sync);
  
  folder_sync->connected = 1;
  error = NULL;
  goto exit;
  
 disconnect:
  mailfolder_disconnect(ep_folder);
 free:
  mailfolder_free(ep_folder);
  folder_sync->ep_folder = NULL;
 exit:
  return error;
}
#endif

static void notify_folder_content_updated(struct etpan_imap_folder_sync * folder_sync);
static void notify_op_end(struct etpan_imap_folder_sync * folder_sync);

struct connect_result {
  struct etpan_error * error;
};

static void sync_state_init(struct etpan_imap_folder_sync * folder_sync);
static void sync_state_done(struct etpan_imap_folder_sync * folder_sync);


#if 0
static void threaded_connect(struct etpan_thread_op * op);
static void threaded_connect_cleanup(struct etpan_thread_op * op);
static void threaded_connect_callback(int cancelled,
    struct connect_result * result,
    void * data);


static void connect_run_op(struct etpan_thread_manager_app * manager,
    struct etpan_imap_folder_sync * folder_sync)
{
  struct etpan_thread_op * op;
  struct etpan_storage * storage;
  struct etpan_imap_sync * imap_sync;
  struct connect_result * result;
  
  op = etpan_thread_op_new();
  op->param = folder_sync;
  result = malloc(sizeof(* result));
  if (result == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  result->error = NULL;
  op->result = result;
  
  op->cancellable = 1;
  op->run = threaded_connect;
  op->callback = (void (*)(int, void *, void *)) threaded_connect_callback;
  op->callback_data = folder_sync;
  op->cleanup = threaded_connect_cleanup;
  
  imap_sync = folder_sync->imap_sync;
  storage = etpan_imap_sync_get_thread_storage(imap_sync);
  etpan_thread_manager_app_storage_schedule(manager, storage, op);
}

static void threaded_connect(struct etpan_thread_op * op)
{
  struct etpan_error * error;
  struct connect_result * result;
  struct etpan_imap_folder_sync * folder_sync;
  
  result = op->result;
  folder_sync = op->param;
  
  error = connect(folder_sync);
  result->error = error;
  if (error == NULL) {
    struct etpan_imap_sync * imap_sync;
    
    imap_sync = folder_sync->imap_sync;
    imap_sync->connected_count ++;
  }
}

static void threaded_connect_callback(int cancelled,
    struct connect_result * result,
    void * data)
{
  struct etpan_imap_folder_sync * folder_sync;
  
  folder_sync = data;
  
  if (result->error != NULL) {
    folder_sync->error = result->error;
    result->error = NULL;
    folder_sync->state = STATE_DISCONNECTED;
    folder_sync->last_sync_date = etpan_get_time();
    sync_state_done(folder_sync);
  }
  else {
    folder_sync->state = STATE_CONNECTED;
  }
  notify_op_end(folder_sync);
}

static void threaded_connect_cleanup(struct etpan_thread_op * op)
{
  struct connect_result * result;
  
  result = op->result;
  ETPAN_ERROR_FREE(result->error);
  free(result);
}
#endif


struct uid_list_result {
  carray * fetch_groups;
  carray * flag_groups;
  uint32_t * body_uid_list;
  unsigned int body_uid_count;
  uint32_t * uid_list;
  unsigned int uid_count;
  struct etpan_error * error;
};

static void threaded_get_uid_list(struct etpan_thread_op * op);
static void threaded_get_uid_list_cleanup(struct etpan_thread_op * op);
static void threaded_get_uid_list_callback(int cancelled, struct uid_list_result * result, void * data);

static void get_uid_list_run_op(struct etpan_thread_manager_app * manager,
    struct etpan_imap_folder_sync * folder_sync)
{
  struct etpan_thread_op * op;
  struct etpan_storage * storage;
  struct etpan_imap_sync * imap_sync;
  struct uid_list_result * result;
    
  if ((folder_sync->fetch_groups != NULL) || (folder_sync->flag_groups != NULL))
    ETPAN_CRASH("fetch groups should be NULL");
    
  ETPAN_LOCAL_LOG("get uid list");
  result = malloc(sizeof(* result));
  if (result == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  result->error = NULL;
  result->fetch_groups = NULL;
  result->flag_groups = NULL;
  result->body_uid_list = NULL;
  result->body_uid_count = 0;
  result->uid_list = NULL;
  result->uid_count = 0;
  
  op = etpan_thread_op_new();
  op->param = folder_sync;
  op->result = result;
    
  op->cancellable = 1;
  op->run = threaded_get_uid_list;
  op->callback = (void (*)(int, void *, void *)) threaded_get_uid_list_callback;
  op->callback_data = folder_sync;
  op->cleanup = threaded_get_uid_list_cleanup;
    
  imap_sync = folder_sync->imap_sync;
  storage = etpan_imap_sync_get_thread_storage(imap_sync);
  etpan_thread_manager_app_storage_schedule(manager, storage, op);
}

static void threaded_get_uid_list(struct etpan_thread_op * op)
{
  uint32_t lastuid;
  uint32_t next_lastuid;
  struct etpan_imap_sync * imap_sync;
  struct etpan_imap_folder_sync * folder_sync;
  struct etpan_error * error;
  uint32_t * uid_list;
  unsigned uid_count;
  carray * fetch_groups;
  carray * flag_groups;
  uint32_t * body_uid_list;
  unsigned int body_uid_count;
  uint32_t env_lastuid;
  uint32_t body_lastuid;
  struct uid_list_result * result;
    
  folder_sync = op->param;
  imap_sync = folder_sync->imap_sync;
  result = op->result;
  
  error = header_db_open(folder_sync);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(folder_sync, 0, ERROR_DOMAIN_SYNC,
        error);
    ETPAN_ERROR_FREE(error);
    error = new_error;
    
    goto exit;
  }
  lastuid = etpan_imap_sync_get_lastuid(imap_sync, folder_sync->location);
  header_db_close(folder_sync);
  
  error = fetch_uid_list(folder_sync,
      lastuid, &next_lastuid);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(folder_sync, 0, ERROR_DOMAIN_SYNC,
        error);
    ETPAN_ERROR_FREE(error);
    error = new_error;
    
    goto exit;
  }
  
  error = header_db_open(folder_sync);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(folder_sync, 0, ERROR_DOMAIN_SYNC,
        error);
    ETPAN_ERROR_FREE(error);
    error = new_error;
    
    goto exit;
  }
  error = etpan_imap_sync_set_lastuid(imap_sync, folder_sync->location,
      next_lastuid);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    header_db_close(folder_sync);
    new_error = rewrite_error(folder_sync, 0, ERROR_DOMAIN_SYNC,
        error);
    ETPAN_ERROR_FREE(error);
    error = new_error;
    
    goto exit;
  }
  
  /* split uid list to n msg per request */
  
  /* split env */
  env_lastuid = etpan_imap_sync_get_envlastuid(imap_sync,
      folder_sync->location);
  folder_sync->env_lastuid = env_lastuid;
  
  get_uid_list(folder_sync->header_db, &uid_list, &uid_count);
  fetch_groups = get_groups(uid_list, uid_count, env_lastuid + 1, 0,
      MSG_ENV_PER_GROUP);
  
  /* split flags */
  if (env_lastuid != 0) {
    flag_groups = get_groups(uid_list, uid_count, 0, env_lastuid,
        MSG_FLAG_PER_GROUP);
  }
  else {
    ETPAN_LOCAL_LOG("don't need to fetch flags");
    flag_groups = carray_new(4);
  }
  
  /* split body */
  body_lastuid = etpan_imap_sync_get_bodylastuid(imap_sync,
      folder_sync->location);
    
  body_uid_count = 0;
  body_uid_list = NULL;
  if (uid_count > 0) {
    unsigned int index;
    unsigned int left;
    unsigned int right;
        
    find_elt(uid_list, uid_count, body_lastuid + 1, 0, uid_count - 1, &left, &right);
        
    if (uid_list[right] >= body_lastuid + 1) {
      index = right;
            
      if (uid_list[left] >= body_lastuid + 1) {
        index = left;
      }
            
      body_uid_count = uid_count - index;
      body_uid_list = malloc(sizeof(* body_uid_list) * body_uid_count);
      if (body_uid_list == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      memcpy(body_uid_list, uid_list + index, sizeof(* body_uid_list) * body_uid_count);
    }
  }
  header_db_close(folder_sync);
  
  result->fetch_groups = fetch_groups;
  result->flag_groups = flag_groups;
  result->body_uid_list = body_uid_list;
  result->body_uid_count = body_uid_count;
  result->uid_list = uid_list;
  result->uid_count = uid_count;
  error = NULL;
  
 exit:
  result->error = error;
}

static void threaded_get_uid_list_callback(int cancelled, struct uid_list_result * result, void * data)
{
  struct etpan_imap_folder_sync * folder_sync;
    
  folder_sync = data;
    
  folder_sync->state = STATE_HAS_UIDLIST;
    
  folder_sync->fetch_groups = result->fetch_groups;
  folder_sync->flag_groups = result->flag_groups;
  folder_sync->body_uid_list = result->body_uid_list;
  folder_sync->body_uid_count = result->body_uid_count;
  folder_sync->uid_list = result->uid_list;
  folder_sync->uid_count = result->uid_count;
  folder_sync->error = result->error;
  result->fetch_groups = NULL;
  result->flag_groups = NULL;
  result->body_uid_list = NULL;
  result->body_uid_count = 0;
  result->uid_list = NULL;
  result->uid_count = 0;
  result->error = NULL;
  
  folder_sync->current_fetch_group_index = 0;
  folder_sync->current_flag_group_index = 0;
  folder_sync->current_body_index = 0;

  if (folder_sync->error != NULL) {
    if (etpan_error_get_code(folder_sync->error) == ERROR_STREAM) {
      folder_sync->state = STATE_STREAM_CLOSED;
      sync_state_done(folder_sync);
    }
  }
  
  notify_op_end(folder_sync);
}

static void threaded_get_uid_list_cleanup(struct etpan_thread_op * op)
{
  struct uid_list_result * result;
  unsigned int i;
    
  result = op->result;
    
  if (result->fetch_groups != NULL) {
    for(i = 0 ; i < carray_count(result->fetch_groups) ; i ++) {
      struct mailimap_set * set;
            
      set = carray_get(result->fetch_groups, i);
      mailimap_set_free(set);
    }
    carray_free(result->fetch_groups);
  }
  if (result->flag_groups != NULL) {
    for(i = 0 ; i < carray_count(result->flag_groups) ; i ++) {
      struct mailimap_set * set;
            
      set = carray_get(result->flag_groups, i);
      mailimap_set_free(set);
    }
    carray_free(result->flag_groups);
  }
  if (result->body_uid_list != NULL) {
    free(result->body_uid_list);
  }
  if (result->uid_list != NULL) {
    free(result->uid_list);
  }
  ETPAN_ERROR_FREE(result->error);
  free(result);
}


struct env_list_result {
  struct etpan_error * error;
  int updated;
};

static void threaded_get_env_list(struct etpan_thread_op * op);
static void threaded_get_env_list_callback(int cancelled,
    struct env_list_result * result, void * data);
static void threaded_get_env_list_cleanup(struct etpan_thread_op * op);

static void get_env_list_run_op(struct etpan_thread_manager_app * manager,
    struct etpan_imap_folder_sync * folder_sync)
{
  struct etpan_thread_op * op;
  struct etpan_storage * storage;
  struct etpan_imap_sync * imap_sync;
  struct env_list_result * result;  
  
  result = malloc(sizeof(* result));
  if (result == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  result->error = NULL;
  result->updated = 0;
  
  op = etpan_thread_op_new();
  op->param = folder_sync;
  op->result = result;
  
  op->cancellable = 1;
  op->run = threaded_get_env_list;
  op->callback = (void (*)(int, void *, void *)) threaded_get_env_list_callback;
  op->callback_data = folder_sync;
  op->cleanup = threaded_get_env_list_cleanup;
  
  imap_sync = folder_sync->imap_sync;
  storage = etpan_imap_sync_get_thread_storage(imap_sync);
  etpan_thread_manager_app_storage_schedule(manager, storage, op);
}

static void threaded_get_env_list(struct etpan_thread_op * op)
{
  struct etpan_imap_folder_sync * folder_sync;
  struct etpan_error * error;
  clistiter * iter;
  struct mailimap_set * set;
  uint32_t next_lastuid;
  struct env_list_result * result;  
  
  folder_sync = op->param;
  result = op->result;
  
  if (folder_sync->current_fetch_group_index >= carray_count(folder_sync->fetch_groups))
    return;
  
  ETPAN_LOCAL_LOG("** fetching");
  set = carray_get(folder_sync->fetch_groups, folder_sync->current_fetch_group_index);
  ETPAN_LOCAL_LOG("%i/%i", folder_sync->current_fetch_group_index,
      carray_count(folder_sync->fetch_groups));
  for(iter = clist_begin(set->set_list) ; iter != NULL ;
      iter = clist_next(iter)) {
    struct mailimap_set_item * item;
    
    item = clist_content(iter);
    ETPAN_LOCAL_LOG("%u-%u", item->set_first, item->set_last);
  }
  error = fetch_env_list(folder_sync, set, &next_lastuid);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(folder_sync, 0, ERROR_DOMAIN_SYNC,
        error);
    ETPAN_ERROR_FREE(error);
    error = new_error;
    
    goto exit;
  }
  
  error = header_db_open(folder_sync);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(folder_sync, 0, ERROR_DOMAIN_SYNC,
        error);
    ETPAN_ERROR_FREE(error);
    error = new_error;
    
    goto exit;
  }
  result->updated = 1;
  error = etpan_imap_sync_set_envlastuid(folder_sync->imap_sync, folder_sync->location,
      next_lastuid);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    header_db_close(folder_sync);
    new_error = rewrite_error(folder_sync, 0, ERROR_DOMAIN_SYNC,
        error);
    ETPAN_ERROR_FREE(error);
    error = new_error;
    
    goto exit;
  }
  header_db_close(folder_sync);
  
  error = NULL;
  
 exit:
  result->error = error;
}

static void threaded_get_env_list_callback(int cancelled,
    struct env_list_result * result, void * data)
{
  struct etpan_imap_folder_sync * folder_sync;
    
  folder_sync = data;
    
  folder_sync->current_fetch_group_index ++;
  if (folder_sync->current_fetch_group_index >= carray_count(folder_sync->fetch_groups)) {
    /* finished */
    folder_sync->state = STATE_HAS_ENVELOPE;
  }
  else {
    /* not finished */
  }
  folder_sync->error = result->error;
  result->error = NULL;
  
  if (folder_sync->error != NULL) {
    if (etpan_error_get_code(folder_sync->error) == ERROR_STREAM) {
      folder_sync->state = STATE_STREAM_CLOSED;
      sync_state_done(folder_sync);
    }
  }
  
  if (result->updated) {
    notify_folder_content_updated(folder_sync);
  }
  
  notify_op_end(folder_sync);
}

static void threaded_get_env_list_cleanup(struct etpan_thread_op * op)
{
  struct env_list_result * result;  
  
  result = op->result;
  ETPAN_ERROR_FREE(result->error);
  
  free(result);
}


struct flags_list_result {
  struct etpan_error * error;
  int updated;
};

static void threaded_get_flag_list(struct etpan_thread_op * op);
static void threaded_get_flag_list_callback(int cancelled, struct flags_list_result * result, void * data);
static void threaded_get_flag_list_cleanup(struct etpan_thread_op * op);

static void get_flag_list_run_op(struct etpan_thread_manager_app * manager,
    struct etpan_imap_folder_sync * folder_sync)
{
  struct etpan_thread_op * op;
  struct etpan_storage * storage;
  struct etpan_imap_sync * imap_sync;
  struct flags_list_result * result;
  
  result = malloc(sizeof(* result));
  if (result == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  result->error = NULL;
  result->updated = 0;
  
  op = etpan_thread_op_new();
  op->param = folder_sync;
  op->result = result;
  
  op->cancellable = 1;
  op->run = threaded_get_flag_list;
  op->callback = (void (*)(int, void *, void *)) threaded_get_flag_list_callback;
  op->callback_data = folder_sync;
  op->cleanup = threaded_get_flag_list_cleanup;
  
  if ((etpan_get_time() - folder_sync->last_flags_sync_date) < FLAG_SYNC_INTERVAL) {
    folder_sync->skip_flags = 1;
  }
  else {
    folder_sync->skip_flags = 0;
  }
  
  imap_sync = folder_sync->imap_sync;
  storage = etpan_imap_sync_get_thread_storage(imap_sync);
  etpan_thread_manager_app_storage_schedule(manager, storage, op);
}

static void threaded_get_flag_list(struct etpan_thread_op * op)
{
  struct etpan_imap_folder_sync * folder_sync;
  struct etpan_error * error;
  clistiter * iter;
  struct mailimap_set * set;
  int r;
  struct flags_list_result * result;
  
  folder_sync = op->param;
  result = op->result;
  
  if (folder_sync->current_flag_group_index >= carray_count(folder_sync->flag_groups))
    return;
  
  if (folder_sync->skip_flags) {
    carray * keys;
    unsigned int i;
    
    error = header_db_open(folder_sync);
    if (error != NULL) {
      struct etpan_error * new_error;
      
      new_error = rewrite_error(folder_sync, 0, ERROR_DOMAIN_SYNC,
          error);
      ETPAN_ERROR_FREE(error);
      error = new_error;
      
      goto exit;
    }
    keys = etpan_sqldb_get_keys(folder_sync->header_db);
    for(i = 0 ; i < carray_count(keys) ; i ++) {
      char * uid;
      void * data;
      size_t length;
      struct etpan_message_flags * flags;
        
      uid = carray_get(keys, i);
      r = etpan_sqldb_get(folder_sync->header_db,
          uid, "flags", &data, &length);
      if (r < 0)
        continue;
      flags = etpan_message_flags_unserialize(data, length);
      free(data);
      update_count_with_flags(folder_sync, flags, 0);
      etpan_message_flags_free(flags);
    }
    etpan_sqldb_keys_free(keys);
    header_db_close(folder_sync);
  }
  else {
    ETPAN_LOCAL_LOG("** fetching flags %u/%u", folder_sync->current_flag_group_index, carray_count(folder_sync->flag_groups));
    set = carray_get(folder_sync->flag_groups, folder_sync->current_flag_group_index);
    for(iter = clist_begin(set->set_list) ; iter != NULL ;
        iter = clist_next(iter)) {
      struct mailimap_set_item * item;
      
      item = clist_content(iter);
      ETPAN_LOCAL_LOG("%u-%u", item->set_first, item->set_last);
    }
    error = fetch_flags_list(folder_sync, set);
    if (error != NULL) {
      struct etpan_error * new_error;
    
      new_error = rewrite_error(folder_sync, 0, ERROR_DOMAIN_SYNC,
          error);
      ETPAN_ERROR_FREE(error);
      error = new_error;
    
      goto exit;
    }
    
    result->updated = 1;
  }
  
  error = NULL;
  
 exit:
  result->error = error;
}

static void threaded_get_flag_list_callback(int cancelled, struct flags_list_result * result, void * data)
{
  struct etpan_imap_folder_sync * folder_sync;
  struct etpan_error * error;
  int done;
  
  folder_sync = data;

  folder_sync->error = result->error;
  result->error = NULL;
  
  done = 0;
  if (folder_sync->skip_flags) {
    done = 1;
  }
  else {
    if (folder_sync->error == NULL) {
      folder_sync->current_flag_group_index ++;
      
      if (folder_sync->current_flag_group_index >= carray_count(folder_sync->flag_groups)) {
        /* finished */
        folder_sync->last_flags_sync_date = etpan_get_time();
        done = 1;
      }
    }
  }
  
  if (done) {
    folder_sync->state = STATE_HAS_FLAGS;
    
    if (folder_sync->error == NULL) {
      error = etpan_imap_sync_set_status(folder_sync->imap_sync,
          folder_sync->location,
          folder_sync->msg_count,
          folder_sync->unseen_count,
          folder_sync->recent_count);
      folder_sync->error = error;
    }
  }
  if (folder_sync->error != NULL) {
    if (etpan_error_get_code(folder_sync->error) == ERROR_STREAM) {
      folder_sync->state = STATE_STREAM_CLOSED;
      sync_state_done(folder_sync);
    }
  }
  
  if (result->updated) {
    notify_folder_content_updated(folder_sync);
  }
  
  notify_op_end(folder_sync);
}

static void threaded_get_flag_list_cleanup(struct etpan_thread_op * op)
{
  struct flags_list_result * result;
  
  result = op->result;
  ETPAN_ERROR_FREE(result->error);
  free(result);
}

struct body_list_result {
  struct etpan_error * error;
};

static void threaded_get_body_list(struct etpan_thread_op * op);
static void threaded_get_body_list_callback(int cancelled, struct body_list_result * result, void * data);
static void threaded_get_body_list_cleanup(struct etpan_thread_op * op);

static void get_body_list_run_op(struct etpan_thread_manager_app * manager,
    struct etpan_imap_folder_sync * folder_sync)
{
  struct etpan_thread_op * op;
  struct etpan_storage * storage;
  struct etpan_imap_sync * imap_sync;
  struct body_list_result * result;
  
  result = malloc(sizeof(* result));
  if (result == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  result->error = NULL;
  
  op = etpan_thread_op_new();
  op->param = folder_sync;
  op->result = result;
  
  op->cancellable = 1;
  op->run = threaded_get_body_list;
  op->callback = (void (*)(int, void *, void *)) threaded_get_body_list_callback;
  op->callback_data = folder_sync;
  op->cleanup = threaded_get_body_list_cleanup;
  
  imap_sync = folder_sync->imap_sync;
  storage = etpan_imap_sync_get_thread_storage(imap_sync);
  etpan_thread_manager_app_storage_schedule(manager, storage, op);
}

static void threaded_get_body_list(struct etpan_thread_op * op)
{
  struct etpan_imap_folder_sync * folder_sync;
  struct etpan_imap_sync * imap_sync;
  struct etpan_error * error;
  uint32_t uid;
  struct body_list_result * result;
  
  folder_sync = op->param;
  result = op->result;
  imap_sync = folder_sync->imap_sync;
  
  if (folder_sync->current_body_index >= folder_sync->body_uid_count) {
    return;
  }
  
  error = body_db_open(folder_sync);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(folder_sync, 0, ERROR_DOMAIN_SYNC,
        error);
    ETPAN_ERROR_FREE(error);
    error = new_error;
    
    goto exit;
  }
  ETPAN_LOCAL_LOG("** fetching body %i/%i",
      folder_sync->current_body_index, folder_sync->body_uid_count);
  uid = folder_sync->body_uid_list[folder_sync->current_body_index];
  body_db_close(folder_sync);
  
  error = fetch_parts(folder_sync, uid);
  if (error != NULL) {
    if (etpan_error_get_code(error) == ERROR_FETCH) {
      /* msg not found should not be critical */
      ETPAN_ERROR_IGNORE(error);
      error = NULL;
    }
  }
  
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(folder_sync, 0, ERROR_DOMAIN_SYNC,
        error);
    ETPAN_ERROR_FREE(error);
    error = new_error;
    
    goto exit;
  }
  
  error = body_db_open(folder_sync);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(folder_sync, 0, ERROR_DOMAIN_SYNC,
        error);
    ETPAN_ERROR_FREE(error);
    error = new_error;
    
    goto exit;
  }
  error = etpan_imap_sync_set_bodylastuid(imap_sync, folder_sync->location, uid);
  body_db_close(folder_sync);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(folder_sync, 0, ERROR_DOMAIN_SYNC,
        error);
    ETPAN_ERROR_FREE(error);
    error = new_error;
    goto exit;
  }
  
  error = NULL;
  
 exit:
  result->error = error;
}

static void threaded_get_body_list_callback(int cancelled, struct body_list_result * result, void * data)
{
  struct etpan_imap_folder_sync * folder_sync;
  
  folder_sync = data;
  
  folder_sync->error = result->error;
  result->error = NULL;
  
  if (folder_sync->error == NULL) {
    folder_sync->current_body_index ++;
    if (folder_sync->current_body_index >= folder_sync->body_uid_count) {
      /* finished */
      folder_sync->state = STATE_HAS_BODY;
    }
  }
  if (folder_sync->error != NULL) {
    if (etpan_error_get_code(folder_sync->error) == ERROR_STREAM) {
      folder_sync->state = STATE_STREAM_CLOSED;
      sync_state_done(folder_sync);
    }
  }
  
  notify_op_end(folder_sync);
}

static void threaded_get_body_list_cleanup(struct etpan_thread_op * op)
{
  struct body_list_result * result;
  
  result = op->result;
  ETPAN_ERROR_FREE(result->error);
  free(result);
}


struct cleanup_result {
  struct etpan_error * error;
};

static void threaded_cleanup(struct etpan_thread_op * op);
static void threaded_cleanup_callback(int cancelled, struct cleanup_result * result, void * data);
static void threaded_cleanup_cleanup(struct etpan_thread_op * op);

static void cleanup_run_op(struct etpan_thread_manager_app * manager,
    struct etpan_imap_folder_sync * folder_sync)
{
  struct etpan_thread_op * op;
  struct etpan_storage * storage;
  struct etpan_imap_sync * imap_sync;
  struct cleanup_result * result;
  
  result = malloc(sizeof(* result));
  if (result == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  result->error = NULL;
  
  op = etpan_thread_op_new();
  op->param = folder_sync;
  op->result = result;
    
  op->cancellable = 0;
  op->run = threaded_cleanup;
  op->callback = (void (*)(int, void *, void *)) threaded_cleanup_callback;
  op->callback_data = folder_sync;
  op->cleanup = threaded_cleanup_cleanup;
    
  imap_sync = folder_sync->imap_sync;
  storage = etpan_imap_sync_get_thread_storage(imap_sync);
  etpan_thread_manager_app_storage_schedule(manager, storage, op);
}

static void threaded_cleanup(struct etpan_thread_op * op)
{
  struct etpan_imap_folder_sync * folder_sync;
  struct etpan_error * error;
  int r;
  carray * keys;
  unsigned int i;
  struct cleanup_result * result;
  int had_error;
  
  folder_sync = op->param;
  result = op->result;
  
  ETPAN_LOCAL_LOG("** cleanup");
  
  if (!folder_sync->skip_flags) {

    /* cleanup header DB */
    
    error = header_db_open(folder_sync);
    if (error != NULL) {
      struct etpan_error * new_error;
      
      new_error = rewrite_error(folder_sync, 0, ERROR_DOMAIN_SYNC,
          error);
      ETPAN_ERROR_FREE(error);
      error = new_error;
      
      goto exit;
    }
    
    had_error = 0;
    r = etpan_sqldb_begin_transaction(folder_sync->header_db);
    if (r < 0)
      had_error = 1;
    keys = etpan_sqldb_get_keys(folder_sync->header_db);
    
    for(i = 0 ; i < carray_count(keys) ; i ++) {
      char * uid;
      chashdatum key;
      chashdatum value;
      
      uid = carray_get(keys, i);
      key.data = uid;
      key.len = strlen(uid) + 1;
      r = chash_get(folder_sync->flags_updated_hash, &key, &value);
      if (r < 0) {
        ETPAN_LOCAL_LOG("remove cached %s", uid);
        r = etpan_sqldb_delete(folder_sync->header_db, uid);
        if (r < 0)
          had_error = 1;
      }
    }
    
    etpan_sqldb_keys_free(keys);
    r = etpan_sqldb_end_transaction(folder_sync->header_db);
    if (r < 0)
      had_error = 1;
    
    header_db_close(folder_sync);
    
    if (had_error) {
      error = etpan_error_new();
      etpan_error_set_code(error, ERROR_DATABASE);
      etpan_error_set_short_description(error, _("Cache could not be written"));
      etpan_error_strf_long_description(error,
          _("An error occurred while synchronizing account %s.\n"
          "Cache of mailbox %s could not be written properly on disk."),
          etpan_imap_sync_get_id(folder_sync->imap_sync),
          folder_sync->location);
      goto exit;
    }
    
    /* cleanup body DB */
    
    error = body_db_open(folder_sync);
    if (error != NULL) {
      struct etpan_error * new_error;
      
      new_error = rewrite_error(folder_sync, 0, ERROR_DOMAIN_SYNC,
          error);
      ETPAN_ERROR_FREE(error);
      error = new_error;
      
      goto exit;
    }
    
    had_error = 0;
    r = etpan_sqldb_begin_transaction(folder_sync->body_db);
    if (r < 0)
      had_error = 1;
    
    keys = etpan_sqldb_get_keys(folder_sync->body_db);
    
    for(i = 0 ; i < carray_count(keys) ; i ++) {
      char * body_id;
      chashdatum key;
      chashdatum value;
      char * msg_uid;
      char * p;
    
      body_id = carray_get(keys, i);
      msg_uid = strdup(body_id);
      if (msg_uid == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      p = strchr(msg_uid, '-');
      if (p != NULL) {
        * p = '\0';
        key.data = msg_uid;
        key.len = strlen(msg_uid) + 1;
        r = chash_get(folder_sync->flags_updated_hash, &key, &value);
        if (r < 0) {
          ETPAN_LOCAL_LOG("remove cached %s", body_id);
          r = etpan_sqldb_delete(folder_sync->body_db, body_id);
          if (r < 0)
            had_error = 1;
        }
      }
      free(msg_uid);
    }
    
    etpan_sqldb_keys_free(keys);
    
    r = etpan_sqldb_end_transaction(folder_sync->body_db);
    if (r < 0)
      had_error = 1;
    
    body_db_close(folder_sync);

    if (had_error) {
      error = etpan_error_new();
      etpan_error_set_code(error, ERROR_DATABASE);
      etpan_error_set_short_description(error, _("Cache could not be written"));
      etpan_error_strf_long_description(error,
          _("An error occurred while synchronizing account %s.\n"
          "Cache of mailbox %s could not be written properly on disk."),
          etpan_imap_sync_get_id(folder_sync->imap_sync),
          folder_sync->location);
      goto exit;
    }
  }
  
  error = NULL;
  ETPAN_LOCAL_LOG("** cleanup done");
  
 exit:
  result->error = error;
}

static void threaded_cleanup_callback(int cancelled, struct cleanup_result * result, void * data)
{
  struct etpan_imap_folder_sync * folder_sync;
  
  folder_sync = data;
  folder_sync->error = result->error;
  result->error = NULL;
  folder_sync->state = STATE_CONNECTED;
  folder_sync->last_sync_date = etpan_get_time();
  sync_state_done(folder_sync);
  
  notify_op_end(folder_sync);
}

static void threaded_cleanup_cleanup(struct etpan_thread_op * op)
{
  struct cleanup_result * result;
  
  result = op->result;
  ETPAN_ERROR_FREE(result->error);
  free(result);
}

static void notify_op_end(struct etpan_imap_folder_sync * folder_sync)
{
  if (folder_sync->error != NULL)
    etpan_error_log(folder_sync->error);
  
  ETPAN_SIGNAL_SEND(folder_sync, ETPAN_IMAP_FOLDER_SYNC_OP_ENDED_SIGNAL);
  
  ETPAN_ERROR_FREE(folder_sync->error);
  folder_sync->error = NULL;
}

static void sync_state_init(struct etpan_imap_folder_sync * folder_sync)
{
  folder_sync->fetch_groups = NULL;
  folder_sync->flag_groups = NULL;
  folder_sync->body_uid_list = NULL;
  folder_sync->body_uid_count = 0;
  folder_sync->uid_list = NULL;
  folder_sync->uid_count = 0;
  
  folder_sync->skip_flags = 0;
  folder_sync->msg_count = 0;
  folder_sync->unseen_count = 0;
  folder_sync->recent_count = 0;
  folder_sync->flags_updated_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (folder_sync->flags_updated_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
}

static void sync_state_done(struct etpan_imap_folder_sync * folder_sync)
{
  unsigned int i;
    
  if (folder_sync->fetch_groups != NULL) {
    for(i = 0 ; i < carray_count(folder_sync->fetch_groups) ; i ++) {
      struct mailimap_set * set;
            
      set = carray_get(folder_sync->fetch_groups, i);
      mailimap_set_free(set);
    }
    folder_sync->fetch_groups = NULL;
  }
  if (folder_sync->flag_groups != NULL) {
    for(i = 0 ; i < carray_count(folder_sync->flag_groups) ; i ++) {
      struct mailimap_set * set;
            
      set = carray_get(folder_sync->flag_groups, i);
      mailimap_set_free(set);
    }
    folder_sync->flag_groups = NULL;
  }
  if (folder_sync->body_uid_list != NULL) {
    free(folder_sync->body_uid_list);
    folder_sync->body_uid_list = NULL;
    folder_sync->body_uid_count = 0;
  }
  if (folder_sync->uid_list != NULL) {
    free(folder_sync->uid_list);
    folder_sync->uid_list = NULL;
    folder_sync->uid_count = 0;
  }
  
  if (folder_sync->flags_updated_hash != NULL) {
    chash_free(folder_sync->flags_updated_hash);
    folder_sync->flags_updated_hash = NULL;
  }
}


static int comp_pair(struct flags_pair ** a, struct flags_pair ** b)
{
  return (* a)->uid - (* b)->uid;
}

static struct etpan_error *
check_flags(struct etpan_imap_folder_sync * folder_sync)
{
  struct etpan_error * error;
  carray * error_list;
  chashiter * iter;
  int r;
  carray * tab;
  unsigned int i;
  uint32_t first;
  uint32_t last;
  int expunge_needed;
  
  error_list = carray_new(4);
  if (error_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  pthread_mutex_lock(&folder_sync->pending_flags_lock);
  tab = carray_new(chash_count(folder_sync->pending_flags));
  if (tab == NULL)
    ETPAN_LOG_MEMORY_ERROR;

  ETPAN_LOCAL_LOG("store flags %u messages %s", chash_count(folder_sync->pending_flags),
      folder_sync->location);
  
  expunge_needed = 0;
  for(iter = chash_begin(folder_sync->pending_flags) ; iter != NULL ;
      iter = chash_next(folder_sync->pending_flags, iter)) {
    chashdatum value;
    struct flags_pair * pair;
    int flag_value;
    
    chash_value(iter, &value);
    pair = value.data;
    flag_value = etpan_message_flags_get_value(pair->flags);
    if ((flag_value & ETPAN_FLAGS_DELETED) != 0)
      expunge_needed = 1;
    r = carray_add(tab, pair, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  chash_clear(folder_sync->pending_flags);
  
  pthread_mutex_unlock(&folder_sync->pending_flags_lock);
  
  /* sort flags by uid */
  qsort(carray_data(tab), carray_count(tab), sizeof(void *),
      (int (*)(const void *, const void *)) comp_pair);
  
  /* send flags */
  if (carray_count(tab) > 0) {
    struct etpan_message_flags * flags;
    struct mail_flags * lep_flags;
    struct flags_pair * pair;
    struct etpan_error * cur_error;
    
    error = select_folder(folder_sync);
    if (error != NULL) {
      goto free;
    }
    
    pair = carray_get(tab, 0);
    first = pair->uid;
    last = pair->uid;
    flags = pair->flags;
    lep_flags = etpan_lep_flags_to_lep(flags);
    
    for(i = 1 ; i < carray_count(tab) ; i ++) {
      pair = carray_get(tab, i);
      
      if ((last + 1 == pair->uid) &&
          etpan_message_flags_equal(flags, pair->flags)) {
        last = pair->uid;
      }
      else {
        cur_error = etpan_imap_store_flags(get_session(folder_sync),
            folder_sync->location, first, last, lep_flags);
        if (cur_error != NULL) {
          r = carray_add(error_list, cur_error, NULL);
          if (r < 0)
            ETPAN_LOG_MEMORY_ERROR;
        }
        
        first = pair->uid;
        last = pair->uid;
        flags = pair->flags;
        lep_flags = etpan_lep_flags_to_lep(flags);
      }
    }
    
    cur_error = etpan_imap_store_flags(get_session(folder_sync),
        folder_sync->location, first, last, lep_flags);
    if (cur_error != NULL) {
      r = carray_add(error_list, cur_error, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
    }
  }
  if (expunge_needed) {
    struct etpan_error * cur_error;
    
    error = select_folder(folder_sync);
    if (error != NULL) {
      goto free;
    }
    
    cur_error = etpan_imap_expunge(get_session(folder_sync),
        folder_sync->location);
    if (cur_error != NULL) {
      r = carray_add(error_list, cur_error, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
    }
  }
  
  /* cleanup */
  
 free:
  for(i = 0 ; i < carray_count(tab) ; i ++) {
    struct flags_pair * pair;
    
    pair = carray_get(tab, i);
    etpan_message_flags_free(pair->flags);
    free(pair);
  }
  
  carray_free(tab);
  
  error = NULL;
  if (carray_count(error_list) > 0) {
    struct etpan_error * new_error;
    
    error = etpan_error_multiple();
    for(i = 0 ; i < carray_count(error_list) ; i ++) {
      struct etpan_error * cur_error;
      
      cur_error = carray_get(error_list, i);
      etpan_error_add_child(error, cur_error);
    }
    
    new_error = rewrite_error(folder_sync, 0, ERROR_DOMAIN_SYNC,
        error);
    ETPAN_ERROR_FREE(error);
    error = new_error;
  }
  
  return error;
}

struct check_flags_result {
  struct etpan_error * error;
};

static void threaded_check_flags(struct etpan_thread_op * op);
static void threaded_check_flags_callback(int cancelled, struct check_flags_result * result, void * data);
static void threaded_check_flags_cleanup(struct etpan_thread_op * op);

static void threaded_check_flags_run_op(struct etpan_thread_manager_app * manager,
    struct etpan_imap_folder_sync * folder_sync)
{
  struct etpan_thread_op * op;
  struct etpan_storage * storage;
  struct etpan_imap_sync * imap_sync;
  struct check_flags_result * result;
  
  result = malloc(sizeof(* result));
  if (result == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  result->error = NULL;
  
  op = etpan_thread_op_new();
  op->param = folder_sync;
  op->result = result;
  
  op->cancellable = 0;
  op->run = threaded_check_flags;
  op->callback = (void (*)(int, void *, void *)) threaded_check_flags_callback;
  op->callback_data = folder_sync;
  op->cleanup = threaded_check_flags_cleanup;
  
  imap_sync = folder_sync->imap_sync;
  storage = etpan_imap_sync_get_thread_storage(imap_sync);
  etpan_thread_manager_app_storage_schedule(manager, storage, op);
}

static void threaded_check_flags(struct etpan_thread_op * op)
{
  struct check_flags_result * result;
  struct etpan_imap_folder_sync * folder_sync;
  
  folder_sync = op->param;
  result = op->result;
  result->error = check_flags(folder_sync);
}

static void threaded_check_flags_callback(int cancelled, struct check_flags_result * result, void * data)
{
  struct etpan_imap_folder_sync * folder_sync;
  
  folder_sync = data;
  folder_sync->error = result->error;
  result->error = NULL;
  
  if (folder_sync->error != NULL) {
    if (etpan_error_get_code(folder_sync->error) == ERROR_STREAM) {
      folder_sync->state = STATE_STREAM_CLOSED;
      sync_state_done(folder_sync);
    }
  }
  
  notify_op_end(folder_sync);
}

static void threaded_check_flags_cleanup(struct etpan_thread_op * op)
{
  struct check_flags_result * result;
  
  result = op->result;
  ETPAN_ERROR_FREE(result->error);
  free(result);
}


static struct etpan_error *
process_append(struct etpan_imap_folder_sync * folder_sync)
{
  int r;
  struct etpan_error * error;
  void * data;
  size_t length;
  char * content;
  size_t content_length;
  char * uid;
  struct etpan_message_flags * flags;
  struct mail_flags * ep_flags;
  int had_error;
  
  error = append_db_open(folder_sync);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(folder_sync, 0, ERROR_DOMAIN_SYNC,
        error);
    ETPAN_ERROR_FREE(error);
    error = new_error;
    
    goto exit;
  }
  
  if (folder_sync->append_queue == NULL) {
    folder_sync->append_queue = etpan_sqldb_get_keys(folder_sync->append_db);
    folder_sync->current_append_index = 0;
    if (carray_count(folder_sync->append_queue) == 0) {
      carray_free(folder_sync->append_queue);
      folder_sync->append_queue = NULL;
      pthread_mutex_lock(&folder_sync->append_lock);
      folder_sync->append_has_items = 0;
      pthread_mutex_unlock(&folder_sync->append_lock);
    }
    else {
      pthread_mutex_lock(&folder_sync->append_lock);
      folder_sync->append_has_items = 1;
      pthread_mutex_unlock(&folder_sync->append_lock);
    }
  }
  
  if (!folder_sync->append_has_items) {
    append_db_close(folder_sync);
    return NULL;
  }
  
  ETPAN_LOG("append %i/%i %i", folder_sync->current_append_index,
      carray_count(folder_sync->append_queue),
      folder_sync->append_has_items);

  if (folder_sync->current_append_index >= carray_count(folder_sync->append_queue)) {
    etpan_sqldb_keys_free(folder_sync->append_queue);
    folder_sync->append_queue = NULL;
    append_db_close(folder_sync);
    return NULL;
  }
  
  ETPAN_LOCAL_LOG("append");
  
  content = NULL;
  uid = carray_get(folder_sync->append_queue,
      folder_sync->current_append_index);
  uid = strdup(uid);
  if (uid == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  r = etpan_sqldb_get(folder_sync->append_db, uid, "content",
      &data, &length);
  if (r < 0) {
    free(uid);
    etpan_sqldb_keys_free(folder_sync->append_queue);
    folder_sync->append_queue = NULL;
    ETPAN_WARN_LOG("database corrupted");
    append_db_close(folder_sync);
    return NULL;
  }
  else {
    decode_str(data, length, &content, &content_length);
    
    r = etpan_sqldb_get(folder_sync->append_db, uid, "flags",
        &data, &length);
    if (r == 0) {
      flags = etpan_message_flags_unserialize(data, length);
    }
    else {
      ETPAN_WARN_LOG("database corrupted");
      flags = etpan_message_flags_new();
    }
    folder_sync->current_append_index ++;
  }
  
  /* if finished, flush the queue */
  if (folder_sync->current_append_index >= carray_count(folder_sync->append_queue)) {
    etpan_sqldb_keys_free(folder_sync->append_queue);
    folder_sync->append_queue = NULL;
  }
  
  append_db_close(folder_sync);
  
  error = select_folder(folder_sync);
  if (error != NULL) {
    etpan_message_flags_free(flags);
    free(content);
    goto free_uid;
  }
  
  ETPAN_ASSERT(flags != NULL, "flags should not be null");
  ep_flags = etpan_lep_flags_to_lep(flags);
  error = etpan_imap_append(get_session(folder_sync),
      folder_sync->location, content,
      content_length, ep_flags);
  mail_flags_free(ep_flags);
  etpan_message_flags_free(flags);
  free(content);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(folder_sync, 0, ERROR_DOMAIN_SYNC,
        error);
    ETPAN_ERROR_FREE(error);
    error = new_error;
    
    goto free_uid;
  }
  
  error = append_db_open(folder_sync);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(folder_sync, 0, ERROR_DOMAIN_SYNC,
        error);
    ETPAN_ERROR_FREE(error);
    error = new_error;
    
    goto free_uid;
  }
  
  had_error = 0;
  r = etpan_sqldb_begin_transaction(folder_sync->append_db);
  if (r < 0)
    had_error = 1;
  r = etpan_sqldb_delete(folder_sync->append_db, uid);
  if (r < 0)
    had_error = 1;
  r = etpan_sqldb_end_transaction(folder_sync->append_db);
  if (r < 0)
    had_error = 1;

  if (had_error) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_DATABASE);
    etpan_error_set_short_description(error, _("Cache could not be written"));
    etpan_error_strf_long_description(error,
        _("An error occurred while synchronizing account %s.\n"
            "Cache of mailbox %s could not be written properly on disk."),
        etpan_imap_sync_get_id(folder_sync->imap_sync),
        folder_sync->location);
    goto close_db;
  }
  
  error = NULL;
  
 close_db:
  append_db_close(folder_sync);
 free_uid:
  free(uid);
 exit:
  return error;
}


struct append_result {
  struct etpan_error * error;
};

static void threaded_append(struct etpan_thread_op * op);
static void threaded_append_callback(int cancelled, struct append_result * result, void * data);
static void threaded_append_cleanup(struct etpan_thread_op * op);

static void threaded_append_run_op(struct etpan_thread_manager_app * manager,
    struct etpan_imap_folder_sync * folder_sync)
{
  struct etpan_thread_op * op;
  struct etpan_storage * storage;
  struct etpan_imap_sync * imap_sync;
  struct append_result * result;
  
  result = malloc(sizeof(* result));
  if (result == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  result->error = NULL;
  
  op = etpan_thread_op_new();
  op->param = folder_sync;
  op->result = result;
    
  op->cancellable = 0;
  op->run = threaded_append;
  op->callback = (void (*)(int, void *, void *)) threaded_append_callback;
  op->callback_data = folder_sync;
  op->cleanup = threaded_append_cleanup;
  
  imap_sync = folder_sync->imap_sync;
  storage = etpan_imap_sync_get_thread_storage(imap_sync);
  etpan_thread_manager_app_storage_schedule(manager, storage, op);
}

static void threaded_append(struct etpan_thread_op * op)
{
  struct check_flags_result * result;
  struct etpan_imap_folder_sync * folder_sync;
  
  folder_sync = op->param;
  result = op->result;
  result->error = process_append(folder_sync);
}

static void threaded_append_callback(int cancelled, struct append_result * result, void * data)
{
  struct etpan_imap_folder_sync * folder_sync;
  
  folder_sync = data;
  folder_sync->error = result->error;
  result->error = NULL;
  if (folder_sync->error != NULL) {
    if (etpan_error_get_code(folder_sync->error) == ERROR_STREAM) {
      folder_sync->state = STATE_STREAM_CLOSED;
      sync_state_done(folder_sync);
    }
  }
  else {
    folder_sync->last_sync_date = 0;
    folder_sync->state = STATE_CONNECTED;
  }
  notify_op_end(folder_sync);
}

static void threaded_append_cleanup(struct etpan_thread_op * op)
{
  struct check_flags_result * result;
  
  result = op->result;
  ETPAN_ERROR_FREE(result->error);
  free(result);
}

void etpan_imap_folder_sync(struct etpan_thread_manager_app * manager,
                            struct etpan_imap_folder_sync * folder_sync)
{
  if (folder_sync->op != NULL)
    ETPAN_CRASH("op should not be launched twice");

  ETPAN_LOCAL_LOG("sync %s", folder_sync->location);
  switch (folder_sync->state) {
  case STATE_CONNECTED:
  case STATE_HAS_UIDLIST:
  case STATE_HAS_ENVELOPE:
  case STATE_HAS_FLAGS:
  case STATE_HAS_BODY:
    {
      int b;
    
      pthread_mutex_lock(&folder_sync->check_requested_lock);
      b = folder_sync->check_requested;
      folder_sync->check_requested = 0;
      pthread_mutex_unlock(&folder_sync->check_requested_lock);
    
      if (b) {
        ETPAN_LOCAL_LOG("check flags (store flags)");
        threaded_check_flags_run_op(manager, folder_sync);
        return;
      }
    
      pthread_mutex_lock(&folder_sync->append_lock);
      b = folder_sync->append_has_items;
      pthread_mutex_unlock(&folder_sync->append_lock);
      ETPAN_LOCAL_LOG("append : %i", b);
    
      if (b) {
        ETPAN_LOCAL_LOG("append %s", folder_sync->location);
        threaded_append_run_op(manager, folder_sync);
        return;
      }
    }
  }

  ETPAN_LOCAL_LOG("folder sync %s", folder_sync->location);
  switch (folder_sync->state) {
  case STATE_DISCONNECTED:
#if 0
    ETPAN_LOCAL_LOG("connect %s %s", etpan_imap_sync_get_id(folder_sync->imap_sync),
        folder_sync->location);
    connect_run_op(manager, folder_sync);
#endif
    folder_sync->state = STATE_CONNECTED;
    notify_op_end(folder_sync);
    break;
  case STATE_CONNECTED:
    sync_state_init(folder_sync);
    ETPAN_LOCAL_LOG("get uid list %s %s",
        etpan_imap_sync_get_id(folder_sync->imap_sync),
        folder_sync->location);
    get_uid_list_run_op(manager, folder_sync);
    break;
  case STATE_STREAM_CLOSED:
#if 0
    ETPAN_LOCAL_LOG("(closed) connect %s %s", etpan_imap_sync_get_id(folder_sync->imap_sync),
        folder_sync->location);
    connect_run_op(manager, folder_sync);
#endif
    folder_sync->state = STATE_CONNECTED;
    notify_op_end(folder_sync);
    break;
  case STATE_HAS_UIDLIST:
    ETPAN_LOCAL_LOG("get env list %s %s",
        etpan_imap_sync_get_id(folder_sync->imap_sync),
        folder_sync->location);
    get_env_list_run_op(manager, folder_sync);
    break;
  case STATE_HAS_ENVELOPE:
    ETPAN_LOCAL_LOG("get flags %s %s",
        etpan_imap_sync_get_id(folder_sync->imap_sync),
        folder_sync->location);
    get_flag_list_run_op(manager, folder_sync);
    break;
  case STATE_HAS_FLAGS:
    if (folder_sync->opened) {
      ETPAN_LOCAL_LOG("get body %s %s",
        etpan_imap_sync_get_id(folder_sync->imap_sync),
        folder_sync->location);
      get_body_list_run_op(manager, folder_sync);
    }
    else {
      ETPAN_LOCAL_LOG("cleanup %s %s",
        etpan_imap_sync_get_id(folder_sync->imap_sync),
        folder_sync->location);
      cleanup_run_op(manager, folder_sync);
    }
    break;
  case STATE_HAS_BODY:
    ETPAN_LOCAL_LOG("cleanup %s %s",
        etpan_imap_sync_get_id(folder_sync->imap_sync),
        folder_sync->location);
    cleanup_run_op(manager, folder_sync);
    break;
  }
}

void etpan_imap_folder_set_sync_now(struct etpan_imap_folder_sync * folder_sync)
{
  folder_sync->last_sync_date = 0;
  folder_sync->last_flags_sync_date = 0;
}

int etpan_imap_folder_sync_get_priority(struct etpan_imap_folder_sync * folder_sync)
{
  int priority_delta;
  int priority;
  int priority_set;
  
  priority_set = 0;
  priority = ETPAN_IMAP_SYNC_PRIORITY_HAS_NOTHING;
  priority_delta = 0;
  if (folder_sync->opened) {
    priority_delta += ETPAN_IMAP_SYNC_PRIORITY_OPENED * folder_sync->opened;
  }
  if (strcasecmp(folder_sync->location, "INBOX") == 0) {
    priority_delta += ETPAN_IMAP_SYNC_PRIORITY_INBOX;
  }
  if (folder_sync->append_has_items) {
    priority_delta += ETPAN_IMAP_SYNC_PRIORITY_HAS_APPEND_ITEMS;
  }
  
  switch (folder_sync->state) {
  case STATE_CONNECTED:
  case STATE_HAS_UIDLIST:
  case STATE_HAS_ENVELOPE:
  case STATE_HAS_FLAGS:
  case STATE_HAS_BODY:
    if (!priority_set) {
      int b;
      
      pthread_mutex_lock(&folder_sync->check_requested_lock);
      b = folder_sync->check_requested;
      pthread_mutex_unlock(&folder_sync->check_requested_lock);
    
      if (b) {
        priority = ETPAN_IMAP_SYNC_PRIORITY_USER_OP + priority_delta;
        priority_set = 1;
      }
    }
    
    if (!priority_set) {
      int b;
      
      pthread_mutex_lock(&folder_sync->append_lock);
      b = folder_sync->append_has_items;
      pthread_mutex_unlock(&folder_sync->append_lock);
      
      if (b) {
        priority = ETPAN_IMAP_SYNC_PRIORITY_USER_OP + priority_delta;
        priority_set = 1;
      }
    }
    break;
  }

  if (!priority_set) {
    switch (folder_sync->state) {
    case STATE_DISCONNECTED:
    case STATE_CONNECTED:
    case STATE_STREAM_CLOSED:
    case STATE_HAS_UIDLIST:
      priority = ETPAN_IMAP_SYNC_PRIORITY_HAS_NOTHING + priority_delta;
      priority_set = 1;
      break;
    
    case STATE_HAS_ENVELOPE:
    case STATE_HAS_FLAGS:
      priority = ETPAN_IMAP_SYNC_PRIORITY_HAS_ENV + priority_delta;
      priority_set = 1;
      break;
    
    case STATE_HAS_BODY:
      priority = ETPAN_IMAP_SYNC_PRIORITY_HAS_BODY + priority_delta;
      priority_set = 1;
      break;
    }
  }
  
  ETPAN_LOCAL_LOG("folder priority %s %s %i",
      etpan_imap_sync_get_id(folder_sync->imap_sync),
      folder_sync->location, priority);
  
  return priority;
}

double etpan_imap_folder_sync_next_op_date(struct etpan_imap_folder_sync * folder_sync)
{
  switch (folder_sync->state) {
  case STATE_CONNECTED:
  case STATE_HAS_UIDLIST:
  case STATE_HAS_ENVELOPE:
  case STATE_HAS_FLAGS:
  case STATE_HAS_BODY:
    {
      int b;
    
      pthread_mutex_lock(&folder_sync->check_requested_lock);
      b = folder_sync->check_requested;
      pthread_mutex_unlock(&folder_sync->check_requested_lock);
  
      if (b)
        return 0;

      pthread_mutex_lock(&folder_sync->append_lock);
      b = folder_sync->append_has_items;
      pthread_mutex_unlock(&folder_sync->append_lock);

      if (b)
        return 0;
    }
    break;
  }
  
  switch (folder_sync->state) {
  case STATE_DISCONNECTED:
  case STATE_CONNECTED:
    return folder_sync->last_sync_date + SYNC_INTERVAL;
  default:
    return 0;
  }
}

char * etpan_imap_folder_sync_get_location(struct etpan_imap_folder_sync * folder_sync)
{
  return folder_sync->location;
}

int etpan_imap_folder_sync_get_noinferiors(struct etpan_imap_folder_sync * folder_sync)
{
  return folder_sync->noinferiors;
}

char etpan_imap_folder_sync_get_imap_separator(struct etpan_imap_folder_sync * folder_sync)
{
  return folder_sync->separator;
}


struct etpan_error *
etpan_imap_folder_sync_fetch_msg_list(struct etpan_folder * folder,
    struct etpan_imap_folder_sync * folder_sync,
    chash * current_msg_list)
{
  int r;
  struct etpan_error * error;
  carray * key_list;
  unsigned int i;
  
  error = header_db_open(folder_sync);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(folder_sync, 0, ERROR_DOMAIN_SYNC,
        error);
    ETPAN_ERROR_FREE(error);
    error = new_error;
    goto exit;
  }
  
  key_list = etpan_sqldb_get_keys(folder_sync->header_db);
  for(i = 0 ; i < carray_count(key_list) ; i ++) {
    chashdatum key;
    chashdatum value;
    struct etpan_message * msg;
    char * uid;
    void * data;
    size_t length;
    struct etpan_message_header * header;
    struct etpan_message_flags * flags;
    
    uid = carray_get(key_list, i);
    msg = etpan_message_imap_sync_new();
    etpan_message_set_folder(msg, folder);
    etpan_message_imap_sync_set_uid(msg, uid);
    r = etpan_sqldb_get(folder_sync->header_db, uid,
        "headers", &data, &length);
    if (r < 0) {
      etpan_message_free(msg);
      continue;
    }
    header = etpan_message_header_unserialize(data, length);
    free(data);
    etpan_message_set_header(msg, header);
    
    r = etpan_sqldb_get(folder_sync->header_db, uid,
        "flags", &data, &length);
    if (r < 0) {
      etpan_message_free(msg);
      continue;
    }
    flags = etpan_message_flags_unserialize(data, length);
    free(data);
    etpan_message_set_flags(msg, flags);
    etpan_message_flags_free(flags);
    
    uid = etpan_message_get_uid(msg); 
    key.data = uid;
    key.len = strlen(uid) + 1;
    value.data = msg;
    value.len = 0;
    
    r = chash_set(current_msg_list, &key, &value, NULL);
    if (r < 0) {
      ETPAN_LOG_MEMORY_ERROR;
    }
  }
  etpan_sqldb_keys_free(key_list);
  error = NULL;
  
  header_db_close(folder_sync);
 exit:
  return error;
}

void etpan_imap_folder_sync_check_msg(struct etpan_imap_folder_sync * folder_sync,
    struct etpan_message * msg,
    struct etpan_message_flags * flags)
{
  char * uid_str;
  uint32_t uid;
  chashdatum key;
  chashdatum value;
  struct etpan_message_flags * dup_flags;
  int r;
  struct flags_pair * pair;
  
  ETPAN_LOCAL_LOG("check msg flags");
  uid_str = etpan_message_imap_sync_get_uid(msg);
  uid = strtoul(uid_str, NULL, 10);
  dup_flags = etpan_message_flags_dup(flags);
  pair = malloc(sizeof(* flags));
  if (pair == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  pair->flags = dup_flags;
  pair->uid = uid;
  key.data = uid_str;
  key.len = strlen(uid_str) + 1;
  
  pthread_mutex_lock(&folder_sync->pending_flags_lock);
  r = chash_get(folder_sync->pending_flags, &key, &value);
  if (r == 0) {
    struct flags_pair * old_pair;
    
    old_pair = value.data;
    etpan_message_flags_free(old_pair->flags);
    free(old_pair);
    chash_delete(folder_sync->pending_flags, &key, NULL);
  }
  
  value.data = pair;
  value.len = 0;
  r = chash_set(folder_sync->pending_flags, &key, &value, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  pthread_mutex_unlock(&folder_sync->pending_flags_lock);
}

static void flush_pending_flags(struct etpan_imap_folder_sync * folder_sync)
{
  chashiter * iter;
  
  for(iter = chash_begin(folder_sync->pending_flags) ; iter != NULL ;
      iter = chash_next(folder_sync->pending_flags, iter)) {
    chashdatum value;
    struct flags_pair * pair;
    
    chash_value(iter, &value);
    pair = value.data;
    etpan_message_flags_free(pair->flags);
    free(pair);
  }
  chash_clear(folder_sync->pending_flags);
}


struct etpan_error *
etpan_imap_folder_sync_status(struct etpan_imap_folder_sync * folder_sync,
    unsigned int * p_count, unsigned int * p_unseen, unsigned int * p_recent)
{
  int r;
  
  r = etpan_imap_sync_get_status(folder_sync->imap_sync,
      folder_sync->location, p_count, p_unseen, p_recent);
  if (r < 0) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_STATUS);
    etpan_error_set_short_description(error, _("Mailbox error"));
    etpan_error_strf_long_description(error,
        _("Information of mailbox %s could not be retrieved."),
        folder_sync->location);
    
    return error;
  }
  
  return NULL;
}


struct body_data {
  struct mailsem * sem;
  struct etpan_imap_folder_sync * folder_sync;
  uint32_t uid_value;
  char * part_id;
  struct mailimap_body * result;
  struct etpan_error * error;
};

static void get_body_in_main_thread(void * data);

struct etpan_error *
etpan_imap_folder_sync_get_body(struct etpan_imap_folder_sync * folder_sync,
    char * uid, struct mailimap_body ** result)
{
  struct etpan_error * error;
  int r;
  char part_id[40];
  void * data;
  size_t length;
  struct mailimap_body * body;
  uint32_t uid_value;
  struct body_data * body_data;
  char * part_id_dup;
  
  uid_value = strtoul(uid, NULL, 10);
  
  error = body_db_open(folder_sync);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(folder_sync, uid_value, ERROR_DOMAIN_FETCH,
        error);
    ETPAN_ERROR_FREE(error);
    error = new_error;
    goto exit;
  }
  
  snprintf(part_id, sizeof(part_id), "%s-body", uid);
  r = etpan_sqldb_get(folder_sync->body_db, part_id,
      "content", &data, &length);
  body_db_close(folder_sync);
  
  if (r == 0) {
    body = etpan_imap_body_unserialize(data, length);
    free(data);
    
    * result = body;
    error = NULL;
    
    goto exit;
  }
  
  part_id_dup = strdup(part_id);
  if (part_id_dup == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  body_data = malloc(sizeof(* body_data));
  if (body_data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  body_data->folder_sync = folder_sync;
  body_data->uid_value = uid_value;
  body_data->part_id = part_id_dup;
  body_data->sem = mailsem_new();
  if (body_data->sem == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  body_data->result = NULL;
  body_data->error = NULL;
  
  etpan_thread_manager_app_run_in_main_thread(etpan_thread_manager_app_get_default(), get_body_in_main_thread, body_data, 0);
  
  mailsem_down(body_data->sem);
  
  body = body_data->result;
  error = body_data->error;
  
  mailsem_free(body_data->sem);
  free(body_data->part_id);
  free(body_data);
  
  if (error != NULL)
    goto exit;
  
  * result = body;
  
 exit:
  return error;
}

static void get_body_in_thread(struct etpan_thread_op * op);
static void get_body_in_thread_callback(int cancelled, void * dummy, void * data);

static void get_body_in_main_thread(void * data)
{
  struct etpan_thread_op * op;
  struct etpan_storage * storage;
  struct etpan_imap_sync * imap_sync;
  struct body_data * body_data;
  struct etpan_imap_folder_sync * folder_sync;
  
  body_data = data;
  folder_sync = body_data->folder_sync;
  
  op = etpan_thread_op_new();
  op->param = body_data;
  op->result = NULL;
  
  op->cancellable = 1;
  op->run = get_body_in_thread;
  op->callback = (void (*)(int, void *, void *)) get_body_in_thread_callback;
  op->callback_data = body_data;
  op->cleanup = NULL;
  
  imap_sync = folder_sync->imap_sync;
  storage = etpan_imap_sync_get_thread_storage(imap_sync);
  etpan_thread_manager_app_storage_schedule(etpan_thread_manager_app_get_default(), storage, op);
}

static void get_body_in_thread(struct etpan_thread_op * op)
{
  struct body_data * body_data;
  struct mailimap_body * body;
  struct etpan_error * error;
  struct etpan_imap_folder_sync * folder_sync;
  uint32_t uid_value;
  char * part_id;
  void * data;
  size_t length;
  int r;
  
  body_data = op->param;
  folder_sync = body_data->folder_sync;
  uid_value = body_data->uid_value;
  part_id = body_data->part_id;
  
  body = NULL;
  
#if 0
  error = connect(folder_sync);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(folder_sync, uid_value, ERROR_DOMAIN_FETCH,
        error);
    ETPAN_ERROR_FREE(error);
    error = new_error;
    goto exit;
  }
#endif
  
  error = fetch_parts(folder_sync, uid_value);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(folder_sync, uid_value, ERROR_DOMAIN_FETCH,
        error);
    ETPAN_ERROR_FREE(error);
    error = new_error;
    goto exit;
  }
  
  error = body_db_open(folder_sync);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(folder_sync, uid_value, ERROR_DOMAIN_FETCH,
        error);
    ETPAN_ERROR_FREE(error);
    error = new_error;
    goto exit;
  }
  
  r = etpan_sqldb_get(folder_sync->body_db, part_id,
      "content", &data, &length);
  body_db_close(folder_sync);
  
  if (r < 0) {
    char * description;
        
    ETPAN_WARN_LOG("database corrupted");
      
    description = get_description(folder_sync, uid_value);
        
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_DATABASE);
    etpan_error_set_short_description(error, _("Cache could not be written"));
    etpan_error_strf_long_description(error,
        _("An error occurred while gettting content of the following message:\n"
            "%s"
            "Cache of mailbox %s could not be written properly on disk."),
        description,
        folder_sync->location);
        
    free(description);
    
    goto exit;
  }
  
  body = etpan_imap_body_unserialize(data, length);
  free(data);
  
  body_data->result = body;
  error = NULL;
  
 exit:
  body_data->error = error;
}

static void get_body_in_thread_callback(int cancelled, void * dummy, void * data)
{
  struct body_data * body_data;
  
  body_data = data;
  mailsem_up(body_data->sem);
}

struct part_data {
  struct mailsem * sem;
  struct etpan_imap_folder_sync * folder_sync;
  uint32_t uid_value;
  char * part_id;
  char * section;
  char * content;
  size_t length;
  int part_type;
  struct etpan_error * error;
};

static void get_part_in_main_thread(void * data);

struct etpan_error *
etpan_imap_folder_sync_get_part(struct etpan_imap_folder_sync * folder_sync,
    char * uid, char * section, int part_type,
    void ** p_data, size_t * p_length)
{
  struct etpan_error * error;
  int r;
  char part_id[256];
  void * data;
  size_t length;
  char * content;
  size_t content_length;
  char * suffix;
  uint32_t uid_value;
  struct part_data * part_data;
  char * part_id_dup;
  
  uid_value = strtoul(uid, NULL, 10);
  
  switch (part_type) {
  case ETPAN_IMAP_PART_TYPE_MAIN:
    suffix = "";
    break;
  case ETPAN_IMAP_PART_TYPE_HEADER:
    suffix = "-header";
    break;
  case ETPAN_IMAP_PART_TYPE_MIME:
    suffix = "-mime";
    break;
  default:
    suffix = NULL;
    ETPAN_CRASH("invalid part type");
    break;
  }
  
  error = body_db_open(folder_sync);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(folder_sync, uid_value, ERROR_DOMAIN_FETCH,
        error);
    ETPAN_ERROR_FREE(error);
    error = new_error;
    goto exit;
  }
  
  snprintf(part_id, sizeof(part_id), "%s-%s%s", uid, section, suffix);
  ETPAN_LOCAL_LOG("part id : %s", part_id);
  r = etpan_sqldb_get(folder_sync->body_db, part_id,
      "content", &data, &length);
  
  body_db_close(folder_sync);
  
  if (r == 0) {
    decode_str(data, length, &content, &content_length);
    free(data);
    ETPAN_LOCAL_LOG("data: %p %lu", content, (unsigned long) content_length);
    
    * p_data = content;
    * p_length = content_length;
    
    error = NULL;
    
    goto exit;
  }

  part_id_dup = strdup(part_id);
  if (part_id_dup == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  part_data = malloc(sizeof(* part_data));
  if (part_data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  part_data->folder_sync = folder_sync;
  part_data->uid_value = uid_value;
  part_data->part_id = part_id_dup;
  part_data->part_type = part_type;
  part_data->section = strdup(section);
  if (part_data->section == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  part_data->sem = mailsem_new();
  if (part_data->sem == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  part_data->content = NULL;
  part_data->length = 0;
  part_data->error = NULL;
  
  etpan_thread_manager_app_run_in_main_thread(etpan_thread_manager_app_get_default(), get_part_in_main_thread, part_data, 0);
  
  mailsem_down(part_data->sem);
  
  content = part_data->content;
  length = part_data->length;
  error = part_data->error;
  
  mailsem_free(part_data->sem);
  free(part_data->section);
  free(part_data->part_id);
  free(part_data);
  
  if (error != NULL)
    goto exit;
  
  * p_data = content;
  * p_length = length;
  
 exit:
  return error;
}

static void get_part_in_thread(struct etpan_thread_op * op);
static void get_part_in_thread_callback(int cancelled, void * dummy, void * data);

static void get_part_in_main_thread(void * data)
{
  struct etpan_thread_op * op;
  struct etpan_storage * storage;
  struct etpan_imap_sync * imap_sync;
  struct part_data * part_data;
  struct etpan_imap_folder_sync * folder_sync;
  
  part_data = data;
  folder_sync = part_data->folder_sync;
  
  op = etpan_thread_op_new();
  op->param = part_data;
  op->result = NULL;
  
  op->cancellable = 1;
  op->run = get_part_in_thread;
  op->callback = (void (*)(int, void *, void *)) get_part_in_thread_callback;
  op->callback_data = part_data;
  op->cleanup = NULL;
  
  imap_sync = folder_sync->imap_sync;
  storage = etpan_imap_sync_get_thread_storage(imap_sync);
  etpan_thread_manager_app_storage_schedule(etpan_thread_manager_app_get_default(), storage, op);
}

static void get_part_in_thread(struct etpan_thread_op * op)
{
  struct mailimap_section * imap_section;
  char * content;
  size_t content_length;
  struct part_data * part_data;
  struct etpan_error * error;
  struct etpan_imap_folder_sync * folder_sync;
  char * part_id;
  uint32_t uid_value;
  int part_type;
  char * section;
  void * data;
  size_t length;
  int had_error;
  int r;
  
  part_data = op->param;
  folder_sync = part_data->folder_sync;
  uid_value = part_data->uid_value;
  part_id = part_data->part_id;
  part_type = part_data->part_type;
  section = part_data->section;
  
  content = NULL;
  content_length = 0;
  
#if 0
  error = connect(folder_sync);
  if (error != NULL) {
    struct etpan_error * new_error;
      
    new_error = rewrite_error(folder_sync, uid_value, ERROR_DOMAIN_FETCH,
        error);
    ETPAN_ERROR_FREE(error);
    error = new_error;
    goto exit;
  }
#endif
  
  error = select_folder(folder_sync);
  if (error != NULL) {
    goto exit;
  }
  
  switch (part_type) {
  case ETPAN_IMAP_PART_TYPE_MAIN:
    imap_section = get_imap_section_from_string(section, EP_SECTION_MESSAGE);
    break;
  case ETPAN_IMAP_PART_TYPE_HEADER:
    imap_section = get_imap_section_from_string(section, EP_SECTION_HEADER);
    break;
  case ETPAN_IMAP_PART_TYPE_MIME:
    imap_section = get_imap_section_from_string(section, EP_SECTION_MIME);
    break;
  default:
    imap_section = NULL;
    ETPAN_CRASH("invalid");
    break;
  }
  
  error = imap_fetch_section(folder_sync,
      uid_value, imap_section, &content, &content_length);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(folder_sync, uid_value, ERROR_DOMAIN_FETCH,
        error);
    ETPAN_ERROR_FREE(error);
    error = new_error;
    goto exit;
  }
  ETPAN_LOCAL_LOG("fetch data %p %lu", content, (unsigned long) content_length);
  encode_str(content, content_length, &data, &length);
  
  error = body_db_open(folder_sync);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(folder_sync, uid_value, ERROR_DOMAIN_FETCH,
        error);
    ETPAN_ERROR_FREE(error);
    error = new_error;
    free(data);
    free(content);
    goto exit;
  }
  had_error = 0;
  r = etpan_sqldb_begin_transaction(folder_sync->body_db);
  if (r < 0)
    had_error = 1;
  r = etpan_sqldb_set(folder_sync->body_db, part_id,
      "content", data, length);
  if (r < 0)
    had_error = 1;
  r = etpan_sqldb_end_transaction(folder_sync->body_db);
  if (r < 0)
    had_error = 1;
  
  body_db_close(folder_sync);
  free(data);
  
  if (had_error) {
    char * description;
    
    description = get_description(folder_sync, uid_value);
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_DATABASE);
    etpan_error_set_short_description(error, _("Cache could not be written"));
    etpan_error_strf_long_description(error,
        _("An error occurred while gettting content of the following message:\n"
            "%s"
            "Cache of mailbox %s could not be written properly on disk."),
        description,
        folder_sync->location);
    
    free(description);
    
    free(content);
    
    goto exit;
  }
  
  ETPAN_LOCAL_LOG("data: %p %lu", content, (unsigned long) content_length);
  
  error = NULL;
  part_data->content = content;
  part_data->length = content_length;
  
 exit:
  part_data->error = error;
}

static void get_part_in_thread_callback(int cancelled, void * dummy, void * data)
{
  struct part_data * part_data;
  
  part_data = data;
  mailsem_up(part_data->sem);
}

struct etpan_error *
etpan_imap_folder_sync_check(struct etpan_imap_folder_sync * folder_sync)
{
  struct etpan_error * error;
  int r;
  unsigned int msg_count;
  unsigned int unseen_count;
  unsigned int recent_count;
  chashiter * iter;
  int had_error;
  
  pthread_mutex_lock(&folder_sync->check_requested_lock);
  folder_sync->check_requested = 1;
  pthread_mutex_unlock(&folder_sync->check_requested_lock);
  
  msg_count = 0;
  unseen_count = 0;
  recent_count = 0;
  etpan_imap_folder_sync_status(folder_sync,
      &msg_count, &unseen_count, &recent_count);
  folder_sync->msg_count = msg_count;
  folder_sync->unseen_count = unseen_count;
  folder_sync->recent_count = recent_count;
  
  error = header_db_open(folder_sync);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(folder_sync, 0, ERROR_DOMAIN_SYNC,
        error);
    ETPAN_ERROR_FREE(error);
    error = new_error;
    goto exit;
  }
  
  pthread_mutex_lock(&folder_sync->pending_flags_lock);
  had_error = 0;
  r = etpan_sqldb_begin_transaction(folder_sync->header_db);
  if (r < 0)
    had_error = 1;
  ETPAN_LOCAL_LOG("checking %u messages %s", chash_count(folder_sync->pending_flags),
      folder_sync->location);
  for(iter = chash_begin(folder_sync->pending_flags) ; iter != NULL ;
      iter = chash_next(folder_sync->pending_flags, iter)) {
    chashdatum key;
    chashdatum value;
    void * data;
    size_t length;
    struct flags_pair * pair;
    char * uid;
    struct etpan_message_flags * original_flags;
    
    chash_key(iter, &key);
    chash_value(iter, &value);
    pair = value.data;
    uid = key.data;
    
    r = etpan_sqldb_get(folder_sync->header_db, uid, "flags", &data, &length);
    if (r < 0) {
      ETPAN_WARN_LOG("database corrupted");
      continue;
    }
    
    original_flags = etpan_message_flags_unserialize(data, length);
    
    etpan_message_flags_serialize(pair->flags, &data, &length);
    r = etpan_sqldb_set(folder_sync->header_db, uid, "flags", data, length);
    if (r < 0)
      had_error = 1;
    free(data);
    
    update_count_with_flags(folder_sync, original_flags, 1);
    update_count_with_flags(folder_sync, pair->flags, 0);
    
    etpan_message_flags_free(original_flags);
  }
  r = etpan_sqldb_end_transaction(folder_sync->header_db);
  if (r < 0)
    had_error = 1;
  pthread_mutex_unlock(&folder_sync->pending_flags_lock);
  
  header_db_close(folder_sync);

  if (had_error) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_DATABASE);
    etpan_error_set_short_description(error, _("Cache could not be written"));
    etpan_error_strf_long_description(error,
        _("Cache of mailbox %s could not be written properly on disk."),
        folder_sync->location);
    
    goto exit;
  }
  
  msg_count = folder_sync->msg_count;
  unseen_count = folder_sync->unseen_count;
  recent_count = folder_sync->recent_count;
  error = etpan_imap_sync_set_status(folder_sync->imap_sync,
      folder_sync->location,
      msg_count, unseen_count, recent_count);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(folder_sync, 0, ERROR_DOMAIN_SYNC,
        error);
    ETPAN_ERROR_FREE(error);
    error = new_error;
    goto exit;
  }
  
  error = NULL;
  
 exit:
  return error;
}


struct etpan_error *
etpan_imap_folder_sync_append(struct etpan_imap_folder_sync * folder_sync,
    char * content, size_t length, struct etpan_message_flags * flags)
{
  int r;
  struct etpan_error * error;
  char * uid;
  void * data;
  size_t data_length;
  int had_error;
  
  error = append_db_open(folder_sync);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(folder_sync, 0, ERROR_DOMAIN_SYNC,
        error);
    ETPAN_ERROR_FREE(error);
    error = new_error;
    goto exit;
  }
  
  uid = etpan_uuid_generate();
  
  had_error = 0;
  r = etpan_sqldb_begin_transaction(folder_sync->append_db);
  if (r < 0)
    had_error = 1;
  
  encode_str(content, length, &data, &data_length);
  r = etpan_sqldb_set(folder_sync->append_db, uid, "content",
      data, data_length);
  if (r < 0)
    had_error = 1;
  free(data);
  
  etpan_message_flags_serialize(flags, &data, &data_length);
  r = etpan_sqldb_set(folder_sync->append_db, uid, "flags",
      data, data_length);
  if (r < 0)
    had_error = 1;
  free(data);
  
  r = etpan_sqldb_end_transaction(folder_sync->append_db);
  if (r < 0)
    had_error = 1;
  
  free(uid);
  
  append_db_close(folder_sync);
  
  if (had_error) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_DATABASE);
    etpan_error_set_short_description(error, _("Cache could not be written"));
    etpan_error_strf_long_description(error,
        _("An error occurred while adding a message to the mailbox.\n"
            "Cache of mailbox %s could not be written properly on disk."),
        folder_sync->location);
    
    goto exit;
  }
  
  pthread_mutex_lock(&folder_sync->append_lock);
  folder_sync->append_has_items = 1;
  pthread_mutex_unlock(&folder_sync->append_lock);
  
  error = NULL;
  
 exit:
  return error;
}

void etpan_imap_folder_sync_open(struct etpan_imap_folder_sync * folder_sync)
{
  folder_sync->opened ++;
}

void etpan_imap_folder_sync_close(struct etpan_imap_folder_sync * folder_sync)
{
  folder_sync->opened --;
}

struct etpan_error * etpan_imap_folder_sync_get_error(struct etpan_imap_folder_sync * folder_sync)
{
  return folder_sync->error;
}

static struct etpan_error *
rewrite_error(struct etpan_imap_folder_sync * folder_sync, uint32_t uid,
    int domain,
    struct etpan_error * error)
{
  struct etpan_error * new_error;
  char * previous_long_description;
  
  new_error = etpan_error_new();
  etpan_error_set_code(new_error, etpan_error_get_code(error));
  etpan_error_set_short_description(new_error,
      etpan_error_get_short_description(error));
  previous_long_description = etpan_error_get_long_description(error);
  
  switch (domain) {
  case ERROR_DOMAIN_FETCH:
    {
      char * description;
      
      description = get_description(folder_sync, uid);
      etpan_error_strf_long_description(new_error,
          _("An error occurred while gettting content of the following message:\n"
              "%s"
              "%s"),
          description,
          previous_long_description);
      free(description);
    }
    break;
  case ERROR_DOMAIN_FETCH_MSG_LIST:
    etpan_error_strf_long_description(new_error,
        _("An error occurred while getting the list of messages of the mailbox %s.\n"
            "%s"),
        folder_sync->location,
        previous_long_description);
    break;
  case ERROR_DOMAIN_SYNC:
    etpan_error_strf_long_description(new_error,
        _("An error occurred while synchronizing account %s.\n"
            "%s"),
        etpan_imap_sync_get_id(folder_sync->imap_sync),
        previous_long_description);
    break;
    
  default:
    etpan_error_strf_long_description(new_error,
        _("An error occurred with account %s.\n"
            "%s"),
        etpan_imap_sync_get_id(folder_sync->imap_sync),
        previous_long_description);
    break;
  }
  
  return new_error;
}

static void disconnect(struct etpan_imap_folder_sync * folder_sync)
{
  struct etpan_imap_sync * imap_sync;
  
  if (!folder_sync->connected)
    return;
  
#if 0
  mailfolder_disconnect(folder_sync->ep_folder);
  mailfolder_free(folder_sync->ep_folder);
  folder_sync->ep_folder = NULL;
#endif
  
  imap_sync = folder_sync->imap_sync;
  imap_sync->connected_count --;
}

void etpan_imap_folder_sync_disconnect_nt(struct etpan_imap_folder_sync * folder_sync)
{
  disconnect(folder_sync);
}

static void notify_folder_content_updated(struct etpan_imap_folder_sync * folder_sync)
{
  ETPAN_SIGNAL_SEND(folder_sync,
      ETPAN_IMAP_FOLDER_SYNC_CONTENT_UPDATED_SIGNAL);
}
