#include "etpan-imap-serialize.h"

#include <string.h>
#include <stdlib.h>

#include "etpan-serialize.h"
#include "etpan-log.h"

static struct etpan_serialize_data * encode_body(struct mailimap_body * body);
static struct mailimap_body * decode_body(struct etpan_serialize_data * hash);
static struct etpan_serialize_data * encode_body_1part(struct mailimap_body_type_1part * b_1part);
static struct mailimap_body_type_1part * decode_body_1part(struct etpan_serialize_data * hash);
static struct etpan_serialize_data * encode_body_mpart(struct mailimap_body_type_mpart * part);
static struct mailimap_body_type_mpart * decode_body_mpart(struct etpan_serialize_data * hash);
static struct etpan_serialize_data * encode_body_list(clist * list);
static clist * decode_body_list(struct etpan_serialize_data * tab);
static struct etpan_serialize_data * encode_body_basic(struct mailimap_body_type_basic * body);
struct mailimap_body_type_basic * decode_body_basic(struct etpan_serialize_data * hash);
static struct etpan_serialize_data * encode_body_msg(struct mailimap_body_type_msg * body);
struct mailimap_body_type_msg * decode_body_msg(struct etpan_serialize_data * hash);
static struct etpan_serialize_data * encode_body_text(struct mailimap_body_type_text * body);
struct mailimap_body_type_text * decode_body_text(struct etpan_serialize_data * hash);
static struct etpan_serialize_data * encode_body_ext_1part(struct mailimap_body_ext_1part * ext);
static struct mailimap_body_ext_1part * decode_body_ext_1part(struct etpan_serialize_data * hash);
static struct etpan_serialize_data * encode_body_ext_mpart(struct mailimap_body_ext_mpart * ext);
static struct mailimap_body_ext_mpart * decode_body_ext_mpart(struct etpan_serialize_data * hash);
static struct etpan_serialize_data * encode_media_basic(struct mailimap_media_basic * media_basic);
struct mailimap_media_basic * decode_media_basic(struct etpan_serialize_data * hash);
static struct etpan_serialize_data * encode_body_fields(struct mailimap_body_fields * body_fields);
static struct mailimap_body_fields * decode_body_fields(struct etpan_serialize_data * hash);
static struct etpan_serialize_data * encode_body_fld_param(struct mailimap_body_fld_param * param);
static struct mailimap_body_fld_param * decode_body_fld_param(struct etpan_serialize_data * tab);
static struct etpan_serialize_data * encode_single_body_fld_param(struct mailimap_single_body_fld_param * param);
static struct mailimap_single_body_fld_param * decode_single_body_fld_param(struct etpan_serialize_data * hash);
static struct etpan_serialize_data * encode_body_fld_enc(struct mailimap_body_fld_enc * enc);
static struct mailimap_body_fld_enc * decode_body_fld_enc(struct etpan_serialize_data * hash);
static struct etpan_serialize_data * encode_envelope(struct mailimap_envelope * envelope);
static struct mailimap_envelope * decode_envelope(struct etpan_serialize_data * hash);
static struct etpan_serialize_data * encode_body_fld_dsp(struct mailimap_body_fld_dsp * dsp);
static struct mailimap_body_fld_dsp * decode_body_fld_dsp(struct etpan_serialize_data * hash);
static struct etpan_serialize_data * encode_body_fld_lang(struct mailimap_body_fld_lang * lang);
static struct mailimap_body_fld_lang * decode_body_fld_lang(struct etpan_serialize_data * hash);
static struct etpan_serialize_data * encode_body_ext(struct mailimap_body_extension * ext);
static struct mailimap_body_extension * decode_body_ext(struct etpan_serialize_data * hash);
static struct etpan_serialize_data * encode_body_ext_list(clist * body_ext_list);
static clist * decode_body_ext_list(struct etpan_serialize_data * tab);
static struct etpan_serialize_data * encode_address(struct mailimap_address * addr);
static struct mailimap_address * decode_address(struct etpan_serialize_data * hash);
static struct etpan_serialize_data * encode_address_list(clist * addr_list);
static clist * decode_address_list(struct etpan_serialize_data * tab);


#if 0
struct mailimap_body {
  int bd_type;
  /* can be MAILIMAP_BODY_1PART or MAILIMAP_BODY_MPART */
  union {
    struct mailimap_body_type_1part * bd_body_1part; /* can be NULL */
    struct mailimap_body_type_mpart * bd_body_mpart; /* can be NULL */
  } bd_data;
};
#endif

static struct etpan_serialize_data * encode_body(struct mailimap_body * body)
{
  struct etpan_serialize_data * hash;
  struct etpan_serialize_data * data;
  
  hash = etpan_serialize_data_new_hash();
  data = NULL;
  switch (body->bd_type) {
  case MAILIMAP_BODY_1PART:
    data = encode_body_1part(body->bd_data.bd_body_1part);
    break;
  case MAILIMAP_BODY_MPART:
    data = encode_body_mpart(body->bd_data.bd_body_mpart);
    break;
  }
  
  etpan_serialize_hash_set_int32(hash, "type", body->bd_type);
  etpan_serialize_hash_set(hash, "part", data);
  
  return hash;
}

static struct mailimap_body * decode_body(struct etpan_serialize_data * hash)
{
  int type;
  struct etpan_serialize_data * data;
  struct mailimap_body_type_1part * b_1part;
  struct mailimap_body_type_mpart * b_mpart;
  struct mailimap_body * body;
  
  type = etpan_serialize_hash_get_int32(hash, "type");
  data = etpan_serialize_hash_get(hash, "part");
  
  b_1part = NULL;
  b_mpart = NULL;
  switch (type) {
  case MAILIMAP_BODY_1PART:
    b_1part = decode_body_1part(data);
    break;
  case MAILIMAP_BODY_MPART:
    b_mpart = decode_body_mpart(data);
    break;
  }
  
  body = mailimap_body_new(type, b_1part, b_mpart);
  if (body == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return body;
}

#if 0
struct mailimap_body_type_1part {
  int bd_type;
  union {
    struct mailimap_body_type_basic * bd_type_basic; /* can be NULL */
    struct mailimap_body_type_msg * bd_type_msg;     /* can be NULL */
    struct mailimap_body_type_text * bd_type_text;   /* can be NULL */
  } bd_data;
  struct mailimap_body_ext_1part * bd_ext_1part;   /* can be NULL */
};
#endif

static struct etpan_serialize_data * encode_body_1part(struct mailimap_body_type_1part * b_1part)
{
  struct etpan_serialize_data * hash;
  struct etpan_serialize_data * data;
  struct etpan_serialize_data * ext;
  
  hash = etpan_serialize_data_new_hash();
  data = NULL;
  switch (b_1part->bd_type) {
  case MAILIMAP_BODY_TYPE_1PART_BASIC:
    data = encode_body_basic(b_1part->bd_data.bd_type_basic);
    break;
  case MAILIMAP_BODY_TYPE_1PART_MSG:
    data = encode_body_msg(b_1part->bd_data.bd_type_msg);
    break;
  case MAILIMAP_BODY_TYPE_1PART_TEXT:
    data = encode_body_text(b_1part->bd_data.bd_type_text);
    break;
  }
  etpan_serialize_hash_set_int32(hash, "type", b_1part->bd_type);
  etpan_serialize_hash_set(hash, "part", data);
  
  if (b_1part->bd_ext_1part != NULL) {
    ext = encode_body_ext_1part(b_1part->bd_ext_1part);
    etpan_serialize_hash_set(hash, "ext", ext);
  }
  
  return hash;
}

static struct mailimap_body_type_1part * decode_body_1part(struct etpan_serialize_data * hash)
{
  int type;
  struct mailimap_body_type_1part * body;
  struct mailimap_body_type_basic * b_basic;
  struct mailimap_body_type_msg * b_msg;
  struct mailimap_body_type_text * b_text;
  struct mailimap_body_ext_1part * b_ext;
  struct etpan_serialize_data * data;
  struct etpan_serialize_data * ext;
  
  type = etpan_serialize_hash_get_int32(hash, "type");
  data = etpan_serialize_hash_get(hash, "part");
  ext = etpan_serialize_hash_get(hash, "ext");
  b_basic = NULL;
  b_msg = NULL;
  b_text = NULL;
  switch (type) {
  case MAILIMAP_BODY_TYPE_1PART_BASIC:
    b_basic = decode_body_basic(data);
    break;
  case MAILIMAP_BODY_TYPE_1PART_MSG:
    b_msg = decode_body_msg(data);
    break;
  case MAILIMAP_BODY_TYPE_1PART_TEXT:
    b_text = decode_body_text(data);
    break;
  }
  
  b_ext = NULL;
  if (ext != NULL)
    b_ext = decode_body_ext_1part(ext);
  
  body = mailimap_body_type_1part_new(type, b_basic, b_msg, b_text, b_ext);
  if (body == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return body;
}

#if 0
struct mailimap_body_type_mpart {
  clist * bd_list; /* list of (struct mailimap_body *) */
                     /* != NULL */
  char * bd_media_subtype; /* != NULL */
  struct mailimap_body_ext_mpart * bd_ext_mpart; /* can be NULL */
};
#endif

static struct etpan_serialize_data * encode_body_mpart(struct mailimap_body_type_mpart * part)
{
  struct etpan_serialize_data * hash;
  struct etpan_serialize_data * encoded_list;
  struct etpan_serialize_data * ext;
  
  hash = etpan_serialize_data_new_hash();
  encoded_list = encode_body_list(part->bd_list);
  etpan_serialize_hash_set(hash, "body-list", encoded_list);
  etpan_serialize_hash_set_str(hash, "subtype", part->bd_media_subtype);
  if (part->bd_ext_mpart != NULL) {
    ext = encode_body_ext_mpart(part->bd_ext_mpart);
    etpan_serialize_hash_set(hash, "ext", ext);
  }
  
  return hash;
}

static struct mailimap_body_type_mpart * decode_body_mpart(struct etpan_serialize_data * hash)
{
  struct etpan_serialize_data * encoded_body_list;
  clist * body_list;
  char * subtype;
  struct mailimap_body_ext_mpart * b_ext;
  struct mailimap_body_type_mpart * body;
  struct etpan_serialize_data * ext;
  
  encoded_body_list = etpan_serialize_hash_get(hash, "body-list");
  body_list = decode_body_list(encoded_body_list);
  subtype = etpan_serialize_hash_get_str(hash, "subtype");
  subtype = strdup(subtype);
  if (subtype == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  b_ext = NULL;
  ext = etpan_serialize_hash_get(hash, "ext");
  if (ext != NULL) {
    b_ext = decode_body_ext_mpart(ext);
  }
  
  body = mailimap_body_type_mpart_new(body_list, subtype, b_ext);
  if (body == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return body;
}

static struct etpan_serialize_data * encode_body_list(clist * list)
{
  struct etpan_serialize_data * tab;
  clistiter * iter;
  
  tab = etpan_serialize_data_new_array();
  for(iter = clist_begin(list) ; iter != NULL ; iter = clist_next(iter)) {
    struct mailimap_body * body;
    struct etpan_serialize_data * data;
    
    body = clist_content(iter);
    data = encode_body(body);
    etpan_serialize_array_add(tab, data);
  }
  
  return tab;
}

static clist * decode_body_list(struct etpan_serialize_data * tab)
{
  clist * list;
  int r;
  unsigned int i;
  
  list = clist_new();
  for(i = 0 ; i < etpan_serialize_array_count(tab) ; i ++) {
    struct etpan_serialize_data * data;
    struct mailimap_body * body;
    
    data = etpan_serialize_array_get(tab, i);
    body = decode_body(data);
    r = clist_append(list, body);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  return list;
}

#if 0
struct mailimap_body_type_basic {
  struct mailimap_media_basic * bd_media_basic; /* != NULL */
  struct mailimap_body_fields * bd_fields; /* != NULL */
};
#endif

static struct etpan_serialize_data * encode_body_basic(struct mailimap_body_type_basic * body)
{
  struct etpan_serialize_data * hash;
  struct etpan_serialize_data * data;
  
  hash = etpan_serialize_data_new_hash();
  data = encode_media_basic(body->bd_media_basic);
  etpan_serialize_hash_set(hash, "media", data);
  data = encode_body_fields(body->bd_fields);
  etpan_serialize_hash_set(hash, "fields", data);
  
  return hash;
}

struct mailimap_body_type_basic * decode_body_basic(struct etpan_serialize_data * hash)
{
  struct etpan_serialize_data * data;
  struct mailimap_media_basic * media;
  struct mailimap_body_fields * fields;
  struct mailimap_body_type_basic * body;
  
  data = etpan_serialize_hash_get(hash, "media");
  media = decode_media_basic(data);
  data = etpan_serialize_hash_get(hash, "fields");
  fields = decode_body_fields(data);
  
  body = mailimap_body_type_basic_new(media, fields);
  if (body == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return body;
}

#if 0
struct mailimap_body_type_msg {
  struct mailimap_body_fields * bd_fields; /* != NULL */
  struct mailimap_envelope * bd_envelope;       /* != NULL */
  struct mailimap_body * bd_body;               /* != NULL */
  uint32_t bd_lines;
};
#endif

static struct etpan_serialize_data * encode_body_msg(struct mailimap_body_type_msg * body)
{
  struct etpan_serialize_data * hash;
  struct etpan_serialize_data * data;
  
  hash = etpan_serialize_data_new_hash();
  data = encode_body_fields(body->bd_fields);
  etpan_serialize_hash_set(hash, "fields", data);
  data = encode_envelope(body->bd_envelope);
  etpan_serialize_hash_set(hash, "envelope", data);
  data = encode_body(body->bd_body);
  etpan_serialize_hash_set(hash, "body", data);
  etpan_serialize_hash_set_uint32(hash, "lines", body->bd_lines);
  
  return hash;
}

struct mailimap_body_type_msg * decode_body_msg(struct etpan_serialize_data * hash)
{
  struct etpan_serialize_data * data;
  struct mailimap_body_fields * bd_fields;
  struct mailimap_envelope * bd_envelope;
  struct mailimap_body * bd_body;
  uint32_t bd_lines;
  struct mailimap_body_type_msg * body;
  
  data = etpan_serialize_hash_get(hash, "fields");
  bd_fields = decode_body_fields(data);
  data = etpan_serialize_hash_get(hash, "envelope");
  bd_envelope = decode_envelope(data);
  data = etpan_serialize_hash_get(hash, "body");
  bd_body = decode_body(data);
  bd_lines = etpan_serialize_hash_get_uint32(hash, "lines");
  
  body = mailimap_body_type_msg_new(bd_fields, bd_envelope, bd_body, bd_lines);
  if (body == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return body;
}

#if 0
struct mailimap_body_type_text {
  char * bd_media_text;                         /* != NULL */
  struct mailimap_body_fields * bd_fields; /* != NULL */
  uint32_t bd_lines;
};
#endif

static struct etpan_serialize_data * encode_body_text(struct mailimap_body_type_text * body)
{
  struct etpan_serialize_data * hash;
  struct etpan_serialize_data * data;
  
  hash = etpan_serialize_data_new_hash();
  etpan_serialize_hash_set_str(hash, "media", body->bd_media_text);
  data = encode_body_fields(body->bd_fields);
  etpan_serialize_hash_set(hash, "fields", data);
  etpan_serialize_hash_set_uint32(hash, "lines", body->bd_lines);
  
  return hash;
}

struct mailimap_body_type_text * decode_body_text(struct etpan_serialize_data * hash)
{
  struct etpan_serialize_data * data;
  char * bd_media_text;
  struct mailimap_body_fields * bd_fields;
  uint32_t bd_lines;
  struct mailimap_body_type_text * body;
  
  bd_media_text = etpan_serialize_hash_get_str(hash, "media");
  bd_media_text = strdup(bd_media_text);
  if (bd_media_text == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  data = etpan_serialize_hash_get(hash, "fields");
  bd_fields = decode_body_fields(data);
  bd_lines = etpan_serialize_hash_get_uint32(hash, "lines");
  
  body = mailimap_body_type_text_new(bd_media_text, bd_fields, bd_lines);
  if (body == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return body;
}

#if 0
struct mailimap_body_ext_1part {
  char * bd_md5;   /* can be NULL */
  struct mailimap_body_fld_dsp * bd_disposition; /* can be NULL */
  struct mailimap_body_fld_lang * bd_language;   /* can be NULL */
  
  clist * bd_extension_list; /* list of (struct mailimap_body_extension *) */
                               /* can be NULL */
};
#endif

static struct etpan_serialize_data * encode_body_ext_1part(struct mailimap_body_ext_1part * ext)
{
  struct etpan_serialize_data * hash;
  struct etpan_serialize_data * data;
  
  hash = etpan_serialize_data_new_hash();
  etpan_serialize_hash_set_str(hash, "md5", ext->bd_md5);
  
  if (ext->bd_disposition != NULL) {
    data = encode_body_fld_dsp(ext->bd_disposition);
    etpan_serialize_hash_set(hash, "disposition", data);
  }
  if (ext->bd_language != NULL) {
    data = encode_body_fld_lang(ext->bd_language);
    etpan_serialize_hash_set(hash, "language", data);
  }
  if (ext->bd_extension_list != NULL) {
    data = encode_body_ext_list(ext->bd_extension_list);
    etpan_serialize_hash_set(hash, "ext-list", data);
  }
  
  return hash;
}

static struct mailimap_body_ext_1part * decode_body_ext_1part(struct etpan_serialize_data * hash)
{
  char * bd_md5;
  struct mailimap_body_fld_dsp * bd_disposition;
  struct mailimap_body_fld_lang * bd_language;
  clist * bd_extension_list;
  struct etpan_serialize_data * data;
  struct mailimap_body_ext_1part * bd_ext_1part;
  
  bd_md5 = NULL;
  bd_disposition = NULL;
  bd_language = NULL;
  bd_extension_list = NULL;
  
  bd_md5 = etpan_serialize_hash_get_str(hash, "md5");
  if (bd_md5 != NULL) {
    bd_md5 = strdup(bd_md5);
    if (bd_md5 == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  data = etpan_serialize_hash_get(hash, "disposition");
  if (data != NULL) {
    bd_disposition = decode_body_fld_dsp(data);
  }
  data = etpan_serialize_hash_get(hash, "language");
  if (data != NULL) {
    bd_language = decode_body_fld_lang(data);
  }
  data = etpan_serialize_hash_get(hash, "ext-list");
  if (data != NULL) {
    bd_extension_list = decode_body_ext_list(data);
  }
  
  bd_ext_1part = mailimap_body_ext_1part_new(bd_md5, bd_disposition,
      bd_language, bd_extension_list);
  if (bd_ext_1part == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return bd_ext_1part;
}

#if 0
struct mailimap_body_ext_mpart {
  struct mailimap_body_fld_param * bd_parameter; /* can be NULL */
  struct mailimap_body_fld_dsp * bd_disposition; /* can be NULL */
  struct mailimap_body_fld_lang * bd_language;   /* can be NULL */
  clist * bd_extension_list; /* list of (struct mailimap_body_extension *) */
                               /* can be NULL */
};
#endif

static struct etpan_serialize_data * encode_body_ext_mpart(struct mailimap_body_ext_mpart * ext)
{
  struct etpan_serialize_data * hash;
  struct etpan_serialize_data * data;
  
  hash = etpan_serialize_data_new_hash();
  
  if (ext->bd_parameter != NULL) {
    data = encode_body_fld_param(ext->bd_parameter);
    etpan_serialize_hash_set(hash, "parameter", data);
  }
  if (ext->bd_disposition != NULL) {
    data = encode_body_fld_dsp(ext->bd_disposition);
    etpan_serialize_hash_set(hash, "disposition", data);
  }
  if (ext->bd_language != NULL) {
    data = encode_body_fld_lang(ext->bd_language);
    etpan_serialize_hash_set(hash, "language", data);
  }
  if (ext->bd_extension_list != NULL) {
    data = encode_body_ext_list(ext->bd_extension_list);
    etpan_serialize_hash_set(hash, "ext-list", data);
  }
  
  return hash;
}

static struct mailimap_body_ext_mpart * decode_body_ext_mpart(struct etpan_serialize_data * hash)
{
  struct etpan_serialize_data * data;
  struct mailimap_body_fld_param * bd_parameter;
  struct mailimap_body_fld_dsp * bd_disposition;
  struct mailimap_body_fld_lang * bd_language;
  clist * bd_extension_list;
  struct mailimap_body_ext_mpart * body;
  
  bd_parameter = NULL;
  bd_disposition = NULL;
  bd_language = NULL;
  bd_extension_list = NULL;
  data = etpan_serialize_hash_get(hash, "parameter");
  if (data != NULL) {
    bd_parameter = decode_body_fld_param(data);
  }
  data = etpan_serialize_hash_get(hash, "disposition");
  if (data != NULL) {
    bd_disposition = decode_body_fld_dsp(data);
  }
  data = etpan_serialize_hash_get(hash, "language");
  if (data != NULL) {
    bd_language = decode_body_fld_lang(data);
  }
  data = etpan_serialize_hash_get(hash, "ext-list");
  if (data != NULL) {
    bd_extension_list = decode_body_ext_list(data);
  }
  
  body = mailimap_body_ext_mpart_new(bd_parameter,
      bd_disposition, bd_language, bd_extension_list);
  if (body == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return body;
}

#if 0
struct mailimap_media_basic {
  int med_type;
  char * med_basic_type; /* can be NULL */
  char * med_subtype;    /* != NULL */
};
#endif

static struct etpan_serialize_data * encode_media_basic(struct mailimap_media_basic * media_basic)
{
  struct etpan_serialize_data * hash;
  
  hash = etpan_serialize_data_new_hash();
  etpan_serialize_hash_set_int32(hash, "type", media_basic->med_type);
  etpan_serialize_hash_set_str(hash, "basic-type",
      media_basic->med_basic_type);
  etpan_serialize_hash_set_str(hash, "subtype",
      media_basic->med_subtype);
  
  return hash;
}

struct mailimap_media_basic * decode_media_basic(struct etpan_serialize_data * hash)
{
  struct mailimap_media_basic * media_basic;
  int type;
  char * basic_type;
  char * subtype;
  
  type = etpan_serialize_hash_get_int32(hash, "type");
  basic_type = etpan_serialize_hash_get_str(hash, "basic-type");
  if (basic_type != NULL) {
    basic_type = strdup(basic_type);
    if (basic_type == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  subtype = etpan_serialize_hash_get_str(hash, "subtype");
  if (subtype != NULL) {
    subtype = strdup(subtype);
    if (subtype == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  media_basic = mailimap_media_basic_new(type, basic_type, subtype);
  if (media_basic == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return media_basic;
}

#if 0
struct mailimap_body_fields {
  struct mailimap_body_fld_param * bd_parameter; /* can be NULL */
  char * bd_id;                                  /* can be NULL */
  char * bd_description;                         /* can be NULL */
  struct mailimap_body_fld_enc * bd_encoding;    /* != NULL */
  uint32_t bd_size;
};
#endif

static struct etpan_serialize_data * encode_body_fields(struct mailimap_body_fields * body_fields)
{
  struct etpan_serialize_data * hash;
  struct etpan_serialize_data * data;
  
  hash = etpan_serialize_data_new_hash();
  if (body_fields->bd_parameter != NULL) {
    data = encode_body_fld_param(body_fields->bd_parameter);
    etpan_serialize_hash_set(hash, "parameter", data);
  }
  etpan_serialize_hash_set_str(hash, "id", body_fields->bd_id);
  etpan_serialize_hash_set_str(hash, "description", body_fields->bd_description);
  data = encode_body_fld_enc(body_fields->bd_encoding);
  etpan_serialize_hash_set(hash, "encoding", data);
  etpan_serialize_hash_set_uint32(hash, "size", body_fields->bd_size);
  
  return hash;
}

static struct mailimap_body_fields * decode_body_fields(struct etpan_serialize_data * hash)
{
  struct etpan_serialize_data * data;
  struct mailimap_body_fld_param * bd_parameter;
  char * bd_id;
  char * bd_description;
  struct mailimap_body_fld_enc * bd_encoding;
  uint32_t bd_size;
  struct mailimap_body_fields * body_fields;
  
  bd_parameter = NULL;
  data = etpan_serialize_hash_get(hash, "parameter");
  if (data != NULL) {
    bd_parameter = decode_body_fld_param(data);
  }
  bd_id = etpan_serialize_hash_get_str(hash, "id");
  if (bd_id != NULL) {
    bd_id = strdup(bd_id);
    if (bd_id == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  bd_description = etpan_serialize_hash_get_str(hash, "description");
  if (bd_description != NULL) {
    bd_description = strdup(bd_description);
    if (bd_description == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  data = etpan_serialize_hash_get(hash, "encoding");
  bd_encoding = decode_body_fld_enc(data);
  bd_size = etpan_serialize_hash_get_uint32(hash, "size");
  
  body_fields = mailimap_body_fields_new(bd_parameter, bd_id,
      bd_description, bd_encoding, bd_size);
  if (body_fields == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return body_fields;
}

#if 0
struct mailimap_body_fld_param {
  clist * pa_list; /* list of (struct mailimap_single_body_fld_param *) */
                /* != NULL */
};
#endif

static struct etpan_serialize_data * encode_body_fld_param(struct mailimap_body_fld_param * param)
{
  struct etpan_serialize_data * tab;
  clistiter * iter;
  
  tab = etpan_serialize_data_new_array();
  for(iter = clist_begin(param->pa_list) ; iter != NULL ; iter = clist_next(iter)) {
    struct mailimap_single_body_fld_param * elt;
    struct etpan_serialize_data * data;
    
    elt = clist_content(iter);
    data = encode_single_body_fld_param(elt);
    etpan_serialize_array_add(tab, data);
  }
  
  return tab;
}

static struct mailimap_body_fld_param * decode_body_fld_param(struct etpan_serialize_data * tab)
{
  unsigned int i;
  clist * list;
  int r;
  struct mailimap_body_fld_param * param;
  
  list = clist_new();
  if (list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  for(i = 0 ; i < etpan_serialize_array_count(tab) ; i ++) {
    struct etpan_serialize_data * data;
    struct mailimap_single_body_fld_param * elt;
    
    data = etpan_serialize_array_get(tab, i);
    elt = decode_single_body_fld_param(data);
    r = clist_append(list, elt);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  param = mailimap_body_fld_param_new(list);
  if (param == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return param;
}

#if 0
struct mailimap_single_body_fld_param {
  char * pa_name;  /* != NULL */
  char * pa_value; /* != NULL */
};
#endif

static struct etpan_serialize_data * encode_single_body_fld_param(struct mailimap_single_body_fld_param * param)
{
  struct etpan_serialize_data * hash;
  
  hash = etpan_serialize_data_new_hash();
  etpan_serialize_hash_set_str(hash, "name", param->pa_name);
  etpan_serialize_hash_set_str(hash, "value", param->pa_value);
  
  return hash;
}

static struct mailimap_single_body_fld_param * decode_single_body_fld_param(struct etpan_serialize_data * hash)
{
  char * pa_name;
  char * pa_value;
  struct mailimap_single_body_fld_param * param;
  
  pa_name = etpan_serialize_hash_get_str(hash, "name");
  if (pa_name != NULL) {
    pa_name = strdup(pa_name);
    if (pa_name == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  pa_value = etpan_serialize_hash_get_str(hash, "value");
  if (pa_value != NULL) {
    pa_value = strdup(pa_value);
    if (pa_value == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  param = mailimap_single_body_fld_param_new(pa_name, pa_value);
  if (param == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return param;
}

#if 0
struct mailimap_body_fld_enc {
  int enc_type;
  char * enc_value; /* can be NULL */
};
#endif

static struct etpan_serialize_data * encode_body_fld_enc(struct mailimap_body_fld_enc * enc)
{
  struct etpan_serialize_data * hash;
  
  hash = etpan_serialize_data_new_hash();
  etpan_serialize_hash_set_int32(hash, "type", enc->enc_type);
  etpan_serialize_hash_set_str(hash, "value", enc->enc_value);
  
  return hash;
}

static struct mailimap_body_fld_enc * decode_body_fld_enc(struct etpan_serialize_data * hash)
{
  struct mailimap_body_fld_enc * enc;
  int type;
  char * value;
  
  type = etpan_serialize_hash_get_int32(hash, "type");
  value = etpan_serialize_hash_get_str(hash, "value");
  if (value != NULL) {
    value = strdup(value);
    if (value == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  enc = mailimap_body_fld_enc_new(type, value);
  if (enc == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return enc;
}

#if 0
struct mailimap_envelope {
  char * env_date;                             /* can be NULL */
  char * env_subject;                          /* can be NULL */
  struct mailimap_env_from * env_from;         /* can be NULL */
  struct mailimap_env_sender * env_sender;     /* can be NULL */
  struct mailimap_env_reply_to * env_reply_to; /* can be NULL */
  struct mailimap_env_to * env_to;             /* can be NULL */
  struct mailimap_env_cc * env_cc;             /* can be NULL */
  struct mailimap_env_bcc * env_bcc;           /* can be NULL */
  char * env_in_reply_to;                      /* can be NULL */
  char * env_message_id;                       /* can be NULL */
};
#endif

static struct etpan_serialize_data * encode_envelope(struct mailimap_envelope * envelope)
{
  struct etpan_serialize_data * hash;
  struct etpan_serialize_data * data;
  
  hash = etpan_serialize_data_new_hash();
  etpan_serialize_hash_set_str(hash, "date", envelope->env_date);
  etpan_serialize_hash_set_str(hash, "subject", envelope->env_subject);
  if (envelope->env_from != NULL) {
    if (envelope->env_from->frm_list != NULL) {
      data = encode_address_list(envelope->env_from->frm_list);
      etpan_serialize_hash_set(hash, "from", data);
    }
  }
  if (envelope->env_sender != NULL) {
    if (envelope->env_sender->snd_list != NULL) {
      data = encode_address_list(envelope->env_sender->snd_list);
      etpan_serialize_hash_set(hash, "sender", data);
    }
  }
  if (envelope->env_reply_to != NULL) {
    if (envelope->env_reply_to->rt_list != NULL) {
      data = encode_address_list(envelope->env_reply_to->rt_list);
      etpan_serialize_hash_set(hash, "reply-to", data);
    }
  }
  if (envelope->env_to != NULL) {
    if (envelope->env_to->to_list != NULL) {
      data = encode_address_list(envelope->env_to->to_list);
      etpan_serialize_hash_set(hash, "to", data);
    }
  }
  if (envelope->env_cc != NULL) {
    if (envelope->env_cc->cc_list != NULL) {
      data = encode_address_list(envelope->env_cc->cc_list);
      etpan_serialize_hash_set(hash, "cc", data);
    }
  }
  if (envelope->env_bcc != NULL) {
    if (envelope->env_bcc->bcc_list != NULL) {
      data = encode_address_list(envelope->env_bcc->bcc_list);
      etpan_serialize_hash_set(hash, "bcc", data);
    }
  }
  etpan_serialize_hash_set_str(hash, "in-reply-to", envelope->env_in_reply_to);
  etpan_serialize_hash_set_str(hash, "message-id", envelope->env_message_id);
  
  return hash;
}

static struct mailimap_envelope * decode_envelope(struct etpan_serialize_data * hash)
{
  struct etpan_serialize_data * data;
  char * env_date;
  char * env_subject;
  struct mailimap_env_from * env_from;
  struct mailimap_env_sender * env_sender;
  struct mailimap_env_reply_to * env_reply_to;
  struct mailimap_env_to * env_to;
  struct mailimap_env_cc * env_cc;
  struct mailimap_env_bcc * env_bcc;
  char * env_in_reply_to;
  char * env_message_id;
  struct mailimap_envelope * envelope;
  
  env_date = NULL;
  env_subject = NULL;
  env_from = NULL;
  env_sender = NULL;
  env_reply_to = NULL;
  env_to = NULL;
  env_cc = NULL;
  env_bcc = NULL;
  env_in_reply_to = NULL;
  env_message_id = NULL;
  
  env_date = etpan_serialize_hash_get_str(hash, "date");
  if (env_date != NULL) {
    env_date = strdup(env_date);
    if (env_date == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  env_subject = etpan_serialize_hash_get_str(hash, "subject");
  if (env_subject != NULL) {
    env_subject = strdup(env_subject);
    if (env_subject == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  data = etpan_serialize_hash_get(hash, "from");
  if (data != NULL) {
    clist * list;
    
    list = decode_address_list(data);
    env_from = mailimap_env_from_new(list);
    if (env_from == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  else {
    env_from = mailimap_env_from_new(NULL);
    if (env_from == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  data = etpan_serialize_hash_get(hash, "sender");
  if (data != NULL) {
    clist * list;
    
    list = decode_address_list(data);
    env_sender = mailimap_env_sender_new(list);
    if (env_from == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  else {
    env_sender = mailimap_env_sender_new(NULL);
    if (env_from == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  data = etpan_serialize_hash_get(hash, "reply-to");
  if (data != NULL) {
    clist * list;
    
    list = decode_address_list(data);
    env_reply_to = mailimap_env_reply_to_new(list);
    if (env_reply_to == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  else {
    env_reply_to = mailimap_env_reply_to_new(NULL);
    if (env_reply_to == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  data = etpan_serialize_hash_get(hash, "to");
  if (data != NULL) {
    clist * list;
    
    list = decode_address_list(data);
    env_to = mailimap_env_to_new(list);
    if (env_to == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  else {
    env_to = mailimap_env_to_new(NULL);
    if (env_to == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  data = etpan_serialize_hash_get(hash, "cc");
  if (data != NULL) {
    clist * list;
    
    list = decode_address_list(data);
    env_cc = mailimap_env_cc_new(list);
    if (env_cc == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  else {
    env_cc = mailimap_env_cc_new(NULL);
    if (env_cc == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  data = etpan_serialize_hash_get(hash, "bcc");
  if (data != NULL) {
    clist * list;
    
    list = decode_address_list(data);
    env_bcc = mailimap_env_bcc_new(list);
    if (env_bcc == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  else {
    env_bcc = mailimap_env_bcc_new(NULL);
    if (env_bcc == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  env_in_reply_to = etpan_serialize_hash_get_str(hash, "in-reply-to");
  if (env_in_reply_to != NULL) {
    env_in_reply_to = strdup(env_in_reply_to);
    if (env_in_reply_to == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  env_message_id = etpan_serialize_hash_get_str(hash, "message-id");
  if (env_message_id != NULL) {
    env_message_id = strdup(env_message_id);
    if (env_message_id == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }

  envelope = mailimap_envelope_new(env_date, env_subject,
      env_from, env_sender, env_reply_to, env_to, env_cc, env_bcc,
      env_in_reply_to, env_message_id);
  if (envelope == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return envelope;
}


#if 0
struct mailimap_body_fld_dsp {
  char * dsp_type;                     /* != NULL */
  struct mailimap_body_fld_param * dsp_attributes; /* can be NULL */
};
#endif

static struct etpan_serialize_data * encode_body_fld_dsp(struct mailimap_body_fld_dsp * dsp)
{
  struct etpan_serialize_data * hash;
  struct etpan_serialize_data * data;
  
  hash = etpan_serialize_data_new_hash();
  etpan_serialize_hash_set_str(hash, "type", dsp->dsp_type);
  if (dsp->dsp_attributes != NULL) {
    data = encode_body_fld_param(dsp->dsp_attributes);
    etpan_serialize_hash_set(hash, "attributes", data);
  }
  
  return hash;
}

static struct mailimap_body_fld_dsp * decode_body_fld_dsp(struct etpan_serialize_data * hash)
{
  struct etpan_serialize_data * data;
  char * dsp_type;
  struct mailimap_body_fld_param * dsp_attributes;
  struct mailimap_body_fld_dsp * dsp;
  
  dsp_type = etpan_serialize_hash_get_str(hash, "type");
  dsp_type = strdup(dsp_type);
  if (dsp_type == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  dsp_attributes = NULL;
  data = etpan_serialize_hash_get(hash, "attributes");
  if (data != NULL) {
    dsp_attributes = decode_body_fld_param(data);
  }
  
  dsp = mailimap_body_fld_dsp_new(dsp_type, dsp_attributes);
  if (dsp == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return dsp;
}

#if 0
struct mailimap_body_fld_lang {
  int lg_type;
  union {
    char * lg_single; /* can be NULL */
    clist * lg_list; /* list of string (char *), can be NULL */
  } lg_data;
};
#endif

static struct etpan_serialize_data * encode_body_fld_lang(struct mailimap_body_fld_lang * lang)
{
  struct etpan_serialize_data * hash;
  
  hash = etpan_serialize_data_new_hash();
  etpan_serialize_hash_set_int32(hash, "type", lang->lg_type);
  switch (lang->lg_type) {
  case MAILIMAP_BODY_FLD_LANG_SINGLE:
    etpan_serialize_hash_set_str(hash, "single", lang->lg_data.lg_single);
    break;
  case MAILIMAP_BODY_FLD_LANG_LIST:
    {
      clistiter * iter;
      struct etpan_serialize_data * tab;
      
      tab = etpan_serialize_data_new_array();
      for(iter = clist_begin(lang->lg_data.lg_list) ; iter != NULL ;
          iter = clist_next(iter)) {
        char * elt;
        
        elt = clist_content(iter);
        etpan_serialize_array_add_str(tab, elt);
      }
      etpan_serialize_hash_set(hash, "list", tab);
    }
    break;
  }
  
  return hash;
}

static struct mailimap_body_fld_lang * decode_body_fld_lang(struct etpan_serialize_data * hash)
{
  int type;
  clist * list;
  char * value;
  struct mailimap_body_fld_lang * lang;
  int r;
  
  list = NULL;
  value = NULL;
  type = etpan_serialize_hash_get_int32(hash, "type");
  
  switch (type) {
  case MAILIMAP_BODY_FLD_LANG_SINGLE:
    {
      value = etpan_serialize_hash_get_str(hash, "single");
      if (value != NULL) {
        value = strdup(value);
        if (value == NULL)
          ETPAN_LOG_MEMORY_ERROR;
      }
    }
    break;
  case MAILIMAP_BODY_FLD_LANG_LIST:
    {
      unsigned int i;
      struct etpan_serialize_data * tab;
      
      list = clist_new();
      if (list == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      
      tab = etpan_serialize_hash_get(hash, "list");
      for(i = 0 ; i < etpan_serialize_array_count(tab) ; i ++) {
        char * elt;
        
        elt = etpan_serialize_array_get_str(tab, i);
        elt = strdup(elt);
        if (elt == NULL)
          ETPAN_LOG_MEMORY_ERROR;
        
        r = clist_append(list, elt);
        if (r < 0)
          ETPAN_LOG_MEMORY_ERROR;
      }
    }
    break;
  }
  
  lang = mailimap_body_fld_lang_new(type, value, list);
  if (lang == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return lang;
};

#if 0
struct mailimap_body_extension {
  int ext_type;
  /*
    can be MAILIMAP_BODY_EXTENSION_NSTRING, MAILIMAP_BODY_EXTENSION_NUMBER
    or MAILIMAP_BODY_EXTENSION_LIST
  */
  union {
    char * ext_nstring;    /* can be NULL */
    uint32_t ext_number;
    clist * ext_body_extension_list;
    /* list of (struct mailimap_body_extension *) */
    /* can be NULL */
  } ext_data;
};
#endif

static struct etpan_serialize_data * encode_body_ext(struct mailimap_body_extension * ext)
{
  struct etpan_serialize_data * hash;
  struct etpan_serialize_data * data;
  
  hash = etpan_serialize_data_new_hash();
  etpan_serialize_hash_set_int32(hash, "type", ext->ext_type);
  switch (ext->ext_type) {
  case MAILIMAP_BODY_EXTENSION_NSTRING:
    etpan_serialize_hash_set_str(hash, "string", ext->ext_data.ext_nstring);
    break;
  case MAILIMAP_BODY_EXTENSION_NUMBER:
    etpan_serialize_hash_set_uint32(hash, "number", ext->ext_data.ext_number);
    break;
  case MAILIMAP_BODY_EXTENSION_LIST:
    data = encode_body_ext_list(ext->ext_data.ext_body_extension_list);
    etpan_serialize_hash_set(hash, "list", data);
    break;
  }
  
  return hash;
}

static struct mailimap_body_extension * decode_body_ext(struct etpan_serialize_data * hash)
{
  int type;
  struct mailimap_body_extension * ext;
  char * ext_nstring;
  uint32_t ext_number;
  clist * ext_body_extension_list;
  struct etpan_serialize_data * data;
  
  type = etpan_serialize_data_get_int32(hash);
  ext_nstring = NULL;
  ext_number = 0;
  ext_body_extension_list = NULL;
  switch (type) {
  case MAILIMAP_BODY_EXTENSION_NSTRING:
    ext_nstring = etpan_serialize_hash_get_str(hash, "string");
    if (ext_nstring != NULL) {
      ext_nstring = strdup(ext_nstring);
      if (ext_nstring == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    break;
  case MAILIMAP_BODY_EXTENSION_NUMBER:
    ext_number = etpan_serialize_hash_get_uint32(hash, "number");
    break;
  case MAILIMAP_BODY_EXTENSION_LIST:
    data = etpan_serialize_hash_get(hash, "list");
    ext_body_extension_list = decode_body_ext_list(data);
    break;
  }
  
  ext = mailimap_body_extension_new(type, ext_nstring, ext_number,
      ext_body_extension_list);
  if (ext == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return ext;
}

static struct etpan_serialize_data * encode_body_ext_list(clist * body_ext_list)
{
  clistiter * iter;
  struct etpan_serialize_data * tab;
  
  tab = etpan_serialize_data_new_array();
  
  for(iter = clist_begin(body_ext_list) ; iter != NULL ;
      iter = clist_next(iter)) {
    struct mailimap_body_extension * ext;
    struct etpan_serialize_data * data;
    
    ext = clist_content(iter);
    data = encode_body_ext(ext);
    etpan_serialize_array_add(tab, data);
  }
  
  return tab;
}

static clist * decode_body_ext_list(struct etpan_serialize_data * tab)
{
  unsigned int i;
  clist * list;
  int r;
  
  list = clist_new();
  if (list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  for(i = 0 ; i < etpan_serialize_array_count(tab) ; i ++) {
    struct etpan_serialize_data * data;
    struct mailimap_body_extension * ext;
    
    data = etpan_serialize_array_get(tab, i);
    ext = decode_body_ext(data);
    r = clist_append(list, ext);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  return list;
}

#if 0
struct mailimap_address {
  char * ad_personal_name; /* can be NULL */
  char * ad_source_route;  /* can be NULL */
  char * ad_mailbox_name;  /* can be NULL */
  char * ad_host_name;     /* can be NULL */
};
#endif

static struct etpan_serialize_data * encode_address(struct mailimap_address * addr)
{
  struct etpan_serialize_data * hash;
  
  hash = etpan_serialize_data_new_hash();
  etpan_serialize_hash_set_str(hash, "personal-name", addr->ad_personal_name);
  etpan_serialize_hash_set_str(hash, "source-route", addr->ad_source_route);
  etpan_serialize_hash_set_str(hash, "mailbox-name", addr->ad_mailbox_name);
  etpan_serialize_hash_set_str(hash, "host-name", addr->ad_host_name);
  
  return hash;
}

static struct mailimap_address * decode_address(struct etpan_serialize_data * hash)
{
  char * ad_personal_name;
  char * ad_source_route;
  char * ad_mailbox_name;
  char * ad_host_name;
  struct mailimap_address * addr;
  
  ad_personal_name = etpan_serialize_hash_get_str(hash, "personal-name");
  if (ad_personal_name != NULL) {
    ad_personal_name = strdup(ad_personal_name);
    if (ad_personal_name == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  ad_source_route = etpan_serialize_hash_get_str(hash, "source-route");
  if (ad_source_route != NULL) {
    ad_source_route = strdup(ad_source_route);
    if (ad_source_route == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  ad_mailbox_name = etpan_serialize_hash_get_str(hash, "mailbox-name");
  if (ad_mailbox_name != NULL) {
    ad_mailbox_name = strdup(ad_mailbox_name);
    if (ad_mailbox_name == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  ad_host_name = etpan_serialize_hash_get_str(hash, "host-name");
  if (ad_host_name != NULL) {
    ad_host_name = strdup(ad_host_name);
    if (ad_host_name == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  addr = mailimap_address_new(ad_personal_name, ad_source_route,
      ad_mailbox_name, ad_host_name);
  if (addr == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return addr;
}

static struct etpan_serialize_data * encode_address_list(clist * addr_list)
{
  clistiter * iter;
  struct etpan_serialize_data * tab;
  
  tab = etpan_serialize_data_new_array();
  
  for(iter = clist_begin(addr_list) ; iter != NULL ;
      iter = clist_next(iter)) {
    struct mailimap_address * addr;
    struct etpan_serialize_data * data;
    
    addr = clist_content(iter);
    data = encode_address(addr);
    etpan_serialize_array_add(tab, data);
  }
  
  return tab;
}

static clist * decode_address_list(struct etpan_serialize_data * tab)
{
  unsigned int i;
  clist * list;
  int r;
  
  list = clist_new();
  if (list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  for(i = 0 ; i < etpan_serialize_array_count(tab) ; i ++) {
    struct etpan_serialize_data * data;
    struct mailimap_address * addr;
    
    data = etpan_serialize_array_get(tab, i);
    addr = decode_address(data);
    r = clist_append(list, addr);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  return list;
}

struct etpan_serialize_data * etpan_imap_serialize_encode_body(struct mailimap_body * body)
{
  return encode_body(body);
}

struct mailimap_body * etpan_imap_serialize_decode_body(struct etpan_serialize_data * hash)
{
  return decode_body(hash);
}

struct etpan_serialize_data * etpan_imap_serialize_encode_envelope(struct mailimap_envelope * envelope)
{
  return encode_envelope(envelope);
}

struct mailimap_envelope * etpan_imap_serialize_decode_envelope(struct etpan_serialize_data * hash)
{
  return decode_envelope(hash);
}

void etpan_imap_body_serialize(struct mailimap_body * body,
    void ** p_data, size_t * p_length)
{
  struct etpan_serialize_data * hash;
  
  hash = etpan_imap_serialize_encode_body(body);
  etpan_serialize_encode(hash, p_data, p_length);
  etpan_serialize_data_free(hash);
}

struct mailimap_body *
etpan_imap_body_unserialize(void * data, size_t length)
{
  struct etpan_serialize_data * hash;
  struct mailimap_body * body;
  
  hash = etpan_serialize_decode(data, length);
  body = etpan_imap_serialize_decode_body(hash);
  etpan_serialize_data_free(hash);
  
  return body;
}

void etpan_imap_envelope_serialize(struct mailimap_envelope * env,
    void ** p_data, size_t * p_length)
{
  struct etpan_serialize_data * hash;
  
  hash = etpan_imap_serialize_encode_envelope(env);
  etpan_serialize_encode(hash, p_data, p_length);
  etpan_serialize_data_free(hash);
}

struct mailimap_envelope *
etpan_imap_envelope_unserialize(void * data, size_t length)
{
  struct etpan_serialize_data * hash;
  struct mailimap_envelope * env;
  
  hash = etpan_serialize_decode(data, length);
  env = etpan_imap_serialize_decode_envelope(hash);
  etpan_serialize_data_free(hash);
  
  return env;
}
