#include "etpan-imap-sync.h"
#include "etpan-imap-sync-private.h"

#define ETPAN_MODULE_LOG_NAME "IMAPSYNC"

#include <stdlib.h>
#include <string.h>

#include "etpan-error.h"
#include "etpan-log.h"
#include "etpan-connection-types.h"
#include "etpan-nls.h"
#include "etpan-imap.h"
#include "etpan-sqldb.h"
#include "etpan-utils.h"
#include "etpan-account-manager.h"
#include "etpan-imap-folder-sync.h"
#include "etpan-imap-folder-sync-private.h"
#include "etpan-thread-manager.h"
#include "etpan-thread-manager-app.h"
#include "etpan-account.h"
#include "etpan-time.h"
#include "etpan-signal.h"
#include "etpan-folder.h"
#include "etpan-folder-imap-sync.h"
#include "etpan-folder-imap-sync-private.h"
#include "etpan-serialize.h"
#include "etpan-mail-manager.h"
#include "etpan-storage.h"

enum {
  ERROR_DOMAIN_CONNECTION,
  ERROR_DOMAIN_FETCH_FOLDER_LIST,
  ERROR_DOMAIN_CREATE,
  ERROR_DOMAIN_DELETE,
  ERROR_DOMAIN_RENAME,
};

enum {
  STATE_DISCONNECTED,
  STATE_CONNECTED,
  STATE_STREAM_CLOSED,
};

#define SYNC_FOLDERLIST_INTERVAL (5 * 60)

static void add_folder(struct etpan_imap_sync * imap_sync,
    char * folder_name, int noinferiors, char separator);

static int update_folder_list_from_db(struct etpan_imap_sync * imap_sync);

static struct etpan_error * rewrite_error(struct etpan_imap_sync * imap_sync,
    int domain,
    struct etpan_error * error);

struct etpan_imap_sync * etpan_imap_sync_new(void)
{
  struct etpan_imap_sync * imap_sync;
  
  /* in a thread */
  imap_sync = malloc(sizeof(* imap_sync));
  if (imap_sync == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  imap_sync->id = NULL;
  imap_sync->account = NULL;
  
  imap_sync->hostname = NULL;
  imap_sync->port = 0;
  imap_sync->connection_type = ETPAN_CONNECTION_PLAIN;
  imap_sync->auth_type = ETPAN_AUTH_PASSWORD;
  imap_sync->username = NULL;
  imap_sync->password = NULL;
  imap_sync->command = NULL;
  imap_sync->base_location = NULL;
  imap_sync->main_imap_separator = '\0';
  
  /* threaded */
  imap_sync->connected = 0;
  imap_sync->connected_count = 0;
  imap_sync->threaded_hostname = NULL;
  imap_sync->threaded_port = 0;
  imap_sync->threaded_connection_type = ETPAN_CONNECTION_PLAIN;
  imap_sync->threaded_auth_type = ETPAN_AUTH_PASSWORD;
  imap_sync->threaded_username = NULL;
  imap_sync->threaded_password = NULL;
  imap_sync->threaded_command = NULL;
  imap_sync->threaded_base_location = NULL;
  imap_sync->threaded_main_imap_separator = '\0';
  imap_sync->should_disconnect = 0;
  imap_sync->threaded_current_location = NULL;
  
  imap_sync->ep_storage = NULL;
  imap_sync->folderlist_db = NULL;
  pthread_mutex_init(&imap_sync->folderlist_db_lock, NULL);
  
  pthread_mutex_init(&imap_sync->folder_hash_lock, NULL);
  imap_sync->folder_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (imap_sync->folder_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  pthread_mutex_init(&imap_sync->pending_folders_lock, NULL);
  imap_sync->pending_folders = carray_new(16);
  if (imap_sync->pending_folders == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  imap_sync->last_sync_date = 0;
  imap_sync->syncing = 0;
  imap_sync->op = NULL;
  imap_sync->error = NULL;
  imap_sync->last_sync_success = 0;
  imap_sync->state = STATE_DISCONNECTED;
  
  imap_sync->dummy_storage = etpan_storage_new();
  
  return imap_sync;
}

void etpan_imap_sync_free(struct etpan_imap_sync * imap_sync)
{
  etpan_storage_free(imap_sync->dummy_storage);
  
  if (chash_count(imap_sync->folder_hash) != 0)
    ETPAN_WARN_LOG("folder not all closed");
  
  carray_free(imap_sync->pending_folders);
  chash_free(imap_sync->folder_hash);
  pthread_mutex_destroy(&imap_sync->pending_folders_lock);
  pthread_mutex_destroy(&imap_sync->folder_hash_lock);
  
  pthread_mutex_destroy(&imap_sync->folderlist_db_lock);
  
  free(imap_sync->threaded_current_location);
  free(imap_sync->threaded_base_location);
  free(imap_sync->threaded_command);
  free(imap_sync->threaded_password);
  free(imap_sync->threaded_username);
  free(imap_sync->threaded_hostname);
  
  free(imap_sync->base_location);
  free(imap_sync->command);
  free(imap_sync->password);
  free(imap_sync->username);
  free(imap_sync->hostname);
  
  free(imap_sync->id);
  
  free(imap_sync);
}

/* properties */

void etpan_imap_sync_set_id(struct etpan_imap_sync * imap_sync,
    char * id)
{
  if (id != imap_sync->id) {
    free(imap_sync->id);
    if (id != NULL) {
      imap_sync->id = strdup(id);
      if (imap_sync->id == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      imap_sync->id = NULL;
  }
}

char * etpan_imap_sync_get_id(struct etpan_imap_sync * imap_sync)
{
  return imap_sync->id;
}

void etpan_imap_sync_set_hostname(struct etpan_imap_sync * imap_sync,
    char * hostname)
{
  if (hostname != imap_sync->hostname) {
    free(imap_sync->hostname);
    if (hostname != NULL) {
      imap_sync->hostname = strdup(hostname);
      if (imap_sync->hostname == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      imap_sync->hostname = NULL;
  }
}

char * etpan_imap_sync_get_hostname(struct etpan_imap_sync * imap_sync)
{
  return imap_sync->hostname;
}

void etpan_imap_sync_set_port(struct etpan_imap_sync * imap_sync, int port)
{
  imap_sync->port = port;
}

int etpan_imap_sync_get_port(struct etpan_imap_sync * imap_sync)
{
  return imap_sync->port;
}

void etpan_imap_sync_set_connection_type(struct etpan_imap_sync * imap_sync,
    int connection_type)
{
  imap_sync->connection_type = connection_type;
}

int etpan_imap_sync_get_connection_type(struct etpan_imap_sync * imap_sync)
{
  return imap_sync->connection_type;
}

void etpan_imap_sync_set_auth_type(struct etpan_imap_sync * imap_sync,
    int auth_type)
{
  imap_sync->auth_type = auth_type;
}

int etpan_imap_sync_get_auth_type(struct etpan_imap_sync * imap_sync)
{
  return imap_sync->auth_type;
}

void etpan_imap_sync_set_username(struct etpan_imap_sync * imap_sync,
    char * username)
{
  if (username != imap_sync->username) {
    free(imap_sync->username);
    if (username != NULL) {
      imap_sync->username = strdup(username);
      if (imap_sync->username == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      imap_sync->username = NULL;
  }
}

char * etpan_imap_sync_get_username(struct etpan_imap_sync * imap_sync)
{
  return imap_sync->username;
}

void etpan_imap_sync_set_password(struct etpan_imap_sync * imap_sync,
    char * password)
{
  if (password != imap_sync->password) {
    free(imap_sync->password);
    if (password != NULL) {
      imap_sync->password = strdup(password);
      if (imap_sync->password == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      imap_sync->password = NULL;
  }
}

char * etpan_imap_sync_get_password(struct etpan_imap_sync * imap_sync)
{
  return imap_sync->password;
}

void etpan_imap_sync_set_base_location(struct etpan_imap_sync * imap_sync,
    char * base_location)
{
  if (base_location != imap_sync->base_location) {
    free(imap_sync->base_location);
    if (base_location != NULL) {
      imap_sync->base_location = strdup(base_location);
      if (imap_sync->base_location == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      imap_sync->base_location = NULL;
  }
}

char * etpan_imap_sync_get_base_location(struct etpan_imap_sync * imap_sync)
{
  return imap_sync->base_location;
}

void etpan_imap_sync_set_command(struct etpan_imap_sync * imap_sync,
    char * command)
{
  if (command != imap_sync->command) {
    free(imap_sync->command);
    if (command != NULL) {
      imap_sync->command = strdup(command);
      if (imap_sync->command == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      imap_sync->command = NULL;
  }
}

char * etpan_imap_sync_get_command(struct etpan_imap_sync * imap_sync)
{
  return imap_sync->command;
}

/* setup */

void etpan_imap_sync_setup(struct etpan_imap_sync * imap_sync)
{
  carray * columns;
  char filename[PATH_MAX];
  char * path;
  int r;

  /* running in a thread */
  free(imap_sync->threaded_base_location);
  imap_sync->threaded_base_location = NULL;
  free(imap_sync->threaded_command);
  imap_sync->threaded_command = NULL;
  free(imap_sync->threaded_password);
  imap_sync->threaded_password = NULL;
  free(imap_sync->threaded_username);
  imap_sync->threaded_username = NULL;
  free(imap_sync->threaded_hostname);
  imap_sync->threaded_hostname = NULL;
  
  if (imap_sync->hostname != NULL) {
    imap_sync->threaded_hostname = strdup(imap_sync->hostname);
    if (imap_sync->threaded_hostname == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  imap_sync->threaded_port = imap_sync->port;
  imap_sync->threaded_connection_type = imap_sync->connection_type;
  imap_sync->threaded_auth_type = imap_sync->auth_type;
  if (imap_sync->username != NULL) {
    imap_sync->threaded_username = strdup(imap_sync->username);
    if (imap_sync->threaded_username == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  if (imap_sync->password != NULL) {
    imap_sync->threaded_password = strdup(imap_sync->password);
    if (imap_sync->threaded_password == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  if (imap_sync->command != NULL) {
    imap_sync->threaded_command = strdup(imap_sync->command);
    if (imap_sync->threaded_command == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  if (imap_sync->base_location != NULL) {
    imap_sync->threaded_base_location = strdup(imap_sync->base_location);
    if (imap_sync->threaded_base_location == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  path = etpan_get_imap_cache_dir(etpan_account_manager_get_default(),
      imap_sync);
  etpan_make_dir(path);
  columns = carray_new(4);
  if (columns == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  r = carray_add(columns, "flags", NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  r = carray_add(columns, "delimiter", NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  r = carray_add(columns, "uidvalidity", NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  r = carray_add(columns, "lastuid", NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  r = carray_add(columns, "envlastuid", NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  r = carray_add(columns, "bodylastuid", NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  r = carray_add(columns, "msg_count", NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  r = carray_add(columns, "unseen_count", NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  r = carray_add(columns, "recent_count", NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  etpan_make_dir(path);
  snprintf(filename, sizeof(filename), "%s/folderlist.db", path);
  ETPAN_LOCAL_LOG("create db %s", filename);
  imap_sync->folderlist_db = etpan_sqldb_new(filename, columns);
  free(path);
  carray_free(columns);
  
  etpan_sqldb_open(imap_sync->folderlist_db);
  
  /* in a thread */
  update_folder_list_from_db(imap_sync);
}

void etpan_imap_sync_unsetup(struct etpan_imap_sync * imap_sync)
{
  etpan_sqldb_close(imap_sync->folderlist_db);
  
  free(imap_sync->threaded_base_location);
  imap_sync->threaded_base_location = NULL;
  free(imap_sync->threaded_command);
  imap_sync->threaded_command = NULL;
  free(imap_sync->threaded_password);
  imap_sync->threaded_password = NULL;
  free(imap_sync->threaded_username);
  imap_sync->threaded_username = NULL;
  free(imap_sync->threaded_hostname);
  imap_sync->threaded_hostname = NULL;
}

static struct etpan_error * ep_storage_connect(struct etpan_imap_sync * imap_sync)
{
  int r;
  struct etpan_error * error;
  
  r = mailstorage_connect(imap_sync->ep_storage);
  if (r == MAIL_ERROR_MEMORY)
    ETPAN_LOG_MEMORY_ERROR;
    
  if (r == MAIL_ERROR_CONNECT) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CONNECT);
    etpan_error_set_short_description(error, "Could not connect");
    etpan_error_strf_long_description(error,
        _("Connection to account %s failed. "
            "The connection to IMAP server %s, port %i could not be established.\n"
            "Check the name of server and port."),
        etpan_imap_sync_get_id(imap_sync),
        etpan_imap_sync_get_hostname(imap_sync),
        etpan_imap_sync_get_port(imap_sync));
    goto exit;
  }
  else if (r == MAIL_ERROR_LOGIN) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_AUTH);
    etpan_error_set_short_description(error, "Authentication failed");
    etpan_error_strf_long_description(error,
        _("Connection to account %s failed. "
            "Check your login, password and authentication type."),
        etpan_imap_sync_get_id(imap_sync));
    goto exit;
  }
  else if (r == MAIL_ERROR_PARSE) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, "Parse error");
    etpan_error_strf_long_description(error, _("Connection to account %s failed. The application did not understand what the server sent."),
        etpan_imap_sync_get_id(imap_sync));
    goto exit;
  }
  else if (r == MAIL_ERROR_STREAM) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_STREAM);
    etpan_error_set_short_description(error, "Connection closed");
    etpan_error_strf_long_description(error,
        _("Connection to account %s failed. "
            "The connection closed unexpectedly."),
        etpan_imap_sync_get_id(imap_sync));
    goto exit;
  }
  else if (r != MAIL_NO_ERROR) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CONNECT);
    etpan_error_set_short_description(error, _("Unexpected error"));
    etpan_error_strf_long_description(error,
        _("Connection to account %s failed. "
            "An unexpected error (libetpan: %i) occurred."),
        etpan_imap_sync_get_id(imap_sync), r);
    goto exit;
  }
  
  error = NULL;
  
 exit:
  return error;
}

static void encode_uint32(uint32_t value, void ** p_data, size_t * p_length)
{
  struct etpan_serialize_data * sdata;
  
  sdata = etpan_serialize_data_new_uint32(value);
  etpan_serialize_encode(sdata, p_data, p_length);
  etpan_serialize_data_free(sdata);
}

static uint32_t decode_uint32(void * data, size_t length)
{
  struct etpan_serialize_data * sdata;
  uint32_t value;
  
  sdata = etpan_serialize_decode(data, length);
  value = etpan_serialize_data_get_uint32(sdata);
  etpan_serialize_data_free(sdata);
  
  return value;
}

static void encode_char(char ch, void ** p_data, size_t * p_length)
{
  encode_uint32((unsigned char) ch, p_data, p_length);
}

static char decode_char(void * data, size_t length)
{
  return (char) ((unsigned char) decode_uint32(data, length));
}

static struct etpan_error * get_main_delimiter(struct etpan_imap_sync * imap_sync)
{
  int r;
  struct etpan_error * error;
  char * location;
  struct mailstorage * ep_storage;
  void * data;
  size_t length;
  char imap_separator;
  carray * imap_mb_list;
  struct etpan_imap_mailbox_info * imap_folder;
  int had_error;
  
  ep_storage = imap_sync->ep_storage;
  
  pthread_mutex_lock(&imap_sync->folderlist_db_lock);
  r = etpan_sqldb_get(imap_sync->folderlist_db,
      "", "delimiter", &data, &length);
  pthread_mutex_unlock(&imap_sync->folderlist_db_lock);
  if (r == 0) {
    free(data);
    return NULL;
  }
  
  location = imap_sync->threaded_base_location;
  if (location == NULL)
    location = "";
  
  error = etpan_imap_list_mailboxes(ep_storage->sto_session,
      location, &imap_mb_list);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(imap_sync, ERROR_DOMAIN_CONNECTION, error);
    ETPAN_ERROR_FREE(error);
    
    error = new_error;
    goto exit;
  }
  
  if (carray_count(imap_mb_list) == 0) {
    etpan_imap_list_mailboxes_free(imap_mb_list);
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FOLDER_NOT_FOUND);
    etpan_error_set_short_description(error, _("Mailbox not found"));
    
    etpan_error_strf_long_description(error,
        _("Connection to account %s failed. "
            "Mailbox (%s) given in mailbox prefix does not exist or is invalid."),
        location);
    
    goto exit;
  }
  
  imap_folder = carray_get(imap_mb_list, 0);
  imap_separator = imap_folder->delimiter;
  
  imap_sync->threaded_main_imap_separator = imap_separator;
  
  pthread_mutex_lock(&imap_sync->folderlist_db_lock);
  had_error = 0;
  r = etpan_sqldb_begin_transaction(imap_sync->folderlist_db);
  if (r < 0)
    had_error = 1;
  encode_char(imap_separator, &data, &length);
  r = etpan_sqldb_set(imap_sync->folderlist_db,
      "", "delimiter", data, length);
  free(data);
  if (r < 0)
    had_error = 1;
  r = etpan_sqldb_end_transaction(imap_sync->folderlist_db);
  if (r < 0)
    had_error = 1;
  pthread_mutex_unlock(&imap_sync->folderlist_db_lock);
  
  if (had_error) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_DATABASE);
    etpan_error_set_short_description(error, _("Cache could not be written"));
    etpan_error_strf_long_description(error,
        _("Connection to account %s failed. "
            "Cache could not be written properly on disk."),
        etpan_imap_sync_get_id(imap_sync));
    goto exit;
  }
  
  etpan_imap_list_mailboxes_free(imap_mb_list);
  
  error = NULL;
  
 exit:
  return error;
}

static struct etpan_error * connect(struct etpan_imap_sync * imap_sync)
{
  int r;
  struct etpan_error * error;
  struct mailstorage * ep_storage;
  int ep_connection_type;
  int ep_auth_type;
  
  if (imap_sync->connected)
    return NULL;
  
  ep_storage = mailstorage_new(NULL);
  if (ep_storage == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  if (imap_sync->threaded_command == NULL) {
    switch (imap_sync->threaded_connection_type) {
    case ETPAN_CONNECTION_STARTTLS:
      ep_connection_type = CONNECTION_TYPE_STARTTLS;
      break;
    case ETPAN_CONNECTION_TRY_STARTTLS:
      ep_connection_type = CONNECTION_TYPE_TRY_STARTTLS;
      break;
    case ETPAN_CONNECTION_TLS:
      ep_connection_type = CONNECTION_TYPE_TLS;
      break;
    default:
      ep_connection_type = CONNECTION_TYPE_PLAIN;
      break;
    }
  }
  else {
    switch (imap_sync->threaded_connection_type) {
    case ETPAN_CONNECTION_STARTTLS:
      ep_connection_type = CONNECTION_TYPE_COMMAND_STARTTLS;
      break;
    case ETPAN_CONNECTION_TRY_STARTTLS:
      ep_connection_type = CONNECTION_TYPE_COMMAND_TRY_STARTTLS;
      break;
    case ETPAN_CONNECTION_TLS:
      ep_connection_type = CONNECTION_TYPE_COMMAND_TLS;
      break;
    default:
      ep_connection_type = CONNECTION_TYPE_COMMAND;
      break;
    }
  }
    
  ep_auth_type = IMAP_AUTH_TYPE_PLAIN;
    
  r = imap_mailstorage_init(ep_storage,
      imap_sync->threaded_hostname, imap_sync->threaded_port,
      imap_sync->threaded_command,
      ep_connection_type, ep_auth_type,
      imap_sync->threaded_username, imap_sync->threaded_password,
      0, NULL);
  if (r != MAIL_NO_ERROR) {
    ETPAN_LOG_MEMORY_ERROR;
  }
    
  imap_sync->ep_storage = ep_storage;
    
  error = ep_storage_connect(imap_sync);
  if (error != NULL)
    goto free_storage;
    
  error = get_main_delimiter(imap_sync);
  if (error != NULL)
    goto disconnect;
  imap_sync->connected = 1;
  
  return NULL;
  
 disconnect:
  mailstorage_disconnect(imap_sync->ep_storage);
 free_storage:
  mailstorage_free(imap_sync->ep_storage);
  imap_sync->ep_storage = NULL;
  return error;
}

static int update_folder_list_from_db(struct etpan_imap_sync * imap_sync)
{
  carray * keys;
  unsigned int i;
  int r;
  chash * folder_hash;
  chashiter * iter;
  carray * folder_to_remove;
  int updated;
  
  updated = 0;
  pthread_mutex_lock(&imap_sync->folderlist_db_lock);
  /* in a thread */
  keys = etpan_sqldb_get_keys(imap_sync->folderlist_db);
  ETPAN_LOCAL_LOG("folder list count : %i", carray_count(keys));
  for(i = 0 ; i < carray_count(keys) ; i ++) {
    char * folder_name;
    void * data;
    size_t length;
    int flags;
    char delimiter;
    chashdatum key;
    chashdatum value;
    
    folder_name = carray_get(keys, i);    
    if (* folder_name == '\0')
      continue;
    
    r = etpan_sqldb_get(imap_sync->folderlist_db,
        folder_name, "flags", &data, &length);
    if (r < 0) {
      ETPAN_WARN_LOG("database corrupted");
      continue;
    }
    flags = decode_uint32(data, length);
    free(data);
    
    r = etpan_sqldb_get(imap_sync->folderlist_db,
        folder_name, "delimiter", &data, &length);
    if (r < 0) {
      ETPAN_WARN_LOG("database corrupted");
      continue;
    }
    delimiter = decode_char(data, length);
    free(data);
    
    key.data = folder_name;
    key.len = strlen(folder_name) + 1;
    
    /* in a thread */
    pthread_mutex_lock(&imap_sync->folder_hash_lock);
    r = chash_get(imap_sync->folder_hash, &key, &value);
    pthread_mutex_unlock(&imap_sync->folder_hash_lock);
    if (r < 0) {
      add_folder(imap_sync, folder_name,
          (flags & ETPAN_IMAP_MB_NOINFERIORS) != 0, delimiter);
      updated = 1;
    }
  }
  pthread_mutex_unlock(&imap_sync->folderlist_db_lock);
  
  folder_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (folder_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;

  /* disconnect lost folders */
  folder_to_remove = carray_new(16);
  if (folder_to_remove == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  for(i = 0 ; i < carray_count(keys) ; i ++) {
    chashdatum key;
    chashdatum value;
    char * folder_name;
    
    folder_name = carray_get(keys, i);
    if (* folder_name == '\0')
      continue;
    
    key.data = folder_name;
    key.len = strlen(folder_name) + 1;
    value.data = NULL;
    value.len = 0;
    r = chash_set(folder_hash, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  pthread_mutex_lock(&imap_sync->folder_hash_lock);
  pthread_mutex_lock(&imap_sync->pending_folders_lock);
  for(iter = chash_begin(imap_sync->folder_hash) ; iter != NULL ;
      iter = chash_next(imap_sync->folder_hash, iter)) {
    chashdatum key;
    chashdatum value;
    
    chash_key(iter, &key);
    r = chash_get(folder_hash, &key, &value);
    if (r < 0) {
      struct etpan_imap_folder_sync * folder_sync;
      
      chash_value(iter, &value);
      folder_sync = value.data;
      
      /* mark for disconnection */
      r = carray_add(folder_to_remove, folder_sync, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
      r = carray_add(imap_sync->pending_folders, folder_sync, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
    }
  }

  for(i = 0 ; i < carray_count(folder_to_remove) ; i ++) {
    struct etpan_imap_folder_sync * folder_sync;
    chashdatum key;
    
    folder_sync = carray_get(folder_to_remove, i);
    key.data = folder_sync->location;
    key.len = strlen(folder_sync->location) + 1;
    chash_delete(imap_sync->folder_hash, &key, NULL);
  }
  
  pthread_mutex_unlock(&imap_sync->pending_folders_lock);
  pthread_mutex_unlock(&imap_sync->folder_hash_lock);
  
  pthread_mutex_lock(&imap_sync->pending_folders_lock);
  imap_sync->disconnecting_folders = 1;
  for(i = 0 ; i < carray_count(folder_to_remove) ; i ++) {
    struct etpan_imap_folder_sync * folder_sync;
    
    folder_sync = carray_get(folder_to_remove, i);
    etpan_imap_folder_sync_disconnect_nt(folder_sync);
  }
  imap_sync->disconnecting_folders = 0;
  pthread_mutex_unlock(&imap_sync->pending_folders_lock);
  
  carray_free(folder_to_remove);
  chash_free(folder_hash);
  
  etpan_sqldb_keys_free(keys);
  
  return updated;
}

static void add_folder_result(carray * imap_mb_list,
    chash * folder_hash, carray * folder_list, char * p_delimiter)
{
  unsigned int i;
  int r;
  
  i = 0;
  while (i < carray_count(imap_mb_list)) {
    struct etpan_imap_mailbox_info * folder;
    chashdatum key;
    chashdatum value;
    
    folder = carray_get(imap_mb_list, i);
    if (p_delimiter != NULL)
      * p_delimiter = folder->delimiter;
    key.data = folder->mb;
    key.len = strlen(folder->mb) + 1;
    r = chash_get(folder_hash, &key, &value);
    if (r < 0) {
      value.data = folder;
      value.len = 0;
      r = chash_set(folder_hash, &key, &value, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
      carray_delete(imap_mb_list, i);
      r = carray_add(folder_list, folder, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else {
      i ++;
    }
  }
}

static struct etpan_error *
recursive_list_folder(struct etpan_imap_sync * imap_sync,
    char * top_level, chash * folder_hash,
    carray * folder_list)
{
  struct etpan_error * error;
  chashdatum key;
  chashdatum value;
  carray * imap_mb_list;
  char * wildcard;
  size_t size;
  struct mailstorage * ep_storage;
  int r;
  
  ep_storage = imap_sync->ep_storage;
  
  if (* top_level != '\0') {
    char delimiter;
    
    delimiter = '\0';
    key.data = top_level;
    key.len = strlen(top_level) + 1;
    r = chash_get(folder_hash, &key, &value);
    if (r < 0) {
      error = etpan_imap_list_mailboxes(ep_storage->sto_session,
          top_level, &imap_mb_list);
      if (error != NULL) {
        struct etpan_error * new_error;
        
        new_error = rewrite_error(imap_sync,
            ERROR_DOMAIN_FETCH_FOLDER_LIST, error);
        ETPAN_ERROR_FREE(error);
        error = new_error;
        
        goto exit;
      }
      
      add_folder_result(imap_mb_list, folder_hash, folder_list, &delimiter);
      etpan_imap_list_mailboxes_free(imap_mb_list);
    }
    else {
      struct etpan_imap_mailbox_info * folder;
    
      folder = value.data;
      delimiter = folder->delimiter;
    }
  
    if (delimiter == '\0')
      return NULL;
  
    size = strlen(top_level) + 2;
    wildcard = malloc(size + 1);
    if (wildcard == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    snprintf(wildcard, size + 1, "%s%c%%", top_level, delimiter);
  }
  else {
    wildcard = strdup("%");
    if (wildcard == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  error = etpan_imap_lsub_mailboxes(ep_storage->sto_session,
      wildcard, &imap_mb_list);
  free(wildcard);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(imap_sync,
        ERROR_DOMAIN_FETCH_FOLDER_LIST, error);
    ETPAN_ERROR_FREE(error);
    error = new_error;
    
    goto exit;
  }
  
  add_folder_result(imap_mb_list, folder_hash, folder_list, NULL);
  
  etpan_imap_list_mailboxes_free(imap_mb_list);
  
  error = NULL;
  
 exit:
  return error;
}

static struct etpan_error *
sync_folder_list(struct etpan_imap_sync * imap_sync)
{
  struct etpan_error * error;
  carray * imap_mb_list;
  unsigned int i;
  int r;
  carray * keys;
  chash * folder_hash;
  unsigned int current_index;
  int had_error;
  char * base_location;
  
  /* in a thread */
  imap_mb_list = carray_new(16);
  if (imap_mb_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  folder_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (folder_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  error = recursive_list_folder(imap_sync, "INBOX",
      folder_hash, imap_mb_list);
  if (error != NULL)
    goto free_mb_list;
  if (imap_sync->threaded_base_location == NULL)
    base_location = "";
  else
    base_location = imap_sync->threaded_base_location;
  error = recursive_list_folder(imap_sync, base_location,
      folder_hash, imap_mb_list);
  if (error != NULL)
    goto free_mb_list;
  
  current_index = 0;
  while (current_index < carray_count(imap_mb_list)) {
    struct etpan_imap_mailbox_info * folder;
    
    folder = carray_get(imap_mb_list, current_index);
    if ((folder->flags & ETPAN_IMAP_MB_NOINFERIORS) != 0) {
      current_index ++;
      continue;
    }
    
    error = recursive_list_folder(imap_sync, folder->mb,
        folder_hash, imap_mb_list);
    if (error != NULL)
      goto free_mb_list;
    current_index ++;
  }
  
  ETPAN_LOCAL_LOG("store results");
  had_error = 0;
  pthread_mutex_lock(&imap_sync->folderlist_db_lock);
  r = etpan_sqldb_begin_transaction(imap_sync->folderlist_db);
  if (r < 0)
    had_error = 1;
  for(i = 0 ; i < carray_count(imap_mb_list) ; i ++) {
    struct etpan_imap_mailbox_info * folder;
    void * data;
    size_t length;
    
    folder = carray_get(imap_mb_list, i);
    if ((folder->flags & ETPAN_IMAP_MB_NOSELECT) != 0) {
      chashdatum key;
      
      key.data = folder->mb;
      key.len = strlen(folder->mb) + 1;
      chash_delete(folder_hash, &key, NULL);
      
      continue;
    }
    
    encode_uint32(folder->flags, &data, &length);
    r = etpan_sqldb_set(imap_sync->folderlist_db,
        folder->mb, "flags",
        data, length);
    free(data);
    if (r < 0)
      had_error = 1;
    encode_char(folder->delimiter, &data, &length);
    r = etpan_sqldb_set(imap_sync->folderlist_db,
        folder->mb, "delimiter",
        data, length);
    free(data);
    if (r < 0)
      had_error = 1;
  }
  /* cleanup non-existant */
  keys = etpan_sqldb_get_keys(imap_sync->folderlist_db);
  for(i = 0 ; i < carray_count(keys) ; i ++) {
    char * folder_name;
    chashdatum key;
    chashdatum value;
    
    folder_name = carray_get(keys, i);
    if (* folder_name == '\0')
      continue;
    
    key.data = folder_name;
    key.len = strlen(folder_name) + 1;
    r = chash_get(folder_hash, &key, &value);
    if (r < 0) {
      r = etpan_sqldb_delete(imap_sync->folderlist_db, folder_name);
      if (r < 0)
        had_error = 1;
    }
  }
  r = etpan_sqldb_end_transaction(imap_sync->folderlist_db);
  if (r < 0)
    had_error = 1;
  etpan_sqldb_keys_free(keys);
  keys = etpan_sqldb_get_keys(imap_sync->folderlist_db);
  pthread_mutex_unlock(&imap_sync->folderlist_db_lock);
  
  if (had_error) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_DATABASE);
    etpan_error_set_short_description(error, _("Cache could not be written"));
    etpan_error_strf_long_description(error,
        _("An error occurred while getting list of mailboxes on account %s.\n"
            "Cache could not be written properly on disk."),
        etpan_imap_sync_get_id(imap_sync));
    
    goto free_keys;
  }
  
  for(i = 0 ; i < carray_count(keys) ; i ++) {
    char * folder_name;
    unsigned int msg_count;
    unsigned int unseen_count;
    unsigned int recent_count;
    void * data;
    size_t length;
    uint32_t flags;
    
    folder_name = carray_get(keys, i);
    if (* folder_name == '\0')
      continue;
    
    r = etpan_sqldb_get(imap_sync->folderlist_db,
        folder_name, "flags",
        &data, &length);
    flags = decode_uint32(data, length);
    free(data);
    if ((flags & ETPAN_IMAP_MB_NOSELECT) != 0)
      continue;
    
    r = etpan_imap_sync_get_status(imap_sync, folder_name,
        &msg_count, &unseen_count, &recent_count);
    if (r < 0) {
      error = etpan_imap_status(imap_sync->ep_storage->sto_session,
          folder_name,
          &msg_count, &unseen_count, &recent_count);
      ETPAN_LOCAL_LOG("status %s %u %u %u %p", folder_name,
          msg_count, unseen_count, recent_count,
          error);
      if (error != NULL) {
        struct etpan_error * new_error;
        
        new_error = rewrite_error(imap_sync,
            ERROR_DOMAIN_FETCH_FOLDER_LIST, error);
        ETPAN_ERROR_FREE(error);
        error = new_error;
        
        goto free_mb_list;
      }
      error = etpan_imap_sync_set_status(imap_sync, folder_name,
          msg_count, unseen_count, recent_count);
      if (error != NULL) {
        struct etpan_error * new_error;
        
        new_error = rewrite_error(imap_sync,
            ERROR_DOMAIN_FETCH_FOLDER_LIST, error);
        ETPAN_ERROR_FREE(error);
        error = new_error;
        
        goto free_mb_list;
      }
    }
  }
  
  error = NULL;
  
 free_keys:
  etpan_sqldb_keys_free(keys);
 free_mb_list:
  chash_free(folder_hash);
  etpan_imap_list_mailboxes_free(imap_mb_list);
  
  return error;
}

static void add_folder(struct etpan_imap_sync * imap_sync,
    char * folder_name, int noinferiors, char separator)
{
  struct etpan_imap_folder_sync * folder_sync;
  chashdatum key;
  chashdatum value;
  int r;
  
  folder_sync = etpan_imap_folder_sync_new(imap_sync, folder_name,
      noinferiors, separator);
  
  key.data = folder_name;
  key.len = strlen(folder_name) + 1;
  value.data = folder_sync;
  value.len = 0;
  /* in a thread */
  pthread_mutex_lock(&imap_sync->folder_hash_lock);
  r = chash_set(imap_sync->folder_hash, &key, &value, NULL);
  pthread_mutex_unlock(&imap_sync->folder_hash_lock);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

void etpan_imap_sync_set_account(struct etpan_imap_sync * imap_sync,
    struct etpan_account * account)
{
  imap_sync->account = account;
}

struct etpan_account *
etpan_imap_sync_get_account(struct etpan_imap_sync * imap_sync)
{
  return imap_sync->account;
}

static void notify_op_end(struct etpan_imap_sync * imap_sync);


struct connect_result {
  struct etpan_error * error;
};

static void threaded_connect(struct etpan_thread_op * op);
static void threaded_connect_cleanup(struct etpan_thread_op * op);
static void threaded_connect_callback(int cancelled,
    struct connect_result * result,
    void * data);


static void connect_run_op(struct etpan_thread_manager_app * manager,
    struct etpan_imap_sync * imap_sync)
{
  struct etpan_thread_op * op;
  struct etpan_storage * storage;
  struct connect_result * result;
  
  op = etpan_thread_op_new();
  op->param = imap_sync;
  result = malloc(sizeof(* result));
  if (result == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  result->error = NULL;
  op->result = result;
  
  op->cancellable = 1;
  op->run = threaded_connect;
  op->callback = (void (*)(int, void *, void *)) threaded_connect_callback;
  op->callback_data = imap_sync;
  op->cleanup = threaded_connect_cleanup;
  
  storage = etpan_imap_sync_get_thread_storage(imap_sync);
  etpan_thread_manager_app_storage_schedule(manager, storage, op);
}

static void threaded_connect(struct etpan_thread_op * op)
{
  struct etpan_error * error;
  struct connect_result * result;
  struct etpan_imap_sync * imap_sync;
  
  result = op->result;
  imap_sync = op->param;
  
  error = connect(imap_sync);
  result->error = error;
}

static void threaded_connect_callback(int cancelled,
    struct connect_result * result,
    void * data)
{
  struct etpan_imap_sync * imap_sync;
  
  imap_sync = data;
  
  if (result->error != NULL) {
    etpan_storage_unsetup(imap_sync->dummy_storage);
    imap_sync->error = result->error;
    result->error = NULL;
    imap_sync->state = STATE_DISCONNECTED;
    imap_sync->last_sync_date = etpan_get_time();
  }
  else {
    imap_sync->state = STATE_CONNECTED;
  }
  notify_op_end(imap_sync);
}

static void threaded_connect_cleanup(struct etpan_thread_op * op)
{
  struct connect_result * result;
  
  result = op->result;
  ETPAN_ERROR_FREE(result->error);
  free(result);
}



struct sync_folder_list_result {
  struct etpan_error * error;
  int updated;
};

static void threaded_sync_folder_list(struct etpan_thread_op * op);
static void threaded_sync_folder_list_callback(int cancelled,
    struct sync_folder_list_result * result, void * data);
static void threaded_sync_folder_list_cleanup(struct etpan_thread_op * op);

static void sync_folder_list_run_op(struct etpan_thread_manager_app * manager,
    struct etpan_imap_sync * imap_sync)
{
  struct etpan_thread_op * op;
  struct etpan_storage * storage;
  struct sync_folder_list_result * result;
  
  ETPAN_LOCAL_LOG("folder list run op");
  ETPAN_ASSERT(imap_sync->op == NULL, "should have one operation %s", imap_sync->id);
  ETPAN_ASSERT(imap_sync->account != NULL, "account not setup for imap %s", imap_sync->id);
  
  result = malloc(sizeof(* result));
  if (result == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  result->error = NULL;
  result->updated = 0;
  
  op = etpan_thread_op_new();
  op->param = imap_sync;
  op->result = result;
  
  op->cancellable = 1;
  op->run = threaded_sync_folder_list;
  op->callback = (void (*)(int, void *, void *)) threaded_sync_folder_list_callback;
  op->callback_data = imap_sync;
  op->cleanup = threaded_sync_folder_list_cleanup;
  
  storage = etpan_imap_sync_get_thread_storage(imap_sync);
  etpan_thread_manager_app_storage_schedule(manager, storage, op);
  
  imap_sync->op = op;
}

static void threaded_sync_folder_list(struct etpan_thread_op * op)
{
  struct etpan_error * error;
  struct etpan_imap_sync * imap_sync;
  struct sync_folder_list_result * result;
  
  result = op->result;
  imap_sync = op->param;
  ETPAN_LOCAL_LOG("folder list sync");
  error = sync_folder_list(imap_sync);
  if (error != NULL) {
    result->error = error;
    return;
  }
  ETPAN_LOCAL_LOG("folder list sync step3");
  
  result->updated = update_folder_list_from_db(imap_sync);
}


static void notify_storage_content_updated(struct etpan_imap_sync * imap_sync);

static void threaded_sync_folder_list_callback(int cancelled,
    struct sync_folder_list_result * result, void * data)
{
  struct etpan_imap_sync * imap_sync;
  
  imap_sync = data;
  imap_sync->op = NULL;
  imap_sync->error = result->error;
  result->error = NULL;

  if (imap_sync->error != NULL) {
    if (etpan_error_get_code(imap_sync->error) == ERROR_STREAM) {
      imap_sync->state = STATE_STREAM_CLOSED;
    }
  }
  
  imap_sync->last_sync_date = etpan_get_time();
  imap_sync->main_imap_separator = imap_sync->threaded_main_imap_separator;
  
  if (result->updated) {
    notify_storage_content_updated(imap_sync);
  }
  
  notify_op_end(imap_sync);
}

static void threaded_sync_folder_list_cleanup(struct etpan_thread_op * op)
{
  struct sync_folder_list_result * result;
  
  result = op->result;
  ETPAN_ERROR_FREE(result->error);
  free(result);
}

static struct etpan_imap_folder_sync *
get_next_folder_sync(struct etpan_imap_sync * imap_sync)
{
  struct etpan_imap_folder_sync * result;  
  double min_date;
  chashiter * iter;
  double current_date;
  int result_priority;
  
  current_date = etpan_get_time();
  min_date = SYNC_FOLDERLIST_INTERVAL + imap_sync->last_sync_date;
  result = NULL;
  result_priority = 0;
  
  /* in main thread */
  pthread_mutex_lock(&imap_sync->folder_hash_lock);
  for(iter = chash_begin(imap_sync->folder_hash) ; iter != NULL ;
      iter = chash_next(imap_sync->folder_hash, iter)) {
    double date;
    struct etpan_imap_folder_sync * folder_sync;
    chashdatum value;
    
    chash_value(iter, &value);
    folder_sync = value.data;
    date = etpan_imap_folder_sync_next_op_date(folder_sync);
    if (date < current_date) {
      if (etpan_imap_folder_sync_get_priority(folder_sync) > result_priority) {
        result = folder_sync;
        result_priority = etpan_imap_folder_sync_get_priority(result);
        min_date = date;
      }
    }
    else if (date < min_date) {
      min_date = date;
      result = folder_sync;
      result_priority = etpan_imap_folder_sync_get_priority(result);
    }
  }
  pthread_mutex_unlock(&imap_sync->folder_hash_lock);
  
  return result;
}


static void folder_sync_op_ended_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);
static void disconnect_run_op(struct etpan_thread_manager_app * manager,
    struct etpan_imap_sync * imap_sync);

static void sync_next_op(struct etpan_thread_manager_app * manager,
                         struct etpan_imap_sync * imap_sync)
{
  switch (imap_sync->state) {
  case STATE_DISCONNECTED:
    connect_run_op(manager, imap_sync);
    break;
  case STATE_CONNECTED:
    if (imap_sync->should_disconnect) {
      disconnect_run_op(manager, imap_sync);
    }
    else {
      if (etpan_get_time() - imap_sync->last_sync_date > SYNC_FOLDERLIST_INTERVAL) {
        sync_folder_list_run_op(manager, imap_sync);
      }
      else {
        struct etpan_imap_folder_sync * folder_sync;
        
        folder_sync = get_next_folder_sync(imap_sync);
        if (folder_sync != NULL) {
          ETPAN_SIGNAL_ADD_HANDLER(folder_sync,
              ETPAN_IMAP_FOLDER_SYNC_OP_ENDED_SIGNAL,
              folder_sync_op_ended_handler, imap_sync);
          
          etpan_imap_folder_sync(manager, folder_sync);
        }
      }
    }
    break;
  case STATE_STREAM_CLOSED:
    connect_run_op(manager, imap_sync);
    break;
  }
}

static void folder_sync_op_ended_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_imap_sync * imap_sync;
  struct etpan_imap_folder_sync * folder_sync;
  struct etpan_error * error;
  
  folder_sync = sender;
  imap_sync = user_data;
  
  ETPAN_SIGNAL_REMOVE_HANDLER(folder_sync,
      ETPAN_IMAP_FOLDER_SYNC_OP_ENDED_SIGNAL,
      folder_sync_op_ended_handler, imap_sync);
  
  error = etpan_imap_folder_sync_get_error(folder_sync);
  if (error != NULL) {
    imap_sync->error = etpan_error_dup(error);
  }
  
  notify_op_end(imap_sync);
}

static void notify_op_end(struct etpan_imap_sync * imap_sync)
{
  imap_sync->syncing = 0;
  
  if (imap_sync->error != NULL)
    imap_sync->last_sync_success = 0;
  else
    imap_sync->last_sync_success = 1;
  
  if (imap_sync->error != NULL)
    etpan_error_log(imap_sync->error);
  
  ETPAN_SIGNAL_SEND(imap_sync, ETPAN_IMAP_SYNC_OP_ENDED_SIGNAL);
  
  ETPAN_ERROR_FREE(imap_sync->error);
  imap_sync->error = NULL;
}

static void remove_pending_folders(struct etpan_imap_sync * imap_sync);

/* return delay for next op */
void etpan_imap_sync(struct etpan_thread_manager_app * manager,
    struct etpan_imap_sync * imap_sync)
{
  ETPAN_ASSERT(!imap_sync->syncing, "already syncing");
  
  ETPAN_LOCAL_LOG("imap sync %s", imap_sync->id);
  imap_sync->syncing = 1;
  remove_pending_folders(imap_sync);
  sync_next_op(manager, imap_sync);
}

double etpan_imap_sync_next_op_date(struct etpan_imap_sync * imap_sync)
{
  struct etpan_imap_folder_sync * folder_sync;
  
  ETPAN_LOCAL_LOG("next op date ? %i %i", imap_sync->state, imap_sync->syncing);
  if (imap_sync->syncing) {
    return etpan_get_time() + 24 * 60 * 60;
  }
  
#if 0
  ETPAN_LOCAL_LOG("next op date %s", imap_sync->id);
#endif
  switch (imap_sync->state) {
  case STATE_DISCONNECTED:
    return SYNC_FOLDERLIST_INTERVAL + imap_sync->last_sync_date;
    
  case STATE_CONNECTED:
    ETPAN_LOCAL_LOG("next op date %i ?", imap_sync->should_disconnect);
    if (imap_sync->should_disconnect) {
      ETPAN_LOCAL_LOG("next op date disconnection");
      return etpan_get_time();
    }
    
    folder_sync = get_next_folder_sync(imap_sync);
    if (folder_sync != NULL) {
      return etpan_imap_folder_sync_next_op_date(folder_sync);
    }
    return SYNC_FOLDERLIST_INTERVAL + imap_sync->last_sync_date;
    
  case STATE_STREAM_CLOSED:
    return 0;
    
  default:
    ETPAN_CRASH("invalid state");
    return etpan_get_time() + 24 * 60 * 60;
  }
}

struct mailstorage * etpan_imap_sync_get_storage(struct etpan_imap_sync * imap_sync)
{
  return imap_sync->ep_storage;
}

char etpan_imap_sync_get_main_imap_separator(struct etpan_imap_sync * imap_sync)
{
  return imap_sync->main_imap_separator;
}

int etpan_imap_sync_get_status(struct etpan_imap_sync * imap_sync, char * location, unsigned int * p_count, unsigned int * p_unseen, unsigned int * p_recent)
{
  unsigned int count;
  unsigned int unseen;
  unsigned int recent;
  void * data;
  size_t length;
  int r;
  
  count = 0;
  unseen = 0;
  recent = 0;
  
  pthread_mutex_lock(&imap_sync->folderlist_db_lock);
  r = etpan_sqldb_get(imap_sync->folderlist_db,
      location, "msg_count", &data, &length);
  if (r < 0)
    goto unlock;
  count = decode_uint32(data, length);
  free(data);
  
  r = etpan_sqldb_get(imap_sync->folderlist_db,
      location, "unseen_count", &data, &length);
  if (r < 0)
    goto unlock;
  unseen = decode_uint32(data, length);
  free(data);
  
  r = etpan_sqldb_get(imap_sync->folderlist_db,
      location, "recent_count", &data, &length);
  if (r < 0)
    goto unlock;
  recent = decode_uint32(data, length);
  free(data);
  ETPAN_LOCAL_LOG("get status %s %s %u", imap_sync->id, location, unseen);
  pthread_mutex_unlock(&imap_sync->folderlist_db_lock);
  
  * p_count = count;
  * p_unseen = unseen;
  * p_recent = recent;
  
  return 0;
  
 unlock:
  pthread_mutex_unlock(&imap_sync->folderlist_db_lock);
  return -1;
}

struct etpan_error *
etpan_imap_sync_set_status(struct etpan_imap_sync * imap_sync, char * location, unsigned int count, unsigned int unseen, unsigned int recent)
{
  int r;
  int had_error;
  void * data;
  size_t length;
  
  ETPAN_LOCAL_LOG("set status %s %s %u %u %u", imap_sync->id, location, count, unseen, recent);
  had_error = 0;
  pthread_mutex_lock(&imap_sync->folderlist_db_lock);
  r = etpan_sqldb_begin_transaction(imap_sync->folderlist_db);
  if (r < 0)
    had_error = 1;
  encode_uint32(count, &data, &length);
  r = etpan_sqldb_set(imap_sync->folderlist_db,
      location, "msg_count", data, length);
  free(data);
  if (r < 0)
    had_error = 1;
  encode_uint32(unseen, &data, &length);
  r = etpan_sqldb_set(imap_sync->folderlist_db,
      location, "unseen_count", data, length);
  free(data);
  if (r < 0)
    had_error = 1;
  encode_uint32(recent, &data, &length);
  r = etpan_sqldb_set(imap_sync->folderlist_db,
      location, "recent_count", data, length);
  free(data);
  if (r < 0)
    had_error = 1;
  r = etpan_sqldb_end_transaction(imap_sync->folderlist_db);
  if (r < 0)
    had_error = 1;
  pthread_mutex_unlock(&imap_sync->folderlist_db_lock);
  
  if (had_error) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_DATABASE);
    etpan_error_set_short_description(error, _("Cache could not be written"));
    etpan_error_strf_long_description(error,
        _("Cache could not be written properly on disk."));
    
    return error;
  }
  
  return NULL;
}

static struct etpan_error *
set_db_value(struct etpan_imap_sync * imap_sync, char * location,
    char * value_name, uint32_t value)
{
  int r;
  int had_error;
  void * data;
  size_t length;
  
  pthread_mutex_lock(&imap_sync->folderlist_db_lock);
  had_error = 0;
  r = etpan_sqldb_begin_transaction(imap_sync->folderlist_db);
  if (r < 0)
    had_error = 1;
  encode_uint32(value, &data, &length);
  r = etpan_sqldb_set(imap_sync->folderlist_db,
      location, value_name, data, length);
  free(data);
  if (r < 0)
    had_error = 1;
  r = etpan_sqldb_end_transaction(imap_sync->folderlist_db);
  if (r < 0)
    had_error = 1;
  pthread_mutex_unlock(&imap_sync->folderlist_db_lock);
  
  if (had_error) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_DATABASE);
    etpan_error_set_short_description(error, _("Cache could not be written"));
    etpan_error_strf_long_description(error,
        _("Cache could not be written properly on disk."));
    
    return error;
  }
  
  return NULL;
}

static uint32_t get_db_value(struct etpan_imap_sync * imap_sync,
    char * location, char * value_name)
{
  uint32_t value;
  void * data;
  size_t length;
  int r;
  
  value = 0;
  pthread_mutex_lock(&imap_sync->folderlist_db_lock);
  r = etpan_sqldb_get(imap_sync->folderlist_db, location, value_name,
      &data, &length);
  pthread_mutex_unlock(&imap_sync->folderlist_db_lock);
  if (r == 0) {
    value = decode_uint32(data, length);
    free(data);
  }
  
  return value;
}

struct etpan_error *
etpan_imap_sync_set_lastuid(struct etpan_imap_sync * imap_sync,
    char * location, uint32_t value)
{
  return set_db_value(imap_sync, location, "lastuid", value);
}

uint32_t etpan_imap_sync_get_lastuid(struct etpan_imap_sync * imap_sync,
    char * location)
{
  return get_db_value(imap_sync, location, "lastuid");
}


struct etpan_error *
etpan_imap_sync_set_envlastuid(struct etpan_imap_sync * imap_sync,
    char * location, uint32_t value)
{
  return set_db_value(imap_sync, location, "envlastuid", value);
}

uint32_t etpan_imap_sync_get_envlastuid(struct etpan_imap_sync * imap_sync,
    char * location)
{
  return get_db_value(imap_sync, location, "envlastuid");
}

struct etpan_error *
etpan_imap_sync_set_bodylastuid(struct etpan_imap_sync * imap_sync,
    char * location, uint32_t value)
{
  return set_db_value(imap_sync, location, "bodylastuid", value);
}

uint32_t etpan_imap_sync_get_bodylastuid(struct etpan_imap_sync * imap_sync,
    char * location)
{
  return get_db_value(imap_sync, location, "bodylastuid");
}

struct etpan_error *
etpan_imap_sync_set_uidvalidity(struct etpan_imap_sync * imap_sync,
    char * location, uint32_t value)
{
  return set_db_value(imap_sync, location, "uidvalidity", value);
}

uint32_t etpan_imap_sync_get_uidvalidity(struct etpan_imap_sync * imap_sync,
    char * location)
{
  return get_db_value(imap_sync, location, "uidvalidity");
}

static struct etpan_error * rewrite_error(struct etpan_imap_sync * imap_sync,
    int domain,
    struct etpan_error * error)
{
  struct etpan_error * new_error;
  char * previous_long_description;
  
  new_error = etpan_error_new();
  etpan_error_set_code(new_error, etpan_error_get_code(error));
  etpan_error_set_short_description(new_error,
      etpan_error_get_short_description(error));
  previous_long_description = etpan_error_get_long_description(error);
  
  switch (domain) {
  case ERROR_DOMAIN_CONNECTION:
    etpan_error_strf_long_description(new_error,
        _("Connection to account %s failed. "
            "%s"),
        etpan_imap_sync_get_id(imap_sync),
        previous_long_description);
    break;
    
  case ERROR_DOMAIN_FETCH_FOLDER_LIST:
    etpan_error_strf_long_description(new_error,
        _("An error occurred while getting list of mailboxes on account %s.\n"
            "%s"),
        etpan_imap_sync_get_id(imap_sync),
        previous_long_description);
    break;

  case ERROR_DOMAIN_CREATE:
    etpan_error_strf_long_description(new_error,
        _("An error occurred while creating a mailbox on account %s.\n"
            "%s"),
        etpan_imap_sync_get_id(imap_sync),
        previous_long_description);
    break;
    
  case ERROR_DOMAIN_DELETE:
    etpan_error_strf_long_description(new_error,
        _("An error occurred while deleting a mailbox on account %s.\n"
            "%s"),
        etpan_imap_sync_get_id(imap_sync),
        previous_long_description);
    break;
    
  case ERROR_DOMAIN_RENAME:
    etpan_error_strf_long_description(new_error,
        _("An error occurred while changing the name of a mailbox on account %s.\n"
            "%s"),
        etpan_imap_sync_get_id(imap_sync),
        previous_long_description);
    break;
    
  default:
    etpan_error_strf_long_description(new_error,
        _("An error occurred with account %s.\n"
            "%s"),
        etpan_imap_sync_get_id(imap_sync),
        previous_long_description);
    break;
  }
  
  return new_error;
}

struct etpan_error * etpan_imap_sync_get_error(struct etpan_imap_sync * imap_sync)
{
  return imap_sync->error;
}


struct etpan_folder * sync_folder_to_folder(struct etpan_storage * storage,
    struct etpan_imap_folder_sync * folder_sync)
{
  char * location;
  struct etpan_folder * folder;
  char * vpath;
  char separator;
  size_t len;
  
  location = etpan_imap_folder_sync_get_location(folder_sync);
  separator = etpan_imap_folder_sync_get_imap_separator(folder_sync);
  
  len = strlen(location) + strlen(storage->id) + 2;
  vpath = malloc(len);
  snprintf(vpath, len, "%s/%s", storage->id, location);
  if (vpath == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  if (separator != '/') {
    char * p;
    
    p = vpath + strlen(storage->id) + 1;
    while ((p = strchr(p, '/')) != NULL) {
      * p = '-';
      p ++;
    }
    
    p = vpath;
    while ((p = strchr(p, separator)) != NULL) {
      * p = '/';
      p ++;
    }
  }
  ETPAN_LOCAL_LOG("vpath %s", vpath);
  
  folder = etpan_folder_imap_sync_new();
  
  etpan_folder_imap_sync_set_folder_sync(folder, folder_sync);
  etpan_folder_set_location(folder, location);
  etpan_folder_set_ui_path(folder, vpath);

  etpan_folder_set_storage(folder, storage);
  
  free(vpath);
  
  return folder;
}

void etpan_imap_sync_fetch_folder_list(struct etpan_storage * storage,
    struct etpan_imap_sync * imap_sync,
    chash * folder_list)
{
  chash * folder_sync_hash;
  chashiter * iter;
  
  /* in a thread */
  pthread_mutex_lock(&imap_sync->folder_hash_lock);
  folder_sync_hash = imap_sync->folder_hash;
  ETPAN_LOCAL_LOG("hash folder list : %i", chash_count(folder_sync_hash));
  for(iter = chash_begin(folder_sync_hash) ; iter != NULL ; iter = chash_next(folder_sync_hash, iter)) {
    chashdatum value;
    chashdatum key;
    struct etpan_imap_folder_sync * folder_sync;
    struct etpan_folder * folder;
    char * location;
    
    chash_key(iter, &key);
    chash_value(iter, &value);
    folder_sync = value.data;
    folder = sync_folder_to_folder(storage, folder_sync);
    location = etpan_imap_folder_sync_get_location(folder_sync);
    ETPAN_LOCAL_LOG("folder %s", location);
    if (strcasecmp(location, "INBOX") == 0) {
      etpan_folder_set_show_unfiltered(folder, 0);
    }
    
    value.data = folder;
    value.len = 0;
    chash_set(folder_list, &key, &value, NULL);
  }
  pthread_mutex_unlock(&imap_sync->folder_hash_lock);
}


static void disconnect(struct etpan_imap_sync * imap_sync)
{
  struct mailstorage * ep_storage;
  
  if (!imap_sync->connected)
    return;
  
  imap_sync->connected = 0;
  
  ep_storage = imap_sync->ep_storage;
  mailstorage_disconnect(ep_storage);
  mailstorage_free(ep_storage);
  imap_sync->ep_storage = NULL;
}

static void disconnect_with_folder(struct etpan_imap_sync * imap_sync)
{
  chash * folder_sync_hash;
  chashiter * iter;
  
  pthread_mutex_lock(&imap_sync->folder_hash_lock);
  folder_sync_hash = imap_sync->folder_hash;
  for(iter = chash_begin(folder_sync_hash) ; iter != NULL ; iter = chash_next(folder_sync_hash, iter)) {
    chashdatum value;
    chashdatum key;
    struct etpan_imap_folder_sync * folder_sync;
    
    chash_key(iter, &key);
    chash_value(iter, &value);
    folder_sync = value.data;
    etpan_imap_folder_sync_disconnect_nt(folder_sync);
  }
  chash_clear(folder_sync_hash);
  
  remove_pending_folders(imap_sync);
  
  pthread_mutex_unlock(&imap_sync->folder_hash_lock);
  
  disconnect(imap_sync);
}

void etpan_imap_sync_disconnect_nt(struct etpan_imap_sync * imap_sync)
{
  disconnect_with_folder(imap_sync);
}

static void threaded_disconnect(struct etpan_thread_op * op);
static void threaded_disconnect_callback(int cancelled,
    void * dummy,
    void * data);

static void disconnect_run_op(struct etpan_thread_manager_app * manager,
    struct etpan_imap_sync * imap_sync)
{
  struct etpan_thread_op * op;
  struct etpan_storage * storage;
  
  ETPAN_LOCAL_LOG("start imap sync disconnect");
  op = etpan_thread_op_new();
  op->param = imap_sync;
  op->result = NULL;
  
  op->cancellable = 1;
  op->run = threaded_disconnect;
  op->callback = (void (*)(int, void *, void *)) threaded_disconnect_callback;
  op->callback_data = imap_sync;
  op->cleanup = NULL;
  
  storage = etpan_imap_sync_get_thread_storage(imap_sync);
  etpan_thread_manager_app_storage_schedule(manager, storage, op);
}

static void threaded_disconnect(struct etpan_thread_op * op)
{
  struct etpan_imap_sync * imap_sync;
  
  imap_sync = op->param;
  
  disconnect_with_folder(imap_sync);
}

static void threaded_disconnect_callback(int cancelled,
    void * dummy,
    void * data)
{
  struct etpan_imap_sync * imap_sync;
  
  imap_sync = data;
  
  imap_sync->state = STATE_DISCONNECTED;
  etpan_storage_unsetup(imap_sync->dummy_storage);
  notify_op_end(imap_sync);
}

static void remove_pending_folders(struct etpan_imap_sync * imap_sync)
{
  unsigned int i;
  
  pthread_mutex_lock(&imap_sync->pending_folders_lock);
  if (imap_sync->disconnecting_folders) {
    pthread_mutex_unlock(&imap_sync->pending_folders_lock);
    return;
  }
  if (carray_count(imap_sync->pending_folders) > 0)
    ETPAN_LOCAL_LOG("remove pending folders %u",carray_count(imap_sync->pending_folders));
  for(i = 0 ; i < carray_count(imap_sync->pending_folders) ; i ++) {
    struct etpan_imap_folder_sync * folder_sync;
    
    folder_sync = carray_get(imap_sync->pending_folders, i);
    etpan_imap_folder_sync_unref(folder_sync);
  }
  carray_set_size(imap_sync->pending_folders, 0);
  pthread_mutex_unlock(&imap_sync->pending_folders_lock);
}

struct etpan_error *
etpan_imap_sync_force_fetch_folder_list(struct etpan_imap_sync * imap_sync)
{
  struct etpan_error * error;
  
  error = sync_folder_list(imap_sync);
  if (error != NULL)
    return error;
  
  update_folder_list_from_db(imap_sync);
  
  return NULL;
}

static void notify_storage_content_updated(struct etpan_imap_sync * imap_sync)
{
  ETPAN_SIGNAL_SEND(imap_sync,
      ETPAN_IMAP_SYNC_STORAGE_CONTENT_UPDATED_SIGNAL);
}

void etpan_imap_sync_disconnect(struct etpan_imap_sync * imap_sync)
{
  imap_sync->should_disconnect = 1;
}

int etpan_imap_sync_is_disconnected(struct etpan_imap_sync * imap_sync)
{
  return (imap_sync->state == STATE_DISCONNECTED);
}

struct etpan_storage * etpan_imap_sync_get_thread_storage(struct etpan_imap_sync * imap_sync)
{
  return imap_sync->dummy_storage;
}
