#include "etpan-message-imap-sync.h"
#include "etpan-message-imap-sync-private.h"

#include <stdlib.h>
#include <string.h>

#define ETPAN_MODULE_LOG_NAME "IMAPSYNC"

#include "etpan-imap-serialize.h"
#include "etpan-error.h"
#include "etpan-log.h"
#include "etpan-message.h"
#include "etpan-part.h"
#include "etpan-part-imap-sync.h"
#include "etpan-sqldb.h"
#include "etpan-folder.h"
#include "etpan-folder-imap-sync.h"
#include "etpan-folder-imap-sync-private.h"
#include "etpan-part-header.h"

static struct etpan_message_header * get_header(struct etpan_message * msg);

static struct etpan_error * check(struct etpan_message * msg,
    struct etpan_message_flags * flags);

static struct etpan_error * fetch_part_list(struct etpan_message * msg,
    struct etpan_part ** p_main_part);

static void clear_part_list(struct etpan_message * msg);

static void free_data(struct etpan_message * msg);

static struct etpan_message_driver imap_driver = {
  .name = "imap",
  
  .get_header = get_header,
  
  .check = check,
  
  .fetch_part_list = fetch_part_list,
  .clear_part_list = clear_part_list,
  
  .free_data = free_data,
};



struct message_data {
  char * uid;
  struct mailimap_body * body;
};

struct etpan_message * etpan_message_imap_sync_new(void)
{
  struct etpan_message * msg;
  struct message_data * data;
  
  msg = etpan_message_new();
  if (msg == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  data = malloc(sizeof(* data));
  if (data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  data->uid = NULL;
  data->body = NULL;
  
  etpan_message_set_data(msg, data);
  etpan_message_set_driver(msg, &imap_driver);
  
  return msg;
}

void etpan_message_imap_sync_set_uid(struct etpan_message * msg, char * uid)
{
  struct message_data * data;
  char * folder_uid;
  char * msg_uid;
  size_t len;
  struct etpan_message_flags * flags;
  
  if (uid == NULL)
    ETPAN_CRASH("uid should not be NULL");
  
  data = etpan_message_get_data(msg);
  free(data->uid);
  data->uid = strdup(uid);
  if (data->uid == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  folder_uid = etpan_folder_get_uid(msg->folder);
  len = strlen(uid) + strlen(folder_uid) + 2;
  msg_uid = malloc(len);
  if (msg_uid == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  snprintf(msg_uid, len, "%s-%s", folder_uid, uid);
  msg_uid[len - 1] = '\0';
  etpan_message_set_uid(msg, msg_uid);
  free(msg_uid);
}

char * etpan_message_imap_sync_get_uid(struct etpan_message * msg)
{
  struct message_data * data;
  
  data = etpan_message_get_data(msg);
  
  return data->uid;
}

static struct etpan_message_header * get_header(struct etpan_message * msg)
{
  ETPAN_CRASH("should not be called");
  
  return NULL;
}

static struct etpan_error * check(struct etpan_message * msg,
    struct etpan_message_flags * flags)
{
  struct etpan_folder * folder;
  
  folder = etpan_message_get_folder(msg);
  etpan_folder_imap_sync_check_msg(folder, msg, flags);
  
  return NULL; 
}

static struct etpan_part *
recursive_part_from_imap_body(struct etpan_message * msg,
    char * base_part_name,
    struct mailimap_body * imap_body);

static struct etpan_part *
imap_body_type_msg_to_body(struct etpan_message * msg,
    char * base_part_name,
    struct mailimap_body_type_msg * imap_type_msg)
{
  struct etpan_part * child;
  struct etpan_part * part;
  char * part_name;
  size_t len;

  part = etpan_part_imap_sync_new();
  etpan_part_set_type(part, ETPAN_PART_TYPE_MESSAGE);
  
  len = strlen(base_part_name) + 2;
  part_name = malloc(len + 1);
  if (part_name == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  if (imap_type_msg->bd_body->bd_type == MAILIMAP_BODY_MPART) {
    /* if child is multipart */
    snprintf(part_name, len + 1, "%s", base_part_name);
  }
  else if (* base_part_name == '\0') {
    snprintf(part_name, len + 1, "1");
  }
  else {
    snprintf(part_name, len + 1, "%s.1", base_part_name);
  }
  child = recursive_part_from_imap_body(msg, part_name,
      imap_type_msg->bd_body);
  etpan_part_add_child(part, child);
  free(part_name);
  
  return part;
}

static struct etpan_part *
imap_body_type_1part_to_body(struct etpan_message * msg,
    char * base_part_name,
    struct mailimap_body_type_1part * type_1part)
{
  struct etpan_part * part;
  
  part = NULL;
  switch (type_1part->bd_type) {
  case MAILIMAP_BODY_TYPE_1PART_BASIC:
    part = etpan_part_imap_sync_new();
    etpan_part_set_type(part, ETPAN_PART_TYPE_SINGLE);
    break;
  case MAILIMAP_BODY_TYPE_1PART_MSG:
    part = imap_body_type_msg_to_body(msg, base_part_name,
        type_1part->bd_data.bd_type_msg);
    break;
  case MAILIMAP_BODY_TYPE_1PART_TEXT:
    part = etpan_part_imap_sync_new();
    etpan_part_set_type(part, ETPAN_PART_TYPE_SINGLE);
    break;
  default:
    ETPAN_CRASH("invalid body");
    break;
  }
  
  return part;
}

static struct etpan_part *
imap_body_type_mpart_to_body(struct etpan_message * msg,
    char * base_part_name,
    struct mailimap_body_type_mpart * type_mpart)
{
  struct etpan_part * child;
  struct etpan_part * part;
  char * part_name;
  size_t len;
  clistiter * cur;
  unsigned int i;
  
  part = etpan_part_imap_sync_new();
  etpan_part_set_type(part, ETPAN_PART_TYPE_MULTIPLE);
  
  len = strlen(base_part_name) + 20;
  part_name = malloc(len + 1);
  if (part_name == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  i = 1;
  for(cur = clist_begin(type_mpart->bd_list) ; cur != NULL ; cur = clist_next(cur)) {
    struct mailimap_body * sub_body;
    
    sub_body = clist_content(cur);
    
    if (* base_part_name == '\0')
      snprintf(part_name, len + 1, "%u", i);
    else
      snprintf(part_name, len + 1, "%s.%u", base_part_name, i);
    child = recursive_part_from_imap_body(msg, part_name, sub_body);
    etpan_part_add_child(part, child);
    i ++;
  }
  free(part_name);
  
  return part;
}


static struct etpan_part *
recursive_part_from_imap_body(struct etpan_message * msg,
    char * base_part_name,
    struct mailimap_body * imap_body)
{
  struct etpan_part * part;
  size_t len;
  char * part_name;
  
  part = NULL;
  switch (imap_body->bd_type) {
  case MAILIMAP_BODY_1PART:
    part = imap_body_type_1part_to_body(msg, base_part_name,
        imap_body->bd_data.bd_body_1part);
    break;
  case MAILIMAP_BODY_MPART:
    part = imap_body_type_mpart_to_body(msg, base_part_name,
        imap_body->bd_data.bd_body_mpart);
    break;
  default:
    ETPAN_CRASH("invalid body");
    break;
  }
  
  len = strlen(base_part_name) + strlen("part-");
  part_name = malloc(len + 1);
  if (part_name == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  snprintf(part_name, len + 1, "part-%s", base_part_name);
  etpan_part_set_uid(part, part_name);
  free(part_name);
  
  etpan_part_imap_sync_set_section(part, base_part_name);
  etpan_part_imap_sync_set_body(part, imap_body);
  etpan_part_set_message(part, msg);
  
  return part;
}

static struct etpan_error * fetch_part_list(struct etpan_message * msg,
    struct etpan_part ** p_main_part)
{
  struct etpan_part * part;
  struct etpan_error * error;
  struct mailimap_body * imap_body;
  struct message_data * data;
  struct etpan_folder * folder;
  struct etpan_part * main_part;
  struct etpan_part_header * header;
  
  data = etpan_message_get_data(msg);
  folder = etpan_message_get_folder(msg);
  error = etpan_folder_imap_sync_get_body(folder, data->uid, &imap_body);
  if (error != NULL) {
    return error;
  }
  
  data->body = imap_body;
  switch (imap_body->bd_type) {
  case MAILIMAP_BODY_1PART:
    part = recursive_part_from_imap_body(msg, "1", imap_body);
    break;
  case MAILIMAP_BODY_MPART:
    part = recursive_part_from_imap_body(msg, "", imap_body);
    break;
  default:
    part = NULL;
    ETPAN_CRASH("invalid body");
    break;
  }
  
  main_part = etpan_part_imap_sync_new();
  etpan_part_set_message(main_part, msg);
  etpan_part_set_type(main_part, ETPAN_PART_TYPE_MESSAGE);
  etpan_part_imap_sync_set_body(main_part, NULL);
  etpan_part_imap_sync_set_section(main_part, "");
  etpan_part_set_uid(main_part, "part-");
  header = etpan_part_header_new();
  etpan_part_header_set_content_type(header, "message/rfc822");
  etpan_part_header_set_mime_encoding(header, ETPAN_PART_MIME_ENCODING_OTHER);
  etpan_part_set_header(main_part, header);
  etpan_part_add_child(main_part, part);
  
  * p_main_part = main_part;
  
  return NULL;
}

static void clear_part_list(struct etpan_message * msg)
{
  struct message_data * data;
  
  data = etpan_message_get_data(msg);
  if (data->body == NULL) {
    ETPAN_LOG("imap_sync clear_part_list should not be called");
  }
  else {
    mailimap_body_free(data->body);
    data->body = NULL;
  }
}

static void free_data(struct etpan_message * msg)
{
  struct message_data * data;
  
  data = etpan_message_get_data(msg);
  if (data->body != NULL)
    mailimap_body_free(data->body);
  free(data->uid);
  free(data);
}

struct etpan_error *
etpan_message_imap_sync_get_part(struct etpan_message * msg,
    char * section, int part_type,
    void ** p_data, size_t * p_length)
{
  char * uid;
  struct etpan_folder * folder;
  struct message_data * data;
  
  data = etpan_message_get_data(msg);
  folder = etpan_message_get_folder(msg);
  
  return etpan_folder_imap_sync_get_part(folder, data->uid, section,
      part_type, p_data, p_length);
}
