#include "etpan-part-imap-sync.h"

#include <stdlib.h>
#include <string.h>

#include "etpan-error.h"
#include "etpan-part-header.h"
#include "etpan-nls.h"
#include "etpan-lep.h"
#include "etpan-imap-folder-sync.h"
#include "etpan-message-imap-sync-private.h"
#include "etpan-imap.h"

#define ETPAN_MODULE_LOG_NAME "IMAPSYNC"
#include "etpan-log.h"

enum {
  ERROR_DOMAIN_FETCH,
};

static struct etpan_part_header * get_header(struct etpan_part * part);

static struct etpan_error * fetch(struct etpan_part * part,
    char ** p_content, size_t * p_length);

static struct etpan_error * fetch_header(struct etpan_part * part,
    char ** p_content, size_t * p_length);

static struct etpan_error * fetch_mime_header(struct etpan_part * part,
    char ** p_content, size_t * p_length);

static void free_data(struct etpan_part * part);

static struct etpan_part_driver imap_driver = {
  .name = "imap-sync",
  
  .get_header = get_header,
  
  .fetch = fetch,
  .fetch_header = fetch_header,
  .fetch_mime_header = fetch_mime_header,
  
  .free_data = free_data,
};

struct part_data {
  struct mailimap_body * body;
  char * section;
};

struct etpan_part * etpan_part_imap_sync_new(void)
{
  struct etpan_part * part;
  struct part_data * data;
  
  part = etpan_part_new();
  if (part == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  data = malloc(sizeof(* data));
  if (data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  data->body = NULL;
  data->section = NULL;
  
  etpan_part_set_data(part, data);
  etpan_part_set_driver(part, &imap_driver);
  
  return part;
}

void etpan_part_imap_sync_set_body(struct etpan_part * part,
    struct mailimap_body * body)
{
  struct part_data * data;
  
  data = etpan_part_get_data(part);
  data->body = body;
}

struct mailimap_body * etpan_part_imap_sync_get_body(struct etpan_part * part)
{
  struct part_data * data;
  
  data = etpan_part_get_data(part);
  return data->body;
}

static struct etpan_error * fetch(struct etpan_part * part,
    char ** p_content, size_t * p_length)
{
  struct part_data * data;
  int r;
  void * content;
  size_t length;
  struct etpan_message * msg;
  struct etpan_error * error;
  
  data = etpan_part_get_data(part);
  
  ETPAN_LOCAL_LOG("fetch body %s", data->section);
  msg = etpan_part_get_message(part);
  error = etpan_message_imap_sync_get_part(msg, data->section,
      ETPAN_IMAP_PART_TYPE_MAIN,
      &content, &length);
  if (error != NULL) {
    return error;
  }
  
  * p_content = content;
  * p_length = length;
  
  return NULL;
}

static struct etpan_error * fetch_header(struct etpan_part * part,
    char ** p_content, size_t * p_length)
{
  struct part_data * data;
  int r;
  void * content;
  size_t length;
  struct etpan_message * msg;
  struct etpan_error * error;
  
  data = etpan_part_get_data(part);
  
  ETPAN_LOCAL_LOG("fetch header %s", data->section);
  msg = etpan_part_get_message(part);
  error = etpan_message_imap_sync_get_part(msg, data->section,
      ETPAN_IMAP_PART_TYPE_HEADER,
      &content, &length);
  if (error != NULL) {
    return error;
  }
  
  * p_content = content;
  * p_length = length;
  
  return NULL;
}

static struct etpan_error * fetch_mime_header(struct etpan_part * part,
    char ** p_content, size_t * p_length)
{
  struct part_data * data;
  int r;
  void * content;
  size_t length;
  struct etpan_message * msg;
  struct etpan_error * error;
  
  data = etpan_part_get_data(part);
  
  msg = etpan_part_get_message(part);
  error = etpan_message_imap_sync_get_part(msg, data->section,
      ETPAN_IMAP_PART_TYPE_MIME,
      &content, &length);
  if (error != NULL) {
    return error;
  }
  
  * p_content = content;
  * p_length = length;
  
  return NULL;
}

static void free_data(struct etpan_part * part)
{
  struct part_data * data;
  
  data = etpan_part_get_data(part);
  free(data->section);
  free(data);
}

static struct etpan_part_header * get_header(struct etpan_part * part)
{
  struct mailmime_fields * mime_fields;
  struct mailmime_content * mime_content;
  struct etpan_part_header * header;
  struct mailimap_body * body;
  uint32_t mime_size;
  struct part_data * data;
  
  data = etpan_part_get_data(part);
  body = data->body;
  mime_fields = NULL;
  mime_content = NULL;
  mime_size = 0;
  etpan_imap_get_fields_content_from_body(body, &mime_fields,
      &mime_content, &mime_size);
  
  header = etpan_mime_header_from_lep(mime_fields, mime_content);
  ETPAN_LOCAL_LOG("section: %s %s", data->section, etpan_part_header_get_content_type(header));
  
  return header;
}


void etpan_part_imap_sync_set_section(struct etpan_part * part,
    char * section)
{
  struct part_data * data;
  
  ETPAN_LOCAL_LOG("set section %s", section);
  data = etpan_part_get_data(part);
  if (section != data->section) {
    free(data->section);
    if (section != NULL) {
      data->section = strdup(section);
      if (data->section == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      data->section = NULL;
  }
}
