#include "etpan-storage-imap.h"
#include "etpan-storage-imap-private.h"

#include <libetpan/libetpan.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#define ETPAN_MODULE_LOG_NAME "IMAPSYNC"

#include "etpan-log.h"
#include "etpan-error.h"
#include "etpan-storage.h"
#include "etpan-storage-private.h"
#include "etpan-folder.h"
#include "etpan-imap-sync.h"
#include "etpan-imap-sync-private.h"
#include "etpan-imap.h"
#include "etpan-nls.h"
#include "etpan-imap-folder-sync.h"
#include "etpan-folder-imap-sync.h"
#include "etpan-thread-manager-app.h"
#include "etpan-folder-imap-sync.h"
#include "etpan-folder-imap-sync-private.h"
#include "etpan-signal.h"

enum {
  ERROR_DOMAIN_CONNECTION,
  ERROR_DOMAIN_FETCH_FOLDER_LIST,
  ERROR_DOMAIN_CREATE,
  ERROR_DOMAIN_DELETE,
  ERROR_DOMAIN_RENAME,
};

static struct etpan_error * rewrite_error(struct etpan_storage * storage,
                                          int domain,
                                          struct etpan_error * error);

static struct etpan_error * setup(struct etpan_storage * storage);

static void free_data(struct etpan_storage * storage);

static struct etpan_error * create_folder(struct etpan_storage * storage,
    char * folder_name);

static struct etpan_error * delete_folder(struct etpan_storage * storage,
    char * folder_name);

static struct etpan_error * rename_folder(struct etpan_storage * storage,
    char * folder_name, char * new_name);

static struct etpan_error * fetch_folder_list(struct etpan_storage * storage,
    chash * folder_list);

static struct etpan_error * connect(struct etpan_storage * storage);

static void disconnect(struct etpan_storage * storage);

static char * get_sub_folder_location(struct etpan_storage * storage,
    struct etpan_folder * parent, char * name);

static int allows_sub_folders(struct etpan_storage * storage,
    struct etpan_folder * parent);

static struct etpan_storage_driver imap_driver = {
  .name = "imap-sync",
  
  .network = 1,
  
  .setup = setup,
  .create_folder = create_folder,
  .delete_folder = delete_folder,
  .rename_folder = rename_folder,
  .fetch_folder_list = fetch_folder_list,
  .connect = connect,
  .disconnect = disconnect,
  .free_data = free_data,
  
  .get_sub_folder_location = get_sub_folder_location,
  .allows_sub_folders = allows_sub_folders,
};

struct storage_data {
  char * hostname;
  int port;
  int connection_type;
  int auth_type;
  char * username;
  char * password;
  char * command;
  char * base_location;
  
  /* threaded calls */
  char * threaded_hostname;
  int threaded_port;
  int threaded_connection_type;
  int threaded_auth_type;
  char * threaded_username;
  char * threaded_password;
  char * threaded_command;
  char * threaded_base_location;
  struct etpan_imap_sync * imap_sync;
};

static void content_updated_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);

static struct etpan_error * connect(struct etpan_storage * storage)
{
  struct storage_data * data;
  struct etpan_imap_sync * imap_sync;
  struct etpan_account * account;
  
  data = etpan_storage_get_data(storage);
  
  if (data->imap_sync != NULL)
    return NULL;
  
  imap_sync = etpan_imap_sync_new();
  
  etpan_imap_sync_set_id(imap_sync, etpan_storage_get_id(storage));
  etpan_imap_sync_set_hostname(imap_sync, data->threaded_hostname);
  etpan_imap_sync_set_port(imap_sync, data->threaded_port);
  etpan_imap_sync_set_connection_type(imap_sync, data->threaded_connection_type);
  etpan_imap_sync_set_auth_type(imap_sync, data->threaded_auth_type);
  etpan_imap_sync_set_username(imap_sync, data->threaded_username);
  etpan_imap_sync_set_password(imap_sync, data->threaded_password);
  etpan_imap_sync_set_command(imap_sync, data->threaded_command);
  etpan_imap_sync_set_base_location(imap_sync, data->threaded_base_location);
  
  account = etpan_storage_get_account(storage);
  etpan_imap_sync_set_account(imap_sync, account);
  
  etpan_imap_sync_setup(imap_sync);
  data->imap_sync = imap_sync;
  
  ETPAN_SIGNAL_ADD_HANDLER(imap_sync,
      ETPAN_IMAP_SYNC_STORAGE_CONTENT_UPDATED_SIGNAL,
      content_updated_handler, storage);
  
  return NULL;
}

static void content_updated_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_storage * storage;
  
  storage = user_data;
  
  ETPAN_SIGNAL_SEND(storage,
      ETPAN_STORAGE_IMAP_SYNC_CONTENT_UPDATED_ENDED_SIGNAL);
}

struct disconnect_data {
  struct etpan_storage * storage;
  struct mailsem * sem;
};

static void disconnect_in_main_thread(void * data);

static void disconnect(struct etpan_storage * storage)
{
  struct storage_data * data;
  struct etpan_imap_sync * imap_sync;
  struct disconnect_data * disconnect_data;
  
  data = etpan_storage_get_data(storage);
  
  if (data->imap_sync == NULL)
    return;
  
  imap_sync = data->imap_sync;
  
  disconnect_data = malloc(sizeof(* data));
  if (disconnect_data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  disconnect_data->sem = mailsem_new();
  if (disconnect_data->sem == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  disconnect_data->storage = storage;
  
  etpan_thread_manager_app_run_in_main_thread(etpan_thread_manager_app_get_default(), disconnect_in_main_thread, disconnect_data, 0);
  
  mailsem_down(disconnect_data->sem);
  mailsem_free(disconnect_data->sem);
  free(disconnect_data);
}

static void disconnecting_op_ended_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);

static void disconnect_in_main_thread(void * data)
{
  struct disconnect_data * disconnect_data;
  struct storage_data * storage_data;
  struct etpan_imap_sync * imap_sync;
  
  disconnect_data = data;
  
  storage_data = etpan_storage_get_data(disconnect_data->storage);
  imap_sync = storage_data->imap_sync;
  ETPAN_LOG("imap sync start disconnection");
  
  if (etpan_imap_sync_is_disconnected(imap_sync)) {
    ETPAN_LOG("already disconnected");
    mailsem_up(disconnect_data->sem);
    return;
  }
  
  ETPAN_SIGNAL_ADD_HANDLER(imap_sync, ETPAN_IMAP_SYNC_OP_ENDED_SIGNAL,
      disconnecting_op_ended_handler, disconnect_data);
  
  etpan_imap_sync_disconnect(imap_sync);
  etpan_imap_sync(etpan_thread_manager_app_get_default(), imap_sync);
}

static void disconnecting_op_ended_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_storage * storage;
  struct etpan_imap_sync * imap_sync;
  struct disconnect_data * disconnect_data;
  struct mailsem * sem;
  
  imap_sync = sender;
  if (!etpan_imap_sync_is_disconnected(imap_sync)) {
    return;
  }
  
  ETPAN_LOG("imap sync disconnected");
  disconnect_data = user_data;
  ETPAN_SIGNAL_REMOVE_HANDLER(imap_sync, ETPAN_IMAP_SYNC_OP_ENDED_SIGNAL,
      disconnecting_op_ended_handler, disconnect_data);
  storage = disconnect_data->storage;
  sem = disconnect_data->sem;
  
  mailsem_up(sem);
}

struct etpan_storage * etpan_storage_imap_sync_new(void)
{
  struct storage_data * data;
  struct etpan_storage * storage;
  
  storage = etpan_storage_new();
  data = malloc(sizeof(* data));
  if (data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  data->hostname = NULL;
  data->port = 0;
  data->connection_type = ETPAN_CONNECTION_PLAIN;
  data->auth_type = ETPAN_AUTH_PASSWORD;
  data->username = NULL;
  data->password = NULL;
  data->command = NULL;
  data->base_location = NULL;
  data->threaded_hostname = NULL;
  data->threaded_port = 0;
  data->threaded_connection_type = ETPAN_CONNECTION_PLAIN;
  data->threaded_auth_type = ETPAN_AUTH_PASSWORD;
  data->threaded_username = NULL;
  data->threaded_password = NULL;
  data->threaded_command = NULL;
  data->threaded_base_location = NULL;
  data->imap_sync = NULL;
  
  etpan_storage_set_data(storage, data);
  etpan_storage_set_driver(storage, &imap_driver);
  
  return storage;
}

static void free_data(struct etpan_storage * storage)
{
  struct storage_data * data;
  struct etpan_imap_sync * imap_sync;
  
  data = etpan_storage_get_data(storage);
  
  imap_sync = data->imap_sync;
  
  ETPAN_SIGNAL_REMOVE_HANDLER(imap_sync,
      ETPAN_IMAP_FOLDER_SYNC_CONTENT_UPDATED_SIGNAL,
      content_updated_handler, storage);
  
  etpan_imap_sync_unsetup(imap_sync);
  etpan_imap_sync_free(imap_sync);
  data->imap_sync = NULL;
  
  free(data->threaded_base_location);
  free(data->threaded_command);
  free(data->threaded_password);
  free(data->threaded_username);
  free(data->threaded_hostname);
  
  free(data->command);
  free(data->password);
  free(data->username);
  free(data->base_location);
  free(data->hostname);
  free(data);
}

static struct etpan_error * create_folder(struct etpan_storage * storage,
    char * folder_name)
{
  struct mailstorage * ep_storage;
  struct storage_data * data;
  struct etpan_error * error;
  struct etpan_error * other_error;
  
  data = etpan_storage_get_data(storage);
  ep_storage = etpan_imap_sync_get_storage(data->imap_sync);
  
  error = etpan_imap_create(ep_storage->sto_session, folder_name);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(storage, ERROR_DOMAIN_CREATE, error);
    ETPAN_ERROR_FREE(error);
    
    error = new_error;
    goto exit;
  }
  
  error = NULL;
  
 exit:
  other_error = etpan_imap_sync_force_fetch_folder_list(data->imap_sync);
  if (other_error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(storage, ERROR_DOMAIN_RENAME, other_error);
    ETPAN_ERROR_FREE(other_error);
    
    if (error != NULL)
      ETPAN_ERROR_FREE(other_error);
    else
      error = new_error;
  }
  
  return error;
}

static struct etpan_error * delete_folder(struct etpan_storage * storage,
    char * folder_name)
{
  struct mailstorage * ep_storage;
  struct storage_data * data;
  struct etpan_error * error;
  struct etpan_error * other_error;
  
  data = etpan_storage_get_data(storage);
  ep_storage = etpan_imap_sync_get_storage(data->imap_sync);
  
  error = etpan_imap_delete(ep_storage->sto_session, folder_name);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(storage, ERROR_DOMAIN_DELETE, error);
    ETPAN_ERROR_FREE(error);
    
    error = new_error;
    goto exit;
  }
  
  error = NULL;
  
 exit:
  other_error = etpan_imap_sync_force_fetch_folder_list(data->imap_sync);
  if (other_error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(storage, ERROR_DOMAIN_RENAME, other_error);
    ETPAN_ERROR_FREE(other_error);
    
    if (error != NULL)
      ETPAN_ERROR_FREE(other_error);
    else
      error = new_error;
  }
  
  return error;
}

void etpan_storage_imap_sync_set_hostname(struct etpan_storage * storage,
    char * hostname)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  if (hostname != data->hostname) {
    free(data->hostname);
    if (hostname != NULL) {
      data->hostname = strdup(hostname);
      if (data->hostname == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      data->hostname = NULL;
  }
}

char * etpan_storage_imap_sync_get_hostname(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  return data->hostname;
}

void etpan_storage_imap_sync_set_port(struct etpan_storage * storage, int port)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  data->port = port;
}

int etpan_storage_imap_sync_get_port(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  return data->port;
}

void etpan_storage_imap_sync_set_connection_type(struct etpan_storage * storage,
    int connection_type)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  data->connection_type = connection_type;
}

int etpan_storage_imap_sync_get_connection_type(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  return data->connection_type;
}

void etpan_storage_imap_sync_set_auth_type(struct etpan_storage * storage,
    int auth_type)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  data->auth_type = auth_type;
}

int etpan_storage_imap_sync_get_auth_type(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  return data->auth_type;
}

void etpan_storage_imap_sync_set_username(struct etpan_storage * storage,
    char * username)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  if (username != data->username) {
    free(data->username);
    if (username != NULL) {
      data->username = strdup(username);
      if (data->username == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      data->username = NULL;
  }
}

char * etpan_storage_imap_sync_get_username(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  return data->username;
}

void etpan_storage_imap_sync_set_password(struct etpan_storage * storage,
    char * password)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  if (password != data->password) {
    free(data->password);
    if (password != NULL) {
      data->password = strdup(password);
      if (data->password == NULL)
         ETPAN_LOG_MEMORY_ERROR;
    }
    else
      data->password = NULL;
  }
}

char * etpan_storage_imap_sync_get_password(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  return data->password;
}

void etpan_storage_imap_sync_set_base_location(struct etpan_storage * storage,
    char * base_location)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  if (base_location != data->base_location) {
    free(data->base_location);
    if (base_location != NULL) {
      data->base_location = strdup(base_location);
      if (data->base_location == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      data->base_location = NULL;
  }
}

char * etpan_storage_imap_sync_get_base_location(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  return data->base_location;
}

void etpan_storage_imap_sync_set_command(struct etpan_storage * storage,
    char * command)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  if (command != data->command) {
    free(data->command);
    if (command != NULL) {
      data->command = strdup(command);
      if (data->command == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      data->command = NULL;
  }
}

char * etpan_storage_imap_sync_get_command(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  return data->command;
}

static struct etpan_error * setup(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  
  free(data->threaded_hostname);
  if (data->hostname != NULL) {
    data->threaded_hostname = strdup(data->hostname);
    if (data->threaded_hostname == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  else {
    data->threaded_hostname = NULL;
  }
  
  data->threaded_port = data->port;
  data->threaded_connection_type = data->connection_type;
  data->threaded_auth_type = data->auth_type;
  
  free(data->threaded_username);
  if (data->username != NULL) {
    data->threaded_username = strdup(data->username);
    if (data->threaded_username == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  else {
    data->threaded_username = NULL;
  }
  
  free(data->threaded_password);
  if (data->password != NULL) {
    data->threaded_password = strdup(data->password);
    if (data->threaded_password == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  else {
    data->threaded_password = NULL;
  }
  
  free(data->threaded_command);
  if (data->command != NULL) {
    data->threaded_command = strdup(data->command);
    if (data->threaded_command == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  else {
    data->threaded_command = NULL;
  }
  
  free(data->threaded_base_location);
  if (data->base_location != NULL) {
    data->threaded_base_location = strdup(data->base_location);
    if (data->threaded_base_location == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  } 
  else {
    data->threaded_base_location = NULL;
  }
  
  return NULL;
}

static char * get_sub_folder_location(struct etpan_storage * storage,
    struct etpan_folder * parent, char * name)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  
  /* because of (1), this function has to be called
     after a fetch of folder list */
  
  if (parent == NULL) {
    char * base_location;
    
    base_location = etpan_storage_imap_get_base_location(storage);
    if (base_location == NULL) {
      char * result;
      
      result = strdup(name);
      if (result == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      
      return result;
    }
    else {
      char * location;
      size_t size;
      char separator;
      
      separator = etpan_imap_sync_get_main_imap_separator(data->imap_sync);
      
      if (separator == 0) {
        ETPAN_WARN_LOG("IMAP separator not fetched yet");
        etpan_crash();
      }
      
      size = strlen(base_location) + strlen(name) + 2;
      location = malloc(size);
      if (location == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      
      snprintf(location, size, "%s%c%s", base_location, separator, name);
      
      return location;
    }
  }
  else {
    char * parent_location;
    char * location;
    size_t size;
    char separator;
    struct etpan_imap_folder_sync * folder_sync;
    
    parent_location = etpan_folder_get_location(parent);
    size = strlen(parent_location) + strlen(name) + 2;
    location = malloc(size);
    if (location == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    
    folder_sync = etpan_folder_imap_sync_get_folder_sync(parent);
    separator = etpan_imap_folder_sync_get_imap_separator(folder_sync);
    snprintf(location, size, "%s%c%s", parent_location, separator, name);
    
    return location;
  }
}

static int allows_sub_folders(struct etpan_storage * storage,
    struct etpan_folder * parent)
{
  (void) storage;
  
  if (parent == NULL)
    return 1;
  else {
    char * location;
    struct storage_data * data;
    struct etpan_imap_folder_sync * folder_sync;
    
    data = etpan_storage_get_data(storage);
    
    location = etpan_folder_get_location(parent);
    folder_sync = etpan_folder_imap_sync_get_folder_sync(parent); 
    return (!etpan_imap_folder_sync_get_noinferiors(folder_sync));
  }
}

static struct etpan_error * rename_folder(struct etpan_storage * storage,
    char * folder_name, char * new_name)
{
  struct mailstorage * ep_storage;
  struct storage_data * data;
  struct etpan_imap_mailbox_info * imap_folder;
  char imap_separator;
  carray * imap_mb_list;
  unsigned int i;
  size_t len;
  char pattern[PATH_MAX];
  struct etpan_error * error;
  struct etpan_error * other_error;
  
  data = etpan_storage_get_data(storage);
  ep_storage = etpan_imap_sync_get_storage(data->imap_sync);
  
  error = etpan_imap_list_mailboxes(ep_storage->sto_session,
      folder_name, &imap_mb_list);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(storage, ERROR_DOMAIN_RENAME, error);
    ETPAN_ERROR_FREE(error);
    
    error = new_error;
    
    goto err;
  }

  if (carray_count(imap_mb_list) == 0) {
    struct etpan_error * new_error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FOLDER_NOT_FOUND);
    etpan_error_set_short_description(error, _("Mailbox not found"));
    etpan_error_strf_long_description(error,
        _("Mailbox %s was not found while changing its name."),
        folder_name);
    
    new_error = rewrite_error(storage, ERROR_DOMAIN_RENAME, error);
    ETPAN_ERROR_FREE(error);
    
    error = new_error;
    
    goto free_mb_list;
  }
  
  imap_folder = carray_get(imap_mb_list, 0);
  imap_separator = imap_folder->delimiter;
  
  etpan_imap_list_mailboxes_free(imap_mb_list);
  
  snprintf(pattern, sizeof(pattern), "%s%c*", folder_name, imap_separator);
  error = etpan_imap_lsub_mailboxes(ep_storage->sto_session,
      pattern, &imap_mb_list);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(storage, ERROR_DOMAIN_RENAME, error);
    ETPAN_ERROR_FREE(error);
    
    error = new_error;
    
    goto err;
  }
  
  ETPAN_LOG("rename %s %s", folder_name, new_name);
  
  error = etpan_imap_unsubscribe(ep_storage->sto_session, folder_name);
  ETPAN_ERROR_IGNORE(error);
  
  error = etpan_imap_rename(ep_storage->sto_session, folder_name, new_name);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(storage, ERROR_DOMAIN_RENAME, error);
    ETPAN_ERROR_FREE(error);
    
    error = new_error;
    
    goto free_mb_list;
  }

  error = etpan_imap_subscribe(ep_storage->sto_session, new_name);
  ETPAN_ERROR_IGNORE(error);
  
  len = strlen(folder_name);
  for(i = 0 ; i < carray_count(imap_mb_list) ; i ++) {
    struct etpan_imap_mailbox_info * subfolder;
    char * postfix;
    char * new_sublocation;
    size_t len_new_sublocation;
    
    subfolder = carray_get(imap_mb_list, i);
    postfix = subfolder->mb + len;
    len_new_sublocation = strlen(postfix) + strlen(new_name) + 1;
    new_sublocation = malloc(len_new_sublocation);
    if (new_sublocation == NULL) {
      ETPAN_LOG_MEMORY_ERROR;
    }
    
    snprintf(new_sublocation, len_new_sublocation, "%s%s", new_name, postfix);
    error = etpan_imap_unsubscribe(ep_storage->sto_session, subfolder->mb);
    ETPAN_ERROR_IGNORE(error);
    error = etpan_imap_subscribe(ep_storage->sto_session, new_sublocation);
    ETPAN_ERROR_IGNORE(error);
  }
  
  etpan_imap_list_mailboxes_free(imap_mb_list);
  
  error = NULL;
  
 free_mb_list:
  etpan_imap_list_mailboxes_free(imap_mb_list);
  
 err:
  other_error = etpan_imap_sync_force_fetch_folder_list(data->imap_sync);
  if (other_error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(storage, ERROR_DOMAIN_RENAME, other_error);
    ETPAN_ERROR_FREE(other_error);
    
    if (error != NULL)
      ETPAN_ERROR_FREE(other_error);
    else
      error = new_error;
  }
  
  return error;
}

static struct etpan_error * fetch_folder_list(struct etpan_storage * storage,
                                              chash * folder_list)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  etpan_imap_sync_fetch_folder_list(storage,
      data->imap_sync, folder_list);
  
  return NULL;
}

static struct etpan_error * rewrite_error(struct etpan_storage * storage,
    int domain,
    struct etpan_error * error)
{
  struct etpan_error * new_error;
  char * previous_long_description;
  
  new_error = etpan_error_new();
  etpan_error_set_code(new_error, etpan_error_get_code(error));
  etpan_error_set_short_description(new_error,
      etpan_error_get_short_description(error));
  previous_long_description = etpan_error_get_long_description(error);
  
  switch (domain) {
  case ERROR_DOMAIN_CONNECTION:
    etpan_error_strf_long_description(new_error,
        _("Connection to account %s failed. "
            "%s"),
        etpan_storage_get_id(storage),
        previous_long_description);
    break;
    
  case ERROR_DOMAIN_FETCH_FOLDER_LIST:
    etpan_error_strf_long_description(new_error,
        _("An error occurred while getting list of mailboxes on account %s.\n"
            "%s"),
        etpan_storage_get_id(storage),
        previous_long_description);
    break;

  case ERROR_DOMAIN_CREATE:
    etpan_error_strf_long_description(new_error,
        _("An error occurred while creating a mailbox on account %s.\n"
            "%s"),
        etpan_storage_get_id(storage),
        previous_long_description);
    break;
    
  case ERROR_DOMAIN_DELETE:
    etpan_error_strf_long_description(new_error,
        _("An error occurred while deleting a mailbox on account %s.\n"
            "%s"),
        etpan_storage_get_id(storage),
        previous_long_description);
    break;
    
  case ERROR_DOMAIN_RENAME:
    etpan_error_strf_long_description(new_error,
        _("An error occurred while changing the name of a mailbox on account %s.\n"
            "%s"),
        etpan_storage_get_id(storage),
        previous_long_description);
    break;
    
  default:
    etpan_error_strf_long_description(new_error,
        _("An error occurred with account %s.\n"
            "%s"),
        etpan_storage_get_id(storage),
        previous_long_description);
    break;
  }
  
  return new_error;
}

struct etpan_imap_sync *
etpan_storage_imap_sync_get_imap_sync(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  return data->imap_sync;
}
