#include "etpan-pop-sync.h"

#include <string.h>
#include <stdlib.h>
#include <libetpan/libetpan.h>

#define ETPAN_MODULE_LOG_NAME "POPSYNC"

#include "etpan-log.h"
#include "etpan-error.h"
#include "etpan-connection-types.h"
#include "etpan-nls.h"
#include "etpan-thread-manager-app.h"
#include "etpan-thread-manager.h"
#include "etpan-account.h"
#include "etpan-time.h"
#include "etpan-utils.h"
#include "etpan-account-manager.h"
#include "etpan-account.h"
#include "etpan-sqldb.h"
#include "etpan-serialize.h"
#include "etpan-storage.h"
#include "etpan-folder.h"
#include "etpan-message.h"
#include "etpan-signal.h"

#define DEFAULT_DAYS 15
#define DEFAULT_SYNC_DELAY (10 * 60)

enum {
  STATE_DISCONNECTED,
  STATE_CONNECTED,
  STATE_LIST_FETCHED,
  STATE_CONTENT_FETCHED,
  STATE_DELETED,
  STATE_CANCELLED,
};

struct etpan_pop_sync * etpan_pop_sync_new(void)
{
  struct etpan_pop_sync * pop_sync;
  
  pop_sync = malloc(sizeof(* pop_sync));
  if (pop_sync == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  pop_sync->id = NULL;
  
  pop_sync->hostname = NULL;
  pop_sync->port = 0;
  pop_sync->connection_type = ETPAN_CONNECTION_PLAIN;
  pop_sync->auth_type = ETPAN_AUTH_PASSWORD;
  pop_sync->username = NULL;
  pop_sync->password = NULL;
  pop_sync->command = NULL;
  pop_sync->days = DEFAULT_DAYS;
  
  /* threaded */
  pop_sync->ep_storage = NULL;
  pop_sync->threaded_hostname = NULL;
  pop_sync->threaded_port = 0;
  pop_sync->threaded_connection_type = ETPAN_CONNECTION_PLAIN;
  pop_sync->threaded_auth_type = ETPAN_AUTH_PASSWORD;
  pop_sync->threaded_username = NULL;
  pop_sync->threaded_password = NULL;
  pop_sync->threaded_command = NULL;
  pop_sync->threaded_days = DEFAULT_DAYS;
  
  pop_sync->cancelled = 0;
  pop_sync->state = STATE_DISCONNECTED;
  pop_sync->error = NULL;
  pop_sync->current_msg = 0;
  pop_sync->callback = NULL;
  pop_sync->cb_data = NULL;
  pop_sync->syncing = 0;
  
  pop_sync->connected = 0;
  pop_sync->last_sync_date = 0;
  pop_sync->uid_db = 0;
  
  pop_sync->dummy_storage = etpan_storage_new();

  return pop_sync;
}

void etpan_pop_sync_free(struct etpan_pop_sync * pop_sync)
{
  etpan_storage_free(pop_sync->dummy_storage);
  free(pop_sync->threaded_command);
  free(pop_sync->threaded_password);
  free(pop_sync->threaded_username);
  free(pop_sync->threaded_hostname);
  free(pop_sync->command);
  free(pop_sync->password);
  free(pop_sync->username);
  free(pop_sync->hostname);
  free(pop_sync->id);
  
  free(pop_sync);
}

void etpan_pop_sync_set_account(struct etpan_pop_sync * pop_sync,
    struct etpan_account * account)
{
  pop_sync->account = account;
}

struct etpan_account *
etpan_pop_sync_get_account(struct etpan_pop_sync * pop_sync)
{
  return pop_sync->account;
}

void etpan_pop_sync_set_id(struct etpan_pop_sync * pop_sync,
    char * id)
{
  if (id != pop_sync->id) {
    free(pop_sync->id);
    if (id != NULL) {
      pop_sync->id = strdup(id);
      if (pop_sync->id == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      pop_sync->id = NULL;
  }
}

char * etpan_pop_sync_get_id(struct etpan_pop_sync * pop_sync)
{
  return pop_sync->id;
}

void etpan_pop_sync_set_hostname(struct etpan_pop_sync * pop_sync,
    char * hostname)
{
  if (hostname != pop_sync->hostname) {
    free(pop_sync->hostname);
    if (hostname != NULL) {
      pop_sync->hostname = strdup(hostname);
      if (pop_sync->hostname == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      pop_sync->hostname = NULL;
  }
}

char * etpan_pop_sync_get_hostname(struct etpan_pop_sync * pop_sync)
{
  return pop_sync->hostname;
}

void etpan_pop_sync_set_port(struct etpan_pop_sync * pop_sync, int port)
{
  pop_sync->port = port;
}

int etpan_pop_sync_get_port(struct etpan_pop_sync * pop_sync)
{
  return pop_sync->port;
}

void etpan_pop_sync_set_connection_type(struct etpan_pop_sync * pop_sync,
    int connection_type)
{
  pop_sync->connection_type = connection_type;
}

int etpan_pop_sync_get_connection_type(struct etpan_pop_sync * pop_sync)
{
  return pop_sync->connection_type;
}

void etpan_pop_sync_set_auth_type(struct etpan_pop_sync * pop_sync,
    int auth_type)
{
  pop_sync->auth_type = auth_type;
}

int etpan_pop_sync_get_auth_type(struct etpan_pop_sync * pop_sync)
{
  return pop_sync->auth_type;
}

void etpan_pop_sync_set_username(struct etpan_pop_sync * pop_sync,
    char * username)
{
  if (username != pop_sync->username) {
    free(pop_sync->username);
    if (username != NULL) {
      pop_sync->username = strdup(username);
      if (pop_sync->username == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      pop_sync->username = NULL;
  }
}

char * etpan_pop_sync_get_username(struct etpan_pop_sync * pop_sync)
{
  return pop_sync->username;
}

void etpan_pop_sync_set_password(struct etpan_pop_sync * pop_sync,
    char * password)
{
  if (password != pop_sync->password) {
    free(pop_sync->password);
    if (password != NULL) {
      pop_sync->password = strdup(password);
      if (pop_sync->password == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      pop_sync->password = NULL;
  }
}

char * etpan_pop_sync_get_password(struct etpan_pop_sync * pop_sync)
{
  return pop_sync->password;
}

void etpan_pop_sync_set_command(struct etpan_pop_sync * pop_sync,
    char * command)
{
  if (command != pop_sync->command) {
    free(pop_sync->command);
    if (command != NULL) {
      pop_sync->command = strdup(command);
      if (pop_sync->command == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      pop_sync->command = NULL;
  }
}

char * etpan_pop_sync_get_command(struct etpan_pop_sync * pop_sync)
{
  return pop_sync->command;
}

void etpan_pop_sync_set_days(struct etpan_pop_sync * pop_sync,
    unsigned int days)
{
  pop_sync->days = days;
}

int etpan_pop_sync_get_days(struct etpan_pop_sync * pop_sync)
{
  return pop_sync->days;
}

void etpan_pop_sync_setup(struct etpan_pop_sync * pop_sync)
{
  /* running in a thread */
  free(pop_sync->threaded_command);
  pop_sync->threaded_command = NULL;
  free(pop_sync->threaded_password);
  pop_sync->threaded_password = NULL;
  free(pop_sync->threaded_username);
  pop_sync->threaded_username = NULL;
  free(pop_sync->threaded_hostname);
  pop_sync->threaded_hostname = NULL;
  
  if (pop_sync->hostname != NULL) {
    pop_sync->threaded_hostname = strdup(pop_sync->hostname);
    if (pop_sync->threaded_hostname == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  pop_sync->threaded_port = pop_sync->port;
  pop_sync->threaded_connection_type = pop_sync->connection_type;
  pop_sync->threaded_auth_type = pop_sync->auth_type;
  if (pop_sync->username != NULL) {
    pop_sync->threaded_username = strdup(pop_sync->username);
    if (pop_sync->threaded_username == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  if (pop_sync->password != NULL) {
    pop_sync->threaded_password = strdup(pop_sync->password);
    if (pop_sync->threaded_password == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  if (pop_sync->command != NULL) {
    pop_sync->threaded_command = strdup(pop_sync->command);
    if (pop_sync->threaded_command == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  pop_sync->threaded_days = pop_sync->days;
}

void etpan_pop_sync_unsetup(struct etpan_pop_sync * pop_sync)
{
  free(pop_sync->threaded_command);
  pop_sync->threaded_command = NULL;
  free(pop_sync->threaded_password);
  pop_sync->threaded_password = NULL;
  free(pop_sync->threaded_username);
  pop_sync->threaded_username = NULL;
  free(pop_sync->threaded_hostname);
  pop_sync->threaded_hostname = NULL;
}

static void next_op(struct etpan_pop_sync * pop_sync);
static void notify_end(struct etpan_pop_sync * pop_sync);

void etpan_pop_sync_start(struct etpan_pop_sync * pop_sync,
    void (* callback)(void *), void * user_data)
{
  ETPAN_ASSERT(!pop_sync->syncing, "should not be used twice");
  
  ETPAN_LOCAL_LOG("pop sync start");
  pop_sync->state = STATE_DISCONNECTED;
  pop_sync->callback = callback;
  pop_sync->cb_data = user_data;
  pop_sync->syncing = 1;
  pop_sync->current_msg = 0;
  
  next_op(pop_sync);
}

static void connect_run_op(struct etpan_thread_manager_app * manager,
    struct etpan_pop_sync * pop_sync);
static void list_run_op(struct etpan_thread_manager_app * manager,
    struct etpan_pop_sync * pop_sync);
static void get_next_content_run_op(struct etpan_thread_manager_app * manager,
    struct etpan_pop_sync * pop_sync);
static void delete_run_op(struct etpan_thread_manager_app * manager,
    struct etpan_pop_sync * pop_sync);
static void disconnect_run_op(struct etpan_thread_manager_app * manager,
    struct etpan_pop_sync * pop_sync);

static struct etpan_error * uid_db_open(struct etpan_pop_sync * pop_sync);
static void uid_db_close(struct etpan_pop_sync * pop_sync);
static int uid_db_get_mark(struct etpan_pop_sync * pop_sync, char * uid,
    time_t date);
static struct etpan_error *
uid_db_set_mark(struct etpan_pop_sync * pop_sync, char * uid, time_t date);

static void next_op(struct etpan_pop_sync * pop_sync)
{
  ETPAN_LOCAL_LOG("pop next op");
  switch (pop_sync->state) {
  case STATE_DISCONNECTED:
    ETPAN_LOCAL_LOG("connect pop sync");
    connect_run_op(etpan_thread_manager_app_get_default(), pop_sync);
    break;
  case STATE_CONNECTED:
    list_run_op(etpan_thread_manager_app_get_default(), pop_sync);
    break;
  case STATE_LIST_FETCHED:
    get_next_content_run_op(etpan_thread_manager_app_get_default(), pop_sync);
    break;
  case STATE_CONTENT_FETCHED:
    delete_run_op(etpan_thread_manager_app_get_default(), pop_sync);
    break;
  case STATE_DELETED:
    ETPAN_LOCAL_LOG("disconnect pop sync");
    disconnect_run_op(etpan_thread_manager_app_get_default(), pop_sync);
    break;
  case STATE_CANCELLED:
    ETPAN_LOCAL_LOG("disconnect pop sync");
    disconnect_run_op(etpan_thread_manager_app_get_default(), pop_sync);
    break;
  }
}

static void notify_cancel(struct etpan_pop_sync * pop_sync)
{
  ETPAN_LOCAL_LOG("notify_cancel");
  
  if (pop_sync->state != STATE_DISCONNECTED) {
    pop_sync->state = STATE_CANCELLED;
    next_op(pop_sync);
    return;
  }
  
  if (pop_sync->error == NULL) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CANCELLED);
    etpan_error_set_short_description(error,
        _("Synchronization of account cancelled"));
    etpan_error_strf_long_description(error,
        _("Synchronization of POP account %s cancelled."),
        pop_sync->id);
  }
  
  notify_end(pop_sync);
}

static void notify_end(struct etpan_pop_sync * pop_sync)
{
  pop_sync->last_sync_date = etpan_get_time();
  pop_sync->syncing = 0;
  
  if (pop_sync->error != NULL)
    etpan_error_log(pop_sync->error);
  
  ETPAN_LOCAL_LOG("end of pop sync");
  ETPAN_SIGNAL_SEND(pop_sync, ETPAN_POP_SYNC_OP_ENDED_SIGNAL);
  pop_sync->callback(pop_sync->cb_data);
  
  pop_sync->callback = NULL;
  pop_sync->cb_data = NULL;
  ETPAN_ERROR_FREE(pop_sync->error);
  pop_sync->error = NULL;
}

void etpan_pop_sync_cancel(struct etpan_pop_sync * pop_sync)
{
  pop_sync->cancelled = 1;
}

struct etpan_error * etpan_pop_sync_get_error(struct etpan_pop_sync * pop_sync)
{
  return pop_sync->error;
}

/* connect */

static struct etpan_error * ep_storage_connect(struct etpan_pop_sync * pop_sync)
{
  int r;
  struct etpan_error * error;

  if (pop_sync->connected)
    return NULL;
  
  r = mailstorage_connect(pop_sync->ep_storage);
  if (r == MAIL_ERROR_MEMORY)
    ETPAN_LOG_MEMORY_ERROR;
    
  if (r == MAIL_ERROR_CONNECT) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CONNECT);
    etpan_error_set_short_description(error, "Could not connect");
    etpan_error_strf_long_description(error,
        _("Connection to account %s failed. "
            "The connection to POP server %s, port %i could not be established.\n"
            "Check the name of server and port."),
        etpan_pop_sync_get_id(pop_sync),
        etpan_pop_sync_get_hostname(pop_sync),
        etpan_pop_sync_get_port(pop_sync));
    goto exit;
  }
  else if (r == MAIL_ERROR_LOGIN) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_AUTH);
    etpan_error_set_short_description(error, "Authentication failed");
    etpan_error_strf_long_description(error,
        _("Connection to account %s failed. "
            "Check your login, password and authentication type."),
        etpan_pop_sync_get_id(pop_sync));
    goto exit;
  }
  else if (r == MAIL_ERROR_PARSE) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, "Parse error");
    etpan_error_strf_long_description(error, _("Connection to account %s failed. The application did not understand what the server sent."),
        etpan_pop_sync_get_id(pop_sync));
    goto exit;
  }
  else if (r == MAIL_ERROR_STREAM) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_STREAM);
    etpan_error_set_short_description(error, "Connection closed");
    etpan_error_strf_long_description(error,
        _("Connection to account %s failed. "
            "The connection closed unexpectedly."),
        etpan_pop_sync_get_id(pop_sync));
    goto exit;
  }
  else if (r != MAIL_NO_ERROR) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CONNECT);
    etpan_error_set_short_description(error, _("Unexpected error"));
    etpan_error_strf_long_description(error,
        _("Connection to account %s failed. "
            "An unexpected error (libetpan: %i) occurred."),
        etpan_pop_sync_get_id(pop_sync), r);
    goto exit;
  }
  
  error = NULL;

  pop_sync->connected = 1;
  
 exit:
  return error;
}

static struct etpan_error * connect(struct etpan_pop_sync * pop_sync)
{
  int r;
  struct etpan_error * error;
  struct mailstorage * ep_storage;
  int ep_connection_type;
  int ep_auth_type;
  
  ep_storage = mailstorage_new(NULL);
  if (ep_storage == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  if (pop_sync->threaded_command == NULL) {
    switch (pop_sync->threaded_connection_type) {
    case ETPAN_CONNECTION_STARTTLS:
      ep_connection_type = CONNECTION_TYPE_STARTTLS;
      break;
    case ETPAN_CONNECTION_TRY_STARTTLS:
      ep_connection_type = CONNECTION_TYPE_TRY_STARTTLS;
      break;
    case ETPAN_CONNECTION_TLS:
      ep_connection_type = CONNECTION_TYPE_TLS;
      break;
    default:
      ep_connection_type = CONNECTION_TYPE_PLAIN;
      break;
    }
  }
  else {
    switch (pop_sync->threaded_connection_type) {
    case ETPAN_CONNECTION_STARTTLS:
      ep_connection_type = CONNECTION_TYPE_COMMAND_STARTTLS;
      break;
    case ETPAN_CONNECTION_TRY_STARTTLS:
      ep_connection_type = CONNECTION_TYPE_COMMAND_TRY_STARTTLS;
      break;
    case ETPAN_CONNECTION_TLS:
      ep_connection_type = CONNECTION_TYPE_COMMAND_TLS;
      break;
    default:
      ep_connection_type = CONNECTION_TYPE_COMMAND;
      break;
    }
  }
    
  ep_auth_type = POP3_AUTH_TYPE_PLAIN;
  
  r = pop3_mailstorage_init(ep_storage,
      pop_sync->threaded_hostname, pop_sync->threaded_port,
      pop_sync->threaded_command,
      ep_connection_type, ep_auth_type,
      pop_sync->threaded_username, pop_sync->threaded_password,
      0, NULL, NULL);
  if (r != MAIL_NO_ERROR) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  pop_sync->ep_storage = ep_storage;
  
  error = ep_storage_connect(pop_sync);
  if (error != NULL)
    goto free_storage;
  
  return NULL;
  
 free_storage:
  mailstorage_free(pop_sync->ep_storage);
  return error;
}

struct connect_result {
  struct etpan_error * error;
};

static void threaded_connect(struct etpan_thread_op * op);
static void threaded_connect_cleanup(struct etpan_thread_op * op);
static void threaded_connect_callback(int cancelled,
    struct connect_result * result,
    void * data);

static void connect_run_op(struct etpan_thread_manager_app * manager,
    struct etpan_pop_sync * pop_sync)
{
  struct etpan_thread_op * op;
  struct etpan_storage * storage;
  struct connect_result * result;
  
  op = etpan_thread_op_new();
  op->param = pop_sync;
  result = malloc(sizeof(* result));
  if (result == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  result->error = NULL;
  op->result = result;
  
  op->cancellable = 1;
  op->run = threaded_connect;
  op->callback = (void (*)(int, void *, void *)) threaded_connect_callback;
  op->callback_data = pop_sync;
  op->cleanup = threaded_connect_cleanup;
  
  storage = pop_sync->dummy_storage;
  etpan_thread_manager_app_storage_schedule(manager, storage, op);
}

static void disconnect(struct etpan_pop_sync * pop_sync);

static void threaded_connect(struct etpan_thread_op * op)
{
  struct etpan_error * error;
  struct connect_result * result;
  struct etpan_pop_sync * pop_sync;
  
  result = op->result;
  pop_sync = op->param;
  
  error = connect(pop_sync);
  if (error != NULL)
    goto exit;
  
  error = uid_db_open(pop_sync);
  if (error != NULL)
    goto disconnect;
  
  error = NULL;  
  goto exit;
  
 disconnect:
  disconnect(pop_sync);
 exit:
  result->error = error;
}

static void threaded_connect_callback(int cancelled,
    struct connect_result * result,
    void * data)
{
  struct etpan_pop_sync * pop_sync;
  
  ETPAN_LOCAL_LOG("connect callback pop sync");
  pop_sync = data;
  
  if (result->error != NULL) {
    etpan_storage_unsetup(pop_sync->dummy_storage);
    pop_sync->error = result->error;
    result->error = NULL;
    if (pop_sync->cancelled) {
      notify_cancel(pop_sync);
    }
    else {
      notify_end(pop_sync);
    }
  }
  else {
    pop_sync->state = STATE_CONNECTED;
    if (pop_sync->cancelled) {
      notify_cancel(pop_sync);
    }
    else {
      next_op(pop_sync);
    }
  }
}

static void threaded_connect_cleanup(struct etpan_thread_op * op)
{
  struct connect_result * result;
  
  result = op->result;
  ETPAN_ERROR_FREE(result->error);
  free(result);
}

/* list run */

static inline struct pop3_session_state_data *
get_data(mailsession * session)
{
  return session->sess_data;
}

static mailpop3 * get_pop3_session(mailsession * session)
{
  return get_data(session)->pop3_session;
}

static mailpop3 * get_pop(struct etpan_pop_sync * pop_sync)
{
  return get_pop3_session(pop_sync->ep_storage->sto_session);
}

struct etpan_error * list(struct etpan_pop_sync * pop_sync, carray ** result)
{
  carray * msg_list;
  mailpop3 * pop;
  
  pop = get_pop(pop_sync);
  mailpop3_list(pop, &msg_list);
  
  * result = msg_list;
  
  return NULL;
}

struct list_result {
  struct etpan_error * error;
  carray * msg_list;
};

static void threaded_list(struct etpan_thread_op * op);
static void threaded_list_callback(int cancelled,
    struct list_result * result,
    void * data);
static void threaded_list_cleanup(struct etpan_thread_op * op);

static void list_run_op(struct etpan_thread_manager_app * manager,
    struct etpan_pop_sync * pop_sync)
{
  struct etpan_thread_op * op;
  struct etpan_storage * storage;
  struct list_result * result;
  
  op = etpan_thread_op_new();
  op->param = pop_sync;
  result = malloc(sizeof(* result));
  if (result == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  result->error = NULL;
  result->msg_list = NULL;
  op->result = result;
  
  op->cancellable = 1;
  op->run = threaded_list;
  op->callback = (void (*)(int, void *, void *)) threaded_list_callback;
  op->callback_data = pop_sync;
  op->cleanup = threaded_list_cleanup;
  
  storage = pop_sync->dummy_storage;
  etpan_thread_manager_app_storage_schedule(manager, storage, op);
}

static void threaded_list(struct etpan_thread_op * op)
{
  struct etpan_error * error;
  struct list_result * result;
  struct etpan_pop_sync * pop_sync;
  carray * msg_list;
  
  result = op->result;
  pop_sync = op->param;
  
  error = list(pop_sync, &msg_list);
  result->error = error;
  result->msg_list = msg_list;
}

static void threaded_list_callback(int cancelled,
    struct list_result * result,
    void * data)
{
  struct etpan_pop_sync * pop_sync;
  
  pop_sync = data;
  
  if (result->error != NULL) {
    pop_sync->error = result->error;
    result->error = NULL;
    if (pop_sync->cancelled) {
      notify_cancel(pop_sync);
    }
    else {
      notify_end(pop_sync);
    }
  }
  else {
    ETPAN_LOG("pop-sync: %u msg", carray_count(result->msg_list));
    pop_sync->msg_list = result->msg_list;
    pop_sync->state = STATE_LIST_FETCHED;
    if (pop_sync->cancelled) {
      notify_cancel(pop_sync);
    }
    else {
      next_op(pop_sync);
    }
  }
}

static void threaded_list_cleanup(struct etpan_thread_op * op)
{
  struct connect_result * result;
  
  result = op->result;
  ETPAN_ERROR_FREE(result->error);
  free(result);
}

/* get next content */

struct get_next_content_result {
  struct etpan_error * error;
};

static void threaded_get_next_content(struct etpan_thread_op * op);
static void threaded_get_next_content_callback(int cancelled,
    struct get_next_content_result * result,
    void * data);
static void threaded_get_next_content_cleanup(struct etpan_thread_op * op);
static void get_next_content_finished(struct etpan_pop_sync * pop_sync,
    struct etpan_error * error);

static void get_next_content_run_op(struct etpan_thread_manager_app * manager,
    struct etpan_pop_sync * pop_sync)
{
  struct etpan_thread_op * op;
  struct etpan_storage * storage;
  struct get_next_content_result * result;
  
  op = etpan_thread_op_new();
  op->param = pop_sync;
  result = malloc(sizeof(* result));
  if (result == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  result->error = NULL;
  op->result = result;
  
  op->cancellable = 1;
  op->run = threaded_get_next_content;
  op->callback = (void (*)(int, void *, void *)) threaded_get_next_content_callback;
  op->callback_data = pop_sync;
  op->cleanup = threaded_get_next_content_cleanup;
  
  storage = pop_sync->dummy_storage;
  etpan_thread_manager_app_storage_schedule(manager, storage, op);
}

static void threaded_get_next_content(struct etpan_thread_op * op)
{
  struct etpan_error * error;
  struct get_next_content_result * result;
  struct etpan_pop_sync * pop_sync;
  int r;
  mailpop3 * pop;
  struct mailpop3_msg_info * info;
  char * content;
  size_t content_size;
  
  result = op->result;
  pop_sync = op->param;
  pop = get_pop(pop_sync);
  
  pop_sync->fetch_result_content = NULL;
  pop_sync->fetch_result_content_size = 0;
  
  if (pop_sync->current_msg >= carray_count(pop_sync->msg_list)) {
    return;
  }
  
  ETPAN_LOCAL_LOG("pop-sync: %u/%u msg", pop_sync->current_msg,
      carray_count(pop_sync->msg_list));
  info = carray_get(pop_sync->msg_list, pop_sync->current_msg);
  
  if (!uid_db_get_mark(pop_sync,info->msg_uidl, 0)) {
    r = mailpop3_retr(pop, info->msg_index, &content, &content_size);
    if (r == MAILPOP3_ERROR_MEMORY) {
      ETPAN_LOG_MEMORY_ERROR;
    }
    else if (r == MAILPOP3_ERROR_STREAM) {
      error = etpan_error_new();
      etpan_error_set_code(error, ERROR_STREAM);
      etpan_error_set_short_description(error, "Connection closed");
      etpan_error_strf_long_description(error,
          _("Connection to account %s failed. "
              "The connection closed unexpectedly."),
          etpan_pop_sync_get_id(pop_sync));
      goto exit;
    }
    else if (r != MAILPOP3_NO_ERROR) {
      /* ignore other errors */
    }
    else {
      pop_sync->fetch_result_content = content;
      pop_sync->fetch_result_content_size = content_size;
    }
  }
  
  error = NULL;
  
 exit:
  pop_sync->error = error;
}

static void appended_callback(int cancelled,
    struct etpan_folder_append_msg_result * result,
    void * cb_data);

static void threaded_get_next_content_callback(int cancelled,
    struct get_next_content_result * result,
    void * data)
{
  struct etpan_pop_sync * pop_sync;
  struct etpan_account * account;
  struct etpan_storage * storage;
  struct etpan_folder * folder;
  struct etpan_message_flags * flags;
  
  pop_sync = data;
  
  if (result->error != NULL) {
    get_next_content_finished(pop_sync, result->error);
    return;
  }
  
  if (pop_sync->fetch_result_content != NULL) {
    /* deliver, filter */
    /* if not delivered, do the following */
    
    account = pop_sync->account;
    storage = etpan_account_get_storage(account);
    folder = etpan_storage_get_folder(storage, "");
    ETPAN_ASSERT(folder != NULL, "INBOX should exist");
    
    flags = etpan_message_flags_new();
    etpan_folder_append(etpan_thread_manager_app_get_default(),
        folder, pop_sync->fetch_result_content,
        pop_sync->fetch_result_content_size,
        flags, appended_callback, pop_sync);
    etpan_message_flags_free(flags);
    
    mailpop3_retr_free(pop_sync->fetch_result_content);
    pop_sync->fetch_result_content = NULL;
    pop_sync->fetch_result_content_size = 0;
  }
  else {
    get_next_content_finished(pop_sync, NULL);
  }
}

static void appended_callback(int cancelled,
    struct etpan_folder_append_msg_result * result,
    void * cb_data)
{
  struct etpan_pop_sync * pop_sync;
  
  pop_sync = cb_data;
  
  if (result->error != NULL) {
    get_next_content_finished(pop_sync, result->error);
    return;
  }
  else {
    struct etpan_error * error;
    struct mailpop3_msg_info * info;
    
    info = carray_get(pop_sync->msg_list, pop_sync->current_msg);
    error = uid_db_set_mark(pop_sync, info->msg_uidl, time(NULL));
    if (error != NULL) {
      get_next_content_finished(pop_sync, error);
      ETPAN_ERROR_FREE(error);
      return;
    }
  }
  
  get_next_content_finished(pop_sync, NULL);
}

static void get_next_content_finished(struct etpan_pop_sync * pop_sync,
    struct etpan_error * error)
{  
  if (error != NULL) {
    pop_sync->error = etpan_error_dup(error);
    if (pop_sync->cancelled) {
      notify_cancel(pop_sync);
    }
    else {
      notify_end(pop_sync);
    }
  }
  else {
    pop_sync->current_msg ++;
    
    if (pop_sync->current_msg >= carray_count(pop_sync->msg_list)) {
      pop_sync->state = STATE_CONTENT_FETCHED;
    }
    
    if (pop_sync->cancelled) {
      notify_cancel(pop_sync);
    }
    else {
      next_op(pop_sync);
    }
  }
}

static void threaded_get_next_content_cleanup(struct etpan_thread_op * op)
{
  struct get_next_content_result * result;
  
  result = op->result;
  ETPAN_ERROR_FREE(result->error);
  free(result);
}


/* delete */

struct delete_result {
  struct etpan_error * error;
};

static void threaded_delete(struct etpan_thread_op * op);
static void threaded_delete_callback(int cancelled,
    struct delete_result * result,
    void * data);
static void threaded_delete_cleanup(struct etpan_thread_op * op);

static void delete_run_op(struct etpan_thread_manager_app * manager,
    struct etpan_pop_sync * pop_sync)
{
  struct etpan_thread_op * op;
  struct etpan_storage * storage;
  struct delete_result * result;
  
  op = etpan_thread_op_new();
  op->param = pop_sync;
  result = malloc(sizeof(* result));
  if (result == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  result->error = NULL;
  op->result = result;
  
  op->cancellable = 1;
  op->run = threaded_delete;
  op->callback = (void (*)(int, void *, void *)) threaded_delete_callback;
  op->callback_data = pop_sync;
  op->cleanup = threaded_delete_cleanup;
  
  storage = pop_sync->dummy_storage;
  etpan_thread_manager_app_storage_schedule(manager, storage, op);
}

static void threaded_delete(struct etpan_thread_op * op)
{
  struct etpan_error * error;
  struct delete_result * result;
  struct etpan_pop_sync * pop_sync;
  carray * list;
  mailpop3 * pop;
  unsigned int i;
  time_t date;
  int r;
  
  result = op->result;
  pop_sync = op->param;
  pop = get_pop(pop_sync);
  
  date = time(NULL) - pop_sync->threaded_days * 24 * 60 * 60;
  mailpop3_list(pop, &list);
  for(i = 0 ; i < carray_count(list) ; i ++) {
    struct mailpop3_msg_info * info;
    
    info = carray_get(list, i);
    if (uid_db_get_mark(pop_sync,info->msg_uidl, 0)) {
      if (!uid_db_get_mark(pop_sync,info->msg_uidl, date)) {
        ETPAN_LOCAL_LOG("delete message %s", info->msg_uidl);
        r = mailpop3_dele(pop, info->msg_index);
        if (r == MAILPOP3_ERROR_STREAM) {
          error = etpan_error_new();
          etpan_error_set_code(error, ERROR_STREAM);
          etpan_error_set_short_description(error, "Connection closed");
          etpan_error_strf_long_description(error,
              _("Connection to account %s failed. "
                  "The connection closed unexpectedly."),
              etpan_pop_sync_get_id(pop_sync));
          goto exit;
        }
        else if (r != MAILPOP3_NO_ERROR) {
          /* ignore other errors */
        }
        else {
          /* success */
        }
      }
    }
  }
  
  error = NULL;
  
 exit:
  result->error = error;
}

static void threaded_delete_callback(int cancelled,
    struct delete_result * result,
    void * data)
{
  struct etpan_pop_sync * pop_sync;
  
  pop_sync = data;
  
  if (result->error != NULL) {
    pop_sync->error = result->error;
    result->error = NULL;
    if (pop_sync->cancelled) {
      notify_cancel(pop_sync);
    }
    else {
      notify_end(pop_sync);
    }
  }
  else {
    pop_sync->state = STATE_DELETED;
    if (pop_sync->cancelled) { 
      notify_cancel(pop_sync);
    }
    else {
      next_op(pop_sync);
    }
  }
}

static void threaded_delete_cleanup(struct etpan_thread_op * op)
{
  struct delete_result * result;
  
  result = op->result;
  ETPAN_ERROR_FREE(result->error);
  free(result);
}


/* disconnect */

static void disconnect(struct etpan_pop_sync * pop_sync)
{
  struct mailstorage * ep_storage;
  
  if (!pop_sync->connected)
    return;
  
  ep_storage = pop_sync->ep_storage;
  mailstorage_disconnect(ep_storage);
  mailstorage_free(ep_storage);
  pop_sync->ep_storage = NULL;
  pop_sync->connected = 0;
}

static void threaded_disconnect(struct etpan_thread_op * op);
static void threaded_disconnect_callback(int cancelled,
    void * dummy, void * data);

static void disconnect_run_op(struct etpan_thread_manager_app * manager,
    struct etpan_pop_sync * pop_sync)
{
  struct etpan_thread_op * op;
  struct etpan_storage * storage;
  
  op = etpan_thread_op_new();
  op->param = pop_sync;
  op->result = NULL;
  
  op->cancellable = 1;
  op->run = threaded_disconnect;
  op->callback = (void (*)(int, void *, void *)) threaded_disconnect_callback;
  op->callback_data = pop_sync;
  op->cleanup = NULL;
  
  storage = pop_sync->dummy_storage;
  etpan_thread_manager_app_storage_schedule(manager, storage, op);
}

static void threaded_disconnect(struct etpan_thread_op * op)
{
  struct etpan_pop_sync * pop_sync;
  
  pop_sync = op->param;
  
  uid_db_close(pop_sync);
  
  disconnect(pop_sync);
}

static void threaded_disconnect_callback(int cancelled,
    void * dummy, void * data)
{
  struct etpan_pop_sync * pop_sync;
  
  pop_sync = data;
  ETPAN_LOCAL_LOG("threaded disconnect callback");
  etpan_storage_unsetup(pop_sync->dummy_storage);
  notify_end(pop_sync);
}

/* uid db */

static struct etpan_error * uid_db_open(struct etpan_pop_sync * pop_sync)
{
  char filename[PATH_MAX];
  char * path;
  carray * columns;
  int r;
  
  path = etpan_get_pop_cache_dir(etpan_account_manager_get_default(),
      pop_sync);
  etpan_make_dir(path);
  
  snprintf(filename, sizeof(filename), "%s/uid.db", path);
  free(path);
  
  columns = carray_new(4);
  if (columns == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  r = carray_add(columns, "date", NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  pop_sync->uid_db = etpan_sqldb_new(filename, columns);
  carray_free(columns);
  
  r = etpan_sqldb_open(pop_sync->uid_db);
  if (r < 0) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_DATABASE);
    etpan_error_set_short_description(error, _("Cache could not be written"));
    etpan_error_strf_long_description(error,
        _("Cache of account %s could not be written properly on disk."),
        pop_sync->id);
  }
  
  return NULL;
}

static void uid_db_close(struct etpan_pop_sync * pop_sync)
{
  etpan_sqldb_close(pop_sync->uid_db);
  etpan_sqldb_free(pop_sync->uid_db);
  pop_sync->uid_db = NULL;
}

static void encode_uint64(uint64_t value, void ** p_data, size_t * p_length)
{
  struct etpan_serialize_data * sdata;
  
  sdata = etpan_serialize_data_new_uint64(value);
  etpan_serialize_encode(sdata, p_data, p_length);
  etpan_serialize_data_free(sdata);
}

static uint64_t decode_uint64(void * data, size_t length)
{
  struct etpan_serialize_data * sdata;
  uint64_t value;
  
  sdata = etpan_serialize_decode(data, length);
  value = etpan_serialize_data_get_uint64(sdata);
  etpan_serialize_data_free(sdata);
  
  return value;
}

static struct etpan_error *
set_db_value(struct etpan_pop_sync * pop_sync, char * uid,
    char * value_name, uint64_t value)
{
  int r;
  int had_error;
  void * data;
  size_t length;
  
  had_error = 0;
  r = etpan_sqldb_begin_transaction(pop_sync->uid_db);
  if (r < 0)
    had_error = 1;
  encode_uint64(value, &data, &length);
  r = etpan_sqldb_set(pop_sync->uid_db,
      uid, value_name, data, length);
  free(data);
  if (r < 0)
    had_error = 1;
  r = etpan_sqldb_end_transaction(pop_sync->uid_db);
  if (r < 0)
    had_error = 1;
  
  if (had_error) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_DATABASE);
    etpan_error_set_short_description(error, _("Cache could not be written"));
    etpan_error_strf_long_description(error,
        _("Cache could not be written properly on disk."));
    
    return error;
  }
  
  return NULL;
}

static uint64_t get_db_value(struct etpan_pop_sync * pop_sync,
    char * uid, char * value_name)
{
  uint64_t value;
  void * data;
  size_t length;
  int r;
  
  value = 0;
  r = etpan_sqldb_get(pop_sync->uid_db, uid, value_name,
      &data, &length);
  if (r == 0) {
    value = decode_uint64(data, length);
  }
  
  return value;
}

static int uid_db_get_mark(struct etpan_pop_sync * pop_sync, char * uid,
    time_t date)
{
  uint64_t value;
  
  value = get_db_value(pop_sync, uid, "date");
  if (value == 0)
    return 0;
  if (value < (uint64_t) date)
    return 0;
  
  return 1;
}

static struct etpan_error *
uid_db_set_mark(struct etpan_pop_sync * pop_sync, char * uid, time_t date)
{
  return set_db_value(pop_sync, uid, "date", (uint64_t) date);
}

void etpan_pop_sync_disconnect_nt(struct etpan_pop_sync * pop_sync)
{
  disconnect(pop_sync);
}

double etpan_pop_sync_next_op_date(struct etpan_pop_sync * pop_sync)
{
  if (pop_sync->syncing)
    return etpan_get_time() + 24 * 60 * 60;
  
  return pop_sync->last_sync_date + DEFAULT_SYNC_DELAY;
}

int etpan_pop_sync_is_syncing(struct etpan_pop_sync * pop_sync)
{
  return pop_sync->syncing;
}
