/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-imap.c,v 1.10 2004/11/12 13:57:02 hoa Exp $
 */

#include "etpan-imap.h"

#include <libetpan/libetpan.h>
#include <stdlib.h>
#include <string.h>
#include "etpan-error.h"
#include "etpan-log.h"
#include "etpan-nls.h"

static struct etpan_imap_mailbox_info * mailbox_info_new(char * mb,
    int flags, char delimiter)
{
  struct etpan_imap_mailbox_info * mb_info;
  
  mb_info = malloc(sizeof(* mb_info));
  if (mb_info == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  mb_info->mb = strdup(mb);
  if (mb_info->mb == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  mb_info->flags = flags;
  mb_info->delimiter = delimiter;
  
  return mb_info;
}

static void mailbox_info_free(struct etpan_imap_mailbox_info * mb_info)
{
  free(mb_info->mb);
  free(mb_info);
}

static mailimap * get_imap(mailsession * session)
{
  mailimap * imap;
  struct imap_session_state_data * data;
  
  if (strcasecmp(session->sess_driver->sess_name, "imap-cached") == 0) {
    struct imap_cached_session_state_data * cached_data;
    
    cached_data = session->sess_data;
    session = cached_data->imap_ancestor;
  }
  
  data = session->sess_data;
  imap = data->imap_session;
  
  return imap;
}  

static int imap_flags_to_flags(struct mailimap_mbx_list_flags * imap_flags)
{
  int flags;
  clistiter * cur;
  
  flags = 0;
  if (imap_flags->mbf_type == MAILIMAP_MBX_LIST_FLAGS_SFLAG) {
    switch (imap_flags->mbf_sflag) {
    case MAILIMAP_MBX_LIST_SFLAG_MARKED:
      flags |= ETPAN_IMAP_MB_MARKED;
      break;
    case MAILIMAP_MBX_LIST_SFLAG_NOSELECT:
      flags |= ETPAN_IMAP_MB_NOSELECT;
      break;
    case MAILIMAP_MBX_LIST_SFLAG_UNMARKED:
      flags |= ETPAN_IMAP_MB_UNMARKED;
      break;
    }
  }
  
  for(cur = clist_begin(imap_flags->mbf_oflags) ; cur != NULL ;
      cur = clist_next(cur)) {
    struct mailimap_mbx_list_oflag * oflag;
    
    oflag = clist_content(cur);
    
    switch (oflag->of_type) {
    case MAILIMAP_MBX_LIST_OFLAG_NOINFERIORS:
      flags |= ETPAN_IMAP_MB_NOINFERIORS;
      break;
    }
  }
  
  return flags;
}

struct etpan_error * etpan_imap_lsub_mailboxes(mailsession * session,
    char * wildcard, carray ** result)
{
  clist * list;
  int r;
  clistiter * cur;
  mailimap * imap;
  carray * info_array;
  struct etpan_error * error;
  
  imap = get_imap(session);
  
  r = mailimap_lsub(imap, "", wildcard, &list);
  if (r == MAILIMAP_ERROR_MEMORY)
    ETPAN_LOG_MEMORY_ERROR;
  
  if (r == MAILIMAP_ERROR_STREAM) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_STREAM);
    etpan_error_set_short_description(error, _("Connection closed"));
    etpan_error_strf_long_description(error, _("Connection was closed while retrieving the list of mailboxes"));
    goto err;
  }
  else if (r == MAILIMAP_ERROR_PARSE) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, _("Parse error"));
    etpan_error_strf_long_description(error, _("The application did not understand what the server sent while it was retrieving the list of mailboxes."));
    goto err;
  }
  else if (r != MAILIMAP_NO_ERROR) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_IMAP_MAILBOX_LIST);
    etpan_error_set_short_description(error, _("Unexpected error"));
    etpan_error_strf_long_description(error, _("An unexpected error (%i) occurred while retrieving the list of mailboxes"), r);
    goto err;
  }

  info_array = carray_new(16);
  if (info_array == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }

  for(cur = clist_begin(list) ; cur != NULL ; cur = cur->next) {
    struct mailimap_mailbox_list * mb_list;
    struct etpan_imap_mailbox_info * mb_info;
    int flags;
    
    mb_list = cur->data;
    
    flags = 0;
    if (mb_list->mb_flag != NULL)
      flags = imap_flags_to_flags(mb_list->mb_flag);
    
    mb_info = mailbox_info_new(mb_list->mb_name, flags, mb_list->mb_delimiter);
    
    r = carray_add(info_array, mb_info, NULL);
    if (r < 0) {
      mailbox_info_free(mb_info);
      ETPAN_LOG_MEMORY_ERROR;
    }
  }
  
  mailimap_list_result_free(list);
  
  * result = info_array;
  
  return NULL;
  
 err:
  return error;
}

struct etpan_error * etpan_imap_list_mailboxes(mailsession * session,
    char * wildcard, carray ** result)
{
  clist * list;
  int r;
  clistiter * cur;
  mailimap * imap;
  carray * info_array;
  struct etpan_error * error;
  
  imap = get_imap(session);
  
  r = mailimap_list(imap, "", wildcard, &list);
  if (r == MAILIMAP_ERROR_MEMORY)
    ETPAN_LOG_MEMORY_ERROR;
  
  if (r == MAILIMAP_ERROR_STREAM) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_STREAM);
    etpan_error_set_short_description(error, _("Connection closed"));
    etpan_error_strf_long_description(error, _("Connection was closed while retrieving the list of mailboxes"));
    goto err;
  }
  else if (r == MAILIMAP_ERROR_PARSE) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, _("Parse error"));
    etpan_error_strf_long_description(error, _("The application did not understand what the server sent while it was retrieving the list of mailboxes."));
    goto err;
  }
  else if (r != MAILIMAP_NO_ERROR) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_IMAP_MAILBOX_LIST);
    etpan_error_set_short_description(error, _("Unexpected error"));
    etpan_error_strf_long_description(error, _("An unexpected error (%i) occurred while retrieving the list of mailboxes"), r);
    goto err;
  }

  info_array = carray_new(16);
  if (info_array == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }

  for(cur = clist_begin(list) ; cur != NULL ; cur = cur->next) {
    struct mailimap_mailbox_list * mb_list;
    struct etpan_imap_mailbox_info * mb_info;
    int flags;
    
    mb_list = cur->data;
    
    flags = 0;
    if (mb_list->mb_flag != NULL)
      flags = imap_flags_to_flags(mb_list->mb_flag);
    
    mb_info = mailbox_info_new(mb_list->mb_name, flags, mb_list->mb_delimiter);
    
    r = carray_add(info_array, mb_info, NULL);
    if (r < 0) {
      mailbox_info_free(mb_info);
      ETPAN_LOG_MEMORY_ERROR;
    }
  }
  
  mailimap_list_result_free(list);
  
  * result = info_array;
  
  return NULL;
  
 err:
  return error;
}

void etpan_imap_list_mailboxes_free(carray * info_array)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(info_array) ; i ++) {
    struct etpan_imap_mailbox_info * mb_info;
    
    mb_info = carray_get(info_array, i);
    mailbox_info_free(mb_info);
  }
  carray_free(info_array);
}

struct etpan_error * etpan_imap_select(mailsession * session,
    char * mailbox)
{
  int r;
  mailimap * imap;
  struct etpan_error * error;
  
  imap = get_imap(session);
  
  r = mailimap_select(imap, mailbox);
  if (r == MAILIMAP_ERROR_MEMORY)
    ETPAN_LOG_MEMORY_ERROR;
  
  if (r == MAILIMAP_ERROR_STREAM) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_STREAM);
    etpan_error_set_short_description(error, _("Connection closed"));
    etpan_error_strf_long_description(error, _("Connection was closed while analyzing mailbox %s."), mailbox);
    goto err;
  }
  else if (r == MAILIMAP_ERROR_PARSE) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, _("Parse error"));
    etpan_error_strf_long_description(error, _("The application did not understand what the server sent while analyzing the mailbox %s."), mailbox);
    goto err;
  }
  else if (r != MAILIMAP_NO_ERROR) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FOLDER_NOT_FOUND);
    etpan_error_set_short_description(error, _("Mailbox not found"));
    etpan_error_strf_long_description(error, _("Mailbox %s was not found."),
        mailbox);
    goto err;
  }
  
  error = etpan_imap_subscribe(session, mailbox);
  ETPAN_ERROR_IGNORE(error);
  
  return NULL;
  
 err:
  return error;
}

struct etpan_error * etpan_imap_create(mailsession * session,
    char * mailbox)
{
  int r;
  mailimap * imap;
  struct etpan_error * error;
  int subscribed;
  
  imap = get_imap(session);
  
  subscribed = 0;
  r = mailimap_create(imap, mailbox);
  if (r == MAILIMAP_ERROR_MEMORY)
    ETPAN_LOG_MEMORY_ERROR;
  
  if ((r != MAILIMAP_NO_ERROR) &&
      (r != MAILIMAP_ERROR_STREAM) &&
      (r != MAILIMAP_ERROR_PARSE)) {
    r = mailimap_subscribe(imap, mailbox);
    if (r == MAILIMAP_ERROR_MEMORY)
      ETPAN_LOG_MEMORY_ERROR;
    subscribed = 1;
  }
  if (r == MAILIMAP_ERROR_STREAM) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_STREAM);
    etpan_error_set_short_description(error, _("Connection closed"));
    etpan_error_strf_long_description(error, _("Connection was closed while creating mailbox %s"), mailbox);
    goto err;
  }
  else if (r == MAILIMAP_ERROR_PARSE) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, _("Parse error"));
    etpan_error_strf_long_description(error, _("The application did not understand what the server sent while creating mailbox %s."), mailbox);
    goto err;
  }
  else if (r != MAILIMAP_NO_ERROR) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_IMAP_CREATE);
    etpan_error_set_short_description(error, _("Could not create mailbox"));
    etpan_error_strf_long_description(error, _("An error occurred while creating mailbox %s. Check whether the mailbox already exists."), mailbox);
    goto err;
  }
  
  if (!subscribed) {
    error = etpan_imap_subscribe(session, mailbox);
    if (error != NULL)
      goto err;
  }
  
  return NULL;
  
 err:
  return error;
}

struct etpan_error * etpan_imap_delete(mailsession * session,
    char * mailbox)
{
  int r;
  mailimap * imap;
  struct etpan_error * error;
  
  imap = get_imap(session);
  
  error = etpan_imap_unsubscribe(session, mailbox);
  ETPAN_ERROR_IGNORE(error);
  
  r = mailimap_delete(imap, mailbox);
  if (r == MAILIMAP_ERROR_MEMORY)
    ETPAN_LOG_MEMORY_ERROR;
  
  if (r == MAILIMAP_ERROR_STREAM) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_STREAM);
    etpan_error_set_short_description(error, _("Connection closed"));
    etpan_error_strf_long_description(error, _("Connection was closed while deleting mailbox %s"), mailbox);
    goto err;
  }
  else if (r == MAILIMAP_ERROR_PARSE) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, _("Parse error"));
    etpan_error_strf_long_description(error, _("The application did not understand what the server sent while deleting mailbox %s."), mailbox);
    goto err;
  }
  else if (r != MAILIMAP_NO_ERROR) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_DELETE);
    etpan_error_set_short_description(error, _("Could not delete mailbox"));
    etpan_error_strf_long_description(error, _("An error occurred while deleting mailbox %s. The mailbox may already be deleted."), mailbox);
    goto err;
  }
  
  return NULL;
  
 err:
  return error;
}

struct etpan_error * etpan_imap_rename(mailsession * session,
    char * mailbox, char * new_mailbox)
{
  int r;
  mailimap * imap;
  struct etpan_error * error;
  
  imap = get_imap(session);
  
  r = mailimap_rename(imap, mailbox, new_mailbox);
  if (r == MAILIMAP_ERROR_MEMORY)
    ETPAN_LOG_MEMORY_ERROR;
  
  if (r == MAILIMAP_ERROR_STREAM) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_STREAM);
    etpan_error_set_short_description(error, _("Connection closed"));
    etpan_error_strf_long_description(error, _("Connection was closed while chaing the name of mailbox %s"), mailbox);
    goto err;
  }
  else if (r == MAILIMAP_ERROR_PARSE) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, _("Parse error"));
    etpan_error_strf_long_description(error, _("The application did not understand what the server sent while changing the name of mailbox %s."), mailbox);
    goto err;
  }
  else if (r != MAILIMAP_NO_ERROR) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_IMAP_RENAME);
    etpan_error_set_short_description(error, _("Could not change the name of the mailbox"));
    etpan_error_strf_long_description(error, _("An error occurred while changing the name of the mailbox %s. Check whether the mailbox still exists or the new name (%s) does not already exists."), mailbox, new_mailbox);
    goto err;
  }
  
  return NULL;
  
 err:
  return error;
}


struct etpan_error * etpan_imap_subscribe(mailsession * session,
    char * mailbox)
{
  int r;
  mailimap * imap;
  struct etpan_error * error;
  
  imap = get_imap(session);
  
  r = mailimap_subscribe(imap, mailbox);
  if (r == MAILIMAP_ERROR_MEMORY)
    ETPAN_LOG_MEMORY_ERROR;
  
  if (r == MAILIMAP_ERROR_STREAM) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_STREAM);
    etpan_error_set_short_description(error, _("Connection closed"));
    etpan_error_strf_long_description(error, _("Connection was closed while opening mailbox %s"), mailbox);
    goto err;
  }
  else if (r == MAILIMAP_ERROR_PARSE) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, _("Parse error"));
    etpan_error_strf_long_description(error, _("The application did not understand what the server sent while opening mailbox %s."), mailbox);
    goto err;
  }
  else if (r != MAILIMAP_NO_ERROR) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_IMAP_SUBSCRIBE);
    etpan_error_set_short_description(error, _("Could open the mailbox"));
    etpan_error_strf_long_description(error, _("An error occurred while opening the mailbox %s. Check whether the mailbox still exists."), mailbox);
    goto err;
  }
  
  return NULL;
  
 err:
  return error;
}

struct etpan_error * etpan_imap_unsubscribe(mailsession * session,
    char * mailbox)
{
  int r;
  mailimap * imap;
  struct etpan_error * error;
  
  imap = get_imap(session);
  
  r = mailimap_unsubscribe(imap, mailbox);
  if (r == MAILIMAP_ERROR_MEMORY)
    ETPAN_LOG_MEMORY_ERROR;
  
  if (r == MAILIMAP_ERROR_STREAM) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_STREAM);
    etpan_error_set_short_description(error, _("Connection closed"));
    etpan_error_strf_long_description(error, _("Connection was closed while closing mailbox %s"), mailbox);
    goto err;
  }
  else if (r == MAILIMAP_ERROR_PARSE) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, _("Parse error"));
    etpan_error_strf_long_description(error, _("The application did not understand what the server sent while closing mailbox %s."), mailbox);
    goto err;
  }
  else if (r != MAILIMAP_NO_ERROR) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_IMAP_UNSUBSCRIBE);
    etpan_error_set_short_description(error, _("Could close the mailbox"));
    etpan_error_strf_long_description(error, _("An error occurred while closing the mailbox %s. Check whether the mailbox still exists."), mailbox);
    goto err;
  }
  
  return NULL;
  
 err:
  return error;
}

struct mail_flags *
etpan_imap_flags_to_lep(struct mailimap_msg_att_dynamic * att_dyn)
{
  struct mail_flags * flags;
  clist * flag_list;
  clistiter * cur;

  flags = mail_flags_new_empty();
  if (flags == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  flags->fl_flags = 0;

  flag_list = att_dyn->att_list;
  if (flag_list != NULL) {
    for(cur = clist_begin(flag_list) ; cur != NULL ;
        cur = clist_next(cur)) {
      struct mailimap_flag_fetch * flag_fetch;

      flag_fetch = clist_content(cur);
      if (flag_fetch->fl_type == MAILIMAP_FLAG_FETCH_RECENT)
	flags->fl_flags |= MAIL_FLAG_NEW;
      else {
	char * keyword;
	int r;

	switch (flag_fetch->fl_flag->fl_type) {
	case MAILIMAP_FLAG_ANSWERED:
	  flags->fl_flags |= MAIL_FLAG_ANSWERED;
	  break;
	case MAILIMAP_FLAG_FLAGGED:
	  flags->fl_flags |= MAIL_FLAG_FLAGGED;
	  break;
	case MAILIMAP_FLAG_DELETED:
	  flags->fl_flags |= MAIL_FLAG_DELETED;
	  break;
	case MAILIMAP_FLAG_SEEN:
	  flags->fl_flags |= MAIL_FLAG_SEEN;
	  break;
	case MAILIMAP_FLAG_DRAFT:
	  keyword = strdup("Draft");
	  if (keyword == NULL)
            ETPAN_LOG_MEMORY_ERROR;
	  r = clist_append(flags->fl_extension, keyword);
	  if (r < 0)
            ETPAN_LOG_MEMORY_ERROR;
	  break;
	case MAILIMAP_FLAG_KEYWORD:
          if (strcasecmp(flag_fetch->fl_flag->fl_data.fl_keyword,
                  "$Forwarded") == 0) {
            flags->fl_flags |= MAIL_FLAG_FORWARDED;
          }
          else {
            keyword = strdup(flag_fetch->fl_flag->fl_data.fl_keyword);
            if (keyword == NULL)
              ETPAN_LOG_MEMORY_ERROR;
            r = clist_append(flags->fl_extension, keyword);
            if (r < 0)
              ETPAN_LOG_MEMORY_ERROR;
          }
	  break;
	case MAILIMAP_FLAG_EXTENSION:
	  /* do nothing */
	  break;
	}
      }
    }
    /*
      MAIL_FLAG_NEW was set for \Recent messages.
      Correct this flag for \Seen messages by unsetting it.
    */
    if ((flags->fl_flags & MAIL_FLAG_SEEN) && (flags->fl_flags & MAIL_FLAG_NEW)) {
      flags->fl_flags &= ~MAIL_FLAG_NEW;
    }
  }
  
  return flags;
}

struct mailimap_flag_list *
etpan_imap_flags_from_lep(struct mail_flags * flags)
{
  struct mailimap_flag * flag;
  struct mailimap_flag_list * flag_list;
  clistiter * cur;
  int r;

  flag_list = mailimap_flag_list_new_empty();
  if (flag_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;

  if ((flags->fl_flags & MAIL_FLAG_DELETED) != 0) {
    flag = mailimap_flag_new_deleted();
    if (flag == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    r = mailimap_flag_list_add(flag_list, flag);
    if (r != MAILIMAP_NO_ERROR)
      ETPAN_LOG_MEMORY_ERROR;
  }

  if ((flags->fl_flags & MAIL_FLAG_FLAGGED) != 0) {
    flag = mailimap_flag_new_flagged();
    if (flag == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    r = mailimap_flag_list_add(flag_list, flag);
    if (r != MAILIMAP_NO_ERROR)
      ETPAN_LOG_MEMORY_ERROR;
  }

  if ((flags->fl_flags & MAIL_FLAG_SEEN) != 0) {
    flag = mailimap_flag_new_seen();
    if (flag == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    r = mailimap_flag_list_add(flag_list, flag);
    if (r != MAILIMAP_NO_ERROR)
      ETPAN_LOG_MEMORY_ERROR;
  }

  if ((flags->fl_flags & MAIL_FLAG_ANSWERED) != 0) {
    flag = mailimap_flag_new_answered();
    if (flag == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    r = mailimap_flag_list_add(flag_list, flag);
    if (r != MAILIMAP_NO_ERROR)
      ETPAN_LOG_MEMORY_ERROR;
  }

  if ((flags->fl_flags & MAIL_FLAG_FORWARDED) != 0) {
    char * flag_str;
    
    flag_str = strdup("$Forwarded");
    if (flag_str == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    flag = mailimap_flag_new_flag_keyword(flag_str);
    if (flag == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    r = mailimap_flag_list_add(flag_list, flag);
    if (r != MAILIMAP_NO_ERROR)
      ETPAN_LOG_MEMORY_ERROR;
  }

  for(cur = clist_begin(flags->fl_extension) ; cur != NULL ;
      cur = clist_next(cur)) {
    char * flag_str;

    flag_str = clist_content(cur);

    if (strcasecmp(flag_str, "Draft") == 0) {
      flag = mailimap_flag_new_draft();
      if (flag == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      r = mailimap_flag_list_add(flag_list, flag);
      if (r != MAILIMAP_NO_ERROR)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else {
      flag_str = strdup(flag_str);
      if (flag_str == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      flag = mailimap_flag_new_flag_keyword(flag_str);
      if (flag == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      r = mailimap_flag_list_add(flag_list, flag);
      if (r != MAILIMAP_NO_ERROR)
        ETPAN_LOG_MEMORY_ERROR;
    }
  }
  
  return flag_list;
}

struct etpan_error *
etpan_imap_store_flags(mailsession * session, char * mailbox,
    uint32_t first, uint32_t last,
    struct mail_flags * flags)
{
  struct mailimap_store_att_flags * att_flags;
  struct mailimap_flag_list * flag_list;
  struct mailimap_set * set;
  int r;
  mailimap * imap;
  struct etpan_error * error;
  
  imap = get_imap(session);
  
  set = mailimap_set_new_interval(first, last);
  if (set == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  flag_list = etpan_imap_flags_from_lep(flags);
  att_flags = mailimap_store_att_flags_new_set_flags_silent(flag_list);
  if (att_flags == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  r = mailimap_uid_store(imap, set, att_flags);
  if (r == MAILIMAP_ERROR_MEMORY)
    ETPAN_LOG_MEMORY_ERROR;
  
  if (r == MAILIMAP_ERROR_STREAM) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_STREAM);
    etpan_error_set_short_description(error, _("Connection closed"));
    etpan_error_strf_long_description(error, _("Connection was closed while storing status of messages in mailbox %s"), mailbox);
    goto exit;
  }
  else if (r == MAILIMAP_ERROR_PARSE) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, _("Parse error"));
    etpan_error_strf_long_description(error, _("The application did not understand what the server sent while storing status of messages in mailbox %s."), mailbox);
    goto exit;
  }
  else if (r != MAILIMAP_NO_ERROR) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_IMAP_STORE);
    etpan_error_set_short_description(error, _("Could save the status of the messages"));
    etpan_error_strf_long_description(error, _("The status of the messages set of mailbox %s could not be saved."), mailbox);
    goto exit;
  }
  
  error = NULL;
  
 exit:
  mailimap_store_att_flags_free(att_flags);
  mailimap_set_free(set);
  return error;
}

struct etpan_error *
etpan_imap_append(mailsession * session, char * mailbox,
    char * content, size_t content_length, struct mail_flags * flags)
{
  mailimap * imap;
  struct etpan_error * error;
  int r;
  struct mailimap_flag_list * flag_list;
  
  imap = get_imap(session);
  
  flag_list = etpan_imap_flags_from_lep(flags);
  r = mailimap_append(imap, mailbox, flag_list, NULL,
      content, content_length);
  mailimap_flag_list_free(flag_list);
  
  if (r == MAILIMAP_ERROR_MEMORY)
    ETPAN_LOG_MEMORY_ERROR;
  
  if (r == MAILIMAP_ERROR_STREAM) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_STREAM);
    etpan_error_set_short_description(error, _("Connection closed"));
    etpan_error_strf_long_description(error, _("Connection was closed while adding a message to mailbox %s"), mailbox);
    goto exit;
  }
  else if (r == MAILIMAP_ERROR_PARSE) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, _("Parse error"));
    etpan_error_strf_long_description(error, _("The application did not understand what the server sent while adding a message to mailbox %s."), mailbox);
    goto exit;
  }
  else if (r != MAILIMAP_NO_ERROR) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_IMAP_APPEND);
    etpan_error_set_short_description(error, _("Could add message"));
    etpan_error_strf_long_description(error, _("The message could not be added to mailbox %s."), mailbox);
    goto exit;
  }
  
  error = NULL;
  
 exit:
  return error;
}

struct etpan_error *
etpan_imap_expunge(mailsession * session, char * mailbox)
{
  mailimap * imap;
  struct etpan_error * error;
  int r;
  
  imap = get_imap(session);
  r = mailimap_expunge(imap);
  if (r == MAILIMAP_ERROR_MEMORY)
    ETPAN_LOG_MEMORY_ERROR;
  
  if (r == MAILIMAP_ERROR_STREAM) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_STREAM);
    etpan_error_set_short_description(error, _("Connection closed"));
    etpan_error_strf_long_description(error, _("Connection was closed while cleaning up mailbox %s"), mailbox);
    goto exit;
  }
  else if (r == MAILIMAP_ERROR_PARSE) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, _("Parse error"));
    etpan_error_strf_long_description(error, _("The application did not understand what the server sent while cleaning up mailbox %s."), mailbox);
    goto exit;
  }
  else if (r != MAILIMAP_NO_ERROR) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_IMAP_EXPUNGE);
    etpan_error_set_short_description(error, _("Could not clean the mailbox"));
    etpan_error_strf_long_description(error, _("The mailbox %s could not be cleaned on server."), mailbox);
    goto exit;
  }
  
  error = NULL;
  
 exit:
  return error;
}

#if 0
static struct mailimap_set_item *
imap_set_item_dup(struct mailimap_set_item * item)
{
  struct mailimap_set_item * result;
  
  result = mailimap_set_item_new(item->set_first, item->set_last);
  if (result == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return result;
}

static struct mailimap_set * imap_set_dup(struct mailimap_set * set)
{
  clistiter * iter;
  struct mailimap_set * result;
  int r;
  
  result = mailimap_set_new_empty();
  if (result == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  for(iter = clist_begin(set->set_list) ; iter != NULL ;
      iter = clist_next(iter)) {
    struct mailimap_set_item * item_dup;
    struct mailimap_set_item * item;
    
    item = clist_content(iter);
    item_dup = imap_set_item_dup(item);
    r = mailimap_set_add(result, item_dup);
    if (r != MAILIMAP_NO_ERROR)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  return result;
}
#endif


struct etpan_error * etpan_imap_status(mailsession * session,
    const char * mailbox,
    unsigned int * result_messages,
    unsigned int * result_unseen,
    unsigned int * result_recent)
{
  struct mailimap_status_att_list * att_list;
  struct mailimap_mailbox_data_status * status;
  int r;
  clistiter * cur;
  mailimap * imap;
  struct etpan_error * error;
  
  imap = get_imap(session);
  
  att_list = mailimap_status_att_list_new_empty();
  if (att_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  r = mailimap_status_att_list_add(att_list, MAILIMAP_STATUS_ATT_MESSAGES);
  if (r != MAILIMAP_NO_ERROR)
    ETPAN_LOG_MEMORY_ERROR;
  
  r = mailimap_status_att_list_add(att_list, MAILIMAP_STATUS_ATT_RECENT);
  if (r != MAILIMAP_NO_ERROR)
    ETPAN_LOG_MEMORY_ERROR;
  
  r = mailimap_status_att_list_add(att_list, MAILIMAP_STATUS_ATT_UNSEEN);
  if (r != MAILIMAP_NO_ERROR)
    ETPAN_LOG_MEMORY_ERROR;
  
  r = mailimap_status(imap, mailbox, att_list, &status);
  mailimap_status_att_list_free(att_list);
  if (r == MAILIMAP_ERROR_MEMORY)
    ETPAN_LOG_MEMORY_ERROR;
  
  if (r == MAILIMAP_ERROR_STREAM) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_STREAM);
    etpan_error_set_short_description(error, _("Connection closed"));
    etpan_error_strf_long_description(error, _("Connection was closed while cleaning up mailbox %s"), mailbox);
    goto exit;
  }
  else if (r == MAILIMAP_ERROR_PARSE) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, _("Parse error"));
    etpan_error_strf_long_description(error, _("The application did not understand what the server sent while cleaning up mailbox %s."), mailbox);
    goto exit;
  }
  else if (r != MAILIMAP_NO_ERROR) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_IMAP_SUBSCRIBE);
    etpan_error_set_short_description(error, _("Could not get information of the mailbox"));
    etpan_error_strf_long_description(error, _("Information of the mailbox %s could not be retrieved from the server."), mailbox);
    goto exit;
  }

  * result_messages = 0;
  * result_recent = 0;
  * result_unseen = 0;
  
  for (cur = clist_begin(status->st_info_list);
       cur != NULL ; cur = clist_next(cur)) {
    struct mailimap_status_info * status_info;
      
    status_info = clist_content(cur);
    switch (status_info->st_att) {
    case MAILIMAP_STATUS_ATT_MESSAGES:
      * result_messages = status_info->st_value;
      break;
    case MAILIMAP_STATUS_ATT_RECENT:
      * result_recent = status_info->st_value;
      break;
    case MAILIMAP_STATUS_ATT_UNSEEN:
      * result_unseen = status_info->st_value;
      break;
    }
  }

  mailimap_mailbox_data_status_free(status); 
  
  error = NULL;  
  
 exit:
  return error;
}


static struct mailmime_disposition *
imap_disposition_to_mime_disposition(struct mailimap_body_fld_dsp * imap_dsp)
{
  size_t cur_token;
  int r;
  struct mailmime_disposition_type * dsp_type;
  struct mailmime_disposition * dsp;
  clist * parameters;
  int res;

  cur_token = 0;
  r = mailmime_disposition_type_parse(imap_dsp->dsp_type,
      strlen(imap_dsp->dsp_type), &cur_token, &dsp_type);
  if (r == MAILIMF_ERROR_MEMORY)
    ETPAN_LOG_MEMORY_ERROR;
  
  if (r != MAILIMF_NO_ERROR)
    return NULL;

  parameters = clist_new();
  if (parameters == NULL)
    ETPAN_LOG_MEMORY_ERROR;

  if (imap_dsp->dsp_attributes != NULL) {
    clistiter * cur;

    for(cur = clist_begin(imap_dsp->dsp_attributes->pa_list) ; cur != NULL ;
	cur = clist_next(cur)) {
      struct mailimap_single_body_fld_param * imap_param;
      struct mailmime_disposition_parm * dsp_param;
      struct mailmime_parameter * param;
      char * filename;
      char * creation_date;
      char * modification_date;
      char * read_date;
      size_t size;
      int type;

      imap_param = clist_content(cur);

      filename = NULL;
      creation_date = NULL;
      modification_date = NULL;
      read_date = NULL;
      size = 0;
      param = NULL;

      type = mailmime_disposition_guess_type(imap_param->pa_name,
          strlen(imap_param->pa_name), 0);

      switch (type) {
      case MAILMIME_DISPOSITION_PARM_FILENAME:
	if (strcasecmp(imap_param->pa_name, "filename") != 0) {
	  type = MAILMIME_DISPOSITION_PARM_PARAMETER;
	  break;
	}
	filename = strdup(imap_param->pa_value);
	if (filename == NULL)
          ETPAN_LOG_MEMORY_ERROR;
	break;

      case MAILMIME_DISPOSITION_PARM_CREATION_DATE:
	if (strcasecmp(imap_param->pa_name, "creation-date") != 0) {
	  type = MAILMIME_DISPOSITION_PARM_PARAMETER;
	  break;
	}
	creation_date = strdup(imap_param->pa_value);
	if (creation_date == NULL)
          ETPAN_LOG_MEMORY_ERROR;
	break;

      case MAILMIME_DISPOSITION_PARM_MODIFICATION_DATE:
	if (strcasecmp(imap_param->pa_name, "modification-date") != 0) {
	  type = MAILMIME_DISPOSITION_PARM_PARAMETER;
	  break;
	}
	modification_date = strdup(imap_param->pa_value);
	if (modification_date == NULL)
          ETPAN_LOG_MEMORY_ERROR;
	break;
    
      case MAILMIME_DISPOSITION_PARM_READ_DATE:
	if (strcasecmp(imap_param->pa_name, "read-date") != 0) {
	  type = MAILMIME_DISPOSITION_PARM_PARAMETER;
	  break;
	}
	read_date = strdup(imap_param->pa_value);
	if (read_date == NULL)
          ETPAN_LOG_MEMORY_ERROR;
	break;

      case MAILMIME_DISPOSITION_PARM_SIZE:
	if (strcasecmp(imap_param->pa_name, "size") != 0) {
	  type = MAILMIME_DISPOSITION_PARM_PARAMETER;
	  break;
	}
	size = strtoul(imap_param->pa_value, NULL, 10);
	break;
      }

      if (type == MAILMIME_DISPOSITION_PARM_PARAMETER) {
	char * name;
	char * value;

	name = strdup(imap_param->pa_name);
	if (name == NULL)
          ETPAN_LOG_MEMORY_ERROR;
	
	value = strdup(imap_param->pa_value);
	if (value == NULL)
          ETPAN_LOG_MEMORY_ERROR;

	param = mailmime_parameter_new(name, value);
	if (param == NULL)
          ETPAN_LOG_MEMORY_ERROR;
      }

      dsp_param = mailmime_disposition_parm_new(type, filename,
						creation_date,
						modification_date,
						read_date,
						size, param);
      if (dsp_param == NULL) {
        ETPAN_LOG_MEMORY_ERROR;
      }

      r = clist_append(parameters, dsp_param);
      if (r != 0) {
        ETPAN_LOG_MEMORY_ERROR;
      }
    }
  }

  dsp = mailmime_disposition_new(dsp_type, parameters);
  if (dsp == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }

  return dsp;
}

static struct mailmime_language *
imap_language_to_mime_language(struct mailimap_body_fld_lang * imap_lang)
{
  clist * list;
  clistiter * cur;
  int res;
  char * single;
  int r;
  struct mailmime_language * lang;

  list = clist_new();
  if (list == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }

  switch (imap_lang->lg_type) {
  case MAILIMAP_BODY_FLD_LANG_SINGLE:
    if (imap_lang->lg_data.lg_single != NULL) {
      single = strdup(imap_lang->lg_data.lg_single);
      if (single == NULL) {
        ETPAN_LOG_MEMORY_ERROR;
      }
      r = clist_append(list, single);
      if (r < 0) {
        ETPAN_LOG_MEMORY_ERROR;
      }
    }

    break;

  case MAILIMAP_BODY_FLD_LANG_LIST:
    for(cur = clist_begin(imap_lang->lg_data.lg_list) ;
        cur != NULL ; cur = clist_next(cur)) {
      char * original_single;
      
      original_single = clist_content(cur);
      
      single = strdup(original_single);
      if (single == NULL) {
        ETPAN_LOG_MEMORY_ERROR;
      }
      r = clist_append(list, single);
      if (r < 0) {
        ETPAN_LOG_MEMORY_ERROR;
      }
    }
  }

  lang = mailmime_language_new(list);
  if (lang == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  return lang;
}

static void
imap_body_fields_to_mime_fields(struct mailimap_body_fields * body_fields,
    struct mailimap_body_fld_dsp * imap_dsp,
    struct mailimap_body_fld_lang * imap_lang,
    struct mailmime_fields ** result,
    uint32_t * pbody_size)
{
  struct mailmime_field * mime_field;
  struct mailmime_fields * mime_fields;
  clist * list;
  char * id;
  struct mailmime_mechanism * encoding;
  char * description;
  struct mailmime_disposition * dsp;
  struct mailmime_language * lang;
  int type;
  int r;
  int res;

  list = clist_new();
  if (list == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
 
  if (body_fields != NULL) {
    
    if (pbody_size != NULL)
      * pbody_size = body_fields->bd_size;
    
    if (body_fields->bd_id != NULL) {
      type = MAILMIME_FIELD_ID;
      id = strdup(body_fields->bd_id);
      if (id == NULL) {
        ETPAN_LOG_MEMORY_ERROR;
      }

      mime_field = mailmime_field_new(type, NULL,
          NULL, id, NULL, 0, NULL, NULL);
      if (mime_field == NULL) {
        ETPAN_LOG_MEMORY_ERROR;
      }

      r = clist_append(list, mime_field);
      if (r != 0) {
        ETPAN_LOG_MEMORY_ERROR;
      }
    }

    if (body_fields->bd_description != NULL) {
      type = MAILMIME_FIELD_DESCRIPTION;
      description = strdup(body_fields->bd_description);
      if (description == NULL) {
        ETPAN_LOG_MEMORY_ERROR;
      }

      mime_field = mailmime_field_new(type, NULL,
          NULL, NULL, description, 0, NULL, NULL);
      if (mime_field == NULL) {
        ETPAN_LOG_MEMORY_ERROR;
      }

      r = clist_append(list, mime_field);
      if (r != 0) {
        ETPAN_LOG_MEMORY_ERROR;
      }
    }
  
    if (body_fields->bd_encoding != NULL) {
      char * encoding_value;
      int encoding_type;

      type = MAILMIME_FIELD_TRANSFER_ENCODING;

      encoding_type = MAILMIME_MECHANISM_8BIT;
      encoding_value = NULL;
      switch (body_fields->bd_encoding->enc_type) {
      case MAILIMAP_BODY_FLD_ENC_7BIT:
	encoding_type = MAILMIME_MECHANISM_7BIT;
	break;
      case MAILIMAP_BODY_FLD_ENC_8BIT:
	encoding_type = MAILMIME_MECHANISM_8BIT;
	break;
      case MAILIMAP_BODY_FLD_ENC_BINARY:
	encoding_type = MAILMIME_MECHANISM_BINARY;
	break;
      case MAILIMAP_BODY_FLD_ENC_BASE64:
	encoding_type = MAILMIME_MECHANISM_BASE64;
	break;
      case MAILIMAP_BODY_FLD_ENC_QUOTED_PRINTABLE:
	encoding_type = MAILMIME_MECHANISM_QUOTED_PRINTABLE;
	break;
      case MAILIMAP_BODY_FLD_ENC_OTHER:
	encoding_type = MAILMIME_MECHANISM_TOKEN;
	encoding_value = strdup(body_fields->bd_encoding->enc_value);
	if (encoding_value == NULL) {
          ETPAN_LOG_MEMORY_ERROR;
	}
	break;
      default:
        ETPAN_LOG_MEMORY_ERROR;
        break;
      }

      encoding = mailmime_mechanism_new(encoding_type, encoding_value);
      if (encoding == NULL) {
        ETPAN_LOG_MEMORY_ERROR;
      }

      mime_field = mailmime_field_new(type, NULL,
          encoding, NULL, NULL, 0, NULL, NULL);
      if (mime_field == NULL) {
        ETPAN_LOG_MEMORY_ERROR;
      }

      r = clist_append(list, mime_field);
      if (r != 0) {
        ETPAN_LOG_MEMORY_ERROR;
      }
    }
  }

  if (imap_dsp != NULL) {
    dsp = imap_disposition_to_mime_disposition(imap_dsp);
    if (dsp != NULL) {
      type = MAILMIME_FIELD_DISPOSITION;
      
      mime_field = mailmime_field_new(type, NULL,
				      NULL, NULL, NULL, 0, dsp, NULL);
      if (mime_field == NULL) {
        ETPAN_LOG_MEMORY_ERROR;
      }
      
      r = clist_append(list, mime_field);
      if (r != 0) {
        ETPAN_LOG_MEMORY_ERROR;
      }
    }
  }

  if (imap_lang != NULL) {
    int has_lang;
    
    has_lang = 1;
    if (imap_lang->lg_type == MAILIMAP_BODY_FLD_LANG_SINGLE)
      if (imap_lang->lg_data.lg_single == NULL)
        has_lang = 0;
    
    if (has_lang) {
      lang = imap_language_to_mime_language(imap_lang);
      
      type = MAILMIME_FIELD_LANGUAGE;

      mime_field = mailmime_field_new(type, NULL,
          NULL, NULL, NULL, 0, NULL, lang);
      if (mime_field == NULL) {
        ETPAN_LOG_MEMORY_ERROR;
      }

      r = clist_append(list, mime_field);
      if (r != 0) {
        ETPAN_LOG_MEMORY_ERROR;
      }
    }
  }

  mime_fields = mailmime_fields_new(list);
  if (mime_fields == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }

  * result = mime_fields;
}

static struct mailmime_content *
imap_body_parameter_to_content(struct mailimap_body_fld_param * body_parameter,
			       char * subtype,
			       struct mailmime_type * mime_type)
{
  clist * parameters;
  char * new_subtype;
  struct mailmime_content * content;
  int r;
  int res;

  new_subtype = strdup(subtype);
  if (new_subtype == NULL)
    ETPAN_LOG_MEMORY_ERROR;

  parameters = clist_new();
  if (parameters == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  if (body_parameter != NULL) {
    clistiter * cur;

    for(cur = clist_begin(body_parameter->pa_list) ; cur != NULL ;
	cur = clist_next(cur)) {
      struct mailimap_single_body_fld_param * imap_param;
      struct mailmime_parameter * param;
      char * name;
      char * value;

      imap_param = clist_content(cur);
      name = strdup(imap_param->pa_name);
      if (name == NULL)
        ETPAN_LOG_MEMORY_ERROR;

      value = strdup(imap_param->pa_value);
      if (value == NULL)
        ETPAN_LOG_MEMORY_ERROR;

      param = mailmime_parameter_new(name, value);
      if (param == NULL)
        ETPAN_LOG_MEMORY_ERROR;

      r = clist_append(parameters, param);
      if (r != 0)
        ETPAN_LOG_MEMORY_ERROR;
    }
  }

  content = mailmime_content_new(mime_type, new_subtype, parameters);
  if (content == NULL)
    ETPAN_LOG_MEMORY_ERROR;

  return content;
}

static void
get_fields_content_from_mpart(struct mailimap_body_type_mpart * type_mpart,
    struct mailmime_fields ** p_mime_fields,
    struct mailmime_content ** p_mime_content)
{
  struct mailmime_composite_type * composite_type;
  struct mailmime_type * mime_type;
  
  if (type_mpart->bd_ext_mpart == NULL) {
    * p_mime_fields = mailmime_fields_new_empty();
  }
  else {
    imap_body_fields_to_mime_fields(NULL,
        type_mpart->bd_ext_mpart->bd_disposition,
        type_mpart->bd_ext_mpart->bd_language,
        p_mime_fields, NULL);
  }
  
  composite_type =
    mailmime_composite_type_new(MAILMIME_COMPOSITE_TYPE_MULTIPART, NULL);
  if (composite_type == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  mime_type = mailmime_type_new(MAILMIME_TYPE_COMPOSITE_TYPE, NULL,
      composite_type);
  if (mime_type == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  if (type_mpart->bd_ext_mpart == NULL) {
    * p_mime_content =
      imap_body_parameter_to_content(NULL,
          type_mpart->bd_media_subtype,
          mime_type);
  }
  else {
    * p_mime_content =
      imap_body_parameter_to_content(type_mpart->bd_ext_mpart->bd_parameter,
          type_mpart->bd_media_subtype,
          mime_type);
  }
}

static struct mailmime_content *
imap_body_media_basic_to_content_type(struct mailimap_media_basic *
    media_basic,
    struct mailimap_body_fld_param * body_parameter)
{
  struct mailmime_content * content;
  struct mailmime_type * mime_type;
  struct mailmime_discrete_type * discrete_type;
  struct mailmime_composite_type * composite_type;
  char * discrete_type_extension;
  int discrete_type_type;
  int composite_type_type;
  int mime_type_type;
  char * subtype;
  int r;
  int res;

  discrete_type = NULL;
  composite_type = NULL;
  discrete_type_extension = NULL;
  subtype = NULL;
  discrete_type_type = 0;
  composite_type_type = 0;
  mime_type_type = 0;

  switch (media_basic->med_type) {
  case MAILIMAP_MEDIA_BASIC_APPLICATION:
    mime_type_type = MAILMIME_TYPE_DISCRETE_TYPE;
    discrete_type_type = MAILMIME_DISCRETE_TYPE_APPLICATION;
    break;

  case MAILIMAP_MEDIA_BASIC_AUDIO:
    mime_type_type = MAILMIME_TYPE_DISCRETE_TYPE;
    discrete_type_type = MAILMIME_DISCRETE_TYPE_APPLICATION;
    break;

  case MAILIMAP_MEDIA_BASIC_IMAGE:
    mime_type_type = MAILMIME_TYPE_DISCRETE_TYPE;
    discrete_type_type = MAILMIME_DISCRETE_TYPE_IMAGE;
    break;

  case MAILIMAP_MEDIA_BASIC_MESSAGE:
    mime_type_type = MAILMIME_TYPE_COMPOSITE_TYPE;
    composite_type_type = MAILMIME_COMPOSITE_TYPE_MESSAGE;
    break;

  case MAILIMAP_MEDIA_BASIC_VIDEO:
    mime_type_type = MAILMIME_TYPE_DISCRETE_TYPE;
    discrete_type_type = MAILMIME_DISCRETE_TYPE_VIDEO;
    break;

  case MAILIMAP_MEDIA_BASIC_OTHER:
    mime_type_type = MAILMIME_TYPE_DISCRETE_TYPE;
    discrete_type_type = MAILMIME_DISCRETE_TYPE_EXTENSION;
    discrete_type_extension = media_basic->med_basic_type;
    if (discrete_type_extension == NULL) {
      ETPAN_CRASH("invalid mime type");
    }

    break;

  default:
    ETPAN_CRASH("invalid");
    break;
  }

  switch (mime_type_type) {
  case MAILMIME_TYPE_DISCRETE_TYPE:
    if (discrete_type_extension != NULL) {
      discrete_type_extension = strdup(discrete_type_extension);
      if (discrete_type_extension == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    
    discrete_type = mailmime_discrete_type_new(discrete_type_type,
					       discrete_type_extension);
    if (discrete_type == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    
    break; 
    
  case MAILMIME_TYPE_COMPOSITE_TYPE:
    composite_type = mailmime_composite_type_new(composite_type_type,
						 NULL);
    if (composite_type == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    
    break; 

  default:
    ETPAN_CRASH("invalid");
    break;
  }

  mime_type = mailmime_type_new(mime_type_type, discrete_type, composite_type);
  if (mime_type == NULL)
    ETPAN_LOG_MEMORY_ERROR;

  return imap_body_parameter_to_content(body_parameter,
      media_basic->med_subtype, mime_type);
}

static void
get_fields_content_from_basic(struct mailimap_body_type_basic *
    imap_type_basic,
    struct mailimap_body_ext_1part * body_ext_1part,
    struct mailmime_fields ** p_mime_fields,
    struct mailmime_content ** p_mime_content,
    uint32_t * p_mime_size)
{
  struct mailimap_body_fld_dsp * imap_dsp;
  struct mailimap_body_fld_lang * imap_lang;
  
  * p_mime_content = imap_body_media_basic_to_content_type(imap_type_basic->bd_media_basic,
      imap_type_basic->bd_fields->bd_parameter);
  
  imap_dsp = NULL;
  imap_lang = NULL;
  if (body_ext_1part != NULL) {
    imap_dsp = body_ext_1part->bd_disposition;
    imap_lang = body_ext_1part->bd_language;
  }
  
  imap_body_fields_to_mime_fields(imap_type_basic->bd_fields,
      imap_dsp, imap_lang,
      p_mime_fields, p_mime_size);
}

static void
get_fields_content_from_msg(struct mailimap_body_type_msg *
    imap_type_msg,
    struct mailimap_body_ext_1part * body_ext_1part,
    struct mailmime_fields ** p_mime_fields,
    struct mailmime_content ** p_mime_content,
    uint32_t * p_mime_size)
{
  struct mailmime_composite_type * composite_type;
  struct mailmime_type * mime_type;
  
  imap_body_fields_to_mime_fields(imap_type_msg->bd_fields,
      body_ext_1part->bd_disposition, body_ext_1part->bd_language,
      p_mime_fields, NULL);
  
  composite_type =
    mailmime_composite_type_new(MAILMIME_COMPOSITE_TYPE_MESSAGE,
        NULL);
  if (composite_type == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  mime_type = mailmime_type_new(MAILMIME_TYPE_COMPOSITE_TYPE,
				NULL, composite_type);
  if (mime_type == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  * p_mime_content =
    imap_body_parameter_to_content(imap_type_msg->bd_fields->bd_parameter,
      "rfc822", mime_type);
  
  * p_mime_size = 0;
}

static struct mailmime_content *
imap_body_type_text_to_content_type(char * subtype,
    struct mailimap_body_fld_param * body_parameter)
{
  struct mailmime_content * content;
  struct mailmime_type * mime_type;
  struct mailmime_discrete_type * discrete_type;
  int r;
  int res;

  discrete_type = NULL;

  discrete_type = mailmime_discrete_type_new(MAILMIME_DISCRETE_TYPE_TEXT,
					     NULL);
  if (discrete_type == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  mime_type = mailmime_type_new(MAILMIME_TYPE_DISCRETE_TYPE,
				discrete_type, NULL);
  if (mime_type == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return imap_body_parameter_to_content(body_parameter, subtype, mime_type);
}

static void get_fields_content_from_text(struct mailimap_body_type_text *
    imap_type_text,
    struct mailimap_body_ext_1part * body_ext_1part,
    struct mailmime_fields ** p_mime_fields,
    struct mailmime_content ** p_mime_content,
    uint32_t * p_mime_size)
{
  struct mailmime_content * content;
  struct mailmime_fields * mime_fields;
  struct mailmime * body;
  int r;
  int res;
  uint32_t mime_size;
  struct mailimap_body_fld_dsp * imap_dsp;
  struct mailimap_body_fld_lang * imap_lang;
  
  * p_mime_content =
    imap_body_type_text_to_content_type(imap_type_text->bd_media_text,
        imap_type_text->bd_fields->bd_parameter);
  
  imap_dsp = NULL;
  imap_lang = NULL;
  if (body_ext_1part != NULL) {
    imap_dsp = body_ext_1part->bd_disposition;
    imap_lang = body_ext_1part->bd_language;
  }
  
  imap_body_fields_to_mime_fields(imap_type_text->bd_fields,
      imap_dsp, imap_lang,
      p_mime_fields, p_mime_size);
}

static void
get_fields_content_from_1part(struct mailimap_body_type_1part * type_1part,
    struct mailmime_fields ** p_mime_fields,
    struct mailmime_content ** p_mime_content,
    uint32_t * p_mime_size)
{
  switch (type_1part->bd_type) {
  case MAILIMAP_BODY_TYPE_1PART_BASIC:
    get_fields_content_from_basic(type_1part->bd_data.bd_type_basic,
        type_1part->bd_ext_1part, p_mime_fields, p_mime_content,
        p_mime_size);
    break;
  case MAILIMAP_BODY_TYPE_1PART_MSG:
    get_fields_content_from_msg(type_1part->bd_data.bd_type_msg,
        type_1part->bd_ext_1part, p_mime_fields, p_mime_content,
        p_mime_size);
    break;
  case MAILIMAP_BODY_TYPE_1PART_TEXT:
    get_fields_content_from_text(type_1part->bd_data.bd_type_text,
        type_1part->bd_ext_1part, p_mime_fields, p_mime_content,
        p_mime_size);
    break;
  }
}

void etpan_imap_get_fields_content_from_body(struct mailimap_body * imap_body,
    struct mailmime_fields ** p_mime_fields,
    struct mailmime_content ** p_mime_content,
    uint32_t * p_mime_size)
{
  switch (imap_body->bd_type) {
  case MAILIMAP_BODY_1PART:
    get_fields_content_from_1part(imap_body->bd_data.bd_body_1part,
        p_mime_fields, p_mime_content, p_mime_size);
    break;
  case MAILIMAP_BODY_MPART:
    get_fields_content_from_mpart(imap_body->bd_data.bd_body_mpart,
        p_mime_fields, p_mime_content);
    * p_mime_size = 0;
    break;
  }
}

/* XXX */
static struct mailmime *
imap_body_type_basic_or_text_to_body(struct mailimap_body * imap_body)
{
  struct mailmime_content * content;
  struct mailmime_fields * mime_fields;
  struct mailmime * body;
  int r;
  int res;
  uint32_t mime_size;
  
  etpan_imap_get_fields_content_from_body(imap_body,
      &mime_fields, &content, &mime_size);
  
  body = mailmime_new(MAILMIME_SINGLE, NULL,
      mime_size, mime_fields, content,
      NULL, NULL, NULL, NULL, NULL, NULL);
  if (body == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return body;
}

static struct mailmime *
imap_body_type_msg_to_body(struct mailimap_body * imap_body,
    struct mailimap_body_type_msg * imap_type_msg)
{
  struct mailmime * body;
  struct mailmime * msg_body;
  struct mailmime_fields * mime_fields;
  struct mailmime_composite_type * composite_type;
  struct mailmime_type * mime_type;
  struct mailmime_content * content;
  struct mailimf_fields * fields;
  int r;
  int res;
  uint32_t mime_size;
  
  etpan_imap_get_fields_content_from_body(imap_body,
      &mime_fields, &content, &mime_size);
  
  fields = etpan_imap_env_to_lep(imap_type_msg->bd_envelope, NULL, 0);
  
  msg_body = etpan_imap_body_to_lep(imap_type_msg->bd_body);
  
  body = mailmime_new(MAILMIME_MESSAGE, NULL,
      mime_size, mime_fields, content,
      NULL, NULL, NULL, NULL, fields, msg_body);
  if (body == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return body;
}


static struct mailmime *
imap_body_type_1part_to_body(struct mailimap_body * imap_body,
    struct mailimap_body_type_1part * type_1part)
{
  struct mailmime * body;
  int r;
  int res;
  
  body = NULL;
  switch (type_1part->bd_type) {
  case MAILIMAP_BODY_TYPE_1PART_BASIC:
  case MAILIMAP_BODY_TYPE_1PART_TEXT:
    body = imap_body_type_basic_or_text_to_body(imap_body);
    break;
  case MAILIMAP_BODY_TYPE_1PART_MSG:
    body = imap_body_type_msg_to_body(imap_body,
        type_1part->bd_data.bd_type_msg);
    break;
  }
  
  return body;
}

static struct mailmime *
imap_body_type_mpart_to_body(struct mailimap_body * imap_body,
    struct mailimap_body_type_mpart * type_mpart)
{
  struct mailmime_fields * mime_fields;
  struct mailmime_composite_type * composite_type;
  struct mailmime_type * mime_type;
  struct mailmime_content * content;
  struct mailmime * body;
  clistiter * cur;
  clist * list;
  int r;
  int res;
  uint32_t mime_size;

  etpan_imap_get_fields_content_from_body(imap_body,
      &mime_fields, &content, &mime_size);

  list = clist_new();
  if (list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  for(cur = clist_begin(type_mpart->bd_list) ; cur != NULL ;
      cur = clist_next(cur)) {
    struct mailimap_body * imap_body;
    struct mailmime * sub_body;

    imap_body = clist_content(cur);

    sub_body = etpan_imap_body_to_lep(imap_body);
    r = clist_append(list, sub_body);
    if (r != 0)
      ETPAN_LOG_MEMORY_ERROR;
  }

  body = mailmime_new(MAILMIME_MULTIPLE, NULL,
      mime_size, mime_fields, content,
      NULL, NULL, NULL, list, NULL, NULL);
  
  return body;
}


struct mailmime * etpan_imap_body_to_lep(struct mailimap_body * imap_body)
{
  struct mailmime * body;
  int r;
  int res;

  body = NULL;
  switch (imap_body->bd_type) {
  case MAILIMAP_BODY_1PART:
    body = imap_body_type_1part_to_body(imap_body,
        imap_body->bd_data.bd_body_1part);
    break;
  case MAILIMAP_BODY_MPART:
    body = imap_body_type_mpart_to_body(imap_body,
        imap_body->bd_data.bd_body_mpart);
    break;
  default:
    ETPAN_CRASH("invalid body type");
    break;
  }
  
  return body;
}



static struct mailimf_mailbox *
imap_address_to_mailbox(struct mailimap_address * imap_addr)
{
  char * dsp_name;
  char * addr;
  struct mailimf_mailbox * mb;
  int res;

  if (imap_addr->ad_personal_name == NULL)
    dsp_name = NULL;
  else {
    dsp_name = strdup(imap_addr->ad_personal_name);
    if (dsp_name == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }

  if (imap_addr->ad_host_name == NULL) {
    if (imap_addr->ad_mailbox_name == NULL) {
      addr = strdup("");
    }
    else {
      addr = strdup(imap_addr->ad_mailbox_name);
    }
    if (addr == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  else {
    addr = malloc(strlen(imap_addr->ad_mailbox_name) +
        strlen(imap_addr->ad_host_name) + 2);
    if (addr == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    strcpy(addr, imap_addr->ad_mailbox_name);
    strcat(addr, "@");
    strcat(addr, imap_addr->ad_host_name);
  }

  mb = mailimf_mailbox_new(dsp_name, addr);
  if (mb == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return mb;
}

static struct mailimf_mailbox_list *
imap_mailbox_list_to_mailbox_list(clist * imap_mailbox_list)
{
  clistiter * cur;
  clist * list;
  struct mailimf_mailbox_list * mb_list;
  int r;
  int res;

  list = clist_new();
  if (list == NULL)
    ETPAN_LOG_MEMORY_ERROR;

  for(cur = clist_begin(imap_mailbox_list) ; cur != NULL ;
      cur = clist_next(cur)) {
    struct mailimap_address * imap_addr;
    struct mailimf_mailbox * mb;

    imap_addr = clist_content(cur);

    if (imap_addr->ad_mailbox_name == NULL)
      continue;

    mb = imap_address_to_mailbox(imap_addr);
    r = clist_append(list, mb);
    if (r != 0)
      ETPAN_LOG_MEMORY_ERROR;
  }

  mb_list = mailimf_mailbox_list_new(list);
  if (mb_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return mb_list;
}

static struct mailimf_address *
imap_address_to_address(struct mailimap_address * imap_addr)
{
  struct mailimf_address * addr;
  struct mailimf_mailbox * mb;
  int r;
  int res;

  mb = imap_address_to_mailbox(imap_addr);
  addr = mailimf_address_new(MAILIMF_ADDRESS_MAILBOX, mb, NULL);
  if (addr == NULL)
    ETPAN_LOG_MEMORY_ERROR;

  return addr;
}

/*
  at exit, imap_mb_list will fall on the last element of the group,
  where mailbox name will be NIL, so that imap_mailbox_list_to_address_list
  can continue
*/

static struct mailimf_group *
imap_mailbox_list_to_group(clist * imap_mb_list, clistiter ** iter)
{
  clistiter * imap_mailbox_listiter;
  clist * list;
  struct mailimf_group * group;
  struct mailimap_address * imap_addr;
  char * group_name;
  clistiter * cur;
  struct mailimf_mailbox_list * mb_list;
  int r;
  int res;

  imap_mailbox_listiter = * iter;

  imap_addr = clist_content(imap_mailbox_listiter);
  ETPAN_ASSERT(imap_addr->ad_mailbox_name != NULL, "invalid imap address");

  group_name = strdup(imap_addr->ad_mailbox_name);
  if (group_name == NULL)
    ETPAN_LOG_MEMORY_ERROR;

  list = clist_new();
  if (list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  for(cur = clist_next(imap_mailbox_listiter) ; cur != NULL ;
      cur = clist_next(cur)) {
    struct mailimf_mailbox * mb;

    imap_addr = clist_content(cur);

    if (imap_addr->ad_mailbox_name == NULL) {
      break;
    }

    mb = imap_address_to_mailbox(imap_addr);
    r = clist_append(list, mb);
    if (r != 0)
      ETPAN_LOG_MEMORY_ERROR;
  }

  mb_list = mailimf_mailbox_list_new(list);
  if (mb_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  group = mailimf_group_new(group_name, mb_list);
  if (group == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  * iter = cur;

  return group;
}

static struct mailimf_address_list *
imap_mailbox_list_to_address_list(clist * imap_mailbox_list)
{
  clistiter * cur;
  clist * list;
  struct mailimf_address_list * addr_list;
  int r;
  int res;

  list = clist_new();
  if (list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  for(cur = clist_begin(imap_mailbox_list) ; cur != NULL ;
      cur = clist_next(cur)) {
    struct mailimap_address * imap_addr;
    struct mailimf_address * addr;

    imap_addr = clist_content(cur);

    if (imap_addr->ad_mailbox_name == NULL)
      continue;

    if ((imap_addr->ad_host_name == NULL) &&
        (imap_addr->ad_mailbox_name != NULL)) {
      struct mailimf_group * group;
      
      group = imap_mailbox_list_to_group(imap_mailbox_list, &cur);
      addr = mailimf_address_new(MAILIMF_ADDRESS_GROUP, NULL, group);
      if (addr == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else {
      addr = imap_address_to_address(imap_addr);
    }
    
    r = clist_append(list, addr);
    if (r != 0)
      ETPAN_LOG_MEMORY_ERROR;
  }

  addr_list = mailimf_address_list_new(list);
  if (addr_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return addr_list;
}


struct mailimf_fields *
etpan_imap_env_to_lep(struct mailimap_envelope * env,
    char * ref_str, size_t ref_size)
{
  clist * list;
  struct mailimf_field * field;
  int r;
  struct mailimf_fields * fields;
  int res;

  list = clist_new();
  if (list == NULL)
    ETPAN_LOG_MEMORY_ERROR;

  if (env->env_date != NULL) {
    size_t cur_token;
    struct mailimf_date_time * date_time;

    cur_token = 0;
    r = mailimf_date_time_parse(env->env_date, strlen(env->env_date),
        &cur_token, &date_time);

    if (r == MAILIMF_NO_ERROR) {
      struct mailimf_orig_date * orig;
      
      orig = mailimf_orig_date_new(date_time);
      if (orig == NULL)
        ETPAN_LOG_MEMORY_ERROR;

      field = mailimf_field_new(MAILIMF_FIELD_ORIG_DATE,
          NULL, NULL, NULL, NULL, NULL, NULL, NULL,
          NULL, orig, NULL,
          NULL, NULL, NULL, NULL, NULL, NULL, NULL,
          NULL, NULL, NULL, NULL, NULL);
      if (field == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      
      r = clist_append(list, field);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
    }
  }

  if (env->env_subject != NULL) {
    char * subject;
    struct mailimf_subject * subject_field;

    subject = strdup(env->env_subject);
    if (subject == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    
    subject_field = mailimf_subject_new(subject);
    if (subject_field == NULL)
      ETPAN_LOG_MEMORY_ERROR;

    field = mailimf_field_new(MAILIMF_FIELD_SUBJECT,
          NULL, NULL, NULL, NULL, NULL, NULL, NULL,
        NULL, NULL, NULL,
        NULL, NULL, NULL, NULL, NULL, NULL, NULL,
        NULL, subject_field, NULL, NULL, NULL);
    if (field == NULL)
      ETPAN_LOG_MEMORY_ERROR;

    r = clist_append(list, field);
    if (r != 0)
      ETPAN_LOG_MEMORY_ERROR;
  }

  if (env->env_from != NULL) {
    if (env->env_from->frm_list != NULL) {
      struct mailimf_mailbox_list * mb_list;
      struct mailimf_from * from;
      
      mb_list = imap_mailbox_list_to_mailbox_list(env->env_from->frm_list);
      from = mailimf_from_new(mb_list);
      if (from == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      
      field = mailimf_field_new(MAILIMF_FIELD_FROM,
          NULL, NULL, NULL, NULL, NULL, NULL, NULL,
          NULL, NULL, from,
          NULL, NULL, NULL, NULL, NULL, NULL, NULL,
          NULL, NULL, NULL, NULL, NULL);
      if (field == NULL)
        ETPAN_LOG_MEMORY_ERROR;

      r = clist_append(list, field);
      if (r != 0)
        ETPAN_LOG_MEMORY_ERROR;
    }
  }

  if (env->env_sender != NULL) {
    if (env->env_sender->snd_list != NULL) {
      struct mailimf_sender * sender;
      struct mailimf_mailbox * mb;
      
      mb = imap_address_to_mailbox(clist_begin(env->env_sender->snd_list)->data);
      sender = mailimf_sender_new(mb);
      if (sender == NULL)
        ETPAN_LOG_MEMORY_ERROR;

      field = mailimf_field_new(MAILIMF_FIELD_SENDER,
          NULL, NULL, NULL, NULL, NULL, NULL, NULL,
          NULL, NULL, NULL,
          sender, NULL, NULL, NULL, NULL, NULL, NULL,
          NULL, NULL, NULL, NULL, NULL);
      if (field == NULL)
        ETPAN_LOG_MEMORY_ERROR;

      r = clist_append(list, field);
      if (r != 0)
        ETPAN_LOG_MEMORY_ERROR;
    }
  }

  if (env->env_reply_to != NULL) {
    if (env->env_reply_to->rt_list != NULL) {
      struct mailimf_address_list * addr_list;
      struct mailimf_reply_to * reply_to;
      
      addr_list = imap_mailbox_list_to_address_list(env->env_reply_to->rt_list);
      reply_to = mailimf_reply_to_new(addr_list);
      if (reply_to == NULL)
        ETPAN_LOG_MEMORY_ERROR;

      field = mailimf_field_new(MAILIMF_FIELD_REPLY_TO,
          NULL, NULL, NULL, NULL, NULL, NULL, NULL,
          NULL, NULL, NULL,
          NULL, reply_to, NULL, NULL, NULL, NULL, NULL,
          NULL, NULL, NULL, NULL, NULL);
      if (field == NULL)
        ETPAN_LOG_MEMORY_ERROR;

      r = clist_append(list, field);
      if (r != 0)
        ETPAN_LOG_MEMORY_ERROR;
    }
  }

  if (env->env_to != NULL) {
    if (env->env_to->to_list != NULL) {
      struct mailimf_address_list * addr_list;
      struct mailimf_to * to;

      addr_list = imap_mailbox_list_to_address_list(env->env_to->to_list);
      to = mailimf_to_new(addr_list);
      if (to == NULL)
        ETPAN_LOG_MEMORY_ERROR;

      field = mailimf_field_new(MAILIMF_FIELD_TO,
          NULL, NULL, NULL, NULL, NULL, NULL, NULL,
          NULL, NULL, NULL,
          NULL, NULL, to, NULL, NULL, NULL, NULL,
          NULL, NULL, NULL, NULL, NULL);
      if (field == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      
      r = clist_append(list, field);
      if (r != 0)
        ETPAN_LOG_MEMORY_ERROR;
    }
  }

  if (env->env_cc != NULL) {
    if (env->env_cc->cc_list != NULL) {
      struct mailimf_address_list * addr_list;
      struct mailimf_cc * cc;

      addr_list = imap_mailbox_list_to_address_list(env->env_cc->cc_list);
      cc = mailimf_cc_new(addr_list);
      if (cc == NULL)
        ETPAN_LOG_MEMORY_ERROR;

      field = mailimf_field_new(MAILIMF_FIELD_CC,
          NULL, NULL, NULL, NULL, NULL, NULL, NULL,
          NULL, NULL, NULL,
          NULL, NULL, NULL, cc, NULL, NULL, NULL,
          NULL, NULL, NULL, NULL, NULL);
      if (field == NULL)
        ETPAN_LOG_MEMORY_ERROR;

      r = clist_append(list, field);
      if (r != 0)
        ETPAN_LOG_MEMORY_ERROR;
    }
  }

  if (env->env_bcc != NULL) {
    if (env->env_bcc->bcc_list != NULL) {
      struct mailimf_address_list * addr_list;
      struct mailimf_bcc * bcc;
      
      addr_list = imap_mailbox_list_to_address_list(env->env_bcc->bcc_list);
      bcc = mailimf_bcc_new(addr_list);
      if (bcc == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      
      field = mailimf_field_new(MAILIMF_FIELD_BCC,
          NULL, NULL, NULL, NULL, NULL, NULL, NULL,
          NULL, NULL, NULL,
          NULL, NULL, NULL, NULL, bcc, NULL, NULL,
          NULL, NULL, NULL, NULL, NULL);
      if (field == NULL)
        ETPAN_LOG_MEMORY_ERROR;

      r = clist_append(list, field);
      if (r != 0)
        ETPAN_LOG_MEMORY_ERROR;
    }
  }

  if (env->env_in_reply_to != NULL) {
    struct mailimf_in_reply_to * in_reply_to;
    size_t cur_token;
    clist * msg_id_list;
      
    cur_token = 0;
    r = mailimf_msg_id_list_parse(env->env_in_reply_to,
        strlen(env->env_in_reply_to), &cur_token, &msg_id_list);

    switch (r) {
    case MAILIMF_NO_ERROR:
      in_reply_to = mailimf_in_reply_to_new(msg_id_list);
      if (in_reply_to == NULL)
        ETPAN_LOG_MEMORY_ERROR;
	
      field = mailimf_field_new(MAILIMF_FIELD_IN_REPLY_TO,
          NULL, NULL, NULL, NULL, NULL, NULL, NULL,
          NULL, NULL, NULL,
          NULL, NULL, NULL, NULL, NULL, NULL,
          in_reply_to,
          NULL, NULL, NULL, NULL, NULL);
      if (field == NULL)
        ETPAN_LOG_MEMORY_ERROR;
	
      r = clist_append(list, field);
      if (r != 0)
        ETPAN_LOG_MEMORY_ERROR;
      break;

    case MAILIMF_ERROR_PARSE:
      break;

    default:
      ETPAN_LOG_MEMORY_ERROR;
      break;
    }
  }

  if (env->env_message_id != NULL) {
    char * id;
    struct mailimf_message_id * msg_id;
    size_t cur_token;

    cur_token = 0;
    r = mailimf_msg_id_parse(env->env_message_id, strlen(env->env_message_id),
        &cur_token, &id);
    switch (r) {
    case MAILIMF_NO_ERROR:

      msg_id = mailimf_message_id_new(id);
      if (msg_id == NULL)
        ETPAN_LOG_MEMORY_ERROR;

      field = mailimf_field_new(MAILIMF_FIELD_MESSAGE_ID,
          NULL, NULL, NULL, NULL, NULL, NULL, NULL,
          NULL, NULL, NULL,
          NULL, NULL, NULL, NULL, NULL, msg_id, NULL,
          NULL, NULL, NULL, NULL, NULL);
      if (field == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      
      r = clist_append(list, field);
      if (r != 0)
        ETPAN_LOG_MEMORY_ERROR;
      break;

    case MAILIMF_ERROR_PARSE:
      break;

    default:
      ETPAN_LOG_MEMORY_ERROR;
      break;
    }
  }

  if (ref_str != NULL) {
    struct mailimf_references * references;
    size_t cur_token;

    cur_token = 0;
    r = mailimf_references_parse(ref_str, ref_size,
        &cur_token, &references);
    switch (r) {
    case MAILIMF_NO_ERROR:
      field = mailimf_field_new(MAILIMF_FIELD_REFERENCES,
          NULL, NULL, NULL, NULL, NULL, NULL, NULL,
          NULL, NULL, NULL,
          NULL, NULL, NULL, NULL, NULL, NULL,
          NULL,
          references, NULL, NULL, NULL, NULL);
      if (field == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      
      r = clist_append(list, field);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
      break;

    case MAILIMF_ERROR_PARSE:
      break;

    default:
      ETPAN_LOG_MEMORY_ERROR;
      break;
    }
  }

  fields = mailimf_fields_new(list);
  if (fields == NULL)
    ETPAN_LOG_MEMORY_ERROR;

  return fields;
}
