#include "etpan-storage-imap.h"

#include <libetpan/libetpan.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/types.h>
#include <dirent.h>
#include <sys/stat.h>

#include "etpan-log.h"
#include "etpan-error.h"
#include "etpan-storage.h"
#include "etpan-storage-private.h"
#include "etpan-folder.h"
#include "etpan-folder-imap.h"
#include "etpan-lep.h"
#include "etpan-imap.h"
#include "etpan-storage-imap-private.h"
#include "etpan-utils.h"
#include "etpan-nls.h"

enum {
  ERROR_DOMAIN_CONNECTION,
  ERROR_DOMAIN_FETCH_FOLDER_LIST,
  ERROR_DOMAIN_CREATE,
  ERROR_DOMAIN_DELETE,
  ERROR_DOMAIN_RENAME,
};

static struct etpan_error * rewrite_error(struct etpan_storage * storage,
    int domain,
    struct etpan_error * error);

static struct etpan_error * setup(struct etpan_storage * storage);

static void free_data(struct etpan_storage * storage);

static struct etpan_error * create_folder(struct etpan_storage * storage,
    char * folder_name);

static struct etpan_error * delete_folder(struct etpan_storage * storage,
    char * folder_name);

static struct etpan_error * rename_folder(struct etpan_storage * storage,
    char * folder_name, char * new_name);

static struct etpan_error * fetch_folder_list(struct etpan_storage * storage,
    chash * folder_list);

static struct etpan_error * connect(struct etpan_storage * storage);

static void disconnect(struct etpan_storage * storage);

static char * get_sub_folder_location(struct etpan_storage * storage,
    struct etpan_folder * parent, char * name);

static int allows_sub_folders(struct etpan_storage * storage,
    struct etpan_folder * parent);

static struct etpan_storage_driver imap_driver = {
  .name = "imap",
  
  .network = 1,
  
  .setup = setup,
  .create_folder = create_folder,
  .delete_folder = delete_folder,
  .rename_folder = rename_folder,
  .fetch_folder_list = fetch_folder_list,
  .connect = connect,
  .disconnect = disconnect,
  .free_data = free_data,
  
  .get_sub_folder_location = get_sub_folder_location,
  .allows_sub_folders = allows_sub_folders,
};

struct storage_data {
  char * hostname;
  int port;
  int connection_type;
  int auth_type;
  char * username;
  char * password;
  char * command;
  char * base_location;
  
  /* threaded */
  struct mailstorage * storage;
  int connected;
  char * threaded_hostname;
  int threaded_port;
  int threaded_connection_type;
  int threaded_auth_type;
  char * threaded_username;
  char * threaded_password;
  char * threaded_command;
  char * threaded_base_location;
  char threaded_main_imap_separator;
};

static struct etpan_error * ep_storage_connect(struct etpan_storage * storage)
{
  struct storage_data * data;
  int r;
  struct etpan_error * error;
  
  data = etpan_storage_get_data(storage);

  r = mailstorage_connect(data->storage);
  if (r == MAIL_ERROR_MEMORY)
    ETPAN_LOG_MEMORY_ERROR;
    
  if (r == MAIL_ERROR_CONNECT) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CONNECT);
    etpan_error_set_short_description(error, "Could not connect");
    etpan_error_strf_long_description(error,
        _("Connection to account %s failed. "
            "The connection to IMAP server %s, port %i could not be established.\n"
            "Check the name of server and port."),
        etpan_storage_get_id(storage),
        etpan_storage_imap_get_hostname(storage),
        etpan_storage_imap_get_port(storage));
    goto err;
  }
  else if (r == MAIL_ERROR_LOGIN) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_AUTH);
    etpan_error_set_short_description(error, "Authentication failed");
    etpan_error_strf_long_description(error,
        _("Connection to account %s failed. "
            "Check your login, password and authentication type."),
        etpan_storage_get_id(storage));
    goto err;
  }
  else if (r == MAIL_ERROR_PARSE) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, "Parse error");
    etpan_error_strf_long_description(error, _("Connection to account %s failed. The application did not understand what the server sent."),
        etpan_storage_get_id(storage));
    goto err;
  }
  else if (r == MAIL_ERROR_STREAM) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_STREAM);
    etpan_error_set_short_description(error, "Connection closed");
    etpan_error_strf_long_description(error,
        _("Connection to account %s failed. "
            "The connection closed unexpectedly."),
        etpan_storage_get_id(storage));
    goto err;
  }
  else if (r != MAIL_NO_ERROR) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CONNECT);
    etpan_error_set_short_description(error, _("Unexpected error"));
    etpan_error_strf_long_description(error,
        _("Connection to account %s failed. "
            "An unexpected error (libetpan: %i) occurred."),
        etpan_storage_get_id(storage), r);
    goto err;
  }
  
  error = NULL;
  
 err:
  return error;
}

static struct etpan_error * connect(struct etpan_storage * storage)
{
  int r;
  struct storage_data * data;
  struct mailstorage * ep_storage;
  int ep_connection_type;
  int ep_auth_type;
  int cached;
  struct etpan_error * error;
  struct etpan_imap_mailbox_info * imap_folder;
  char imap_separator;
  char * location;
  carray * imap_mb_list;
  
  data = etpan_storage_get_data(storage);
  
  if (data->connected) {
    error = ep_storage_connect(storage);
    if (error != NULL) {
      goto err;
    }
    
    r = mailsession_noop(data->storage->sto_session);
    if (r == MAIL_ERROR_MEMORY)
      ETPAN_LOG_MEMORY_ERROR;
    
    if (r == MAIL_ERROR_PARSE) {
      error = etpan_error_new();
      etpan_error_set_code(error, ERROR_PARSE);
      etpan_error_set_short_description(error, "Parse error");
      etpan_error_strf_long_description(error, _("Connection to account %s failed. The application did not understand what the server sent."),
          etpan_storage_get_id(storage));
      goto err;
    }
    else if (r == MAIL_ERROR_STREAM) {
      error = etpan_error_new();
      etpan_error_set_code(error, ERROR_STREAM);
      etpan_error_set_short_description(error, "Connection closed");
      etpan_error_strf_long_description(error,
          _("Connection to account %s failed. "
              "The connection closed unexpectedly."),
          etpan_storage_get_id(storage));
      goto err;
    }
    else if (r != MAIL_NO_ERROR) {
      error = etpan_error_new();
      etpan_error_set_code(error, ERROR_CONNECT);
      etpan_error_set_short_description(error, _("Unexpected error"));
      etpan_error_strf_long_description(error,
          _("Connection to account %s failed. "
              "An unexpected error (libetpan: %i) occurred."),
          etpan_storage_get_id(storage), r);
      goto err;
    }
    
    return NULL;
  }
  
  if ((data->threaded_command == NULL) && (data->threaded_hostname == NULL)) {
    error = etpan_error_internal_strf(_("Account information of %s are not valid. Neither command nor hostname are defined."),
        etpan_storage_get_id(storage));
    
    goto err;
  }
  
  if (data->threaded_command == NULL) {
    switch (data->threaded_connection_type) {
    case ETPAN_CONNECTION_STARTTLS:
      ep_connection_type = CONNECTION_TYPE_STARTTLS;
      break;
    case ETPAN_CONNECTION_TRY_STARTTLS:
      ep_connection_type = CONNECTION_TYPE_TRY_STARTTLS;
      break;
    case ETPAN_CONNECTION_TLS:
      ep_connection_type = CONNECTION_TYPE_TLS;
      break;
    default:
      ep_connection_type = CONNECTION_TYPE_PLAIN;
      break;
    }
  }
  else {
    switch (data->threaded_connection_type) {
    case ETPAN_CONNECTION_STARTTLS:
      ep_connection_type = CONNECTION_TYPE_COMMAND_STARTTLS;
      break;
    case ETPAN_CONNECTION_TRY_STARTTLS:
      ep_connection_type = CONNECTION_TYPE_COMMAND_TRY_STARTTLS;
      break;
    case ETPAN_CONNECTION_TLS:
      ep_connection_type = CONNECTION_TYPE_COMMAND_TLS;
      break;
    default:
      ep_connection_type = CONNECTION_TYPE_COMMAND;
      break;
    }
  }
  
  ep_auth_type = IMAP_AUTH_TYPE_PLAIN;
  
  ep_storage = mailstorage_new(NULL);
  if (ep_storage == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  cached = 1;
  if (etpan_storage_get_threaded_cache_path(storage) == NULL)
    cached = 0;
  
  r = imap_mailstorage_init(ep_storage,
    data->threaded_hostname, data->threaded_port,
    data->threaded_command,
    ep_connection_type, ep_auth_type,
    data->threaded_username, data->threaded_password,
    cached, etpan_storage_get_threaded_cache_path(storage));
  if (r != MAIL_NO_ERROR) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  data->storage = ep_storage;
  error = ep_storage_connect(storage);
  if (error != NULL) {
    goto free_storage;
  }
  
  /* get separator */
  location = data->threaded_base_location;
  if (location == NULL)
    location = "";
    
  error = etpan_imap_list_mailboxes(ep_storage->sto_session,
      location, &imap_mb_list);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(storage, ERROR_DOMAIN_CONNECTION, error);
    ETPAN_ERROR_FREE(error);
    
    error = new_error;
    goto disconnect;
  }
  
  if (carray_count(imap_mb_list) == 0) {
    etpan_imap_list_mailboxes_free(imap_mb_list);
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FOLDER_NOT_FOUND);
    etpan_error_set_short_description(error, _("Mailbox not found"));
    
    etpan_error_strf_long_description(error,
        _("Mailbox (%s) given in mailbox prefix does not exist or is invalid."),
        location);
    
    goto disconnect;
  }
  
  imap_folder = carray_get(imap_mb_list, 0);
  imap_separator = imap_folder->delimiter;
  
  data->threaded_main_imap_separator = imap_separator;
  
  etpan_imap_list_mailboxes_free(imap_mb_list);
  
  data->connected = 1;
  
  return NULL;
  
 disconnect:
  mailstorage_disconnect(ep_storage);
 free_storage:
  mailstorage_free(ep_storage);
  data->storage = NULL;
 err:
  return error;
}

static void disconnect(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  
  if (!data->connected)
    return;
  
  data->connected = 0;
  
  mailstorage_disconnect(data->storage);
  mailstorage_free(data->storage);
  data->storage = NULL;
}

struct etpan_storage * etpan_storage_imap_new(void)
{
  struct storage_data * data;
  struct etpan_storage * storage;
  
  storage = etpan_storage_new();
  data = malloc(sizeof(* data));
  if (data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  data->hostname = NULL;
  data->port = 0;
  data->connection_type = ETPAN_CONNECTION_PLAIN;
  data->auth_type = ETPAN_AUTH_PASSWORD;
  data->username = NULL;
  data->password = NULL;
  data->command = NULL;
  data->base_location = NULL;
  data->storage = NULL;
  data->connected = 0;
  data->threaded_hostname = NULL;
  data->threaded_port = 0;
  data->threaded_connection_type = ETPAN_CONNECTION_PLAIN;
  data->threaded_auth_type = ETPAN_AUTH_PASSWORD;
  data->threaded_username = NULL;
  data->threaded_password = NULL;
  data->threaded_command = NULL;
  data->threaded_base_location = NULL;
  data->threaded_main_imap_separator = 0;
  
  etpan_storage_set_data(storage, data);
  etpan_storage_set_driver(storage, &imap_driver);
  
  return storage;
}

static struct etpan_error * get_imap_list(struct etpan_storage * storage,
    char * base_vpath, char * pathname, carray ** result);
static void free_array(carray * list);

struct path_info {
  char * mb;
  char * vpath;
  int noinferiors;
  char imap_separator;
};

static struct etpan_folder * folder_from_info(struct etpan_storage * storage,
    struct path_info * info)
{
  struct etpan_folder * folder;
  
  /* create the folder */
  
  folder = etpan_folder_imap_new();
  
  etpan_folder_set_ui_path(folder, info->vpath);
  etpan_folder_set_location(folder, info->mb);
  etpan_folder_imap_set_noinferiors(folder, info->noinferiors);
  etpan_folder_imap_set_separator(folder, info->imap_separator);
  
  etpan_folder_set_storage(folder, storage);
  
  return folder;
}

static struct etpan_error * fetch_folder_list(struct etpan_storage * storage,
    chash * folder_list)
{
  carray * path_list;
  int r;
  unsigned int i;
  struct storage_data * data;
  struct etpan_error * error;
  
  error = connect(storage);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(storage, ERROR_DOMAIN_FETCH_FOLDER_LIST, error);
    ETPAN_ERROR_FREE(error);
    
    error = new_error;
    return error;
  }
  
  data = etpan_storage_get_data(storage);
  
  error = get_imap_list(storage, storage->id,
      data->threaded_base_location, &path_list);
  ETPAN_ERROR_IGNORE(error);
  
  for(i = 0 ; i < carray_count(path_list) ; i ++) {
    struct path_info * info;
    struct etpan_folder * folder;
    chashdatum key;
    chashdatum value;
    
    info = carray_get(path_list, i);
    folder = folder_from_info(storage, info);
    
    /* there can be double, they must be removed */
    
    key.data = folder->location;
    key.len = strlen(folder->location) + 1;
    
    r = chash_get(folder_list, &key, &value);
    if (r == 0) {
      etpan_folder_unref(folder);
      continue;
    }
    
    value.data = folder;
    value.len = 0;
    r = chash_set(folder_list, &key, &value, NULL);
    if (r < 0) {
      etpan_folder_unref(folder);
      ETPAN_LOG_MEMORY_ERROR;
    }
  }
  
  free_array(path_list);
  
  return NULL;
}

static void free_data(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  
  free(data->threaded_base_location);
  free(data->threaded_command);
  free(data->threaded_password);
  free(data->threaded_username);
  free(data->threaded_hostname);
  
  free(data->command);
  free(data->password);
  free(data->username);
  free(data->base_location);
  free(data->hostname);
  free(data);
}

static struct etpan_error * create_folder(struct etpan_storage * storage,
    char * folder_name)
{
  struct mailstorage * ep_storage;
  struct storage_data * data;
  struct etpan_error * error;
  
  data = etpan_storage_get_data(storage);
  ep_storage = data->storage;
  
  error = etpan_imap_create(ep_storage->sto_session, folder_name);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(storage, ERROR_DOMAIN_CREATE, error);
    ETPAN_ERROR_FREE(error);
    
    error = new_error;
  }
  
  return error;
}

static struct etpan_error * delete_folder(struct etpan_storage * storage,
    char * folder_name)
{
  struct mailstorage * ep_storage;
  struct storage_data * data;
  struct etpan_error * error;
  
  data = etpan_storage_get_data(storage);
  ep_storage = data->storage;
  
  error = etpan_imap_delete(ep_storage->sto_session, folder_name);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(storage, ERROR_DOMAIN_DELETE, error);
    ETPAN_ERROR_FREE(error);
    
    error = new_error;
  }
  
  return error;
}



static struct path_info * path_info_new(char * mb, char * vpath,
    int noinferiors, int imap_separator)
{
  struct path_info * info;
  
  info = malloc(sizeof(* info));
  if (info == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  info->mb = mb;
  info->vpath = vpath;
  info->noinferiors = noinferiors;
  info->imap_separator = imap_separator;
  
  return info;
}

static void add_to_array(carray * list, char * mb, char * vpath,
    int noinferiors, char imap_separator)
{
  struct path_info * info;
  int r;
  
  info = path_info_new(mb, vpath, noinferiors, imap_separator);
  if (info == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  r = carray_add(list, info, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

static void free_array(carray * list)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(list) ; i ++) {
    struct path_info * info;
    
    info = carray_get(list, i);
    free(info->vpath);
    free(info->mb);
    free(info);
  }
  carray_free(list);
}

static int compare_path(const void * elt1, const void * elt2)
{
  struct path_info * const * ppath1;
  struct path_info * const * ppath2;
  struct path_info * path1;
  struct path_info * path2;
  
  ppath1 = elt1;
  ppath2 = elt2;
  path1 = * ppath1;
  path2 = * ppath2;
  
  return strcmp(path1->mb, path2->mb);
}

/* imap */

static void imap_add_to_list(struct etpan_storage * storage,
    carray * list, char * base_vpath,
    char * dir, carray * error_list)
{
  carray * imap_mb_list;
  int r;
  unsigned int i;
  char pattern[PATH_MAX];
  char tmp_vpath[PATH_MAX];
  struct storage_data * data;
  struct mailstorage * ep_storage;
  struct etpan_error * error;
  
  data = etpan_storage_get_data(storage);
  ep_storage = data->storage;
  
  snprintf(pattern, sizeof(pattern), "*");
  
  if (dir != NULL) {
    if (* dir != '\0') {
      struct etpan_imap_mailbox_info * imap_folder;
      char imap_separator;
      
      error = etpan_imap_list_mailboxes(ep_storage->sto_session,
          dir, &imap_mb_list);
      if (error != NULL) {
        struct etpan_error * new_error;
        
        new_error = rewrite_error(storage,
            ERROR_DOMAIN_FETCH_FOLDER_LIST, error);
        ETPAN_ERROR_FREE(error);
        
        r = carray_add(error_list, error, NULL);
        if (r < 0)
          ETPAN_LOG_MEMORY_ERROR;
        
        goto err;
      }
      
      if (carray_count(imap_mb_list) == 0) {
        etpan_imap_list_mailboxes_free(imap_mb_list);
        
        error = etpan_error_new();
        etpan_error_set_code(error, ERROR_FOLDER_NOT_FOUND);
        etpan_error_set_short_description(error, _("Mailbox not found"));
        
        etpan_error_strf_long_description(error,
            _("Mailbox %s (%s) was not found while getting the list of mailboxes of account %s."),
            dir, base_vpath, etpan_storage_get_id(storage));
        r = carray_add(error_list, error, NULL);
        if (r < 0)
          ETPAN_LOG_MEMORY_ERROR;
        
        goto err;
      }
      
      imap_folder = carray_get(imap_mb_list, 0);
      imap_separator = imap_folder->delimiter;
      
      etpan_imap_list_mailboxes_free(imap_mb_list);
      
      snprintf(pattern, sizeof(pattern), "%s%c*", dir, imap_separator);
    }
  }
  
  
  error = etpan_imap_lsub_mailboxes(ep_storage->sto_session,
      pattern, &imap_mb_list);
  if (error != NULL) {
    struct etpan_error * new_error;
        
    new_error = rewrite_error(storage,
        ERROR_DOMAIN_FETCH_FOLDER_LIST, error);
    ETPAN_ERROR_FREE(error);
        
    r = carray_add(error_list, new_error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    goto err;
  }
  
  for(i = 0 ; i < carray_count(imap_mb_list) ; i ++) {
    struct etpan_imap_mailbox_info * info;
    char mb[PATH_MAX];
    char * vpath;
    char * name;
    int noinferiors;
    char imap_separator;
    carray * presence_mb_list;
    
    info = carray_get(imap_mb_list, i);
    
    if ((info->flags & ETPAN_IMAP_MB_NOSELECT) != 0)
      continue;
    
    error = etpan_imap_list_mailboxes(ep_storage->sto_session,
        info->mb, &presence_mb_list);
    if (error != NULL) {
      struct etpan_error * new_error;
      
      new_error = rewrite_error(storage,
          ERROR_DOMAIN_FETCH_FOLDER_LIST, error);
      ETPAN_ERROR_FREE(error);
      
      r = carray_add(error_list, new_error, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
      
      goto free_list;
    }
    
    if (carray_count(presence_mb_list) == 0) {
      /* fix IMAP subscription list */
      error = etpan_imap_unsubscribe(ep_storage->sto_session, info->mb);
      ETPAN_ERROR_IGNORE(error);
      etpan_imap_list_mailboxes_free(presence_mb_list);
      continue;
    }
    
    etpan_imap_list_mailboxes_free(presence_mb_list);
    
    snprintf(mb, sizeof(mb), "%s", info->mb);
    
    if (info->delimiter != '/') {
      char * p;
      
      p = mb;
      while ((p = strchr(p, '/')) != NULL) {
        * p = '-';
        p ++;
      }
      
      p = mb;
      while ((p = strchr(p, info->delimiter)) != NULL) {
        * p = '/';
        p ++;
      }
    }
    
    name = strdup(info->mb);
    if (name == NULL) {
      ETPAN_LOG_MEMORY_ERROR;
    }
    
    if (* base_vpath == '\0')
      snprintf(tmp_vpath, sizeof(tmp_vpath), "%s", mb);
    else
      snprintf(tmp_vpath, sizeof(tmp_vpath), "%s/%s", base_vpath, mb);
    
    vpath = strdup(tmp_vpath);
    if (vpath == NULL) {
      free(name);
      ETPAN_LOG_MEMORY_ERROR;
    }
    
    noinferiors = (info->flags & ETPAN_IMAP_MB_NOINFERIORS) != 0;
    imap_separator = info->delimiter;
    add_to_array(list, name, vpath, noinferiors, imap_separator);
  }
  
  etpan_imap_list_mailboxes_free(imap_mb_list);
  
  return;
  
 free_list:
  etpan_imap_list_mailboxes_free(imap_mb_list);
 err:
  return;
}

static struct etpan_error * get_imap_list(struct etpan_storage * storage,
    char * base_vpath, char * pathname, carray ** result)
{
  int r;
  carray * mb_list;
  struct etpan_error * error;
  carray * error_list;
  carray * imap_mb_list;
  struct mailstorage * ep_storage;
  struct storage_data * data;
  unsigned int i;
  
  mb_list = carray_new(8);
  if (mb_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  error_list = carray_new(4);
  if (error_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  imap_add_to_list(storage, mb_list, base_vpath, pathname, error_list);
  
  /* INBOX */
  data = etpan_storage_get_data(storage);
  ep_storage = data->storage;
  error = etpan_imap_list_mailboxes(ep_storage->sto_session,
      "INBOX", &imap_mb_list);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(storage,
        ERROR_DOMAIN_FETCH_FOLDER_LIST, error);
    ETPAN_ERROR_FREE(error);
    
    r = carray_add(error_list, new_error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  else {
    if (carray_count(imap_mb_list) > 0) {
      struct etpan_imap_mailbox_info * imap_folder;
      char * vpath;
      char * name;
      int noinferiors;
      char imap_separator;
      char tmp_vpath[PATH_MAX];
      
      imap_folder = carray_get(imap_mb_list, 0);
    
      name = strdup("INBOX");
      if (name == NULL) {
        etpan_imap_list_mailboxes_free(imap_mb_list);
        ETPAN_LOG_MEMORY_ERROR;
      }
    
      if (* base_vpath == '\0')
        snprintf(tmp_vpath, sizeof(tmp_vpath), "INBOX");
      else
        snprintf(tmp_vpath, sizeof(tmp_vpath), "%s/INBOX", base_vpath);
    
      vpath = strdup(tmp_vpath);
      if (vpath == NULL) {
        free(name);
        etpan_imap_list_mailboxes_free(imap_mb_list);
        ETPAN_LOG_MEMORY_ERROR;
      }
    
      noinferiors = (imap_folder->flags & ETPAN_IMAP_MB_NOINFERIORS) != 0;
      imap_separator = imap_folder->delimiter;
      add_to_array(mb_list, name, vpath, noinferiors, imap_separator);
    }
    etpan_imap_list_mailboxes_free(imap_mb_list);
  }
  
  qsort(carray_data(mb_list),
      carray_count(mb_list),
      sizeof(void *),
      compare_path);
  
  * result = mb_list;
  
  if (carray_count(error_list) == 0) {
    error = NULL;
  }
  else if (carray_count(error_list) == 1) {
    error = carray_get(error_list, 0);
  }
  else {
    error = etpan_error_multiple();
    for(i = 0 ; i < carray_count(error_list) ; i ++) {
      struct etpan_error * suberror;
      
      suberror = carray_get(error_list, i);
      etpan_error_add_child(error, suberror);
    }
  }
  carray_free(error_list);
  
  return error;
}


void etpan_storage_imap_set_hostname(struct etpan_storage * storage,
    char * hostname)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  if (hostname != data->hostname) {
    free(data->hostname);
    if (hostname != NULL) {
      data->hostname = strdup(hostname);
      if (data->hostname == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      data->hostname = NULL;
  }
}

char * etpan_storage_imap_get_hostname(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  return data->hostname;
}

void etpan_storage_imap_set_port(struct etpan_storage * storage, int port)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  data->port = port;
}

int etpan_storage_imap_get_port(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  return data->port;
}

void etpan_storage_imap_set_connection_type(struct etpan_storage * storage,
    int connection_type)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  data->connection_type = connection_type;
}

int etpan_storage_imap_get_connection_type(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  return data->connection_type;
}

void etpan_storage_imap_set_auth_type(struct etpan_storage * storage,
    int auth_type)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  data->auth_type = auth_type;
}

int etpan_storage_imap_get_auth_type(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  return data->auth_type;
}

void etpan_storage_imap_set_username(struct etpan_storage * storage,
    char * username)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  if (username != data->username) {
    free(data->username);
    if (username != NULL) {
      data->username = strdup(username);
      if (data->username == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      data->username = NULL;
  }
}

char * etpan_storage_imap_get_username(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  return data->username;
}

void etpan_storage_imap_set_password(struct etpan_storage * storage,
    char * password)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  if (password != data->password) {
    free(data->password);
    if (password != NULL) {
      data->password = strdup(password);
      if (data->password == NULL)
         ETPAN_LOG_MEMORY_ERROR;
    }
    else
      data->password = NULL;
  }
}

char * etpan_storage_imap_get_password(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  return data->password;
}

void etpan_storage_imap_set_base_location(struct etpan_storage * storage,
    char * base_location)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  if (base_location != data->base_location) {
    free(data->base_location);
    if (base_location != NULL) {
      data->base_location = strdup(base_location);
      if (data->base_location == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      data->base_location = NULL;
  }
}

char * etpan_storage_imap_get_base_location(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  return data->base_location;
}

void etpan_storage_imap_set_command(struct etpan_storage * storage,
    char * command)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  if (command != data->command) {
    free(data->command);
    if (command != NULL) {
      data->command = strdup(command);
      if (data->command == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      data->command = NULL;
  }
}

char * etpan_storage_imap_get_command(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  return data->command;
}

static struct etpan_error * setup(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  
  free(data->threaded_hostname);
  if (data->hostname != NULL) {
    data->threaded_hostname = strdup(data->hostname);
    if (data->threaded_hostname == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  else {
    data->threaded_hostname = NULL;
  }
  
  data->threaded_port = data->port;
  data->threaded_connection_type = data->connection_type;
  data->threaded_auth_type = data->auth_type;
  
  free(data->threaded_username);
  if (data->username != NULL) {
    data->threaded_username = strdup(data->username);
    if (data->threaded_username == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  else {
    data->threaded_username = NULL;
  }
  
  free(data->threaded_password);
  if (data->password != NULL) {
    data->threaded_password = strdup(data->password);
    if (data->threaded_password == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  else {
    data->threaded_password = NULL;
  }
  
  free(data->threaded_command);
  if (data->command != NULL) {
    data->threaded_command = strdup(data->command);
    if (data->threaded_command == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  else {
    data->threaded_command = NULL;
  }
  
  free(data->threaded_base_location);
  if (data->base_location != NULL) {
    data->threaded_base_location = strdup(data->base_location);
    if (data->threaded_base_location == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  } 
  else {
    data->threaded_base_location = NULL;
  }
  
  data->threaded_main_imap_separator = 0;
  
  return NULL;
}

struct mailstorage *
etpan_storage_imap_get_ep_storage(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  return data->storage;
}

static char * get_sub_folder_location(struct etpan_storage * storage,
    struct etpan_folder * parent, char * name)
{
  /* because of (1), this function has to be called
     after a fetch of folder list */
  
  if (parent == NULL) {
    char * base_location;
    
    base_location = etpan_storage_imap_get_base_location(storage);
    if (base_location == NULL) {
      char * result;
      
      result = strdup(name);
      if (result == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      
      return result;
    }
    else {
      struct storage_data * data;
      char * location;
      size_t size;
      char separator;
      
      data = etpan_storage_get_data(storage);
      
      if (data->threaded_main_imap_separator == 0) {
        ETPAN_LOG("IMAP separator not fetched yet");
        etpan_crash();
      }
      
      size = strlen(base_location) + strlen(name) + 2;
      location = malloc(size);
      if (location == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      
      /* TODO - should have a separator computed in main thread for
         correctness (1) */
      separator = data->threaded_main_imap_separator;
      snprintf(location, size, "%s%c%s", base_location, separator, name);
      
      return location;
    }
  }
  else {
    char * parent_location;
    char * location;
    size_t size;
    char separator;
    
    parent_location = etpan_folder_get_location(parent);
    size = strlen(parent_location) + strlen(name) + 2;
    location = malloc(size);
    if (location == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    
    separator = etpan_folder_imap_get_separator(parent);
    snprintf(location, size, "%s%c%s", parent_location, separator, name);
    
    return location;
  }
}

static int allows_sub_folders(struct etpan_storage * storage,
    struct etpan_folder * parent)
{
  (void) storage;
  
  if (parent == NULL)
    return 1;
  else 
    return (!etpan_folder_imap_get_noinferiors(parent));
}

static struct etpan_error * rename_folder(struct etpan_storage * storage,
    char * folder_name, char * new_name)
{
  struct mailstorage * ep_storage;
  struct storage_data * data;
  struct etpan_imap_mailbox_info * imap_folder;
  char imap_separator;
  carray * imap_mb_list;
  unsigned int i;
  size_t len;
  char pattern[PATH_MAX];
  struct etpan_error * error;
  
  data = etpan_storage_get_data(storage);
  ep_storage = data->storage;
  
  error = etpan_imap_list_mailboxes(ep_storage->sto_session,
      folder_name, &imap_mb_list);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(storage, ERROR_DOMAIN_RENAME, error);
    ETPAN_ERROR_FREE(error);
    
    error = new_error;
    
    goto err;
  }

  if (carray_count(imap_mb_list) == 0) {
    struct etpan_error * new_error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FOLDER_NOT_FOUND);
    etpan_error_set_short_description(error, _("Mailbox not found"));
    etpan_error_strf_long_description(error,
        _("Mailbox %s was not found while changing its name."),
        folder_name);
    
    new_error = rewrite_error(storage, ERROR_DOMAIN_RENAME, error);
    ETPAN_ERROR_FREE(error);
    
    error = new_error;
    
    goto free_mb_list;
  }
  
  imap_folder = carray_get(imap_mb_list, 0);
  imap_separator = imap_folder->delimiter;
  
  etpan_imap_list_mailboxes_free(imap_mb_list);
  
  snprintf(pattern, sizeof(pattern), "%s%c*", folder_name, imap_separator);
  error = etpan_imap_lsub_mailboxes(ep_storage->sto_session,
      pattern, &imap_mb_list);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(storage, ERROR_DOMAIN_RENAME, error);
    ETPAN_ERROR_FREE(error);
    
    error = new_error;
    
    goto err;
  }
  
  error = etpan_imap_rename(ep_storage->sto_session, folder_name, new_name);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(storage, ERROR_DOMAIN_RENAME, error);
    ETPAN_ERROR_FREE(error);
    
    error = new_error;
    
    goto free_mb_list;
  }
  
  len = strlen(folder_name);
  for(i = 0 ; i < carray_count(imap_mb_list) ; i ++) {
    struct etpan_imap_mailbox_info * subfolder;
    char * postfix;
    char * new_sublocation;
    size_t len_new_sublocation;
    
    subfolder = carray_get(imap_mb_list, i);
    postfix = subfolder->mb + len;
    len_new_sublocation = strlen(postfix) + strlen(new_name) + 1;
    new_sublocation = malloc(len_new_sublocation);
    if (new_sublocation == NULL) {
      ETPAN_LOG_MEMORY_ERROR;
    }
    
    snprintf(new_sublocation, len_new_sublocation, "%s%s", new_name, postfix);
    error = etpan_imap_unsubscribe(ep_storage->sto_session, subfolder->mb);
    ETPAN_ERROR_IGNORE(error);
    error = etpan_imap_subscribe(ep_storage->sto_session, new_sublocation);
    ETPAN_ERROR_IGNORE(error);
  }
  
  etpan_imap_list_mailboxes_free(imap_mb_list);
  
  return NULL;
  
 free_mb_list:
  etpan_imap_list_mailboxes_free(imap_mb_list);
 err:
  return error;
}


static struct etpan_error * rewrite_error(struct etpan_storage * storage,
    int domain,
    struct etpan_error * error)
{
  struct etpan_error * new_error;
  char * previous_long_description;
  
  new_error = etpan_error_new();
  etpan_error_set_code(new_error, etpan_error_get_code(error));
  etpan_error_set_short_description(new_error,
      etpan_error_get_short_description(error));
  previous_long_description = etpan_error_get_long_description(error);
  
  switch (domain) {
  case ERROR_DOMAIN_CONNECTION:
    etpan_error_strf_long_description(new_error,
        _("Connection to account %s failed. "
            "%s"),
        etpan_storage_get_id(storage),
        previous_long_description);
    break;
    
  case ERROR_DOMAIN_FETCH_FOLDER_LIST:
    etpan_error_strf_long_description(new_error,
        _("An error occurred while getting list of mailboxes on account %s.\n"
            "%s"),
        etpan_storage_get_id(storage),
        previous_long_description);
    break;

  case ERROR_DOMAIN_CREATE:
    etpan_error_strf_long_description(new_error,
        _("An error occurred while creating a mailbox on account %s.\n"
            "%s"),
        etpan_storage_get_id(storage),
        previous_long_description);
    break;
    
  case ERROR_DOMAIN_DELETE:
    etpan_error_strf_long_description(new_error,
        _("An error occurred while deleting a mailbox on account %s.\n"
            "%s"),
        etpan_storage_get_id(storage),
        previous_long_description);
    break;
    
  case ERROR_DOMAIN_RENAME:
    etpan_error_strf_long_description(new_error,
        _("An error occurred while changing the name of a mailbox on account %s.\n"
            "%s"),
        etpan_storage_get_id(storage),
        previous_long_description);
    break;
    
  default:
    etpan_error_strf_long_description(new_error,
        _("An error occurred with account %s.\n"
            "%s"),
        etpan_storage_get_id(storage),
        previous_long_description);
    break;
  }
  
  return new_error;
}
