#include "etpan-folder-maildir.h"

#include <stdlib.h>
#include <string.h>

#define ETPAN_MODULE_LOG_NAME "MAILDIR"

#include "etpan-folder.h"
#include "etpan-storage.h"
#include "etpan-storage-private.h"
#include "etpan-storage-maildir.h"
#include "etpan-message-lep.h"
#include "etpan-error.h"
#include "etpan-lep.h"
#include "etpan-log.h"
#include "etpan-nls.h"

enum {
  ERROR_DOMAIN_FETCH,
  ERROR_DOMAIN_APPEND,
  ERROR_DOMAIN_CHECK,
  ERROR_DOMAIN_STATUS,
  ERROR_DOMAIN_EXPUNGE,
};

static struct etpan_error * rewrite_error(struct etpan_folder * folder,
    int domain,
    struct etpan_error * error);

static struct etpan_error * fetch_msg_list(struct etpan_folder * folder,
    chash * current_msg_list);

static struct etpan_error * append_msg(struct etpan_folder * folder,
    char * message, size_t size,
    struct etpan_message_flags * flags);

static struct etpan_error * check(struct etpan_folder * folder);

static struct etpan_error * status(struct etpan_folder * folder,
    unsigned int * p_count, unsigned int * p_unseen, unsigned int * p_recent);

static struct etpan_error * expunge(struct etpan_folder * folder);

static void free_data(struct etpan_folder * folder);

static struct etpan_error * connect(struct etpan_folder * folder);

static void disconnect(struct etpan_folder * folder);

static int is_sub_folder(struct etpan_folder * parent,
    struct etpan_folder * child);

static struct etpan_folder_driver maildir_driver = {
  .name = "maildir",
  
  .setup = NULL,
  .fetch_msg_list = fetch_msg_list,
  .append_msg = append_msg,
  .check = check,
  .status = status,
  .expunge = expunge,
  .free_data = free_data,
  .connect = connect,
  .disconnect = disconnect,
  .is_sub_folder = is_sub_folder,
};

struct folder_data {
  /* modified by thread */
  int connected;
  struct mailfolder * folder;
  struct mailstorage * storage;
};

static struct mailfolder * get_mailfolder(struct etpan_folder * folder)
{
  struct folder_data * data;

  data = etpan_folder_get_data(folder);
  return data->folder;
}

static struct mailstorage * get_mailstorage(struct etpan_folder * folder)
{
  struct folder_data * data;

  data = etpan_folder_get_data(folder);
  return data->storage;
}

static struct etpan_error * connect(struct etpan_folder * folder)
{
  int r;
  struct folder_data * data;
  struct mailstorage * ep_storage;
  struct mailfolder * ep_folder;
  char * path;
  char * mb_cache_directory;
  char * mb_flags_directory;
  int cached;
  char * base_path;
  struct etpan_storage * storage;
  char tmp_path[PATH_MAX];
  struct etpan_error * error;
  
  data = etpan_folder_get_data(folder);
  
  if (data->connected)
    return NULL;
  
  ETPAN_LOCAL_LOG("connecting to maildir folder %s", etpan_folder_get_ui_path(folder));
  
  storage = etpan_folder_get_storage(folder);
  base_path = etpan_storage_maildir_get_path(storage);
  path = etpan_folder_get_threaded_location(folder);
  snprintf(tmp_path, sizeof(tmp_path), "%s/.%s", base_path, path);
  
  ep_storage = mailstorage_new(NULL);
  if (ep_storage == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  mb_cache_directory =
    etpan_storage_get_threaded_cache_path(folder->storage);
  mb_flags_directory =
    etpan_storage_get_threaded_flags_path(folder->storage);
   
  cached = 1;
  if ((mb_cache_directory == NULL) || (mb_flags_directory == NULL))
    cached = 0;

  ETPAN_LOCAL_LOG("cached maildir %i", cached);
  
  r = maildir_mailstorage_init(ep_storage, tmp_path, cached,
      mb_cache_directory, mb_flags_directory);
  if (r != MAIL_NO_ERROR) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  ep_folder = mailfolder_new(ep_storage, tmp_path, NULL);
  if (ep_folder == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }

  r = mailstorage_connect(ep_storage);
  if (r == MAIL_ERROR_MEMORY)
    ETPAN_LOG_MEMORY_ERROR;
  
  if (r != MAIL_NO_ERROR) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FILE);
    etpan_error_set_short_description(error, _("Could not open the mailbox"));
    etpan_error_strf_long_description(error, _("An error occurred while opening the mailbox %s. Check whether the file exists (%s) or the permissions."), path, tmp_path);
    goto free_folder;
  }

  ETPAN_LOCAL_LOG("storage connected");
  
  r = mailfolder_connect(ep_folder);
  if (r == MAIL_ERROR_MEMORY)
    ETPAN_LOG_MEMORY_ERROR;
  
  if (r != MAIL_NO_ERROR) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FILE);
    etpan_error_set_short_description(error, _("Could not open the mailbox"));
    etpan_error_strf_long_description(error, _("An error occurred while opening the mailbox %s. Check whether the file exists (%s) or the permissions."), path, tmp_path);
    goto disconnect_storage;
  }
  
  ETPAN_LOCAL_LOG("folder connected");
  
  data->storage = ep_storage;
  data->folder = ep_folder;
  data->connected = 1;
  
  etpan_set_lep_folder(folder, ep_folder);
  
  return NULL;
  
 disconnect_storage:
  mailstorage_disconnect(ep_storage);
 free_folder:
  mailfolder_free(ep_folder);
  mailstorage_free(ep_storage);
  return error;
}


static void disconnect(struct etpan_folder * folder)
{
  struct folder_data * data;
  
  data = etpan_folder_get_data(folder);
  
  if (!data->connected)
    return;
  
  mailfolder_disconnect(data->folder);
  mailstorage_disconnect(data->storage);
  mailfolder_free(data->folder);
  mailstorage_free(data->storage);
  
  data->connected = 0;
  data->folder = NULL;
  data->storage = NULL;
}

struct etpan_folder * etpan_folder_maildir_new(void)
{
  struct etpan_folder * folder;
  struct folder_data * data;
  
  folder = etpan_folder_new();
  data = malloc(sizeof(* data));
  if (data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  data->folder = NULL;
  data->storage = NULL;
  data->connected = 0;
  
  etpan_folder_set_data(folder, data);
  etpan_folder_set_driver(folder, &maildir_driver);
  
  return folder;
}

static void free_data(struct etpan_folder * folder)
{
  struct folder_data * data;
  
  disconnect(folder);
  
  data = etpan_folder_get_data(folder);
  
  free(data);
}

static struct etpan_error * fetch_msg_list(struct etpan_folder * folder,
    chash * current_msg_list)
{
  struct mailfolder * ep_folder;
  struct etpan_error * error;
  
  error = connect(folder);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(folder, ERROR_DOMAIN_FETCH, error);
    ETPAN_ERROR_FREE(error);
    
    return new_error;
  }
  
  ep_folder = get_mailfolder(folder);
  
  return etpan_lep_fetch_msg_list(folder, ep_folder, current_msg_list);
}

static struct etpan_error * append_msg(struct etpan_folder * folder,
    char * message, size_t size,
    struct etpan_message_flags * flags)
{
  struct mailfolder * ep_folder;
  struct mail_flags * ep_flags;
  int r;
  struct etpan_error * error;
  
  error = connect(folder);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(folder, ERROR_DOMAIN_APPEND, error);
    ETPAN_ERROR_FREE(error);
    
    return new_error;
  }
  
  ep_folder = get_mailfolder(folder);
  
  ep_flags = etpan_lep_flags_to_lep(flags);
  r = mailfolder_append_message_flags(ep_folder, message, size, ep_flags);
  mail_flags_free(ep_flags);
  if (r == MAIL_ERROR_MEMORY)
    ETPAN_LOG_MEMORY_ERROR;
  
  if ((r == MAIL_ERROR_DISKSPACE) || (r == MAIL_ERROR_FILE)) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FILE);
    etpan_error_set_short_description(error, _("Not enough disk space"));
    etpan_error_strf_long_description(error, _("An error occurred while adding a message to the mailbox %s. Disk space is insufficient."), etpan_folder_get_ui_path(folder));
    goto error;
  }
  else if (r == MAIL_ERROR_PARSE) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, _("The mailbox is not valid"));
    etpan_error_strf_long_description(error, _("An error occurred while adding a message to the mailbox %s. The mailbox is not valid."), etpan_folder_get_ui_path(folder));
    goto error;
  }
  else if (r != MAIL_NO_ERROR) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_APPEND);
    etpan_error_set_short_description(error, _("Unexpected error"));
    etpan_error_strf_long_description(error, _("An unexpected error (%i) occurred while adding a message to the mailbox %s."), r, etpan_folder_get_ui_path(folder));
    goto error;
  }
  
  return NULL;
  
 error:
  return error;
}

static struct etpan_error * check(struct etpan_folder * folder)
{
  struct mailfolder * ep_folder;
  int r;
  struct etpan_error * error;
  
  error = connect(folder);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(folder, ERROR_DOMAIN_CHECK, error);
    ETPAN_ERROR_FREE(error);
    
    return new_error;
  }
  
  ep_folder = get_mailfolder(folder);
  r = mailfolder_check(ep_folder);
  if (r == MAIL_ERROR_MEMORY)
    ETPAN_LOG_MEMORY_ERROR;
  
  if ((r == MAIL_ERROR_DISKSPACE) || (r == MAIL_ERROR_FILE)) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FILE);
    etpan_error_set_short_description(error, _("Not enough disk space"));
    etpan_error_strf_long_description(error, _("An error occurred while saving mailbox %s information to disk. Disk space is insufficient."), etpan_folder_get_ui_path(folder));
    goto err;
  }
  else if (r == MAIL_ERROR_PARSE) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, _("The mailbox is not valid"));
    etpan_error_strf_long_description(error, _("An error occurred while saving mailbox %s information to disk. The mailbox is not valid."), etpan_folder_get_ui_path(folder));
    goto err;
  }
  else if (r != MAIL_NO_ERROR) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CHECK);
    etpan_error_set_short_description(error, _("Unexpected error"));
    etpan_error_strf_long_description(error, _("An unexpected error occurred while saving mailbox %s information to disk."), r, etpan_folder_get_ui_path(folder));
    goto err;
  }
  
  return NULL;
  
 err:
  return error;
}

static struct etpan_error * status(struct etpan_folder * folder,
    unsigned int * p_count, unsigned int * p_unseen, unsigned int * p_recent)
{
  struct mailfolder * ep_folder;
  uint32_t count;
  uint32_t unseen;
  uint32_t recent;
  int r;
  struct etpan_error * error;
  
  error = connect(folder);
  if (error != NULL) {
    struct etpan_error * new_error;
      
    new_error = rewrite_error(folder, ERROR_DOMAIN_STATUS, error);
    ETPAN_ERROR_FREE(error);
      
    return new_error;
  }
  
  ep_folder = get_mailfolder(folder);
  r = mailfolder_status(ep_folder, &count, &recent, &unseen);
  if (r == MAIL_ERROR_MEMORY)
    ETPAN_LOG_MEMORY_ERROR;
  
  if ((r == MAIL_ERROR_DISKSPACE) || (r == MAIL_ERROR_FILE)) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FILE);
    etpan_error_set_short_description(error, _("Not enough disk space"));
    etpan_error_strf_long_description(error, _("An error occurred while getting information of mailbox %s. Disk space is insufficient."), etpan_folder_get_ui_path(folder));
    goto err;
  }
  else if (r == MAIL_ERROR_PARSE) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, _("The mailbox is not valid"));
    etpan_error_strf_long_description(error, _("An error occurred while getting information of mailbox %s. The mailbox is not valid."), etpan_folder_get_ui_path(folder));
    goto err;
  }
  else if (r != MAIL_NO_ERROR) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_STATUS);
    etpan_error_set_short_description(error, _("Unexpected error"));
    etpan_error_strf_long_description(error, _("An unexpected error occurred while getting information of mailbox %s."), r, etpan_folder_get_ui_path(folder));
    goto err;
  }
  
  * p_count = count;
  * p_unseen = unseen;
  * p_recent = recent;
  
  return NULL;
  
 err:
  return error;
}

static struct etpan_error * expunge(struct etpan_folder * folder)
{
  struct mailfolder * ep_folder;
  int r;
  struct etpan_error * error;
  
  error = connect(folder);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(folder, ERROR_DOMAIN_EXPUNGE, error);
    ETPAN_ERROR_FREE(error);
    
    return new_error;
  }
  
  ep_folder = get_mailfolder(folder);
  r = mailfolder_expunge(ep_folder);
  if (r == MAIL_ERROR_MEMORY)
    ETPAN_LOG_MEMORY_ERROR;
  
  if ((r == MAIL_ERROR_DISKSPACE) || (r == MAIL_ERROR_FILE)) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FILE);
    etpan_error_set_short_description(error, _("Not enough disk space"));
    etpan_error_strf_long_description(error, _("An error occurred while compacting mailbox %s. Disk space is insufficient."), etpan_folder_get_ui_path(folder));
    goto err;
  }
  else if (r == MAIL_ERROR_PARSE) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, _("The mailbox is not valid"));
    etpan_error_strf_long_description(error, _("An error occurred while compacting mailbox %s. The mailbox is not valid."), etpan_folder_get_ui_path(folder));
    goto err;
  }
  else if (r != MAIL_NO_ERROR) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_EXPUNGE);
    etpan_error_set_short_description(error, _("Unexpected error"));
    etpan_error_strf_long_description(error, _("An unexpected error occurred while compacting mailbox %s."), r, etpan_folder_get_ui_path(folder));
    goto err;
  }
  
  return NULL;
  
 err:
  return error;
}

static int is_sub_folder(struct etpan_folder * parent,
    struct etpan_folder * child)
{
  char * location;
  char * child_location;
  char * prefix;
  size_t size;
  int result;
  
  location = etpan_folder_get_location(parent);
  child_location = etpan_folder_get_location(child);
  size = strlen(location) + 2;
  prefix = malloc(size);
  if (prefix == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  snprintf(prefix, size, "%s.", location);
  result = strncmp(prefix, child_location, size - 1);
  free(prefix);
  if (result == 0)
    return 1;
  
  return 0;
}

static struct etpan_error * rewrite_error(struct etpan_folder * folder,
    int domain,
    struct etpan_error * error)
{
  struct etpan_error * new_error;
  char * previous_long_description;
  
  new_error = etpan_error_new();
  etpan_error_set_code(new_error, etpan_error_get_code(error));
  etpan_error_set_short_description(new_error,
      etpan_error_get_short_description(error));
  previous_long_description = etpan_error_get_long_description(error);
  
  switch (domain) {
  case ERROR_DOMAIN_FETCH:
    etpan_error_strf_long_description(new_error,
        _("An error occurred while getting the list of messages of the mailbox %s.\n"
            "%s"),
        etpan_folder_get_ui_path(folder),
        previous_long_description);
    break;
  case ERROR_DOMAIN_APPEND:
    etpan_error_strf_long_description(new_error,
        _("An error occurred while adding a message to the mailbox %s.\n"
            "%s"),
        etpan_folder_get_ui_path(folder),
        previous_long_description);
    break;
  case ERROR_DOMAIN_CHECK:
    etpan_error_strf_long_description(new_error,
        _("An error occurred while saving mailbox %s information.\n"
            "%s"),
        etpan_folder_get_ui_path(folder),
        previous_long_description);
    break;
  case ERROR_DOMAIN_STATUS:
    etpan_error_strf_long_description(new_error,
        _("An error occurred while getting information of mailbox %s.\n"
            "%s"),
        etpan_folder_get_ui_path(folder),
        previous_long_description);
    break;
  case ERROR_DOMAIN_EXPUNGE:
    etpan_error_strf_long_description(new_error,
        _("An error occurred while getting information of mailbox %s.\n"
            "%s"),
        etpan_folder_get_ui_path(folder),
        previous_long_description);
    break;
    
  default:
    etpan_error_strf_long_description(new_error,
        _("An error occurred with the mailbox %s.\n"
            "%s"),
        etpan_folder_get_ui_path(folder),
        previous_long_description);
    break;
  }
  
  return new_error;
}
