#include "etpan-storage-maildir.h"

#include <libetpan/libetpan.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/types.h>
#include <dirent.h>
#include <sys/stat.h>

#include "etpan-log.h"
#include "etpan-error.h"
#include "etpan-storage.h"
#include "etpan-folder.h"
#include "etpan-folder-maildir.h"
#include "etpan-utils.h"
#include "etpan-nls.h"

static struct etpan_error * setup(struct etpan_storage * storage);

static void free_data(struct etpan_storage * storage);

static struct etpan_error * create_folder(struct etpan_storage * storage,
    char * folder_name);

static struct etpan_error * delete_folder(struct etpan_storage * storage,
    char * folder_name);

static struct etpan_error * rename_folder(struct etpan_storage * storage,
    char * folder_name, char * new_name);

static struct etpan_error * fetch_folder_list(struct etpan_storage * storage,
    chash * folder_list);

static char * get_sub_folder_location(struct etpan_storage * storage,
    struct etpan_folder * parent, char * name);

static int allows_sub_folders(struct etpan_storage * storage,
    struct etpan_folder * parent);

static struct etpan_storage_driver maildir_driver = {
  .name = "maildir",
  
  .network = 0,
  
  .setup = setup,
  .create_folder = create_folder,
  .delete_folder = delete_folder,
  .rename_folder = rename_folder,
  .fetch_folder_list = fetch_folder_list,
  .connect = NULL,
  .disconnect = NULL,
  .free_data = free_data,
  .get_sub_folder_location = get_sub_folder_location,
  .allows_sub_folders = allows_sub_folders,
};

struct storage_data {
  char * path;
  char * threaded_path;
};

static char * get_threaded_path(struct etpan_storage * storage);

struct etpan_storage * etpan_storage_maildir_new(void)
{
  struct storage_data * data;
  struct etpan_storage * storage;
  
  storage = etpan_storage_new();
  data = malloc(sizeof(* data));
  if (data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  data->path = NULL;
  data->threaded_path = NULL;
  
  etpan_storage_set_data(storage, data);
  etpan_storage_set_driver(storage, &maildir_driver);
  
  return storage;
}

void etpan_storage_maildir_set_path(struct etpan_storage * storage, char * path)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  if (path != data->path) {
    free(data->path);
    if (path != NULL) {
      data->path = strdup(path);
      if (data->path == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      data->path = NULL;
  }
}

char * etpan_storage_maildir_get_path(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  return data->path;
}

static struct etpan_error *
get_maildir_list(char * base_vpath, char * pathname,
    carray ** result);
static void free_array(carray * list);

struct path_info {
  char * mb;
  char * vpath;
};

static struct etpan_folder * folder_from_info(struct etpan_storage * storage,
    struct path_info * info)
{
  struct etpan_folder * folder;
  
  /* create the folder */
  
  folder = etpan_folder_maildir_new();
  
  etpan_folder_set_ui_path(folder, info->vpath);
  etpan_folder_set_location(folder, info->mb);
  etpan_folder_set_storage(folder, storage);
  
  return folder;
}

static struct etpan_error * fetch_folder_list(struct etpan_storage * storage,
    chash * folder_list)
{
  carray * path_list;
  int r;
  unsigned int i;
  struct etpan_error * error;
  
  if (get_threaded_path(storage) == NULL) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_INVAL);
    etpan_error_set_short_description(error, _("Invalid path"));
    etpan_error_strf_long_description(error, _("Path for account %s has not been defined."), etpan_storage_get_id(storage));
    
    goto err;
  }
  
  error = get_maildir_list(storage->id,
      get_threaded_path(storage), &path_list);
  ETPAN_ERROR_IGNORE(error);
  
  for(i = 0 ; i < carray_count(path_list) ; i ++) {
    struct path_info * info;
    struct etpan_folder * folder;
    chashdatum key;
    chashdatum value;
    
    info = carray_get(path_list, i);
    folder = folder_from_info(storage, info);
    if (folder == NULL) {
      continue;
    }
    
    key.data = folder->location;
    key.len = strlen(folder->location) + 1;
    value.data = folder;
    value.len = 0;
    r = chash_set(folder_list, &key, &value, NULL);
    if (r < 0) {
      etpan_folder_unref(folder);
      ETPAN_LOG_MEMORY_ERROR;
    }
  }
  
  free_array(path_list);
  
  return NULL;
  
 err:
  return error;
}

static void free_data(struct etpan_storage * storage)
{
  struct storage_data * data;

  data = etpan_storage_get_data(storage);
  free(data->threaded_path);
  free(data->path);
  free(data);
}

static char * get_absolute_path(struct etpan_storage * storage,
    char * folder_name)
{
  char * storage_path;
  size_t len;
  char * absolute_path;
  
  storage_path = get_threaded_path(storage);
  
  len = strlen(folder_name) + strlen(storage_path) + 3;
  absolute_path = malloc(len);
  if (absolute_path == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  snprintf(absolute_path, len, "%s/.%s", storage_path, folder_name);
  
  return absolute_path;
}

static struct etpan_error * create_folder(struct etpan_storage * storage,
    char * folder_name)
{
  char * absolute_path;
  int r;
  struct etpan_error * error;
  char subfolder[PATH_MAX];
  
  ETPAN_LOG("folder: %s", folder_name);
  
  if (folder_name[0] == '\0') {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_INVAL);
    etpan_error_set_short_description(error, _("Invalid mailbox name"));
    etpan_error_strf_long_description(error, _("The name of mailbox to create on account %s is not valid. The mailbox name is empty."), etpan_storage_get_id(storage));
    
    goto err;
  }
  
  absolute_path = get_absolute_path(storage, folder_name);
  r = mkdir(absolute_path, 0700);
  if (r < 0) {
    goto error_create;
  }
  
  snprintf(subfolder, sizeof(subfolder), "%s/cur", absolute_path);
  r = mkdir(subfolder, 0700);
  if (r < 0) {
    goto error_create;
  }
  snprintf(subfolder, sizeof(subfolder), "%s/new", absolute_path);
  r = mkdir(subfolder, 0700);
  if (r < 0) {
    goto error_create;
  }
  snprintf(subfolder, sizeof(subfolder), "%s/tmp", absolute_path);
  r = mkdir(subfolder, 0700);
  if (r < 0) {
    goto error_create;
  }
  
  free(absolute_path);
  
  return NULL;
  
 error_create:
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CREATE);
    etpan_error_set_short_description(error, _("Could not create"));
    etpan_error_strf_long_description(error, _("The mailbox %s could not be created on account %s. Check whether the mailbox already exists or the permissions."), folder_name, etpan_storage_get_id(storage));
    goto free;
    
 free:
  free(absolute_path);
 err:
  return error;
}

static struct etpan_error * delete_content(char * pathname)
{
  DIR * dir;
  struct dirent * ent;
  int r;
  struct etpan_error * error;
  
  dir = opendir(pathname);
  if (dir == NULL) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FOLDER_NOT_FOUND);
    etpan_error_set_short_description(error, _("Could not open folder"));
    etpan_error_strf_long_description(error, _("The content of folder %s could not be read."), pathname);
    
    goto err;
  }
  
  while ((ent = readdir(dir)) != NULL) {
    struct stat stat_buf;
    int file_type;
    char filename[PATH_MAX];
    
    if (ent->d_name[0] == '.')
      continue;
    
    snprintf(filename, sizeof(filename), "%s/%s", pathname, ent->d_name);
    
    r = stat(filename, &stat_buf);
    if (r < 0)
      continue;
    
    file_type = stat_buf.st_mode & S_IFMT;
    if (file_type == S_IFREG) {
      unlink(filename);
    }
  }
  
  closedir(dir);
  
  return NULL;
  
 err:
  return error;
}

static struct etpan_error * delete_folder(struct etpan_storage * storage,
    char * folder_name)
{
  char * absolute_path;
  int r;
  struct etpan_error * error;
  char subdir[PATH_MAX];
  
  if (folder_name[0] == '\0') {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_INVAL);
    etpan_error_set_short_description(error, _("Invalid mailbox name"));
    etpan_error_strf_long_description(error, _("The name of mailbox to delete on account %s is not valid. The mailbox name is empty."), etpan_storage_get_id(storage));
    
    goto err;
  }
  
  absolute_path = get_absolute_path(storage, folder_name);
  snprintf(subdir, sizeof(subdir), "%s/cur", absolute_path);
  error = delete_content(subdir);
  if (error != NULL) {
    goto free;
  }
  snprintf(subdir, sizeof(subdir), "%s/new", absolute_path);
  error = delete_content(subdir);
  if (error != NULL) {
    goto free;
  }
  snprintf(subdir, sizeof(subdir), "%s/tmp", absolute_path);
  error = delete_content(subdir);
  if (error != NULL) {
    goto free;
  }
  error = delete_content(absolute_path);
  if (error != NULL) {
    goto free;
  }
  snprintf(subdir, sizeof(subdir), "%s/cur", absolute_path);
  r = rmdir(subdir);
  if (r < 0) {
    goto err_delete;
  }
  snprintf(subdir, sizeof(subdir), "%s/new", absolute_path);
  r = rmdir(subdir);
  if (r < 0) {
    goto err_delete;
  }
  snprintf(subdir, sizeof(subdir), "%s/tmp", absolute_path);
  r = rmdir(subdir);
  if (r < 0) {
    goto err_delete;
  }
  
  r = rmdir(absolute_path);
  if (r < 0) {
    goto err_delete;
  }
  
  free(absolute_path);
  
  return NULL;
  
 err_delete:
  error = etpan_error_new();
  etpan_error_set_code(error, ERROR_DELETE);
  etpan_error_set_short_description(error, _("Could not delete"));
  etpan_error_strf_long_description(error, _("The mailbox %s could not be deleted on account %s. Check the permissions."), folder_name, etpan_storage_get_id(storage));
 free:
  free(absolute_path);
 err:
  return error;
}



static struct path_info * path_info_new(char * mb, char * vpath)
{
  struct path_info * info;
  
  info = malloc(sizeof(* info));
  if (info == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  info->mb = mb;
  info->vpath = vpath;
  
  return info;
}

static void add_to_array(carray * list, char * mb, char * vpath)
{
  struct path_info * info;
  int r;
  
  info = path_info_new(mb, vpath);
  r = carray_add(list, info, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

static void free_array(carray * list)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(list) ; i ++) {
    struct path_info * info;
    
    info = carray_get(list, i);
    free(info->vpath);
    free(info->mb);
    free(info);
  }
  carray_free(list);
}

static int compare_path(const void * elt1, const void * elt2)
{
  struct path_info * const * ppath1;
  struct path_info * const * ppath2;
  struct path_info * path1;
  struct path_info * path2;
  
  ppath1 = elt1;
  ppath2 = elt2;
  path1 = * ppath1;
  path2 = * ppath2;
  
  return strcmp(path1->mb, path2->mb);
}

/* maildir */

static void maildir_add_to_list(carray * list, char * base_vpath,
    char * pathname, carray * error_list)
{
  DIR * dir;
  struct dirent * ent;
  int r;
  int file_type;
  struct stat stat_buf;
  char filename[PATH_MAX];
  struct etpan_error * error;
  
  snprintf(filename, sizeof(filename), "%s/cur", pathname);
  r = stat(filename, &stat_buf);
  if (r == 0) {
    file_type = stat_buf.st_mode & S_IFMT;
    if (file_type == S_IFDIR) {
      char * name;
      char * vpath;
      char tmp_vpath[PATH_MAX];
      
      name = strdup("");
      if (name == NULL) {
        ETPAN_LOG_MEMORY_ERROR;
      }
      
      if (* base_vpath == '\0')
        snprintf(tmp_vpath, sizeof(tmp_vpath), "INBOX");
      else
        snprintf(tmp_vpath, sizeof(tmp_vpath), "%s/INBOX", base_vpath);
      vpath = strdup(tmp_vpath);
      if (vpath == NULL) {
        free(name);
        ETPAN_LOG_MEMORY_ERROR;
      }
      
      add_to_array(list, name, vpath);
    }
  }
  
  dir = opendir(pathname);
  if (dir == NULL) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FOLDER_NOT_FOUND);
    etpan_error_set_short_description(error, _("Folder not found"));
    etpan_error_strf_long_description(error, _("The folder containing the mbox files was not found (%s)."), pathname);
    
    r = carray_add(error_list, error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    return;
  }
  
  while ((ent = readdir(dir)) != NULL) {
    if (ent->d_name[0] == '.') {
      if (ent->d_name[1] == '\0')
        continue;
      else if (ent->d_name[1] == '.') {
        if (ent->d_name[2] == '\0')
          continue;
      }
      else {
        snprintf(filename, sizeof(filename), "%s/%s/cur",
            pathname, ent->d_name);
        r = stat(filename, &stat_buf);
        if (r == 0) {
          file_type = stat_buf.st_mode & S_IFMT;
          if (file_type == S_IFDIR) {
            char tmp_vpath[PATH_MAX];
            char * name;
            char * vpath;
            char tmp_dname[PATH_MAX];
            char * p;
            
            snprintf(tmp_dname, sizeof(tmp_dname), "%s", ent->d_name + 1);
            p = tmp_dname;
            while ((p = strchr(p, '.')) != NULL) {
              * p = '/';
              p ++;
            }
            
            if (* base_vpath == '\0')
              snprintf(tmp_vpath, sizeof(tmp_vpath), "%s", tmp_dname);
            else
              snprintf(tmp_vpath, sizeof(tmp_vpath), "%s/%s",
                  base_vpath, tmp_dname);
            
            name = strdup(ent->d_name + 1);
            if (name == NULL) {
              ETPAN_LOG_MEMORY_ERROR;
            }
            
            vpath = strdup(tmp_vpath);
            if (vpath == NULL) {
              free(name);
              ETPAN_LOG_MEMORY_ERROR;
            }
            
            add_to_array(list, name, vpath);
          }
        }
      }
    }
  }
  
  closedir(dir);
}

static struct etpan_error *
get_maildir_list(char * base_vpath, char * pathname,
    carray ** result)
{
  carray * mb_list;
  carray * error_list;
  struct etpan_error * error;
  unsigned int i;
  
  mb_list = carray_new(8);
  if (mb_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  error_list = carray_new(4);
  if (error_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  maildir_add_to_list(mb_list, base_vpath, pathname, error_list);
  
  qsort(carray_data(mb_list),
      carray_count(mb_list),
      sizeof(void *),
      compare_path);
  
  * result = mb_list;
  
  if (carray_count(error_list) == 0) {
    error = NULL;
  }
  else if (carray_count(error_list) == 1) {
    error = carray_get(error_list, 0);
  }
  else {
    error = etpan_error_multiple();
    for(i = 0 ; i < carray_count(error_list) ; i ++) {
      struct etpan_error * suberror;
      
      suberror = carray_get(error_list, i);
      etpan_error_add_child(error, suberror);
    }
  }
  carray_free(error_list);
  
  return error;
}


static struct etpan_error * setup(struct etpan_storage * storage)
{
  struct storage_data * data;
  char subfolder[PATH_MAX];
  
  data = etpan_storage_get_data(storage);
  
  mkdir(data->path, 0700);
  snprintf(subfolder, sizeof(subfolder), "%s/cur", data->path);
  mkdir(subfolder, 0700);
  snprintf(subfolder, sizeof(subfolder), "%s/new", data->path);
  mkdir(subfolder, 0700);
  snprintf(subfolder, sizeof(subfolder), "%s/tmp", data->path);
  mkdir(subfolder, 0700);
  
  free(data->threaded_path);
  if (data->path != NULL) {
    data->threaded_path = strdup(data->path);
    if (data->threaded_path == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  else {
    data->threaded_path = NULL;
  }
  
  return NULL;
}


static char * get_threaded_path(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  return data->threaded_path;
}

static char * get_sub_folder_location(struct etpan_storage * storage,
    struct etpan_folder * parent, char * name)
{
  (void) storage;
  
  if (parent == NULL) {
    char * subfolder;
    
    subfolder = strdup(name);
    if (subfolder == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    
    return subfolder;
  }
  else {
    size_t size;
    char * location;
    char * parent_location;
    
    parent_location = etpan_folder_get_location(parent);
    size = strlen(parent_location) + 1 + strlen(name) + 1;
    location = malloc(size);
    if (location == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    
    snprintf(location, size, "%s.%s", parent_location, name);
    
    return location;
  }
}

static int allows_sub_folders(struct etpan_storage * storage,
    struct etpan_folder * parent)
{
  (void) storage;
  (void) parent;
  
  return 1;
}

static struct etpan_error * rename_folder(struct etpan_storage * storage,
    char * folder_name, char * new_name)
{
  char * absolute_new_path;
  char * absolute_path;
  int r;
  struct etpan_error * error;
  struct stat stat_buf;
  
  if (folder_name[0] == '\0') {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_INVAL);
    etpan_error_set_short_description(error, _("Invalid mailbox name"));
    etpan_error_strf_long_description(error, _("Mailbox to rename on account %s is not valid. The mailbox name is empty."), etpan_storage_get_id(storage));
    goto err;
  }
  if (new_name[0] == '\0') {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_INVAL);
    etpan_error_set_short_description(error, _("Invalid mailbox name"));
    etpan_error_strf_long_description(error, _("An error occurred while changing the name of mailbox %s on account %s. New name of the mailbox is empty."), folder_name, etpan_storage_get_id(storage));
    goto err;
  }
  
  absolute_path = NULL;
  absolute_new_path = NULL;
  
  absolute_path = get_absolute_path(storage, folder_name);
  absolute_new_path = get_absolute_path(storage, new_name);
  
  r = stat(absolute_new_path, &stat_buf);
  if (r == 0) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FILE);
    etpan_error_set_short_description(error, _("Mailbox already exists"));
    etpan_error_strf_long_description(error, _("An error occurred while changing the name of mailbox from %s to %s on account %s. Mailbox %s already exists."),
        folder_name, new_name, etpan_storage_get_id(storage),
        new_name);
    
    goto free;
  }
  
  r = rename(absolute_path, absolute_new_path);
  if (r < 0) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FILE);
    etpan_error_set_short_description(error, _("Rename failed"));
    etpan_error_strf_long_description(error, _("An error occurred while changing the name of mailbox from %s to %s on account %s. Check if the mailbox %s exists or the permissions."),
        folder_name, new_name, etpan_storage_get_id(storage),
        folder_name);
    
    goto free;
  }
  
  free(absolute_new_path);
  free(absolute_path);
  
  return NULL;
  
 free:
  free(absolute_new_path);
  free(absolute_path);
 err:
  return error;
}
