#include "etpan-mbox-cache.h"

#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "etpan-log.h"
#include "etpan-error.h"
#include "etpan-tc.h"
#include "etpan-log.h"
#include "etpan-nls.h"

static struct etpan_mbox_cache * default_manager = NULL;

struct msg_count {
  unsigned int count;
  unsigned int unseen;
  unsigned int recent;
  time_t mtime;
};

void etpan_mbox_cache_set_default(struct etpan_mbox_cache * cache)
{
  default_manager = cache;
}

struct etpan_mbox_cache * etpan_mbox_cache_get_default(void)
{
  return default_manager;
}

struct etpan_mbox_cache * etpan_mbox_cache_new(char * cache_path)
{
  struct etpan_mbox_cache * cache;
  
  cache = malloc(sizeof(* cache));
  if (cache == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  cache->filename = strdup(cache_path);
  if (cache->filename == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  cache->db = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (cache->db == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return cache;
}

void etpan_mbox_cache_free(struct etpan_mbox_cache * cache)
{
  chashiter * iter;
  
  for(iter = chash_begin(cache->db) ; iter != NULL ;
      iter = chash_next(cache->db, iter)) {
    chashdatum value;
    struct msg_count * count_data;
    
    chash_value(iter, &value);
    count_data = value.data;
    free(count_data);
  }
  
  chash_free(cache->db);
  free(cache->filename);
  free(cache);
}

static struct etpan_error * cache_load(struct etpan_mbox_cache * cache)
{
  void * iter;
  int r;
  tcconf_section_t * config;

  config = tcconf_load_file(NULL, cache->filename);
  if (config == NULL) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, _("Parse error"));
    etpan_error_strf_long_description(error, _("Cache file for mbox at %s could not be parsed."), cache->filename);
    
    return error;
  }
  
  iter = NULL;
  do {
    tcconf_section_t * folder_info;
    
    folder_info = tcconf_nextsection(config, "folder", &iter);
    if (folder_info != NULL) {
      unsigned int count;
      unsigned int unseen;
      unsigned int recent;
      char * location;
      
      r = tcconf_getvalue(folder_info, "location", "%s", &location);
      if (r <= 0)
        goto next_location;

      fprintf(stderr, "%i %s\n", r, location);
      r = tcconf_getvalue(folder_info, "count", "%ui", &count);
      if (r <= 0)
        goto next;

      r = tcconf_getvalue(folder_info, "unseen", "%ui", &unseen);
      if (r <= 0)
        goto next;
      
      r = tcconf_getvalue(folder_info, "recent", "%ui", &recent);
      if (r <= 0)
        goto next;
      
#if 0
      r = tcconf_getvalue(folder_info, "mtime", "%lui", &value);
      if (r <= 0)
        goto next;
#endif
      
      fprintf(stderr, "%i %s\n", r, location);
      etpan_mbox_cache_set_count(cache, location, count,
          unseen, recent);
      
    next:
      free(location);
    next_location:
      tcfree(folder_info);
      /* ignore error */
    }
    else {
      break;
    }
  } while (1);
  
  tcfree(config);
  
  return NULL;
}

static char * get_quoted_name(char * str)
{
  unsigned int len;
  char * p;
  char * dupstr;
  char * dest_p;
  
  len = 0;
  p = str;
  while (* p != '\0') {
    if ((* p == '\\') || (* p == '\"'))
      len ++;
    len ++;
    p ++;
  }
  
  dupstr = malloc(len + 1);
  if (dupstr == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  dest_p = dupstr;
  p = str;
  while (* p != '\0') {
    if ((* p == '\\') || (* p == '\"'))
      * dest_p = '\\';
    
    * dest_p = * p;
    
    p ++;
    dest_p ++;
  }
  
  return dupstr;
}

static struct etpan_error * cache_save(struct etpan_mbox_cache * cache)
{
  chashiter * iter;
  FILE * f;
  
  f = fopen(cache->filename, "w");
  if (f == NULL) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FILE);
    etpan_error_set_short_description(error, _("Cache file could not be written"));
    etpan_error_strf_long_description(error, _("Cache file for mbox at %s could not be written to disk."), cache->filename);
    
    return error;
  }
  
  for(iter = chash_begin(cache->db) ; iter != NULL ;
      iter = chash_next(cache->db, iter)) {
    chashdatum value;
    struct msg_count * count_data;
    char * quoted;
    chashdatum key;
    char * location;
    
    chash_key(iter, &key);
    chash_value(iter, &value);
    count_data = value.data;
    location = key.data;
    
    if (count_data->mtime == (time_t) -1)
      continue;
    
    quoted = get_quoted_name(key.data);
    
    fprintf(f, "folder [\n");
    fprintf(f, "  location \"%s\"\n", location);
    fprintf(f, "  count %u\n", count_data->count);
    fprintf(f, "  unseen %u\n", count_data->unseen);
    fprintf(f, "  recent %u\n", count_data->recent);
    fprintf(f, "  mtime %lu\n", (unsigned long int) count_data->mtime);
    fprintf(f, "]\n");
    fprintf(f, "\n");
    free(quoted);
  }
  
  fclose(f);
  
  return NULL;
}

void etpan_mbox_cache_setup(struct etpan_mbox_cache * cache)
{
  struct etpan_error * error;
  
  error = cache_load(cache);
  ETPAN_ERROR_IGNORE(error);
}

void etpan_mbox_cache_unsetup(struct etpan_mbox_cache * cache)
{
  struct etpan_error * error;
  
  error = cache_save(cache);
  ETPAN_ERROR_IGNORE(error);
}

static void set_count(struct etpan_mbox_cache * cache,
    char * path, unsigned int count,
    unsigned int unseen, unsigned int recent,
    time_t mtime)
{
  chashdatum key;
  chashdatum value;
  int r;
  struct msg_count * count_data;
  
  key.data = path;
  key.len = strlen(path) + 1;
  r = chash_get(cache->db, &key, &value);
  if (r < 0) {
    count_data = malloc(sizeof(* count_data));
    if (count_data == NULL) {
      ETPAN_LOG("etpan_mbox_cache_set_count - not enough memory");
      return;
    }
    
    value.data = count_data;
    value.len = 0;
    
    r = chash_set(cache->db, &key, &value, NULL);
    if (r < 0) {
      free(count_data);
      ETPAN_LOG("etpan_mbox_cache_set_count - not enough memory");
      return;
    }
  }
  else {
    count_data = value.data;
  }
  
  count_data->count = count;
  count_data->unseen = unseen;
  count_data->recent = recent;
  count_data->mtime = mtime;
}

void etpan_mbox_cache_set_count(struct etpan_mbox_cache * cache,
    char * path, unsigned int count,
    unsigned int unseen, unsigned int recent)
{
  struct stat stat_info;
  int r;
  
  r = stat(path, &stat_info);
  if (r < 0) {
    etpan_mbox_cache_invalidate(cache, path);
    return;
  }
  
  set_count(cache, path, count, unseen, recent, stat_info.st_mtime);
}

int etpan_mbox_cache_get_count(struct etpan_mbox_cache * cache,
    char * path, unsigned int * p_count,
    unsigned int * p_unseen, unsigned int * p_recent)
{
  chashdatum key;
  chashdatum value;
  int r;
  struct msg_count * count_data;
  struct stat stat_info;
  
  key.data = path;
  key.len = strlen(path) + 1;
  
  r = chash_get(cache->db, &key, &value);
  if (r < 0)
    return -1;
  
  count_data = value.data;
  
  if (count_data->mtime == (time_t) -1)
    return -1;
  
  r = stat(path, &stat_info);
  if (r < 0)
    return -1;
  
  if (stat_info.st_mtime != count_data->mtime)
    return ERROR_INVAL;
  
  * p_count = count_data->count;
  * p_unseen = count_data->unseen;
  * p_recent = count_data->recent;
  
  return 0;
}

void etpan_mbox_cache_invalidate(struct etpan_mbox_cache * cache,
    char * path)
{
  set_count(cache, path, 0, 0, 0, (time_t) -1);
}
