#include "etpan-sender-news.h"

#include <libetpan/libetpan.h>
#include <libetpan/mailsem.h>
#include <stdlib.h>

#include "etpan-sender.h"
#include "etpan-storage-news.h"
#include "etpan-error.h"
#include "etpan-nls.h"
#include "etpan-log.h"

static struct etpan_error * send_message(struct etpan_sender * sender,
    char * message, size_t size);
static void free_data(struct etpan_sender * sender);

struct sender_data {
  struct etpan_storage * storage;
  struct mailsem * sem;
};

static struct etpan_sender_driver news_driver = {
  .name = "news",
  .setup = NULL,
  .free_data = free_data,
  .send_message = send_message,
  .connect = NULL,
  .disconnect = NULL,
};

struct append_cb_data {
  struct mailsem * sem;
  struct etpan_storage * storage;
  char * message;
  size_t size;
  struct etpan_sender * sender;
  struct etpan_error * error;
};

static void append_callback(int cancelled,
    struct etpan_storage_news_append_msg_result * result,
    void * cb_data)
{
  struct sender_data * data;
  struct append_cb_data * a_cb_data;
  struct etpan_sender * sender;
  
  a_cb_data = cb_data;
  sender = a_cb_data->sender;
  
  data = etpan_sender_get_data(sender);

  if (cancelled) {
    struct etpan_error * error;

    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CANCELLED);
    etpan_error_set_short_description(error,
        _("Message was not posted"));
    etpan_error_strf_long_description(error,
        _("Post of the message from account %s was interrupted."),
        etpan_sender_get_id(a_cb_data->sender));
    
    a_cb_data->error = error;
  }
  else {
    a_cb_data->error = result->error;
    result->error = NULL;
  }
  
  mailsem_up(data->sem);
}

static void send_message_main_thread(void * data)
{
  struct append_cb_data * a_cb_data;
  struct etpan_thread_op * op;
  
  a_cb_data = data;
  
  op = etpan_storage_news_append(etpan_thread_manager_app_get_default(),
      a_cb_data->storage, a_cb_data->message, a_cb_data->size,
      append_callback, a_cb_data);
}

static struct etpan_error * send_message(struct etpan_sender * sender,
    char * message, size_t size)
{
  struct sender_data * data;
  struct append_cb_data a_cb_data;
  
  data = etpan_sender_get_data(sender);
  
  a_cb_data.sem = data->sem;
  a_cb_data.storage = data->storage;
  a_cb_data.message = message;
  a_cb_data.size = size;
  a_cb_data.sender = sender;
  a_cb_data.error = NULL;
  
  etpan_thread_manager_app_run_in_main_thread(etpan_thread_manager_app_get_default(), send_message_main_thread, &a_cb_data, 1);
  
  mailsem_down(data->sem);
  
  return a_cb_data.error;
}

static void free_data(struct etpan_sender * sender)
{
  struct sender_data * data;
  
  data = etpan_sender_get_data(sender);
  
  mailsem_free(data->sem);
  
  free(data);
}

struct etpan_sender * etpan_sender_news_new(void)
{
  struct etpan_sender * sender;
  struct sender_data * data;
  
  sender = etpan_sender_new();
  if (sender == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  data = malloc(sizeof(* data));
  if (data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  data->sem = mailsem_new();
  if (data->sem == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  data->storage = NULL;
  
  etpan_sender_set_data(sender, data);
  etpan_sender_set_driver(sender, &news_driver);
  
  return sender;
}

void etpan_sender_news_set_storage(struct etpan_sender * sender,
    struct etpan_storage * storage)
{
  struct sender_data * data;
  
  data = etpan_sender_get_data(sender);
  data->storage = storage;
}
