#include "etpan-storage-news.h"

#include <libetpan/libetpan.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/types.h>
#include <dirent.h>
#include <sys/stat.h>
#include <sys/mman.h>

#include "etpan-log.h"
#include "etpan-error.h"
#include "etpan-storage.h"
#include "etpan-folder.h"
#include "etpan-folder-news.h"
#include "etpan-lep.h"
#include "etpan-storage-private.h"
#include "etpan-storage-news-private.h"
#include "etpan-utils.h"
#include "etpan-connection-types.h"
#include "etpan-nls.h"

enum {
  ERROR_DOMAIN_APPEND,
};

static struct etpan_error * rewrite_error(struct etpan_storage * storage,
    int domain,
    struct etpan_error * error);

static struct etpan_error * setup(struct etpan_storage * storage);

static void free_data(struct etpan_storage * storage);

static struct etpan_error * create_folder(struct etpan_storage * storage,
    char * folder_name);

static struct etpan_error * delete_folder(struct etpan_storage * storage,
    char * folder_name);

static struct etpan_error * fetch_folder_list(struct etpan_storage * storage,
    chash * folder_list);

static struct etpan_error * connect(struct etpan_storage * storage);

static void disconnect(struct etpan_storage * storage);

static char * get_sub_folder_location(struct etpan_storage * storage,
    struct etpan_folder * parent, char * name);

static struct etpan_storage_driver news_driver = {
  .name = "news",
  
  .network = 1,
  
  .setup = setup,
  .create_folder = create_folder,
  .delete_folder = delete_folder,
  .rename_folder = NULL,
  .fetch_folder_list = fetch_folder_list,
  .connect = connect,
  .disconnect = disconnect,
  .free_data = free_data,
  .get_sub_folder_location = get_sub_folder_location,
  .allows_sub_folders = NULL,
};

struct storage_data {
  char * hostname;
  int port;
  int connection_type;
  char * username;
  char * password;
  
  /* threaded */
  struct mailstorage * storage;
  int connected;
  char * threaded_hostname;
  int threaded_port;
  int threaded_connection_type;
  char * threaded_username;
  char * threaded_password;
};

static struct etpan_error * ep_storage_connect(struct etpan_storage * storage)
{
  struct etpan_error * error;
  int r;
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  r = mailstorage_connect(data->storage);
  if (r == MAIL_ERROR_MEMORY)
    ETPAN_LOG_MEMORY_ERROR;
    
  if (r == MAIL_ERROR_CONNECT) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CONNECT);
    etpan_error_set_short_description(error, "Could not connect");
    etpan_error_strf_long_description(error,
        _("Connection to account %s failed."
            "The connection to IMAP server %s, port %i could not be established.\n"
            "Check the name of server and port."),
        etpan_storage_get_id(storage),
        etpan_storage_news_get_hostname(storage),
        etpan_storage_news_get_port(storage));
    goto err;
  }
  else if (r == MAIL_ERROR_LOGIN) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_AUTH);
    etpan_error_set_short_description(error, "Authentication failed");
    etpan_error_strf_long_description(error,
        _("Connection to account %s failed."
            "Check your login, password and authentication type."),
        etpan_storage_get_id(storage));
    goto err;
  }
  else if (r == MAIL_ERROR_PARSE) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, "Parse error");
    etpan_error_strf_long_description(error, _("Connection to account %s failed. The application did not understand what the server sent."),
        etpan_storage_get_id(storage));
    goto err;
  }
  else if (r == MAIL_ERROR_STREAM) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_STREAM);
    etpan_error_set_short_description(error, "Connection closed");
    etpan_error_strf_long_description(error,
        _("Connection to account %s failed."
            "The connection closed unexpectedly."),
        etpan_storage_get_id(storage));
    goto err;
  }
  else if (r != MAIL_NO_ERROR) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CONNECT);
    etpan_error_set_short_description(error, _("Unexpected error"));
    etpan_error_strf_long_description(error,
        _("Connection to account %s failed."
            "An unexpected error (libetpan: %i) occurred."),
        etpan_storage_get_id(storage), r);
    goto err;
  }
    
  error = NULL;
    
 err:
  return error;
}

static struct etpan_error * connect(struct etpan_storage * storage)
{
  int r;
  struct storage_data * data;
  struct mailstorage * ep_storage;
  int ep_connection_type;
  int ep_auth_type;
  int cached;
  struct etpan_error * error;
  
  data = etpan_storage_get_data(storage);
  
  if (data->connected) {
    error = ep_storage_connect(storage);
    if (error != NULL) {
      goto err;
    }
    
    r = mailsession_noop(data->storage->sto_session);
    if (r == MAIL_ERROR_MEMORY)
      ETPAN_LOG_MEMORY_ERROR;
    
    if (r == MAIL_ERROR_LOGIN) {
      error = etpan_error_new();
      etpan_error_set_code(error, ERROR_AUTH);
      etpan_error_set_short_description(error, "Authentication failed");
      etpan_error_strf_long_description(error,
          _("Connection to account %s failed."
              "Check your login, password and authentication type."),
          etpan_storage_get_id(storage));
      goto err;
    }
    else if (r == MAIL_ERROR_PARSE) {
      error = etpan_error_new();
      etpan_error_set_code(error, ERROR_PARSE);
      etpan_error_set_short_description(error, "Parse error");
      etpan_error_strf_long_description(error, _("Connection to account %s failed. The application did not understand what the server sent."),
          etpan_storage_get_id(storage));
      goto err;
    }
    else if (r == MAIL_ERROR_STREAM) {
      error = etpan_error_new();
      etpan_error_set_code(error, ERROR_STREAM);
      etpan_error_set_short_description(error, "Connection closed");
      etpan_error_strf_long_description(error,
          _("Connection to account %s failed."
              "The connection closed unexpectedly."),
          etpan_storage_get_id(storage));
      goto err;
    }
    else if (r != MAIL_NO_ERROR) {
      error = etpan_error_new();
      etpan_error_set_code(error, ERROR_CONNECT);
      etpan_error_set_short_description(error, _("Unexpected error"));
      etpan_error_strf_long_description(error,
          _("Connection to account %s failed."
              "An unexpected error (libetpan: %i) occurred."),
          etpan_storage_get_id(storage), r);
      goto err;
    }
    
    return NULL;
  }

  if (data->threaded_hostname == NULL) {
    error = etpan_error_internal_strf(_("Account information of %s are not valid. Neither command nor hostname are defined."),
        etpan_storage_get_id(storage));
    
    goto err;
  }
  
  switch (data->threaded_connection_type) {
  case ETPAN_CONNECTION_TLS:
    ep_connection_type = CONNECTION_TYPE_TLS;
    break;
  default:
    ep_connection_type = CONNECTION_TYPE_PLAIN;
    break;
  }
  
  ep_auth_type = NNTP_AUTH_TYPE_PLAIN;
  
  ep_storage = mailstorage_new(NULL);
  if (ep_storage == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  cached = 1;
  if (etpan_storage_get_threaded_cache_path(storage) == NULL)
    cached = 0;
  if (etpan_storage_get_threaded_flags_path(storage) == NULL)
    cached = 0;
  
  r = nntp_mailstorage_init(ep_storage,
      data->threaded_hostname, data->threaded_port,
      NULL,
      ep_connection_type, ep_auth_type,
      data->threaded_username, data->threaded_password,
      cached, etpan_storage_get_threaded_cache_path(storage),
      etpan_storage_get_threaded_flags_path(storage));
  if (r == MAIL_ERROR_MEMORY)
    ETPAN_LOG_MEMORY_ERROR;
  
  data->storage = ep_storage;
  error = ep_storage_connect(storage);
  if (error != NULL) {
    goto free_storage;
  }
  
  data->connected = 1;
  
  return NULL;
  
 free_storage:
  mailstorage_free(ep_storage);
  data->storage = NULL;
 err:
  return error;
}

static void disconnect(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  
  if (!data->connected)
    return;
  
  data->connected = 0;
  
  mailstorage_disconnect(data->storage);
  mailstorage_free(data->storage);
  data->storage = NULL;
}

struct etpan_storage * etpan_storage_news_new(void)
{
  struct storage_data * data;
  struct etpan_storage * storage;
  
  storage = etpan_storage_new();
  data = malloc(sizeof(* data));
  if (data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  data->hostname = NULL;
  data->port = 0;
  data->connection_type = ETPAN_CONNECTION_PLAIN;
  data->username = NULL;
  data->password = NULL;
  data->storage = NULL;
  data->connected = 0;
  data->threaded_hostname = NULL;
  data->threaded_port = 0;
  data->threaded_connection_type = ETPAN_CONNECTION_PLAIN;
  data->threaded_username = NULL;
  data->threaded_password = NULL;
  
  etpan_storage_set_data(storage, data);
  etpan_storage_set_driver(storage, &news_driver);
  
  return storage;
}

static struct etpan_error * create_folder(struct etpan_storage * storage,
    char * folder_name)
{
  struct etpan_error * error;
  char * home;
  FILE * f;
  char filename[PATH_MAX];
  struct storage_data * data;
  
  home = getenv("HOME");
  if (home == NULL) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FILE);
    etpan_error_set_short_description(error, _("Home directory not found"));
    etpan_error_strf_long_description(error, _("Error while subscribing to %s on account %s. Home directory is not defined. It should be defined in environment variable HOME."), folder_name, etpan_storage_get_id(storage));
    
    goto err;
  }
  
  data = etpan_storage_get_data(storage);
  snprintf(filename, sizeof(filename), "%s/.newsrc-%s", home,
      data->threaded_hostname);
  f = fopen(filename, "a");
  if (f == NULL) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FILE);
    etpan_error_set_short_description(error, _("Could not subscribe"));
    etpan_error_strf_long_description(error, _("Error while subscribing to %s on account %s. The subscription file could not be written (%s)."), folder_name, etpan_storage_get_id(storage), filename);
    
    goto err;
  }
  
  fprintf(f, "%s:\n", folder_name);
  
  fclose(f);
  
  return NULL;
  
 err:
  return error;
}

static struct etpan_error * delete_folder(struct etpan_storage * storage,
    char * folder_name)
{
  struct etpan_error * error;
  char * home;
  FILE * f;
  char filename[PATH_MAX];
  carray * group_list;
  char buf[1024];
  unsigned int i;
  char * prefix;
  size_t size;
  struct storage_data * data;
  int r;
  
  home = getenv("HOME");
  if (home == NULL) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FILE);
    etpan_error_set_short_description(error, _("Home directory not found"));
    etpan_error_strf_long_description(error, _("Error while subscribing to %s on account %s. Home directory is not defined. It should be defined in environment variable HOME."), folder_name, etpan_storage_get_id(storage));
    
    goto err;
  }
  
  group_list = carray_new(16);
  if (group_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  data = etpan_storage_get_data(storage);
  snprintf(filename, sizeof(filename), "%s/.newsrc-%s", home,
      data->threaded_hostname);
  f = fopen(filename, "r");
  if (f == NULL) {
    carray_free(group_list);
    return NULL;
  }
  
  while (fgets(buf, sizeof(buf), f)) {
    char * item;
    
    item = strdup(buf);
    if (item == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    
    r = carray_add(group_list, item, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  fclose(f);
  
  size = strlen(folder_name) + 2;
  prefix = malloc(size);
  if (prefix == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  snprintf(prefix, size, "%s:", folder_name);
  
  f = fopen(filename, "w");
  if (f == NULL) {
    carray_free(group_list);
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FILE);
    etpan_error_set_short_description(error, _("Could not unsubscribe"));
    etpan_error_strf_long_description(error, _("Error while unsubscribing from %s on account %s. The subscription file could not be written (%s)."), folder_name, etpan_storage_get_id(storage), filename);
    
    goto err;
  }
  
  for(i = 0 ; i < carray_count(group_list) ; i ++) {
    char * item;
    
    item = carray_get(group_list, i);
    if (strncmp(prefix, folder_name, size - 1) != 0) {
      fputs(item, f);
    }
  }
  
  fclose(f);
  
  carray_free(group_list);
  
  return NULL;
  
 err:
  return error;
}

static struct etpan_error * fetch_folder_list(struct etpan_storage * storage,
    chash * folder_list)
{
  FILE * f;
  char filename[PATH_MAX];
  char * home;
  char buffer[1024];
  int new_line;
  int r;
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  
  home = getenv("HOME");
  if (home == NULL) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FILE);
    etpan_error_set_short_description(error, _("Home directory not found"));
    etpan_error_strf_long_description(error, _("Error while getting folder list on account %s. Home directory is not defined. It should be defined in environment variable HOME."), etpan_storage_get_id(storage));
    
    return error;
  }
  
  snprintf(filename, sizeof(filename), "%s/.newsrc-%s", home,
      data->threaded_hostname);
  f = fopen(filename, "r");
  if (f == NULL) {
    return NULL;
  }
  
  new_line = 1;
  while (fgets(buffer, sizeof(buffer), f) != NULL) {
    size_t len;
    int process;
    
    process = 0;
    if (new_line)
      process = 1;
    
    new_line = 0;
    len = strlen(buffer);
    if (len > 0)
      if (buffer[len - 1] == '\n')
        new_line = 1;
    
    if (process) {
      char * p;

      p = strchr(buffer, ':');
      if (p != NULL) {
        chashdatum key;
        chashdatum value;
        size_t name_len;
        struct etpan_folder * folder;
        char vpath[PATH_MAX];
        
        * p = '\0';
        
        folder = etpan_folder_news_new();
        snprintf(vpath, sizeof(vpath), "%s/%s",
            etpan_storage_get_id(storage), buffer);
        etpan_folder_set_ui_path(folder, vpath);
        etpan_folder_set_location(folder, buffer);
        etpan_folder_set_storage(folder, storage);
        
        name_len = p - buffer;
        key.data = buffer;
        key.len = name_len + 1;
        value.data = folder;
        value.len = 0;
        
        r = chash_set(folder_list, &key, &value, NULL);
        if (r < 0) {
          etpan_folder_unref(folder);
          ETPAN_LOG_MEMORY_ERROR;
        }
      }
    }
    
  }
  
  fclose(f);
  
  return NULL;
}

static void free_data(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  
  free(data->threaded_password);
  free(data->threaded_username);
  free(data->threaded_hostname);
  
  free(data->password);
  free(data->username);
  free(data->hostname);
  free(data);
}

void etpan_storage_news_set_hostname(struct etpan_storage * storage,
    char * hostname)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  if (hostname != data->hostname) {
    free(data->hostname);
    if (hostname != NULL) {
      data->hostname = strdup(hostname);
      if (data->hostname == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      data->hostname = NULL;
  }
}

char * etpan_storage_news_get_hostname(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  return data->hostname;
}

void etpan_storage_news_set_port(struct etpan_storage * storage, int port)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  data->port = port;
}

int etpan_storage_news_get_port(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  return data->port;
}

void etpan_storage_news_set_connection_type(struct etpan_storage * storage,
    int connection_type)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  data->connection_type = connection_type;
}

int etpan_storage_news_get_connection_type(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  return data->connection_type;
}

void etpan_storage_news_set_username(struct etpan_storage * storage,
    char * username)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  if (username != data->username) {
    free(data->username);
    if (username != NULL) {
      data->username = strdup(username);
      if (data->username == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      data->username = NULL;
  }
}

char * etpan_storage_news_get_username(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  return data->username;
}

void etpan_storage_news_set_password(struct etpan_storage * storage,
    char * password)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  if (password != data->password) {
    free(data->password);
    if (password != NULL) {
      data->password = strdup(password);
      if (data->password == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      data->password = NULL;
  }
}

char * etpan_storage_news_get_password(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  return data->password;
}

static struct etpan_error * setup(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  
  free(data->threaded_hostname);
  if (data->hostname != NULL) {
    data->threaded_hostname = strdup(data->hostname);
    if (data->threaded_hostname == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  else {
    data->threaded_hostname = NULL;
  }
  
  data->threaded_port = data->port;
  data->threaded_connection_type = data->connection_type;
  
  free(data->threaded_username);
  if (data->username != NULL) {
    data->threaded_username = strdup(data->username);
    if (data->threaded_username == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  else {
    data->threaded_username = NULL;
  }
  
  free(data->threaded_password);
  if (data->password != NULL) {
    data->threaded_password = strdup(data->password);
    if (data->threaded_password == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  else {
    data->threaded_password = NULL;
  }
  
  return NULL;
}

struct mailstorage *
etpan_storage_news_get_ep_storage(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  return data->storage;
}


/* append */

struct etpan_storage_news_append_msg_param {
  struct etpan_storage * storage;
  MMAPString * message;
};

static void
storage_news_append_msg_cleanup(struct etpan_thread_op * op)
{
  struct etpan_storage_news_append_msg_result * result;
  struct etpan_storage_news_append_msg_param * param;
  
  param = op->param;
  mmap_string_free(param->message);
  
  free(op->param);
  
  result = op->result;
  ETPAN_ERROR_FREE(result->error);
  free(op->result);
}

static void
storage_news_append_msg_run(struct etpan_thread_op * op)
{
  struct etpan_storage_news_append_msg_param * param;
  struct etpan_storage_news_append_msg_result * result;
  struct etpan_error * error;
  struct storage_data * data;
  int r;
  
  param = op->param;
  result = op->result;
  
  data = etpan_storage_get_data(param->storage);
  
  error = etpan_storage_connect_nt(param->storage);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(param->storage, ERROR_DOMAIN_APPEND, error);
    result->error = new_error;
    ETPAN_ERROR_FREE(error);
    error = new_error;
    
    goto err;
  }
  
  r = mailsession_append_message(data->storage->sto_session,
      param->message->str, param->message->len);
  if (r == MAIL_ERROR_MEMORY)
    ETPAN_LOG_MEMORY_ERROR;
    
  if (r == MAIL_ERROR_LOGIN) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_AUTH);
    etpan_error_set_short_description(error, "Authentication failed");
    etpan_error_strf_long_description(error,
        _("An error occurred while posting a message to account %s. "
            "Check your login, password and authentication type."),
        etpan_storage_get_id(param->storage));
    goto err;
  }
  else if (r == MAIL_ERROR_PARSE) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, "Parse error");
    etpan_error_strf_long_description(error, _("An error occurred while posting a message to account %s. The application did not understand what the server sent."),
        etpan_storage_get_id(param->storage));
    goto err;
  }
  else if (r == MAIL_ERROR_STREAM) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_STREAM);
    etpan_error_set_short_description(error, "Connection closed");
    etpan_error_strf_long_description(error,
        _("An error occurred while posting a message to account %s. "
            "The connection closed unexpectedly."),
        etpan_storage_get_id(param->storage));
    goto err;
  }
  else if (r == MAIL_ERROR_READONLY) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_READONLY);
    etpan_error_set_short_description(error, _("Group is read-only"));
    etpan_error_strf_long_description(error,
        _("An error occurred while posting a message to account %s. "
            "The newsgroup is read-only."),
        etpan_storage_get_id(param->storage));
    goto err;
  }
  else if (r == MAIL_ERROR_APPEND) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_APPEND);
    etpan_error_set_short_description(error, _("Could not post the message"));
    etpan_error_strf_long_description(error,
        _("An error occurred while posting a message to account %s. "
            "The message could not be posted to this group."),
        etpan_storage_get_id(param->storage));
    goto err;
  }
  else if (r != MAIL_NO_ERROR) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_APPEND);
    etpan_error_set_short_description(error, _("Unexpected error"));
    etpan_error_strf_long_description(error,
        _("An error occurred while posting a message to account %s. "
            "An unexpected error (libetpan: %i) occurred."),
        etpan_storage_get_id(param->storage), r);
    goto err;
  }
  
  result->error = NULL;
  
  return;
  
 err:
  result->error = error;
}

struct etpan_thread_op *
etpan_storage_news_append(struct etpan_thread_manager_app * manager,
    struct etpan_storage * storage,
    char * message, size_t length,
    void (* callback)(int, struct etpan_storage_news_append_msg_result *, void *),
    void * cb_data)
{
  struct etpan_thread_op * op;
  struct etpan_storage_news_append_msg_param * param;
  struct etpan_storage_news_append_msg_result * result;
  
  param = malloc(sizeof(* param));
  if (param == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  param->storage = storage;
  param->message = mmap_string_new_len(message, length);
  if (param->message == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  result = malloc(sizeof(* result));
  if (result == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  op = etpan_thread_op_new();
  
  op->param = param;
  op->result = result;
  
  op->cancellable = 0;
  op->run = storage_news_append_msg_run;
  op->callback = (void (*)(int, void *, void *)) callback;
  op->callback_data = cb_data;
  op->cleanup = storage_news_append_msg_cleanup;
  
  etpan_thread_manager_app_storage_schedule(manager, storage, op);
  
  return op;
}



/* append file */

struct etpan_storage_news_append_file_msg_param {
  struct etpan_storage * storage;
  char * filename;
};

static void
storage_news_append_file_msg_cleanup(struct etpan_thread_op * op)
{
  struct etpan_storage_news_append_file_msg_param * param;
  struct etpan_storage_news_append_msg_result * result;
  
  param = op->param;
  free(param->filename);
  free(op->param);
  
  result = op->result;
  ETPAN_ERROR_FREE(result->error);
  free(op->result);
}

static struct etpan_error * append_file(struct etpan_storage * storage,
    char * filename)
{
  int r;
  struct stat stat_buf;
  int fd;
  char * data;
  size_t len;
  struct storage_data * storage_data;
  struct etpan_error * error;
  
  r = stat(filename, &stat_buf);
  if (r < 0) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FILE);
    etpan_error_set_short_description(error, "File not found");
    etpan_error_strf_long_description(error,
        _("An error occurred while posting a message to account %s. "
            "The file was not found (%s)."),
        etpan_storage_get_id(storage), filename);
    goto err;
  }
  
  len = stat_buf.st_size;
  
  fd = open(filename, O_RDONLY);
  if (fd < 0) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FILE);
    etpan_error_set_short_description(error, "File could not be read");
    etpan_error_strf_long_description(error,
        _("An error occurred while posting a message to account %s. "
            "The file could not be read (%s)."),
        etpan_storage_get_id(storage), filename);
    goto err;
  }
  
  data = mmap(NULL, len, PROT_READ, MAP_PRIVATE, fd, 0);
  if (data == MAP_FAILED) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FILE);
    etpan_error_set_short_description(error, "File could not be read");
    etpan_error_strf_long_description(error,
        _("An error occurred while posting a message to account %s. "
            "The file could not be read (%s)."),
        etpan_storage_get_id(storage), filename);
    goto close;
  }
  
  storage_data = etpan_storage_get_data(storage);
  
  r = mailsession_append_message(storage_data->storage->sto_session,
      data, len);
  if (r == MAIL_ERROR_MEMORY)
    ETPAN_LOG_MEMORY_ERROR;
    
  if (r == MAIL_ERROR_LOGIN) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_AUTH);
    etpan_error_set_short_description(error, "Authentication failed");
    etpan_error_strf_long_description(error,
        _("An error occurred while posting a message to account %s. "
            "Check your login, password and authentication type."),
        etpan_storage_get_id(storage));
    goto unmap;
  }
  else if (r == MAIL_ERROR_PARSE) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, "Parse error");
    etpan_error_strf_long_description(error, _("An error occurred while posting a message to account %s. The application did not understand what the server sent."),
        etpan_storage_get_id(storage));
    goto unmap;
  }
  else if (r == MAIL_ERROR_STREAM) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_STREAM);
    etpan_error_set_short_description(error, "Connection closed");
    etpan_error_strf_long_description(error,
        _("An error occurred while posting a message to account %s. "
            "The connection closed unexpectedly."),
        etpan_storage_get_id(storage));
    goto unmap;
  }
  else if (r == MAIL_ERROR_READONLY) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_READONLY);
    etpan_error_set_short_description(error, _("Group is read-only"));
    etpan_error_strf_long_description(error,
        _("An error occurred while posting a message to account %s. "
            "The newsgroup is read-only."),
        etpan_storage_get_id(storage));
    goto unmap;
  }
  else if (r == MAIL_ERROR_APPEND) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_APPEND);
    etpan_error_set_short_description(error, _("Could not post the message"));
    etpan_error_strf_long_description(error,
        _("An error occurred while posting a message to account %s. "
            "The message could not be posted to this group."),
        etpan_storage_get_id(storage));
    goto unmap;
  }
  else if (r != MAIL_NO_ERROR) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_APPEND);
    etpan_error_set_short_description(error, _("Unexpected error"));
    etpan_error_strf_long_description(error,
        _("An error occurred while posting a message to account %s. "
            "An unexpected error (libetpan: %i) occurred."),
        etpan_storage_get_id(storage), r);
    goto unmap;
  }
  
  munmap(data, len);
  close(fd);
  
  return NULL;
  
 unmap:
  munmap(data, len);
 close:
  close(fd);
 err:
  return error;
}

static void
storage_news_append_file_msg_run(struct etpan_thread_op * op)
{
  struct etpan_storage_news_append_file_msg_param * param;
  struct etpan_storage_news_append_msg_result * result;
  struct etpan_error * error;
  
  param = op->param;
  result = op->result;
  
  error = etpan_storage_connect_nt(param->storage);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(param->storage, ERROR_DOMAIN_APPEND, error);
    result->error = new_error;
    ETPAN_ERROR_FREE(error);
    error = new_error;
    
    goto err;
  }
  
  error = append_file(param->storage, param->filename);
  if (error != NULL) {
    goto err;
  }
  
  result->error = NULL;
  return;
  
 err:
  result->error = error;
}

struct etpan_thread_op *
etpan_storage_news_append_file(struct etpan_thread_manager_app * manager,
    struct etpan_storage * storage,
    char * filename,
    void (* callback)(int, struct etpan_storage_news_append_msg_result *, void *),
    void * cb_data)
{
  struct etpan_thread_op * op;
  struct etpan_storage_news_append_file_msg_param * param;
  struct etpan_storage_news_append_msg_result * result;
  
  param = malloc(sizeof(* param));
  if (param == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  param->storage = storage;
  param->filename = strdup(filename);
  if (param->filename == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  result = malloc(sizeof(* result));
  if (result == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  op = etpan_thread_op_new();
  
  op->param = param;
  op->result = result;
  
  op->cancellable = 0;
  op->run = storage_news_append_file_msg_run;
  op->callback = (void (*)(int, void *, void *)) callback;
  op->callback_data = cb_data;
  op->cleanup = storage_news_append_file_msg_cleanup;
  
  etpan_thread_manager_app_storage_schedule(manager, storage, op);
  
  return op;
}

static char * get_sub_folder_location(struct etpan_storage * storage,
    struct etpan_folder * parent, char * name)
{
  (void) storage;
  
  if (parent == NULL) {
    char * sublocation;
    
    sublocation = strdup(name);
    if (sublocation == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    
    return sublocation;
  }
  else {
    return NULL;
  }
}

static struct etpan_error * rewrite_error(struct etpan_storage * storage,
    int domain,
    struct etpan_error * error)
{
  struct etpan_error * new_error;
  char * previous_long_description;
  
  new_error = etpan_error_new();
  etpan_error_set_code(new_error, etpan_error_get_code(error));
  etpan_error_set_short_description(new_error,
      etpan_error_get_short_description(error));
  previous_long_description = etpan_error_get_long_description(error);
  
  switch (domain) {
  case ERROR_DOMAIN_APPEND:
    etpan_error_strf_long_description(new_error,
        _("An error occurred while posting a message on account %s.\n"
            "%s"),
        etpan_storage_get_id(storage),
        previous_long_description);
    break;
    
  default:
    etpan_error_strf_long_description(new_error,
        _("An error occurred with account %s.\n"
            "%s"),
        etpan_storage_get_id(storage),
        previous_long_description);
    break;
  }
  
  return new_error;
}
