#include "etpan-folder-pop.h"

#include <stdlib.h>
#include <string.h>

#include "etpan-folder.h"
#include "etpan-storage.h"
#include "etpan-storage-pop.h"
#include "etpan-storage-pop-private.h"
#include "etpan-message-lep.h"
#include "etpan-error.h"
#include "etpan-lep.h"
#include "etpan-log.h"
#include "etpan-nls.h"

enum {
  ERROR_DOMAIN_CHECK,
  ERROR_DOMAIN_STATUS,
  ERROR_DOMAIN_EXPUNGE,
};

static struct etpan_error * rewrite_error(struct etpan_folder * folder,
    int domain,
    struct etpan_error * error);

static struct etpan_error * fetch_msg_list(struct etpan_folder * folder,
    chash * current_msg_list);

static struct etpan_error * check(struct etpan_folder * folder);

static struct etpan_error * status(struct etpan_folder * folder,
    unsigned int * p_count, unsigned int * p_unseen, unsigned int * p_recent);

static struct etpan_error * expunge(struct etpan_folder * folder);

static void free_data(struct etpan_folder * folder);

static struct etpan_error * connect(struct etpan_folder * folder);
static void disconnect(struct etpan_folder * folder);

static struct etpan_folder_driver pop_driver = {
  .name = "pop",
  
  .fetch_msg_list = fetch_msg_list,
  .append_msg = NULL,
  .check = check,
  .status = status,
  .expunge = expunge,
  .free_data = free_data,
  .connect = connect,
  .disconnect = disconnect,
  .is_sub_folder = NULL,
};

struct folder_data {
  /* threaded */
  struct mailfolder * folder;
  int connected;
};

static struct mailfolder * get_mailfolder(struct etpan_folder * folder)
{
  struct folder_data * data;

  data = etpan_folder_get_data(folder);
  return data->folder;
}

static struct etpan_error * connect(struct etpan_folder * folder)
{
  int r;
  struct folder_data * data;
  struct mailstorage * ep_storage;
  struct mailfolder * ep_folder;
  struct etpan_error * error;
  
  data = etpan_folder_get_data(folder);
  
  if (data->connected) {
    r = mailfolder_connect(data->folder);
    if (r == MAIL_ERROR_MEMORY)
      ETPAN_LOG_MEMORY_ERROR;
    
    if (r == MAIL_ERROR_CONNECT) {
      error = etpan_error_new();
      etpan_error_set_code(error, ERROR_CONNECT);
      etpan_error_set_short_description(error, "Could not connect");
      etpan_error_strf_long_description(error,
          _("Connection to mailbox %s failed. "
              "Check whether the mailbox exists."),
          etpan_folder_get_ui_path(folder));
      goto err;
    }
    else if (r == MAIL_ERROR_LOGIN) {
      error = etpan_error_new();
      etpan_error_set_code(error, ERROR_AUTH);
      etpan_error_set_short_description(error, "Authentication failed");
      etpan_error_strf_long_description(error,
          _("Connection to mailbox %s failed. "
              "Check your login, password and authentication type."),
          etpan_folder_get_ui_path(folder));
      goto err;
    }
    else if (r == MAIL_ERROR_PARSE) {
      error = etpan_error_new();
      etpan_error_set_code(error, ERROR_PARSE);
      etpan_error_set_short_description(error, "Parse error");
      etpan_error_strf_long_description(error, _("Connection to mailbox %s failed. The application did not understand what the server sent."),
          etpan_folder_get_ui_path(folder));
      goto err;
    }
    else if (r == MAIL_ERROR_STREAM) {
      error = etpan_error_new();
      etpan_error_set_code(error, ERROR_STREAM);
      etpan_error_set_short_description(error, "Connection closed");
      etpan_error_strf_long_description(error,
          _("Connection to mailbox %s failed. "
              "The connection closed unexpectedly."),
          etpan_folder_get_ui_path(folder));
      goto err;
    }
    else if (r != MAIL_NO_ERROR) {
      error = etpan_error_new();
      etpan_error_set_code(error, ERROR_CONNECT);
      etpan_error_set_short_description(error, _("Unexpected error"));
      etpan_error_strf_long_description(error,
          _("Connection to mailbox %s failed. "
              "An unexpected error (libetpan: %i) occurred."),
          etpan_folder_get_ui_path(folder), r);
      goto err;
    }
    
    r = mailfolder_noop(data->folder);
    if (r == MAIL_ERROR_MEMORY)
      ETPAN_LOG_MEMORY_ERROR;
    
    if (r == MAIL_ERROR_LOGIN) {
      error = etpan_error_new();
      etpan_error_set_code(error, ERROR_AUTH);
      etpan_error_set_short_description(error, "Authentication failed");
      etpan_error_strf_long_description(error,
          _("Connection to mailbox %s failed."
              "Check your login, password and authentication type."),
          etpan_folder_get_ui_path(folder));
      goto err;
    }
    else if (r == MAIL_ERROR_PARSE) {
      error = etpan_error_new();
      etpan_error_set_code(error, ERROR_PARSE);
      etpan_error_set_short_description(error, "Parse error");
      etpan_error_strf_long_description(error, _("Connection to mailbox %s failed. The application did not understand what the server sent."),
          etpan_folder_get_ui_path(folder));
      goto err;
    }
    else if (r == MAIL_ERROR_STREAM) {
      error = etpan_error_new();
      etpan_error_set_code(error, ERROR_STREAM);
      etpan_error_set_short_description(error, "Connection closed");
      etpan_error_strf_long_description(error,
          _("Connection to mailbox %s failed. "
              "The connection closed unexpectedly."),
          etpan_folder_get_ui_path(folder));
      goto err;
    }
    else if (r != MAIL_NO_ERROR) {
      error = etpan_error_new();
      etpan_error_set_code(error, ERROR_CONNECT);
      etpan_error_set_short_description(error, _("Unexpected error"));
      etpan_error_strf_long_description(error,
          _("Connection to mailbox %s failed. "
              "An unexpected error (libetpan: %i) occurred."),
          etpan_folder_get_ui_path(folder), r);
      goto err;
    }
    
    return NULL;
  }
  
  ep_storage = etpan_storage_pop_get_ep_storage(folder->storage);
  if (ep_storage == NULL) {
    ETPAN_LOG("storage has no ep_storage");
    etpan_crash();
  }
  
  ep_folder = mailfolder_new(ep_storage,
      etpan_folder_get_threaded_location(folder), NULL);
  if (ep_folder == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  r = mailfolder_connect(ep_folder);
  if (r == MAIL_ERROR_CONNECT) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CONNECT);
    etpan_error_set_short_description(error, "Could not connect");
    etpan_error_strf_long_description(error,
        _("Connection to mailbox %s failed. "
            "Check whether the mailbox exists."),
        etpan_folder_get_ui_path(folder));
    goto free_folder;
  }
  else if (r == MAIL_ERROR_LOGIN) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_AUTH);
    etpan_error_set_short_description(error, "Authentication failed");
    etpan_error_strf_long_description(error,
        _("Connection to mailbox %s failed. "
            "Check your login, password and authentication type."),
        etpan_folder_get_ui_path(folder));
    goto free_folder;
  }
  else if (r == MAIL_ERROR_PARSE) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, "Parse error");
    etpan_error_strf_long_description(error, _("Connection to mailbox %s failed. The application did not understand what the server sent."),
        etpan_folder_get_ui_path(folder));
    goto free_folder;
  }
  else if (r == MAIL_ERROR_STREAM) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_STREAM);
    etpan_error_set_short_description(error, "Connection closed");
    etpan_error_strf_long_description(error,
        _("Connection to mailbox %s failed. "
            "The connection closed unexpectedly."),
        etpan_folder_get_ui_path(folder));
    goto free_folder;
  }
  else if (r != MAIL_NO_ERROR) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CONNECT);
    etpan_error_set_short_description(error, _("Unexpected error"));
    etpan_error_strf_long_description(error,
        _("Connection to mailbox %s failed. "
            "An unexpected error (libetpan: %i) occurred."),
        etpan_folder_get_ui_path(folder), r);
    goto free_folder;
  }
  
  data->connected = 1;
  data->folder = ep_folder;
  
  etpan_set_lep_folder(folder, ep_folder);
  
  return NULL;
  
 free_folder:
  mailfolder_free(ep_folder);
 err:
  return error;
}

static void disconnect(struct etpan_folder * folder)
{
  struct folder_data * data;
  
  data = etpan_folder_get_data(folder);
  
  if (!data->connected)
    return;
  
  data->connected = 0;
  
  mailfolder_disconnect(data->folder);
  mailfolder_free(data->folder);
  data->folder = NULL;
  etpan_set_lep_folder(folder, NULL);
}

struct etpan_folder * etpan_folder_pop_new(void)
{
  struct etpan_folder * folder;
  struct folder_data * data;
  
  folder = etpan_folder_new();
  data = malloc(sizeof(* data));
  if (data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  data->folder = NULL;
  data->connected = 0;
  
  etpan_folder_set_data(folder, data);
  etpan_folder_set_driver(folder, &pop_driver);
  
  return folder;
}

static void free_data(struct etpan_folder * folder)
{
  struct folder_data * data;
  
  data = etpan_folder_get_data(folder);
  
  free(data);
}

static struct etpan_error * fetch_msg_list(struct etpan_folder * folder,
    chash * current_msg_list)
{
  struct mailfolder * ep_folder;
  
  ep_folder = get_mailfolder(folder);
  
  return etpan_lep_fetch_msg_list(folder, ep_folder, current_msg_list);
}

static struct etpan_error * check(struct etpan_folder * folder)
{
  struct mailfolder * ep_folder;
  int r;
  struct etpan_error * error;
  
  error = connect(folder);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(folder, ERROR_DOMAIN_CHECK, error);
    ETPAN_ERROR_FREE(error);
    
    return new_error;
  }
  
  ep_folder = get_mailfolder(folder);
  r = mailfolder_check(ep_folder);
  if (r == MAIL_ERROR_MEMORY)
    ETPAN_LOG_MEMORY_ERROR;
  
  if (r == MAIL_ERROR_PARSE) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, "Parse error");
    etpan_error_strf_long_description(error, _("An error occurred while saving mailbox %s information. The application did not understand what the server sent."),
        etpan_folder_get_ui_path(folder));
    goto err;
  }
  else if (r == MAIL_ERROR_STREAM) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_STREAM);
    etpan_error_set_short_description(error, "Connection closed");
    etpan_error_strf_long_description(error,
        _("An error occurred while saving mailbox %s information. "
            "The connection closed unexpectedly."),
        etpan_folder_get_ui_path(folder));
    goto err;
  }
  else if (r != MAIL_NO_ERROR) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CHECK);
    etpan_error_set_short_description(error, _("Unexpected error"));
    etpan_error_strf_long_description(error,
        _("An error occurred while saving mailbox %s information. "
            "An unexpected error (libetpan: %i) occurred."),
        etpan_folder_get_ui_path(folder), r);
    goto err;
  }
  
  return NULL;
  
 err:
  return error;
}

static struct etpan_error * status(struct etpan_folder * folder,
    unsigned int * p_count, unsigned int * p_unseen, unsigned int * p_recent)
{
  struct mailfolder * ep_folder;
  uint32_t count;
  uint32_t unseen;
  uint32_t recent;
  int r;
  struct etpan_error * error;
  
  error = connect(folder);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(folder, ERROR_DOMAIN_STATUS, error);
    ETPAN_ERROR_FREE(error);
    
    return new_error;
  }
  
  ep_folder = get_mailfolder(folder);
  r = mailfolder_status(ep_folder, &count, &recent, &unseen);
  if (r == MAIL_ERROR_MEMORY)
    ETPAN_LOG_MEMORY_ERROR;
  
  if (r == MAIL_ERROR_PARSE) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, "Parse error");
    etpan_error_strf_long_description(error, _("An error occurred while getting mailbox %s information. The application did not understand what the server sent."),
        etpan_folder_get_ui_path(folder));
    goto err;
  }
  else if (r == MAIL_ERROR_STREAM) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_STREAM);
    etpan_error_set_short_description(error, "Connection closed");
    etpan_error_strf_long_description(error,
        _("An error occurred while getting mailbox %s information. "
            "The connection closed unexpectedly."),
        etpan_folder_get_ui_path(folder));
    goto err;
  }
  else if (r != MAIL_NO_ERROR) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_STATUS);
    etpan_error_set_short_description(error, _("Unexpected error"));
    etpan_error_strf_long_description(error,
        _("An error occurred while getting mailbox %s information. "
            "An unexpected error (libetpan: %i) occurred."),
        etpan_folder_get_ui_path(folder), r);
    goto err;
  }
  
  * p_count = count;
  * p_unseen = unseen;
  * p_recent = recent;
  
  return NULL;
  
 err:
  return error;
}

static struct etpan_error * expunge(struct etpan_folder * folder)
{
  struct mailfolder * ep_folder;
  int r;
  struct etpan_error * error;
  
  error = connect(folder);
  if (error != NULL) {
    struct etpan_error * new_error;
    
    new_error = rewrite_error(folder, ERROR_DOMAIN_EXPUNGE, error);
    ETPAN_ERROR_FREE(error);
    
    return new_error;
  }
  
  ep_folder = get_mailfolder(folder);
  r = mailfolder_expunge(ep_folder);
  if (r == MAIL_ERROR_MEMORY)
    ETPAN_LOG_MEMORY_ERROR;
  
  if (r == MAIL_ERROR_PARSE) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, "Parse error");
    etpan_error_strf_long_description(error, _("An error occurred while saving mailbox %s information. The application did not understand what the server sent."),
        etpan_folder_get_ui_path(folder));
    goto err;
  }
  else if (r == MAIL_ERROR_STREAM) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_STREAM);
    etpan_error_set_short_description(error, "Connection closed");
    etpan_error_strf_long_description(error,
        _("An error occurred while saving mailbox %s information. "
            "The connection closed unexpectedly."),
        etpan_folder_get_ui_path(folder));
    goto err;
  }
  else if (r != MAIL_NO_ERROR) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_EXPUNGE);
    etpan_error_set_short_description(error, _("Unexpected error"));
    etpan_error_strf_long_description(error,
        _("An error occurred while saving mailbox %s information. "
            "An unexpected error (libetpan: %i) occurred."),
        etpan_folder_get_ui_path(folder), r);
    goto err;
  }
  
  return NULL;
  
 err:
  return error;
}

static struct etpan_error * rewrite_error(struct etpan_folder * folder,
    int domain,
    struct etpan_error * error)
{
  struct etpan_error * new_error;
  char * previous_long_description;
  
  new_error = etpan_error_new();
  etpan_error_set_code(new_error, etpan_error_get_code(error));
  etpan_error_set_short_description(new_error,
      etpan_error_get_short_description(error));
  previous_long_description = etpan_error_get_long_description(error);
  
  switch (domain) {
  case ERROR_DOMAIN_CHECK:
    etpan_error_strf_long_description(new_error,
        _("An error occurred while saving mailbox %s information.\n"
            "%s"),
        etpan_folder_get_ui_path(folder),
        previous_long_description);
    break;
  case ERROR_DOMAIN_STATUS:
    etpan_error_strf_long_description(new_error,
        _("An error occurred while getting information of mailbox %s.\n"
            "%s"),
        etpan_folder_get_ui_path(folder),
        previous_long_description);
    break;
  case ERROR_DOMAIN_EXPUNGE:
    etpan_error_strf_long_description(new_error,
        _("An error occurred while getting information of mailbox %s.\n"
            "%s"),
        etpan_folder_get_ui_path(folder),
        previous_long_description);
    break;
    
  default:
    etpan_error_strf_long_description(new_error,
        _("An error occurred with the mailbox %s.\n"
            "%s"),
        etpan_folder_get_ui_path(folder),
        previous_long_description);
    break;
  }
  
  return new_error;
}
