#include "etpan-storage-pop.h"
#include "etpan-storage-pop-private.h"

#include <libetpan/libetpan.h>
#include <stdlib.h>
#include <string.h>

#include "etpan-error.h"
#include "etpan-storage.h"
#include "etpan-storage-private.h"
#include "etpan-folder.h"
#include "etpan-folder-pop.h"
#include "etpan-lep.h"
#include "etpan-utils.h"
#include "etpan-connection-types.h"
#include "etpan-log.h"
#include "etpan-nls.h"

static struct etpan_error * setup(struct etpan_storage * storage);

static void free_data(struct etpan_storage * storage);

static struct etpan_error * fetch_folder_list(struct etpan_storage * storage,
    chash * folder_list);

static struct etpan_error * connect(struct etpan_storage * storage);

static void disconnect(struct etpan_storage * storage);

static struct etpan_storage_driver pop_driver = {
  .name = "pop",
  
  .network = 1,
  
  .setup = setup,
  .create_folder = NULL,
  .delete_folder = NULL,
  .rename_folder = NULL,
  .fetch_folder_list = fetch_folder_list,
  .connect = connect,
  .disconnect = disconnect,
  .free_data = free_data,
  .get_sub_folder_location = NULL,
  .allows_sub_folders = NULL,
};

struct storage_data {
  char * hostname;
  int port;
  int connection_type;
  int auth_type;
  char * username;
  char * password;
  char * command;
  
  /* threaded */
  struct mailstorage * storage;
  int connected;
  char * threaded_hostname;
  int threaded_port;
  int threaded_connection_type;
  int threaded_auth_type;
  char * threaded_username;
  char * threaded_password;
  char * threaded_command;
};

static struct etpan_error * ep_storage_connect(struct etpan_storage * storage)
{
  int r;
  struct storage_data * data;
  struct etpan_error * error;
  
  data = etpan_storage_get_data(storage);
  
  r = mailstorage_connect(data->storage);
  if (r == MAIL_ERROR_MEMORY)
    ETPAN_LOG_MEMORY_ERROR;
    
  if (r == MAIL_ERROR_CONNECT) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CONNECT);
    etpan_error_set_short_description(error, "Could not connect");
    etpan_error_strf_long_description(error,
        _("Connection to account %s failed."
            "The connection to POP server %s, port %i could not be established.\n"
            "Check the name of server and port."),
        etpan_storage_get_id(storage),
        etpan_storage_pop_get_hostname(storage),
        etpan_storage_pop_get_port(storage));
    goto err;
  }
  else if (r == MAIL_ERROR_LOGIN) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_AUTH);
    etpan_error_set_short_description(error, "Authentication failed");
    etpan_error_strf_long_description(error,
        _("Connection to account %s failed."
            "Check your login, password and authentication type."),
        etpan_storage_get_id(storage));
    goto err;
  }
  else if (r == MAIL_ERROR_PARSE) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, "Parse error");
    etpan_error_strf_long_description(error, _("Connection to account %s failed. The application did not understand what the server sent."),
        etpan_storage_get_id(storage));
    goto err;
  }
  else if (r == MAIL_ERROR_STREAM) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_STREAM);
    etpan_error_set_short_description(error, "Connection closed");
    etpan_error_strf_long_description(error,
        _("Connection to account %s failed."
            "The connection closed unexpectedly."),
        etpan_storage_get_id(storage));
    goto err;
  }
  else if (r != MAIL_NO_ERROR) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CONNECT);
    etpan_error_set_short_description(error, _("Unexpected error"));
    etpan_error_strf_long_description(error,
        _("Connection to account %s failed."
            "An unexpected error (libetpan: %i) occurred."),
        etpan_storage_get_id(storage), r);
    goto err;
  }
    
  error = NULL;

 err:
  return error;
}

static struct etpan_error * connect(struct etpan_storage * storage)
{
  int r;
  struct storage_data * data;
  struct mailstorage * ep_storage;
  int ep_connection_type;
  int ep_auth_type;
  int cached;
  struct etpan_error * error;
  
  data = etpan_storage_get_data(storage);
  
  if (data->connected) {
    error = ep_storage_connect(storage);
    if (error != NULL) {
      goto err;
    }
    
    r = mailsession_noop(data->storage->sto_session);
    if (r == MAIL_ERROR_MEMORY)
      ETPAN_LOG_MEMORY_ERROR;
    
    if (r == MAIL_ERROR_PARSE) {
      error = etpan_error_new();
      etpan_error_set_code(error, ERROR_PARSE);
      etpan_error_set_short_description(error, "Parse error");
      etpan_error_strf_long_description(error, _("Connection to account %s failed. The application did not understand what the server sent."),
          etpan_storage_get_id(storage));
      goto err;
    }
    else if (r == MAIL_ERROR_STREAM) {
      error = etpan_error_new();
      etpan_error_set_code(error, ERROR_STREAM);
      etpan_error_set_short_description(error, "Connection closed");
      etpan_error_strf_long_description(error,
          _("Connection to account %s failed."
              "The connection closed unexpectedly."),
          etpan_storage_get_id(storage));
      goto err;
    }
    else if (r != MAIL_NO_ERROR) {
      error = etpan_error_new();
      etpan_error_set_code(error, ERROR_CONNECT);
      etpan_error_set_short_description(error, _("Unexpected error"));
      etpan_error_strf_long_description(error,
          _("Connection to account %s failed."
              "An unexpected error (libetpan: %i) occurred."),
          etpan_storage_get_id(storage), r);
      goto err;
    }
    
    return NULL;
  }
  
  if ((data->threaded_command == NULL) && (data->threaded_hostname == NULL)) {
    error = etpan_error_internal_strf(_("Account information of %s are not valid. Neither command nor hostname are defined."),
        etpan_storage_get_id(storage));
    
    goto err;
  }
  
  if (data->threaded_command == NULL) {
    switch (data->threaded_connection_type) {
    case ETPAN_CONNECTION_STARTTLS:
      ep_connection_type = CONNECTION_TYPE_STARTTLS;
      break;
    case ETPAN_CONNECTION_TRY_STARTTLS:
      ep_connection_type = CONNECTION_TYPE_TRY_STARTTLS;
      break;
    case ETPAN_CONNECTION_TLS:
      ep_connection_type = CONNECTION_TYPE_TLS;
      break;
    default:
      ep_connection_type = CONNECTION_TYPE_PLAIN;
      break;
    }
  }
  else {
    switch (data->threaded_connection_type) {
    case ETPAN_CONNECTION_STARTTLS:
      ep_connection_type = CONNECTION_TYPE_COMMAND_STARTTLS;
      break;
    case ETPAN_CONNECTION_TRY_STARTTLS:
      ep_connection_type = CONNECTION_TYPE_COMMAND_TRY_STARTTLS;
      break;
    case ETPAN_CONNECTION_TLS:
      ep_connection_type = CONNECTION_TYPE_COMMAND_TLS;
      break;
    default:
      ep_connection_type = CONNECTION_TYPE_COMMAND;
      break;
    }
  }
  
  ep_auth_type = POP3_AUTH_TYPE_PLAIN;
  
  ep_storage = mailstorage_new(NULL);
  if (ep_storage == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  cached = 1;
  if (etpan_storage_get_threaded_cache_path(storage) == NULL)
    cached = 0;
  if (etpan_storage_get_threaded_flags_path(storage) == NULL)
    cached = 0;
  
  r = pop3_mailstorage_init(ep_storage,
    data->threaded_hostname, data->threaded_port,
    data->threaded_command,
    ep_connection_type, ep_auth_type,
    data->threaded_username, data->threaded_password,
    cached, etpan_storage_get_threaded_cache_path(storage),
      etpan_storage_get_threaded_flags_path(storage));
  if (r != MAIL_NO_ERROR) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  data->storage = ep_storage;
  error = ep_storage_connect(storage);
  if (error != NULL) {
    goto free_storage;
  }
  
  data->connected = 1;
  
  return NULL;
  
 free_storage:
  mailstorage_free(ep_storage);
  data->storage = NULL;
 err:
  return error;
}

static void disconnect(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  
  if (!data->connected)
    return;
  
  data->connected = 0;
  
  mailstorage_disconnect(data->storage);
  mailstorage_free(data->storage);
  data->storage = NULL;
}

struct etpan_storage * etpan_storage_pop_new(void)
{
  struct storage_data * data;
  struct etpan_storage * storage;
  
  storage = etpan_storage_new();
  data = malloc(sizeof(* data));
  if (data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  data->hostname = NULL;
  data->port = 0;
  data->connection_type = ETPAN_CONNECTION_PLAIN;
  data->auth_type = ETPAN_AUTH_PLAIN;
  data->username = NULL;
  data->password = NULL;
  data->command = NULL;
  data->storage = NULL;
  data->connected = 0;
  data->threaded_hostname = NULL;
  data->threaded_port = 0;
  data->threaded_connection_type = ETPAN_CONNECTION_PLAIN;
  data->threaded_auth_type = ETPAN_AUTH_PLAIN;
  data->threaded_username = NULL;
  data->threaded_password = NULL;
  data->threaded_command = NULL;
  
  etpan_storage_set_data(storage, data);
  etpan_storage_set_driver(storage, &pop_driver);
  
  return storage;
}

static struct etpan_error * fetch_folder_list(struct etpan_storage * storage,
    chash * folder_list)
{
  int r;
  struct storage_data * data;
  struct etpan_folder * folder;
  char vpath[PATH_MAX];
  chashdatum key;
  chashdatum value;
  
  data = etpan_storage_get_data(storage);
  
  snprintf(vpath, sizeof(vpath), "%s/inbox", etpan_storage_get_id(storage));
  
  folder = etpan_folder_pop_new();
  etpan_folder_set_storage(folder, storage);
  etpan_folder_set_ui_path(folder, vpath);
  etpan_folder_set_location(folder, "INBOX");
  
  key.data = "inbox";
  key.len = strlen("inbox") + 1;
  value.data = folder;
  value.len = 0;
  r = chash_set(folder_list, &key, &value, NULL);
  if (r < 0) {
    etpan_folder_unref(folder);
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  return NULL;
}

static void free_data(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  
  free(data->threaded_command);
  free(data->threaded_password);
  free(data->threaded_username);
  free(data->threaded_hostname);
  
  free(data->command);
  free(data->password);
  free(data->username);
  free(data->hostname);
  free(data);
}



void etpan_storage_pop_set_hostname(struct etpan_storage * storage,
    char * hostname)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  if (hostname != data->hostname) {
    free(data->hostname);
    if (hostname != NULL) {
      data->hostname = strdup(hostname);
      if (data->hostname == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      data->hostname = NULL;
  }
}

char * etpan_storage_pop_get_hostname(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  return data->hostname;
}

void etpan_storage_pop_set_port(struct etpan_storage * storage, int port)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  data->port = port;
}

int etpan_storage_pop_get_port(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  return data->port;
}

void etpan_storage_pop_set_connection_type(struct etpan_storage * storage,
    int connection_type)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  data->connection_type = connection_type;
}

int etpan_storage_pop_get_connection_type(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  return data->connection_type;
}

void etpan_storage_pop_set_auth_type(struct etpan_storage * storage,
    int auth_type)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  data->auth_type = auth_type;
}

int etpan_storage_pop_get_auth_type(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  return data->auth_type;
}

void etpan_storage_pop_set_username(struct etpan_storage * storage,
    char * username)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  if (username != data->username) {
    free(data->username);
    if (username != NULL) {
      data->username = strdup(username);
      if (data->username == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      data->username = NULL;
  }
}

char * etpan_storage_pop_get_username(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  return data->username;
}

void etpan_storage_pop_set_password(struct etpan_storage * storage,
    char * password)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  if (password != data->password) {
    free(data->password);
    if (password != NULL) {
      data->password = strdup(password);
      if (data->password == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      data->password = NULL;
  }
}

char * etpan_storage_pop_get_password(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  return data->password;
}


void etpan_storage_pop_set_command(struct etpan_storage * storage,
    char * command)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  if (command != data->command) {
    free(data->command);
    if (command != NULL) {
      data->command = strdup(command);
      if (data->command == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      data->command = NULL;
  }
}

char * etpan_storage_pop_get_command(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  return data->command;
}

static struct etpan_error * setup(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  
  free(data->threaded_hostname);
  if (data->hostname != NULL) {
    data->threaded_hostname = strdup(data->hostname);
    if (data->threaded_hostname == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  else {
    data->threaded_hostname = NULL;
  }
  
  data->threaded_port = data->port;
  data->threaded_connection_type = data->connection_type;
  data->threaded_auth_type = data->auth_type;
  
  free(data->threaded_username);
  if (data->username != NULL) {
    data->threaded_username = strdup(data->username);
    if (data->threaded_username == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  else {
    data->threaded_username = NULL;
  }
  
  free(data->threaded_password);
  if (data->password != NULL) {
    data->threaded_password = strdup(data->password);
    if (data->threaded_password == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  else {
    data->threaded_password = NULL;
  }
  
  free(data->threaded_command);
  if (data->command != NULL) {
    data->threaded_command = strdup(data->command);
    if (data->threaded_command == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  else {
    data->threaded_command = NULL;
  }
  
  return NULL;
}

struct mailstorage *
etpan_storage_pop_get_ep_storage(struct etpan_storage * storage)
{
  struct storage_data * data;
  
  data = etpan_storage_get_data(storage);
  return data->storage;
}
