#include <stdlib.h>
#include <signal.h>
#include <limits.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <locale.h>
#include <time.h>

#include "etpan-gtk-ui.h"
#include "etpan-backend.h"

enum {
  STATE_INIT,
  STATE_SETUP,
  STATE_RUNNING,
  STATE_STOPPING,
  STATE_UNSETUP,
  STATE_DONE,
};

struct etpan_app {
  int state;
};

static struct etpan_error * init(struct etpan_app * app, int * argc, char *** argv);
static void setup(struct etpan_app * app);
static void run(struct etpan_app * app);
static void quit_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);
static void quit(struct etpan_app * app);
static void stop(struct etpan_app * app);
static void stop_callback(void * cb_data);
static void unsetup(struct etpan_app * app);
static void done(struct etpan_app * app);

static struct etpan_error * init(struct etpan_app * app, int * argc, char *** argv)
{
  struct etpan_error * error;
  
  app->state = STATE_INIT;
  
  error = etpan_backend_init();
  if (error != NULL) {
    goto err;
  }
  error = etpan_gtk_ui_init(argc, argv);
  if (error != NULL) {
    goto done_backend;
  }
  
  return NULL;
  
 done_backend:
  etpan_backend_done();
 err:
  return error;
}

static void setup(struct etpan_app * app)
{
  app->state = STATE_SETUP;
  
  etpan_backend_setup();
  etpan_gtk_ui_setup();
  
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_GTK_UI_MAIN_CLOSE_SIGNAL, NULL, app,
      quit_handler);
}

static void run(struct etpan_app * app)
{
  app->state = STATE_RUNNING;
  
  etpan_gtk_ui_run();
}

static void quit_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_app * app;
  (void) signal_name;
  (void) signal_data;
  (void) sender;
  
  app = user_data;
  
  quit(app);
}

static void quit(struct etpan_app * app)
{
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_GTK_UI_MAIN_CLOSE_SIGNAL, NULL, app,
      quit_handler);
  
  etpan_gtk_ui_show_quit_panel();
  etpan_gtk_ui_stop();
  
  /* try to quit */
  stop(app);
}

static void stop(struct etpan_app * app)
{
  etpan_backend_stop(stop_callback, app);
  app->state = STATE_STOPPING;
}

static void stop_callback(void * cb_data)
{
  struct etpan_app * app;
  
  app = cb_data;
  etpan_gtk_ui_quit();
}

static void unsetup(struct etpan_app * app)
{
  app->state = STATE_UNSETUP;
  
  etpan_gtk_ui_unsetup();
  etpan_backend_unsetup();
}

static void done(struct etpan_app * app)
{
  app->state = STATE_DONE;
  
  etpan_gtk_ui_done();
  etpan_backend_done();
}

int main(int argc, char *argv[])
{
  struct etpan_error * error;
  struct etpan_app app;
  
  error = init(&app, &argc, &argv);
  if (error != NULL) {
    etpan_error_log(error);
    ETPAN_ERROR_FREE(error);
    exit(EXIT_FAILURE);
  }
  
  setup(&app);
  
  /* running */
  
  run(&app);
  
  /* done */
  unsetup(&app);
  
  done(&app);
  
  exit(EXIT_SUCCESS);
}
