#include "etpan-bookmark-view.h"

#include <stdlib.h>
#include <string.h>
#include "etpan-backend.h"
#include "etpan-bookmark.h"
#include "etpan-message-list.h"

struct etpan_bookmark_view * etpan_bookmark_view_new(void)
{
  struct etpan_bookmark_view * bookmark_view;
  GtkWidget * bookmark_widget;
  
  bookmark_view = malloc(sizeof(* bookmark_view));
  if (bookmark_view == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  bookmark_view->button_list = carray_new(8);
  if (bookmark_view->button_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  bookmark_widget = gtk_toolbar_new();
  gtk_widget_show(bookmark_widget);
  gtk_toolbar_set_style(GTK_TOOLBAR(bookmark_widget), GTK_TOOLBAR_BOTH);
  
  bookmark_view->bookmark = bookmark_widget;
  
  bookmark_view->message_list = NULL;
  
  return bookmark_view;
}

void etpan_bookmark_view_free(struct etpan_bookmark_view * bookmark_view)
{
  etpan_bookmark_view_clear(bookmark_view);
  carray_free(bookmark_view->button_list);
  gtk_widget_destroy(bookmark_view->bookmark);
  free(bookmark_view);
}

GtkWidget * etpan_bookmark_view_get_main_widget(struct etpan_bookmark_view *
    bookmark_view)
{
  return bookmark_view->bookmark;
}

struct button_item {
  GtkWidget * button;
  struct etpan_bookmark * bookmark;
  struct etpan_bookmark_view * bookmark_view;
  gulong signal_id;
};

static struct button_item * button_item_new(void)
{
  struct button_item * item;
  
  item = malloc(sizeof(* item));
  if (item == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  item->button = NULL;
  item->bookmark = NULL;
  
  return item;
}

static void button_item_free(struct button_item * item)
{
  free(item);
}

static void button_item_set_button(struct button_item * item,
    GtkWidget * button)
{
  item->button = button;
}

static GtkWidget * button_item_get_button(struct button_item * item)
{
  return item->button;
}

static void button_item_set_bookmark(struct button_item * item,
    struct etpan_bookmark * bookmark)
{
  item->bookmark = bookmark;
}

static void button_item_action(struct etpan_bookmark_view * bookmark_view,
    struct button_item * item);

static void button_clicked(GtkToolButton * toolbutton,
    gpointer user_data)
{
  struct button_item * item;
  (void) toolbutton;
  
  item = (struct button_item *) user_data;
  button_item_action(item->bookmark_view, item);
}

static void button_item_register_action(struct etpan_bookmark_view *
    bookmark_view,
    struct button_item * item)
{
  item->bookmark_view = bookmark_view;
  
  item->signal_id = g_signal_connect(item->button, "clicked",
      (GCallback) button_clicked, (gpointer) item);
}

static void button_item_unregister_action(struct etpan_bookmark_view *
    bookmark_view,
    struct button_item * item)
{
  (void) bookmark_view;
  
  g_signal_handler_disconnect(item->button, item->signal_id);
}

static void button_item_action(struct etpan_bookmark_view * bookmark_view,
    struct button_item * item)
{
  struct etpan_folder * folder;

  if (item->bookmark == NULL)
    return;
  
  folder = etpan_bookmark_get_folder(etpan_account_manager_get_default(),
      item->bookmark);
  if (folder == NULL)
    return;

  if (bookmark_view->message_list == NULL)
    return;
  
  etpan_message_list_set_folder(bookmark_view->message_list, folder);
}

void etpan_bookmark_view_add_item(struct etpan_bookmark_view * bookmark_view,
    struct etpan_bookmark * bookmark)
{
  GtkWidget * button;
  int r;
  struct button_item * item;
  char * label;
  
  label = etpan_bookmark_get_display_name(bookmark);
  if (label == NULL)
    label = _("Unknown");
  
  button = (GtkWidget *) gtk_tool_button_new(NULL, label);
  gtk_widget_show(button);
  gtk_container_add(GTK_CONTAINER(bookmark_view->bookmark), button);
  
  item = button_item_new();
  if (item == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  button_item_set_button(item, button);
  button_item_set_bookmark(item, bookmark);
  
  button_item_register_action(bookmark_view, item);
  
  r = carray_add(bookmark_view->button_list, item, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

void etpan_bookmark_view_clear(struct etpan_bookmark_view * bookmark_view)
{
  unsigned int i;
  int r;
  
  for(i = 0 ; i < carray_count(bookmark_view->button_list) ; i ++) {
    struct button_item * item;
    
    item = carray_get(bookmark_view->button_list, i);
    button_item_unregister_action(bookmark_view, item);
    gtk_widget_destroy(button_item_get_button(item));
    button_item_free(item);
  }
  
  r = carray_set_size(bookmark_view->button_list, 0);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

void etpan_bookmark_view_set_message_list(struct etpan_bookmark_view * bookmark_view, struct etpan_message_list * msg_list)
{
  bookmark_view->message_list = msg_list;
}

void etpan_bookmark_view_setup(struct etpan_bookmark_view * bookmark_view)
{
  struct etpan_bookmark_manager * manager;
  carray * bookmark_list;
  unsigned int i;
  
  manager = etpan_bookmark_manager_get_default();
  bookmark_list = etpan_bookmark_manager_get_list(manager);
  
  for(i = 0 ; i < carray_count(bookmark_list) ; i ++) {
    struct etpan_bookmark * item;
    
    item = carray_get(bookmark_list, i);
    etpan_bookmark_view_add_item(bookmark_view, item);
  }
}

void etpan_bookmark_view_unsetup(struct etpan_bookmark_view * bookmark_view)
{
  etpan_bookmark_view_clear(bookmark_view);
}
