#include "etpan-completion.h"

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <stdlib.h>

#include "etpan-backend.h"
#include "etpan-gtk-tree-model.h"

/* completion - begin */

static unsigned int address_get_n_columns(struct etpan_gtk_tree_data_source *
    datasource)
{
  (void) datasource;
  
  return 4;
}

static GType address_get_column_type(struct etpan_gtk_tree_data_source * datasource,
    unsigned int column_index)
{
  static GType column_types[4] = {
    G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING,
    G_TYPE_POINTER
  };
  (void) datasource;
  
  return column_types[column_index];
}

static unsigned int address_get_children_count(struct etpan_gtk_tree_data_source *
    datasource, void * item)
{
  struct etpan_completion * completion;
  
  completion = datasource->data;
  
  if (item == NULL) {
    carray * list;
    
    list = etpan_abook_request_get_result(completion->abook_request);
    return carray_count(list);
  }
  else {
    return 0;
  }
}

static int address_item_has_child(struct etpan_gtk_tree_data_source *
    datasource, void * item)
{
  struct etpan_completion * completion;
  
  completion = datasource->data;
  
  if (item == NULL) {
    carray * list;
    
    list = etpan_abook_request_get_result(completion->abook_request);
    return (carray_count(list) > 0);
  }
  else
    return 0;
}

static void * address_get_child_item(struct etpan_gtk_tree_data_source *
    datasource, void * item, unsigned int index)
{
  struct etpan_completion * completion;
  carray * list;
  struct etpan_abook_entry * entry;
  (void) item;
  
  completion = datasource->data;
  
  list = etpan_abook_request_get_result(completion->abook_request);
  
  entry = carray_get(list, index);
  
  return entry;
}

static void address_get_item_value(struct etpan_gtk_tree_data_source *
    datasource, void * item, unsigned int column_index,
    GValue * value)
{
  struct etpan_completion * completion;
  struct etpan_abook_entry * entry;
  
  completion = datasource->data;
  
  entry = item;
  
  switch (column_index) {
  case 0:
    {
      g_value_init(value, G_TYPE_STRING);
      g_value_set_string(value, entry->fullname);
    }
    break;
  case 1:
    {
      g_value_init(value, G_TYPE_STRING);
      g_value_set_string(value, entry->address);
    }
    break;
  case 2:
    {
      char str[1024];
      char entry_str[1024];
      
      etpan_abook_entry_str(entry_str, sizeof(entry_str), entry);
      
      switch (entry->match_type) {
      case ETPAN_ABOOK_MATCH_ADDRESS:
        snprintf(str, sizeof(str), "%s (%s)", entry->address, entry_str);
        break;
        
      case ETPAN_ABOOK_MATCH_FIRSTNAME:
        snprintf(str, sizeof(str), "%s (%s)", entry->firstname, entry_str);
        break;
      
      case ETPAN_ABOOK_MATCH_LASTNAME:
        snprintf(str, sizeof(str), "%s (%s)", entry->lastname, entry_str);
        break;
        
      case ETPAN_ABOOK_MATCH_NICKNAME:
        snprintf(str, sizeof(str), "%s (%s)", entry->nickname, entry_str);
        break;
        
      default:
      case ETPAN_ABOOK_MATCH_NONE:
      case ETPAN_ABOOK_MATCH_FULLNAME:
        snprintf(str, sizeof(str), "%s", entry_str);
        break;
      }
      
      g_value_init(value, G_TYPE_STRING);
      g_value_set_string(value, str);
    }
    break;
  case 3:
    {
      g_value_init(value, G_TYPE_POINTER);
      g_value_set_pointer(value, entry);
    }
    break;
  }
}

/* completion - end */


static void size_requested(GtkWidget * widget,
    GtkRequisition * requisition,
    gpointer user_data);

static void text_changed(GtkTextBuffer * textbuffer,
    gpointer user_data);

static gboolean
completion_key_press(GtkWidget * widget,
    GdkEventKey * event,
    gpointer user_data);

static gboolean
completion_button_press(GtkWidget * widget,
    GdkEventButton * event,
    gpointer user_data);

static gboolean
entry_key_press(GtkWidget * widget,
    GdkEventKey * event,
    gpointer user_data);

static gboolean
completion_list_button_press(GtkWidget * widget,
    GdkEventButton * event,
    gpointer user_data);

struct etpan_completion * etpan_completion_new(void)
{
  GtkWidget * window;
  GtkWidget * treeview;
  GtkWidget * scrolledwindow;
  etpan_gtk_tree_model * completion_model;
  struct etpan_completion * completion;
  GtkWidget * scrolledtext;
  GtkWidget * textview;
  GtkTextBuffer * buffer;
  GtkTreeSelection * sel;
  GtkTreeViewColumn * col_text;
  GtkCellRenderer * col_text_renderer;
  struct etpan_abook_request * abook_request;
  
  completion = malloc(sizeof(* completion));
  if (completion == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  abook_request = etpan_abook_request_new(etpan_abook_manager_get_default());
  if (abook_request == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  completion->abook_request = abook_request;
  
  window = gtk_window_new(GTK_WINDOW_POPUP);
  
  scrolledwindow = gtk_scrolled_window_new(NULL, NULL);
  gtk_widget_show(scrolledwindow);
  gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrolledwindow),
      GTK_SHADOW_ETCHED_IN);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledwindow),
      GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
  
  treeview = gtk_tree_view_new();
  
  completion_model = etpan_gtk_tree_model_new();
  completion->completion_datasource.data = completion;
  completion->completion_datasource.get_n_columns = address_get_n_columns;
  completion->completion_datasource.get_column_type = address_get_column_type;
  completion->completion_datasource.get_children_count = address_get_children_count;
  completion->completion_datasource.item_has_child = address_item_has_child;
  completion->completion_datasource.get_child_item = address_get_child_item;
  completion->completion_datasource.get_item_value = address_get_item_value;
  
  etpan_gtk_tree_model_set_datasource(completion_model,
      &completion->completion_datasource);
  
  gtk_tree_view_set_model(GTK_TREE_VIEW(treeview),
      GTK_TREE_MODEL(completion_model));
  
  col_text = gtk_tree_view_column_new();
  col_text_renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_column_pack_start(col_text, col_text_renderer, TRUE);
  gtk_tree_view_column_set_attributes(col_text, col_text_renderer,
      "text", 2, NULL);
  gtk_tree_view_column_set_sizing(col_text, GTK_TREE_VIEW_COLUMN_AUTOSIZE);
  gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), col_text);
  
  gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(treeview), FALSE);
  gtk_tree_view_set_hover_selection(GTK_TREE_VIEW(treeview), TRUE);
  
  sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
  gtk_tree_selection_set_mode (sel, GTK_SELECTION_BROWSE);
  
  gtk_widget_show(treeview);
  gtk_container_add(GTK_CONTAINER(scrolledwindow), treeview);
  gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(treeview), FALSE);
  gtk_tree_view_set_enable_search(GTK_TREE_VIEW(treeview), FALSE);
  
  gtk_container_add(GTK_CONTAINER(window), scrolledwindow);
  
  scrolledtext = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrolledtext),
      GTK_SHADOW_IN);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledtext),
      GTK_POLICY_NEVER, GTK_POLICY_NEVER);
  
  textview = gtk_text_view_new();
  gtk_text_view_set_left_margin(GTK_TEXT_VIEW(textview), 2);
  gtk_text_view_set_right_margin(GTK_TEXT_VIEW(textview), 2);
  gtk_text_view_set_pixels_above_lines(GTK_TEXT_VIEW(textview), 2);
  gtk_text_view_set_pixels_below_lines(GTK_TEXT_VIEW(textview), 2);
  
  buffer = gtk_text_buffer_new(NULL);
  gtk_text_view_set_buffer(GTK_TEXT_VIEW(textview), buffer);
  gtk_text_view_set_accepts_tab(GTK_TEXT_VIEW(textview), FALSE);
  gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(textview), GTK_WRAP_WORD_CHAR);
  gtk_widget_show(textview);
  gtk_container_add(GTK_CONTAINER(scrolledtext), textview);
  
  completion->window = window;
  completion->treeview = treeview;
  completion->scrolledwindow = scrolledwindow;
  completion->textview = textview;
  completion->buffer = buffer;
  completion->scrolledtext = scrolledtext;
  completion->col_text = col_text;
  completion->completion_model = completion_model;
  completion->visible = 0;
  completion->requesting = 0;
  completion->scheduled = 0;
  
  return completion;
}

void etpan_completion_free(struct etpan_completion * completion)
{
  g_object_unref(completion->buffer);
  gtk_widget_destroy(completion->textview);
  gtk_widget_destroy(completion->scrolledtext);
  
  g_object_unref(completion->completion_model);
  gtk_widget_destroy(completion->treeview);
  gtk_widget_destroy(completion->scrolledwindow);
  
  gtk_widget_destroy(completion->window);
  
  etpan_abook_request_free(completion->abook_request);
  free(completion);
}

GtkWidget * etpan_completion_get_main_widget(struct etpan_completion * completion)
{
  return completion->scrolledtext;
}

GtkWidget * etpan_completion_get_textview(struct etpan_completion * completion)
{
  return completion->textview;
}

static void etpan_completion_hide_completion(struct etpan_completion * completion);

static void etpan_completion_show_completion(struct etpan_completion * completion)
{
  gint x;
  gint y;
  gint width;
  gint height;
  unsigned int items;
  GtkRequisition req;
  int was_visible;
  
  was_visible = 0;
  if (completion->visible) {
    was_visible = 1;
  }
  items = gtk_tree_model_iter_n_children(GTK_TREE_MODEL(completion->completion_model), NULL);
  
  if (items == 0) {
    etpan_completion_hide_completion(completion);
    return;
  }
  
  completion->visible = 1;
  
  gdk_window_get_deskrelative_origin(completion->scrolledtext->window, &x, &y);
  x += completion->scrolledtext->allocation.x;
  y += completion->scrolledtext->allocation.y;
  
  width = completion->scrolledtext->allocation.width;
  height = completion->scrolledtext->allocation.height;
  
  gtk_window_move(GTK_WINDOW(completion->window), x, y + height);
  
  gtk_tree_view_column_cell_get_size (completion->col_text, NULL,
                                      NULL, NULL, NULL, &height);
  
  if (!was_visible) {
    if (items > 0) {
      GtkTreePath * path;
      
      path = gtk_tree_path_new_first();
      gtk_tree_view_set_cursor(GTK_TREE_VIEW(completion->treeview),
          path, NULL, FALSE);
      gtk_tree_path_free(path);
    }
  }
  
  if (items > 15)
    items = 15;
  
  gtk_widget_set_size_request(completion->treeview, width, height * items);
  gtk_widget_size_request(completion->window, &req);
  gtk_widget_set_size_request(completion->window, width, req.height);
  gtk_widget_show(completion->window);
  gtk_grab_add(completion->window);
  gdk_pointer_grab(completion->window->window, TRUE,
      GDK_BUTTON_PRESS_MASK |
      GDK_BUTTON_RELEASE_MASK |
      GDK_POINTER_MOTION_MASK,
      NULL, NULL, GDK_CURRENT_TIME);
}

static void etpan_completion_hide_completion(struct etpan_completion * completion)
{
  if (!completion->visible)
    return;
  
  gdk_pointer_ungrab(GDK_CURRENT_TIME);
  gtk_grab_remove(completion->window);
  gtk_widget_hide(completion->window);
  completion->visible = 0;
}

static void get_completion_value(struct etpan_completion * completion,
    unsigned int * p_begin, unsigned int * p_end);

static void request_completion(struct etpan_completion * completion);
static void schedule_request(struct etpan_completion * completion);

static void size_requested(GtkWidget * widget,
    GtkRequisition * requisition,
    gpointer user_data)
{
  struct etpan_completion * completion;
  gint width;
  gint height;
  GtkRequisition r;
  GtkRequisition r2;
  guint border;
  GtkTextView * textview;
  GtkTextIter iter;
  unsigned int line_count;
  gint line_height;
  (void) widget;
  (void) requisition;
  
  completion = user_data;
  
  textview = GTK_TEXT_VIEW(completion->textview);
  gtk_widget_size_request(completion->textview, &r);
  gtk_widget_size_request(completion->scrolledtext, &r2);
  border = 2;
  
  width = r2.width;
  height = r.height + border * 2;
 
  gtk_text_buffer_get_iter_at_offset(completion->buffer,
      &iter, 0);
 
  line_count = 1;
  while (1) {
    if (!gtk_text_view_forward_display_line(GTK_TEXT_VIEW(completion->textview),
            &iter))
      break;
    line_count ++;
  }
  line_height = (height - 4) / line_count;
  
  if (line_count >= 4) {
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(completion->scrolledtext),
        GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
    
    height = 4 * line_height + 4;
  }
  else {
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(completion->scrolledtext),
        GTK_POLICY_NEVER, GTK_POLICY_NEVER);
  }
  
  gtk_widget_set_size_request(completion->scrolledtext, width, height);
}


static void text_changed(GtkTextBuffer * textbuffer,
    gpointer user_data)
{
  struct etpan_completion * completion;
  (void) textbuffer;
  
  completion = user_data;
  
  schedule_request(completion);
}

static int completion_key_handler(struct etpan_completion * completion,
    GdkEventKey * event);

static gboolean
completion_key_press(GtkWidget * widget,
    GdkEventKey * event,
    gpointer user_data)
{
  struct etpan_completion * completion;
  unsigned int items;
  (void) widget;
  
  completion = user_data;
  
  items = gtk_tree_model_iter_n_children(GTK_TREE_MODEL(completion->completion_model), NULL);
  
  if (completion_key_handler(completion, event))
    return TRUE;
  
  gtk_widget_event(completion->textview, (GdkEvent *) event);
  
  return TRUE;
}

static gboolean
entry_key_press(GtkWidget * widget,
    GdkEventKey * event,
    gpointer user_data)
{
  struct etpan_completion * completion;
  (void) widget;
  
  completion = user_data;
  
  if ((event->keyval == GDK_ISO_Enter) ||
      (event->keyval == GDK_KP_Enter) ||
      (event->keyval == GDK_Return)) {
    return TRUE;
  }
  
  return FALSE;
}

static void cancel_request(struct etpan_completion * completion);

static gboolean
completion_button_press(GtkWidget * widget,
    GdkEventButton * event,
    gpointer user_data)
{
  struct etpan_completion * completion;
  (void) widget;
  (void) event;
  
  completion = user_data;
  
  cancel_request(completion);
  etpan_completion_hide_completion(completion);
  
  return TRUE;
}

static void perform_completion(struct etpan_completion * completion);

static gboolean
completion_list_button_press(GtkWidget * widget,
    GdkEventButton * event,
    gpointer user_data)
{
  struct etpan_completion * completion;
  GtkTreePath * path;
  gboolean r;
  
  path = NULL;
  completion = user_data;
  
  r = gtk_tree_view_get_path_at_pos(GTK_TREE_VIEW (widget),
      event->x, event->y, &path, NULL, NULL, NULL);
  if (r) {
    cancel_request(completion);
    etpan_completion_hide_completion(completion);
    
    perform_completion(completion);
    
    return TRUE;
  }
  
  return FALSE;
}


static int keyval_is_cursor_move(guint keyval)
{
  if ((keyval == GDK_Up) || (keyval == GDK_KP_Up))
    return TRUE;
  
  if ((keyval == GDK_Down) || (keyval == GDK_KP_Down))
    return TRUE;

  if (keyval == GDK_Page_Up)
    return TRUE;

  if (keyval == GDK_Page_Down)
    return TRUE;

  return FALSE;
}

static int completion_key_handler(struct etpan_completion * completion,
    GdkEventKey * event)
{
  if (keyval_is_cursor_move(event->keyval)) {
    GtkTreePath * path;
    unsigned int selected;
    unsigned int items;
    unsigned int display_max;
    
    items = gtk_tree_model_iter_n_children(GTK_TREE_MODEL(completion->completion_model), NULL);
    if (items > 15)
      display_max = 15;
    else
      display_max = items;
    
    gtk_tree_view_get_cursor(GTK_TREE_VIEW(completion->treeview), &path, NULL);
    
    if (path == NULL) {
      selected = 0;
    }
    else {
      gint * indexes;
      
      indexes = gtk_tree_path_get_indices(path);
      selected = indexes[0];
    
      if (path != NULL)
        gtk_tree_path_free(path);
    
      if (event->keyval == GDK_Up || event->keyval == GDK_KP_Up) {
        if (selected == 0)
          selected = 0;
        else
          selected --;
      }
      else if (event->keyval == GDK_Down || event->keyval == GDK_KP_Down) {
        if (selected >= items - 1)
          selected = items - 1;
        else
          selected ++;
      }
      else if (event->keyval == GDK_Page_Up) {
        if (selected <= display_max - 1)
          selected = 0;
        else
          selected -= display_max - 1;
      }
      else if (event->keyval == GDK_Page_Down) {
        if (selected + display_max - 1 >= items - 1)
          selected = items - 1;
        else
          selected += display_max - 1;
      }
    }
    
    path = gtk_tree_path_new_from_indices(selected, -1);
    gtk_tree_view_set_cursor(GTK_TREE_VIEW(completion->treeview),
        path, NULL, FALSE);
    gtk_tree_path_free (path);
    
    return TRUE;
  }
  else if (event->keyval == GDK_Escape) {
    cancel_request(completion);
    etpan_completion_hide_completion(completion);
    
    return TRUE;
  }
  else if ((event->keyval == GDK_Tab) || (event->keyval == GDK_KP_Tab) ||
      (event->keyval == GDK_ISO_Left_Tab)) {
    GtkDirectionType dir;
    
    if (event->keyval == GDK_ISO_Left_Tab)
      dir = GTK_DIR_TAB_BACKWARD;
    else
      dir = GTK_DIR_TAB_FORWARD;
    
    cancel_request(completion);
    etpan_completion_hide_completion(completion);
    
    gtk_widget_child_focus(gtk_widget_get_toplevel(completion->textview), dir);
    
    return TRUE;
  }
  else if ((event->keyval == GDK_ISO_Enter) ||
      (event->keyval == GDK_KP_Enter) ||
      (event->keyval == GDK_Return)) {
    
    cancel_request(completion);
    etpan_completion_hide_completion(completion);
    
    perform_completion(completion);
    return TRUE;
  }
  
  return FALSE;
}

static void get_completion_value(struct etpan_completion * completion,
    unsigned int * p_begin, unsigned int * p_end)
{
  char * begin;
  char * end;
  size_t remaining;
  char * p;
  GtkTextIter iter_begin;
  GtkTextIter iter_end;
  GtkTextMark * mark_end;
  char * text;
  size_t begin_offset;
  size_t end_offset;
  size_t skipped;
  
  gtk_text_buffer_get_iter_at_offset(completion->buffer, &iter_begin, 0);
  mark_end = gtk_text_buffer_get_insert(completion->buffer);
  gtk_text_buffer_get_iter_at_mark(completion->buffer, &iter_end, mark_end);
  
  text = gtk_text_buffer_get_slice(completion->buffer,
      &iter_begin, &iter_end, 0);
  
  end = text + strlen(text);
  begin = text;
  remaining = g_utf8_strlen(begin, -1);
  
  p = begin;
  skipped = 0;
  begin_offset = 0;
  while (remaining > 0) {
    if (* p == ',') {
      begin = p + 1;
      begin_offset = skipped + 1;
    }
    p = g_utf8_next_char(p);
    remaining --;
    skipped ++;
  }
  
  while (* begin == ' ') {
    begin ++;
    skipped ++;
    begin_offset ++;
  }
  
  end_offset = g_utf8_strlen(begin, -1) + begin_offset;
  
  * p_begin = begin_offset;
  * p_end = end_offset;
  
  g_free(text);
}

static void perform_completion(struct etpan_completion * completion)
{
  unsigned int begin_offset;
  unsigned int end_offset;
  GtkTextIter iter_begin;
  GtkTextIter iter_end;
  struct etpan_abook_entry * entry;
  GValue value;
  char buffer[1024];
  GtkTreePath * path;
  GtkTreeIter iter;
  char entry_str[1024];
  
  cancel_request(completion);
  etpan_completion_hide_completion(completion);
  
  gtk_tree_view_get_cursor(GTK_TREE_VIEW(completion->treeview), &path, NULL);
  if (path == NULL)
    return;
  
  gtk_tree_model_get_iter(GTK_TREE_MODEL(completion->completion_model),
      &iter, path);
  
  get_completion_value(completion, &begin_offset, &end_offset);
  gtk_text_buffer_get_iter_at_offset(completion->buffer,
      &iter_begin, begin_offset);
  gtk_text_buffer_get_iter_at_offset(completion->buffer,
      &iter_end, end_offset);
  
  gtk_text_buffer_delete(completion->buffer, &iter_begin, &iter_end);
  
  memset(&value, 0, sizeof(value));
  gtk_tree_model_get_value(GTK_TREE_MODEL(completion->completion_model),
      &iter, 3, &value);
  entry = g_value_get_pointer(&value);
  
  etpan_abook_entry_str(entry_str, sizeof(entry_str), entry);
  snprintf(buffer, sizeof(buffer), "%s, ", entry_str);
  gtk_text_buffer_insert_at_cursor(completion->buffer,
      buffer, strlen(buffer));
}

static void cancel_request(struct etpan_completion * completion)
{
  /* cancel scheduled request */
  if (completion->scheduled) {
    g_source_remove(completion->timer_id);
    completion->scheduled = 0;
  }
  
  if (completion->requesting) {
    etpan_abook_request_cancel(completion->abook_request);
    completion->requesting = 0;
  }
}

static gboolean timer_handler(gpointer data)
{
  struct etpan_completion * completion;
  
  completion = data;
  
  cancel_request(completion);
  request_completion(completion);
  
  return FALSE;
}

//#define COMPLETION_DELAY (1 * 1000) /* 1 second */
#define COMPLETION_DELAY (300) /* 1 second */

static void schedule_request(struct etpan_completion * completion)
{
  cancel_request(completion);
  
  completion->scheduled = 1;
  completion->timer_id = g_timeout_add(COMPLETION_DELAY,
      timer_handler, completion);
}

static void request_completion(struct etpan_completion * completion)
{
  unsigned int begin_offset;
  unsigned int end_offset;
  GtkTextIter iter_begin;
  GtkTextIter iter_end;
  char * text;
  
  get_completion_value(completion, &begin_offset, &end_offset);
  if (begin_offset == end_offset) {
    etpan_completion_hide_completion(completion);
    return;
  }
  
  gtk_text_buffer_get_iter_at_offset(completion->buffer,
      &iter_begin, begin_offset);
  gtk_text_buffer_get_iter_at_offset(completion->buffer,
      &iter_end, end_offset);
  
  text = gtk_text_buffer_get_slice(completion->buffer,
      &iter_begin, &iter_end, 0);
  ETPAN_LOG("complete : %s", text);
  etpan_abook_request_lookup(completion->abook_request, text);
  etpan_gtk_tree_model_reload(completion->completion_model);
  completion->requesting = 1;
  
  g_free(text);
}

static void lookup_finished(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_abook_request * request;
  struct etpan_completion * completion;
  (void) signal_name;
  (void) signal_data;
  
  completion = user_data;
  request = sender;
  
  completion->requesting = 0;
  
  ETPAN_LOG("lookup finished");
}

static void lookup_updated(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_completion * completion;
  (void) signal_name;
  (void) signal_data;
  (void) sender;
  
  completion = user_data;
  
  etpan_gtk_tree_model_reload(completion->completion_model);
  etpan_completion_show_completion(completion);
  ETPAN_LOG("updated result");
}

static gboolean focus_out_handler(GtkWidget * widget,
    GdkEventFocus * event,
    gpointer user_data)
{
  struct etpan_completion * completion;
  (void) widget;
  (void) event;
  
  completion = user_data;
  cancel_request(completion);
  
  return FALSE;
}

void etpan_completion_setup(struct etpan_completion * completion)
{
  /* gtk signals */
  completion->size_signal_id =
    g_signal_connect(completion->textview, "size-request",
        G_CALLBACK(size_requested), (gpointer) completion);
  completion->changed_signal_id =
    g_signal_connect(completion->buffer, "changed",
        G_CALLBACK(text_changed), (gpointer) completion);
  completion->window_keypress_signal_id =
    g_signal_connect(completion->window, "key_press_event",
        G_CALLBACK(completion_key_press),
        completion);
  completion->window_button_signal_id =
    g_signal_connect(completion->window, "button_press_event",
        G_CALLBACK (completion_button_press),
        completion);
  completion->entry_keypress_signal_id =
    g_signal_connect(completion->textview, "key_press_event",
        G_CALLBACK(entry_key_press),
        completion);
  completion->list_button_signal_id =
    g_signal_connect(completion->treeview, "button_press_event",
        G_CALLBACK (completion_list_button_press),
        completion);
  completion->focus_out_signal_id =
    g_signal_connect(completion->textview, "focus-out-event",
        G_CALLBACK (focus_out_handler),
        completion);
  
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_ABOOK_REQUEST_UPDATED_RESULT, completion->abook_request, completion,
      lookup_updated);
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_ABOOK_REQUEST_FINISHED, completion->abook_request, completion,
      lookup_finished);
}

void etpan_completion_unsetup(struct etpan_completion * completion)
{
  cancel_request(completion);
  
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_ABOOK_REQUEST_FINISHED, completion->abook_request, completion,
      lookup_finished);
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_ABOOK_REQUEST_UPDATED_RESULT, completion->abook_request, completion,
      lookup_updated);
  
  g_signal_handler_disconnect(completion->textview,
      completion->focus_out_signal_id);
  g_signal_handler_disconnect(completion->treeview,
      completion->list_button_signal_id);
  g_signal_handler_disconnect(completion->textview,
      completion->entry_keypress_signal_id);
  g_signal_handler_disconnect(completion->window,
      completion->window_button_signal_id);
  g_signal_handler_disconnect(completion->window,
      completion->window_keypress_signal_id);
  g_signal_handler_disconnect(completion->buffer,
      completion->changed_signal_id);
  g_signal_handler_disconnect(completion->textview,
      completion->size_signal_id);
}

carray * etpan_completion_get_address_list(struct etpan_completion * completion)
{
  GtkTextIter start_iter;
  GtkTextIter end_iter;
  char * str;
  carray * list;
  
  gtk_text_buffer_get_start_iter(completion->buffer, &start_iter);
  gtk_text_buffer_get_end_iter(completion->buffer, &end_iter);
  
  str = gtk_text_buffer_get_text(completion->buffer,
      &start_iter, &end_iter, 0);
  
  list = etpan_address_list_to_from_str(str);
  
  g_free(str);
  
  return list;
}
