#include "etpan-contextual-menu.h"

#include <stdlib.h>

#include "etpan-backend.h"
#include "etpan-log.h"
#include "etpan-error.h"

enum {
  ETPAN_MENU_ITEM_TYPE_SIMPLE,
  ETPAN_MENU_ITEM_TYPE_SEPARATOR,
  ETPAN_MENU_ITEM_TYPE_CHECK,
  ETPAN_MENU_ITEM_TYPE_SUBMENU,
};

struct menu_item {
  int type;
  struct etpan_contextual_menu * menu;
  GtkWidget * menu_item;
  GtkWidget * label_widget;
  int has_signal;
  gulong signal_id;
  int index;
  struct etpan_contextual_menu * submenu;
};

struct etpan_contextual_menu * etpan_contextual_menu_new(void)
{
  struct etpan_contextual_menu * menu;
  GtkWidget * menu_widget;
  
  menu = malloc(sizeof(* menu));
  if (menu == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  menu->item_list = carray_new(8);
  if (menu->item_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  menu_widget = gtk_menu_new();
  gtk_widget_show(menu_widget);
  
  menu->menu = menu_widget;
  
  return menu;
}

void etpan_contextual_menu_free(struct etpan_contextual_menu * menu)
{
  etpan_contextual_menu_clear(menu);
  carray_free(menu->item_list);
  gtk_widget_destroy(menu->menu);
  free(menu);
}

static void menu_handler(GtkMenuItem * menuitem,
    gpointer user_data)
{
  struct etpan_contextual_menu * menu;
  struct menu_item * item_data;
  (void) menuitem;
  
  item_data = user_data;
  menu = item_data->menu;
  
  etpan_signal_send(etpan_signal_manager_get_default(),
      ETPAN_CONTEXTUAL_MENU_CLICKED_SIGNAL, menu, &item_data->index);
}

void etpan_contextual_menu_add_item(struct etpan_contextual_menu * menu,
    char * label, GtkWidget * image)
{
  GtkWidget * label_widget;
  GtkWidget * hbox;
  GtkWidget * item;
  int label_index;
  struct menu_item * item_data;
  int r;
  
  label_index = carray_count(menu->item_list);
  
  item_data = malloc(sizeof(* item_data));
  if (item_data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  item_data->menu = menu;
  item_data->type = ETPAN_MENU_ITEM_TYPE_SIMPLE;
  item_data->index = label_index;
  item_data->submenu = NULL;
  item_data->label_widget = NULL;
  
  r = carray_add(menu->item_list, item_data, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  item = gtk_menu_item_new();
  gtk_widget_show(item);
  
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_widget_show(hbox);
  
  gtk_container_add(GTK_CONTAINER(item), hbox);
  
  if (label != NULL) {
    label_widget = gtk_label_new(label);
    gtk_widget_show(label_widget);
    gtk_box_pack_start(GTK_BOX(hbox), label_widget, FALSE, FALSE, 0);
    item_data->label_widget = label_widget;
  }
  if (image != NULL) {
    gtk_box_pack_start(GTK_BOX(hbox), image, FALSE, FALSE, 0);
  }
  
  gtk_menu_attach(GTK_MENU(menu->menu), item,
      0, 1, label_index, label_index + 1);
  
  item_data->has_signal = 1;
  item_data->signal_id = g_signal_connect(item,
      "activate", G_CALLBACK(menu_handler),
      (gpointer) item_data);
  item_data->menu_item = item;
}

void etpan_contextual_menu_add_check_item(struct etpan_contextual_menu * menu,
    char * label, GtkWidget * image)
{
  GtkWidget * label_widget;
  GtkWidget * hbox;
  GtkWidget * item;
  int label_index;
  struct menu_item * item_data;
  int r;
  
  label_index = carray_count(menu->item_list);
  
  item_data = malloc(sizeof(* item_data));
  if (item_data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  item_data->menu = menu;
  item_data->type = ETPAN_MENU_ITEM_TYPE_CHECK;
  item_data->index = label_index;
  item_data->submenu = NULL;
  item_data->label_widget = NULL;
  
  r = carray_add(menu->item_list, item_data, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  item = gtk_check_menu_item_new();
  gtk_widget_show(item);
  
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_widget_show(hbox);
  
  gtk_container_add(GTK_CONTAINER(item), hbox);
  
  if (label != NULL) {
    label_widget = gtk_label_new(label);
    gtk_widget_show(label_widget);
    gtk_box_pack_start(GTK_BOX(hbox), label_widget, FALSE, FALSE, 0);
    item_data->label_widget = label_widget;
  }
  if (image != NULL) {
    gtk_box_pack_start(GTK_BOX(hbox), image, FALSE, FALSE, 0);
  }
  
  gtk_menu_attach(GTK_MENU(menu->menu), item,
      0, 1, label_index, label_index + 1);
  
  item_data->has_signal = 1;
  item_data->signal_id = g_signal_connect(item,
      "activate", G_CALLBACK(menu_handler),
      (gpointer) item_data);
  item_data->menu_item = item;
}

void etpan_contextual_menu_add_separator(struct etpan_contextual_menu * menu)
{
  GtkWidget * item;
  int label_index;
  struct menu_item * item_data;
  int r;
  
  label_index = carray_count(menu->item_list);
  
  item_data = malloc(sizeof(* item_data));
  if (item_data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  item_data->menu = menu;
  item_data->type = ETPAN_MENU_ITEM_TYPE_SEPARATOR;
  item_data->index = label_index;
  item_data->submenu = NULL;
  item_data->label_widget = NULL;
  
  r = carray_add(menu->item_list, item_data, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  item = gtk_separator_menu_item_new();
  gtk_widget_show(item);
  
  gtk_menu_attach(GTK_MENU(menu->menu), item,
      0, 1, label_index, label_index + 1);
  
  item_data->has_signal = 0;
  item_data->signal_id = 0;
  item_data->menu_item = item;
}

void etpan_contextual_menu_add_sub(struct etpan_contextual_menu * menu,
    char * label, GtkWidget * image,
    struct etpan_contextual_menu * submenu)
{
  GtkWidget * label_widget;
  GtkWidget * hbox;
  GtkWidget * item;
  int label_index;
  struct menu_item * item_data;
  int r;
  GtkWidget * submenu_widget;
  
  label_index = carray_count(menu->item_list);
  
  item_data = malloc(sizeof(* item_data));
  if (item_data == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  item_data->menu = menu;
  item_data->type = ETPAN_MENU_ITEM_TYPE_SUBMENU;
  item_data->index = label_index;
  item_data->submenu = submenu;
  item_data->label_widget = NULL;
  
  r = carray_add(menu->item_list, item_data, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  item = gtk_menu_item_new();
  gtk_widget_show(item);
  
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_widget_show(hbox);
  
  gtk_container_add(GTK_CONTAINER(item), hbox);
  
  if (label != NULL) {
    label_widget = gtk_label_new(label);
    gtk_widget_show(label_widget);
    gtk_box_pack_start(GTK_BOX(hbox), label_widget, FALSE, FALSE, 0);
    item_data->label_widget = label_widget;
  }
  if (image != NULL) {
    gtk_box_pack_start(GTK_BOX(hbox), image, FALSE, FALSE, 0);
  }
  
  submenu_widget = etpan_contextual_menu_get_main_widget(submenu);
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), submenu_widget);
  
  gtk_menu_attach(GTK_MENU(menu->menu), item,
      0, 1, label_index, label_index + 1);
  
  item_data->has_signal = 1;
  item_data->signal_id = g_signal_connect(item,
      "activate", G_CALLBACK(menu_handler),
      (gpointer) item_data);
  item_data->menu_item = item;
}

void etpan_contextual_menu_set_check_item(struct etpan_contextual_menu * menu,
    int index, int value)
{
  struct menu_item * item;
  
  item = carray_get(menu->item_list, index);
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item->menu_item), value);
}

int etpan_contextual_menu_get_check_item(struct etpan_contextual_menu * menu,
    int index)
{
  struct menu_item * item;
  
  item = carray_get(menu->item_list, index);
  return gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(item->menu_item));
}

void etpan_contextual_menu_set_label(struct etpan_contextual_menu * menu,
    int index, char * text)
{
  struct menu_item * item;
  
  item = carray_get(menu->item_list, index);
  gtk_label_set_text(GTK_LABEL(item->label_widget), text);
}

void etpan_contextual_menu_clear(struct etpan_contextual_menu * menu)
{
  unsigned int i;
  int r;
  
  for(i = 0 ; i < carray_count(menu->item_list) ; i ++) {
    struct menu_item * item;
    
    item = carray_get(menu->item_list, i);
    
    if (item->submenu != NULL)
      etpan_contextual_menu_free(item->submenu);
    
    if (item->signal_id != 0)
      g_signal_handler_disconnect(item->menu_item, item->signal_id);
    
    gtk_widget_destroy(item->menu_item);
    
    free(item);
  }
  r = carray_set_size(menu->item_list, 0);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

GtkWidget *
etpan_contextual_menu_get_main_widget(struct etpan_contextual_menu *
    menu)
{
  return menu->menu;
}

void etpan_contextual_menu_set_enabled(struct etpan_contextual_menu * menu,
    int index, int enabled)
{
  struct menu_item * item;
  
  item = carray_get(menu->item_list, index);
  ETPAN_LOG("disable widget");
  gtk_widget_set_sensitive(item->menu_item, enabled);
}
