#include "etpan-folder-list.h"

#include <stdlib.h>
#include <string.h>
#include <libetpan/libetpan.h>
#include <gdk/gdkkeysyms.h>

#include "etpan-backend.h"
#include "etpan-gtk-tree-model.h"
#include "etpan-gtk-workaround.h"
#include "etpan-message-list-types.h"
#include "etpan-ui-config.h"
#include "etpan-icon-manager.h"
#include "etpan-contextual-menu.h"
#include "etpan-input-dialog.h"

#include "gtktreeprivate.h"

enum {
  STORAGE_DRIVER_TYPE_POP,
  STORAGE_DRIVER_TYPE_IMAP,
  STORAGE_DRIVER_TYPE_NEWS,
  STORAGE_DRIVER_TYPE_MAILDIR,
  STORAGE_DRIVER_TYPE_MH,
  STORAGE_DRIVER_TYPE_MBOX,
};

struct folder_tree;
static void folder_tree_free_recursive(struct folder_tree * tree);

static void drag_drop_setup(struct etpan_folder_list * folder_list);
static void drag_drop_unsetup(struct etpan_folder_list * folder_list);

static void drag_may_start(struct etpan_folder_list * folder_list,
    gdouble x, gdouble y, guint32 time);
static void drag_cancel_may_start(struct etpan_folder_list * folder_list);
static int drag_may_be_starting(struct etpan_folder_list * folder_list);
static int drag_is_valid(struct etpan_folder_list * folder_list,
    gdouble x, gdouble y, guint32 time);
static void drag_start(struct etpan_folder_list * folder_list,
    GdkEvent * event);
static void drag_end(struct etpan_folder_list * folder_list);
static void drag_set_icon(struct etpan_folder_list * folder_list);

static void popup_menu_setup(struct etpan_folder_list * folder_list);
static void popup_menu_unsetup(struct etpan_folder_list * folder_list);
static void show_folder_popup_menu(struct etpan_folder_list * folder_list,
    GtkWidget * widget, GdkEventButton * event);
static void folder_popup_menu_item_clicked(char * signal_name, void * sender,
    void * signal_data, void * user_data);

void etpan_folder_list_new_folder(struct etpan_folder_list * folder_list,
    struct etpan_storage * storage,
    struct etpan_folder * folder);
void etpan_folder_list_delete_mailbox(struct etpan_folder_list * folder_list,
    struct etpan_folder * folder);

static void reorder(struct etpan_folder_list * folder_list,
    struct folder_tree * source, int type,
    struct folder_tree * dest);
static struct folder_tree *
get_tree_by_ui_path(struct etpan_folder_list * folder_list, char * path);

static void copy_msg_setup(struct etpan_folder_list * folder_list);
static void copy_msg_unsetup(struct etpan_folder_list * folder_list);

static void copy_folder_setup(struct etpan_folder_list * folder_list);
static void copy_folder_unsetup(struct etpan_folder_list * folder_list);

static void notify_folder_selection(struct etpan_folder_list * folder_list);

enum {
  STORE_INDEX_NAME,
  STORE_INDEX_UNREAD,
  STORE_INDEX_INFO,
  STORE_INDEX_IMAGE,
  STORE_INDEX_IMAGESIZE,
  STORE_INDEX_PROGRESS,
  STORE_INDEX_ICON,
};

static unsigned int get_n_columns(struct etpan_gtk_tree_data_source *
    datasource);
static GType get_column_type(struct etpan_gtk_tree_data_source * datasource,
    unsigned int column_index);
static unsigned int get_children_count(struct etpan_gtk_tree_data_source *
    datasource, void * item);
static int item_has_child(struct etpan_gtk_tree_data_source *
    datasource, void * item);
static void * get_child_item(struct etpan_gtk_tree_data_source *
    datasource, void * item, unsigned int index);
static void get_item_value(struct etpan_gtk_tree_data_source *
    datasource, void * item, unsigned int column_index,
    GValue * value);

#define PROGRESS_ANIM_COUNT 10

struct anim_info {
  int image_index;
  int count;
};

static void progress_init(struct etpan_folder_list * folder_list)
{
  folder_list->progress_anim = malloc(PROGRESS_ANIM_COUNT *
      sizeof(* folder_list->progress_anim));
  if (folder_list->progress_anim == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
#define ICON(num) \
  etpan_icon_manager_get_pixbuf(etpan_icon_manager_get_default(), "progress-" num)
  
  folder_list->progress_anim[0] = ICON("0");
  folder_list->progress_anim[1] = ICON("1");
  folder_list->progress_anim[2] = ICON("2");
  folder_list->progress_anim[3] = ICON("3");
  folder_list->progress_anim[4] = ICON("4");
  folder_list->progress_anim[5] = ICON("5");
  folder_list->progress_anim[6] = ICON("4");
  folder_list->progress_anim[7] = ICON("3");
  folder_list->progress_anim[8] = ICON("2");
  folder_list->progress_anim[9] = ICON("1");
#undef ICON
}

static void progress_done(struct etpan_folder_list * folder_list)
{
  free(folder_list->progress_anim);
}

struct etpan_folder_list * etpan_folder_list_new(void)
{
  GtkWidget * scrolledwindow;
  GtkWidget * treeview;
  struct etpan_folder_list * folder_list;
  etpan_gtk_tree_model * folder_model;
  GtkTreeViewColumn * col_name;
  GtkCellRenderer * col_icon_renderer;
  GtkCellRenderer * col_name_renderer;
  GtkTreeViewColumn * col_unread;
  GtkCellRenderer * col_unread_renderer;
  GtkTreeSelection * selection;
  GtkTreeViewColumn * col_progress;
  GtkCellRenderer * col_progress_renderer;
  
  folder_list = malloc(sizeof(* folder_list));
  if (folder_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  folder_list->folder_progress = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYALL);
  if (folder_list->folder_progress == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  folder_list->path_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (folder_list->path_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  folder_list->treeinfo = NULL;
  
  folder_model = etpan_gtk_tree_model_new();
  
  folder_list->folder_datasource.data = folder_list;
  folder_list->folder_datasource.get_n_columns = get_n_columns;
  folder_list->folder_datasource.get_column_type = get_column_type;
  folder_list->folder_datasource.get_children_count = get_children_count;
  folder_list->folder_datasource.item_has_child = item_has_child;
  folder_list->folder_datasource.get_child_item = get_child_item;
  folder_list->folder_datasource.get_item_value = get_item_value;
  
  etpan_gtk_tree_model_set_datasource(folder_model,
      &folder_list->folder_datasource);
  
  scrolledwindow = gtk_scrolled_window_new(NULL, NULL);
  gtk_widget_show(scrolledwindow);
  gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrolledwindow),
      GTK_SHADOW_IN);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledwindow),
      GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  
  treeview = gtk_tree_view_new();
  
  gtk_tree_view_set_enable_search(GTK_TREE_VIEW(treeview), 0);
  gtk_tree_view_set_model(GTK_TREE_VIEW(treeview),
      GTK_TREE_MODEL(folder_model));
  
  gtk_widget_show(treeview);
  gtk_container_add(GTK_CONTAINER(scrolledwindow), treeview);
  gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(treeview), FALSE);
  gtk_tree_view_set_enable_search(GTK_TREE_VIEW(treeview), FALSE);
  
  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
  gtk_tree_selection_set_mode(selection, GTK_SELECTION_SINGLE);
  
  /* column name */
  col_name = gtk_tree_view_column_new();
  gtk_tree_view_column_set_title(col_name, _("folder"));
  col_icon_renderer = gtk_cell_renderer_pixbuf_new();
  gtk_tree_view_column_pack_start(col_name, col_icon_renderer, FALSE);
  gtk_tree_view_column_set_attributes(col_name, col_icon_renderer,
      "pixbuf", STORE_INDEX_ICON,
      NULL);
  col_name_renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_column_pack_start(col_name, col_name_renderer, TRUE);
  gtk_tree_view_column_set_attributes(col_name, col_name_renderer,
      "text", STORE_INDEX_NAME, NULL);
  gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), col_name);
  
  /* column unread */
  col_unread = gtk_tree_view_column_new();
  gtk_tree_view_column_set_title(col_unread, _("unread"));
  col_unread_renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_column_pack_start(col_unread, col_unread_renderer, TRUE);
  gtk_tree_view_column_set_attributes(col_unread, col_unread_renderer,
      "text", STORE_INDEX_UNREAD, NULL);
  gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), col_unread);
  
  /* column progress */
  col_progress = gtk_tree_view_column_new();
  gtk_tree_view_column_set_title(col_unread, _(""));
  col_progress_renderer = gtk_cell_renderer_pixbuf_new();
  g_object_set(col_progress_renderer, "xalign", 1.0, NULL);
  gtk_tree_view_column_pack_start(col_progress, col_progress_renderer, TRUE);
  gtk_tree_view_column_set_attributes(col_progress, col_progress_renderer,
      "pixbuf", STORE_INDEX_PROGRESS,
      NULL);
  gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), col_progress);
  
  gtk_tree_view_set_expander_column(GTK_TREE_VIEW(treeview), col_name);
  
  folder_list->treeview = treeview;
  folder_list->scrolledwindow = scrolledwindow;
  folder_list->folder_model = folder_model;
  folder_list->current_selection = NULL;
  folder_list->progress_animation = 0;
  
  folder_list->drag_may_start = 0;
  
  folder_list->copy_msg_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (folder_list->copy_msg_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  folder_list->copy_folder_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (folder_list->copy_folder_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  folder_list->copy_folder_storage_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (folder_list->copy_folder_storage_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  progress_init(folder_list);
  
  return folder_list;
}

void etpan_folder_list_free(struct etpan_folder_list * folder_list)
{
  chash_free(folder_list->copy_folder_storage_hash);
  chash_free(folder_list->copy_folder_hash);
  chash_free(folder_list->copy_msg_hash);
  
  if (folder_list->current_selection != NULL)
    free(folder_list->current_selection);
  
  if (folder_list->treeinfo != NULL)
    folder_tree_free_recursive(folder_list->treeinfo);
  folder_list->treeinfo = NULL;
  
  progress_done(folder_list);
  
  gtk_tree_view_set_model(GTK_TREE_VIEW(folder_list->treeview), NULL);
  gtk_widget_destroy(folder_list->treeview);
  
  g_object_unref(folder_list->folder_model);
  
  gtk_widget_destroy(folder_list->scrolledwindow);
  
  chash_free(folder_list->path_hash);
  chash_free(folder_list->folder_progress);
  free(folder_list);
}

GtkWidget * etpan_folder_list_get_main_widget(struct etpan_folder_list *
    folder_list)
{
  return folder_list->scrolledwindow;
}

static void start_animation_folder(struct etpan_folder_list * folder_list,
    struct etpan_folder * folder);
static void stop_animation_folder(struct etpan_folder_list * folder_list,
    struct etpan_folder * folder);

static void start_animation_storage(struct etpan_folder_list * folder_list,
    struct etpan_storage * storage);
static void stop_animation_storage(struct etpan_folder_list * folder_list,
    struct etpan_storage * storage);

static void stop_animation(struct etpan_folder_list * folder_list);

struct folder_tree {
  int type;
  char * ui_path;
  struct etpan_account * account;
  struct etpan_folder * folder;
  struct folder_tree * parent;
  carray * children;
  
  /* gtk specific */
  char * gtk_path;
};

static int get_account_type(struct etpan_account * account)
{
  struct etpan_storage * storage;
  struct etpan_storage_driver * driver;
  int type;
  
  storage = etpan_account_get_storage(account);
  if (storage == NULL) {
    ETPAN_LOG("account has no storage");
    etpan_crash();
  }
  
  driver = etpan_storage_get_driver(storage);
  type = STORAGE_DRIVER_TYPE_POP;
  if (strcasecmp(driver->name, "pop-sync") == 0)
    type = STORAGE_DRIVER_TYPE_POP;
  else if (strcasecmp(driver->name, "imap-sync") == 0)
    type = STORAGE_DRIVER_TYPE_IMAP;
  else if (strcasecmp(driver->name, "news") == 0)
    type = STORAGE_DRIVER_TYPE_NEWS;
  else if (strcasecmp(driver->name, "maildir") == 0)
    type = STORAGE_DRIVER_TYPE_MAILDIR;
  else if (strcasecmp(driver->name, "mh") == 0)
    type = STORAGE_DRIVER_TYPE_MH;
  else if (strcasecmp(driver->name, "mbox") == 0)
    type = STORAGE_DRIVER_TYPE_MBOX;
  
  return type;
}

static struct folder_tree * folder_tree_new(char * ui_path)
{
  struct folder_tree * tree;
  
  tree = malloc(sizeof(* tree));
  if (tree == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  tree->ui_path = strdup(ui_path);
  if (tree->ui_path == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  tree->children = carray_new(4);
  if (tree->children == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  tree->parent = NULL;
  
  tree->folder = NULL;
  tree->account = NULL;
  tree->gtk_path = NULL;
  tree->type = ETPAN_FOLDER_LIST_SELECTION_TYPE_GROUP;
  
  return tree;
}

static void folder_tree_free(struct folder_tree * tree)
{
  free(tree->gtk_path);
  carray_free(tree->children);
  free(tree->ui_path);
  free(tree);
}

static void folder_tree_set_gtk_path(struct folder_tree * tree,
    char * gtk_path)
{
  if (gtk_path != tree->gtk_path) {
    free(tree->gtk_path);
    if (gtk_path != NULL) {
      tree->gtk_path = strdup(gtk_path);
      if (tree->gtk_path == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      tree->gtk_path = NULL;
  }
}

static void folder_tree_set_type(struct folder_tree * tree, int type)
{
  tree->type = type;
}

static int folder_tree_get_account_type(struct folder_tree * tree)
{
  return get_account_type(tree->account);
}

static int folder_tree_get_type(struct folder_tree * tree)
{
  return tree->type;
}

static char * folder_tree_get_gtk_path(struct folder_tree * tree)
{
  return tree->gtk_path;
}

static void folder_tree_set_folder(struct folder_tree * tree,
    struct etpan_folder * folder)
{
  if (folder != NULL)
    folder_tree_set_type(tree, ETPAN_FOLDER_LIST_SELECTION_TYPE_FOLDER);
  tree->folder = folder;
}

static struct etpan_folder * folder_tree_get_folder(struct folder_tree * tree)
{
  return tree->folder;
}

static void folder_tree_set_account(struct folder_tree * tree,
    struct etpan_account * account)
{
  tree->account = account;
}

static struct folder_tree * folder_tree_get_parent(struct folder_tree * tree)
{
  return tree->parent;
}

static struct etpan_account * folder_tree_get_account(struct folder_tree * tree)
{
  return tree->account;
}

static void folder_tree_add_child(struct folder_tree * parent,
    struct folder_tree * child)
{
  int r;
  
  r = carray_add(parent->children, child, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  child->parent = parent;
}

static carray * folder_tree_get_children(struct folder_tree * parent)
{
  return parent->children;
}

static char * folder_tree_get_ui_path(struct folder_tree * tree)
{
  return tree->ui_path;
}

static void folder_tree_free_recursive(struct folder_tree * tree)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(tree->children) ; i ++) {
    struct folder_tree * child;
    
    child = carray_get(tree->children, i);
    folder_tree_free_recursive(child);
  }
  folder_tree_free(tree);
}

static struct folder_tree * add_tree(chash * tree_hash,
    char * ui_path, struct etpan_folder * folder)
{
  chashdatum key;
  chashdatum value;
  struct folder_tree * tree;
  struct folder_tree * parent;
  int r;
  char parent_path[PATH_MAX];
  char * sep_pos;
  
  key.data = ui_path;
  key.len = strlen(ui_path) + 1;
  
  r = chash_get(tree_hash, &key, &value);
  if (r == 0) {
    tree = value.data;
    
    if (folder != NULL)
      folder_tree_set_folder(tree, folder);
    
    return tree;
  }
  
  tree = folder_tree_new(ui_path);
  
  folder_tree_set_folder(tree, folder);
  
  value.data = tree;
  value.len = 0;
  r = chash_set(tree_hash, &key, &value, NULL);
  if (r < 0) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  /* root */
  if (* ui_path == '\0')
    return tree;
  
  /* get parent path */
  strncpy(parent_path, ui_path, sizeof(parent_path));
  parent_path[sizeof(parent_path) - 1] = '\0';
  
  sep_pos = strrchr(parent_path, '/');
  if (sep_pos == NULL)
    sep_pos = parent_path;
  
  * sep_pos = '\0';
  
  parent = add_tree(tree_hash, parent_path, NULL);
  
  folder_tree_add_child(parent, tree);
  
  return tree;
}

static struct folder_tree * add_storage(struct etpan_folder_list * folder_list,
    chash * tree_hash,
    struct etpan_storage * storage)
{
  struct folder_tree * root;
  char * storage_id;
  struct etpan_storage_folder_order * folder_order;
  carray * list;
  unsigned int i;
  
  storage_id = etpan_storage_get_id(storage);
  
  root = add_tree(tree_hash, storage_id, NULL);
  
  folder_order = etpan_mail_manager_storage_get_order(etpan_mail_manager_get_default(), storage);
  
  list = etpan_storage_folder_order_get_folder_order(folder_order);
  
  for(i = 0 ; i < carray_count(list) ; i ++) {
    char * folder_name;
    struct folder_tree * tree;
    struct etpan_folder * folder;
    
    folder_name = carray_get(list, i);
    folder = etpan_storage_folder_order_get_folder(folder_order, folder_name);
    tree = add_tree(tree_hash, folder_name, folder);
  }
  
  return root;
}

static void folder_tree_set_account_recursive(struct folder_tree * tree,
    struct etpan_account * account)
{
  carray * children;
  unsigned int i;
  
  folder_tree_set_account(tree, account);
  
  children = folder_tree_get_children(tree);
  for(i = 0 ; i < carray_count(children) ; i ++) {
    struct folder_tree * child;
    
    child = carray_get(children, i);
    folder_tree_set_account_recursive(child, account);
  }
}

static struct folder_tree * add_account(struct etpan_folder_list * folder_list,
    chash * tree_hash,
    struct etpan_account * account)
{
  struct folder_tree * root;
  struct etpan_storage * storage;
  
  storage = etpan_account_get_storage(account);
  if (storage == NULL) {
    ETPAN_LOG("account has no storage");
    etpan_crash();
  }
  
  root = add_storage(folder_list, tree_hash, storage);
  folder_tree_set_account_recursive(root, account);
  folder_tree_set_type(root, ETPAN_FOLDER_LIST_SELECTION_TYPE_ACCOUNT);
  
  return root;
}

static struct folder_tree * build_tree(struct etpan_folder_list * folder_list,
    struct etpan_account_manager * manager)
{
  chash * tree_hash;
  chashdatum key;
  chashdatum value;
  int r;
  struct folder_tree * root;
  carray * list;
  unsigned int i;
  
  tree_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (tree_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  list = etpan_account_manager_get_ordered_list(manager);
  for(i = 0 ; i < carray_count(list) ; i ++) {
    struct etpan_account * account;
    struct folder_tree * tree;
    
    account = carray_get(list, i);
    
    tree = add_account(folder_list, tree_hash, account);
  }
  
  key.data = "";
  key.len = 1;
  r = chash_get(tree_hash, &key, &value);
  if (r < 0) {
    root = add_tree(tree_hash, "", NULL);
  }
  else {
    root = value.data;
  }
  
  chash_free(tree_hash);
  
  return root;
}

static void update_folder_list_treeview(struct etpan_folder_list * folder_list)
{
  etpan_gtk_tree_model_reload(folder_list->folder_model);
  gtk_tree_view_expand_all(GTK_TREE_VIEW(folder_list->treeview));
}

static void update_gtk_path(struct etpan_folder_list * folder_list,
    char * gtk_path,
    struct folder_tree * parent)
{
  carray * children;
  unsigned int i;
  int first;
  int r;
  
  first = 1;
  
  children = folder_tree_get_children(parent);
  for(i = 0 ; i < carray_count(children) ; i ++) {
    struct folder_tree * child;
    char * path;
    char current_gtk_path[PATH_MAX];
    chashdatum key;
    chashdatum value;
    
    child = carray_get(children, i);
    path = folder_tree_get_ui_path(child);
    
    if (* gtk_path == '\0')
      snprintf(current_gtk_path, sizeof(current_gtk_path),
          "%u", i);
    else
      snprintf(current_gtk_path, sizeof(current_gtk_path),
          "%s:%u", gtk_path, i);

    folder_tree_set_gtk_path(child, current_gtk_path);
    
    key.data = path;
    key.len = strlen(path);
    value.data = child;
    value.len = 0;
    
    r = chash_set(folder_list->path_hash, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    update_gtk_path(folder_list, current_gtk_path, child);
  }
}

static int comp_folder_tree(struct folder_tree ** a, struct folder_tree ** b)
{
  return strcmp(folder_tree_get_ui_path(* a),
      folder_tree_get_ui_path(* b));
}

static void sort_tree(struct folder_tree * root)
{
  unsigned int i;
  
  qsort(carray_data(root->children), carray_count(root->children),
      sizeof(struct folder_tree *),
      (int (*)(const void *, const void *)) comp_folder_tree);
  
  for(i = 0 ; i < carray_count(root->children) ; i ++) {
    struct folder_tree * child;
    
    child = carray_get(root->children, i);
    sort_tree(child);
  }
}

static void diff_tree(struct etpan_folder_list * folder_list,
    struct folder_tree * old_root, struct folder_tree * new_root,
    chash * new_folders,
    carray * removed_folders_list)
{
  carray * old_children;
  carray * new_children;
  unsigned int i;
  chash * old_folder_hash;
  int r;
  
  old_children = NULL;
  new_children = NULL;
  
  if (old_root != NULL)
    old_children = old_root->children;
  if (new_root != NULL)
    new_children = new_root->children;
  
  old_folder_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYALL);
  if (old_folder_hash == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  if (old_children != NULL) {
    for(i = 0 ; i < carray_count(old_children) ; i ++) {
      struct folder_tree * child;
      char * path;
      chashdatum key;
      chashdatum value;
      
      child = carray_get(old_children, i);
      path = folder_tree_get_ui_path(child);
      
      key.data = path;
      key.len = strlen(path) + 1;
      value.data = &i;
      value.len = sizeof(i);
      r = chash_set(old_folder_hash, &key, &value, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
    }
  }
  
  if (new_children != NULL) {
    for(i = 0 ; i < carray_count(new_children) ; i ++) {
      struct folder_tree * child;
      char * path;
      chashdatum key;
      chashdatum value;
      
      child = carray_get(new_children, i);
      path = folder_tree_get_ui_path(child);
      key.data = path;
      key.len = strlen(path) + 1;
      
      r = chash_get(old_folder_hash, &key, &value);
      if (r == 0) {
        unsigned int old_index;
        struct folder_tree * old_child;
        struct folder_tree tmp_child;
        
        memcpy(&old_index, value.data, sizeof(old_index));
        old_child = carray_get(old_children, old_index);
        
        diff_tree(folder_list, old_child, child,
            new_folders, removed_folders_list);
        
        memcpy(&tmp_child, old_child, sizeof(tmp_child));
        memcpy(old_child, child, sizeof(tmp_child));
        memcpy(child, &tmp_child, sizeof(tmp_child));
        
        carray_set(new_children, i, old_child);
        carray_set(old_children, old_index, child);
        chash_delete(old_folder_hash, &key, NULL);
      }
    }
  }
  
  chash_free(old_folder_hash);
}

static void fix_children(struct folder_tree * root)
{
  carray * children;
  unsigned int i;
  
  children = root->children;
  if (children == NULL)
    return;
  
  for(i = 0 ; i < carray_count(children) ; i ++) {
    struct folder_tree * child;
    
    child = carray_get(children, i);
    child->parent = root;
    fix_children(child);
  }
}

static void update_folder_list(struct etpan_folder_list * folder_list)
{
  struct folder_tree * root;
  
  root = build_tree(folder_list, etpan_account_manager_get_default());
  
  diff_tree(folder_list, folder_list->treeinfo, root, NULL, NULL);
  fix_children(root);
  
  if (folder_list->treeinfo != NULL)
    folder_tree_free_recursive(folder_list->treeinfo);
  folder_list->treeinfo = root;
  
  update_folder_list_treeview(folder_list);
  
  chash_clear(folder_list->path_hash);
  update_gtk_path(folder_list, "", folder_list->treeinfo);
}



/* update of folder list */

struct update_list_data {
  struct etpan_folder_list * folder_list;
  struct etpan_storage * storage;
};

static void folder_list_updated_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_folder_list * folder_list;
  (void) signal_name;
  (void) signal_data;
  
  folder_list = user_data;
  update_folder_list(folder_list);
  notify_folder_selection(folder_list);
}

static struct folder_tree *
get_selection_data_from_gtk_treepath(struct etpan_folder_list * folder_list,
    GtkTreePath * path)
{
  GtkTreeIter iter;
  struct folder_tree * tree;
  GValue value;
  
  gtk_tree_model_get_iter(GTK_TREE_MODEL(folder_list->folder_model), &iter,
      path);
  
  memset(&value, 0, sizeof(value));
  
  gtk_tree_model_get_value(GTK_TREE_MODEL(folder_list->folder_model), &iter,
      STORE_INDEX_INFO, &value);
  
  tree = g_value_peek_pointer(&value);
  
  return tree;
}

static struct folder_tree *
get_selection_data(struct etpan_folder_list * folder_list,
    char * selection_path)
{
  chashdatum key;
  chashdatum value;
  int r;
  
  key.data = selection_path;
  key.len = strlen(selection_path);
  
  r = chash_get(folder_list->path_hash, &key, &value);
  if (r < 0)
    return NULL;
  
  ETPAN_FOLDERLIST_LOG("info %p", value.data);
  return value.data;
}

int
etpan_folder_list_get_selection_type(struct etpan_folder_list * folder_list,
    char * selection_path)
{
  struct folder_tree * tree;
  
  tree = get_selection_data(folder_list, selection_path);
  ETPAN_FOLDERLIST_LOG("info %p %s", tree, selection_path);
  if (tree == NULL)
    return ETPAN_FOLDER_LIST_SELECTION_TYPE_INVALID;
  
  ETPAN_FOLDERLIST_LOG("info %p", tree);
  
  if (folder_tree_get_folder(tree) != NULL)
    return ETPAN_FOLDER_LIST_SELECTION_TYPE_FOLDER;
  else
    return ETPAN_FOLDER_LIST_SELECTION_TYPE_ACCOUNT;
}

struct etpan_folder *
etpan_folder_list_get_folder(struct etpan_folder_list * folder_list,
    char * path)
{
  struct folder_tree * tree;
  
  tree = get_selection_data(folder_list, path);
  if (tree == NULL)
    return NULL;
  
  return folder_tree_get_folder(tree);
}

struct etpan_account *
etpan_folder_list_get_account(struct etpan_folder_list * folder_list,
    char * path)
{
  struct folder_tree * tree;
  
  tree = get_selection_data(folder_list, path);
  if (tree == NULL)
    return NULL;
  
  return folder_tree_get_account(tree);
}

static void
etpan_folder_list_expand_selection(struct etpan_folder_list * folder_list,
    char * selection_path)
{
  GtkTreePath * path;
  struct folder_tree * tree;
  char * gtk_path;
  
  tree = get_selection_data(folder_list, selection_path);
  if (tree == NULL)
    return;
  
  gtk_path = folder_tree_get_gtk_path(tree);
  if (gtk_path == NULL)
    return;
  
  path = gtk_tree_path_new_from_string((gchar *) gtk_path);
  if (path == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  gtk_tree_view_expand_row(GTK_TREE_VIEW(folder_list->treeview), path, FALSE);
  
  gtk_tree_path_free(path);
}

static void notify_folder_selection(struct etpan_folder_list * folder_list)
{
  GtkTreePath * path;
  GtkTreeViewColumn * column;
  int changed;
  
  changed = 0;
  
  gtk_tree_view_get_cursor(GTK_TREE_VIEW(folder_list->treeview),
      &path, &column);
  
  if (path != NULL) {
    struct folder_tree * tree;
    struct etpan_folder * folder;
    char * selection;
    
    tree = get_selection_data_from_gtk_treepath(folder_list, path);
    
    selection = folder_tree_get_ui_path(tree);
    if ((folder_list->current_selection != NULL) && (selection != NULL)) {
      if (strcmp(selection, folder_list->current_selection) != 0) {
        changed = 1;
      }
    }
    else if (folder_list->current_selection != NULL) {
      changed = 1;
    }
    else if (selection != NULL) {
      changed = 1;
    }
    
    free(folder_list->current_selection);
    folder_list->current_selection = NULL;
    
    ETPAN_FOLDERLIST_LOG("current selection : %p", tree);
    if (tree != NULL) {
      folder_list->current_selection = strdup(folder_tree_get_ui_path(tree));
      if (folder_list->current_selection == NULL)
        ETPAN_LOG_MEMORY_ERROR;
      ETPAN_FOLDERLIST_LOG("selection : %s", folder_list->current_selection);
      
#if 0
      etpan_folder_list_expand_selection(folder_list,
          folder_list->current_selection);
#endif
    }
    gtk_tree_path_free(path);
  }
  else {
    if (folder_list->current_selection != NULL)
      changed = 1;
    
    free(folder_list->current_selection);
    folder_list->current_selection = NULL;
  }
  
  if (folder_list->current_selection == NULL)
    ETPAN_FOLDERLIST_LOG("no selection");
  
  if (changed) {
    etpan_signal_send(etpan_signal_manager_get_default(),
        ETPAN_FOLDER_LIST_SELECTIONCHANGED_SIGNAL,
        folder_list, folder_list->current_selection);
  }
}


static void folder_selected_handler(GtkTreeView * treeview,
    gpointer user_data)
{
  struct etpan_folder_list * folder_list;
  (void) treeview;
  
  ETPAN_FOLDERLIST_LOG("selected");
  folder_list = (struct etpan_folder_list *) user_data;
  notify_folder_selection(folder_list);
}

static void folder_check_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_folder_list * folder_list;
  struct etpan_folder * folder;
  (void) signal_name;
  (void) sender;
  
  folder_list = user_data;
  folder = signal_data;
  
  ETPAN_FOLDERLIST_LOG("folder list check");
  etpan_folder_list_check(folder_list, folder);
}

static void update_list_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_folder_list * folder_list;
  (void) signal_name;
  (void) signal_data;
  (void) sender;
  
  folder_list = user_data;
  
  update_folder_list(folder_list);
}

static void update_font(struct etpan_folder_list * folder_list)
{
  PangoFontDescription * font_desc;
  
  font_desc = pango_font_description_from_string(etpan_ui_config_get_font_list(etpan_ui_config_get_default()));
  gtk_widget_modify_font(folder_list->treeview, font_desc);
  pango_font_description_free(font_desc);
}

static void font_changed_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_folder_list * folder_list;
  (void) signal_name;
  (void) signal_data;
  (void) sender;
  
  folder_list = user_data;
  update_font(folder_list);
}

static gboolean motion_handler(GtkWidget * widget,
    GdkEventMotion * event,
    gpointer user_data)
{
  struct etpan_folder_list * folder_list;
  (void) widget;
  (void) event;
  
  folder_list = user_data;
  
  /*
  if (!gtk_widget_is_focus(folder_list->treeview))
    gtk_widget_grab_focus(folder_list->treeview);
  */
  
  return FALSE;
}

static gboolean keypress_handler(GtkWidget * widget,
    GdkEventKey * event,
    gpointer user_data)
{
  struct etpan_folder_list * folder_list;
  
  folder_list = user_data;
  switch (event->keyval) {
  case GDK_F:
  case GDK_f:
    if ((event->state & GDK_CONTROL_MASK) != 0) {
      return TRUE;
    }
    break;
  }
  
  return FALSE;
}

static void folder_list_status_updated_handler(char * signal_name,
    void * sender, void * signal_data, void * user_data)
{
  struct etpan_folder_list * folder_list;
  (void) signal_name;
  (void) sender;
  (void) signal_data;
  
  folder_list = user_data;
  update_folder_list_treeview(folder_list);
}

void etpan_folder_list_setup(struct etpan_folder_list * folder_list)
{
  GtkTreeIter iter;
  GtkTreePath * path;
  
  folder_list->unsetup = 0;
  
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_ACCOUNT_MANAGER_MODIFICATION_SIGNAL,
      etpan_account_manager_get_default(), folder_list, update_list_handler);
  
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_MESSAGE_LIST_FOLDERCHECK_SIGNAL,
      NULL, folder_list, folder_check_handler);

  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_UI_CONFIG_FONT_CHANGED_SIGNAL,
      etpan_ui_config_get_default(), folder_list,
      font_changed_handler);
  
  ETPAN_SIGNAL_ADD_HANDLER(etpan_mail_manager_get_default(),
      ETPAN_MAIL_MANAGER_FOLDER_LIST_UPDATED_SIGNAL,
      folder_list_updated_handler,
      folder_list);
  
  ETPAN_SIGNAL_ADD_HANDLER(etpan_mail_manager_get_default(),
      ETPAN_MAIL_MANAGER_STATUS_UPDATED_SIGNAL,
      folder_list_status_updated_handler,
      folder_list);
  
  update_font(folder_list);
  
  folder_list->motion_signal_id =
    g_signal_connect(folder_list->treeview,
        "motion-notify-event", G_CALLBACK(motion_handler),
        (gpointer) folder_list);
  
  folder_list->folder_selected_signal_id =
    g_signal_connect(folder_list->treeview,
        "cursor-changed", G_CALLBACK(folder_selected_handler),
        (gpointer) folder_list);

  folder_list->keypress_signal_id =
    g_signal_connect(folder_list->treeview,
        "key-press-event", G_CALLBACK(keypress_handler),
        (gpointer) folder_list);
  
  update_folder_list(folder_list);
  
  etpan_gtk_tree_model_get_iter_from_item(folder_list->folder_model,
      &iter, NULL);
  
  path = gtk_tree_model_get_path(GTK_TREE_MODEL(folder_list->folder_model),
      &iter);
  
  gtk_tree_view_set_cursor(GTK_TREE_VIEW(folder_list->treeview),
      path, NULL, FALSE);
  
  gtk_tree_path_free(path);

  popup_menu_setup(folder_list);
  
  drag_drop_setup(folder_list);
  copy_msg_setup(folder_list);
  copy_folder_setup(folder_list);
}

void etpan_folder_list_unsetup(struct etpan_folder_list * folder_list)
{
  chashiter * iter;
  
  ETPAN_FOLDERLIST_LOG("folder list unsetup");
  
  chash_clear(folder_list->folder_progress);
  stop_animation(folder_list);
  
  copy_folder_unsetup(folder_list);
  copy_msg_unsetup(folder_list);
  drag_drop_unsetup(folder_list);
  
  popup_menu_unsetup(folder_list);
  
  g_signal_handler_disconnect(folder_list->treeview,
      folder_list->keypress_signal_id);
  
  g_signal_handler_disconnect(folder_list->treeview,
      folder_list->folder_selected_signal_id);

  g_signal_handler_disconnect(folder_list->treeview,
      folder_list->motion_signal_id);
  
  ETPAN_SIGNAL_REMOVE_HANDLER(etpan_mail_manager_get_default(),
      ETPAN_MAIL_MANAGER_STATUS_UPDATED_SIGNAL,
      folder_list_status_updated_handler,
      folder_list);
  
  ETPAN_SIGNAL_REMOVE_HANDLER(etpan_mail_manager_get_default(),
      ETPAN_MAIL_MANAGER_FOLDER_LIST_UPDATED_SIGNAL,
      folder_list_updated_handler, folder_list);
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_UI_CONFIG_FONT_CHANGED_SIGNAL,
      etpan_ui_config_get_default(), folder_list,
      font_changed_handler);
  
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_MESSAGE_LIST_FOLDERCHECK_SIGNAL,
      NULL, folder_list, folder_check_handler);

  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_ACCOUNT_MANAGER_MODIFICATION_SIGNAL,
      etpan_account_manager_get_default(), folder_list, update_list_handler);
  
  folder_list->unsetup = 1;
}

void etpan_folder_list_select(struct etpan_folder_list * folder_list,
    char * selection)
{
  struct folder_tree * tree;
  char * gtk_path;
  GtkTreePath * treepath;
  
  tree = get_selection_data(folder_list, selection);
  if (tree == NULL)
    return;
  
  gtk_path = folder_tree_get_gtk_path(tree);
  if (gtk_path == NULL)
    return;
  
  treepath = gtk_tree_path_new_from_string((gchar *) gtk_path);
  
  gtk_tree_view_expand_to_path(GTK_TREE_VIEW(folder_list->treeview), treepath);
  
  gtk_tree_view_set_cursor(GTK_TREE_VIEW(folder_list->treeview),
      treepath, NULL, FALSE);
  
  /* XXX */
  
  gtk_tree_path_free(treepath);
}

char * etpan_folder_list_get_selection(struct etpan_folder_list * folder_list)
{
  return folder_list->current_selection;
}

/* data source */

static unsigned int get_n_columns(struct etpan_gtk_tree_data_source *
      datasource)
{
  (void) datasource;
  
  return 7;
}

static GType get_column_type(struct etpan_gtk_tree_data_source * datasource,
    unsigned int column_index)
{
  static GType column_types[] = {
    G_TYPE_STRING, G_TYPE_STRING, G_TYPE_POINTER, G_TYPE_STRING, G_TYPE_UINT,
    G_TYPE_OBJECT, G_TYPE_OBJECT,
  };
  (void) datasource;
  
  return column_types[column_index];
}

static unsigned int get_children_count(struct etpan_gtk_tree_data_source *
    datasource, void * item)
{
  struct folder_tree * node;
  struct etpan_folder_list * folder_list;
  
  folder_list = datasource->data;
  
  if (item == NULL)
    node = folder_list->treeinfo;
  else
    node = item;
  
  if (node == NULL)
    return 0;
  
  return carray_count(node->children);
}

static int item_has_child(struct etpan_gtk_tree_data_source *
    datasource, void * item)
{
  int result;
  
  result = (get_children_count(datasource, item) != 0);
  
  return result;
}
  
static void * get_child_item(struct etpan_gtk_tree_data_source *
    datasource, void * item, unsigned int index)
{
  struct folder_tree * node;
  struct etpan_folder_list * folder_list;
  
  folder_list = datasource->data;
  
  if (item == NULL)
    node = folder_list->treeinfo;
  else
    node = item;
  
  if (node == NULL)
    return NULL;
  
  return carray_get(node->children, index);
}


static void get_item_value_icon(struct etpan_gtk_tree_data_source *
    datasource, void * item, GValue * value);
static void get_item_value_name(struct etpan_gtk_tree_data_source *
    datasource, void * item, GValue * value);
static void get_item_value_unread(struct etpan_gtk_tree_data_source *
    datasource, void * item, GValue * value);
static void get_item_value_info(struct etpan_gtk_tree_data_source *
    datasource, void * item, GValue * value);
static void get_item_value_image(struct etpan_gtk_tree_data_source *
    datasource, void * item, GValue * value);
static void get_item_value_imagesize(struct etpan_gtk_tree_data_source *
    datasource, void * item, GValue * value);
static void get_item_value_progress(struct etpan_gtk_tree_data_source *
    datasource, void * item, GValue * value);

static void get_item_value(struct etpan_gtk_tree_data_source *
    datasource, void * item, unsigned int column_index,
    GValue * value)
{
  switch (column_index) {
  case STORE_INDEX_NAME:
    get_item_value_name(datasource, item, value);
    break;
  case STORE_INDEX_UNREAD:
    get_item_value_unread(datasource, item, value);
    break;
  case STORE_INDEX_INFO:
    get_item_value_info(datasource, item, value);
    break;
  case STORE_INDEX_IMAGE:
    get_item_value_image(datasource, item, value);
    break;
  case STORE_INDEX_IMAGESIZE:
    get_item_value_imagesize(datasource, item, value);
    break;

  case STORE_INDEX_PROGRESS:
    get_item_value_progress(datasource, item, value);
    break;

  case STORE_INDEX_ICON:
    get_item_value_icon(datasource, item, value);
    break;
  }
}

static void get_item_value_name(struct etpan_gtk_tree_data_source *
    datasource, void * item, GValue * value)
{
  struct folder_tree * node;
  struct etpan_folder_list * folder_list;
  char ui_path[PATH_MAX];
  char * path;
  char * name;
  
  folder_list = datasource->data;
  
  if (item == NULL)
    node = folder_list->treeinfo;
  else
    node = item;
  
  path = folder_tree_get_ui_path(node);
  strncpy(ui_path, path, sizeof(ui_path));
  name = strrchr(ui_path, '/');
  if (name == NULL)
    name = ui_path;
  else
    name ++;
  
  g_value_init(value, G_TYPE_STRING);
  g_value_set_string(value, name);
}

static void get_item_value_unread(struct etpan_gtk_tree_data_source *
    datasource, void * item, GValue * value)
{
  struct folder_tree * node;
  struct etpan_folder_list * folder_list;
  char count_str[256];
  
  folder_list = datasource->data;
  
  if (item == NULL)
    node = folder_list->treeinfo;
  else
    node = item;
  
  if (folder_tree_get_folder(node) != NULL) {
    unsigned int unseen;
    struct etpan_folder * folder;
    
    folder = folder_tree_get_folder(node);
    if (etpan_mail_manager_folder_has_count(etpan_mail_manager_get_default(),
            folder)) {
      unseen = etpan_mail_manager_folder_get_unread_count(etpan_mail_manager_get_default(), folder);
      if (unseen == 0) {
        count_str[0] = '\0';
      }
      else {
        snprintf(count_str, sizeof(count_str), "%u", unseen);
      }
    }
    else {
      snprintf(count_str, sizeof(count_str), _("..."));
    }
  }
  else
    count_str[0] = '\0';
  
  g_value_init(value, G_TYPE_STRING);
  g_value_set_string(value, count_str);
}

static void get_item_value_info(struct etpan_gtk_tree_data_source *
    datasource, void * item, GValue * value)
{
  struct folder_tree * node;
  struct etpan_folder_list * folder_list;
  
  folder_list = datasource->data;
  
  if (item == NULL)
    node = folder_list->treeinfo;
  else
    node = item;
  
  g_value_init(value, G_TYPE_POINTER);
  g_value_set_pointer(value, node);
}

static void get_item_value_icon(struct etpan_gtk_tree_data_source *
    datasource, void * item, GValue * value)
{
  struct folder_tree * node;
  struct etpan_folder_list * folder_list;
  GdkPixbuf * pixbuf;
  
  folder_list = datasource->data;
  
  if (item == NULL)
    node = folder_list->treeinfo;
  else
    node = item;

#define ICON(name) etpan_icon_manager_get_scaled_pixbuf(etpan_icon_manager_get_default(), name, 16)

  switch (folder_tree_get_type(node)) {
  case ETPAN_FOLDER_LIST_SELECTION_TYPE_ACCOUNT:
    {
      int type;
      
      type = folder_tree_get_account_type(node);
      switch (type) {
      case STORAGE_DRIVER_TYPE_POP:
      case STORAGE_DRIVER_TYPE_IMAP:
      case STORAGE_DRIVER_TYPE_NEWS:
        pixbuf = ICON("account-remote");
        break;
      case STORAGE_DRIVER_TYPE_MAILDIR:
      case STORAGE_DRIVER_TYPE_MH:
      case STORAGE_DRIVER_TYPE_MBOX:
        pixbuf = ICON("account");
        break;
      default:
        pixbuf = NULL;
        break;
      }
    }
    break;
  case ETPAN_FOLDER_LIST_SELECTION_TYPE_FOLDER:
    pixbuf = ICON("folder");
    break;
  default:
    pixbuf = NULL;
    break;
  }
#undef ICON

  g_value_init(value, G_TYPE_OBJECT);
  g_value_set_object(value, pixbuf);
}

static void get_item_value_image(struct etpan_gtk_tree_data_source *
    datasource, void * item, GValue * value)
{
  struct folder_tree * node;
  struct etpan_folder_list * folder_list;
  char * stock_id;
  
  folder_list = datasource->data;
  
  if (item == NULL)
    node = folder_list->treeinfo;
  else
    node = item;
  
  switch (folder_tree_get_type(node)) {
  case ETPAN_FOLDER_LIST_SELECTION_TYPE_ACCOUNT:
    stock_id = GTK_STOCK_HOME;
    break;
  case ETPAN_FOLDER_LIST_SELECTION_TYPE_FOLDER:
    stock_id = GTK_STOCK_DIRECTORY;
    break;
  default:
    stock_id = GTK_STOCK_DND_MULTIPLE;
    break;
  }
  
  g_value_init(value, G_TYPE_STRING);
  g_value_set_string(value, stock_id);
}

static void get_item_value_imagesize(struct etpan_gtk_tree_data_source *
    datasource, void * item, GValue * value)
{
  struct folder_tree * node;
  struct etpan_folder_list * folder_list;
  
  folder_list = datasource->data;
  
  if (item == NULL)
    node = folder_list->treeinfo;
  else
    node = item;
  
  g_value_init(value, G_TYPE_UINT);
  g_value_set_uint(value, GTK_ICON_SIZE_MENU);
}

static void get_item_value_progress(struct etpan_gtk_tree_data_source *
    datasource, void * item, GValue * value)
{
  struct folder_tree * node;
  struct etpan_folder_list * folder_list;
  GdkPixbuf * pixbuf;
  
  folder_list = datasource->data;
  
  if (item == NULL)
    node = folder_list->treeinfo;
  else
    node = item;

  switch (folder_tree_get_type(node)) {
  case ETPAN_FOLDER_LIST_SELECTION_TYPE_ACCOUNT:
    {
      int r;
      struct etpan_storage * storage;
      chashdatum key;
      chashdatum value;
      
      storage = etpan_account_get_storage(node->account);
      key.data = &storage;
      key.len = sizeof(storage);
      
      r = chash_get(folder_list->folder_progress, &key, &value);
      if (r < 0) {
        pixbuf = NULL;
      }
      else {
        struct anim_info * info;
        
        info = value.data;
        pixbuf = folder_list->progress_anim[info->image_index];
      }
    }
    break;
  case ETPAN_FOLDER_LIST_SELECTION_TYPE_FOLDER:
    {
      int r;
      struct etpan_folder * folder;
      chashdatum key;
      chashdatum value;
      
      folder = node->folder;
      key.data = &folder;
      key.len = sizeof(folder);
      
      r = chash_get(folder_list->folder_progress, &key, &value);
      if (r < 0) {
        pixbuf = NULL;
      }
      else {
        struct anim_info * info;
        
        info = value.data;
        pixbuf = folder_list->progress_anim[info->image_index];
      }
    }
    break;
  default:
    pixbuf = NULL;
    break;
  }
  
  g_value_init(value, G_TYPE_OBJECT);
  g_value_set_object(value, pixbuf);
}

static carray * get_folder_list(struct etpan_folder_list * folder_list)
{
  struct etpan_account_manager * manager;
  carray * result;
  unsigned int i;
  int r;
  carray * account_list;
  
  result = carray_new(16);
  if (result == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  manager = etpan_account_manager_get_default();
  
  account_list = etpan_account_manager_get_ordered_list(manager);
  for(i = 0 ; i < carray_count(account_list) ; i ++) {
    struct etpan_account * account;
    chash * folder_hash;
    chashiter * iter;
    struct etpan_storage * storage;
    
    account = carray_get(account_list, i);
    storage = etpan_account_get_storage(account);
    if (storage == NULL) {
      ETPAN_LOG("account has no storage");
      etpan_crash();
    }
    
    folder_hash = etpan_storage_get_folder_list(storage);
    for(iter = chash_begin(folder_hash) ; iter != NULL ;
        iter = chash_next(folder_hash, iter)) {
      chashdatum value;
      struct etpan_folder * folder;
      
      chash_value(iter, &value);
      folder = value.data;
      r = carray_add(result, folder, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
    }
  }
  
  return result;
}

void etpan_folder_list_check(struct etpan_folder_list * folder_list,
    struct etpan_folder * folder)
{
  etpan_mail_manager_folder_check(etpan_mail_manager_get_default(), folder);
}



/* progress animation */

#define ANIMATION_INTERVAL 100

static gboolean do_animation(gpointer data)
{
  struct etpan_folder_list * folder_list;
  chashiter * iter;
  int r;
  
  folder_list = data;
  
  for(iter = chash_begin(folder_list->folder_progress) ;
      iter != NULL ; iter = chash_next(folder_list->folder_progress, iter)) {
    chashdatum key;
    chashdatum value;
    struct anim_info info;
    
    chash_key(iter, &key);
    chash_value(iter, &value);
    memcpy(&info, value.data, sizeof(info));
    info.image_index ++;
    info.image_index %= PROGRESS_ANIM_COUNT;
    value.data = &info;
    r = chash_set(folder_list->folder_progress, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  update_folder_list_treeview(folder_list);
  
  return TRUE;
}

static void start_animation(struct etpan_folder_list * folder_list)
{
  if (folder_list->progress_animation)
    return;
  
  folder_list->progress_animation = 1;
  
  folder_list->progress_timer_id = g_timeout_add(ANIMATION_INTERVAL,
      (GSourceFunc) do_animation, (gpointer) folder_list);
}

static void stop_animation(struct etpan_folder_list * folder_list)
{
  if (!folder_list->progress_animation)
    return;
  
  if (chash_count(folder_list->folder_progress) > 0)
    return;
  
  g_source_remove(folder_list->progress_timer_id);
  
  folder_list->progress_animation = 0;
}


static void start_animation_object(struct etpan_folder_list * folder_list,
    void * object);
static void stop_animation_object(struct etpan_folder_list * folder_list,
    void * object);

static void start_animation_storage(struct etpan_folder_list * folder_list,
    struct etpan_storage * storage)
{
  start_animation_object(folder_list, storage);
}

static void stop_animation_storage(struct etpan_folder_list * folder_list,
    struct etpan_storage * storage)
{
  stop_animation_object(folder_list, storage);
}

static void start_animation_folder(struct etpan_folder_list * folder_list,
    struct etpan_folder * folder)
{
  start_animation_object(folder_list, folder);
}

static void stop_animation_folder(struct etpan_folder_list * folder_list,
    struct etpan_folder * folder)
{
  stop_animation_object(folder_list, folder);
}

static void start_animation_object(struct etpan_folder_list * folder_list,
    void * object)
{
  chashdatum key;
  chashdatum value;
  struct anim_info info;
  int r;
  
  key.data = &object;
  key.len = sizeof(object);
  r = chash_get(folder_list->folder_progress, &key, &value);
  if (r >= 0) {
    memcpy(&info, value.data, sizeof(info));
    info.count ++;
    value.data = &info;
    r = chash_set(folder_list->folder_progress, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    return;
  }
  
  key.data = &object;
  key.len = sizeof(object);
  info.image_index = 0;
  info.count = 1;
  value.data = &info;
  value.len = sizeof(info);
  
  r = chash_set(folder_list->folder_progress, &key, &value, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  start_animation(folder_list);
}

static void stop_animation_object(struct etpan_folder_list * folder_list,
    void * object)
{
  chashdatum key;
  chashdatum value;
  struct anim_info info;
  int r;
  
  key.data = &object;
  key.len = sizeof(object);
  r = chash_get(folder_list->folder_progress, &key, &value);
  if (r < 0)
    return;
  
  memcpy(&info, value.data, sizeof(info));
  info.count --;
  if (info.count == 0) {
    chash_delete(folder_list->folder_progress, &key, NULL);
  }
  else {
    value.data = &info;
    value.len = sizeof(info);
    r = chash_set(folder_list->folder_progress, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  update_folder_list_treeview(folder_list);
  
  stop_animation(folder_list);
}

void etpan_folder_list_start_animation_folder(struct etpan_folder_list * folder_list,
    struct etpan_folder * folder)
{
#if 0
  ETPAN_LOG("start anim folder %p", folder);
#endif
  start_animation_folder(folder_list, folder);
}

void etpan_folder_list_stop_animation_folder(struct etpan_folder_list * folder_list,
    struct etpan_folder * folder)
{
#if 0
  ETPAN_LOG("stop anim folder %p", folder);
#endif
  stop_animation_folder(folder_list, folder);
}


/* drag and drop */

/* drop */

/* extracted from GTK - begin */
#define AUTO_EXPAND_TIMEOUT 500
#define AUTO_SCROLL_TIMEOUT 150
#define SCROLL_EDGE_SIZE 15

static void gtk_tree_view_vertical_autoscroll (GtkTreeView *tree_view)
{
  GdkRectangle visible_rect;
  gint y;
  gint offset;
  gfloat value;

  gdk_window_get_pointer (tree_view->priv->bin_window, NULL, &y, NULL);
  y += tree_view->priv->dy;

  gtk_tree_view_get_visible_rect (tree_view, &visible_rect);

  /* see if we are near the edge. */
  offset = y - (visible_rect.y + 2 * SCROLL_EDGE_SIZE);
  if (offset > 0)
    {
      offset = y - (visible_rect.y + visible_rect.height - 2 * SCROLL_EDGE_SIZE);
      if (offset < 0)
	return;
    }

  value = CLAMP (tree_view->priv->vadjustment->value + offset, 0.0,
		 tree_view->priv->vadjustment->upper - tree_view->priv->vadjustment->page_size);
  gtk_adjustment_set_value (tree_view->priv->vadjustment, value);
}

static gint scroll_row_timeout (gpointer data)
{
  GtkTreeView * tree_view;
  
  tree_view = data;
  
  gtk_tree_view_vertical_autoscroll (tree_view);
  
  return TRUE;
}

static gboolean auto_expand_timeout (gpointer data)
{
#if 0
  GtkTreeView *tree_view = GTK_TREE_VIEW (data);
#endif
  struct etpan_folder_list * folder_list;
  GtkTreePath * path;
  
  folder_list = data;
  
  gtk_tree_view_get_dest_row_at_pos(GTK_TREE_VIEW(folder_list->treeview),
      folder_list->drag_auto_expand_x, folder_list->drag_auto_expand_y,
      &path, NULL);
  if (path != NULL) {
    gtk_tree_view_expand_row(GTK_TREE_VIEW(folder_list->treeview),
        path, FALSE);
    
    /* XXX - should remember opened folder and close them back */
#if 0
    tree = get_selection_data_from_gtk_treepath(folder_list, path);
    chash_set(folder_list->drag_expanded_hash, &key, &value, NULL);
#endif
    
    gtk_tree_path_free(path);
  }
  
  return FALSE;
}
/* extracted from GTK - end */

enum {
  DROP_TYPE_UNKNOWN,
  DROP_TYPE_FOLDER,
  DROP_TYPE_MSGLIST,
};

static gboolean drag_motion_handler(GtkWidget * widget,
    GdkDragContext * drag_context,
    gint x,
    gint y,
    guint time,
    gpointer user_data)
{
  struct etpan_folder_list * folder_list;
  GList * targets;
  (void) widget;
  
  folder_list = user_data;
  if (!folder_list->drag_entered) {
    folder_list->drag_entered = 1;
    gtk_drag_highlight(folder_list->scrolledwindow);
    folder_list->drag_scroll_timeout_id =
      g_timeout_add(AUTO_SCROLL_TIMEOUT, scroll_row_timeout, folder_list->treeview);
    
    targets = drag_context->targets;
    while (targets != NULL) {
      char * name;
      
      name = gdk_atom_name(GDK_POINTER_TO_ATOM(targets->data));
      
      if (strcmp(name, ETPAN_FOLDER_LIST_DRAG_FOLDER_NAME) == 0) {
        folder_list->drag_drop_type = DROP_TYPE_FOLDER;
      }
      else if (strcmp(name, ETPAN_MESSAGE_LIST_DRAG_MSGLIST_NAME) == 0) {
        folder_list->drag_drop_type = DROP_TYPE_MSGLIST;
      }
      else {
        folder_list->drag_drop_type = DROP_TYPE_UNKNOWN;
      }
      
      g_free (name);
      
      targets = targets->next;
    }
  }
  else {
    g_source_remove(folder_list->drag_auto_expand_timeout_id);
  }
  
  switch (folder_list->drag_drop_type) {
  case DROP_TYPE_MSGLIST:
    {
      GtkTreePath * path;
      struct folder_tree * tree;
      struct etpan_folder * folder;
      
      gtk_tree_view_get_dest_row_at_pos(GTK_TREE_VIEW(folder_list->treeview),
          x, y, &path, NULL);
      
      folder_list->drag_has_dest_folder = 0;
      if (path != NULL) {
        tree = get_selection_data_from_gtk_treepath(folder_list, path);
        folder = folder_tree_get_folder(tree);
        
        if (folder != NULL) {
          folder_list->drag_has_dest_folder = 1;
        }
      }
      
      if (folder_list->drag_has_dest_folder)
        gtk_tree_view_set_drag_dest_row(GTK_TREE_VIEW(folder_list->treeview),
            path, GTK_TREE_VIEW_DROP_INTO_OR_AFTER);
      else
        gtk_tree_view_set_drag_dest_row(GTK_TREE_VIEW(folder_list->treeview),
            NULL, GTK_TREE_VIEW_DROP_INTO_OR_AFTER);
      
      gtk_tree_path_free(path);
      
      if (!folder_list->drag_has_dest_folder) {
        gdk_drag_status(drag_context, 0, time);
      }
      else {
        if (drag_context->actions == GDK_ACTION_MOVE)
          gdk_drag_status(drag_context, GDK_ACTION_MOVE, time);
        else if (drag_context->actions == GDK_ACTION_COPY)
          gdk_drag_status(drag_context, GDK_ACTION_COPY, time);
        else if (folder_list->drag_has_dest_folder)
          gdk_drag_status(drag_context, GDK_ACTION_MOVE, time);
        else
          gdk_drag_status(drag_context, 0, time);
      }
    }
    break;
  case DROP_TYPE_FOLDER:
    {
      GtkTreePath * path;
      GtkTreeViewDropPosition pos;
      
      gtk_tree_view_get_dest_row_at_pos(GTK_TREE_VIEW(folder_list->treeview),
          x, y, &path, &pos);
      
      folder_list->drag_has_dest_folder = 0;
      
      if (path != NULL) {
        if ((pos == GTK_TREE_VIEW_DROP_INTO_OR_BEFORE) ||
            (pos == GTK_TREE_VIEW_DROP_INTO_OR_AFTER)) {
          /* pos into */
          struct folder_tree * tree;
          struct etpan_folder * folder;
          struct etpan_account * account;
          struct etpan_storage * storage;
          int allow;
          
          tree = get_selection_data_from_gtk_treepath(folder_list, path);
          ETPAN_LOG("tree : %p %s", tree, path);
          folder = folder_tree_get_folder(tree);
          account = folder_tree_get_account(tree);
          storage = NULL;
          if (account != NULL)
            storage = etpan_account_get_storage(account);
          /*
            depending on type of tree node : account or folder
            depending on type of of account
            allow drag & drop or not.
          */
          allow = 1;
          switch (folder_tree_get_type(tree)) {
          case ETPAN_FOLDER_LIST_SELECTION_TYPE_ACCOUNT:
            if (!etpan_storage_allows_sub_folders(storage, NULL))
              allow = 0;
            break;
          case ETPAN_FOLDER_LIST_SELECTION_TYPE_FOLDER:
            if (!etpan_storage_allows_sub_folders(storage, folder))
              allow = 0;
            break;
          }
          
          if (allow) {
            /* drop at same place ? */
            struct folder_tree * source_tree;
            
            source_tree = get_selection_data(folder_list,
                folder_list->current_selection);
            if (folder_tree_get_type(source_tree) ==
                ETPAN_FOLDER_LIST_SELECTION_TYPE_ACCOUNT)
              allow = 0;
          }
          
          if (allow) {
            /* drop at same place ? */
            struct folder_tree * source_tree;
            struct folder_tree * parent_tree;
            
            source_tree = get_selection_data(folder_list,
                folder_list->current_selection);
            parent_tree = folder_tree_get_parent(source_tree);
            if (parent_tree == tree) {
              allow = 0;
            }
          }
          
          if (allow) {
            /* drop in children ? */
            struct folder_tree * source_tree;
            
            source_tree = get_selection_data(folder_list,
                folder_list->current_selection);
            
            while (tree != NULL) {
              if (tree == source_tree)
                break;
              
              tree = folder_tree_get_parent(tree);
            }
            
            if (tree != NULL) {
              allow = 0;
            }
          }
          
          folder_list->drag_has_dest_folder = allow;
        }
        else {
          /* pos before or after */
          struct folder_tree * tree;
          struct folder_tree * parent_tree;
          struct folder_tree * source_tree;
          struct folder_tree * source_parent;
          
          source_tree = get_selection_data(folder_list,
              folder_list->current_selection);
          source_parent = folder_tree_get_parent(source_tree);
          
          folder_list->drag_has_dest_folder = 0;
          
          tree = get_selection_data_from_gtk_treepath(folder_list, path);
          parent_tree = folder_tree_get_parent(tree);
          
          if (parent_tree != NULL) {
            struct etpan_account * account;
            struct etpan_storage * storage;
            int allow;
            carray * sibling;
            
            account = folder_tree_get_account(parent_tree);
            storage = NULL;
            if (account != NULL)
              storage = etpan_account_get_storage(account);
            allow = 1;
            
            if (parent_tree == folder_list->treeinfo)
              allow = 0;
            
            if (folder_tree_get_type(source_tree) ==
                ETPAN_FOLDER_LIST_SELECTION_TYPE_ACCOUNT) {
              if (source_parent != parent_tree)
                allow = 0;
            }
            else {
              struct etpan_folder * parent_folder;
              
              parent_folder = folder_tree_get_folder(parent_tree);
              switch (folder_tree_get_type(parent_tree)) {
              case ETPAN_FOLDER_LIST_SELECTION_TYPE_ACCOUNT:
                if (!etpan_storage_allows_sub_folders(storage, NULL))
                  allow = 0;
                break;
              case ETPAN_FOLDER_LIST_SELECTION_TYPE_FOLDER:
                if (!etpan_storage_allows_sub_folders(storage, parent_folder))
                  allow = 0;
                break;
              }
            }
            
            if (allow) {
              /* drop in children ? */
              if (source_tree != NULL) {
                struct folder_tree * cur_tree;
                
                cur_tree = tree;
                while (cur_tree != NULL) {
                  if (cur_tree == source_tree)
                    break;
                  
                  cur_tree = folder_tree_get_parent(cur_tree);
                }
                
                if (cur_tree != NULL) {
                  allow = 0;
                }
              }
            }
            
            sibling = folder_tree_get_children(parent_tree);
            if (allow) {
              if (pos == GTK_TREE_VIEW_DROP_BEFORE) {
                if (tree != carray_get(sibling, 0))
                  allow = 0;
              }
            }
            
            if (allow) {
              if (pos == GTK_TREE_VIEW_DROP_AFTER) {
                unsigned int tree_index;
                unsigned int i;
                
                tree_index = 0;
                for(i = 0 ; i < carray_count(sibling) ; i ++) {
                  if (carray_get(sibling, i) == tree) {
                    tree_index = i;
                    break;
                  }
                }
                if (tree_index + 1 < carray_count(sibling)) {
                  if (carray_get(sibling, tree_index + 1) == source_tree) {
                    allow = 0;
                  }
                }
              }
            }
            
            folder_list->drag_has_dest_folder = allow;
          }
        }
      }
      
      if (folder_list->drag_has_dest_folder)
        gtk_tree_view_set_drag_dest_row(GTK_TREE_VIEW(folder_list->treeview),
            path, pos);
      else
        gtk_tree_view_set_drag_dest_row(GTK_TREE_VIEW(folder_list->treeview),
            NULL, GTK_TREE_VIEW_DROP_INTO_OR_AFTER);
      
      gtk_tree_path_free(path);
      
      if (!folder_list->drag_has_dest_folder) {
        gdk_drag_status(drag_context, 0, time);
      }
      else {
        if (drag_context->actions == GDK_ACTION_MOVE)
          gdk_drag_status(drag_context, GDK_ACTION_MOVE, time);
        else if (drag_context->actions == GDK_ACTION_COPY)
          gdk_drag_status(drag_context, GDK_ACTION_COPY, time);
        else if (folder_list->drag_has_dest_folder)
          gdk_drag_status(drag_context, GDK_ACTION_MOVE, time);
        else
          gdk_drag_status(drag_context, 0, time);
      }
    }
    break;
  }
  
  folder_list->drag_auto_expand_x = x;
  folder_list->drag_auto_expand_y = y;
  folder_list->drag_auto_expand_timeout_id = 
    g_timeout_add(AUTO_EXPAND_TIMEOUT, auto_expand_timeout, folder_list);
  
  return TRUE;
}

static void drag_leave_handler(GtkWidget * widget,
    GdkDragContext * drag_context,
    guint time,
    gpointer user_data)
{
  struct etpan_folder_list * folder_list;
  (void) widget;
  (void) drag_context;
  (void) time;
  
  ETPAN_LOG("leave");
  folder_list = user_data;
  folder_list->drag_entered = 0;
  gtk_drag_unhighlight(folder_list->scrolledwindow);
  g_source_remove(folder_list->drag_scroll_timeout_id);
  g_source_remove(folder_list->drag_auto_expand_timeout_id);
  gtk_tree_view_set_drag_dest_row(GTK_TREE_VIEW(folder_list->treeview), NULL,
      GTK_TREE_VIEW_DROP_INTO_OR_AFTER);
}

/* drag */

static gboolean drag_button_press_handler(GtkWidget * widget,
    GdkEventButton * event,
    gpointer user_data)
{
  struct etpan_folder_list * folder_list;
  GtkTreePath * path;
  int do_free_path;
  gboolean result;
  GtkTreeView * tree_view;
  (void) tree_view;
  (void) widget;
  
  folder_list = user_data;
  
  result = FALSE;
  do_free_path = 0;
  if (!result) {
    if (event->button == 1) {
      if (gtk_tree_view_get_path_at_pos(GTK_TREE_VIEW(folder_list->treeview),
              event->x, event->y, &path, NULL, NULL, NULL)) {
        GtkTreeSelection * selection;
        
        do_free_path = 1;
        selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(folder_list->treeview));
        if (gtk_tree_selection_path_is_selected(selection, path)) {
          ETPAN_LOG("start drag");
          drag_may_start(folder_list, event->x, event->y, event->time);
          result = TRUE;
        }
        else if ((event->state & (GDK_SHIFT_MASK | GDK_CONTROL_MASK)) != 0) {
          /* do nothing */
        }
        else {
          drag_may_start(folder_list, event->x, event->y, event->time);
        }
      }
    }
    else if (event->button == 3) {
      if (gtk_tree_view_get_path_at_pos(GTK_TREE_VIEW(folder_list->treeview),
              event->x, event->y, &path, NULL, NULL, NULL)) {
        GtkTreeSelection * selection;
        
        do_free_path = 1;
        selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(folder_list->treeview));
        if (gtk_tree_selection_path_is_selected(selection, path)) {
          show_folder_popup_menu(folder_list, folder_list->treeview, event);
          
          result = TRUE;
        }
      }
    }
  }
  
  if (do_free_path)
    gtk_tree_path_free(path);
  
  return result;
}

static gboolean drag_button_release_handler(GtkWidget * widget,
    GdkEventButton * event,
    gpointer user_data)
{
  struct etpan_folder_list * folder_list;
  (void) widget;
  (void) event;
  
  folder_list = user_data;
  
  if (drag_may_be_starting(folder_list)) {
    drag_cancel_may_start(folder_list);
  }
  
  return FALSE;
}

static gboolean drag_mouse_motion_handler(GtkWidget * widget,
    GdkEventMotion * event,
    gpointer user_data)
{
  struct etpan_folder_list * folder_list;
  (void) widget;
  
  folder_list = user_data;
  
  if (drag_may_be_starting(folder_list)) {
    /* check if real drag can start */
    if (drag_is_valid(folder_list, event->x, event->y, event->time)) {
      drag_start(folder_list, (GdkEvent *) event);
    }
  }
  
  return FALSE;
}

static void drag_end_handler(GtkWidget * widget,
    GdkDragContext * drag_context,
    gpointer user_data)
{
  struct etpan_folder_list * folder_list;
  (void) widget;
  (void) drag_context;
  
  folder_list = user_data;
  drag_end(folder_list);
}

static gboolean drag_drop_handler(GtkWidget * widget,
    GdkDragContext * drag_context,
    gint x, gint y,
    guint time,
    gpointer user_data)
{
  struct etpan_folder_list * folder_list;
  (void) widget;
  
  folder_list = user_data;
  
  ETPAN_LOG("drop handler");
  
  switch (folder_list->drag_drop_type) {
  case DROP_TYPE_MSGLIST:
    if (drag_context->action == GDK_ACTION_MOVE) {
      struct folder_tree * tree;
      struct etpan_folder * folder;
      GtkTreePath * path;
      
      gtk_tree_view_get_dest_row_at_pos(GTK_TREE_VIEW(folder_list->treeview),
          x, y, &path, NULL);
      
      tree = get_selection_data_from_gtk_treepath(folder_list, path);
      folder = folder_tree_get_folder(tree);
      ETPAN_LOG("move %p %p", tree, folder);
      
      gtk_drag_finish(drag_context, TRUE, TRUE, time);
      
      gtk_tree_path_free(path);
      
      etpan_signal_send(etpan_signal_manager_get_default(),
          ETPAN_FOLDER_LIST_DRAG_MOVE_MSGLIST_SIGNAL, folder_list,
          folder);
    }
    else if (drag_context->action == GDK_ACTION_COPY) {
      struct folder_tree * tree;
      struct etpan_folder * folder;
      GtkTreePath * path;
      
      gtk_tree_view_get_dest_row_at_pos(GTK_TREE_VIEW(folder_list->treeview),
          x, y, &path, NULL);
      
      tree = get_selection_data_from_gtk_treepath(folder_list, path);
      folder = folder_tree_get_folder(tree);
      
      gtk_drag_finish(drag_context, TRUE, FALSE, time);
      
      gtk_tree_path_free(path);
      
      etpan_signal_send(etpan_signal_manager_get_default(),
          ETPAN_FOLDER_LIST_DRAG_COPY_MSGLIST_SIGNAL, folder_list,
          folder);
    }
    break;
  case DROP_TYPE_FOLDER:
    if ((drag_context->action == GDK_ACTION_MOVE) ||
        (drag_context->action == GDK_ACTION_COPY)) {
      struct folder_tree * dest_tree;
      struct folder_tree * dest_parent;
      struct etpan_folder * dest_folder;
      struct folder_tree * source_parent;
      struct folder_tree * source_tree;
      struct etpan_folder * source_folder;
      int drop_type;
      GtkTreePath * path;
      GtkTreeViewDropPosition pos;
      struct etpan_account * dest_account;
      struct etpan_storage * dest_storage;
      int do_delete;
      
      if (drag_context->action == GDK_ACTION_MOVE) {
        do_delete = 1;
      }
      else {
        do_delete = 0;
      }
      
      /* source */
      source_tree = get_selection_data(folder_list,
          folder_list->current_selection);
      source_folder = folder_tree_get_folder(source_tree);
      source_parent = folder_tree_get_parent(source_tree);
      
      /* destination */
      gtk_tree_view_get_dest_row_at_pos(GTK_TREE_VIEW(folder_list->treeview),
          x, y, &path, &pos);
      dest_tree = get_selection_data_from_gtk_treepath(folder_list, path);
      dest_folder = folder_tree_get_folder(dest_tree);
      dest_account = folder_tree_get_account(dest_tree);
      dest_storage = etpan_account_get_storage(dest_account);
      dest_parent = folder_tree_get_parent(dest_tree);
      gtk_tree_path_free(path);
      
      drop_type = ETPAN_FOLDER_LIST_DROP_INVALID;
      switch (pos) {
      case GTK_TREE_VIEW_DROP_BEFORE:
        drop_type = ETPAN_FOLDER_LIST_DROP_BEFORE;
        break;
      case GTK_TREE_VIEW_DROP_AFTER:
        drop_type = ETPAN_FOLDER_LIST_DROP_AFTER;
        break;
      case GTK_TREE_VIEW_DROP_INTO_OR_BEFORE:
      case GTK_TREE_VIEW_DROP_INTO_OR_AFTER:
        drop_type = ETPAN_FOLDER_LIST_DROP_INTO;
        break;
      }
      
      if (drop_type == ETPAN_FOLDER_LIST_DROP_INTO) {
        etpan_folder_list_copy_folder(folder_list,
            source_folder, dest_storage, dest_folder, do_delete);
        folder_list->copy_dest_path = NULL;
        folder_list->copy_dest_type = drop_type;
      }
      else if (drop_type == ETPAN_FOLDER_LIST_DROP_BEFORE) {
        if (dest_parent != source_parent) {
          struct etpan_folder * dest_parent_folder;
          char * dest_ui_path;
          
          dest_parent_folder = folder_tree_get_folder(dest_parent);
          etpan_folder_list_copy_folder(folder_list,
              source_folder, dest_storage, dest_parent_folder, do_delete);
          /* reorder after callback */
          dest_ui_path = folder_tree_get_ui_path(dest_tree);
          folder_list->copy_dest_path = strdup(dest_ui_path);
          folder_list->copy_dest_type = drop_type;
        }
        else {
          /* reorder immediately */
          reorder(folder_list,
              source_tree, ETPAN_FOLDER_LIST_DROP_BEFORE, dest_tree);
        }
      }
      else if (drop_type == ETPAN_FOLDER_LIST_DROP_AFTER) {
        if (dest_parent != source_parent) {
          struct etpan_folder * dest_parent_folder;
          char * dest_ui_path;
          
          dest_parent_folder = folder_tree_get_folder(dest_parent);
          etpan_folder_list_copy_folder(folder_list,
              source_folder, dest_storage, dest_parent_folder, do_delete);
          /* reorder after callback */
          dest_ui_path = folder_tree_get_ui_path(dest_tree);
          folder_list->copy_dest_path = strdup(dest_ui_path);
          folder_list->copy_dest_type = drop_type;
        }
        else {
          /* reorder immediately */
          reorder(folder_list,
              source_tree, ETPAN_FOLDER_LIST_DROP_AFTER, dest_tree);
        }
      }
      
      gtk_drag_finish(drag_context, TRUE, TRUE, time);
    }
    else {
      gtk_drag_finish(drag_context, FALSE, FALSE, time);
    }
    break;
  }
  ETPAN_LOG("drop ok");
  
  return TRUE;
}

static void reorder(struct etpan_folder_list * folder_list,
    struct folder_tree * source, int type,
    struct folder_tree * dest)
{
  struct folder_tree * parent;
  carray * sibling_list;
  carray * new_order;
  struct etpan_account * account;
  struct etpan_storage * storage;
  chashdatum key;
  chashdatum value;
  char * storage_id;
  int r;
  unsigned int i;
  struct etpan_storage_folder_order * folder_order;
  
  parent = folder_tree_get_parent(source);
  account = folder_tree_get_account(parent);
  storage = etpan_account_get_storage(account);
  sibling_list = folder_tree_get_children(parent);
  new_order = carray_new(carray_count(sibling_list));
  if (new_order == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  for(i = 0 ; i < carray_count(sibling_list) ; i ++) {
    struct folder_tree * sibling;
    
    sibling = carray_get(sibling_list, i);
    
    if (sibling == source)
      continue;
      
    if (sibling == dest) {
      char * source_ui_path;
      char * dest_ui_path;
        
      dest_ui_path = folder_tree_get_ui_path(dest);
      source_ui_path = folder_tree_get_ui_path(source);

      switch (type) {
      case ETPAN_FOLDER_LIST_DROP_BEFORE:
        r = carray_add(new_order, source_ui_path, NULL);
        if (r < 0)
          ETPAN_LOG_MEMORY_ERROR;
        r = carray_add(new_order, dest_ui_path, NULL);
        if (r < 0)
          ETPAN_LOG_MEMORY_ERROR;
        break;
      case ETPAN_FOLDER_LIST_DROP_AFTER:
        r = carray_add(new_order, dest_ui_path, NULL);
        if (r < 0)
          ETPAN_LOG_MEMORY_ERROR;
        r = carray_add(new_order, source_ui_path, NULL);
        if (r < 0)
          ETPAN_LOG_MEMORY_ERROR;
        break;
      }
    }
    else {
      char * ui_path;
      
      ui_path = folder_tree_get_ui_path(sibling);
      r = carray_add(new_order, ui_path, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
    }
  }
  
  folder_order = etpan_mail_manager_storage_get_order(etpan_mail_manager_get_default(), storage);
  
  etpan_storage_folder_order_set_folder_order(folder_order, new_order);
  
  carray_free(new_order);
}

static void drag_begin_handler(GtkWidget * widget,
    GdkDragContext * drag_context,
    gpointer user_data)
{
  struct etpan_folder_list * folder_list;
  (void) widget;
  
  folder_list = user_data;
  
  folder_list->drag_context = drag_context;
  drag_set_icon(folder_list);
}

static void drag_drop_setup(struct etpan_folder_list * folder_list)
{
  GtkTargetList * targets;
  GdkAtom atom;
  
  /* drop */
  gtk_drag_dest_set(folder_list->scrolledwindow,
      GTK_DEST_DEFAULT_MOTION, /* flags */
      NULL, 0,
      GDK_ACTION_COPY | GDK_ACTION_MOVE);
  
  targets = gtk_target_list_new(NULL, 0);
  atom = gdk_atom_intern(ETPAN_MESSAGE_LIST_DRAG_MSGLIST_NAME, 0);
  gtk_target_list_add(targets, atom, GTK_TARGET_SAME_APP, 0);
  atom = gdk_atom_intern(ETPAN_FOLDER_LIST_DRAG_FOLDER_NAME, 0);
  gtk_target_list_add(targets, atom, GTK_TARGET_SAME_APP, 0);
  gtk_drag_dest_set_target_list(folder_list->scrolledwindow, targets);
  gtk_target_list_unref(targets);
  
  folder_list->drag_motion_signal_id =
    g_signal_connect(folder_list->scrolledwindow,
        "drag-motion", G_CALLBACK(drag_motion_handler),
        (gpointer) folder_list);

  folder_list->drag_leave_signal_id =
    g_signal_connect(folder_list->scrolledwindow,
        "drag-leave", G_CALLBACK(drag_leave_handler),
        (gpointer) folder_list);

  folder_list->drag_drop_signal_id =
    g_signal_connect(folder_list->scrolledwindow,
        "drag-drop", G_CALLBACK(drag_drop_handler),
        (gpointer) folder_list);
  
  folder_list->drag_entered = 0;
  
  /* drag */
  folder_list->drag_button_press_signal_id =
    g_signal_connect(folder_list->treeview,
        "button-press-event", G_CALLBACK(drag_button_press_handler),
        (gpointer) folder_list);
  
  folder_list->drag_mouse_motion_signal_id =
    g_signal_connect(folder_list->treeview,
        "motion-notify-event", G_CALLBACK(drag_mouse_motion_handler),
        (gpointer) folder_list);
  
  folder_list->drag_button_release_signal_id =
    g_signal_connect(folder_list->treeview,
        "button-release-event", G_CALLBACK(drag_button_release_handler),
        (gpointer) folder_list);
  
  folder_list->drag_begin_signal_id =
    g_signal_connect(folder_list->treeview,
        "drag-begin", G_CALLBACK(drag_begin_handler),
        (gpointer) folder_list);
  
  folder_list->drag_end_signal_id =
    g_signal_connect(folder_list->treeview,
        "drag-end", G_CALLBACK(drag_end_handler),
        (gpointer) folder_list);
}

static void drag_drop_unsetup(struct etpan_folder_list * folder_list)
{
  /* drag */
  g_signal_handler_disconnect(folder_list->treeview,
      folder_list->drag_end_signal_id);
  g_signal_handler_disconnect(folder_list->treeview,
      folder_list->drag_begin_signal_id);
  
  g_signal_handler_disconnect(folder_list->treeview,
      folder_list->drag_button_press_signal_id);
  g_signal_handler_disconnect(folder_list->treeview,
      folder_list->drag_button_release_signal_id);
  g_signal_handler_disconnect(folder_list->treeview,
      folder_list->drag_mouse_motion_signal_id);
  
  /* drop */
  if (folder_list->drag_entered) {
    g_source_remove(folder_list->drag_scroll_timeout_id);
    g_source_remove(folder_list->drag_auto_expand_timeout_id);
  }
  
  g_signal_handler_disconnect(folder_list->scrolledwindow,
      folder_list->drag_drop_signal_id);
  g_signal_handler_disconnect(folder_list->scrolledwindow,
      folder_list->drag_leave_signal_id);
  g_signal_handler_disconnect(folder_list->scrolledwindow,
      folder_list->drag_motion_signal_id);
  
  gtk_drag_dest_unset(folder_list->treeview);
}



/* drag */

static void drag_may_start(struct etpan_folder_list * folder_list,
    gdouble x, gdouble y, guint32 time)
{
  (void) time;
  
  folder_list->drag_may_start = 1;
  folder_list->drag_may_start_x = x;
  folder_list->drag_may_start_y = y;
  folder_list->drag_expanded = 0;
}

static void drag_cancel_may_start(struct etpan_folder_list * folder_list)
{
  folder_list->drag_may_start = 0;
}

static int drag_may_be_starting(struct etpan_folder_list * folder_list)
{
  return folder_list->drag_may_start;
}

static int drag_is_valid(struct etpan_folder_list * folder_list,
    gdouble x, gdouble y, guint32 time)
{
  (void) time;
  
  return gtk_drag_check_threshold(folder_list->treeview,
      folder_list->drag_may_start_x,
      folder_list->drag_may_start_y,
      x, y);
}

static void drag_start(struct etpan_folder_list * folder_list,
    GdkEvent * event)
{
  GtkTargetList * targets;
  GdkAtom atom;
  
  folder_list->drag_may_start = 0;
  folder_list->drag_started = 1;
  
  targets = gtk_target_list_new(NULL, 0);
  atom = gdk_atom_intern(ETPAN_FOLDER_LIST_DRAG_FOLDER_NAME, 0);
  gtk_target_list_add(targets, atom, GTK_TARGET_SAME_APP, 0);
  folder_list->drag_context = gtk_drag_begin(folder_list->treeview,
      targets,
      GDK_ACTION_MOVE | GDK_ACTION_COPY,
      1, event);
  
  gtk_target_list_unref(targets);
}

static void drag_end(struct etpan_folder_list * folder_list)
{
  folder_list->drag_context = NULL;
  folder_list->drag_started = 0;
}

static void drag_set_icon(struct etpan_folder_list * folder_list)
{
  GdkPixbuf * pixbuf;
  
  /* must be called on drag-begin signal */
  pixbuf = etpan_icon_manager_get_scaled_pixbuf(etpan_icon_manager_get_default(),
      "drag-folder", 32);
  gtk_drag_set_icon_pixbuf(folder_list->drag_context,
      pixbuf, 0, 0);
}

static void copy_folder_finished_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);

static void copy_folder_setup(struct etpan_folder_list * folder_list)
{
  ETPAN_SIGNAL_ADD_HANDLER(etpan_mail_manager_get_default(),
      ETPAN_MAIL_MANAGER_COPY_FOLDER_FINISHED_SIGNAL,
      copy_folder_finished_handler, folder_list);
}

static void copy_folder_unsetup(struct etpan_folder_list * folder_list)
{
  ETPAN_SIGNAL_REMOVE_HANDLER(etpan_mail_manager_get_default(),
      ETPAN_MAIL_MANAGER_COPY_FOLDER_FINISHED_SIGNAL,
      copy_folder_finished_handler, folder_list);
}

void etpan_folder_list_copy_folder(struct etpan_folder_list * folder_list,
    struct etpan_folder * source_folder,
    struct etpan_storage * dest_storage,
    struct etpan_folder * dest_folder,
    int delete)
{
  unsigned int copy_id;
  chashdatum key;
  chashdatum value;
  int r;
  
  if (!etpan_mail_manager_operation_is_allowed(etpan_mail_manager_get_default(),
          source_folder)) {
    ETPAN_WARN_LOG("could not copy %s because an operation is in progress",
        etpan_folder_get_ui_path(source_folder));
    return;
  }
  
  if (dest_folder == NULL)
    start_animation_storage(folder_list, dest_storage);
  else
    start_animation_folder(folder_list, dest_folder);
  
  if (delete)
    ETPAN_LOG("move folder %s", etpan_folder_get_ui_path(source_folder));
  else 
    ETPAN_LOG("copy folder %s", etpan_folder_get_ui_path(source_folder));
  
  copy_id = etpan_mail_manager_get_next_copy_folder_id(etpan_mail_manager_get_default());
  
  if (dest_folder != NULL) {
    key.data = &copy_id;
    key.len = sizeof(copy_id);
    value.data = dest_folder;
    value.len = 0;
    r = chash_set(folder_list->copy_folder_hash, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  else {
    key.data = &copy_id;
    key.len = sizeof(copy_id);
    value.data = dest_storage;
    value.len = 0;
    r = chash_set(folder_list->copy_folder_storage_hash, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  if (delete)
    etpan_mail_manager_move_folder(etpan_mail_manager_get_default(),
        dest_storage, dest_folder, source_folder);
  else
    etpan_mail_manager_copy_folder(etpan_mail_manager_get_default(),
        dest_storage, dest_folder, source_folder);
}

static void copy_folder_finished_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  chashdatum key;
  chashdatum value;
  unsigned int * copy_id;
  struct etpan_folder_list * folder_list;
  int r;
  
  folder_list = user_data;
  copy_id = signal_data;
  key.data = copy_id;
  key.len = sizeof(* copy_id);
  r = chash_get(folder_list->copy_folder_hash, &key, &value);
  if (r == 0) {
    struct etpan_folder * folder;
    
    folder = value.data;
    chash_delete(folder_list->copy_folder_hash, &key, NULL);
    
    stop_animation_folder(folder_list, folder);
  }
  
  r = chash_get(folder_list->copy_folder_storage_hash, &key, &value);
  if (r == 0) {
    struct etpan_storage * storage;
    
    storage = value.data;
    chash_delete(folder_list->copy_folder_storage_hash, &key, NULL);
    
    stop_animation_storage(folder_list, storage);
  }
}

static void popup_menu_setup(struct etpan_folder_list * folder_list)
{
  struct etpan_contextual_menu * menu;
  
  menu = etpan_contextual_menu_new();
  etpan_contextual_menu_add_item(menu, _("New mailbox..."), NULL);
  etpan_contextual_menu_add_item(menu, _("Rename..."), NULL);
  etpan_contextual_menu_add_item(menu, _("Delete"), NULL);
#if 0
  etpan_contextual_menu_add_item(menu, _("Refresh"), NULL);
#endif
  
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_CONTEXTUAL_MENU_CLICKED_SIGNAL, menu,
      folder_list,
      folder_popup_menu_item_clicked);
  
  folder_list->popup_menu = menu;
}

static void popup_menu_unsetup(struct etpan_folder_list * folder_list)
{
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_CONTEXTUAL_MENU_CLICKED_SIGNAL, folder_list->popup_menu,
      folder_list,
      folder_popup_menu_item_clicked);
  
  etpan_contextual_menu_free(folder_list->popup_menu);
}

static void show_folder_popup_menu(struct etpan_folder_list * folder_list,
    GtkWidget * widget, GdkEventButton * event)
{
  GtkWidget * menu;
  int button, event_time;
  (void) widget;
  
  if (event) {
    button = event->button;
    event_time = event->time;
  }
  else {
    button = 0;
    event_time = gtk_get_current_event_time();
  }
  
  menu = etpan_contextual_menu_get_main_widget(folder_list->popup_menu);
  
  gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL, 
      button, event_time);
}


static void folder_popup_menu_item_clicked(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_contextual_menu * menu;
  struct etpan_folder_list * folder_list;
  int * pindex;
  struct etpan_folder * cur_folder;
  struct folder_tree * cur_tree;
  struct etpan_account * cur_account;
  struct etpan_storage * cur_storage;
  struct etpan_folder * parent_folder;
  struct folder_tree * parent_tree;
  (void) signal_name;
  
  folder_list = user_data;
  menu = sender;
  pindex = signal_data;
  
  ETPAN_MSGLIST_LOG("index : %i", * pindex);
  
  cur_tree = get_selection_data(folder_list,
      folder_list->current_selection);
  cur_folder = folder_tree_get_folder(cur_tree);
  cur_account = folder_tree_get_account(cur_tree);
  if (cur_tree == NULL)
    return;
  cur_storage = etpan_account_get_storage(cur_account);
  if (cur_storage == NULL)
    return;
  
  switch (* pindex) {
  case 0:
    /* new */
    etpan_folder_list_new_folder(folder_list, cur_storage, cur_folder);
    break;
  case 1:
    if (cur_folder == NULL)
      return;
    /* rename */
    parent_tree = folder_tree_get_parent(cur_tree);
    parent_folder = folder_tree_get_folder(parent_tree);
    etpan_folder_list_rename_mailbox(folder_list, cur_folder, parent_folder);
    break;
  case 2:
    if (cur_folder == NULL)
      return;
    
    /* delete */
    etpan_folder_list_delete_mailbox(folder_list, cur_folder);
    break;
#if 0
  case 3:
    etpan_mail_manager_storage_update(etpan_mail_manager_get_default(),
        cur_storage);
    break;
#endif
  }
}

/* create folder */

static void create_input_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);
static void created_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);

void etpan_folder_list_new_folder(struct etpan_folder_list * folder_list,
    struct etpan_storage * storage,
    struct etpan_folder * folder)
{
  struct etpan_input_dialog * dialog;
  
  folder_list->create_parent_folder = folder;
  if (folder != NULL)
    etpan_folder_ref(folder);
  folder_list->create_parent_storage = storage;
  
  dialog = etpan_input_dialog_new();
  etpan_input_dialog_set_title(dialog, "Create mailbox");
  etpan_input_dialog_set_label(dialog, "Enter name of the mailbox to create");
  etpan_input_dialog_set_text(dialog, "New folder");
  etpan_input_dialog_setup(dialog);
  
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_INPUT_DIALOG_CLOSED_SIGNAL, dialog, folder_list,
      create_input_handler);
  
  etpan_input_dialog_show(dialog);
}

static void create_input_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_folder_list * folder_list;
  struct etpan_input_dialog * dialog;
  char * name;
  struct etpan_folder_create * folder_create;
  int r;
  struct etpan_storage * storage;
  struct etpan_folder * parent_folder;
  (void) signal_name;
  (void) signal_data;
  
  dialog = sender;
  folder_list = user_data;
  
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_INPUT_DIALOG_CLOSED_SIGNAL, dialog, folder_list,
      create_input_handler);
  
  etpan_input_dialog_hide(dialog);
  r = etpan_input_dialog_get_result(dialog);
  if (r != ETPAN_INPUT_DIALOG_DONE) {
    etpan_input_dialog_unsetup(dialog);
    etpan_input_dialog_free(dialog);
    if (folder_list->create_parent_folder != NULL)
      etpan_folder_unref(folder_list->create_parent_folder);
    folder_list->create_parent_folder = NULL;
    folder_list->create_parent_storage = NULL;
    return;
  }
  
  storage = folder_list->create_parent_storage;
  parent_folder = folder_list->create_parent_folder;
  name = etpan_input_dialog_get_text(dialog);
  
  folder_create = etpan_folder_create_new();
  etpan_folder_create_set_folder_name(folder_create, name);
  etpan_folder_create_set_parent_folder(folder_create, parent_folder);
  etpan_folder_create_set_storage(folder_create, storage);
  etpan_folder_create_setup(folder_create);
  
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_FOLDER_CREATE_FINISHED_SIGNAL, folder_create, folder_list,
      created_handler);
  
  etpan_folder_create_run(folder_create);
  
  /* destroy dialog */
  etpan_input_dialog_unsetup(dialog);
  etpan_input_dialog_free(dialog);
}

static void created_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_folder_list * folder_list;
  struct etpan_folder_create * folder_create;
  struct etpan_folder * created_folder;
  struct etpan_error * error;
  (void) signal_name;
  (void) signal_data;
  
  folder_create = sender;
  folder_list = user_data;
  
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_FOLDER_CREATE_FINISHED_SIGNAL, folder_create, folder_list,
      created_handler);
  
  error = etpan_folder_create_get_error(folder_create);
  if (error != NULL) {
    etpan_error_log(error);
  }
  
  if (folder_list->create_parent_folder != NULL)
    etpan_folder_unref(folder_list->create_parent_folder);
  folder_list->create_parent_folder = NULL;
  folder_list->create_parent_storage = NULL;
  
  created_folder = etpan_folder_create_get_created_folder(folder_create);
  
  /* XXX - select folder ? */
  
  etpan_folder_create_unref(folder_create); 
}

/* delete folder */

static void delete_finished_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);

void etpan_folder_list_delete_mailbox(struct etpan_folder_list * folder_list,
    struct etpan_folder * folder)
{
  struct etpan_folder_delete * folder_delete;

  if (!etpan_mail_manager_operation_is_allowed(etpan_mail_manager_get_default(),
          folder)) {
    ETPAN_WARN_LOG("could not delete %s because an operation is in progress",
        etpan_folder_get_ui_path(folder));
    return;
  }
  
  start_animation_folder(folder_list, folder);
  
  folder_delete = etpan_folder_delete_new();
  
  etpan_folder_delete_set_folder(folder_delete, folder);
  etpan_folder_delete_setup(folder_delete);
  
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_FOLDER_DELETE_FINISHED_SIGNAL, folder_delete, folder_list,
      delete_finished_handler);
  
  etpan_folder_delete_run(folder_delete);
}

static void delete_finished_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_folder_delete * folder_delete;
  struct etpan_folder_list * folder_list;
  struct etpan_folder * folder;
  struct etpan_error * error;
  (void) signal_name;
  (void) signal_data;
  
  folder_delete = sender;
  folder_list = user_data;

  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_FOLDER_DELETE_FINISHED_SIGNAL, folder_delete, folder_list,
      delete_finished_handler);
  
  error = etpan_folder_delete_get_error(folder_delete);
  if (error != NULL) {
    etpan_error_log(error);
  }
  
  folder = etpan_folder_delete_get_folder(folder_delete);
  stop_animation_folder(folder_list, folder);
  etpan_folder_delete_unref(folder_delete);
}

static struct folder_tree * find_folder(struct folder_tree * root,
    struct etpan_folder * folder)
{
  carray * children;
  unsigned int i;
  struct etpan_folder * cur_folder;
  
  cur_folder = folder_tree_get_folder(root);
  if (cur_folder == folder)
    return root;
  
  children = folder_tree_get_children(root);
  for(i = 0 ; i < carray_count(children) ; i ++) {
    struct folder_tree * found;
    
    found = find_folder(carray_get(children, i), folder);
    if (found != NULL)
      return found;
  }
  
  return NULL;
}

/* rename folder */

static void rename_input_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);

void etpan_folder_list_rename_mailbox(struct etpan_folder_list * folder_list,
    struct etpan_folder * folder, struct etpan_folder * parent_folder)
{
  struct etpan_input_dialog * dialog;
  char * name;
  
  folder_list->rename_folder = folder;
  etpan_folder_ref(folder);
  folder_list->rename_parent_folder = parent_folder;
  if (parent_folder != NULL)
    etpan_folder_ref(parent_folder);
  
  dialog = etpan_input_dialog_new();
  etpan_input_dialog_set_title(dialog, "Change the name of a mailbox");
  etpan_input_dialog_set_label(dialog, "Enter name of the new name of the mailbox");
  name = etpan_folder_get_ui_name(folder);
  etpan_input_dialog_set_text(dialog, name);
  etpan_input_dialog_setup(dialog);
  
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_INPUT_DIALOG_CLOSED_SIGNAL, dialog, folder_list,
      rename_input_handler);
  
  etpan_input_dialog_show(dialog);
}

static void renamed_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);

static void rename_input_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_folder_list * folder_list;
  struct etpan_input_dialog * dialog;
  struct etpan_folder_rename * folder_rename;
  int r;
  char * new_name;
  char * new_location;
  struct etpan_storage * storage;
  (void) signal_name;
  (void) signal_data;
  
  dialog = sender;
  folder_list = user_data;
  
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_INPUT_DIALOG_CLOSED_SIGNAL, dialog, folder_list,
      rename_input_handler);
  
  etpan_input_dialog_hide(dialog);
  r = etpan_input_dialog_get_result(dialog);
  if (r != ETPAN_INPUT_DIALOG_DONE) {
    etpan_input_dialog_unsetup(dialog);
    etpan_input_dialog_free(dialog);
    etpan_folder_unref(folder_list->rename_folder);
    if (folder_list->rename_parent_folder != NULL)
      etpan_folder_unref(folder_list->rename_parent_folder);
    folder_list->rename_folder = NULL;
    folder_list->rename_parent_folder = NULL;
    return;
  }
  
  storage = etpan_folder_get_storage(folder_list->rename_folder);
  new_name = etpan_input_dialog_get_text(dialog);
  new_location = etpan_storage_get_sub_folder_location(storage,
    folder_list->rename_parent_folder, new_name);
  
  folder_rename = etpan_folder_rename_new();
  etpan_folder_rename_set_new_name(folder_rename, new_location);
  etpan_folder_rename_set_folder(folder_rename, folder_list->rename_folder);
  etpan_folder_rename_setup(folder_rename);
  
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_FOLDER_RENAME_FINISHED_SIGNAL, folder_rename, folder_list,
      renamed_handler);
  
  etpan_folder_rename_run(folder_rename);
  
  free(new_location);
  
  /* destroy dialog */
  etpan_input_dialog_unsetup(dialog);
  etpan_input_dialog_free(dialog);
}

static void renamed_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_folder_list * folder_list;
  struct etpan_folder_rename * folder_rename;
  struct etpan_folder * renamed_folder;
  struct etpan_error * error;
  (void) signal_name;
  (void) signal_data;
  
  folder_rename = sender;
  folder_list = user_data;
  
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_FOLDER_RENAME_FINISHED_SIGNAL, folder_rename, folder_list,
      renamed_handler);
  
  error = etpan_folder_rename_get_error(folder_rename);
  if (error != NULL) {
    etpan_error_log(error);
  }
  
  if (folder_list->rename_parent_folder != NULL)
    etpan_folder_unref(folder_list->rename_parent_folder);
  etpan_folder_unref(folder_list->rename_folder);
  folder_list->rename_parent_folder = NULL;
  folder_list->rename_folder = NULL;
  
  renamed_folder = etpan_folder_rename_get_renamed_folder(folder_rename);
  
  /* select folder ? */
  
  etpan_folder_rename_unref(folder_rename);
}

/* TODO : performance : implement with hash */
static struct folder_tree *
get_tree_by_ui_path_relative(struct folder_tree * tree,
    char * path)
{
  char * current_path;
  int traverse;
  
  current_path = folder_tree_get_ui_path(tree);
  if (current_path != NULL) {
    traverse = 0;
    if (strcmp(path, current_path) == 0)
      return tree;
    
    if (strncmp(path, current_path, strlen(current_path)) == 0)
      traverse = 1;
  }
  else {
    traverse = 1;
  }
  
  if (traverse) {
    unsigned int i;
    
    for(i = 0 ; i < carray_count(tree->children) ; i ++) {
      struct folder_tree * child;
      struct folder_tree * found;
    
      child = carray_get(tree->children, i);
      found = get_tree_by_ui_path_relative(child, path);
      if (found != NULL)
        return found;
    }
  }
  
  return NULL;
}

static struct folder_tree *
get_tree_by_ui_path(struct etpan_folder_list * folder_list, char * path)
{
  return get_tree_by_ui_path_relative(folder_list->treeinfo, path);
}


static void copy_msg_finished_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);

static void copy_msg_setup(struct etpan_folder_list * folder_list)
{
  ETPAN_SIGNAL_ADD_HANDLER(etpan_mail_manager_get_default(),
      ETPAN_MAIL_MANAGER_COPY_MSG_FINISHED_SIGNAL,
      copy_msg_finished_handler, folder_list);
}

static void copy_msg_unsetup(struct etpan_folder_list * folder_list)
{
  ETPAN_SIGNAL_REMOVE_HANDLER(etpan_mail_manager_get_default(),
      ETPAN_MAIL_MANAGER_COPY_MSG_FINISHED_SIGNAL,
      copy_msg_finished_handler, folder_list);
}

void etpan_folder_copy_messages_with_delete(struct etpan_folder_list * folder_list,
    struct etpan_folder * folder, chash * msg_hash, int delete)
{
  unsigned int copy_id;
  chashdatum key;
  chashdatum value;
  int r;
  
  start_animation_folder(folder_list, folder);
  
  if (delete)
    ETPAN_LOG("move %i messages", chash_count(msg_hash));
  else 
    ETPAN_LOG("copy %i messages", chash_count(msg_hash));
  
  copy_id = etpan_mail_manager_get_next_copy_messages_id(etpan_mail_manager_get_default());
  
  key.data = &copy_id;
  key.len = sizeof(copy_id);
  value.data = folder;
  value.len = 0;
  r = chash_set(folder_list->copy_msg_hash, &key, &value, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  if (delete)
    etpan_mail_manager_move_messages(etpan_mail_manager_get_default(),
         folder, msg_hash);
  else
    etpan_mail_manager_copy_messages(etpan_mail_manager_get_default(),
        folder, msg_hash);
}

static void copy_msg_finished_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  chashdatum key;
  chashdatum value;
  unsigned int * copy_id;
  struct etpan_folder * folder;
  struct etpan_folder_list * folder_list;
  int r;
  
  folder_list = user_data;
  copy_id = signal_data;
  key.data = copy_id;
  key.len = sizeof(* copy_id);
  r = chash_get(folder_list->copy_msg_hash, &key, &value);
  if (r < 0) {
    return;
  }
  
  folder = value.data;
  chash_delete(folder_list->copy_msg_hash, &key, NULL);
  
  stop_animation_folder(folder_list, folder);
}
