#include "etpan-gtk-ui-main.h"

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "etpan-backend.h"
#include "etpan-icon-manager.h"
#include "etpan-main-window.h"
#include "etpan-ui-config.h"
#include "etpan-preferences-window.h"
#include "etpan-tabbed-message-list.h"

static GIOChannel * gtk_ui_io_channel = NULL;
static gint gtk_ui_delete_signal_id = 0;
static gint gtk_ui_key_pressed_signal_id = 0;
static GtkWidget * gtk_ui_quit_panel = NULL;

static gboolean main_window_close_handler(GtkWidget * widget,
    GdkEvent * event,
    gpointer user_data);
static gboolean key_handler(GtkWidget * widget,
    GdkEventKey * event,
    gpointer user_data);

static void log_handler(const gchar * log_domain,
    GLogLevelFlags log_level,
    const gchar * message,
    gpointer user_data)
{
  (void) log_level;
  (void) user_data;
  
  ETPAN_LOG("%s - %s", log_domain, message);
  etpan_log_stack();
}

static void schedule_mail_manager(void);

static gboolean thread_manager_event(GIOChannel * source,
    GIOCondition condition,
    gpointer data)
{
  struct etpan_thread_manager_app * thread_manager;
  struct etpan_main_window * main_window;
  (void) source;
  (void) condition;
  (void) data;
  
  thread_manager = etpan_thread_manager_app_get_default();
  main_window = etpan_main_window_get_default();
  
  etpan_thread_manager_app_loop(thread_manager); 
  
  return TRUE;
}

static int mail_manager_timer_set = 0;
static gulong mail_manager_timer_id;

static void schedule_mail_manager_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);
static void cancel_scheduled_mail_manager(void);

static void mail_manager_setup(void)
{
  schedule_mail_manager();
  ETPAN_SIGNAL_ADD_HANDLER(etpan_mail_manager_get_default(),
      ETPAN_MAIL_MANAGER_SCHEDULE_SIGNAL, schedule_mail_manager_handler, NULL);
}

static void mail_manager_unsetup(void)
{
  ETPAN_SIGNAL_REMOVE_HANDLER(etpan_mail_manager_get_default(),
      ETPAN_MAIL_MANAGER_SCHEDULE_SIGNAL, schedule_mail_manager_handler, NULL);
  cancel_scheduled_mail_manager();
}

static void schedule_mail_manager_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  schedule_mail_manager();
}

static gboolean mail_manager_timer_handler(gpointer data)
{
#if 0
  ETPAN_LOG("perform next task");
#endif
  mail_manager_timer_set = 0;
  etpan_mail_manager_perform_next_task(etpan_mail_manager_get_default());
  schedule_mail_manager();
  
  return FALSE;
}

static void cancel_scheduled_mail_manager(void)
{
  if (mail_manager_timer_set) {
    g_source_remove(mail_manager_timer_id);
    mail_manager_timer_set = 0;
  }
}

static void schedule_mail_manager(void)
{
  double date;
  unsigned int msec_date;
  double current;
  double delta;
  
  cancel_scheduled_mail_manager();
  
  date = etpan_mail_manager_next_task_date(etpan_mail_manager_get_default());
  
  current = etpan_get_time();
  delta = date - current;
#if 0
  ETPAN_LOG("mail manager : %g", delta);
#endif
  if (delta < 0)
    delta = 0;
  
  msec_date = (unsigned int) (delta * 1000);
  mail_manager_timer_set = 1;
  mail_manager_timer_id = g_timeout_add(msec_date,
      mail_manager_timer_handler, NULL);
}

struct etpan_error * etpan_gtk_ui_init(int * argc, char *** argv)
{
  gboolean result;
  char * datadir;
  struct etpan_preferences_window * preferences;
  struct etpan_main_window * main_window;
  struct etpan_ui_config * ui_config;
  struct etpan_icon_manager * icon_manager;
  struct etpan_error * error;
  
  g_log_set_handler ("Gtk", G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL
      | G_LOG_FLAG_RECURSION, log_handler, NULL);
  g_log_set_handler ("GLib", G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL
      | G_LOG_FLAG_RECURSION, log_handler, NULL);
  g_log_set_handler ("GLib-GObject", G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL
      | G_LOG_FLAG_RECURSION, log_handler, NULL);
  g_log_set_handler(NULL, G_LOG_LEVEL_WARNING | G_LOG_FLAG_FATAL
      | G_LOG_FLAG_RECURSION, log_handler, NULL);
  
  g_thread_init(NULL);
  gdk_threads_init();
  
#ifdef ENABLE_NLS
  bindtextdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
  bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
  textdomain (GETTEXT_PACKAGE);
#endif
  
  gtk_set_locale();

  result = gtk_init_check(argc, argv);
  if (!result) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_DISPLAY);
    etpan_error_set_short_description(error, _("Could connect to display"));
    etpan_error_strf_long_description(error, _("The application could not connect to the display. Check the DISPLAY environment variable. It should be set to a correct value. Or maybe you don't have the authorization on the display."));
    
    return error;
  }

  datadir = getenv("ETPAN_DATADIR");
  if (datadir == NULL) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_INVAL);
    etpan_error_set_short_description(error, _("Application resource directory not found"));
    etpan_error_strf_long_description(error, _("Application resource directory must be defined. It should be defined in environment variable ETPAN_DATADIR."));
    
    goto err;
  }
  
  /* icon manager */
  icon_manager = etpan_icon_manager_new();
  etpan_icon_manager_set_path(icon_manager, datadir);
  etpan_icon_manager_set_default(icon_manager);
  
  /* ui config */
  ui_config = etpan_ui_config_new();
  etpan_ui_config_set_default(ui_config);
  
  /* main window */
  main_window = etpan_main_window_new();
  etpan_main_window_set_default(main_window);
  
  /* preferences window */
  preferences = etpan_preferences_window_new();
  etpan_preferences_window_set_default(preferences);
  
  return NULL;
  
 err:
  return error;
}

void etpan_gtk_ui_done(void)
{
  struct etpan_preferences_window * preferences;
  struct etpan_main_window * main_window;
  struct etpan_ui_config * ui_config;
  struct etpan_icon_manager * icon_manager;
  
  preferences = etpan_preferences_window_get_default();
  etpan_preferences_window_free(preferences);
  etpan_preferences_window_set_default(NULL);
  main_window = etpan_main_window_get_default();
  etpan_main_window_free(main_window);
  etpan_main_window_set_default(NULL);
  ui_config = etpan_ui_config_get_default();
  etpan_ui_config_free(ui_config);
  etpan_ui_config_set_default(NULL);
  icon_manager = etpan_icon_manager_get_default();
  etpan_icon_manager_free(icon_manager);
  etpan_icon_manager_set_default(NULL);
}

  
void etpan_gtk_ui_setup(void)
{
  struct etpan_main_window * main_window;
  GtkWidget * window;
  int fd_thread_manager;
  GIOChannel * io_channel;
  struct etpan_preferences_window * preferences;
  struct etpan_thread_manager_app * thread_manager;
  
  /* config read */
  etpan_ui_config_read_default();
  
  /* plug thread manager in gtk */
  thread_manager = etpan_thread_manager_app_get_default();
  fd_thread_manager = etpan_thread_manager_app_get_fd(thread_manager);
  io_channel = g_io_channel_unix_new(fd_thread_manager);
  gtk_ui_io_channel = io_channel;
  g_io_add_watch_full(io_channel, 0, G_IO_IN, thread_manager_event,
      NULL, NULL);
  
  /* main window setup */
  main_window = etpan_main_window_get_default();
  etpan_main_window_setup(main_window);
  window = etpan_main_window_get_main_widget(main_window);
  gtk_ui_key_pressed_signal_id = g_signal_connect(window, "key-press-event",
      G_CALLBACK(key_handler), NULL);
  gtk_ui_delete_signal_id = g_signal_connect(window, "delete-event",
      G_CALLBACK(main_window_close_handler), NULL);
  gtk_widget_show(window);
  
  /* preferences */
  preferences = etpan_preferences_window_get_default();
  etpan_preferences_window_setup(preferences);
  
  /* mail manager */
  mail_manager_setup();
}

void etpan_gtk_ui_run(void)
{
  gtk_main();
}

static gboolean main_window_close_handler(GtkWidget * widget,
    GdkEvent * event,
    gpointer user_data)
{
  (void) widget;
  (void) event;
  (void) user_data;
  
  etpan_signal_send(etpan_signal_manager_get_default(),
    ETPAN_GTK_UI_MAIN_CLOSE_SIGNAL, NULL, NULL);
  
  return TRUE;
}

static gboolean key_handler(GtkWidget * widget,
    GdkEventKey * event,
    gpointer user_data)
{
  unsigned int state;
  int alt;
  int ctrl;
  (void) widget;
  (void) user_data;
  
  state = event->state & (GDK_SHIFT_MASK | GDK_CONTROL_MASK | GDK_MOD1_MASK |
      GDK_MOD2_MASK | GDK_MOD3_MASK | GDK_MOD4_MASK |
      GDK_MOD5_MASK);
  
  alt = (state & GDK_MOD1_MASK) != 0;
  ctrl = (state & GDK_CONTROL_MASK) != 0;
  
#if 0
  ETPAN_LOG("key handler %x alt: %i ctrl: %i %i",
      event->keyval, alt, ctrl, state);
#endif
  
  if ((event->keyval == GDK_q) && (alt || ctrl)) {
    etpan_signal_send(etpan_signal_manager_get_default(),
        ETPAN_GTK_UI_MAIN_CLOSE_SIGNAL, NULL, NULL);
    
    return TRUE;
  }
  else {
    return FALSE;
  }
}

void etpan_gtk_ui_show_quit_panel(void)
{
  GtkWidget * window;
  struct etpan_main_window * main_window;
  GtkWidget * label;
  
  main_window = etpan_main_window_get_default();
  window = etpan_main_window_get_main_widget(main_window);

#if 0  
  gtk_widget_set_sensitive(window, FALSE);
#endif
  gtk_widget_hide(window);
  
  /* display quit window */
  window = gtk_window_new(GTK_WINDOW_POPUP);
  gtk_window_set_default_size(GTK_WINDOW(window), 400, 100);
  label = gtk_label_new(_("disconnecting from account"));
  gtk_widget_show(label);
  gtk_container_add(GTK_CONTAINER(window), label);
  gtk_window_set_position(GTK_WINDOW(window), GTK_WIN_POS_CENTER_ALWAYS);
  gtk_widget_show(window);
}

void etpan_gtk_ui_hide_quit_panel(void)
{
  GtkWidget * window;
  
  window = gtk_ui_quit_panel;
  gtk_widget_hide(window);
  gtk_widget_destroy(window);
  gtk_ui_quit_panel = NULL;
}

void etpan_gtk_ui_quit(void)
{
  gtk_main_quit();
}

static void preferences_close(void)
{
  struct etpan_preferences_window * prefs;
  GtkWidget * window;
  
  prefs = etpan_preferences_window_get_default();
  window = etpan_preferences_panel_get_main_widget(prefs);
  gtk_widget_hide(window);
}

void etpan_gtk_ui_stop(void)
{
  struct etpan_tabbed_message_list * tabbed;
  struct etpan_main_window * main_window;
  
  preferences_close();
  main_window = etpan_main_window_get_default();
  tabbed = etpan_main_window_get_tabbed_message_list(main_window);
  etpan_tabbed_message_list_close_all(tabbed);
}

void etpan_gtk_ui_unsetup(void)
{
  struct etpan_preferences_window * preferences;
  struct etpan_tabbed_message_list * tabbed;
  struct etpan_main_window * main_window;
  GtkWidget * window;
  
  mail_manager_unsetup();
  
  preferences = etpan_preferences_window_get_default();
  etpan_preferences_window_unsetup(preferences);
  
#if 0
  main_window = etpan_main_window_get_default();
  tabbed = etpan_main_window_get_tabbed_message_list(main_window);
  etpan_tabbed_message_list_close_all(tabbed);
#endif
  
  main_window = etpan_main_window_get_default();
  window = etpan_main_window_get_main_widget(main_window);
  g_signal_handler_disconnect(window, gtk_ui_delete_signal_id);
  g_signal_handler_disconnect(window, gtk_ui_key_pressed_signal_id);
  
  etpan_main_window_unsetup(main_window);
  
  g_io_channel_unref(gtk_ui_io_channel);
  gtk_ui_io_channel = NULL;
  
  etpan_ui_config_save_default();
}
