#include "etpan-icon-manager.h"

#include "etpan-backend.h"
#include "etpan-error.h"
#include "etpan-log.h"

static struct etpan_icon_manager * defaut_manager = NULL;

struct etpan_icon_manager * etpan_icon_manager_new(void)
{
  struct etpan_icon_manager * manager;
  
  manager = malloc(sizeof(* manager));
  if (manager == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  manager->path = NULL;
  
  manager->pixbuf_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (manager->pixbuf_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return manager;
}

void etpan_icon_manager_free(struct etpan_icon_manager * manager)
{
  chashiter * iter;
  
  for(iter = chash_begin(manager->pixbuf_hash) ; iter != NULL ;
      iter = chash_next(manager->pixbuf_hash, iter)) {
    chashdatum value;
    GdkPixbuf * pixbuf;
    
    chash_value(iter, &value);
    pixbuf = value.data;
    g_object_unref(pixbuf);
  }

  chash_free(manager->pixbuf_hash);
  free(manager->path);
  free(manager);
}

void etpan_icon_manager_set_default(struct etpan_icon_manager * manager)
{
  defaut_manager = manager;
}

struct etpan_icon_manager * etpan_icon_manager_get_default(void)
{
  return defaut_manager;
}

void etpan_icon_manager_set_path(struct etpan_icon_manager * manager,
    char * path)
{
  if (path != manager->path) {
    free(manager->path);
    if (path != NULL) {
      manager->path = strdup(path);
      if (manager->path == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      manager->path = NULL;
  }
}

static void cache_image(struct etpan_icon_manager * manager, char * name)
{
  GdkPixbuf * pixbuf;
  char filename[PATH_MAX];
  chashdatum key;
  chashdatum value;
  int r;
  
  key.data = name;
  key.len = strlen(name);
  r = chash_get(manager->pixbuf_hash, &key, &value);
  if (r == 0)
    return;
  
  if (manager->path == NULL)
    snprintf(filename, sizeof(filename), "%s.png", name);
  else
    snprintf(filename, sizeof(filename), "%s/%s.png", manager->path, name);
  
  pixbuf = gdk_pixbuf_new_from_file(filename, NULL);
  if (pixbuf == NULL) {
    ETPAN_LOG("either icon %s was not found, either, we ran out of memory",
        filename);
    etpan_crash();
  }
  
  value.data = pixbuf;
  value.len = 0;
  
  r = chash_set(manager->pixbuf_hash, &key, &value, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

GtkWidget * etpan_icon_manager_new_image(struct etpan_icon_manager * manager,
    char * name)
{
  GdkPixbuf * pixbuf;
  GtkWidget * image;
  
  pixbuf = etpan_icon_manager_get_pixbuf(manager, name);
  if (pixbuf == NULL)
    return NULL;
  
  image = gtk_image_new_from_pixbuf(pixbuf);
  
  return image;
}

GtkWidget * etpan_icon_manager_new_scaled_image(struct etpan_icon_manager * manager,
    char * name, int size)
{
  GtkWidget * image;
  GdkPixbuf * pixbuf;
  
  pixbuf = etpan_icon_manager_get_scaled_pixbuf(manager, name, size);
  if (pixbuf == NULL)
    return NULL;
  
  image = gtk_image_new_from_pixbuf(pixbuf);
  
  return image;
}

GdkPixbuf * etpan_icon_manager_get_pixbuf(struct etpan_icon_manager * manager,
    char * name)
{
  chashdatum key;
  chashdatum value;
  int r;
  
  cache_image(manager, name);
  key.data = name;
  key.len = strlen(name);
  
  r = chash_get(manager->pixbuf_hash, &key, &value);
  if (r < 0) {
    ETPAN_LOG("should have cached %s", name);
    etpan_crash();
  }
  
  return value.data;
}

GdkPixbuf * etpan_icon_manager_get_scaled_pixbuf(struct etpan_icon_manager * manager,
    char * name, int size)
{
  GdkPixbuf * scaled;
  GdkPixbuf * pixbuf;
  char identifier[PATH_MAX];
  chashdatum key;
  chashdatum value;
  int r;
  
  snprintf(identifier, sizeof(identifier), "%s/%i", name, size);
  key.data = identifier;
  key.len = strlen(identifier);
  r = chash_get(manager->pixbuf_hash, &key, &value);
  if (r == 0) {
    scaled = value.data;
    
    return scaled;
  }
  
  pixbuf = etpan_icon_manager_get_pixbuf(manager, name);
  if (pixbuf == NULL) {
    ETPAN_LOG("should have cached %s", name);
    etpan_crash();
  }
  
  scaled = gdk_pixbuf_scale_simple(pixbuf, size, size, GDK_INTERP_HYPER);
  
  key.data = identifier;
  key.len = strlen(identifier);
  value.data = scaled;
  value.len = 0;
  
  r = chash_set(manager->pixbuf_hash, &key, &value, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  return scaled;
}
