#include "etpan-main-window.h"

#include <stdlib.h>
#include <gdk/gdkkeysyms.h>

#include "etpan-toolbar.h"
#include "etpan-bookmark-view.h"
#include "etpan-folder-list.h"
#include "etpan-status-bar.h"
#include "etpan-backend.h"
#include "etpan-tabbed-message-list.h"
#include "etpan-message-composer-window.h"
#include "etpan-message-list.h"
#include "etpan-ui-config.h"
#include "etpan-preferences-window.h"
#include "etpan-icon-manager.h"
#include "etpan-search-field.h"
#include "etpan-ui-config.h"
#include "etpan-contextual-menu.h"

struct etpan_main_window * etpan_main_window_new(void)
{
  struct etpan_main_window * main_window;
  struct etpan_toolbar * toolbar;
  struct etpan_bookmark_view * bookmark_view;
  struct etpan_folder_list * folder_list;
  struct etpan_status_bar * status_bar;
  GtkWidget * window;
  GtkWidget * vbox;
  GtkWidget * toolbar_widget;
  GtkWidget * hpaned;
  GtkWidget * folder_list_widget;
  GtkWidget * vbox_msg;
  GtkWidget * bookmark_widget;
  GtkWidget * status_bar_widget;
  struct etpan_tabbed_message_list * tabbed;
  GtkWidget * tabbed_widget;
  GdkPixbuf * icon;
  
  main_window = malloc(sizeof(* main_window));
  if (main_window == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  /* window */
  window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW(window), _("etPanX"));

  gtk_window_set_default_size(GTK_WINDOW(window), 1000, 800);
  
  /* box for toolbar + folder list view + bookmarks +
     message list + message view */
  vbox = gtk_vbox_new(FALSE, 0);
  gtk_widget_show(vbox);
  gtk_container_add(GTK_CONTAINER(window), vbox);
  
  /* toolbar */
  toolbar = etpan_toolbar_new();
  
  toolbar_widget = etpan_toolbar_get_main_widget(toolbar);
  gtk_box_pack_start(GTK_BOX(vbox), toolbar_widget, FALSE, FALSE, 0);
  
  /* split view for folder / (bookmarks + message list + message view) */
  hpaned = gtk_hpaned_new();
  gtk_widget_show(hpaned);
  gtk_box_pack_start(GTK_BOX(vbox), hpaned, TRUE, TRUE, 0);
  gtk_paned_set_position(GTK_PANED(hpaned), 200);
  
  /* folder list view */
  folder_list = etpan_folder_list_new();
  
  folder_list_widget = etpan_folder_list_get_main_widget(folder_list);
  gtk_paned_pack1(GTK_PANED(hpaned), folder_list_widget, FALSE, TRUE);
  
  /* box for bookmarks + message list + message view */
  vbox_msg = gtk_vbox_new(FALSE, 0);
  gtk_widget_show(vbox_msg);
  gtk_paned_pack2(GTK_PANED(hpaned), vbox_msg, TRUE, TRUE);
  
  bookmark_view = etpan_bookmark_view_new();
  
  bookmark_widget = etpan_bookmark_view_get_main_widget(bookmark_view);
  gtk_box_pack_start(GTK_BOX(vbox_msg), bookmark_widget, FALSE, FALSE, 0);
  
  tabbed = etpan_tabbed_message_list_new();
  tabbed_widget = etpan_tabbed_message_list_get_main_widget(tabbed);
  gtk_box_pack_start(GTK_BOX(vbox_msg), tabbed_widget, TRUE, TRUE, 0);
  
  status_bar = etpan_status_bar_new();
  
  main_window->search_field = etpan_search_field_new();
  
  status_bar_widget = etpan_status_bar_get_main_widget(status_bar);
  gtk_box_pack_start(GTK_BOX(vbox), status_bar_widget, FALSE, TRUE, 0);
  
  main_window->toolbar = toolbar;
  main_window->bookmark_view = bookmark_view;
  main_window->folder_list = folder_list;
  main_window->tabbed_message_list = tabbed;
  main_window->status_bar = status_bar;
  
  main_window->window = window;
  main_window->vbox = vbox;
  main_window->hpaned = hpaned;
  main_window->vbox_msg = vbox_msg;
  main_window->saved_split_size = 0;
  
  main_window->searching = 0;
  
  icon = etpan_icon_manager_get_pixbuf(etpan_icon_manager_get_default(),
      "etpanX");
  gtk_window_set_icon(GTK_WINDOW(window), icon);
  
  return main_window;
}

void etpan_main_window_free(struct etpan_main_window * main_window)
{
  etpan_search_field_free(main_window->search_field);
  etpan_status_bar_free(main_window->status_bar);
  etpan_tabbed_message_list_free(main_window->tabbed_message_list);
  etpan_bookmark_view_free(main_window->bookmark_view);
  gtk_widget_destroy(main_window->vbox_msg);
  etpan_folder_list_free(main_window->folder_list);
  gtk_widget_destroy(main_window->hpaned);
  etpan_toolbar_free(main_window->toolbar);
  gtk_widget_destroy(main_window->vbox);
  gtk_widget_destroy(main_window->window);
  free(main_window);
}

GtkWidget * etpan_main_window_get_main_widget(struct etpan_main_window *
    main_window)
{
  return main_window->window;
}

static void new_message(struct etpan_main_window * main_window)
{
  struct etpan_message_composer_window * composer_window;
  GtkWidget * window;
  struct etpan_account * account;
  char * selection;
  struct etpan_message_composer * composer;
  
  composer_window = etpan_message_composer_window_new();
  etpan_message_composer_window_setup(composer_window);
  
  composer = etpan_message_composer_new();
  etpan_message_composer_add_signature(composer);
  etpan_message_composer_window_set_composer(composer_window, composer);
  etpan_message_composer_free(composer);
  
  account = NULL;
  selection = etpan_folder_list_get_selection(main_window->folder_list);

  if (selection != NULL) {
    account = etpan_folder_list_get_account(main_window->folder_list,
        selection);
  }
  else {
    carray * list;
    
    list = etpan_account_manager_get_ordered_list(etpan_account_manager_get_default());
    if (carray_count(list) > 0)
      account = carray_get(list, 0);
  }
  
  if (account != NULL)
    etpan_message_composer_window_set_account(composer_window, account);
  
  window = etpan_message_composer_window_get_main_widget(composer_window);
  gtk_widget_show(window);
}

static void reply_message(struct etpan_main_window * main_window)
{
  struct etpan_message_composer_window * composer;
  GtkWidget * window;
  struct etpan_message_list * msg_list;
  
  composer = etpan_message_composer_window_new();
  etpan_message_composer_window_setup(composer);
  window = etpan_message_composer_window_get_main_widget(composer);
  gtk_widget_show(window);

  msg_list = etpan_tabbed_message_list_get_current_message_list(main_window->tabbed_message_list);
  
  etpan_message_list_reply(msg_list, composer);
}

static void forward_message(struct etpan_main_window * main_window)
{
  struct etpan_message_composer_window * composer;
  GtkWidget * window;
  struct etpan_message_list * msg_list;
  
  composer = etpan_message_composer_window_new();
  etpan_message_composer_window_setup(composer);
  window = etpan_message_composer_window_get_main_widget(composer);
  gtk_widget_show(window);
  
  msg_list = etpan_tabbed_message_list_get_current_message_list(main_window->tabbed_message_list);
  
  etpan_message_list_forward(msg_list, composer);
}

static void delete_message(struct etpan_main_window * main_window)
{
  struct etpan_message_list * msg_list;
  
  msg_list = etpan_tabbed_message_list_get_current_message_list(main_window->tabbed_message_list);
  
  etpan_message_list_delete(msg_list);
}

static void spam_message(struct etpan_main_window * main_window)
{
  static int spam_state = 1;
  GtkWidget * tmp_image;
  
  spam_state = !spam_state;
  
#define ICON(name) \
  etpan_icon_manager_new_scaled_image(etpan_icon_manager_get_default(), \
      name, 32)

  if (spam_state)
    tmp_image = ICON("spam");
  else
    tmp_image = ICON("non-spam");

#undef ICON

  gtk_widget_show(tmp_image);
  if (spam_state)
    etpan_toolbar_replace_item(main_window->toolbar, 6, _("Junk"), tmp_image);
  else
    etpan_toolbar_replace_item(main_window->toolbar, 6, _("Not Junk"), tmp_image);
}

static void preferences_open(struct etpan_main_window * main_window)
{
  struct etpan_preferences_window * prefs;
  GtkWidget * window;
  (void) main_window;
  
  prefs = etpan_preferences_window_get_default();
  window = etpan_preferences_panel_get_main_widget(prefs);
  gtk_widget_show(window);
  gtk_window_present(GTK_WINDOW(window));
}

static void button_clicked(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  int button_index;
  struct etpan_main_window * main_window;
  (void) signal_name;
  (void) sender;
  
  main_window = user_data;
  
  button_index = * (int *) signal_data;
  
  switch (button_index) {
  case 0:
    ETPAN_LOG("new message");
    new_message(main_window);
    break;
  case 2:
    ETPAN_LOG("reply");
    reply_message(main_window);
    break;
  case 3:
    ETPAN_LOG("forward");
    forward_message(main_window);
    break;
  case 5:
    ETPAN_LOG("delete");
    delete_message(main_window);
    break;
#if 0
  case 6:
    ETPAN_LOG("spam");
    spam_message(main_window);
    break;
#endif
#if 0
  case 8:
    ETPAN_LOG("preferences");
    preferences_open(main_window);
    break;
#endif
  }
}

static void search_done_signal_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);

static void search_field_modified(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_message_list * msg_list;
  struct etpan_main_window * main_window;
  const char * value;
  int search_type;
  (void) signal_name;
  (void) signal_data;
  (void) sender;
  
  main_window = user_data;
  msg_list = etpan_tabbed_message_list_get_current_message_list(main_window->tabbed_message_list);

  value = etpan_search_field_get_value(main_window->search_field);
  search_type = etpan_search_field_get_search_type(main_window->search_field);
  etpan_message_list_search(msg_list,
      search_type, value);
  
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_MESSAGE_LIST_SEARCH_DONE_SIGNAL, msg_list, main_window,
      search_done_signal_handler);
  
  if (main_window->searching)
    return;
  
  main_window->searching = 1;
  
  /* XXX - start progress animation */
}

static void search_done_signal_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_message_list * msg_list;
  struct etpan_main_window * main_window;
  (void) signal_name;
  (void) signal_data;
  
  msg_list = sender;
  main_window = user_data;
  
  /* XXX - stop progress animation */
  
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_MESSAGE_LIST_SEARCH_DONE_SIGNAL, msg_list, main_window,
      search_done_signal_handler);
  main_window->searching = 0;
}

static void menu_item_clicked(char * signal_name, void * sender,
    void * signal_data, void * user_data);
static void message_menu_item_clicked(char * signal_name, void * sender,
    void * signal_data, void * user_data);
static void folder_menu_item_clicked(char * signal_name, void * sender,
    void * signal_data, void * user_data);

static void setup_menu(struct etpan_main_window * main_window)
{
  struct etpan_contextual_menu * menu;
  struct etpan_contextual_menu * message_menu;
  struct etpan_contextual_menu * folder_menu;
  
  menu = etpan_contextual_menu_new();
  
  message_menu = etpan_contextual_menu_new();
  etpan_contextual_menu_add_item(message_menu, _("Mark as read"), NULL);
  etpan_contextual_menu_add_item(message_menu, _("Mark"), NULL);
  etpan_contextual_menu_add_separator(message_menu);
  etpan_contextual_menu_add_item(message_menu, _("Delete"), NULL);
  etpan_contextual_menu_add_separator(message_menu);
  etpan_contextual_menu_add_item(message_menu, _("Reply"), NULL);
  etpan_contextual_menu_add_item(message_menu, _("Forward"), NULL);
  etpan_contextual_menu_add_sub(menu, _("Message"), NULL, message_menu);
  
  folder_menu = etpan_contextual_menu_new();
  etpan_contextual_menu_add_item(folder_menu, _("New mailbox..."), NULL);
  etpan_contextual_menu_add_item(folder_menu, _("Rename..."), NULL);
  etpan_contextual_menu_add_item(folder_menu, _("Delete"), NULL);
  etpan_contextual_menu_add_sub(menu, _("Folder"), NULL, folder_menu);
  
  etpan_contextual_menu_add_separator(menu);
  
  etpan_contextual_menu_add_item(menu, _("Preferences"), NULL);
  
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
    ETPAN_CONTEXTUAL_MENU_CLICKED_SIGNAL, menu, main_window,
      menu_item_clicked);
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
    ETPAN_CONTEXTUAL_MENU_CLICKED_SIGNAL, message_menu, main_window,
      message_menu_item_clicked);
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
    ETPAN_CONTEXTUAL_MENU_CLICKED_SIGNAL, folder_menu, main_window,
      folder_menu_item_clicked);
  
  main_window->menu = menu;
  main_window->message_menu = menu;
  main_window->folder_menu = menu;
}

static void unsetup_menu(struct etpan_main_window * main_window)
{
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
    ETPAN_CONTEXTUAL_MENU_CLICKED_SIGNAL, main_window->folder_menu,
      main_window,
      folder_menu_item_clicked);
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_CONTEXTUAL_MENU_CLICKED_SIGNAL, main_window->message_menu,
      main_window,
      message_menu_item_clicked);
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
    ETPAN_CONTEXTUAL_MENU_CLICKED_SIGNAL, main_window->menu,
      main_window,
      menu_item_clicked);
  
  etpan_contextual_menu_free(main_window->menu);
  main_window->folder_menu = NULL;
  main_window->message_menu = NULL;
  main_window->menu = NULL;
}

static void menu_item_clicked(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_main_window * main_window;
  struct etpan_contextual_menu * menu;
  int * pindex;
  (void) signal_name;
  
  menu = sender;
  main_window = user_data;
  pindex = signal_data;
  
  switch (* pindex) {
  case 3:
    preferences_open(main_window);
    break;
  }
}

static void message_menu_item_clicked(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_main_window * main_window;
  struct etpan_contextual_menu * menu;
  int * pindex;
  (void) signal_name;
  
  menu = sender;
  main_window = user_data;
  pindex = signal_data;
  
  switch (* pindex) {
  case 0:
    break;
  }
}

static void folder_menu_item_clicked(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_main_window * main_window;
  struct etpan_contextual_menu * menu;
  int * pindex;
  (void) signal_name;
  
  menu = sender;
  main_window = user_data;
  pindex = signal_data;
  
  switch (* pindex) {
  case 0:
    break;
  }
}

static void setup_toolbar(struct etpan_main_window * main_window)
{
  GtkWidget * tmp_image;
  GtkWidget * toolbar;
  GtkWidget * entry;
  GtkWidget * search_field_widget;
  
  toolbar = etpan_toolbar_get_main_widget(main_window->toolbar);
  etpan_toolbar_clear(main_window->toolbar);
  
#define ICON(name) \
  etpan_icon_manager_new_scaled_image(etpan_icon_manager_get_default(), \
      name, 32)
  
  /* compose button */
  tmp_image = ICON("compose");
  gtk_widget_show(tmp_image);
  etpan_toolbar_add_item(main_window->toolbar, _("Compose"), tmp_image);
  /* separator */
  etpan_toolbar_add_separator(main_window->toolbar);
  /* reply button */
  tmp_image = ICON("reply");
  gtk_widget_show(tmp_image);
  etpan_toolbar_add_item(main_window->toolbar, _("Reply"), tmp_image);
  /* forward button */
  tmp_image = ICON("forward");
  gtk_widget_show(tmp_image);
  etpan_toolbar_add_item(main_window->toolbar, _("Forward"), tmp_image);
  /* separator */
  etpan_toolbar_add_separator(main_window->toolbar);
  /* delete button */
  tmp_image = ICON("delete");
  gtk_widget_show(tmp_image);
  etpan_toolbar_add_item(main_window->toolbar, _("Delete"), tmp_image);
  /* junk button */
#if 0
  tmp_image = ICON("spam");
  gtk_widget_show(tmp_image);
  etpan_toolbar_add_item(main_window->toolbar, _("Junk"), tmp_image);
#endif
  /* separator */
  etpan_toolbar_add_separator(main_window->toolbar);
#if 0
  /* preferences button */
  tmp_image = ICON("preferences");
  gtk_widget_show(tmp_image);
  etpan_toolbar_add_item(main_window->toolbar, _("Preferences"), tmp_image);
#endif
  /* menu */
  tmp_image = ICON("menu");
  gtk_widget_show(tmp_image);
  etpan_toolbar_add_item_menu(main_window->toolbar, _("Menu"), tmp_image,
      main_window->menu);
  /* expander */
  etpan_toolbar_add_expander(main_window->toolbar);
  /* search field */
  etpan_search_field_setup(main_window->search_field);
  entry = etpan_search_field_get_entry(main_window->search_field);
  gtk_entry_set_width_chars(GTK_ENTRY(entry), 40);
  search_field_widget =
    etpan_search_field_get_main_widget(main_window->search_field);
  etpan_toolbar_add_widget(main_window->toolbar, search_field_widget);
  /* expander */
  etpan_toolbar_add_space(main_window->toolbar);
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_SEARCH_FIELD_MODIFIED_SIGNAL, main_window->search_field,
      main_window, search_field_modified);
#undef ICON
  
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_TOOLBAR_BUTTONCLICKED_SIGNAL, main_window->toolbar,
      main_window, button_clicked);
}

static void unsetup_toolbar(struct etpan_main_window * main_window)
{
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_TOOLBAR_BUTTONCLICKED_SIGNAL, main_window->toolbar,
      main_window, button_clicked);

  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_SEARCH_FIELD_MODIFIED_SIGNAL, main_window->search_field,
      main_window, search_field_modified);
  
  etpan_search_field_unsetup(main_window->search_field);
  
  etpan_toolbar_clear(main_window->toolbar);
}

struct etpan_toolbar *
etpan_main_window_get_toolbar(struct etpan_main_window * main_window)
{
  return main_window->toolbar;
}

struct etpan_bookmark_view *
etpan_main_window_get_bookmark_view(struct etpan_main_window * main_window)
{
  return main_window->bookmark_view;
}

struct etpan_folder_list *
etpan_main_window_get_folder_list(struct etpan_main_window * main_window)
{
  return main_window->folder_list;
}

struct etpan_status_bar *
etpan_main_window_get_status_bar(struct etpan_main_window * main_window)
{
  return main_window->status_bar;
}

static gboolean split_press_handler(GtkWidget * widget,
    GdkEventButton * event,
    gpointer user_data)
{
  struct etpan_main_window * main_window;
  GtkPaned * paned;
  
  main_window = (struct etpan_main_window *) user_data;
  
  paned = GTK_PANED(widget);
  if (event->type == GDK_2BUTTON_PRESS) {
    if (paned->handle == event->window) {
      int current_size;
      
      paned->in_drag = 0;
      gdk_display_pointer_ungrab(gtk_widget_get_display(GTK_WIDGET(paned)),
          GDK_CURRENT_TIME);
        
      current_size = gtk_paned_get_position(GTK_PANED(main_window->hpaned));
      if (current_size == 0) {
        if (main_window->saved_split_size == 0)
          main_window->saved_split_size = 200;
        gtk_paned_set_position(GTK_PANED(main_window->hpaned),
            main_window->saved_split_size);
        ETPAN_MAINWINDOW_LOG("set to non 0");
      }
      else {
        main_window->saved_split_size = current_size;
        gtk_paned_set_position(GTK_PANED(main_window->hpaned), 0);
        ETPAN_MAINWINDOW_LOG("set to 0");
      }
      
      return TRUE;
    }
  }
  
  return FALSE;
}

static void add_new_tab(struct etpan_main_window * main_window)
{
  unsigned int tabbed_index;
  struct etpan_tabbed_message_list * tabbed;
  
  tabbed = main_window->tabbed_message_list;
  
  etpan_tabbed_message_list_add_page(tabbed, &tabbed_index);
  
  etpan_tabbed_message_list_set_current_page(tabbed, tabbed_index);
}

static void close_current_tab(struct etpan_main_window * main_window)
{
  unsigned int count;
  int tabbed_index;
  struct etpan_tabbed_message_list * tabbed;
  
  tabbed = main_window->tabbed_message_list;
  count = etpan_tabbed_message_list_get_count(tabbed);
  
  if (count <= 1)
    return;
  
  tabbed_index =
    etpan_tabbed_message_list_get_current_page(tabbed);
  if (tabbed_index < 0)
    return;
  
  if ((unsigned int) tabbed_index == count - 1)
    etpan_tabbed_message_list_set_current_page(tabbed, count - 1);
  else
    etpan_tabbed_message_list_set_current_page(tabbed, tabbed_index + 1);
  
  etpan_tabbed_message_list_close_page(tabbed, tabbed_index);
}

static gboolean key_handler(GtkWidget * widget,
    GdkEventKey * event,
    gpointer user_data)
{
  struct etpan_main_window * main_window;
  unsigned int state;
  int alt;
  int ctrl;
  (void) widget;
  
  main_window = (struct etpan_main_window *) user_data;
  ETPAN_MAINWINDOW_LOG("main_window %p", main_window);
  
  state = event->state & (GDK_SHIFT_MASK | GDK_CONTROL_MASK | GDK_MOD1_MASK |
      GDK_MOD2_MASK | GDK_MOD3_MASK | GDK_MOD4_MASK |
      GDK_MOD5_MASK);
  
  alt = (state & GDK_MOD1_MASK) != 0;
  ctrl = (state & GDK_CONTROL_MASK) != 0;
  
  if ((event->keyval == GDK_t) && (alt || ctrl)) {
    add_new_tab(main_window);
    return TRUE;
  }
  if ((event->keyval == GDK_w) && (alt || ctrl)) {
    close_current_tab(main_window);
    return TRUE;
  }
  else {
    return FALSE;
  }
}

void etpan_main_window_setup(struct etpan_main_window * main_window)
{
  int r;
  unsigned int tabbed_index;
  int top;
  int left;
  int width;
  int height;
  int slider;
  
  setup_menu(main_window);
  setup_toolbar(main_window);
  
  etpan_folder_list_setup(main_window->folder_list);
  etpan_tabbed_message_list_setup(main_window->tabbed_message_list);
  etpan_status_bar_setup(main_window->status_bar);
  etpan_bookmark_view_setup(main_window->bookmark_view);
  
  etpan_tabbed_message_list_set_folder_list(main_window->tabbed_message_list,
      main_window->folder_list);
  etpan_tabbed_message_list_set_bookmark_view(main_window->tabbed_message_list,
      main_window->bookmark_view);
  etpan_tabbed_message_list_set_status_bar(main_window->tabbed_message_list,
      main_window->status_bar);
  etpan_tabbed_message_list_set_search_field(main_window->tabbed_message_list,
      main_window->search_field);
  
  etpan_tabbed_message_list_add_page(main_window->tabbed_message_list,
      &tabbed_index);
  etpan_tabbed_message_list_set_current_page(main_window->tabbed_message_list,
      tabbed_index);
  
  main_window->hpaned_press_signal_id =
    g_signal_connect(main_window->hpaned,
        "button-press-event", G_CALLBACK(split_press_handler),
        (gpointer) main_window);

  main_window->key_press_signal_id =
    g_signal_connect(main_window->window, "key-press-event",
      G_CALLBACK(key_handler), (gpointer) main_window);
  
  etpan_ui_window_set(etpan_ui_config_get_default(), "main",
      main_window->window);
  etpan_ui_slider_set(etpan_ui_config_get_default(), "folder-list-slider",
    main_window->hpaned);
}

void etpan_main_window_unsetup(struct etpan_main_window * main_window)
{
  etpan_ui_set_from_slider(etpan_ui_config_get_default(), "folder-list-slider",
    main_window->hpaned);
  etpan_ui_set_from_window(etpan_ui_config_get_default(), "main",
      main_window->window);
  
  g_signal_handler_disconnect(main_window->window,
      main_window->key_press_signal_id);
  
  g_signal_handler_disconnect(main_window->hpaned,
      main_window->hpaned_press_signal_id);
  
  etpan_tabbed_message_list_set_search_field(main_window->tabbed_message_list,
      NULL);
  etpan_tabbed_message_list_set_folder_list(main_window->tabbed_message_list,
      NULL);
  etpan_tabbed_message_list_set_bookmark_view(main_window->tabbed_message_list,
      NULL);
  etpan_tabbed_message_list_set_status_bar(main_window->tabbed_message_list,
      NULL);
  
  etpan_bookmark_view_unsetup(main_window->bookmark_view);
  etpan_status_bar_unsetup(main_window->status_bar);
  etpan_tabbed_message_list_unsetup(main_window->tabbed_message_list);
  etpan_folder_list_unsetup(main_window->folder_list);
  
  unsetup_toolbar(main_window);
  unsetup_menu(main_window);
}

struct etpan_tabbed_message_list *
etpan_main_window_get_tabbed_message_list(struct etpan_main_window *
    main_window)
{
  return main_window->tabbed_message_list;
}

static struct etpan_main_window * default_main_window = NULL;

void etpan_main_window_set_default(struct etpan_main_window * main_window)
{
  default_main_window = main_window;
}

struct etpan_main_window * etpan_main_window_get_default(void)
{
  return default_main_window;
}

#if 0
static void menu_position(GtkMenu * menu, gint * px, gint * py,
    gboolean * ppush_in, gpointer user_data);

static void show_menu(struct etpan_main_window * main_window)
{
  int button;
  int event_time;
  struct etpan_contextual_menu * menu;
  GtkWidget * widget;
  
  menu = etpan_contextual_menu_new();
  etpan_contextual_menu_add_item(menu, _("Preferences"), NULL);
  etpan_contextual_menu_add_item(menu, _("Message"), NULL);
  etpan_contextual_menu_add_item(menu, _("Folder"), NULL);
  
  widget = etpan_contextual_menu_get_main_widget(menu);
  
  button = 0;
  event_time = gtk_get_current_event_time();
  gtk_menu_popup(GTK_MENU(widget), NULL, NULL, menu_position, main_window,
      button, event_time);
}

static void menu_position(GtkMenu * menu, gint * px, gint * py,
    gboolean * ppush_in, gpointer user_data)
{
  gint x;
  gint y;
  gint width;
  gint height;
  gint depth;
  GtkWidget * button;
  struct etpan_main_window * main_window;
  GtkWidget * widget;
  
  main_window = user_data;
  button = etpan_toolbar_get_item_at_position(main_window->toolbar, 9);
  gdk_window_get_deskrelative_origin(button->window, &x, &y);
  x += button->allocation.x;
  y += button->allocation.y;
  width = button->allocation.width;
  height = button->allocation.height;
  
  * px = x;
  * py = y + height;
  
  * ppush_in = 0;
}
#endif
