#include "etpan-message-composer-window.h"

#include <stdlib.h>
#include <gdk/gdkkeysyms.h>
#include <libgen.h>
#include <unistd.h>

#include "etpan-toolbar.h"
#include "etpan-backend.h"
#include "etpan-ui-config.h"
#include "etpan-gtk-tree-model.h"
#include "etpan-completion.h"
#include "etpan-status-bar.h"
#include "etpan-icon-manager.h"

/* TODO : reparse newsgroup list */

static void set_reference_message(struct etpan_message_composer_window * composer_window, struct etpan_message * msg);

static unsigned int get_n_columns(struct etpan_gtk_tree_data_source *
    datasource)
{
  (void) datasource;
  
  return 2;
}

static GType get_column_type(struct etpan_gtk_tree_data_source * datasource,
    unsigned int column_index)
{
  static GType column_types[2] = {
    G_TYPE_STRING, G_TYPE_POINTER
  };
  (void) datasource;
  
  return column_types[column_index];
}
  
static unsigned int get_children_count(struct etpan_gtk_tree_data_source *
    datasource, void * item)
{
  carray * list;
  (void) datasource;
  
  list = etpan_account_manager_get_ordered_list(etpan_account_manager_get_default());
  
  if (item == NULL)
    return carray_count(list);
  else
    return 0;
}

static int item_has_child(struct etpan_gtk_tree_data_source *
    datasource, void * item)
{
  carray * list;
  (void) datasource;
  
  list = etpan_account_manager_get_ordered_list(etpan_account_manager_get_default());
  
  if (item == NULL)
    return carray_count(list) > 0;
  else
    return 0;
}

static void * get_child_item(struct etpan_gtk_tree_data_source *
    datasource, void * item, unsigned int index)
{
  carray * list;
  (void) datasource;
  (void) item;
  
  list = etpan_account_manager_get_ordered_list(etpan_account_manager_get_default());
  
  return carray_get(list, index);
}

static void get_item_value(struct etpan_gtk_tree_data_source *
    datasource, void * item, unsigned int column_index,
    GValue * value)
{
  struct etpan_account * account;
  (void) datasource;
  
  account = item;
    
  switch (column_index) {
  case 0:
    {
      char * str;
      char * display_name;
      char * mail;
      
      display_name = etpan_account_get_display_name(account);
      mail = etpan_account_get_mail(account);
      if (mail == NULL)
        mail = "";
      
      if (display_name == NULL) {
        size_t len;
        
        len = strlen(mail) + 2;
        str = malloc(len + 1);
        if (str != NULL)
          snprintf(str, len + 1, "<%s>", mail);
      }
      else {
        size_t len;
        
        len = strlen(display_name) + strlen(mail) + 3;
        str = malloc(len + 1);
        if (str != NULL)
          snprintf(str, len + 1, "%s <%s>", display_name, mail);
      }
      
      if (str == NULL) {
        g_value_init(value, G_TYPE_STRING);
        g_value_set_string(value, "");
      }
      else {
        g_value_init(value, G_TYPE_STRING);
        g_value_set_string(value, str);
        
        free(str);
      }
    }
    break;
  case 1:
    {
      g_value_init(value, G_TYPE_POINTER);
      g_value_set_pointer(value, account);
    }
    break;
  }
}

struct etpan_message_composer_window * etpan_message_composer_window_new(void)
{
  struct etpan_message_composer_window * composer;
  struct etpan_toolbar * toolbar;
  GtkWidget * toolbar_widget;
  GtkWidget * window;
  GtkWidget * vbox;
  GtkWidget * table;
  etpan_gtk_tree_model * treemodel;
  GtkCellRenderer * col_name_renderer;
  GtkWidget * account_label;
  GtkWidget * account_combo;
  GtkWidget * to_label;
  GtkWidget * subject_label;
  GtkWidget * subject_entry;
  GtkWidget * cc_label;
  GtkWidget * bcc_label;
  GtkWidget * scrolledwindow;
  GtkWidget * textview;
  GtkTextBuffer * textbuffer;
  unsigned int line;
  struct etpan_completion * to_completion;
  struct etpan_completion * cc_completion;
  struct etpan_completion * bcc_completion;
  GtkWidget * entry;
  GtkWidget * reply_to_label;
  struct etpan_completion * reply_to_completion;
  GtkWidget * newsgroups_label;
  GtkWidget * newsgroups_entry;
  struct etpan_status_bar * status_bar;
  GtkWidget * status_bar_widget;
  GdkPixbuf * icon;
  
  composer = malloc(sizeof(* composer));
  if (composer == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  composer->attach_list = carray_new(4);
  if (composer->attach_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  composer->attach_index = 0;
  composer->references = etpan_message_id_list_new();
  
  /* window */
  window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW(window), _("etPanX - Composer"));
  gtk_window_set_default_size(GTK_WINDOW(window), 500, 400);
  
  /* box for toolbar + recipient + text */
  vbox = gtk_vbox_new(FALSE, 0);
  gtk_widget_show(vbox);
  gtk_container_add(GTK_CONTAINER(window), vbox);
  
  /* toolbar */
  toolbar = etpan_toolbar_new();
  toolbar_widget = etpan_toolbar_get_main_widget(toolbar);
  gtk_box_pack_start(GTK_BOX(vbox), toolbar_widget, FALSE, FALSE, 0);
  
  /* recipient and subject */
  table = gtk_table_new(7, 3, FALSE);
  gtk_widget_show(table);
  gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, 0);
  line = 0;
  
  /* Account field */
  treemodel = etpan_gtk_tree_model_new();
  composer->account_datasource.data = composer;
  composer->account_datasource.get_n_columns = get_n_columns;
  composer->account_datasource.get_column_type = get_column_type;
  composer->account_datasource.get_children_count = get_children_count;
  composer->account_datasource.item_has_child = item_has_child;
  composer->account_datasource.get_child_item = get_child_item;
  composer->account_datasource.get_item_value = get_item_value;
  etpan_gtk_tree_model_set_datasource(treemodel,
      &composer->account_datasource);
  
  account_label = gtk_label_new(_("Account: "));
  gtk_label_set_justify(GTK_LABEL(account_label), GTK_JUSTIFY_CENTER);
  gtk_widget_show(account_label);
  gtk_table_attach(GTK_TABLE(table),
      account_label, 1, 2, line, line + 1, GTK_FILL, GTK_FILL, 0, 0);
  
  account_combo = gtk_combo_box_new();
  
  gtk_combo_box_set_model(GTK_COMBO_BOX(account_combo),
      GTK_TREE_MODEL(treemodel));
  
  col_name_renderer = gtk_cell_renderer_text_new();
  gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(account_combo),
      col_name_renderer, TRUE);
  gtk_cell_layout_set_attributes(GTK_CELL_LAYOUT(account_combo),
      col_name_renderer, "text", 0,
      NULL);
  
  gtk_widget_show(account_combo);
  gtk_table_attach(GTK_TABLE(table),
      account_combo, 2, 3, line, line + 1, GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
  line ++;
  
  /* Newsgroups field */
  newsgroups_label = gtk_label_new(_("Newsgroups: "));
  gtk_label_set_justify(GTK_LABEL(newsgroups_label), GTK_JUSTIFY_RIGHT);
  gtk_widget_show(newsgroups_label);
  gtk_table_attach(GTK_TABLE(table),
      newsgroups_label, 1, 2, line, line + 1, GTK_FILL, GTK_FILL, 0, 0);
  
  newsgroups_entry = gtk_entry_new();
  gtk_widget_show(newsgroups_entry);
  gtk_table_attach(GTK_TABLE(table),
      newsgroups_entry, 2, 3, line, line + 1, GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
  line ++;
  
  /* To field */
  to_label = gtk_label_new(_("To: "));
  gtk_label_set_justify(GTK_LABEL(to_label), GTK_JUSTIFY_CENTER);
  gtk_widget_show(to_label);
  gtk_table_attach(GTK_TABLE(table),
      to_label, 1, 2, line, line + 1, GTK_FILL, GTK_FILL, 0, 0);
  
  to_completion = etpan_completion_new();
  entry = etpan_completion_get_main_widget(to_completion);
  gtk_widget_show(entry);
  gtk_table_attach(GTK_TABLE(table),
      entry, 2, 3, line, line + 1, GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
  line ++;
  
  /* Cc field */
  cc_label = gtk_label_new(_("Cc: "));
  gtk_label_set_justify(GTK_LABEL(cc_label), GTK_JUSTIFY_RIGHT);
  gtk_widget_show(cc_label);
  gtk_table_attach(GTK_TABLE(table),
      cc_label, 1, 2, line, line + 1, GTK_FILL, GTK_FILL, 0, 0);
  
  cc_completion = etpan_completion_new();
  entry = etpan_completion_get_main_widget(cc_completion);
  gtk_widget_show(entry);
  gtk_table_attach(GTK_TABLE(table),
      entry, 2, 3, line, line + 1, GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
  line ++;
  
  /* Bcc field */
  bcc_label = gtk_label_new(_("Bcc: "));
  gtk_label_set_justify(GTK_LABEL(bcc_label), GTK_JUSTIFY_RIGHT);
  gtk_widget_show(bcc_label);
  gtk_table_attach(GTK_TABLE(table),
      bcc_label, 1, 2, line, line + 1, GTK_FILL, GTK_FILL, 0, 0);
  
  bcc_completion = etpan_completion_new();
  entry = etpan_completion_get_main_widget(bcc_completion);
  gtk_widget_show(entry);
  gtk_table_attach(GTK_TABLE(table),
      entry, 2, 3, line, line + 1, GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
  line ++;
  
  /* Reply-To field */
  reply_to_label = gtk_label_new(_("Reply-To: "));
  gtk_label_set_justify(GTK_LABEL(reply_to_label), GTK_JUSTIFY_RIGHT);
  gtk_widget_show(reply_to_label);
  gtk_table_attach(GTK_TABLE(table),
      reply_to_label, 1, 2, line, line + 1, GTK_FILL, GTK_FILL, 0, 0);
  
  reply_to_completion = etpan_completion_new();
  entry = etpan_completion_get_main_widget(reply_to_completion);
  gtk_widget_show(entry);
  gtk_table_attach(GTK_TABLE(table),
      entry, 2, 3, line, line + 1, GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
  line ++;
  
  /* Subject field */
  subject_label = gtk_label_new(_("Subject: "));
  gtk_label_set_justify(GTK_LABEL(subject_label), GTK_JUSTIFY_RIGHT);
  gtk_widget_show(subject_label);
  gtk_table_attach(GTK_TABLE(table),
      subject_label, 1, 2, line, line + 1, GTK_FILL, GTK_FILL, 0, 0);
  
  subject_entry = gtk_entry_new();
  gtk_widget_show(subject_entry);
  gtk_table_attach(GTK_TABLE(table),
      subject_entry, 2, 3, line, line + 1, GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
  line ++;
  
  scrolledwindow = gtk_scrolled_window_new(NULL, NULL);
  gtk_widget_show(scrolledwindow);
  gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrolledwindow),
      GTK_SHADOW_IN);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledwindow),
      GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_box_pack_start(GTK_BOX(vbox), scrolledwindow, TRUE, TRUE, 0);
  
  textview = gtk_text_view_new();
  gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(textview), GTK_WRAP_WORD);
  gtk_widget_show(textview);
  
  textbuffer = gtk_text_buffer_new(NULL);
  gtk_text_view_set_buffer(GTK_TEXT_VIEW(textview), textbuffer);
  
  gtk_container_add(GTK_CONTAINER(scrolledwindow), textview);
  
  gtk_text_buffer_create_tag(textbuffer, "attach",
      NULL);
  
  status_bar = etpan_status_bar_new();
  status_bar_widget = etpan_status_bar_get_main_widget(status_bar);
  gtk_box_pack_start(GTK_BOX(vbox), status_bar_widget, FALSE, TRUE, 0);
  
  composer->window = window;
  composer->vbox = vbox;
  composer->toolbar = toolbar;
  composer->table = table;
  composer->account_label = account_label;
  composer->account_combo = account_combo;
  composer->treemodel = treemodel;
  composer->to_label = to_label;
  composer->to_completion = to_completion;
  composer->subject_label = subject_label;
  composer->subject_entry = subject_entry;
  composer->cc_label = cc_label;
  composer->cc_completion = cc_completion;
  composer->bcc_label = bcc_label;
  composer->bcc_completion = bcc_completion;
  composer->reply_to_label = reply_to_label;
  composer->reply_to_completion = reply_to_completion;
  composer->newsgroups_label = newsgroups_label;
  composer->newsgroups_entry = newsgroups_entry;
  composer->scrolledwindow = scrolledwindow;
  composer->textview = textview;
  composer->textbuffer = textbuffer;
  composer->additional_header_shown = 0;
  composer->status_bar = status_bar;
  
  composer->composer = NULL;
  
  composer->error = NULL;
  composer->reference_msg = NULL;
  composer->compose_type = ETPAN_COMPOSE_TYPE_NEW;
  
  icon = etpan_icon_manager_get_pixbuf(etpan_icon_manager_get_default(),
      "etpanX");
  gtk_window_set_icon(GTK_WINDOW(window), icon);
  
  return composer;
}

static void clear_attach_list(struct etpan_message_composer_window * composer);

void etpan_message_composer_window_free(struct etpan_message_composer_window * composer_window)
{
  ETPAN_ERROR_FREE(composer_window->error);
  composer_window->error = NULL;
  
  clear_attach_list(composer_window);
  
  g_object_unref(composer_window->textbuffer);
  gtk_widget_destroy(composer_window->textview);
  
  gtk_widget_destroy(composer_window->newsgroups_entry);
  gtk_widget_destroy(composer_window->newsgroups_label);
  etpan_completion_free(composer_window->reply_to_completion);
  gtk_widget_destroy(composer_window->reply_to_label);
  etpan_completion_free(composer_window->bcc_completion);
  gtk_widget_destroy(composer_window->bcc_label);
  etpan_completion_free(composer_window->cc_completion);
  gtk_widget_destroy(composer_window->cc_label);
  gtk_widget_destroy(composer_window->subject_entry);
  gtk_widget_destroy(composer_window->subject_label);
  etpan_completion_free(composer_window->to_completion);
  gtk_widget_destroy(composer_window->to_label);
  
  gtk_widget_destroy(composer_window->table);
  etpan_toolbar_free(composer_window->toolbar);
  gtk_widget_destroy(composer_window->vbox);
  gtk_widget_destroy(composer_window->window);
  
  etpan_message_id_list_free(composer_window->references);
  carray_free(composer_window->attach_list);
  free(composer_window);
}

struct attach_item {
  GtkTextMark * start_mark;
  GtkTextMark * end_mark;
  char * filename;
  gint start_offset;
  gint end_offset;
  int temporary;
};

static void clear_attach_list(struct etpan_message_composer_window * composer)
{
  unsigned int i;
  int r;
  
  for(i = 0 ; i < carray_count(composer->attach_list) ; i ++) {
    struct attach_item * item;
    
    item = carray_get(composer->attach_list, i);
    
    if (item->temporary)
      unlink(item->filename);
    free(item->filename);
    gtk_text_buffer_delete_mark(composer->textbuffer, item->end_mark);
    gtk_text_buffer_delete_mark(composer->textbuffer, item->start_mark);
    free(item);
  }
  
  r = carray_set_size(composer->attach_list, 0);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

static void sort_attach(struct etpan_message_composer_window * composer);

static void add_attach(struct etpan_message_composer_window * composer,
    char * filename,
    GtkTextMark * start_mark, GtkTextMark * end_mark,
    int temporary)
{
  struct attach_item * item;
  int r;
  char * dup_filename;
  
  dup_filename = strdup(filename);
  if (dup_filename == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  item = malloc(sizeof(* item));
  if (item == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  item->start_mark = start_mark;
  item->end_mark = end_mark;
  item->filename = dup_filename;
  r = carray_add(composer->attach_list, item, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  item->temporary = temporary;
  
  sort_attach(composer);
}

static int comp_item(const void * a, const void * b)
{
  struct attach_item ** item_a;
  struct attach_item ** item_b;
  gint offset_a;
  gint offset_b;
  
  item_a = (struct attach_item **) a;
  item_b = (struct attach_item **) b;
  
  offset_a = (* item_b)->start_offset;
  offset_b = (* item_a)->start_offset;
  
  return offset_b - offset_a;
}

static void
compute_attach_offset(struct etpan_message_composer_window * composer)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(composer->attach_list) ; i ++) {
    struct attach_item * item;
    GtkTextIter iter;
    
    item = carray_get(composer->attach_list, i);
    
    gtk_text_buffer_get_iter_at_mark(composer->textbuffer,
        &iter, item->start_mark);
    item->start_offset = gtk_text_iter_get_offset(&iter);
    gtk_text_buffer_get_iter_at_mark(composer->textbuffer,
        &iter, item->end_mark);
    item->end_offset = gtk_text_iter_get_offset(&iter);
  }
}

static void sort_attach(struct etpan_message_composer_window * composer)
{
  compute_attach_offset(composer);
  
  qsort(carray_data(composer->attach_list),
      carray_count(composer->attach_list),
      sizeof(void *), comp_item);
}

static void delete_attach_text(struct etpan_message_composer_window * composer,
    struct attach_item * item)
{
  GtkTextIter start_iter;
  GtkTextIter end_iter;
  
  gtk_text_buffer_get_iter_at_mark(composer->textbuffer,
      &start_iter, item->start_mark);
  gtk_text_buffer_get_iter_at_mark(composer->textbuffer,
      &end_iter, item->end_mark);
  gtk_text_buffer_delete(composer->textbuffer, &start_iter, &end_iter);
}

static void delete_attach(struct etpan_message_composer_window * composer,
    struct attach_item * item)
{
  /* TODO : very inefficient */
  unsigned int i;
  
  delete_attach_text(composer, item);
  
  free(item->filename);
  gtk_text_buffer_delete_mark(composer->textbuffer, item->end_mark);
  gtk_text_buffer_delete_mark(composer->textbuffer, item->start_mark);
  free(item);
  
  for(i = 0 ; i < carray_count(composer->attach_list) ; i ++) {
    struct attach_item * cur_item;
    
    cur_item = carray_get(composer->attach_list, i);
    
    if (item == cur_item) {
      carray_delete_slow(composer->attach_list, i);
      break;
    }
  }
  
  compute_attach_offset(composer);
}

static void
delete_attach_in_bound(struct etpan_message_composer_window * composer,
    GtkTextIter * first, GtkTextIter * second)
{
  unsigned int i;
  gint start_offset;
  gint end_offset;
  GtkTextIter * start_iter;
  GtkTextIter * end_iter;
  
  if (gtk_text_iter_compare(first, second) <= 0) {
    start_iter = first;
    end_iter = second;
  }
  else {
    end_iter = first;
    start_iter = second;
  }
  
  start_offset = gtk_text_iter_get_offset(start_iter);
  end_offset = gtk_text_iter_get_offset(end_iter);
  compute_attach_offset(composer);
  
  i = 0;
  while (i < carray_count(composer->attach_list)) {
    struct attach_item * item;
    
    item = carray_get(composer->attach_list, i);
    
    if ((item->start_offset >= start_offset) &&
        (item->end_offset <= end_offset)) {
      delete_attach_text(composer, item);
      
      free(item->filename);
      gtk_text_buffer_delete_mark(composer->textbuffer, item->end_mark);
      gtk_text_buffer_delete_mark(composer->textbuffer, item->start_mark);
      free(item);
      
      carray_delete(composer->attach_list, i);
    }
    else {
      i ++;
    }
  }
}

enum {
  BOUND_TYPE_LEFT,
  BOUND_TYPE_RIGHT,
  BOUND_TYPE_LEFT_INCLUSIVE,
  BOUND_TYPE_RIGHT_INCLUSIVE,
  BOUND_TYPE_EXCLUSIVE,
  BOUND_TYPE_INCLUSIVE,
};

static struct attach_item *
get_attach(struct etpan_message_composer_window * composer,
    GtkTextIter * iter,
    int bound_type)
{
  int offset;
  unsigned int i;
  
  compute_attach_offset(composer);
  
  offset = gtk_text_iter_get_offset(iter);
  
  /* TODO : do a better lookup */
  for(i = 0 ; i < carray_count(composer->attach_list) ; i ++) {
    struct attach_item * item;
    
    item = carray_get(composer->attach_list, i);
    
    switch (bound_type) {
    case BOUND_TYPE_LEFT:
      if (offset == item->start_offset)
        return item;
      break;
    case BOUND_TYPE_RIGHT:
      if (offset == item->end_offset)
        return item;
      break;
    case BOUND_TYPE_LEFT_INCLUSIVE:
      if ((item->start_offset <= offset) && (offset < item->end_offset))
        return item;
      break;
    case BOUND_TYPE_RIGHT_INCLUSIVE:
      if ((item->start_offset < offset) && (offset <= item->end_offset))
        return item;
      break;
    case BOUND_TYPE_EXCLUSIVE:
      if ((item->start_offset < offset) && (offset < item->end_offset))
        return item;
      break;
    case BOUND_TYPE_INCLUSIVE:
      if ((item->start_offset <= offset) && (offset <= item->end_offset))
        return item;
      break;
    }
  }
  
  return NULL;
}

static gboolean delete_handler(GtkWidget *widget,
    GdkEvent *event,
    gpointer user_data)
{
  struct etpan_message_composer_window * composer;
  (void) widget;
  (void) event;
  
  composer = user_data;
  
  gtk_widget_hide(composer->window);
  etpan_message_composer_window_unsetup(composer);
  etpan_message_composer_window_free(composer);
  
  return TRUE;
}

#define WRAP_COUNT 72

static void add_text(struct etpan_message_composer * composer,
    char * text, size_t size)
{
  char * wrapped;
  size_t wrapped_size;
  
  etpan_wrap_text(text, size, &wrapped, &wrapped_size, WRAP_COUNT);
  etpan_message_composer_add_text(composer, wrapped, wrapped_size);
  
  free(wrapped);
}

#if 0
enum {
  OUTBOX_TYPE_NONE,
  OUTBOX_TYPE_SMTP,
  OUTBOX_TYPE_SENT_FOLDER,
  OUTBOX_TYPE_NEWS,
};

static int get_outbox_type(struct etpan_outbox * outbox)
{
  struct etpan_sender * sender;
  struct etpan_sender_driver * driver;
  
  sender = etpan_outbox_get_sender(outbox);
  if (sender == NULL)
    return OUTBOX_TYPE_NONE;
  
  driver = etpan_sender_get_driver(sender);
  if (strcasecmp(driver->name, "smtp") == 0)
    return OUTBOX_TYPE_SMTP;
  else if (strcasecmp(driver->name, "news") == 0)
    return OUTBOX_TYPE_NEWS;
  else if (strcasecmp(driver->name, "sent-folder") == 0)
    return OUTBOX_TYPE_SENT_FOLDER;
  else
    return OUTBOX_TYPE_NONE;
}
#endif

#if 0
static void message_sent(struct etpan_message_composer_window * composer_window);
static void outbox_added(struct etpan_outbox * outbox, void * user_data);
static void outbox_processed(struct etpan_outbox * outbox, void * user_data);
#endif

static void message_sent_callback(struct etpan_error * error, void * cb_data);


static void send_message(struct etpan_message_composer_window * composer_window)
{
  struct etpan_message_composer * composer;
  GtkTextIter start_iter;
  GtkTextIter end_iter;
  gint start_offset;
  gint end_offset;
#if 0
  char * result_content;
  size_t result_length;
#endif
  unsigned int i;
  gint previous_offset;
  int one_text_part;
  carray * list;
  gint selected_account_index;
  struct etpan_account * account;
  struct etpan_message_header * header;
  char * mail;
  char * display_name;
  struct etpan_address * from_elt;
  carray * from;
  const char * subject;
  const char * newsgroups;
#if 0
  int has_recipient;
  int has_newsgroups;
  carray * outbox_list;
#endif
  carray * references;
  
  ETPAN_LOG("sending message");
  
  list = etpan_account_manager_get_ordered_list(etpan_account_manager_get_default());
  selected_account_index = gtk_combo_box_get_active(GTK_COMBO_BOX(composer_window->account_combo));
  account = carray_get(list, selected_account_index);
  
  composer = etpan_message_composer_new();
  etpan_message_composer_set_account(composer, account);
  
  header = etpan_message_composer_get_header(composer);
  
  references = etpan_message_composer_window_get_references(composer_window);
  etpan_message_composer_set_references(composer, references);
  
  /* from */
  display_name = etpan_account_get_display_name(account);
  mail = etpan_account_get_mail(account);
  from_elt = etpan_address_new();
  etpan_address_set_display_name(from_elt, display_name);
  etpan_address_set_address(from_elt, mail);
  
  from = etpan_address_list_new();
  etpan_address_list_add_address(from, from_elt);
  etpan_message_header_set_from(header, from);
  etpan_address_list_free(from);
  
#if 0
  has_recipient = 0;
  has_newsgroups = 0;
#endif
  
  /* reply to */
  list = etpan_completion_get_address_list(composer_window->reply_to_completion);
  etpan_message_header_set_replyto(header, list);
  etpan_address_list_free(list);
  
  /* to */
  list = etpan_completion_get_address_list(composer_window->to_completion);
#if 0
  if (carray_count(list) > 0)
    has_recipient = 1;
#endif
  etpan_message_header_set_to(header, list);
  etpan_address_list_free(list);
  
  /* cc */
  list = etpan_completion_get_address_list(composer_window->cc_completion);
#if 0
  if (carray_count(list) > 0)
    has_recipient = 1;
#endif
  etpan_message_header_set_cc(header, list);
  etpan_address_list_free(list);
  
  /* bcc */
  list = etpan_completion_get_address_list(composer_window->bcc_completion);
#if 0
  if (carray_count(list) > 0)
    has_recipient = 1;
#endif
  etpan_message_header_set_bcc(header, list);
  etpan_address_list_free(list);
  
  /* subject */
  subject = gtk_entry_get_text(GTK_ENTRY(composer_window->subject_entry));
  etpan_message_header_set_subject(header, subject);
  
  /* newsgroups */
  newsgroups = gtk_entry_get_text(GTK_ENTRY(composer_window->newsgroups_entry));
  if (newsgroups[0] != '\0') {
    ETPAN_LOG("set newsgroups %s", newsgroups);
    etpan_message_header_set_newsgroups(header, newsgroups);
#if 0
    has_newsgroups = 1;
#endif
  }
  
  gtk_text_buffer_get_start_iter(composer_window->textbuffer, &start_iter);
  gtk_text_buffer_get_end_iter(composer_window->textbuffer, &end_iter);
  start_offset = gtk_text_iter_get_offset(&start_iter);
  end_offset = gtk_text_iter_get_offset(&end_iter);
  
  compute_attach_offset(composer_window);
  previous_offset = start_offset;
  
  one_text_part = 0;
  
  for(i = 0 ; i < carray_count(composer_window->attach_list) ; i ++) {
    GtkTextIter part_start_iter;
    GtkTextIter part_end_iter;
    gint part_start_offset;
    gint part_end_offset;
    struct attach_item * item;
    
    item = carray_get(composer_window->attach_list, i);
    
    part_start_offset = previous_offset;
    part_end_offset = item->start_offset;
    
    if (part_start_offset != part_end_offset) {
      gchar * part_text_content;
      gint part_text_length;
      
      gtk_text_buffer_get_iter_at_offset(composer_window->textbuffer,
          &part_start_iter, part_start_offset);
      gtk_text_buffer_get_iter_at_offset(composer_window->textbuffer,
          &part_end_iter, part_end_offset);
      
      part_text_content = gtk_text_buffer_get_text(composer_window->textbuffer,
          &part_start_iter, &part_end_iter, 0);
      
      /* (part_end_offset - part_start_offset) is the number of UTF-8 chars */
      part_text_length = strlen(part_text_content);
      
      add_text(composer, part_text_content, part_text_length);
      ETPAN_LOG("add text part %i", part_text_length);
      
      g_free(part_text_content);
    
      one_text_part = 1;
    }
    
    etpan_message_composer_add_file(composer, item->filename, 0);
    ETPAN_LOG("add attachment %s", item->filename);
    
    previous_offset = item->end_offset;
  }
  
  if (end_offset == previous_offset) {
    if ((carray_count(composer_window->attach_list) >= 2) || one_text_part) {
      /* do nothing */
    }
    else {
      /* add empty text part */
      add_text(composer, "", 0);
      ETPAN_LOG("add empty text part");
    }
  }
  else {
    GtkTextIter part_start_iter;
    GtkTextIter part_end_iter;
    gint part_start_offset;
    gint part_end_offset;
    gchar * part_text_content;
    size_t part_text_length;
    
    part_start_offset = previous_offset;
    part_end_offset = end_offset;
    
    gtk_text_buffer_get_iter_at_offset(composer_window->textbuffer,
        &part_start_iter, part_start_offset);
    gtk_text_buffer_get_iter_at_offset(composer_window->textbuffer,
        &part_end_iter, part_end_offset);
    
    part_text_content = gtk_text_buffer_get_text(composer_window->textbuffer,
        &part_start_iter, &part_end_iter, 0);
    
    /* (part_end_offset - part_start_offset) is the number of UTF-8 chars */
    part_text_length = strlen(part_text_content);
    
    add_text(composer, part_text_content, part_text_length);
    
    ETPAN_LOG("add remaining text part %i", part_text_length);
    g_free(part_text_content);
  }
  
#if 0
  etpan_message_composer_render(composer, &result_content, &result_length);
  
  outbox_list = etpan_account_get_outbox_list(account);
  composer_window->outbox_count = 0;
  for(i = 0 ; i < carray_count(outbox_list) ; i ++) {
    struct etpan_outbox * outbox;
    int type;
    
    outbox = carray_get(outbox_list, i);
    type = get_outbox_type(outbox);
    
    switch (type) {
    case OUTBOX_TYPE_NEWS:
      if (has_newsgroups) {
        composer_window->outbox_count ++;
        etpan_outbox_add(outbox, result_content, result_length,
            outbox_added, composer_window);
      }
      break;
    case OUTBOX_TYPE_SMTP:
      if (has_recipient) {
        composer_window->outbox_count ++;
        etpan_outbox_add(outbox, result_content, result_length,
            outbox_added, composer_window);
      }
      break;
    case OUTBOX_TYPE_SENT_FOLDER:
      composer_window->outbox_count ++;
      etpan_outbox_add(outbox, result_content, result_length,
          outbox_added, composer_window);
      break;
    }
  }
  
  if (has_newsgroups || has_recipient) {
    etpan_message_composer_set_reference_message(composer,
        composer_window->reference_msg, composer_window->compose_type);
    etpan_message_composer_mark_done(composer);
  }
  
  free(result_content);
#endif
  
  etpan_message_composer_set_reference_message(composer,
      composer_window->reference_msg, composer_window->compose_type);
  etpan_message_composer_send(composer,
      message_sent_callback, composer_window);
  
  etpan_message_composer_free(composer);
  
  gtk_widget_hide(composer_window->window);
}

static void message_sent_callback(struct etpan_error * error, void * cb_data)
{
  struct etpan_message_composer_window * composer_window;
  
  composer_window = cb_data;
  
  if (error != NULL) {
    etpan_error_log(error);
    composer_window->error = etpan_error_dup(error);
  }
  
  etpan_signal_send(etpan_signal_manager_get_default(),
    ETPAN_MESSAGE_COMPOSER_WINDOW_FINISHED_SIGNAL, composer_window,
      NULL);
}

#if 0
static void outbox_added(struct etpan_outbox * outbox, void * user_data)
{
  struct etpan_message_composer_window * composer_window;
  struct etpan_error * error;
  int r;
  
  composer_window = user_data;
  
  error = etpan_outbox_added_get_error(outbox);
  if (error != NULL) {
    error = etpan_error_dup(error);
    r = carray_add(composer_window->error_list, error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    composer_window->outbox_count --;
    message_sent(composer_window);
    return;
  }
  
  etpan_outbox_process(outbox, outbox_processed, user_data);
}

static void outbox_processed(struct etpan_outbox * outbox, void * user_data)
{
  struct etpan_message_composer_window * composer_window;
  struct etpan_error * error;
  int r;
  
  composer_window = user_data;
  
  error = etpan_outbox_processed_get_error(outbox);
  if (error != NULL) {
    error = etpan_error_dup(error);
    r = carray_add(composer_window->error_list, error, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    composer_window->outbox_count --;
    message_sent(composer_window);
    return;
  }
  
  composer_window->outbox_count --;
  message_sent(composer_window);
}

static void message_sent(struct etpan_message_composer_window * composer_window)
{
  struct etpan_error * error;
  int r;
  unsigned int i;
  
  if (composer_window->outbox_count != 0)
    return;
  
  if (carray_count(composer_window->error_list) == 0) {
    error = NULL;
  }
  else if (carray_count(composer_window->error_list) == 1) {
    error = carray_get(composer_window->error_list, 0);
  }
  else {
    error = etpan_error_multiple();
    for(i = 0 ; i < carray_count(composer_window->error_list) ; i ++) {
      struct etpan_error * suberror;
      
      suberror = carray_get(composer_window->error_list, i);
      etpan_error_add_child(error, suberror);
    }
  }
  
  r = carray_set_size(composer_window->error_list, 0);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  composer_window->error = error;
  
  if (error != NULL)
    etpan_error_log(error);
  
  etpan_signal_send(etpan_signal_manager_get_default(),
    ETPAN_MESSAGE_COMPOSER_WINDOW_FINISHED_SIGNAL, composer_window,
      NULL);
  
#if 0
  /* close ourselves */
  etpan_message_composer_window_unsetup(composer_window);
  etpan_message_composer_window_free(composer_window);
#endif
}
#endif

static void add_attachment(struct etpan_message_composer_window * composer,
    char * filename, int temporary)
{
  GtkTextIter start_iter;
  GtkTextIter end_iter;
  GtkTextIter iter;
  GdkPixbuf * pixbuf;
  GtkTextMark * start_mark;
  GtkTextMark * end_mark;
  struct attach_item * item;
  char basename_buffer[PATH_MAX];
  char * base;
  GError * error;
  
  strncpy(basename_buffer, filename, sizeof(basename_buffer));
  base = basename(basename_buffer);
  
  ETPAN_LOG("add attachment");
  
  gtk_text_buffer_get_iter_at_mark(composer->textbuffer,
      &iter, gtk_text_buffer_get_insert(composer->textbuffer));
  
  item = get_attach(composer, &iter, BOUND_TYPE_EXCLUSIVE);
  if (item != NULL)
    return;
  
  gtk_text_buffer_get_iter_at_mark(composer->textbuffer, &iter,
      gtk_text_buffer_get_insert(composer->textbuffer));
  
  start_mark = gtk_text_buffer_create_mark(composer->textbuffer,
      NULL, &iter, TRUE);
  
  error = NULL;
  pixbuf = gdk_pixbuf_new_from_file(filename, &error);
  if (pixbuf != NULL) {
    /* do nothing */
  }
  else {
    GtkIconSet * iconset;
    GtkStyle * style;
    
    g_error_free(error);
    /* insert icon */
    iconset = gtk_icon_factory_lookup_default("gtk-file");
    style = gtk_widget_get_style(composer->textview);
    pixbuf = gtk_icon_set_render_icon(iconset, style, GTK_TEXT_DIR_NONE,
        GTK_STATE_ACTIVE, GTK_ICON_SIZE_LARGE_TOOLBAR,
        composer->textview, NULL);
  }
  
  gtk_text_buffer_insert_pixbuf(composer->textbuffer, &iter, pixbuf);
  g_object_unref(pixbuf);
  gtk_text_buffer_insert(composer->textbuffer, &iter, "\n", -1);
  
  gtk_text_buffer_insert(composer->textbuffer, &iter, "[", -1);
  gtk_text_buffer_insert(composer->textbuffer, &iter, base, -1);
  gtk_text_buffer_insert(composer->textbuffer, &iter, "]", -1);
  
  gtk_text_buffer_get_iter_at_mark(composer->textbuffer, &iter,
      gtk_text_buffer_get_insert(composer->textbuffer));
  
  end_mark = gtk_text_buffer_create_mark(composer->textbuffer,
      NULL, &iter, TRUE);
  
  gtk_text_buffer_get_iter_at_mark(composer->textbuffer, &start_iter,
      start_mark);
  gtk_text_buffer_get_iter_at_mark(composer->textbuffer, &end_iter,
      end_mark);
  gtk_text_buffer_apply_tag_by_name(composer->textbuffer, "attach",
      &start_iter, &end_iter);

  gtk_text_buffer_insert(composer->textbuffer, &iter, "\n", -1);
  
  add_attach(composer, filename, start_mark, end_mark, temporary);
}

static void dialog_attach_filename(struct etpan_message_composer_window * composer_window)
{
  GtkWidget * dialog;
  int r;
  
  dialog = gtk_file_chooser_dialog_new(_("Select the file to attach."),
      GTK_WINDOW(composer_window->window),
      GTK_FILE_CHOOSER_ACTION_OPEN,
      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
      GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
      NULL);
  
  r = gtk_dialog_run(GTK_DIALOG(dialog));
  if (r == GTK_RESPONSE_ACCEPT) {
    char * filename;
    
    filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog));
    
    add_attachment(composer_window, filename, 0);
    
    g_free(filename);
  }
  
  gtk_widget_destroy(dialog);
}

/* extracted from etpan-preferences-account.c - begin */
enum {
  STORAGE_DRIVER_TYPE_POP,
  STORAGE_DRIVER_TYPE_IMAP,
  STORAGE_DRIVER_TYPE_NEWS,
  STORAGE_DRIVER_TYPE_MAILDIR,
  STORAGE_DRIVER_TYPE_MH,
  STORAGE_DRIVER_TYPE_MBOX,
};

static int get_account_type(struct etpan_account * account)
{
  struct etpan_storage * storage;
  struct etpan_storage_driver * driver;
  int type;
  
  storage = etpan_account_get_storage(account);
  if (storage == NULL) {
    ETPAN_LOG("account has no storage");
    etpan_crash();
  }
  
  driver = etpan_storage_get_driver(storage);
  type = STORAGE_DRIVER_TYPE_POP;
  if (strcasecmp(driver->name, "pop-sync") == 0)
    type = STORAGE_DRIVER_TYPE_POP;
  else if (strcasecmp(driver->name, "imap-sync") == 0)
    type = STORAGE_DRIVER_TYPE_IMAP;
  else if (strcasecmp(driver->name, "news") == 0)
    type = STORAGE_DRIVER_TYPE_NEWS;
  else if (strcasecmp(driver->name, "maildir") == 0)
    type = STORAGE_DRIVER_TYPE_MAILDIR;
  else if (strcasecmp(driver->name, "mh") == 0)
    type = STORAGE_DRIVER_TYPE_MH;
  else if (strcasecmp(driver->name, "mbox") == 0)
    type = STORAGE_DRIVER_TYPE_MBOX;
  
  return type;
}
/* extracted from etpan-preferences-account.c - end */

static void additional_headers_set(struct etpan_message_composer_window * composer)
{
  carray * list;
  gint selected_account_index;
  struct etpan_account * account;
  int type;
  
  list = etpan_account_manager_get_ordered_list(etpan_account_manager_get_default());
  selected_account_index = gtk_combo_box_get_active(GTK_COMBO_BOX(composer->account_combo));
  if (selected_account_index == -1)
    return;
  
  account = carray_get(list, selected_account_index);
  type = get_account_type(account);
  
  switch (type) {
  case STORAGE_DRIVER_TYPE_NEWS:
    {
      GtkWidget * entry;
      
      gtk_widget_show(composer->newsgroups_label);
      gtk_widget_show(composer->newsgroups_entry);
      
      if (composer->additional_header_shown) {
        gtk_widget_show(composer->to_label);
        entry = etpan_completion_get_main_widget(composer->to_completion);
        gtk_widget_show(entry);
        
        gtk_widget_show(composer->cc_label);
        entry = etpan_completion_get_main_widget(composer->cc_completion);
        gtk_widget_show(entry);
        
        gtk_widget_show(composer->bcc_label);
        entry = etpan_completion_get_main_widget(composer->bcc_completion);
        gtk_widget_show(entry);
    
        gtk_widget_show(composer->reply_to_label);
        entry = etpan_completion_get_main_widget(composer->reply_to_completion);
        gtk_widget_show(entry);
      }
      else {
        gtk_widget_hide(composer->to_label);
        entry = etpan_completion_get_main_widget(composer->to_completion);
        gtk_widget_hide(entry);
        
        gtk_widget_hide(composer->cc_label);
        entry = etpan_completion_get_main_widget(composer->cc_completion);
        gtk_widget_hide(entry);
        
        gtk_widget_hide(composer->bcc_label);
        entry = etpan_completion_get_main_widget(composer->bcc_completion);
        gtk_widget_hide(entry);
    
        gtk_widget_hide(composer->reply_to_label);
        entry = etpan_completion_get_main_widget(composer->reply_to_completion);
        gtk_widget_hide(entry);
      }
    }
    break;
  default:
    {
      GtkWidget * entry;
      
      gtk_entry_set_text(GTK_ENTRY(composer->newsgroups_entry), "");
      gtk_widget_hide(composer->newsgroups_label);
      gtk_widget_hide(composer->newsgroups_entry);
      
      gtk_widget_show(composer->to_label);
      entry = etpan_completion_get_main_widget(composer->to_completion);
      gtk_widget_show(entry);
      
      gtk_widget_show(composer->cc_label);
      entry = etpan_completion_get_main_widget(composer->cc_completion);
      gtk_widget_show(entry);
      
      if (composer->additional_header_shown) {
        gtk_widget_show(composer->bcc_label);
        entry = etpan_completion_get_main_widget(composer->bcc_completion);
        gtk_widget_show(entry);
    
        gtk_widget_show(composer->reply_to_label);
        entry = etpan_completion_get_main_widget(composer->reply_to_completion);
        gtk_widget_show(entry);
      }
      else {
        gtk_widget_hide(composer->bcc_label);
        entry = etpan_completion_get_main_widget(composer->bcc_completion);
        gtk_widget_hide(entry);
    
        gtk_widget_hide(composer->reply_to_label);
        entry = etpan_completion_get_main_widget(composer->reply_to_completion);
        gtk_widget_hide(entry);
      }
    }
    break;
  }
}

static void button_clicked(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  int button_index;
  struct etpan_message_composer_window * composer;
  (void) signal_name;
  (void) sender;
  
  composer = user_data;
  
  button_index = * (int *) signal_data;
  
  switch (button_index) {
  case 0:
    send_message(composer);
    break;
  case 1:
    dialog_attach_filename(composer);
    break;
  case 3:
    composer->additional_header_shown = !composer->additional_header_shown;
    additional_headers_set(composer);
    break;
  }
}

static void insert_new_line_before(struct etpan_message_composer_window * composer, struct attach_item * item)
{
  GtkTextIter iter;
  GtkTextMark * mark;
  GtkTextIter start_iter;
  GtkTextIter end_iter;
  
  gtk_text_buffer_get_iter_at_mark(composer->textbuffer, &iter,
      gtk_text_buffer_get_insert(composer->textbuffer));
  
  mark = gtk_text_buffer_create_mark(composer->textbuffer, NULL,
      &iter, TRUE);
  
  gtk_text_buffer_insert(composer->textbuffer, &iter,
      "\n", 1);
  
  gtk_text_buffer_get_iter_at_mark(composer->textbuffer, &start_iter,
      mark);
  gtk_text_buffer_get_iter_at_mark(composer->textbuffer, &end_iter,
      gtk_text_buffer_get_insert(composer->textbuffer));
  
  gtk_text_buffer_remove_tag_by_name(composer->textbuffer, "attach",
      &start_iter, &end_iter);
  
  gtk_text_buffer_move_mark(composer->textbuffer,
      item->start_mark, &end_iter);
  
  gtk_text_buffer_delete_mark(composer->textbuffer, mark);
}

static int has_selection(struct etpan_message_composer_window * composer)
{
  GtkTextIter start;
  GtkTextIter end;
  
  if (gtk_text_buffer_get_selection_bounds(composer->textbuffer, &start, &end))
    return 1;
  else
    return 0;
}

static void delete_selection(struct etpan_message_composer_window * composer)
{
  GtkTextMark * insert_mark;
  GtkTextMark * selection_bound_mark;
  GtkTextIter insert_iter;
  GtkTextIter selection_bound_iter;

  insert_mark =
    gtk_text_buffer_get_insert(composer->textbuffer);
  selection_bound_mark =
    gtk_text_buffer_get_selection_bound(composer->textbuffer);
  
  gtk_text_buffer_get_iter_at_mark(composer->textbuffer,
      &insert_iter, insert_mark);
  gtk_text_buffer_get_iter_at_mark(composer->textbuffer,
      &selection_bound_iter, selection_bound_mark);
  
  delete_attach_in_bound(composer, &insert_iter, &selection_bound_iter);

  gtk_text_buffer_get_iter_at_mark(composer->textbuffer,
      &insert_iter, insert_mark);
  gtk_text_buffer_get_iter_at_mark(composer->textbuffer,
      &selection_bound_iter, selection_bound_mark);
  
  gtk_text_buffer_delete(composer->textbuffer,
      &insert_iter, &selection_bound_iter);
}

/* fix selection of attachments */
static void mark_set_handler(GtkTextBuffer * textbuffer,
    GtkTextIter * iter,
    GtkTextMark * mark,
    gpointer user_data)
{
  struct etpan_message_composer_window * composer;
  GtkTextMark * insert_mark;
  GtkTextMark * selection_bound_mark;
  GtkTextIter insert_iter;
  GtkTextIter selection_bound_iter;
  struct attach_item * item;
  GtkTextMark * lower_mark;
  GtkTextMark * upper_mark;
  GtkTextIter upper_iter;
  GtkTextIter lower_iter;
  gint comp_value;
  (void) textbuffer;
  (void) iter;
  
  composer = user_data;
  
  if (composer->moving_mark)
    return;
  
  insert_mark =
    gtk_text_buffer_get_insert(composer->textbuffer);
  selection_bound_mark =
    gtk_text_buffer_get_selection_bound(composer->textbuffer);
  
  if ((mark != insert_mark) && (mark != selection_bound_mark))
    return;
  
  gtk_text_buffer_get_iter_at_mark(composer->textbuffer,
      &insert_iter, insert_mark);
  gtk_text_buffer_get_iter_at_mark(composer->textbuffer,
      &selection_bound_iter, selection_bound_mark);
  
  comp_value = gtk_text_iter_compare(&insert_iter, &selection_bound_iter);
  if (comp_value == 0) {
    return;
  }
  else if (comp_value < 0) {
    lower_mark = insert_mark;
    upper_mark = selection_bound_mark;
  }
  else {
    upper_mark = insert_mark;
    lower_mark = selection_bound_mark;
  }
  
  gtk_text_buffer_get_iter_at_mark(composer->textbuffer,
      &lower_iter, lower_mark);
  gtk_text_buffer_get_iter_at_mark(composer->textbuffer,
      &upper_iter, upper_mark);
  
  composer->moving_mark = 1;
  
  item = get_attach(composer, &lower_iter, BOUND_TYPE_EXCLUSIVE);
  if (item != NULL) {
    GtkTextIter iter;
    
    gtk_text_buffer_get_iter_at_mark(composer->textbuffer,
        &iter, item->start_mark);
    gtk_text_buffer_move_mark(composer->textbuffer, lower_mark,
        &iter);
  }
  
  item = get_attach(composer, &upper_iter, BOUND_TYPE_EXCLUSIVE);
  if (item != NULL) {
    GtkTextIter iter;
    
    gtk_text_buffer_get_iter_at_mark(composer->textbuffer,
        &iter, item->end_mark);
    gtk_text_buffer_move_mark(composer->textbuffer, upper_mark,
        &iter);
  }
  
  composer->moving_mark = 0;
}

static gboolean key_handler(GtkWidget * widget,
    GdkEventKey * event,
    gpointer user_data)
{
  unsigned int state;
  int alt;
  int ctrl;
  struct etpan_message_composer_window * composer;
  (void) widget;
  
  composer = user_data;
  
  state = event->state & (GDK_SHIFT_MASK | GDK_CONTROL_MASK | GDK_MOD1_MASK |
      GDK_MOD2_MASK | GDK_MOD3_MASK | GDK_MOD4_MASK |
      GDK_MOD5_MASK);
  
  alt = (state & GDK_MOD1_MASK) != 0;
  ctrl = (state & GDK_CONTROL_MASK) != 0;

  if ((event->keyval == GDK_Return) && ((!alt) && (!ctrl))) {
    struct attach_item * item;
    GtkTextIter iter;
    
    gtk_text_buffer_get_iter_at_mark(composer->textbuffer,
        &iter, gtk_text_buffer_get_insert(composer->textbuffer));
    
    item = get_attach(composer, &iter, BOUND_TYPE_LEFT);
    if (item != NULL) {
      insert_new_line_before(composer, item);
      return TRUE;
    }
  }
  
  if ((event->keyval == GDK_BackSpace) && ((!alt) && (!ctrl))) {
    struct attach_item * item;
    GtkTextIter iter;
    
    if (has_selection(composer)) {
      delete_selection(composer);
      return TRUE;
    }
    
    gtk_text_buffer_get_iter_at_mark(composer->textbuffer,
        &iter, gtk_text_buffer_get_insert(composer->textbuffer));
    
    item = get_attach(composer, &iter, BOUND_TYPE_RIGHT_INCLUSIVE);
    if (item != NULL) {
      delete_attach(composer, item);
      return TRUE;
    }
  }

  if ((event->keyval == GDK_Delete) && ((!alt) && (!ctrl))) {
    struct attach_item * item;
    GtkTextIter iter;
    
    if (has_selection(composer)) {
      delete_selection(composer);
      return TRUE;
    }
    
    gtk_text_buffer_get_iter_at_mark(composer->textbuffer,
        &iter, gtk_text_buffer_get_insert(composer->textbuffer));
    
    item = get_attach(composer, &iter, BOUND_TYPE_LEFT_INCLUSIVE);
    if (item != NULL) {
      delete_attach(composer, item);
      return TRUE;
    }
  }
  
  return FALSE;
}

static void update_font(struct etpan_message_composer_window * composer_window)
{
  GtkTextTagTable * table;
  PangoFontDescription * font_desc;
  GtkTextTag * tag;
  
  font_desc = pango_font_description_from_string(etpan_ui_config_get_font_message(etpan_ui_config_get_default()));
  gtk_widget_modify_font(composer_window->textview, font_desc);
  pango_font_description_free(font_desc);
  
  table = gtk_text_buffer_get_tag_table(composer_window->textbuffer);
  
  tag = gtk_text_tag_table_lookup(table, "attach");
  g_object_set(tag,
      "font", etpan_ui_config_get_font_other(etpan_ui_config_get_default()),
      "editable", FALSE,
      NULL);
}

static void font_changed_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_message_composer_window * composer_window;
  (void) signal_name;
  (void) signal_data;
  (void) sender;
  
  composer_window = user_data;
  update_font(composer_window);
}

static void update_account(struct etpan_message_composer_window * composer_window);

static void account_modified_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_message_composer_window * composer_window;
  (void) signal_name;
  (void) signal_data;
  (void) sender;
  
  composer_window = user_data;
  update_account(composer_window);
}

static void account_set_handler(GtkComboBox * widget, gpointer user_data)
{
  struct etpan_message_composer_window * composer_window;
  (void) widget;
  
  composer_window = user_data;
  additional_headers_set(composer_window);
}

void etpan_message_composer_window_setup(struct etpan_message_composer_window * composer_window)
{
  GtkWidget * tmp_image;
  GtkIconSize tmp_toolbar_icon_size;
  GtkWidget * toolbar;
  
  composer_window->moving_mark = 0;
  
  toolbar = etpan_toolbar_get_main_widget(composer_window->toolbar);
  etpan_toolbar_clear(composer_window->toolbar);
  
  tmp_toolbar_icon_size = gtk_toolbar_get_icon_size(GTK_TOOLBAR(toolbar));
  
#define ICON(name) \
  etpan_icon_manager_new_scaled_image(etpan_icon_manager_get_default(), \
      name, 32)
  
  /* send button */
  tmp_image = ICON("send");
  gtk_widget_show(tmp_image);
  etpan_toolbar_add_item(composer_window->toolbar, _("Send"), tmp_image);
  /* attach button */
  tmp_image = ICON("attach");
  gtk_widget_show(tmp_image);
  etpan_toolbar_add_item(composer_window->toolbar, _("Attach"), tmp_image);
  /* separator */
  etpan_toolbar_add_separator(composer_window->toolbar);
  /* headers button */
  tmp_image = ICON("additional-fields");
  gtk_widget_show(tmp_image);
  etpan_toolbar_add_item(composer_window->toolbar, _("Fields"), tmp_image);
  
#undef ICON
  
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_TOOLBAR_BUTTONCLICKED_SIGNAL, composer_window->toolbar,
      composer_window, button_clicked);
  
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_UI_CONFIG_FONT_CHANGED_SIGNAL,
      etpan_account_manager_get_default(), composer_window,
      font_changed_handler);

  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_ACCOUNT_MANAGER_MODIFICATION_SIGNAL,
      etpan_account_manager_get_default(), composer_window,
      account_modified_handler);
  
  update_font(composer_window);
  update_account(composer_window);
  
  composer_window->delete_signal_id = g_signal_connect(composer_window->window,
        "delete-event", G_CALLBACK(delete_handler),
      (gpointer) composer_window);
  
  composer_window->key_press_signal_id =
    g_signal_connect(composer_window->textview, "key-press-event",
      G_CALLBACK(key_handler), (gpointer) composer_window);

  composer_window->mark_set_signal_id =
    g_signal_connect(composer_window->textbuffer, "mark-set",
      G_CALLBACK(mark_set_handler), (gpointer) composer_window);
  
  composer_window->account_changed_signal_id =
    g_signal_connect(composer_window->account_combo, "changed",
      G_CALLBACK(account_set_handler), (gpointer) composer_window);
  
  etpan_ui_window_set(etpan_ui_config_get_default(), "composer",
      composer_window->window);
  
  etpan_completion_setup(composer_window->to_completion);
  etpan_completion_setup(composer_window->cc_completion);
  etpan_completion_setup(composer_window->bcc_completion);
  etpan_completion_setup(composer_window->reply_to_completion);
  etpan_status_bar_setup(composer_window->status_bar);
  
  additional_headers_set(composer_window);
}

void etpan_message_composer_window_unsetup(struct etpan_message_composer_window * composer_window)
{
  etpan_ui_set_from_window(etpan_ui_config_get_default(), "composer",
      composer_window->window);
  
  set_reference_message(composer_window, NULL);
  
  etpan_status_bar_unsetup(composer_window->status_bar);
  etpan_completion_unsetup(composer_window->reply_to_completion);
  etpan_completion_unsetup(composer_window->bcc_completion);
  etpan_completion_unsetup(composer_window->cc_completion);
  etpan_completion_unsetup(composer_window->to_completion);
  
  if (composer_window->composer != NULL)
    etpan_message_composer_window_reply_cancel(composer_window);
  
  g_signal_handler_disconnect(composer_window->account_combo,
      composer_window->account_changed_signal_id);
  
  g_signal_handler_disconnect(composer_window->textbuffer,
      composer_window->mark_set_signal_id);
  
  g_signal_handler_disconnect(composer_window->textview,
      composer_window->key_press_signal_id);
  
  g_signal_handler_disconnect(composer_window->window,
      composer_window->delete_signal_id);

  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_ACCOUNT_MANAGER_MODIFICATION_SIGNAL,
      etpan_account_manager_get_default(), composer_window,
      account_modified_handler);
  
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_UI_CONFIG_FONT_CHANGED_SIGNAL,
      etpan_ui_config_get_default(), composer_window,
      font_changed_handler);
  
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_TOOLBAR_BUTTONCLICKED_SIGNAL, composer_window->toolbar,
      composer_window, button_clicked);
  
  etpan_toolbar_clear(composer_window->toolbar);
}

GtkWidget *
etpan_message_composer_window_get_main_widget(struct etpan_message_composer_window *
    composer_window)
{
  return composer_window->window;
}

void etpan_message_composer_window_set_references(struct etpan_message_composer_window * composer_window, carray * references)
{
  carray * dup_ref;
  
  dup_ref = etpan_message_id_list_dup(references);
  etpan_message_id_list_free(composer_window->references);
  composer_window->references = dup_ref;
}

carray * etpan_message_composer_window_get_references(struct etpan_message_composer_window * composer_window)
{
  return composer_window->references;
}

static void address_str(char * str, size_t len,
    struct etpan_address * addr)
{
  if (etpan_address_get_display_name(addr) != NULL) {
    if (etpan_address_get_address(addr) != NULL) {
      snprintf(str, len, "%s <%s>",
          etpan_address_get_display_name(addr),
          etpan_address_get_address(addr));
    }
    else {
      snprintf(str, len, "%s <>", etpan_address_get_display_name(addr));
    }
  }
  else {
    if (etpan_address_get_address(addr) != NULL) {
      snprintf(str, len, "<%s>", etpan_address_get_address(addr));
    }
    else {
      snprintf(str, len, "<>");
    }
  }
}

void etpan_message_composer_window_set_composer(struct etpan_message_composer_window * composer_window, struct etpan_message_composer * composer)
{
  carray * part_list;
  unsigned int i;
  GtkTextIter iter;
  GtkTextMark * mark;
  int cursor;
  struct etpan_completion * completion;
  GtkWidget * entry;
  char * subject;
  struct etpan_message_header * header;
  carray * addr_list;
  GtkWidget * textview;
  GtkTextBuffer * buffer;
  GtkTextIter iter_begin;
  GtkTextIter iter_end;
  carray * references;
  
  header = etpan_message_composer_get_header(composer);
  
  references = etpan_message_header_get_references(header);
  etpan_message_composer_window_set_references(composer_window, references);
  
  /* fill to */
  addr_list = etpan_message_header_get_to(header);
  completion = composer_window->to_completion;
  textview = etpan_completion_get_textview(completion);
  buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(textview));
  gtk_text_buffer_get_start_iter(buffer, &iter_begin);
  gtk_text_buffer_get_end_iter(buffer, &iter_end);
  gtk_text_buffer_delete(completion->buffer, &iter_begin, &iter_end);
  for(i = 0 ; i < carray_count(addr_list) ; i ++) {
    struct etpan_address * addr;
    char addr_str[1024];
    
    addr = carray_get(addr_list, i);
    address_str(addr_str, sizeof(addr_str), addr);
    gtk_text_buffer_insert_at_cursor(buffer, addr_str, strlen(addr_str));
    gtk_text_buffer_insert_at_cursor(buffer, ", ", 2);
  }
  
  /* fill cc */
  addr_list = etpan_message_header_get_cc(header);
  completion = composer_window->cc_completion;
  textview = etpan_completion_get_textview(completion);
  buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(textview));
  gtk_text_buffer_get_start_iter(buffer, &iter_begin);
  gtk_text_buffer_get_end_iter(buffer, &iter_end);
  gtk_text_buffer_delete(completion->buffer, &iter_begin, &iter_end);
  for(i = 0 ; i < carray_count(addr_list) ; i ++) {
    struct etpan_address * addr;
    char addr_str[1024];
    
    addr = carray_get(addr_list, i);
    address_str(addr_str, sizeof(addr_str), addr);
    gtk_text_buffer_insert_at_cursor(buffer, addr_str, strlen(addr_str));
    gtk_text_buffer_insert_at_cursor(buffer, ", ", 2);
  }
  
  /* fill subject */
  subject = etpan_message_header_get_subject(header);
  entry = composer_window->subject_entry;
  if (subject != NULL)
    gtk_entry_set_text(GTK_ENTRY(entry), subject);
  
  gtk_text_buffer_get_iter_at_mark(composer_window->textbuffer, &iter,
      gtk_text_buffer_get_insert(composer_window->textbuffer));
  cursor = etpan_message_composer_get_cursor(composer);
  mark = NULL;
  
  part_list = etpan_message_composer_get_part_list(composer);
  for(i = 0 ; i < carray_count(part_list) ; i ++) {
    struct etpan_attach_item * item;
    int type;
    
    if (cursor == (int) i)
      mark = gtk_text_buffer_create_mark(composer_window->textbuffer,
          NULL, &iter, 1);
    
    item = carray_get(part_list, i);
    type = etpan_attach_item_get_type(item);
    
    if (type == ETPAN_ATTACH_TYPE_TEXT) {
      char * content;
      size_t content_length;
      char * unwrapped;
      size_t unwrapped_size;
      
      etpan_attach_item_get_text_content(item, &content, &content_length);
      etpan_unwrap_text(content, content_length,
          &unwrapped, &unwrapped_size);
      
      gtk_text_buffer_insert(composer_window->textbuffer,
          &iter, unwrapped, -1);
      
      free(unwrapped);
    }
    else if (type == ETPAN_ATTACH_TYPE_FILE) {
      int temporary;
      char * filename;
      
      temporary = etpan_attach_item_file_is_temporary(item);
      etpan_attach_item_file_set_temporary(item, 0);
      filename = etpan_attach_item_file_get_name(item);
      add_attachment(composer_window, filename, temporary);
    }
  }
  
  if (mark != NULL) {
    gtk_text_buffer_get_iter_at_mark(composer_window->textbuffer, &iter, mark);
    gtk_text_buffer_place_cursor(composer_window->textbuffer, &iter);
    
    gtk_text_buffer_delete_mark(composer_window->textbuffer, mark);
  }
}

/* reply */

static void reply_callback(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_message_composer_window * composer_window;
  struct etpan_message_composer * composer;
  struct etpan_error * error;
  (void) signal_name;
  (void) signal_data;
  (void) sender;
  
  composer_window = user_data;
  composer = composer_window->composer;
  etpan_status_bar_stop_animation(composer_window->status_bar);
  
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_MESSAGE_COMPOSER_READY_SIGNAL, composer, composer_window,
      reply_callback);
  
  error = etpan_message_composer_reply_get_error(composer);
  if (error != NULL) {
    etpan_error_log(error);
    goto free_composer;
  }
  
  etpan_message_composer_window_set_composer(composer_window, composer);
  
 free_composer:
  etpan_message_composer_free(composer);
  composer_window->composer = NULL;

  set_reference_message(composer_window, NULL);
}

void etpan_message_composer_window_reply(struct etpan_message_composer_window * composer_window, struct etpan_message * msg)
{
  struct etpan_message_composer * composer;
  struct etpan_folder * folder;
  struct etpan_storage *storage;
  struct etpan_account * account;
  
  composer_window->compose_type = ETPAN_COMPOSE_TYPE_REPLY;
  set_reference_message(composer_window, msg);
  
  composer = etpan_message_composer_new();
  folder = etpan_message_get_folder(msg);
  storage = etpan_folder_get_storage(folder);
  account = etpan_storage_get_account(storage);
  etpan_message_composer_set_account(composer, account);
  
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_MESSAGE_COMPOSER_READY_SIGNAL, composer, composer_window,
      reply_callback);
  
  composer_window->composer = composer;
  etpan_message_composer_reply(composer, msg, ETPAN_REPLY_TYPE_AUTO);
  
  etpan_status_bar_start_animation(composer_window->status_bar);
}

void etpan_message_composer_window_reply_cancel(struct etpan_message_composer_window * composer_window)
{
  etpan_message_composer_reply_cancel(composer_window->composer);
  set_reference_message(composer_window, NULL);
}




/* forward */

static void forward_callback(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_message_composer_window * composer_window;
  struct etpan_message_composer * composer;
  struct etpan_error * error;
  (void) signal_name;
  (void) signal_data;
  (void) sender;
  
  composer_window = user_data;
  composer = composer_window->composer;
  etpan_status_bar_stop_animation(composer_window->status_bar);
  
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_MESSAGE_COMPOSER_READY_SIGNAL, composer, composer_window,
      forward_callback);
  
  error = etpan_message_composer_forward_get_error(composer);
  if (error != NULL) {
    etpan_error_log(error);
    goto free_composer;
  }
  
  etpan_message_composer_window_set_composer(composer_window, composer);
  
 free_composer:
  etpan_message_composer_free(composer);
  composer_window->composer = NULL;
  
  set_reference_message(composer_window, NULL);
}

void etpan_message_composer_window_forward(struct etpan_message_composer_window * composer_window, struct etpan_message * msg)
{
  struct etpan_message_composer * composer;
  
  composer_window->compose_type = ETPAN_COMPOSE_TYPE_FORWARD;
  set_reference_message(composer_window, msg);
  
  composer = etpan_message_composer_new();
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_MESSAGE_COMPOSER_READY_SIGNAL, composer, composer_window,
      forward_callback);
  
  composer_window->composer = composer;
  etpan_message_composer_forward(composer, msg, ETPAN_FORWARD_TYPE_NORMAL);
  etpan_status_bar_start_animation(composer_window->status_bar);
}

void etpan_message_composer_window_forward_cancel(struct etpan_message_composer_window * composer_window)
{
  etpan_message_composer_forward_cancel(composer_window->composer);
  set_reference_message(composer_window, NULL);
}

void etpan_message_composer_window_set_account(struct etpan_message_composer_window * composer_window, struct etpan_account * account)
{
  carray * list;
  unsigned int i;
  
  list = etpan_account_manager_get_ordered_list(etpan_account_manager_get_default());
  for(i = 0 ; i < carray_count(list) ; i ++) {
    struct etpan_account * current_account;
    
    current_account = carray_get(list, i);
    if (current_account == account)
      gtk_combo_box_set_active(GTK_COMBO_BOX(composer_window->account_combo), i);
  }
  additional_headers_set(composer_window);
}

static void update_account(struct etpan_message_composer_window * composer_window)
{
  struct etpan_account_manager * manager;
  chash * accounthash;
  
  manager = etpan_account_manager_get_default();
  
  accounthash = etpan_account_manager_get_account_hash(manager);
  if (chash_count(accounthash) > 1) {
    gtk_widget_show(composer_window->account_label);
    gtk_widget_show(composer_window->account_combo);
  }
  else {
    gtk_widget_hide(composer_window->account_label);
    gtk_widget_hide(composer_window->account_combo);
  }
  etpan_gtk_tree_model_reload(composer_window->treemodel);
  
  if (gtk_combo_box_get_active(GTK_COMBO_BOX(composer_window->account_combo)) == -1)
    gtk_combo_box_set_active(GTK_COMBO_BOX(composer_window->account_combo), 0);
}

struct etpan_error * etpan_message_composer_window_get_error(struct etpan_message_composer_window * composer_window)
{
  return composer_window->error;
}

void etpan_message_composer_window_show(struct etpan_message_composer_window * composer_window)
{
  gtk_widget_show(composer_window->window);
}

static void set_reference_message(struct etpan_message_composer_window * composer_window, struct etpan_message * msg)
{
  if (composer_window->reference_msg != NULL) {
    struct etpan_folder * folder;
    
    folder = etpan_message_get_folder(composer_window->reference_msg);
    etpan_message_unref(composer_window->reference_msg);
    etpan_folder_unref_msg_list(folder);
    composer_window->reference_msg = NULL;
  }
  
  if (msg != NULL) {
    struct etpan_folder * folder;
    
    folder = etpan_message_get_folder(msg);
    etpan_folder_ref_msg_list(folder);
    etpan_message_ref(msg);
  }
  composer_window->reference_msg = msg;
}
