#include "etpan-message-list.h"

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <gdk/gdkkeysyms.h>

#include "etpan-backend.h"
#include "etpan-message-view.h"
#include "etpan-folder-list.h"
#include "etpan-gtk-workaround.h"
#include "etpan-message-fetcher.h"
#include "etpan-message-composer-window.h"

#include "etpan-gtk-tree-model.h"
#include "etpan-message-tree-lep.h"
#include "etpan-contextual-menu.h"
#include "etpan-message-color.h"
#include "etpan-ui-config.h"
#include "etpan-icon-manager.h"
#include "etpan-search-field.h"

#include "gtktreeprivate.h"

enum {
  SELECT_MODE_NONE,
  SELECT_MODE_FIRST_OPEN,
  SELECT_MODE_KEEP_IN_VIEW,
};

static void drag_drop_setup(struct etpan_message_list * msg_list);
static void drag_drop_unsetup(struct etpan_message_list * msg_list);
static void drag_may_start(struct etpan_message_list * msg_list,
    gdouble x, gdouble y, guint32 time);
static void drag_cancel_may_start(struct etpan_message_list * msg_list);
static int drag_may_be_starting(struct etpan_message_list * msg_list);
static int drag_is_valid(struct etpan_message_list * msg_list,
    gdouble x, gdouble y, guint32 time);
static void drag_start(struct etpan_message_list * msg_list,
    GdkEvent * event);
static void drag_end(struct etpan_message_list * msg_list);
static void drag_set_icon(struct etpan_message_list * msg_list);

#if 0
static void copy_finished_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);
#endif

static void copy_with_delete(struct etpan_message_list * msg_list,
    struct etpan_folder * folder, int delete);

static void next_unread(struct etpan_message_list * msg_list);

static void go_to_message(struct etpan_message_list * msg_list,
    struct etpan_message_tree * elt, int select_mode);

static void start_animation_folder(struct etpan_message_list * msg_list,
    struct etpan_folder * folder);
static void stop_animation_folder(struct etpan_message_list * msg_list,
    struct etpan_folder * folder);

static void apply_search(struct etpan_message_list * msg_list);
#if 0
static void apply_search_callback(int cancelled,
    struct etpan_folder_indexer_search_result * result, void * cb_data);
#endif
static void set_search_state(struct etpan_message_list * msg_list);
static void start_searching(struct etpan_message_list * msg_list);
static void stop_searching(struct etpan_message_list * msg_list);
static void search_setup(struct etpan_message_list * msg_list);
static void search_unsetup(struct etpan_message_list * msg_list);
static void search_update_progress(struct etpan_message_list * msg_list);

#define UNREAD_COL_WIDTH 40
#define FROM_COL_WIDTH 150
#define SUBJECT_COL_WIDTH 400
#define DATE_COL_WIDTH 80

#define FROM_MIN_WIDTH 10
#define SUBJECT_MIN_WIDTH 10
#define DATE_MIN_WIDTH 10

enum {
  STORE_INDEX_FROM,
  STORE_INDEX_SUBJECT,
  STORE_INDEX_DATE,
  STORE_INDEX_INFO,
  STORE_INDEX_IMAGE,
  STORE_INDEX_COLOR,
};

#define DEFAULT_INCOMING_CHARSET "iso-8859-1"


static unsigned int get_n_columns(struct etpan_gtk_tree_data_source *
    datasource);
static GType get_column_type(struct etpan_gtk_tree_data_source * datasource,
    unsigned int column_index);
static unsigned int get_children_count(struct etpan_gtk_tree_data_source *
    datasource, void * item);
static int item_has_child(struct etpan_gtk_tree_data_source *
    datasource, void * item);
static void * get_child_item(struct etpan_gtk_tree_data_source *
    datasource, void * item, unsigned int index);
static void get_item_value(struct etpan_gtk_tree_data_source *
    datasource, void * item, unsigned int column_index,
    GValue * value);


struct etpan_message_list * etpan_message_list_new(void)
{
  GtkWidget * treeview;
  GtkWidget * scrolledwindow;
  struct etpan_message_list * msg_list;
  GtkTreeViewColumn * col_unread;
  GtkCellRenderer * col_unread_renderer;
  GtkTreeViewColumn * col_from;
  GtkCellRenderer * col_from_renderer;
  GtkTreeViewColumn * col_subject;
  GtkCellRenderer * col_subject_renderer;
  GtkTreeViewColumn * col_date;
  GtkCellRenderer * col_date_renderer;
  etpan_gtk_tree_model * msglist_model;
  GtkTreeSelection * selection;
  chash * selected_thread;
  GtkWidget * vbox;
  GtkWidget * search_label;
  
  msg_list = malloc(sizeof(* msg_list));
  if (msg_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  selected_thread = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (selected_thread == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  vbox = gtk_vbox_new(FALSE, 0);
  gtk_widget_show(vbox);
  
  search_label = gtk_label_new("Results of the search");
  gtk_widget_show(search_label);
  
  gtk_box_pack_start(GTK_BOX(vbox), search_label,
      FALSE, FALSE, 0);
  
  gtk_tree_store_update_lock();
  msglist_model = etpan_gtk_tree_model_new();
  gtk_tree_store_update_unlock();
  
  msg_list->msglist_datasource.data = msg_list;
  msg_list->msglist_datasource.get_n_columns = get_n_columns;
  msg_list->msglist_datasource.get_column_type = get_column_type;
  msg_list->msglist_datasource.get_children_count = get_children_count;
  msg_list->msglist_datasource.item_has_child = item_has_child;
  msg_list->msglist_datasource.get_child_item = get_child_item;
  msg_list->msglist_datasource.get_item_value = get_item_value;
  
  etpan_gtk_tree_model_set_datasource(msglist_model,
      &msg_list->msglist_datasource);
  
  msg_list->msg_tree = NULL;
  
  scrolledwindow = gtk_scrolled_window_new(NULL, NULL);
  gtk_widget_show(scrolledwindow);
  gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrolledwindow),
      GTK_SHADOW_IN);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledwindow),
      GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  
  gtk_box_pack_start(GTK_BOX(vbox), scrolledwindow,
      TRUE, TRUE, 0);
  
  treeview = gtk_tree_view_new();
  
  gtk_tree_view_set_enable_search(GTK_TREE_VIEW(treeview), 0);
  gtk_tree_view_set_model(GTK_TREE_VIEW(treeview),
      GTK_TREE_MODEL(msglist_model));
  
  gtk_widget_show(treeview);
  gtk_container_add(GTK_CONTAINER(scrolledwindow), treeview);
  gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(treeview), TRUE);
  gtk_tree_view_set_enable_search(GTK_TREE_VIEW(treeview), FALSE);
  
  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
  gtk_tree_selection_set_mode(selection, GTK_SELECTION_MULTIPLE);
  
  /* column unread */
  col_unread = gtk_tree_view_column_new();
  col_unread_renderer = gtk_cell_renderer_pixbuf_new();
  gtk_tree_view_column_pack_start(col_unread, col_unread_renderer, TRUE);
  gtk_tree_view_column_set_attributes(col_unread, col_unread_renderer,
      "pixbuf", STORE_INDEX_IMAGE,
      "cell-background", STORE_INDEX_COLOR,
      NULL);
  gtk_tree_view_column_set_resizable(col_unread, FALSE);
  gtk_tree_view_column_set_sizing(col_unread, GTK_TREE_VIEW_COLUMN_FIXED);
  gtk_tree_view_column_set_fixed_width(col_unread, UNREAD_COL_WIDTH);
  gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), col_unread);
  
  /* column from */
  col_from = gtk_tree_view_column_new();
  gtk_tree_view_column_set_title(col_from, _("From"));
  col_from_renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_column_pack_start(col_from, col_from_renderer, TRUE);
  gtk_tree_view_column_set_attributes(col_from, col_from_renderer,
      "text", STORE_INDEX_FROM,
      "cell-background", STORE_INDEX_COLOR,
      NULL);
  gtk_tree_view_column_set_resizable(col_from, TRUE);
  gtk_tree_view_column_set_sizing(col_from, GTK_TREE_VIEW_COLUMN_FIXED);
  gtk_tree_view_column_set_fixed_width(col_from, FROM_COL_WIDTH);
  gtk_tree_view_column_set_min_width(col_from, FROM_MIN_WIDTH);
  gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), col_from);
  
  /* column subject */
  col_subject = gtk_tree_view_column_new();
  gtk_tree_view_column_set_title(col_subject, _("Subject"));
  col_subject_renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_column_pack_start(col_subject, col_subject_renderer, TRUE);
  gtk_tree_view_column_set_attributes(col_subject, col_subject_renderer,
      "text", STORE_INDEX_SUBJECT,
      "cell-background", STORE_INDEX_COLOR,
      NULL);
  gtk_tree_view_column_set_resizable(col_subject, TRUE);
  gtk_tree_view_column_set_sizing(col_subject, GTK_TREE_VIEW_COLUMN_FIXED);
  gtk_tree_view_column_set_fixed_width(col_subject, SUBJECT_COL_WIDTH);
  gtk_tree_view_column_set_min_width(col_subject, SUBJECT_MIN_WIDTH);
  gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), col_subject);
  
  /* column date */
  col_date = gtk_tree_view_column_new();
  gtk_tree_view_column_set_title(col_date, _("Date"));
  col_date_renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_column_pack_start(col_date, col_date_renderer, TRUE);
  gtk_tree_view_column_set_attributes(col_date, col_date_renderer,
      "text", STORE_INDEX_DATE,
      "cell-background", STORE_INDEX_COLOR,
      NULL);
  gtk_tree_view_column_set_resizable(col_date, TRUE);
  gtk_tree_view_column_set_sizing(col_date, GTK_TREE_VIEW_COLUMN_FIXED);
  gtk_tree_view_column_set_fixed_width(col_date, DATE_COL_WIDTH);
  gtk_tree_view_column_set_min_width(col_date, DATE_MIN_WIDTH);
  gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), col_date);
  
  gtk_tree_view_set_expander_column(GTK_TREE_VIEW(treeview), col_subject);
  
  msg_list->main_widget = vbox;
  msg_list->search_label = search_label;
  msg_list->scrolledwindow = scrolledwindow;
  msg_list->treeview = treeview;
  msg_list->msglist_model = msglist_model;
  msg_list->msg_tree = NULL;
  msg_list->folder = NULL;
  msg_list->fetch_list_op = NULL;
  msg_list->current_selection = NULL;
  msg_list->folder_list = NULL;
  
  msg_list->col_unread = col_unread;
  msg_list->col_from = col_from;
  msg_list->col_subject = col_subject;
  msg_list->col_date = col_date;
  
  msg_list->selected_thread = selected_thread;
  
  msg_list->search_filter_type =
    ETPAN_FOLDER_INDEXER_SEARCH_TYPE_FROM_AND_SUBJECT;
  msg_list->search_filter_set = 0;
  msg_list->search_filter_text = NULL;
  msg_list->searching = 0;
  msg_list->indexing = 0;
  msg_list->filtering_search_result = 0;
  msg_list->search_has_progress = 0;
  
  msg_list->drag_may_start = 0;
  msg_list->drag_expanded = 0;
  msg_list->drag_may_start_x = 0;
  msg_list->drag_may_start_y = 0;
  msg_list->drag_context = NULL;
  msg_list->drag_started = 0;
  msg_list->drag_timer_set = 0;
  
  msg_list->reaching_message = 0;
  msg_list->expanding = 0;
  msg_list->selected_thread_count = 0;
  msg_list->reloading = 0;
  
  msg_list->folder_view = NULL;
  msg_list->has_user_action = 0;
  
  return msg_list;
}

void etpan_message_list_free(struct etpan_message_list * msg_list)
{
  if (msg_list->current_selection != NULL)
    free(msg_list->current_selection);
  
  gtk_tree_view_set_model(GTK_TREE_VIEW(msg_list->treeview), NULL);
  gtk_widget_destroy(msg_list->treeview);
  if (msg_list->msg_tree != NULL) {
    /* in most case, this will not be called */
    etpan_message_tree_free_recursive(msg_list->msg_tree);
  }
  gtk_tree_store_update_lock();
  g_object_unref(msg_list->msglist_model);
  gtk_tree_store_update_unlock();
  gtk_widget_destroy(msg_list->scrolledwindow);
  gtk_widget_destroy(msg_list->search_label);
  gtk_widget_destroy(msg_list->main_widget);
  chash_free(msg_list->selected_thread);
  free(msg_list);
}

GtkWidget * etpan_message_list_get_main_widget(struct etpan_message_list *
    msg_list)
{
  return msg_list->main_widget;
}

static void check_msg_list(struct etpan_message_list * msg_list,
    struct etpan_folder * folder)
{
  etpan_mail_manager_folder_check(etpan_mail_manager_get_default(), folder);
}

static void notify_message_selection(struct etpan_message_list * msg_list);
static void update_msg_list_treeview(struct etpan_message_list * msg_list);
static void select_thread(struct etpan_message_list * msg_list,
    GtkTreeIter * iter);

static void set_search_state(struct etpan_message_list * msg_list);

static void indexer_updated(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_message_list * msg_list;
  (void) signal_name;
  (void) signal_data;
  (void) sender;
  
  msg_list = user_data;
  if (signal_data != msg_list->folder)
    return;
  
  if (msg_list->search_filter_set)
    set_search_state(msg_list);
}

static void msg_tree_updated(char * signal_name, void * sender,
    void * signal_data, void * user_data);

void etpan_message_list_set_folder(struct etpan_message_list * msg_list,
    struct etpan_folder * folder)
{
  if (msg_list->folder != folder) {
    msg_list->msg_tree = NULL;
    update_msg_list_treeview(msg_list);
    
    if (msg_list->folder != NULL) {
      struct etpan_msg_list_view * view;
      
      check_msg_list(msg_list, msg_list->folder);
      if (msg_list->has_user_action) {
        msg_list->has_user_action = 0;
        stop_animation_folder(msg_list, msg_list->folder);
      }
      
      msg_list->indexing = 0;
      
      ETPAN_SIGNAL_REMOVE_HANDLER(etpan_mail_manager_get_default(),
          ETPAN_MAIL_MANAGER_INDEXER_UPDATED_SIGNAL,
          indexer_updated, msg_list);
      
      view = etpan_folder_view_get_msg_list_view(msg_list->folder_view);
      ETPAN_SIGNAL_REMOVE_HANDLER(view,
          ETPAN_MSG_LIST_VIEW_UPDATED_SIGNAL,
          msg_tree_updated, msg_list);
      
      etpan_folder_view_unsetup(msg_list->folder_view);
      etpan_folder_view_free(msg_list->folder_view);
      msg_list->folder_view = NULL;
    }
    
    msg_list->folder = folder;
    if (msg_list->folder != NULL) {
      struct etpan_msg_list_view * view;
      
      msg_list->folder_view = etpan_folder_view_new();
      etpan_folder_view_set_folder(msg_list->folder_view, msg_list->folder);
      etpan_folder_view_setup(msg_list->folder_view);
      
      view = etpan_folder_view_get_msg_list_view(msg_list->folder_view);
      msg_list->has_user_action = 1;
      start_animation_folder(msg_list, msg_list->folder);
      
      ETPAN_SIGNAL_ADD_HANDLER(view,
          ETPAN_MSG_LIST_VIEW_UPDATED_SIGNAL,
          msg_tree_updated, msg_list);
      
      ETPAN_SIGNAL_ADD_HANDLER(etpan_mail_manager_get_default(),
          ETPAN_MAIL_MANAGER_INDEXER_UPDATED_SIGNAL,
          indexer_updated, msg_list);
      
      apply_search(msg_list);
    }
    else {
      update_msg_list_treeview(msg_list);
    }
    
    notify_message_selection(msg_list);
  }
}

static void update_selected_thread(struct etpan_message_list * msg_list);

static void update_msg_list_treeview(struct etpan_message_list * msg_list)
{
  msg_list->current_time = time(NULL);
  msg_list->reloading = 1;
  etpan_gtk_tree_model_reload(msg_list->msglist_model);
  msg_list->reloading = 0;
  update_selected_thread(msg_list);
  gtk_widget_queue_draw(msg_list->treeview);
}

static void mark_messages(chash * existing, struct etpan_message_tree * tree)
{
  struct etpan_message * msg;
  carray * list;
  int r;
  
  if (tree == NULL)
    return;
  
  list = etpan_message_tree_get_children(tree);
  msg = etpan_message_tree_get_message(tree);
  if (msg != NULL) {
    char * uid;
    chashdatum key;
    chashdatum value;
    
    uid = etpan_message_get_uid(msg);
    key.data = uid;
    key.len = strlen(uid);
    value.data = tree;
    value.len = 0;
    r = chash_set(existing, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  if (list != NULL) {
    unsigned int i;
    
    for(i = 0 ; i < carray_count(list) ; i ++) {
      struct etpan_message_tree * child;
      
      child = carray_get(list, i);
      mark_messages(existing, child);
    }
  }
}

static void msg_tree_updated(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_msg_list_view * view;
  struct etpan_message_tree * tree;
  struct etpan_message_list * msg_list;
  carray * new_messages;
  unsigned int i;
  
  msg_list = user_data;
  
  view = etpan_folder_view_get_msg_list_view(msg_list->folder_view);
  if (msg_list->has_user_action) {
    stop_animation_folder(msg_list, msg_list->folder);
    msg_list->has_user_action = 0;
  }
  
  if (msg_list->filtering_search_result) {
    msg_list->filtering_search_result = 0;
    search_update_progress(msg_list);
  }
  
  tree = etpan_msg_list_view_get_tree(view);
  msg_list->msg_tree = tree;
  update_msg_list_treeview(msg_list);
  
  new_messages = etpan_msg_list_view_get_new_messages(view);
  if (new_messages != NULL) {
    for(i = 0 ; i < carray_count(new_messages) ; i ++) {
      struct etpan_message_tree * node;
    
      node = carray_get(new_messages, i);
      go_to_message(msg_list, node, SELECT_MODE_NONE);
    }
  }
}

static struct etpan_message *
get_selection_data(struct etpan_message_list * msg_list,
    char * selection_path)
{
  struct etpan_message_tree * tree;
  GtkTreeIter iter;
  
  if (!gtk_tree_model_get_iter_from_string(GTK_TREE_MODEL(msg_list->msglist_model),
          &iter, (gchar *) selection_path))
    return NULL;
  
  tree = etpan_gtk_tree_model_get_item(msg_list->msglist_model, &iter);
  if (tree == NULL)
    tree = msg_list->msg_tree;
  
  return etpan_message_tree_get_message(tree);
}

static struct etpan_message_tree *
get_selection_tree(struct etpan_message_list * msg_list,
    char * selection_path)
{
  struct etpan_message_tree * tree;
  GtkTreeIter iter;
  
  if (!gtk_tree_model_get_iter_from_string(GTK_TREE_MODEL(msg_list->msglist_model),
          &iter, (gchar *) selection_path))
    return NULL;
  
  tree = etpan_gtk_tree_model_get_item(msg_list->msglist_model, &iter);
  if (tree == NULL)
    tree = msg_list->msg_tree;
  
  return tree;
}

static void
etpan_message_list_expand_selection(struct etpan_message_list * msg_list,
    char * selection_path);


struct etpan_message_tree *
get_root_thread(struct etpan_message_list * msg_list,
    struct etpan_message_tree * tree)
{
  struct etpan_message_tree * parent;
  
  parent = etpan_message_tree_get_parent(tree);
  ETPAN_MSGLIST_LOG("parent : %p", parent);
  if (parent == msg_list->msg_tree)
    return tree;
  
  return get_root_thread(msg_list, parent);
}

static void mark_thread(struct etpan_message_list * msg_list,
    struct etpan_message_tree * tree)
{
  chashdatum key;
  chashdatum value;
  int r;
  carray * children;
  
  key.data = &tree;
  key.len = sizeof(tree);
  value.data = NULL;
  value.len = 0;
  r = chash_set(msg_list->selected_thread, &key ,&value, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  children = etpan_message_tree_get_children(tree);
  if (children != NULL) {
    unsigned int i;
    
    for(i = 0 ; i < carray_count(children) ; i ++) {
      struct etpan_message_tree * child;
      
      child = carray_get(children, i);
      mark_thread(msg_list, child);
    }
  }
}

static int
is_root_and_not_expanded(struct etpan_message_list * msg_list,
    char * selection_path)
{
  GtkTreePath * path;
  int root;
  int expanded;
  
  root = 0;
  expanded = 0;
  
  path = gtk_tree_path_new_from_string((gchar *) selection_path);
  if (path == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  if (gtk_tree_path_get_depth(path) == 1)
    root = 1;
  
  if (gtk_tree_view_row_expanded(GTK_TREE_VIEW(msg_list->treeview), path))
    expanded = 1;
  
  gtk_tree_path_free(path);
  
  return root && (!expanded);
}

static void notify_message_selection(struct etpan_message_list * msg_list)
{
  struct etpan_message * msg;
  GtkTreePath * path;
  GtkTreeViewColumn * column;
  GtkTreeSelection * selection;
  
  gtk_tree_view_get_cursor(GTK_TREE_VIEW(msg_list->treeview), &path, &column);
  
  free(msg_list->current_selection);
  msg_list->current_selection = NULL;
  
  msg = NULL;
  if (path != NULL) {
    gchar * selection_path;
    struct etpan_message_tree * tree;
    
    selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(msg_list->treeview));
    if (!gtk_tree_selection_path_is_selected(selection, path)) {
      gtk_tree_path_free(path);
      return;
    }
    
    selection_path = gtk_tree_path_to_string(path);
    ETPAN_MSGLIST_LOG("selection path : %s", selection_path);
    
    etpan_message_list_expand_selection(msg_list, selection_path);
    
    msg = get_selection_data(msg_list, selection_path);
    if (msg != NULL) {
      char * uid;
      
      uid = etpan_message_get_uid(msg);
      if (uid != NULL) {
        msg_list->current_selection = strdup(uid);
        if (msg_list->current_selection == NULL)
          ETPAN_LOG_MEMORY_ERROR;
        ETPAN_MSGLIST_LOG("selection : %s", msg_list->current_selection);
      }
    }
    
    g_free(selection_path);
    gtk_tree_path_free(path);
  }
  
  if (msg_list->current_selection == NULL)
    ETPAN_MSGLIST_LOG("no selection");
  
  etpan_signal_send(etpan_signal_manager_get_default(),
      ETPAN_MESSAGE_LIST_SELECTIONCHANGED_SIGNAL,
      msg_list, msg);
}

static void message_selected_handler(GtkTreeView * treeview,
    gpointer user_data)
{
  struct etpan_message_list * msg_list;
  (void) treeview;
  
  msg_list = (struct etpan_message_list *) user_data;
  
  msg_list->reaching_message = 0;
  notify_message_selection(msg_list);
}

static void flags_changed_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_message * msg;
  struct etpan_message_list * msg_list;
  (void) signal_name;
  (void) signal_data;
  (void) sender;
  
  msg_list = user_data;
  msg = signal_data;
  
  check_msg_list(msg_list, msg_list->folder);
  
  gtk_widget_queue_draw(msg_list->treeview);
}

static void message_popup_submenu_item_clicked(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_contextual_menu * menu;
  struct etpan_message_list * msg_list;
  int * pindex;
  (void) signal_name;
  
  msg_list = user_data;
  menu = sender;
  pindex = signal_data;
  
  switch (* pindex) {
  case 0:
    /* red */
    etpan_message_list_mark_color(msg_list, "#ff8080");
    break;
  case 1:
    /* blue */
    etpan_message_list_mark_color(msg_list, "#8080ff");
    break;
  case 2:
    /* green */
    etpan_message_list_mark_color(msg_list, "#80ff80");
    break;
  }
}

static void reply_message(struct etpan_message_list * msg_list)
{
  struct etpan_message_composer_window * composer;
  GtkWidget * window;
  
  composer = etpan_message_composer_window_new();
  etpan_message_composer_window_setup(composer);
  window = etpan_message_composer_window_get_main_widget(composer);
  gtk_widget_show(window);
  
  etpan_message_list_reply(msg_list, composer);
}

static void forward_message(struct etpan_message_list * msg_list)
{
  struct etpan_message_composer_window * composer;
  GtkWidget * window;
  
  composer = etpan_message_composer_window_new();
  etpan_message_composer_window_setup(composer);
  window = etpan_message_composer_window_get_main_widget(composer);
  gtk_widget_show(window);
  
  etpan_message_list_forward(msg_list, composer);
}

static void message_popup_menu_item_clicked(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_contextual_menu * menu;
  struct etpan_message_list * msg_list;
  int * pindex;
  (void) signal_name;
  
  msg_list = user_data;
  menu = sender;
  pindex = signal_data;
  
  ETPAN_MSGLIST_LOG("index : %i", * pindex);
  
  switch (* pindex) {
  case 0:
    /* mark as read */
    etpan_message_list_mark_as_read(msg_list);
    break;
  case 1:
    /* mark */
    etpan_message_list_mark_as_flagged(msg_list);
    break;
  case 2:
    /* label */
    break;
  case 3:
    /* separator */
    break;
  case 4:
    /* delete */
    etpan_message_list_delete(msg_list);
    break;
  case 5:
    /* separator */
    break;
  case 6:
    /* reply */
    reply_message(msg_list);
    break;
  case 7:
    /* forward */
    forward_message(msg_list);
    break;
  }
}

static void popup_menu_setup(struct etpan_message_list * msg_list)
{
  struct etpan_contextual_menu * menu;
  struct etpan_contextual_menu * submenu;
  
  menu = etpan_contextual_menu_new();
  etpan_contextual_menu_add_item(menu, _("Mark as read"), NULL);
  etpan_contextual_menu_add_item(menu, _("Mark"), NULL);
  
#if 0
  submenu = etpan_contextual_menu_new();
  etpan_contextual_menu_add_sub(menu, _("Label"), NULL, submenu);
  etpan_contextual_menu_add_item(submenu, _("Red"), NULL);
  etpan_contextual_menu_add_item(submenu, _("Blue"), NULL);
  etpan_contextual_menu_add_item(submenu, _("Green"), NULL);
  etpan_contextual_menu_add_item(submenu, _("Other ..."), NULL);
#endif
  
  etpan_contextual_menu_add_separator(menu);
  etpan_contextual_menu_add_item(menu, _("Delete"), NULL);
  etpan_contextual_menu_add_separator(menu);
  etpan_contextual_menu_add_item(menu, _("Reply"), NULL);
  etpan_contextual_menu_add_item(menu, _("Forward"), NULL);
  
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
    ETPAN_CONTEXTUAL_MENU_CLICKED_SIGNAL, menu, msg_list,
      message_popup_menu_item_clicked);
  
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
    ETPAN_CONTEXTUAL_MENU_CLICKED_SIGNAL, submenu, msg_list,
      message_popup_submenu_item_clicked);
  
  msg_list->popup_menu = menu;
  msg_list->popup_submenu = submenu;
}

static void popup_menu_unsetup(struct etpan_message_list * msg_list)
{
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_CONTEXTUAL_MENU_CLICKED_SIGNAL, msg_list->popup_submenu, msg_list,
      message_popup_menu_item_clicked);
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_CONTEXTUAL_MENU_CLICKED_SIGNAL, msg_list->popup_menu, msg_list,
      message_popup_menu_item_clicked);
  
  etpan_contextual_menu_free(msg_list->popup_menu);
}

static void show_message_popup_menu(struct etpan_message_list * msg_list,
    GtkWidget * widget, GdkEventButton * event)
{
  GtkWidget * menu;
  int button, event_time;
  (void) widget;
  
  if (event) {
    button = event->button;
    event_time = event->time;
  }
  else {
    button = 0;
    event_time = gtk_get_current_event_time();
  }
  
  menu = etpan_contextual_menu_get_main_widget(msg_list->popup_menu);
  
  gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL, 
      button, event_time);
}

static gboolean button_press_handler(GtkWidget * widget,
    GdkEventButton * event,
    gpointer user_data)
{
  struct etpan_message_list * msg_list;
  GtkTreePath * path;
  int do_free_path;
  gboolean result;
  (void) widget;
  
  ETPAN_MSGLIST_LOG("press");
  
  msg_list = user_data;
  
  result = FALSE;
  do_free_path = 0;
  if (!result) {
    if (event->button == 3) {
      if (gtk_tree_view_get_path_at_pos(GTK_TREE_VIEW(msg_list->treeview),
              event->x, event->y, &path, NULL, NULL, NULL)) {
        GtkTreeSelection * selection;
        
        do_free_path = 1;
        selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(msg_list->treeview));
        if (gtk_tree_selection_path_is_selected(selection, path)) {
          ETPAN_MSGLIST_LOG("click on selected");
      
          show_message_popup_menu(msg_list, msg_list->treeview, event);
      
          result = TRUE;
        }
      }
    }
  }
  
  /* headers */
  msg_list->changing = 1;
  msg_list->current_from_size =
    gtk_tree_view_column_get_width(msg_list->col_from);
  msg_list->current_subject_size =
    gtk_tree_view_column_get_width(msg_list->col_subject);
  msg_list->current_date_size =
    gtk_tree_view_column_get_width(msg_list->col_date);
  
  ETPAN_MSGLIST_LOG("%i", msg_list->current_from_size +
      msg_list->current_subject_size);
  
  if (do_free_path)
    gtk_tree_path_free(path);
  
  return result;
}

static gboolean button_release_handler(GtkWidget * widget,
    GdkEventButton * event,
    gpointer user_data)
{
  struct etpan_message_list * msg_list;
  gint size;
  int from_changed;
  int subject_changed;
  int date_changed;
  (void) widget; 
  (void) event;
  
  ETPAN_MSGLIST_LOG("release");
  msg_list = user_data;
  
  if (!msg_list->changing)
    return FALSE;
  
  from_changed = 0;
  subject_changed = 0;
  date_changed = 0;
  size = gtk_tree_view_column_get_width(msg_list->col_from);
  if (size != msg_list->current_from_size)
    from_changed = 1;
  
  size = gtk_tree_view_column_get_width(msg_list->col_subject);
  if (size != msg_list->current_subject_size)
    subject_changed = 1;
  
  size = gtk_tree_view_column_get_width(msg_list->col_date);
  if (size != msg_list->current_date_size)
    date_changed = 1;
  
  if (from_changed) {
    int diff;
    
    size = gtk_tree_view_column_get_width(msg_list->col_from);
    if (size > msg_list->current_subject_size + msg_list->current_from_size - SUBJECT_MIN_WIDTH) {
      size = msg_list->current_subject_size + msg_list->current_from_size - SUBJECT_MIN_WIDTH;
      gtk_tree_view_column_set_fixed_width(msg_list->col_from, size);
    }
    
    diff = size - msg_list->current_from_size;
    
    gtk_tree_view_column_set_fixed_width(msg_list->col_subject,
        msg_list->current_subject_size - diff);
    gtk_tree_view_column_set_fixed_width(msg_list->col_date,
        msg_list->current_date_size);
  }
  else if (subject_changed || date_changed) {
    int diff;
    
    size = gtk_tree_view_column_get_width(msg_list->col_subject);

    if (size > msg_list->current_subject_size + msg_list->current_date_size - DATE_MIN_WIDTH) {
      size = msg_list->current_subject_size + msg_list->current_date_size - DATE_MIN_WIDTH;
      gtk_tree_view_column_set_fixed_width(msg_list->col_subject, size);
    }
    
    diff = size - msg_list->current_subject_size;
    
    gtk_tree_view_column_set_fixed_width(msg_list->col_date,
        msg_list->current_date_size - diff);
    gtk_tree_view_column_set_fixed_width(msg_list->col_from,
        msg_list->current_from_size);
  }
  
  msg_list->changing = 0;
  return FALSE;
}

static gboolean motion_handler(GtkWidget * widget,
    GdkEventMotion * event,
    gpointer user_data)
{
  struct etpan_message_list * msg_list;
  (void) widget;
  (void) event;
  
  msg_list = user_data;
  
  if (!msg_list->changing) {
    /*
    if (!gtk_widget_is_focus(msg_list->treeview))
      gtk_widget_grab_focus(msg_list->treeview);
    */
    return FALSE;
  }
  
  return FALSE;
}

static void go_to_message(struct etpan_message_list * msg_list,
    struct etpan_message_tree * elt, int select_mode)
{
  struct etpan_message_tree * parent;
  GtkTreeIter iter;
  GtkTreePath * path;
  
  if (select_mode != SELECT_MODE_NONE) {
    ETPAN_MSGLIST_LOG("go to message %p", elt);
  }
  
  parent = etpan_message_tree_get_parent(elt);
  if (parent == NULL) {
    return;
  }
  
  if (etpan_message_tree_get_parent(parent) != NULL)
    go_to_message(msg_list, parent, SELECT_MODE_NONE);
  
  etpan_gtk_tree_model_get_iter_from_item(msg_list->msglist_model,
      &iter, elt);
    
  path =
    gtk_tree_model_get_path(GTK_TREE_MODEL(msg_list->msglist_model),
        &iter);
  
  if (path == NULL) {
    struct etpan_message * msg;
    
    msg = etpan_message_tree_get_message(elt);
    if (msg != NULL)
      ETPAN_WARN_LOG("path is null %s", etpan_message_get_description(msg));
  }
  else {
    gtk_tree_view_expand_to_path(GTK_TREE_VIEW(msg_list->treeview),
        path);
    
    if (select_mode != SELECT_MODE_NONE) {
      char * str;
    
      str = gtk_tree_path_to_string(path);
      ETPAN_MSGLIST_LOG("select %s", str);
      g_free(str);
      switch (select_mode) {
      case SELECT_MODE_FIRST_OPEN:
        gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(msg_list->treeview),
            path,
            NULL, TRUE, 1, 0);
        break;
      case SELECT_MODE_KEEP_IN_VIEW:
        gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(msg_list->treeview),
            path,
            NULL, FALSE, 0, 0);
        break;
      }
      gtk_tree_view_set_cursor(GTK_TREE_VIEW(msg_list->treeview),
          path, NULL, 0);
      msg_list->reaching_message = select_mode;
    }
  
    gtk_tree_path_free(path);
  }
}

static void get_unread_messages(carray * msg_list,
    struct etpan_message_tree * elt,
    struct etpan_message * current)
{
  carray * children;
  int r;
  struct etpan_message * msg;
  unsigned int i;
  
  msg = etpan_message_tree_get_message(elt);
  if (msg != NULL) {
    struct etpan_message_flags * flags;
    int flags_value;
      
    if (current == msg) {
      r = carray_add(msg_list, elt, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else {
      flags = etpan_message_get_flags(msg);
      flags_value = etpan_message_flags_get_value(flags);
      if ((flags_value & ETPAN_FLAGS_SEEN) == 0) {
        r = carray_add(msg_list, elt, NULL);
        if (r < 0)
          ETPAN_LOG_MEMORY_ERROR;
      }
    }
  }
  
  children = etpan_message_tree_get_children(elt);
  if (children != NULL) {
    for(i = 0 ; i < carray_count(children) ; i ++) {
      struct etpan_message_tree * child;
      
      child = carray_get(children, i);
      
      get_unread_messages(msg_list, child, current);
    }
  }
}

static void next_unread(struct etpan_message_list * msg_list)
{
  carray * msg_tab;
  struct etpan_message * msg;
  GtkTreePath * path;
  GtkTreeViewColumn * column;
  unsigned int i;
  unsigned int next;
  
  gtk_tree_view_get_cursor(GTK_TREE_VIEW(msg_list->treeview), &path, &column);
  
  msg = NULL;
  if (path != NULL) {
    gchar * selection_path;
    
    selection_path = gtk_tree_path_to_string(path);
    ETPAN_MSGLIST_LOG("selection path : %s", selection_path);
    msg = get_selection_data(msg_list, selection_path);
    
    gtk_tree_path_free(path);
    g_free(selection_path);
  }
  
  msg_tab = carray_new(16);
  if (msg_tab == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  get_unread_messages(msg_tab, msg_list->msg_tree, msg);
  
  ETPAN_MSGLIST_LOG("%i unread messages", carray_count(msg_tab));
  next = 0;
  for(i = 0 ; i < carray_count(msg_tab) ; i ++) {
    struct etpan_message_tree * elt;
    struct etpan_message * cur_msg;
    
    elt = carray_get(msg_tab, i);
    cur_msg = etpan_message_tree_get_message(elt);
    if (cur_msg == msg) {
      next = i + 1;
    }
  }
  
  if (next < carray_count(msg_tab)) {
    struct etpan_message_tree * elt;
    
    elt = carray_get(msg_tab, next);
    
    go_to_message(msg_list, elt, 1);
  }
  
  carray_free(msg_tab);
}

static gboolean keypress_handler(GtkWidget * widget,
    GdkEventKey * event,
    gpointer user_data)
{
  struct etpan_message_list * msg_list;
  (void) widget;
  
  msg_list = user_data;
  
  switch (event->keyval) {
  case GDK_Escape:
    {
      GtkTreeSelection * selection;
      
      selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(msg_list->treeview));
      gtk_tree_selection_unselect_all(selection);
      
      return TRUE;
    }
    break;
    
  case GDK_space:
    {
      /*
      next_unread(msg_list);
      */
      
      etpan_signal_send(etpan_signal_manager_get_default(),
          ETPAN_MESSAGE_LIST_NEXT_SIGNAL,
          msg_list, NULL);
    }
    break;

  case GDK_F:
  case GDK_f:
    if ((event->state & GDK_CONTROL_MASK) != 0) {
      return TRUE;
    }
    break;
  }
  
  return FALSE;
}

static void headers_setup(struct etpan_message_list * msg_list)
{
  msg_list->changing = 0;
  
  msg_list->button_press_signal_id =
    g_signal_connect(msg_list->treeview,
        "button-press-event", G_CALLBACK(button_press_handler),
        (gpointer) msg_list);
  
  msg_list->motion_signal_id =
    g_signal_connect(msg_list->treeview,
        "motion-notify-event", G_CALLBACK(motion_handler),
        (gpointer) msg_list);
  
  msg_list->button_release_signal_id =
    g_signal_connect(msg_list->treeview,
        "button-release-event", G_CALLBACK(button_release_handler),
        (gpointer) msg_list);

  popup_menu_setup(msg_list);
}

static void headers_unsetup(struct etpan_message_list * msg_list)
{
  popup_menu_unsetup(msg_list);
  
  g_signal_handler_disconnect(msg_list->treeview,
      msg_list->button_press_signal_id);
  g_signal_handler_disconnect(msg_list->treeview,
      msg_list->button_release_signal_id);
  g_signal_handler_disconnect(msg_list->treeview,
      msg_list->motion_signal_id);
}  

static void update_font(struct etpan_message_list * msg_list)
{
  PangoFontDescription * font_desc;
  
  font_desc = pango_font_description_from_string(etpan_ui_config_get_font_list(etpan_ui_config_get_default()));
  gtk_widget_modify_font(msg_list->treeview, font_desc);
  pango_font_description_free(font_desc);
}

static void font_changed_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_message_list * msg_list;
  (void) signal_name;
  (void) signal_data;
  (void) sender;
  
  msg_list = user_data;
  update_font(msg_list);
}

static gboolean selection_handler(GtkTreeSelection * selection,
    GtkTreeModel * model,
    GtkTreePath * path,
    gboolean path_currently_selected,
    gpointer data)
{
  (void) selection;
  (void) model;
  (void) path;
  (void) path_currently_selected;
  (void) data;
  
  return TRUE;
}

void message_view_size_allocated_handler(GtkWidget * widget,
    GtkAllocation * allocation,
    gpointer user_data)
{
  struct etpan_message_list * msg_list;
  GtkTreePath * path;
  
  msg_list = user_data;

  if (msg_list->reaching_message == SELECT_MODE_NONE)
    return;
  
  gtk_tree_view_get_cursor(GTK_TREE_VIEW(msg_list->treeview),
       &path, NULL);
  if (path != NULL) {
    switch (msg_list->reaching_message) {
    case SELECT_MODE_FIRST_OPEN:
      gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(msg_list->treeview),
          path,
          NULL, TRUE, 1, 0);
      break;
    case SELECT_MODE_KEEP_IN_VIEW:
      gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(msg_list->treeview),
          path,
          NULL, FALSE, 0, 0);
      break;
    }
    gtk_tree_path_free(path);
  }
}

static void expand_row_handler(GtkTreeView * tree_view,
    GtkTreeIter * iter,
    GtkTreePath * path,
    gpointer user_data)
{
  struct etpan_message_list * msg_list;
  
  msg_list = user_data;
  
  if (msg_list->expanding)
    return;
  
  msg_list->expanding = 1;
  gtk_tree_view_expand_row(GTK_TREE_VIEW(msg_list->treeview), path, TRUE);
  msg_list->expanding = 0;
}

static gboolean test_collapse_row_handler(GtkTreeView * tree_view,
    GtkTreeIter * iter,
    GtkTreePath * path,
    gpointer user_data)
{
  struct etpan_message_list * msg_list;
  
  msg_list = user_data;
  if (gtk_tree_path_get_depth(path) == 1)
    return FALSE;
  
  return TRUE;
}

static void select_thread(struct etpan_message_list * msg_list,
    GtkTreeIter * iter)
{
  unsigned int i;
  unsigned int count;
  GtkTreeSelection * selection;
  
  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(msg_list->treeview));
  gtk_tree_selection_select_iter(selection, iter);
  
  count = gtk_tree_model_iter_n_children(GTK_TREE_MODEL(msg_list->msglist_model), iter);
  for(i = 0 ; i < count ; i ++) {
    GtkTreeIter child;
    
    gtk_tree_model_iter_nth_child(GTK_TREE_MODEL(msg_list->msglist_model),
        &child, iter, i);
    select_thread(msg_list, &child);
  }
}

static void update_selected_thread(struct etpan_message_list * msg_list)
{
  GList * list;
  carray * tab;
  GList * cur;
  GtkTreeSelection * selection;
  int r;
  
  msg_list->selected_thread_count = 0;
  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(msg_list->treeview));
  list = gtk_tree_selection_get_selected_rows(selection, NULL);
  chash_clear(msg_list->selected_thread);
  
  ETPAN_MSGLIST_LOG("selected number : %i", g_list_length(list));
  for(cur = g_list_first(list) ; cur != NULL ; cur = g_list_next(cur)) {
    GtkTreePath * path;
    GtkTreeIter iter;
    
    path = cur->data;
    
    if (gtk_tree_model_get_iter(GTK_TREE_MODEL(msg_list->msglist_model),
            &iter, path)) {
      struct etpan_message_tree * tree;
      
      tree = etpan_gtk_tree_model_get_item(msg_list->msglist_model, &iter);
      
      if (tree != NULL) {
        chashdatum key;
        chashdatum value;
        
        tree = get_root_thread(msg_list, tree);
        
        key.data = &tree;
        key.len = sizeof(tree);
        r = chash_get(msg_list->selected_thread, &key ,&value);
        if (r == 0)
          continue;
        
        msg_list->selected_thread_count ++;
        mark_thread(msg_list, tree);
      }
    }
  }
  
  g_list_foreach(list, (GFunc) gtk_tree_path_free, NULL);
  g_list_free(list);
}

static void selection_changed_handler(GtkTreeSelection * treeselection,
    gpointer user_data)
{
  struct etpan_message_list * msg_list;
  
  msg_list = user_data;
  
  if (msg_list->reloading)
    return;
  
  update_selected_thread(msg_list);
  gtk_widget_queue_draw(msg_list->treeview);
}

void etpan_message_list_setup(struct etpan_message_list * msg_list)
{
  GtkTreeSelection * selection;
  int total_size;
  GtkTreeViewColumn * column;
  
  gtk_widget_realize(msg_list->treeview);
  msg_list->current_from_size =
    gtk_tree_view_column_get_width(msg_list->col_from);
  msg_list->current_subject_size =
    gtk_tree_view_column_get_width(msg_list->col_subject);
  msg_list->current_date_size =
    gtk_tree_view_column_get_width(msg_list->col_date);
  
  etpan_ui_column_set(etpan_ui_config_get_default(), "column-from-size",
      msg_list->treeview, 1);
  etpan_ui_column_set(etpan_ui_config_get_default(), "column-subject-size",
      msg_list->treeview, 2);
  
  total_size = msg_list->current_from_size + msg_list->current_subject_size + msg_list->current_date_size;
  msg_list->current_from_size =
    gtk_tree_view_column_get_width(msg_list->col_from);
  msg_list->current_subject_size =
    gtk_tree_view_column_get_width(msg_list->col_subject);
  msg_list->current_date_size = total_size - msg_list->current_from_size - msg_list->current_subject_size;
  column = gtk_tree_view_get_column(GTK_TREE_VIEW(msg_list->treeview), 3);
  gtk_tree_view_column_set_fixed_width(column, msg_list->current_date_size);
  
  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(msg_list->treeview));
  
  gtk_tree_selection_set_select_function(selection,
      selection_handler,
      msg_list, NULL);
  
  msg_list->keypress_signal_id =
    g_signal_connect(msg_list->treeview,
        "key-press-event", G_CALLBACK(keypress_handler),
        (gpointer) msg_list);
  
  msg_list->msg_selected_signal_id =
    g_signal_connect(msg_list->treeview,
        "cursor-changed", G_CALLBACK(message_selected_handler),
        (gpointer) msg_list);
  
  msg_list->size_changed_signal_id =
    g_signal_connect(msg_list->treeview,
        "size-allocate", G_CALLBACK(message_view_size_allocated_handler),
        (gpointer) msg_list);

  msg_list->test_collapse_row_signal_id =
    g_signal_connect(msg_list->treeview,
        "test-collapse-row", G_CALLBACK(test_collapse_row_handler),
        (gpointer) msg_list);

  msg_list->expand_row_signal_id =
    g_signal_connect(msg_list->treeview,
        "row-expanded", G_CALLBACK(expand_row_handler),
        (gpointer) msg_list);

  msg_list->selection_changed_signal_id =
    g_signal_connect(selection,
        "changed", G_CALLBACK(selection_changed_handler),
        (gpointer) msg_list);
  
  headers_setup(msg_list);
  
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
    ETPAN_MESSAGE_VIEW_FLAGSCHANGED_SIGNAL, NULL, msg_list,
      flags_changed_handler);
  
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_UI_CONFIG_FONT_CHANGED_SIGNAL,
      etpan_ui_config_get_default(), msg_list,
      font_changed_handler);
  
  update_font(msg_list);
  
  search_setup(msg_list);
  
  drag_drop_setup(msg_list);
  
  update_msg_list_treeview(msg_list);
}

void etpan_message_list_unsetup(struct etpan_message_list * msg_list)
{
  GtkTreeSelection * selection;

  etpan_ui_set_from_column(etpan_ui_config_get_default(), "column-from-size",
      msg_list->treeview, 1);
  etpan_ui_set_from_column(etpan_ui_config_get_default(), "column-subject-size",
      msg_list->treeview, 2);
  
  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(msg_list->treeview));
  
  drag_drop_unsetup(msg_list);
  search_unsetup(msg_list);
  
  etpan_message_list_set_folder_list(msg_list, NULL);

  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_UI_CONFIG_FONT_CHANGED_SIGNAL,
      etpan_ui_config_get_default(), msg_list,
      font_changed_handler);
  
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
    ETPAN_MESSAGE_VIEW_FLAGSCHANGED_SIGNAL, NULL, msg_list,
      flags_changed_handler);
  
  etpan_message_list_set_folder(msg_list, NULL);

  headers_unsetup(msg_list);
  
  g_signal_handler_disconnect(selection,
      msg_list->selection_changed_signal_id);
  
  g_signal_handler_disconnect(msg_list->treeview,
      msg_list->expand_row_signal_id);
  
  g_signal_handler_disconnect(msg_list->treeview,
      msg_list->test_collapse_row_signal_id);
  
  g_signal_handler_disconnect(msg_list->treeview,
      msg_list->size_changed_signal_id);
  
  g_signal_handler_disconnect(msg_list->treeview,
      msg_list->keypress_signal_id);
  
  g_signal_handler_disconnect(msg_list->treeview,
      msg_list->msg_selected_signal_id);
}

static void
etpan_message_list_expand_selection(struct etpan_message_list * msg_list,
    char * selection_path)
{
  GtkTreePath * path;
  
  if (!is_root_and_not_expanded(msg_list, selection_path))
    return;
  
  path = gtk_tree_path_new_from_string((gchar *) selection_path);
  if (path == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  msg_list->expanding = 1;
  gtk_tree_view_expand_row(GTK_TREE_VIEW(msg_list->treeview), path, TRUE);
  msg_list->expanding = 0;
  
  gtk_tree_path_free(path);
}

static void start_animation_folder(struct etpan_message_list * msg_list,
    struct etpan_folder * folder)
{
  if (msg_list->folder_list != NULL)
    etpan_folder_list_start_animation_folder(msg_list->folder_list, folder);
}

static void stop_animation_folder(struct etpan_message_list * msg_list,
    struct etpan_folder * folder)
{
  if (msg_list->folder_list != NULL)
    etpan_folder_list_stop_animation_folder(msg_list->folder_list, folder);
}



static void folder_selected_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  char * selection_path;
  struct etpan_message_list * msg_list;
  struct etpan_folder_list * folder_list;
  struct etpan_folder * folder;
  (void) signal_name;
  
  msg_list = user_data;
  folder_list = msg_list->folder_list;
  selection_path = signal_data;
  
  ETPAN_LOG("folder selected %s", selection_path);
  folder = NULL;
  if (selection_path != NULL) {
    int selection_type;
    
    ETPAN_MSGLIST_LOG("row selected %p %s", sender, selection_path);
    
    selection_type =
      etpan_folder_list_get_selection_type(folder_list,
          selection_path);
    
    if (selection_type == ETPAN_FOLDER_LIST_SELECTION_TYPE_FOLDER) {
      folder = etpan_folder_list_get_folder(folder_list,
          selection_path);
      
      ETPAN_MSGLIST_LOG("got folder %s", etpan_folder_get_ui_path(folder));
    }
  }
  
  if (folder == NULL) {
    ETPAN_MSGLIST_LOG("no selection");
  }
  
  etpan_message_list_set_folder(msg_list, folder);
}

struct etpan_message *
etpan_message_list_get_message(struct etpan_message_list * msg_list,
    char * uid)
{
  return etpan_folder_get_message(msg_list->folder, uid);
}

void etpan_message_list_set_search_field(struct etpan_message_list * msg_list,
    struct etpan_search_field * search_field)
{
  msg_list->search_field = search_field;
}

void etpan_message_list_set_folder_list(struct etpan_message_list * msg_list,
    struct etpan_folder_list * folder_list)
{
  msg_list->folder_list = folder_list;
}

void etpan_message_list_set_activated(struct etpan_message_list * msg_list,
    struct etpan_tabbed_message_list * tabbed, int value)
{
  if (value)
    etpan_signal_add_handler(etpan_signal_manager_get_default(),
        ETPAN_TABBED_MESSAGE_LIST_SELECTIONCHANGED_SIGNAL,
        tabbed, msg_list, folder_selected_handler);
  else
    etpan_signal_remove_handler(etpan_signal_manager_get_default(),
        ETPAN_TABBED_MESSAGE_LIST_SELECTIONCHANGED_SIGNAL,
        tabbed, msg_list, folder_selected_handler);
}

/* data source */

static unsigned int get_n_columns(struct etpan_gtk_tree_data_source *
    datasource)
{
  (void) datasource;
  
  return 6;
}

static GType get_column_type(struct etpan_gtk_tree_data_source * datasource,
    unsigned int column_index)
{
  static GType column_types[] = {
    G_TYPE_STRING, G_TYPE_STRING,
    G_TYPE_STRING, G_TYPE_POINTER, G_TYPE_OBJECT, /*G_TYPE_UINT, */
    G_TYPE_STRING,
  };
  (void) datasource;
  
  return column_types[column_index];
}

static unsigned int get_children_count(struct etpan_gtk_tree_data_source *
    datasource, void * item)
{
  struct etpan_message_tree * tree;
  struct etpan_message_list * msg_list;
  
  msg_list = datasource->data;
  tree = item;
  if (tree == NULL)
    tree = msg_list->msg_tree;
  
  if (tree == NULL)
    return 0;
  
  if (tree->children == NULL)
    return 0;
  
  return carray_count(tree->children);
}

static int item_has_child(struct etpan_gtk_tree_data_source *
    datasource, void * item)
{
  return get_children_count(datasource, item);
}

static void * get_child_item(struct etpan_gtk_tree_data_source *
    datasource, void * item, unsigned int index)
{
  struct etpan_message_tree * tree;
  struct etpan_message_list * msg_list;
  
  msg_list = datasource->data;
  
  tree = item;
  if (tree == NULL)
    tree = msg_list->msg_tree;
  
  if (tree == NULL)
    return NULL;
  
  if (tree->children == NULL)
    return NULL;
  
  if (index >= carray_count(tree->children))
    return NULL;
  
  return carray_get(tree->children, index);
}

static void get_item_value_from(struct etpan_gtk_tree_data_source *
    datasource, void * item, GValue * value)
{
  struct etpan_message_tree * tree;
  char * from_str;
  struct etpan_message_list * msg_list;
  
  msg_list = datasource->data;
  
  tree = item;
  if (tree == NULL)
    tree = msg_list->msg_tree;
  
  from_str = NULL;
  
  if (tree != NULL) {
    struct etpan_message * msg;
    
    msg = etpan_message_tree_get_message(tree);
    if (msg != NULL) {
      struct etpan_message_header * header;
      
      header = etpan_message_get_header(msg);
      if (header != NULL) {
        carray * from_list;
        
        /* from */
        from_list = etpan_message_header_get_from(header);
        if (carray_count(from_list) > 0) {
          struct etpan_address * addr;
          char * display_name;
          char * email;
        
          addr = carray_get(from_list, 0);
          
          display_name = etpan_address_get_display_name(addr);
          email = etpan_address_get_address(addr);
          
          if (display_name != NULL)
            from_str = display_name;
          else if (email != NULL)
            from_str = email;
        }
      }
    }
  }
  
  if (from_str == NULL)
    from_str = "";
  
  g_value_init(value, G_TYPE_STRING);
  g_value_set_string(value, from_str);
}

    
static void get_item_value_subject(struct etpan_gtk_tree_data_source *
    datasource, void * item,
    GValue * value)
{
  char * subject_str;
  struct etpan_message_tree * tree;
  struct etpan_message_list * msg_list;
  
  msg_list = datasource->data;
  
  tree = item;
  if (tree == NULL)
    tree = msg_list->msg_tree;
  
  subject_str = NULL;
  
  if (tree != NULL) {
    struct etpan_message * msg;
    
    msg = etpan_message_tree_get_message(tree);
    if (msg != NULL) {
      struct etpan_message_header * header;
      
      header = etpan_message_get_header(msg);
      if (header != NULL) {
        /* subject */
        subject_str = etpan_message_header_get_subject(header);
      }
    }
  }
  
  if (subject_str == NULL)
    subject_str = "";
  
  g_value_init(value, G_TYPE_STRING);
  g_value_set_string(value, subject_str);
}

#define DATE_STR_MAXLEN 32

static void get_item_value_date(struct etpan_gtk_tree_data_source *
    datasource, void * item,
    GValue * value)
{
  struct etpan_message_tree * tree;
  char * date_str;
  struct etpan_message_list * msg_list;
  int alloc;
  
  alloc = 0;
  msg_list = datasource->data;
  
  tree = item;
  if (tree == NULL)
    tree = msg_list->msg_tree;
  
  date_str = NULL;
  
  if (tree != NULL) {
    struct etpan_message * msg;
    
    msg = etpan_message_tree_get_message(tree);
    
    if (msg != NULL) {
      struct etpan_message_header * header;
      
      header = etpan_message_get_header(msg);
      if (header != NULL) {
        time_t timestamp;
        
        timestamp = etpan_message_header_get_date(header);
        if (timestamp != (time_t) -1) {
          date_str = etpan_message_get_short_date_str(timestamp,
              msg_list->current_time);
          alloc = 1;
        }
      }
    }
  }
  
  if (date_str == NULL)
    date_str = "";
  
  g_value_init(value, G_TYPE_STRING);
  g_value_set_string(value, date_str);
  
  if (alloc)
    free(date_str);
}

static void get_item_value_msg(struct etpan_gtk_tree_data_source *
    datasource, void * item,
    GValue * value)
{
  struct etpan_message_tree * tree;
  struct etpan_message * msg;
  struct etpan_message_list * msg_list;
  
  msg_list = datasource->data;
  
  tree = item;
  if (tree == NULL)
    tree = msg_list->msg_tree;
  
  msg = NULL;
  
  if (tree != NULL) {
    msg = etpan_message_tree_get_message(tree);
  }
  
  g_value_init(value, G_TYPE_POINTER);
  g_value_set_pointer(value, msg);
}

static void get_item_value_image(struct etpan_gtk_tree_data_source *
    datasource, void * item,
    GValue * value)
{
  struct etpan_message_tree * tree;
  struct etpan_message_list * msg_list;
  GdkPixbuf * pixbuf;
  
  msg_list = datasource->data;
  
  tree = item;
  if (tree == NULL)
    tree = msg_list->msg_tree;
  
  pixbuf = NULL;
  
  if (tree != NULL) {
    struct etpan_message * msg;
    
    msg = etpan_message_tree_get_message(tree);
    
    if (msg != NULL) {
      struct etpan_message_flags * flags;
      
      flags = etpan_message_get_flags(msg);
      if (flags != NULL) {
        int value;
        
#define ICON(name) \
    etpan_icon_manager_get_scaled_pixbuf(etpan_icon_manager_get_default(), \
       name, 16)
        
        value = etpan_message_flags_get_value(flags);
        if ((value & ETPAN_FLAGS_FLAGGED) != 0)
          pixbuf = ICON("msg-flagged");
        else if ((value & ETPAN_FLAGS_SEEN) == 0)
          pixbuf = ICON("msg-unseen");
        else if ((value & ETPAN_FLAGS_ANSWERED) != 0)
          pixbuf = ICON("msg-replied-to");
        else if ((value & ETPAN_FLAGS_FORWARDED) != 0)
          pixbuf = ICON("msg-forwarded");
      }
      
#undef ICON
    }
  }
  
  g_value_init(value, G_TYPE_OBJECT);
  g_value_set_object(value, pixbuf);
}

static void get_item_value_color(struct etpan_gtk_tree_data_source *
    datasource, void * item,
    GValue * value)
{
  struct etpan_message_tree * tree;
  struct etpan_message_list * msg_list;
  char * color;
  int r;
  
  msg_list = datasource->data;
  
  tree = item;
  if (tree == NULL)
    tree = msg_list->msg_tree;

  color = "#ffffff";
  if (tree != NULL) {
    chashdatum key;
    chashdatum value;
    struct etpan_message * msg;
    
    /* custom color */
    msg = etpan_message_tree_get_message(tree);
    if (msg != NULL) {
      struct etpan_message_header * header;
      
      header = etpan_message_get_header(msg);
      if (header != NULL) {
        char * msgid;
        
        msgid = etpan_message_header_get_msgid(header);
        if (msgid != NULL) {
          char * custom_color;
          
          custom_color =
            etpan_message_color_get_color(etpan_message_color_get_default(),
                msgid);
          if (custom_color != NULL)
            color = custom_color;
        }
      }
    }
    
    /* selected thread */
    key.data = &tree;
    key.len = sizeof(tree);
    r = chash_get(msg_list->selected_thread, &key, &value);
    if (r == 0) {
      if (msg_list->selected_thread_count > 1) {
        color = "#c0c0ff";
      }
      else {
        color = "#f0f0ff";
      }
    }
  }
  
  g_value_init(value, G_TYPE_STRING);
  g_value_set_string(value, color);
}

static void get_item_value(struct etpan_gtk_tree_data_source *
    datasource, void * item, unsigned int column_index,
    GValue * value)
{
  switch (column_index) {
  case STORE_INDEX_FROM:
    return get_item_value_from(datasource, item, value);
  case STORE_INDEX_SUBJECT:
    return get_item_value_subject(datasource, item, value);
  case STORE_INDEX_DATE:
    return get_item_value_date(datasource, item, value);
  case STORE_INDEX_INFO:
    return get_item_value_msg(datasource, item, value);
  case STORE_INDEX_IMAGE:
    return get_item_value_image(datasource, item, value);
  case STORE_INDEX_COLOR:
    return get_item_value_color(datasource, item, value);
  }
}

void etpan_message_list_reply(struct etpan_message_list * msg_list,
    struct etpan_message_composer_window * composer_window)
{
  struct etpan_message * msg;
  struct etpan_folder * folder;
  struct etpan_storage * storage;
  struct etpan_account * account;
  GtkTreePath * path;
  GtkTreeViewColumn * column;
  char * selection_path;
  
  gtk_tree_view_get_cursor(GTK_TREE_VIEW(msg_list->treeview), &path, &column);
  selection_path = gtk_tree_path_to_string(path);
  msg = get_selection_data(msg_list, selection_path);
  if (msg == NULL)
    return;
  
  account = NULL;
  folder = etpan_message_get_folder(msg);
  storage = etpan_folder_get_storage(folder);
  account = etpan_storage_get_account(storage);
  etpan_message_composer_window_set_account(composer_window, account);
  
  etpan_message_composer_window_reply(composer_window, msg);
}

void etpan_message_list_forward(struct etpan_message_list * msg_list,
    struct etpan_message_composer_window * composer_window)
{
  struct etpan_message * msg;
  struct etpan_folder * folder;
  struct etpan_storage * storage;
  struct etpan_account * account;
  GtkTreePath * path;
  GtkTreeViewColumn * column;
  char * selection_path;
  
  gtk_tree_view_get_cursor(GTK_TREE_VIEW(msg_list->treeview), &path, &column);
  selection_path = gtk_tree_path_to_string(path);
  msg = get_selection_data(msg_list, selection_path);
  if (msg == NULL)
    return;
  
  account = NULL;
  folder = etpan_message_get_folder(msg);
  storage = etpan_folder_get_storage(folder);
  account = etpan_storage_get_account(storage);
  etpan_message_composer_window_set_account(composer_window, account);
  
  etpan_message_composer_window_forward(composer_window, msg);
}

static void add_thread(carray * tab, struct etpan_message_tree * tree)
{
  int r;
  unsigned int i;
  carray * children;
  
  r = carray_add(tab, tree, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  children = etpan_message_tree_get_children(tree);
  if (children == NULL)
    return;
  
  for(i = 0 ; i < carray_count(children) ; i ++) {
    struct etpan_message_tree * child;
    
    child = carray_get(children, i);
    add_thread(tab, child);
  }
}

static carray * get_selection(struct etpan_message_list * msg_list)
{
  GList * list;
  carray * tab;
  GList * cur;
  GtkTreeSelection * selection;
  int r;
  chash * marked_node;
  
  marked_node = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (marked_node == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  tab = carray_new(16);
  if (tab == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(msg_list->treeview));
  list = gtk_tree_selection_get_selected_rows(selection, NULL);
  
  for(cur = g_list_first(list) ; cur != NULL ; cur = g_list_next(cur)) {
    GtkTreePath * path;
    GtkTreeIter iter;
    
    path = cur->data;
    
    if (gtk_tree_model_get_iter(GTK_TREE_MODEL(msg_list->msglist_model),
            &iter, path)) {
      struct etpan_message_tree * tree;
      chashdatum key;
      chashdatum value;
      
      tree = etpan_gtk_tree_model_get_item(msg_list->msglist_model, &iter);

      key.data = &tree;
      key.len = sizeof(tree);
      r = chash_get(marked_node, &key, &value);
      if (r == 0)
        continue;
      
      if (msg_list->selected_thread_count > 1) {
        tree = get_root_thread(msg_list, tree);
        add_thread(tab, tree);
      }
      else {
        if ((gtk_tree_path_get_depth(path) == 1) && (!gtk_tree_view_row_expanded(GTK_TREE_VIEW(msg_list->treeview), path))) {
          add_thread(tab, tree);
        }
        else {
          r = carray_add(tab, tree, NULL);
          if (r < 0)
            ETPAN_LOG_MEMORY_ERROR;
        }
      }
      value.data = NULL;
      value.len = 0;
      r = chash_set(marked_node, &key, &value, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
    }
  }
  
  g_list_foreach(list, (GFunc) gtk_tree_path_free, NULL);
  g_list_free(list);
  
  chash_free(marked_node);
  
  return tab;
}

static void remove_message_from_treeview(struct etpan_message_list * msg_list,
    carray * selected);
static chash * get_selection_hash(carray * selected);

void etpan_message_list_delete(struct etpan_message_list * msg_list)
{
  carray * selected;
  chash * selection_hash;
  
  if (msg_list->folder == NULL)
    return;
  
  selected = get_selection(msg_list);
  selection_hash = get_selection_hash(selected);
  remove_message_from_treeview(msg_list, selected);
  etpan_mail_manager_delete_messages(etpan_mail_manager_get_default(),
      selection_hash);
}

static void mark_flag(struct etpan_message_list * msg_list, int flag_value)
{
  carray * selected;
  unsigned int i;
  int has_flag;
  int r;
  
  selected = get_selection(msg_list);

  ETPAN_MSGLIST_LOG("mark %i messages as read", carray_count(selected));
  has_flag = 0;
  for(i = 0 ; i < carray_count(selected) ; i ++) {
    struct etpan_message_tree * tree;
    struct etpan_message * msg;
    struct etpan_message_flags * flags;
    
    tree = carray_get(selected, i);
    msg = etpan_message_tree_get_message(tree);
    if (msg == NULL)
      continue;
    
    flags = etpan_message_get_flags(msg);
    if (flags != NULL) {
      int value;
      
      value = etpan_message_flags_get_value(flags);
      if ((value & flag_value) == 0)
        has_flag = 1;
    }
  }
  
  for(i = 0 ; i < carray_count(selected) ; i ++) {
    struct etpan_message_tree * tree;
    struct etpan_message * msg;
    struct etpan_message_flags * flags;
    int value;
    
    tree = carray_get(selected, i);
    msg = etpan_message_tree_get_message(tree);
    if (msg == NULL)
      continue;
    
    flags = etpan_message_get_flags(msg);
    flags = etpan_message_flags_dup(flags);
    value = etpan_message_flags_get_value(flags);
    if (has_flag)
      value |= flag_value;
    else
      value &= ~flag_value;
    etpan_message_flags_set_value(flags, value);
    etpan_message_set_flags(msg, flags);
      
    etpan_message_flags_free(flags);
  }
  
  gtk_widget_queue_draw(msg_list->treeview);
  
  /* check & update count */
  if (carray_count(selected) > 0) {
    chash * folder_hash;
    chashiter * iter;
    
    folder_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
    if (folder_hash == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    for(i = 0 ; i < carray_count(selected) ; i ++) {
      chashdatum key;
      chashdatum value;
      struct etpan_message_tree * tree;
      struct etpan_message * msg;
      struct etpan_folder * folder;
      
      tree = carray_get(selected, i);
      msg = etpan_message_tree_get_message(tree);
      if (msg == NULL)
        continue;
      
      folder = etpan_message_get_folder(msg);
      
      key.data = &folder;
      key.len = sizeof(folder);
      value.data = folder;
      value.len = 0;
      r = chash_set(folder_hash, &key, &value, NULL);
      if (r < 0)
        ETPAN_LOG_MEMORY_ERROR;
    }
    
    for(iter = chash_begin(folder_hash) ; iter != NULL ;
        iter = chash_next(folder_hash, iter)) {
      struct etpan_folder * folder;
      chashdatum value;
      
      chash_value(iter, &value);
      folder = value.data;
      
      check_msg_list(msg_list, folder);
    }
    
    chash_free(folder_hash);
  }
  
  carray_free(selected);
}

void etpan_message_list_mark_as_read(struct etpan_message_list * msg_list)
{
  return mark_flag(msg_list, ETPAN_FLAGS_SEEN);
}

void etpan_message_list_mark_as_flagged(struct etpan_message_list * msg_list)
{
  return mark_flag(msg_list, ETPAN_FLAGS_FLAGGED);
}

void etpan_message_list_mark_color(struct etpan_message_list * msg_list,
    char * color)
{
  carray * selected;
  unsigned int i;
  
  selected = get_selection(msg_list);

  ETPAN_MSGLIST_LOG("mark %i messages as colored", carray_count(selected));
  for(i = 0 ; i < carray_count(selected) ; i ++) {
    struct etpan_message_tree * tree;
    struct etpan_message * msg;
    struct etpan_message_header * header;
    
    tree = carray_get(selected, i);
    msg = etpan_message_tree_get_message(tree);
    if (msg == NULL)
      continue;
    
    header = etpan_message_get_header(msg);
    if (header != NULL) {
      char * msgid;
      
      msgid = etpan_message_header_get_msgid(header);
      if (msgid != NULL) {
        etpan_message_color_set_color(etpan_message_color_get_default(),
            msgid, color);
      }
    }
  }
  
  update_msg_list_treeview(msg_list);
  
  carray_free(selected);
}

GtkTreeViewColumn *
etpan_message_list_get_col_unread(struct etpan_message_list * msg_list)
{
  return msg_list->col_unread;
}

GtkTreeViewColumn *
etpan_message_list_get_col_from(struct etpan_message_list * msg_list)
{
  return msg_list->col_from;
}

GtkTreeViewColumn *
etpan_message_list_get_col_subject(struct etpan_message_list * msg_list)
{
  return msg_list->col_subject;
}

GtkTreeViewColumn *
etpan_message_list_get_col_date(struct etpan_message_list * msg_list)
{
  return msg_list->col_date;
}

void etpan_message_list_next_unread(struct etpan_message_list * msg_list)
{
  next_unread(msg_list);
}

/* search */

void etpan_message_list_search(struct etpan_message_list * msg_list,
    int filter_type, const char * text)
{
  const char * p;
  
  free(msg_list->search_filter_text);
  msg_list->search_filter_text = NULL;
  
  p = text;
  while (* p == ' ') {
    p ++;
  }
  if (p[0] == '\0') {
    msg_list->search_filter_set = 0;
    apply_search(msg_list);
    return;
  }
  
  msg_list->search_filter_type = filter_type;
  msg_list->search_filter_set = 1;
  msg_list->search_filter_text = strdup(text);
  search_update_progress(msg_list);
  
  apply_search(msg_list);
}


static void search_finished_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data);

static void apply_search(struct etpan_message_list * msg_list)
{
  struct etpan_folder_indexer * indexer;
  carray * empty;
  
  if (msg_list->folder_view == NULL) {
    gtk_widget_hide(msg_list->search_label);
    return;
  }
  
  if (!msg_list->search_filter_set) {
    gtk_widget_hide(msg_list->search_label);
    /* no filter */
    
#if 0
    etpan_folder_view_clear_filter(msg_list->folder_view);
#endif
    etpan_folder_view_cancel_search(msg_list->folder_view);
    search_update_progress(msg_list);
    return;
  }
  
  gtk_widget_show(msg_list->search_label);
  /* has filter */
  
#if 0
  indexer = etpan_mail_manager_get_indexer(etpan_mail_manager_get_default(),
      msg_list->folder);
#endif
  
  if (msg_list->searching) {
#if 0
    etpan_folder_indexer_cancel_search(indexer);
#endif
    etpan_folder_view_cancel_search(msg_list->folder_view);
  }
  
  msg_list->searching = 1;
  search_update_progress(msg_list);
  
#if 0
  etpan_folder_search_simple(indexer,
      msg_list->search_filter_type, msg_list->search_filter_text,
      apply_search_callback,
      msg_list);
#endif
  ETPAN_SIGNAL_ADD_HANDLER(msg_list->folder_view,
      ETPAN_FOLDER_VIEW_SEARCH_FINISHED, search_finished_handler,
      msg_list);
  
  etpan_folder_view_search_simple(msg_list->folder_view,
      msg_list->search_filter_type, msg_list->search_filter_text);
}

static void search_finished_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_message_list * msg_list;
  
  msg_list = user_data;

  ETPAN_SIGNAL_REMOVE_HANDLER(msg_list->folder_view,
      ETPAN_FOLDER_VIEW_SEARCH_FINISHED, search_finished_handler,
      msg_list);
  
  msg_list->searching = 0;
  search_update_progress(msg_list);
}

#if 0
static void apply_search_callback(int cancelled,
    struct etpan_folder_indexer_search_result * result, void * cb_data)
{
  struct etpan_message_list * msg_list;
  
  msg_list = cb_data;
  msg_list->searching = 0;
  search_update_progress(msg_list);
  
  if (cancelled)
    return;
  
  etpan_folder_view_set_filter(msg_list->folder_view, result->messages);
}
#endif

static void set_search_state(struct etpan_message_list * msg_list)
{
  int indexer_state;
  struct etpan_folder_indexer * indexer;
  int headers;
  int body;
  int update_needed;
  
#if 0
  indexer = etpan_mail_manager_get_indexer(etpan_mail_manager_get_default(),
      msg_list->folder);
  indexer_state = etpan_folder_indexer_indexing_state(indexer);
  
  headers = 0;
  body = 0;
  switch (indexer_state) {
  case ETPAN_FOLDER_INDEXER_STATE_IDLE:
    msg_list->indexing = 0;
    break;
  case ETPAN_FOLDER_INDEXER_STATE_INDEXING_HEADERS:
    msg_list->indexing = 1;
    headers = 1;
    break;
  case ETPAN_FOLDER_INDEXER_STATE_INDEXING_BODY:
    msg_list->indexing = 1;
    body = 1;
    break;
  }
#endif
  headers = 0;
  body = 0;
  msg_list->indexing = 0;
  if (etpan_folder_view_is_indexing(msg_list->folder_view)) {
    headers = 1;
    msg_list->indexing = 1;
  }
  if (etpan_folder_view_is_indexing_body(msg_list->folder_view)) {
    body = 1;
    msg_list->indexing = 1;
  }
  
  search_update_progress(msg_list);
  
  update_needed = 0;
  switch (msg_list->search_filter_type) {
  case ETPAN_FOLDER_INDEXER_SEARCH_TYPE_ALL:
    if (headers || body)
      update_needed = 1;
    break;
  case ETPAN_FOLDER_INDEXER_SEARCH_TYPE_FROM_AND_SUBJECT:
    if (headers)
      update_needed = 1;
    break;
  case ETPAN_FOLDER_INDEXER_SEARCH_TYPE_RECIPIENT_AND_SUBJECT:
    if (headers)
      update_needed = 1;
    break;
  case ETPAN_FOLDER_INDEXER_SEARCH_TYPE_FROM:
    if (headers)
      update_needed = 1;
    break;
  case ETPAN_FOLDER_INDEXER_SEARCH_TYPE_SUBJECT:
    if (headers)
      update_needed = 1;
    break;
  case ETPAN_FOLDER_INDEXER_SEARCH_TYPE_RECIPIENT:
    if (headers)
      update_needed = 1;
    break;
  case ETPAN_FOLDER_INDEXER_SEARCH_TYPE_BODY:
    if (body)
      update_needed = 1;
    break;
  }
  
  if (update_needed)
    apply_search(msg_list);
}

static void start_searching(struct etpan_message_list * msg_list)
{
  etpan_search_field_start_animation(msg_list->search_field);
}

static void stop_searching(struct etpan_message_list * msg_list)
{
  etpan_search_field_stop_animation(msg_list->search_field);
}

static void search_setup(struct etpan_message_list * msg_list)
{
  gtk_widget_hide(msg_list->search_label);
}

static void search_unsetup(struct etpan_message_list * msg_list)
{
}

static void search_update_progress(struct etpan_message_list * msg_list)
{
  int previous_value;
  
  previous_value = msg_list->search_has_progress;
  
  msg_list->search_has_progress = 0;
  if (msg_list->search_filter_set) {
    if (msg_list->searching) {
      msg_list->search_has_progress = 1;
    }
    else if (msg_list->indexing) {

#if 0
      int indexer_state;
      struct etpan_folder_indexer * indexer;
        
      indexer = etpan_mail_manager_get_indexer(etpan_mail_manager_get_default(),
          msg_list->folder);
      indexer_state = etpan_folder_indexer_indexing_state(indexer);
      
      if ((msg_list->search_filter_type == ETPAN_FOLDER_INDEXER_SEARCH_TYPE_ALL) && (indexer_state == ETPAN_FOLDER_INDEXER_STATE_INDEXING_BODY)) {
        msg_list->search_has_progress = 1;
      }
      else if (indexer_state == ETPAN_FOLDER_INDEXER_STATE_INDEXING_HEADERS) {
        msg_list->search_has_progress = 1;
      }
#endif
      if ((msg_list->search_filter_type == ETPAN_FOLDER_INDEXER_SEARCH_TYPE_ALL) && (etpan_folder_view_is_indexing_body(msg_list->folder_view))) {
        msg_list->search_has_progress = 1;
      }
      else if (etpan_folder_view_is_indexing(msg_list->folder_view)) {
        msg_list->search_has_progress = 1;
      }
    }
    else if (msg_list->filtering_search_result) {
      msg_list->search_has_progress = 1;
    }
  }
  
  if (previous_value != msg_list->search_has_progress) {
    if (msg_list->search_has_progress)
      start_searching(msg_list);
    else
      stop_searching(msg_list);
  }
}

/* drag and drop */


static void drag_end_handler(GtkWidget * widget,
    GdkDragContext * drag_context,
    gpointer user_data)
{
  struct etpan_message_list * msg_list;
  (void) widget;
  (void) drag_context;
  
  msg_list = user_data;
  drag_end(msg_list);
}

  
static void drag_row_expanded_handler(GtkTreeView * treeview,
    GtkTreeIter * arg1,
    GtkTreePath * arg2,
    gpointer user_data)
{
  struct etpan_message_list * msg_list;
  (void) treeview;
  (void) arg1;
  (void) arg2;
  
  msg_list = user_data;
  msg_list->drag_expanded = 1;
}

static void drag_row_collapsed_handler(GtkTreeView * treeview,
    GtkTreeIter * arg1,
    GtkTreePath * arg2,
    gpointer user_data)
{
  struct etpan_message_list * msg_list;
  (void) treeview;
  (void) arg1;
  (void) arg2;
  
  msg_list = user_data;
  msg_list->drag_expanded = 1;
}

static gboolean drag_button_press_handler(GtkWidget * widget,
    GdkEventButton * event,
    gpointer user_data)
{
  struct etpan_message_list * msg_list;
  GtkTreePath * path;
  int do_free_path;
  gboolean result;
  GtkTreeView * tree_view;
  
  msg_list = user_data;
  
  if (GTK_TREE_VIEW(msg_list->treeview)->priv->bin_window != event->window)
    return FALSE;
  
  result = FALSE;
  do_free_path = 0;
  if (!result) {
    if (event->button == 1) {
      if (gtk_tree_view_get_path_at_pos(GTK_TREE_VIEW(msg_list->treeview),
              event->x, event->y, &path, NULL, NULL, NULL)) {
        struct etpan_message_tree * tree;
        chashdatum key;
        chashdatum value;
        GtkTreeIter iter;
        unsigned int i;
        int is_selected;
        carray * selection_tab;
        
        do_free_path = 1;
        gtk_tree_model_get_iter(GTK_TREE_MODEL(msg_list->msglist_model),
            &iter, path);
        tree = etpan_gtk_tree_model_get_item(msg_list->msglist_model, &iter);
        
        is_selected = 0;
        selection_tab = get_selection(msg_list);
        for(i = 0 ; i < carray_count(selection_tab) ; i ++) {
          if (carray_get(selection_tab, i) == tree)
            is_selected = 1;
        }
        carray_free(selection_tab);
        /* if (gtk_tree_selection_path_is_selected(selection, path)) { */
        if (is_selected) {
          ETPAN_LOG("start drag");
          drag_may_start(msg_list, event->x, event->y, event->time);
          result = TRUE;
        }
        else if ((event->state & (GDK_SHIFT_MASK | GDK_CONTROL_MASK)) != 0) {
          /* do nothing */
        }
        else {
          drag_may_start(msg_list, event->x, event->y, event->time);
        }
      }
    }
    
    tree_view = GTK_TREE_VIEW(msg_list->treeview);
    /* extracted from GTK - begin */
    if (tree_view->priv->prelight_node &&
        GTK_TREE_VIEW_FLAG_SET (tree_view, GTK_TREE_VIEW_ARROW_PRELIT))
    {
      if (event->button == 1)
      {
        gtk_grab_add (widget);
        tree_view->priv->button_pressed_node = tree_view->priv->prelight_node;
        tree_view->priv->button_pressed_tree = tree_view->priv->prelight_tree;
#if 0 /* disabled by DVH */
        gtk_tree_view_draw_arrow (GTK_TREE_VIEW (widget),
            tree_view->priv->prelight_tree,
            tree_view->priv->prelight_node,
            event->x,
            event->y);
#endif
      }

#if 0 /* disabled by DVH */
      grab_focus_and_unset_draw_keyfocus (tree_view);
#endif
      return TRUE;
    }
    /* extracted from GTK - end */
  }
  
  if (do_free_path)
    gtk_tree_path_free(path);
  
  return result;
}

static gboolean drag_button_release_handler(GtkWidget * widget,
    GdkEventButton * event,
    gpointer user_data)
{
  struct etpan_message_list * msg_list;
  (void) widget;
  (void) event;
  
  msg_list = user_data;
  
  if (drag_may_be_starting(msg_list)) {
    drag_cancel_may_start(msg_list);
  }
  
  return FALSE;
}

static gboolean drag_motion_handler(GtkWidget * widget,
    GdkEventMotion * event,
    gpointer user_data)
{
  struct etpan_message_list * msg_list;
  (void) widget;
  
  msg_list = user_data;
  
  if (drag_may_be_starting(msg_list)) {
    /* check if real drag can start */
    if (drag_is_valid(msg_list, event->x, event->y, event->time)) {
      drag_start(msg_list, (GdkEvent *) event);
    }
  }
  
  return FALSE;
}

static void drag_begin_handler(GtkWidget * widget,
    GdkDragContext * drag_context,
    gpointer user_data)
{
  struct etpan_message_list * msg_list;
  (void) widget;
  
  msg_list = user_data;
  
  msg_list->drag_context = drag_context;
  drag_set_icon(msg_list);
}

static void move_msglist_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_message_list * msg_list;
  struct etpan_folder * folder;
  (void) signal_name;
  (void) signal_data;
  (void) sender;
  
  msg_list = user_data;
  if (!msg_list->drag_started)
    return;
  
  folder = signal_data;
  ETPAN_LOG("move message to %s", etpan_folder_get_ui_path(folder));
  etpan_message_list_move(msg_list, folder);
}

static void copy_msglist_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_message_list * msg_list;
  struct etpan_folder * folder;
  (void) signal_name;
  (void) sender;
  
  msg_list = user_data;
  if (!msg_list->drag_started)
    return;
  
  folder = signal_data;
  ETPAN_LOG("copy message to %s", etpan_folder_get_ui_path(folder));
  etpan_message_list_copy(msg_list, folder);
}

static void drag_drop_setup(struct etpan_message_list * msg_list)
{
  msg_list->drag_button_press_signal_id =
    g_signal_connect(msg_list->treeview,
        "button-press-event", G_CALLBACK(drag_button_press_handler),
        (gpointer) msg_list);
  
  msg_list->drag_motion_signal_id =
    g_signal_connect(msg_list->treeview,
        "motion-notify-event", G_CALLBACK(drag_motion_handler),
        (gpointer) msg_list);
  
  msg_list->drag_button_release_signal_id =
    g_signal_connect(msg_list->treeview,
        "button-release-event", G_CALLBACK(drag_button_release_handler),
        (gpointer) msg_list);
  
  msg_list->drag_begin_signal_id =
    g_signal_connect(msg_list->treeview,
        "drag-begin", G_CALLBACK(drag_begin_handler),
        (gpointer) msg_list);
  msg_list->drag_end_signal_id =
    g_signal_connect(msg_list->treeview,
        "drag-end", G_CALLBACK(drag_end_handler),
        (gpointer) msg_list);
  msg_list->drag_row_expanded_signal_id =
    g_signal_connect(msg_list->treeview,
        "row-expanded", G_CALLBACK(drag_row_expanded_handler),
        (gpointer) msg_list);
  msg_list->drag_row_collapsed_signal_id =
    g_signal_connect(msg_list->treeview,
        "row-collapsed", G_CALLBACK(drag_row_collapsed_handler),
        (gpointer) msg_list);
  
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_FOLDER_LIST_DRAG_MOVE_MSGLIST_SIGNAL, NULL /* sender */,
      msg_list, move_msglist_handler);
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_FOLDER_LIST_DRAG_COPY_MSGLIST_SIGNAL, NULL /* sender */,
      msg_list, copy_msglist_handler);
  
  msg_list->drag_may_start = 0;
  msg_list->drag_started = 0;
  msg_list->drag_timer_set = 0;
}

static void drag_drop_unsetup(struct etpan_message_list * msg_list)
{
  if (msg_list->drag_timer_set)
    g_source_remove(msg_list->drag_timer_id);
  
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_FOLDER_LIST_DRAG_COPY_MSGLIST_SIGNAL, NULL /* sender */,
      msg_list, copy_msglist_handler);
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_FOLDER_LIST_DRAG_MOVE_MSGLIST_SIGNAL, NULL /* sender */,
      msg_list, move_msglist_handler);
  
  g_signal_handler_disconnect(msg_list->treeview,
      msg_list->drag_row_collapsed_signal_id);
  g_signal_handler_disconnect(msg_list->treeview,
      msg_list->drag_row_expanded_signal_id);
  g_signal_handler_disconnect(msg_list->treeview,
      msg_list->drag_end_signal_id);
  g_signal_handler_disconnect(msg_list->treeview,
      msg_list->drag_begin_signal_id);

  g_signal_handler_disconnect(msg_list->treeview,
      msg_list->drag_button_press_signal_id);
  g_signal_handler_disconnect(msg_list->treeview,
      msg_list->drag_button_release_signal_id);
  g_signal_handler_disconnect(msg_list->treeview,
      msg_list->drag_motion_signal_id);
}

static void drag_may_start(struct etpan_message_list * msg_list,
    gdouble x, gdouble y, guint32 time)
{
  (void) time;
  
  msg_list->drag_may_start = 1;
  msg_list->drag_may_start_x = x;
  msg_list->drag_may_start_y = y;
  msg_list->drag_expanded = 0;
}

static gboolean drag_cancel_may_start_handler(gpointer data)
{
  struct etpan_message_list * msg_list;
  GtkTreePath * path;
  
  msg_list = data;
  msg_list->drag_timer_set = 0;
  
  if (msg_list->drag_expanded)
    return FALSE;
  
  if (gtk_tree_view_get_path_at_pos(GTK_TREE_VIEW(msg_list->treeview),
          msg_list->drag_may_start_x, msg_list->drag_may_start_y,
          &path, NULL, NULL, NULL)) {
    gtk_tree_view_set_cursor(GTK_TREE_VIEW(msg_list->treeview),
        path, NULL, 0);
    gtk_tree_path_free(path);
  }
  
  return FALSE;
}

static void drag_cancel_may_start(struct etpan_message_list * msg_list)
{
  msg_list->drag_may_start = 0;

  msg_list->drag_timer_set = 1;
  msg_list->drag_timer_id = g_timeout_add(0,
      drag_cancel_may_start_handler, msg_list);
}

static int drag_may_be_starting(struct etpan_message_list * msg_list)
{
  return msg_list->drag_may_start;
}

static int drag_is_valid(struct etpan_message_list * msg_list,
    gdouble x, gdouble y, guint32 time)
{
  (void) time;
  
  return gtk_drag_check_threshold(msg_list->treeview,
      msg_list->drag_may_start_x,
      msg_list->drag_may_start_y,
      x, y);
}

static void drag_start(struct etpan_message_list * msg_list,
    GdkEvent * event)
{
  GtkTargetList * targets;
  GdkAtom atom;
  
  msg_list->drag_may_start = 0;
  msg_list->drag_started = 1;
  
  ETPAN_LOG("drag start");
  targets = gtk_target_list_new(NULL, 0);
  atom = gdk_atom_intern(ETPAN_MESSAGE_LIST_DRAG_MSGLIST_NAME, 0);
  gtk_target_list_add(targets, atom, GTK_TARGET_SAME_APP, 0);
  msg_list->drag_context = gtk_drag_begin(msg_list->treeview,
      targets,
      GDK_ACTION_COPY | GDK_ACTION_MOVE,
      1, event);
  
  gtk_target_list_unref(targets);
  ETPAN_LOG("drag ok");
}

static void drag_set_icon(struct etpan_message_list * msg_list)
{
  GdkPixbuf * pixbuf;
  
  /* must be called on drag-begin signal */
  pixbuf = etpan_icon_manager_get_scaled_pixbuf(etpan_icon_manager_get_default(),
      "drag-msglist", 32);
  gtk_drag_set_icon_pixbuf(msg_list->drag_context,
      pixbuf, 0, 0);
}

static void drag_end(struct etpan_message_list * msg_list)
{
  msg_list->drag_context = NULL;
  msg_list->drag_started = 0;
}

/* move */

void etpan_message_list_move(struct etpan_message_list * msg_list,
    struct etpan_folder * folder)
{
  copy_with_delete(msg_list, folder, 1);
}

void etpan_message_list_copy(struct etpan_message_list * msg_list,
    struct etpan_folder * folder)
{
  copy_with_delete(msg_list, folder, 0);
}

static chash * get_selection_hash(carray * selected)
{
  chash * selection_hash;
  unsigned int i;
  int r;
  
  selection_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (selection_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  for(i = 0 ; i < carray_count(selected) ; i ++) {
    struct etpan_message_tree * tree;
    struct etpan_message * msg;
    char * uid;
    chashdatum key;
    chashdatum value;
    
    tree = carray_get(selected, i);
    msg = etpan_message_tree_get_message(tree);
    uid = etpan_message_get_uid(msg);
    key.data = uid;
    key.len = strlen(uid) + 1;
    value.data = msg;
    value.len = 0;
    
    r = chash_set(selection_hash, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  return selection_hash;
}

static void remove_message_from_treeview(struct etpan_message_list * msg_list,
    carray * selected)
{
  chash * selected_hash;
  unsigned int i;
  int r;
  
  selected_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (selected_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  for(i = 0 ; i < carray_count(selected) ; i ++) {
    struct etpan_message_tree * tree;
    chashdatum key;
    chashdatum value;
    
    tree = carray_get(selected, i);
    key.data = &tree;
    key.len = sizeof(tree);
    value.data = NULL;
    value.len = 0;
    
    r = chash_set(selected_hash, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
    
  etpan_message_tree_remove_child_list(msg_list->msg_tree, selected_hash);
  
  update_msg_list_treeview(msg_list);
  notify_message_selection(msg_list);
    
  chash_free(selected_hash);
}

static void copy_with_delete(struct etpan_message_list * msg_list,
    struct etpan_folder * folder, int delete)
{
  carray * selected;
  unsigned int i;
  int r;
  chash * selection_hash;
  
  selected = get_selection(msg_list);
  selection_hash = get_selection_hash(selected);
  
  if (delete)
    ETPAN_LOG("move %i messages", carray_count(selected));
  else 
    ETPAN_LOG("copy %i messages", carray_count(selected));
  
  if (delete) {
    remove_message_from_treeview(msg_list, selected);
  }
  
  etpan_folder_copy_messages_with_delete(msg_list->folder_list,
      folder, selection_hash, delete);
  
  chash_free(selection_hash); 
  carray_free(selected);
}
