#include "etpan-message-view.h"

#include <stdlib.h>
#include <math.h>
#include <gdk/gdkkeysyms.h>
#include <libgen.h>
#include <sys/types.h>
#include <sys/wait.h>

#include "etpan-backend.h"
#include "etpan-status-bar.h"
#include "etpan-lep.h"
#include "etpan-message-list.h"
#include "etpan-contextual-menu.h"
#include "etpan-message-composer-window.h"
#include "etpan-ui-config.h"

#define DEFAULT_INCOMING_CHARSET "iso-8859-1"

enum {
  LINK_TYPE_NONE,
  LINK_TYPE_ADDRESS,
  LINK_TYPE_URL,
};

struct link_elt {
  int link_type;
  char * address_info;
  GtkTextBuffer * textbuffer;
  GtkTextMark * start_mark;
  GtkTextMark * end_mark;
  int is_over;
};

static struct link_elt * link_new(void)
{
  struct link_elt * link;
  
  link = malloc(sizeof(* link));
  if (link == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  link->link_type = LINK_TYPE_NONE;
  link->address_info = NULL;
  link->textbuffer = NULL;
  link->start_mark = NULL;
  link->end_mark = NULL;
  link->is_over = 0;
  
  return link;
}

static void link_free(struct link_elt * link)
{
  if (link->start_mark != NULL) {
    gtk_text_buffer_delete_mark(link->textbuffer, link->end_mark);
    gtk_text_buffer_delete_mark(link->textbuffer, link->start_mark);
  }
  free(link->address_info);
  free(link);
}

static void link_set(struct link_elt * link,
    int type, char * address_info,
    GtkTextBuffer * textbuffer,
    GtkTextMark * start_mark, GtkTextMark * end_mark)
{
  (void) type;
  
  link->link_type = LINK_TYPE_ADDRESS;
  link->address_info = address_info;
  link->textbuffer = textbuffer;
  link->start_mark = start_mark;
  link->end_mark = end_mark;
}

static int link_is_over(struct link_elt * link, GtkTextIter * iter)
{
  GtkTextIter start_iter;
  GtkTextIter end_iter;
  
  gtk_text_buffer_get_iter_at_mark(link->textbuffer, &start_iter,
      link->start_mark);
  gtk_text_buffer_get_iter_at_mark(link->textbuffer, &end_iter,
      link->end_mark);
  
  if (gtk_text_iter_compare(&start_iter, iter) > 0)
    return 0;
  
  if (gtk_text_iter_compare(iter, &end_iter) >= 0)
    return 0;
  
  return 1;
}

static void link_set_over(struct link_elt * link, int is_over)
{
  GtkTextIter start_iter;
  GtkTextIter end_iter;
  
  gtk_text_buffer_get_iter_at_mark(link->textbuffer, &start_iter,
      link->start_mark);
  gtk_text_buffer_get_iter_at_mark(link->textbuffer, &end_iter,
      link->end_mark);
  
  if (is_over) {
    if (!link->is_over) {
      gtk_text_buffer_remove_tag_by_name(link->textbuffer, "header-value",
          &start_iter, &end_iter);
      gtk_text_buffer_apply_tag_by_name(link->textbuffer, "header-value-over",
          &start_iter, &end_iter);
      link->is_over = 1;
    }
  }
  else {
    if (link->is_over) {
      gtk_text_buffer_remove_tag_by_name(link->textbuffer, "header-value-over",
          &start_iter, &end_iter);
      gtk_text_buffer_apply_tag_by_name(link->textbuffer, "header-value",
          &start_iter, &end_iter);
      link->is_over = 0;
    }
  }
}

static void start_progress_animation(struct etpan_message_view * msg_view);
static void stop_progress_animation(struct etpan_message_view * msg_view);

static void update_font(struct etpan_message_view * msg_view);

struct etpan_message_view * etpan_message_view_new(void)
{
  GtkWidget * textview;
  GtkWidget * scrolledwindow;
  struct etpan_message_view * msg_view;
  GtkTextBuffer * textbuffer;
  
  msg_view = malloc(sizeof(* msg_view));
  if (msg_view == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  scrolledwindow = gtk_scrolled_window_new(NULL, NULL);
  gtk_widget_show(scrolledwindow);
  gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrolledwindow),
      GTK_SHADOW_IN);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledwindow),
      GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
  
  textview = gtk_text_view_new();
  gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(textview), GTK_WRAP_WORD_CHAR);
  gtk_text_view_set_editable(GTK_TEXT_VIEW(textview), FALSE);
  gtk_widget_show(textview);
  gtk_container_add(GTK_CONTAINER(scrolledwindow), textview);
  
  textbuffer = gtk_text_buffer_new(NULL);
  
  gtk_text_buffer_create_tag(textbuffer, "normal-over",
      NULL);
  
  gtk_text_buffer_create_tag(textbuffer, "fixed",
      NULL);
  
  gtk_text_buffer_create_tag(textbuffer, "fixed-over",
      NULL);
  
  gtk_text_buffer_create_tag(textbuffer, "header-value",
      NULL);
  
  gtk_text_buffer_create_tag(textbuffer, "header-value-over",
      NULL);
  
  gtk_text_buffer_create_tag(textbuffer, "header-name",
      NULL);
  
  gtk_text_buffer_create_tag(textbuffer, "header-name-over",
      NULL);

  gtk_text_view_set_buffer(GTK_TEXT_VIEW(textview), textbuffer);
  
  msg_view->scrolledwindow = scrolledwindow;
  msg_view->textview = textview;
  msg_view->textbuffer = textbuffer;
  
  msg_view->msg_list = NULL;
  
  msg_view->status_bar = NULL;
  msg_view->message = NULL;
  msg_view->fetcher = etpan_message_fetcher_new();
  
  msg_view->parts_list = carray_new(16);
  if (msg_view->parts_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  msg_view->links_list = carray_new(16);
  if (msg_view->links_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  msg_view->mode = ETPAN_MESSAGE_VIEW_MODE_NORMAL;
  msg_view->fetcher_mode = 0;
  
  msg_view->save_op_list = carray_new(4);
  if (msg_view->save_op_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  msg_view->separator = NULL;

  return msg_view;
}

static void update_font(struct etpan_message_view * msg_view)
{
  GtkTextTagTable * table;
  PangoFontDescription * font_desc;
  GtkTextTag * tag;
  
  font_desc = pango_font_description_from_string(etpan_ui_config_get_font_other(etpan_ui_config_get_default()));
  gtk_widget_modify_font(msg_view->textview, font_desc);
  pango_font_description_free(font_desc);
  
  table = gtk_text_buffer_get_tag_table(msg_view->textbuffer);
  
  tag = gtk_text_tag_table_lookup(table, "normal-over");
  g_object_set(tag,
      "background", "#f0f0ff",
      NULL);
  
  tag = gtk_text_tag_table_lookup(table, "fixed");
  g_object_set(tag,
      "font", etpan_ui_config_get_font_message(etpan_ui_config_get_default()),
      NULL);
  
  tag = gtk_text_tag_table_lookup(table, "fixed-over");
  g_object_set(tag,
      "font", etpan_ui_config_get_font_message(etpan_ui_config_get_default()),
      "background", "#f0f0ff",
      NULL);
  
  tag = gtk_text_tag_table_lookup(table, "header-value");
  g_object_set(tag,
      "font", etpan_ui_config_get_font_other(etpan_ui_config_get_default()),
      "foreground", "#000000",
      NULL);
  
  tag = gtk_text_tag_table_lookup(table, "header-value-over");
  g_object_set(tag,
      "font", etpan_ui_config_get_font_other(etpan_ui_config_get_default()),
      "foreground", "#000000",
      "background", "#f0f0ff",
      NULL);
  
  tag = gtk_text_tag_table_lookup(table, "header-name");
  g_object_set(tag,
      "font", etpan_ui_config_get_font_other(etpan_ui_config_get_default()),
      "foreground", "#606060",
      NULL);
  
  tag = gtk_text_tag_table_lookup(table, "header-name-over");
  g_object_set(tag,
      "font", etpan_ui_config_get_font_other(etpan_ui_config_get_default()),
      "foreground", "#f0f0ff",
      NULL);
}

void etpan_message_view_free(struct etpan_message_view * msg_view)
{
  carray_free(msg_view->save_op_list);
  carray_free(msg_view->links_list);
  carray_free(msg_view->parts_list);
#if 0
  etpan_message_fetcher_unsetup(msg_view->fetcher);
#endif
  etpan_message_fetcher_unref(msg_view->fetcher);
  if (msg_view->textbuffer != NULL)
    g_object_unref(msg_view->textbuffer);
  gtk_widget_destroy(msg_view->textview);
  gtk_widget_destroy(msg_view->scrolledwindow);
  free(msg_view);
}

static void font_changed_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_message_view * msg_view;
  (void) signal_name;
  (void) sender;
  (void) signal_data;
  
  msg_view = user_data;
  update_font(msg_view);
}


enum {
  PART_INFO_TYPE_NONE,
  PART_INFO_TYPE_HEADER,
  PART_INFO_TYPE_FILE,
  PART_INFO_TYPE_IMAGE,
  PART_INFO_TYPE_TEXT,
};

struct part_info {
  int type;
  struct etpan_part_fetch_info * fetch_info;
  int is_over;
  GtkTextBuffer * textbuffer;
  GtkTextMark * start_mark;
  GtkTextMark * end_mark;
};

static struct part_info * part_info_new(void)
{
  struct part_info * info;
  
  info = malloc(sizeof(* info));
  if (info == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  info->is_over = 0;
  info->fetch_info = NULL;
  info->type = PART_INFO_TYPE_NONE;
  info->textbuffer = NULL;
  info->start_mark = NULL;
  info->end_mark = NULL;
  
  return info;
}

static void part_info_set_part(struct part_info * info,
    struct etpan_part_fetch_info * fetch_info)
{
  info->fetch_info = fetch_info;
}

static void part_info_set_type(struct part_info * info, int type)
{
  info->type = type;
}

static void part_info_set_mark(struct part_info * info,
    GtkTextBuffer * textbuffer,
    GtkTextMark * start_mark, GtkTextMark * end_mark)
{
  info->textbuffer = textbuffer;
  info->start_mark = start_mark;
  info->end_mark = end_mark;
}

static void part_info_free(struct part_info * info)
{
  if (info->start_mark != NULL) {
    gtk_text_buffer_delete_mark(info->textbuffer, info->start_mark);
    gtk_text_buffer_delete_mark(info->textbuffer, info->end_mark);
  }
  free(info);
}

static int part_info_is_over(struct part_info * info, GtkTextIter * iter)
{
  GtkTextIter start_iter;
  GtkTextIter end_iter;
  
  gtk_text_buffer_get_iter_at_mark(info->textbuffer, &start_iter,
      info->start_mark);
  gtk_text_buffer_get_iter_at_mark(info->textbuffer, &end_iter,
      info->end_mark);
  
  if (gtk_text_iter_compare(&start_iter, iter) > 0)
    return 0;
  
  if (gtk_text_iter_compare(iter, &end_iter) >= 0)
    return 0;
  
  return 1;
}

static void part_info_set_over(struct part_info * info, int is_over)
{
  GtkTextIter start_iter;
  GtkTextIter end_iter;
  
  if (info->type != PART_INFO_TYPE_FILE)
    return;
  
  gtk_text_buffer_get_iter_at_mark(info->textbuffer, &start_iter,
      info->start_mark);
  gtk_text_buffer_get_iter_at_mark(info->textbuffer, &end_iter,
      info->end_mark);
  
  if (is_over) {
    if (!info->is_over) {
      gtk_text_buffer_apply_tag_by_name(info->textbuffer, "normal-over",
          &start_iter, &end_iter);
      info->is_over = 1;
    }
  }
  else {
    if (info->is_over) {
      gtk_text_buffer_remove_tag_by_name(info->textbuffer, "normal-over",
          &start_iter, &end_iter);
      info->is_over = 0;
    }
  }
}

void etpan_message_view_set_status_bar(struct etpan_message_view * msg_view,
    struct etpan_status_bar * status_bar)
{
  msg_view->status_bar = status_bar;
}

GtkWidget * etpan_message_view_get_main_widget(struct etpan_message_view *
    msg_view)
{
  return msg_view->scrolledwindow;
}

void etpan_message_view_set_message(struct etpan_message_view * msg_view,
    struct etpan_message * msg)
{
  GtkTextIter start_iter;
  GtkTextIter end_iter;
  int fetcher_flags;
  
  if (msg_view->message == msg)
    return;
  
  msg_view->separator = NULL;
  gtk_text_buffer_get_start_iter(msg_view->textbuffer,
      &start_iter);
  gtk_text_buffer_get_end_iter(msg_view->textbuffer,
      &end_iter);
  gtk_text_buffer_delete(msg_view->textbuffer,
      &start_iter, &end_iter);
  
  if (msg_view->message != NULL) {
    unsigned int i;
    struct etpan_folder * folder;
    
    /* free link info */
    for(i = 0 ; i < carray_count(msg_view->links_list) ; i ++) {
      struct link_elt * link;
      
      link = carray_get(msg_view->links_list, i);
      link_free(link);
    }
    carray_set_size(msg_view->links_list, 0);
    
    /* free part info */
    for(i = 0 ; i < carray_count(msg_view->parts_list) ; i ++) {
      struct part_info * info;
      
      info = carray_get(msg_view->parts_list, i);
      part_info_free(info);
    }
    carray_set_size(msg_view->parts_list, 0);
    
    folder = etpan_message_get_folder(msg_view->message);
    etpan_message_unref(msg_view->message);
    etpan_folder_unref_msg_list(folder);
  }
  
  msg_view->message = msg;
  if (msg != NULL) {
    struct etpan_folder * folder;
    
    folder = etpan_message_get_folder(msg);
    etpan_folder_ref_msg_list(folder);
    etpan_message_ref(msg);
  }
  
  msg_view->fetcher_mode = msg_view->mode;
  switch (msg_view->mode) {
  case ETPAN_MESSAGE_VIEW_MODE_NORMAL:
    fetcher_flags = ETPAN_MESSAGE_FETCHER_FLAGS_IMAGE |
      ETPAN_MESSAGE_FETCHER_FLAGS_DECODE_ATTACHMENT_CHARSET;
    etpan_message_fetcher_set_flags(msg_view->fetcher, fetcher_flags);
    break;
  case ETPAN_MESSAGE_VIEW_MODE_SOURCE:
    fetcher_flags = ETPAN_MESSAGE_FETCHER_FLAGS_SOURCE;
    etpan_message_fetcher_set_flags(msg_view->fetcher, fetcher_flags);
    break;
  }
  
  etpan_message_fetcher_set_message(msg_view->fetcher, msg);
  
  if (msg != NULL) {
    msg_view->parts_fetched = 0;
    etpan_message_fetcher_run(msg_view->fetcher);
    start_progress_animation(msg_view);
  }
  else {
#if 0
    etpan_message_fetcher_unsetup(msg_view->fetcher);
#endif
  }
}

#if 0
static int header_displayed(struct etpan_message_view * msg_view, char * name)
{
  if ((strcasecmp(name, "From") == 0) ||
      (strcasecmp(name, "Reply-To") == 0) ||
      (strcasecmp(name, "To") == 0) ||
      (strcasecmp(name, "Cc") == 0) ||
      (strcasecmp(name, "Bcc") == 0) ||
      (strcasecmp(name, "Date") == 0) ||
      (strcasecmp(name, "Subject") == 0))
    return 0;
  
  if ((strcasecmp(name, "X-Mailer") == 0) ||
      (strcasecmp(name, "X-Newsreader") == 0) ||
      (strcasecmp(name, "User-Agent") == 0)) {
    return 1;
  }
  
  return 0;
}
#endif

static void show_address_list(struct etpan_message_view * msg_view,
    GtkTextBuffer * textbuffer,
    GtkTextIter * iter,
    carray * address_list)
{
  unsigned int i;
  int first;
  int r;
  
  first = 1;
  for(i = 0 ; i < carray_count(address_list) ; i ++) {
    char * name;
    char * mail;
    struct etpan_address * address;
    GtkTextMark * start_mark;
    GtkTextMark * end_mark;
    struct link_elt * link;
    
    address = carray_get(address_list, i);
    name = etpan_address_get_display_name(address);
    mail = etpan_address_get_address(address);
    
    if (first)
      first = 0;
    else
      gtk_text_buffer_insert_with_tags_by_name(textbuffer, iter, ", ", -1,
          "header-value", NULL);
    
    start_mark = gtk_text_buffer_create_mark(textbuffer, NULL,
        iter, TRUE);
    if (name != NULL) {
      gtk_text_buffer_insert_with_tags_by_name(textbuffer, iter, name, -1,
          "header-value", NULL);
    }
    else if (mail != NULL) {
      gtk_text_buffer_insert_with_tags_by_name(textbuffer, iter, mail, -1,
          "header-value", NULL);
    }
    end_mark = gtk_text_buffer_create_mark(textbuffer, NULL,
        iter, TRUE);
    
    link = link_new();
    
    link_set(link, LINK_TYPE_ADDRESS, strdup(mail),
        textbuffer, start_mark, end_mark);
    
    r = carray_add(msg_view->links_list, link, NULL);
    if (r < 0) {
      link_free(link);
      ETPAN_LOG_MEMORY_ERROR;
    }
  }
}

#define TYPOGRAPHY_WRAP_COUNT 66

static void show_message_part(struct etpan_message_view * msg_view,
    GtkTextBuffer * textbuffer,
    GtkTextIter * iter,
    struct etpan_part_fetch_info * info,
    struct part_info * part_info)
{
  struct etpan_part_header * header;
  struct etpan_part * part;
  char * content;
  size_t content_length;
  char * filename;
  int type;
  char * content_type;
  int inserted;
  GdkPixbuf * pixbuf;
  GtkIconSet * iconset;
  GtkStyle * style;
  
  inserted = 0;
  
  part = etpan_part_fetch_info_get_part(info);
  type = etpan_part_fetch_info_get_type(info);
  
  content_type = NULL;
  content = NULL;
  content_length = 0;
  filename = NULL;
  
  if (type == ETPAN_PART_FETCH_INFO_TYPE_TEXT) {
    etpan_part_fetch_info_get_content(info, &content, &content_length);
  }
  else if (type == ETPAN_PART_FETCH_INFO_TYPE_FILE) {
    filename = etpan_part_fetch_info_get_filename(info);
  }
  
  if (msg_view->fetcher_mode == ETPAN_MESSAGE_VIEW_MODE_SOURCE) {
    if (type == ETPAN_PART_FETCH_INFO_TYPE_TEXT) {
      part_info_set_type(part_info, PART_INFO_TYPE_TEXT);
      
      if (g_utf8_validate(content, content_length, NULL)) {
        gtk_text_buffer_insert_with_tags_by_name(textbuffer, iter, content, -1,
            "fixed", NULL);
      }
      else {
        size_t converted_length;
        char * converted;
        
        charconv_buffer("utf-8", "iso-8859-1",
            content, content_length,
            &converted, &converted_length);
        
        gtk_text_buffer_insert_with_tags_by_name(textbuffer, iter, converted, -1,
            "fixed", NULL);
        
        mmap_string_unref(converted);
      }
    }
  }
  else {
    header = etpan_part_get_header(part);
    if (header == NULL) {
      ETPAN_MSGVIEW_LOG("show_message_part - part has no header");
      goto exit;
    }
    
    content_type = etpan_part_header_get_content_type(header);
  
    if (type == ETPAN_PART_FETCH_INFO_TYPE_TEXT) {
      if (strncasecmp(content_type, "text/", 5) == 0) {
        char * filename;
        size_t output_size;
        char * output;
        char * wrapped;
        size_t wrapped_length;
        
        part_info_set_type(part_info, PART_INFO_TYPE_TEXT);
      
        filename = etpan_part_header_get_filename(header);
        if (filename != NULL) {
          gtk_text_buffer_insert(textbuffer, iter, "--- ", -1);
          gtk_text_buffer_insert(textbuffer, iter, filename, -1);
          gtk_text_buffer_insert(textbuffer, iter, " ---\n", -1);
        }
        
        etpan_unwrap_text(content, content_length,
            &output, &output_size);
        
        etpan_wrap_text(output, output_size,
            &wrapped, &wrapped_length, TYPOGRAPHY_WRAP_COUNT);
          
        gtk_text_buffer_insert_with_tags_by_name(textbuffer, iter, wrapped, -1,
            "fixed", NULL);
        
        gtk_text_buffer_insert_with_tags_by_name(textbuffer, iter, "\n", -1,
            "fixed", NULL);
        
        free(wrapped);
        free(output);
        
        inserted = 1;
      }
      else if (strcasecmp(content_type, "message/rfc822") == 0) {
        carray * unparsed_header;
        unsigned int i;
        time_t date;
        char * subject;
        carray * from;
        carray * replyto;
        carray * to;
        carray * cc;
        carray * bcc;
        struct etpan_message_header * msg_header;
      
        part_info_set_type(part_info, PART_INFO_TYPE_HEADER);
      
        msg_header = etpan_message_header_parse(content, content_length);
      
        from = etpan_message_header_get_from(msg_header);
        if (from != NULL) {
          if (carray_count(from) > 0) {
            gtk_text_buffer_insert_with_tags_by_name(textbuffer,
                iter, _("From: "), -1, "header-name", NULL);
            show_address_list(msg_view, textbuffer, iter, from);
            gtk_text_buffer_insert_with_tags_by_name(textbuffer, iter, "\n", -1,
                "header-value", NULL);
          }
        }
        replyto = etpan_message_header_get_replyto(msg_header);
        if (replyto != NULL) {
          if (carray_count(replyto) > 0) {
            gtk_text_buffer_insert_with_tags_by_name(textbuffer,
                iter, _("Reply-To: "), -1, "header-name", NULL);
            show_address_list(msg_view, textbuffer, iter, replyto);
            gtk_text_buffer_insert_with_tags_by_name(textbuffer, iter, "\n", -1,
                "header-value", NULL);
          }
        }
        subject = etpan_message_header_get_subject(msg_header);
        if (subject != NULL) {
          gtk_text_buffer_insert_with_tags_by_name(textbuffer,
              iter, _("Subject: "), -1, "header-name", NULL);
          gtk_text_buffer_insert_with_tags_by_name(textbuffer, iter, subject, -1,
              "header-value", NULL);
          gtk_text_buffer_insert_with_tags_by_name(textbuffer, iter, "\n", -1,
              "header-value", NULL);
        }
        
        date = etpan_message_header_get_date(msg_header);
        if (date != (time_t) -1) {
          char * date_str;
          
          date_str = etpan_message_get_long_date_str(date);
          gtk_text_buffer_insert_with_tags_by_name(textbuffer,
              iter, _("Date: "), -1, "header-name", NULL);
          gtk_text_buffer_insert_with_tags_by_name(textbuffer, iter, date_str, -1,
              "header-value", NULL);
          gtk_text_buffer_insert_with_tags_by_name(textbuffer, iter, "\n", -1,
              "header-value", NULL);
          free(date_str);
        }
        
        to = etpan_message_header_get_to(msg_header);
        if (to != NULL) {
          if (carray_count(to) > 0) {
            gtk_text_buffer_insert_with_tags_by_name(textbuffer,
                iter, _("To: "), -1, "header-name", NULL);
            show_address_list(msg_view, textbuffer, iter, to);
            gtk_text_buffer_insert_with_tags_by_name(textbuffer, iter, "\n", -1,
                "header-value", NULL);
          }
        }
        cc = etpan_message_header_get_cc(msg_header);
        if (cc != NULL) {
          if (carray_count(cc) > 0) {
            gtk_text_buffer_insert_with_tags_by_name(textbuffer,
                iter, _("Cc: "), -1, "header-name", NULL);
            show_address_list(msg_view, textbuffer, iter, cc);
            gtk_text_buffer_insert_with_tags_by_name(textbuffer, iter, "\n", -1,
                "header-value", NULL);
          }
        }
        bcc = etpan_message_header_get_bcc(msg_header);
        if (bcc != NULL) {
          if (carray_count(bcc) > 0) {
            gtk_text_buffer_insert_with_tags_by_name(textbuffer,
                iter, _("Bcc: "), -1, "header-name", NULL);
            show_address_list(msg_view, textbuffer, iter, bcc);
            gtk_text_buffer_insert_with_tags_by_name(textbuffer, iter, "\n", -1,
                "header-value", NULL);
          }
        }
      
        etpan_message_header_free(msg_header);
      
        unparsed_header = etpan_unparsed_header_parse(content, content_length);
        for(i = 0 ; i < carray_count(unparsed_header) ; i ++) {
          struct etpan_unparsed_header_item * item;
          char * name;
        
          item = carray_get(unparsed_header, i);
          name = etpan_unparsed_header_item_get_name(item);
          
          if (etpan_message_is_header_displayed(name)) {
            char * value;
            
            value = etpan_unparsed_header_item_get_value(item);
            value = etpan_decode_mime_header(value);
            
            gtk_text_buffer_insert_with_tags_by_name(textbuffer,
                iter, name, -1, "header-name", NULL);
            gtk_text_buffer_insert_with_tags_by_name(textbuffer, iter, ": ", -1,
                "header-name", NULL);
            gtk_text_buffer_insert_with_tags_by_name(textbuffer, iter, value, -1,
                "header-value", NULL);
            gtk_text_buffer_insert_with_tags_by_name(textbuffer, iter, "\n", -1,
                "header-value", NULL);
            
            free(value);
          }
        }
#if 0
        gtk_text_buffer_insert(textbuffer, iter, "\n", -1);
#endif
      
        etpan_unparsed_header_free(unparsed_header);
      
        inserted = 1;
      }
    }
    else if (type == ETPAN_PART_FETCH_INFO_TYPE_FILE) {
      if (strncasecmp(content_type, "image/", 6) == 0) {
        GError * error;
      
        part_info_set_type(part_info, PART_INFO_TYPE_IMAGE);
        
        gtk_text_buffer_insert(textbuffer, iter, content_type, -1);
        gtk_text_buffer_insert(textbuffer, iter, "\n", -1);
      
        error = NULL;
        pixbuf = gdk_pixbuf_new_from_file(filename, &error);
        if (pixbuf != NULL) {
          gtk_text_buffer_insert_pixbuf(textbuffer, iter, pixbuf);
          g_object_unref(pixbuf);
          gtk_text_buffer_insert(textbuffer, iter, "\n", -1);
          inserted = 1;
        }
      }
    }
    
  exit:
    
#if 0
    if (content_type != NULL) {
      gtk_text_buffer_insert(textbuffer, iter, content_type, -1);
      gtk_text_buffer_insert(textbuffer, iter, "\n", -1);
    }
#endif
    
    if (!inserted) {
      part_info_set_type(part_info, PART_INFO_TYPE_FILE);
    
      /* insert icon */
      iconset = gtk_icon_factory_lookup_default("gtk-file");
      style = gtk_widget_get_style(msg_view->textview);
      pixbuf = gtk_icon_set_render_icon(iconset, style, GTK_TEXT_DIR_NONE,
          GTK_STATE_ACTIVE, GTK_ICON_SIZE_LARGE_TOOLBAR,
          msg_view->textview, NULL);
    
      gtk_text_buffer_insert_pixbuf(textbuffer, iter, pixbuf);
      g_object_unref(pixbuf);
      gtk_text_buffer_insert(textbuffer, iter, "\n", -1);
    
      filename = etpan_part_header_get_filename(header);
      if (filename != NULL) {
        gtk_text_buffer_insert(textbuffer, iter, filename, -1);
        gtk_text_buffer_insert(textbuffer, iter, "\n", -1);
      }
    }
    
    gtk_text_buffer_insert(textbuffer, iter, "\n", -1);
  }
}

static void show_message(struct etpan_message_view * msg_view)
{
  GtkTextIter iter;
  GtkTextBuffer * textbuffer;
  unsigned int i;
  carray * part_list;
  int r;
  GtkTextIter start_iter;
  GtkTextIter end_iter;
  GdkRectangle visible_rect;
  GtkTextIter visible_iter;
  
  gtk_text_view_get_visible_rect(GTK_TEXT_VIEW(msg_view->textview),
      &visible_rect);
  
  {
    msg_view->separator = NULL;
    
    /* free link info */
    for(i = 0 ; i < carray_count(msg_view->links_list) ; i ++) {
      struct link_elt * link;
      
      link = carray_get(msg_view->links_list, i);
      link_free(link);
    }
    r = carray_set_size(msg_view->links_list, 0);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
    
    /* free part info */
    for(i = 0 ; i < carray_count(msg_view->parts_list) ; i ++) {
      struct part_info * info;
      
      info = carray_get(msg_view->parts_list, i);
      part_info_free(info);
    }
    r = carray_set_size(msg_view->parts_list, 0);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  gtk_text_buffer_get_start_iter(msg_view->textbuffer,
      &start_iter);
  gtk_text_buffer_get_end_iter(msg_view->textbuffer,
      &end_iter);
  gtk_text_buffer_delete(msg_view->textbuffer,
      &start_iter, &end_iter);
    
  ETPAN_MSGVIEW_LOG("fetching parts");
  msg_view->parts_fetched = 1;
  
#if 0
  if (msg_view->fetcher_mode == ETPAN_MESSAGE_VIEW_MODE_SOURCE) {
    struct etpan_part_fetch_info * info;
    
    part_list = etpan_message_fetcher_get_part_list(msg_view->fetcher);
    info = carray_get(part_list, i);
  }
#endif
  
  part_list = etpan_message_fetcher_get_part_list(msg_view->fetcher);
  for(i = 0 ; i < carray_count(part_list) ; i ++) {
    struct part_info * info;
    
    info = part_info_new();
    r = carray_add(msg_view->parts_list, info, NULL);
    if (r < 0) {
      part_info_free(info);
      ETPAN_LOG_MEMORY_ERROR;
    }
  }
  
  textbuffer = msg_view->textbuffer;
  gtk_text_buffer_get_iter_at_offset(textbuffer, &iter, 0);
  
  part_list = etpan_message_fetcher_get_part_list(msg_view->fetcher);
  for(i = 0 ; i < carray_count(part_list) ; i ++) {
    struct etpan_part_fetch_info * info;
    struct part_info * part_info;
    GtkTextMark * start_mark;
    GtkTextMark * end_mark;
    
    info = carray_get(part_list, i);
    
    start_mark = gtk_text_buffer_create_mark(textbuffer, NULL,
        &iter, TRUE);
    
    part_info = carray_get(msg_view->parts_list, i);
    part_info_set_part(part_info, info);
    
    show_message_part(msg_view, textbuffer, &iter, info, part_info);
    
    end_mark = gtk_text_buffer_create_mark(textbuffer, NULL,
        &iter, TRUE);
    
    if ((i == 0) && (msg_view->mode == ETPAN_MESSAGE_VIEW_MODE_NORMAL)) {
      GtkTextChildAnchor * anchor;
      GtkWidget * separator;
      
      anchor = gtk_text_buffer_create_child_anchor(textbuffer, &iter);
      separator = gtk_hseparator_new();
      gtk_widget_set_size_request(separator,
          msg_view->scrolledwindow->allocation.width - 100, 5);
      msg_view->separator = separator;
      
      gtk_widget_show(separator);
      
      gtk_text_view_add_child_at_anchor(GTK_TEXT_VIEW(msg_view->textview),
          separator, anchor);
      gtk_text_buffer_insert(textbuffer, &iter, "\n\n", -1);
    }
    
    part_info_set_mark(part_info, textbuffer, start_mark, end_mark);
  }
  
  /* place text view previous position */
  gtk_text_buffer_get_iter_at_line(textbuffer,
      &visible_iter, visible_rect.y);
  gtk_text_view_scroll_to_iter(GTK_TEXT_VIEW(msg_view->textview),
      &visible_iter, 0, TRUE, 0, 1);
}

static void part_fetched_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_message_view * msg_view;
  (void) signal_name;
  (void) signal_data;
  (void) sender;
  
  msg_view = user_data;
  
  ETPAN_MSGVIEW_LOG("part fetched");
  show_message(msg_view);
}

static void finished_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_message_view * msg_view;
  struct etpan_message * msg;
  struct etpan_error * error;
  (void) signal_name;
  (void) sender;
  (void) signal_data;
  
  msg_view = user_data;
  stop_progress_animation(msg_view);
  
  ETPAN_MSGVIEW_LOG("finished");
  error = etpan_message_fetcher_get_error(msg_view->fetcher);
  if (error != NULL) {
    etpan_error_log(error);
    return;
  }
  
  show_message(msg_view);
  
  ETPAN_MSGVIEW_LOG("mark message as read");
  /* mark message as read */
  msg = msg_view->message;
  if (msg != NULL) {
    struct etpan_message_flags * flags;
    int value;
    
    flags = etpan_message_get_flags(msg);
    flags = etpan_message_flags_dup(flags);
    value = etpan_message_flags_get_value(flags);
    if ((value & ETPAN_FLAGS_SEEN) == 0) {
      value |= ETPAN_FLAGS_SEEN;
        
      etpan_message_flags_set_value(flags, value);
      etpan_message_set_flags(msg, flags);
        
      etpan_signal_send(etpan_signal_manager_get_default(),
          ETPAN_MESSAGE_VIEW_FLAGSCHANGED_SIGNAL,
          msg_view, msg);
    }
    etpan_message_flags_free(flags);
  }
}

static gboolean motion_handler(GtkWidget * widget,
    GdkEventMotion * event,
    gpointer user_data)
{
  struct etpan_message_view * msg_view;
  GtkTextIter iter;
  unsigned int i;
  gint bx;
  gint by;
  
  msg_view = user_data;
  
  /*
  if (!gtk_widget_is_focus(msg_view->textview))
    gtk_widget_grab_focus(msg_view->textview);
  */
  
  gtk_text_view_window_to_buffer_coords(GTK_TEXT_VIEW(msg_view->textview), 
      GTK_TEXT_WINDOW_WIDGET,
      event->x, event->y, &bx, &by);
  
  gtk_text_view_get_iter_at_location(GTK_TEXT_VIEW(msg_view->textview),
      &iter, bx, by);
  
  for(i = 0 ; i < carray_count(msg_view->links_list) ; i ++) {
    struct link_elt * link;
    
    link = carray_get(msg_view->links_list, i);
    if (link_is_over(link, &iter)) {
      link_set_over(link, 1);
    }
    else {
      link_set_over(link, 0);
    }
  }

  for(i = 0 ; i < carray_count(msg_view->parts_list) ; i ++) {
    struct part_info * info;
    
    info = carray_get(msg_view->parts_list, i);
    if (part_info_is_over(info, &iter)) {
      part_info_set_over(info, 1);
    }
    else {
      part_info_set_over(info, 0);
    }
  }
  
  /* must be here to get all motion events */
  gdk_window_get_pointer(widget->window, NULL, NULL, NULL);
  
  return FALSE;
}

static void address_show_popup_menu(struct etpan_message_view * msg_view,
    GtkWidget * widget, GdkEventButton * event)
{
  GtkWidget * menu;
  int button, event_time;
  (void) widget;
  
  if (event) {
    button = event->button;
    event_time = event->time;
  }
  else {
    button = 0;
    event_time = gtk_get_current_event_time();
  }
  
  menu = etpan_contextual_menu_get_main_widget(msg_view->address_popup_menu);
  
  gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL, 
      button, event_time);
}

static void part_show_popup_menu(struct etpan_message_view * msg_view,
    GtkWidget * widget, GdkEventButton * event)
{
  GtkWidget * menu;
  int button, event_time;
  (void) widget;
  
  if (event) {
    button = event->button;
    event_time = event->time;
  }
  else {
    button = 0;
    event_time = gtk_get_current_event_time();
  }
  
  menu = etpan_contextual_menu_get_main_widget(msg_view->part_popup_menu);
  
  gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL, 
      button, event_time);
}

gboolean button_press_handler(GtkWidget *widget,
    GdkEventButton * event,
    gpointer user_data)
{
  struct etpan_message_view * msg_view;
  unsigned int i;
  GtkTextIter iter;
  gint bx;
  gint by;
  
  msg_view = user_data;
  
  gtk_text_view_window_to_buffer_coords(GTK_TEXT_VIEW(msg_view->textview), 
      GTK_TEXT_WINDOW_WIDGET,
      event->x, event->y, &bx, &by);
  
  gtk_text_view_get_iter_at_location(GTK_TEXT_VIEW(msg_view->textview),
      &iter, bx, by);
  
  /* address in the message */
  for(i = 0 ; i < carray_count(msg_view->links_list) ; i ++) {
    struct link_elt * link;
    
    link = carray_get(msg_view->links_list, i);
    if (link_is_over(link, &iter)) {
      ETPAN_MSGVIEW_LOG("address clicked : %s", link->address_info);
      
      etpan_contextual_menu_set_label(msg_view->address_popup_menu,
          0, link->address_info);
      etpan_contextual_menu_set_enabled(msg_view->address_popup_menu, 0, 0);
      
      msg_view->current_address = link;
      address_show_popup_menu(msg_view, widget, event);
      return TRUE;
    }
  }
  
  /* part popup menu */
  for(i = 0 ; i < carray_count(msg_view->parts_list) ; i ++) {
    struct part_info * info;
    
    info = carray_get(msg_view->parts_list, i);
    switch (info->type) {
    case PART_INFO_TYPE_FILE:
    case PART_INFO_TYPE_IMAGE:
      if (part_info_is_over(info, &iter)) {
        if (info->fetch_info != NULL) {
          struct etpan_part * part;
          
          part = etpan_part_fetch_info_get_part(info->fetch_info);
          if (part != NULL) {
            char * filename;
            struct etpan_part_header * header;
            
            filename = NULL;
            header = etpan_part_get_header(part);
            if (header != NULL)
              filename = etpan_part_header_get_filename(header);
            
            if (filename != NULL) {
              etpan_contextual_menu_set_label(msg_view->part_popup_menu,
                  0, filename);
            }
            else {
              etpan_contextual_menu_set_label(msg_view->part_popup_menu,
                  0, _("(no filename)"));
            }
            etpan_contextual_menu_set_enabled(msg_view->part_popup_menu, 0, 0);
            if (filename != NULL)
              etpan_contextual_menu_set_enabled(msg_view->part_popup_menu, 1, 1);
            else
              etpan_contextual_menu_set_enabled(msg_view->part_popup_menu, 1, 0);
            
            msg_view->current_part = info->fetch_info;
            
            part_show_popup_menu(msg_view, widget, event);
            
            return TRUE;
          }
        }
      }
      break;
    }
  }
  
  return FALSE;
}

static gboolean keypress_handler(GtkWidget * widget,
    GdkEventKey * event,
    gpointer user_data)
{
  struct etpan_message_view * msg_view;
  (void) widget;
  
  msg_view = user_data;
  
  switch (event->keyval) {
  case GDK_space:
    {
      etpan_message_view_scroll_page_down(msg_view);
    }
    break;
  case GDK_s:
    {
      struct etpan_message * msg;
      
      ETPAN_MSGVIEW_LOG("switch mode");
      if (etpan_message_view_get_mode(msg_view) == ETPAN_MESSAGE_VIEW_MODE_SOURCE)
        etpan_message_view_set_mode(msg_view, ETPAN_MESSAGE_VIEW_MODE_NORMAL);
      else
        etpan_message_view_set_mode(msg_view, ETPAN_MESSAGE_VIEW_MODE_SOURCE);
      
      msg = msg_view->message;
      etpan_message_ref(msg);
      etpan_message_view_set_message(msg_view, NULL);
      etpan_message_view_set_message(msg_view, msg);
      etpan_message_unref(msg);
    }
  }
  
  return FALSE;
}

static void new_message(struct etpan_message_view * msg_view)
{
  struct etpan_message_composer_window * composer;
  GtkWidget * window;
  (void) msg_view;
  
  composer = etpan_message_composer_window_new();
  etpan_message_composer_window_setup(composer);
  window = etpan_message_composer_window_get_main_widget(composer);
  gtk_widget_show(window);
}

static void address_popup_menu_item_clicked(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_contextual_menu * menu;
  struct etpan_message_view * msg_view;
  int * pindex;
  (void) signal_name;
  
  msg_view = user_data;
  menu = sender;
  pindex = signal_data;
  
  ETPAN_MSGVIEW_LOG("index : %i", * pindex);
  
  switch (* pindex) {
  case 1:
    /* copy address */
    {
      if (msg_view->current_address != NULL) {
        struct link_elt * link;
        GtkClipboard * clipboard;
        
        link = msg_view->current_address;
        clipboard = gtk_clipboard_get(GDK_SELECTION_PRIMARY);
        gtk_clipboard_set_text(clipboard,
            link->address_info, strlen(link->address_info));
        gtk_clipboard_store(clipboard);
      }
    }
    break;
  case 2:
    /* new message */
    new_message(msg_view);
    break;
  case 3:
    /* add to address book */
    break;
  }
}

static void address_popup_menu_setup(struct etpan_message_view * msg_view)
{
  struct etpan_contextual_menu * menu;
  
  menu = etpan_contextual_menu_new();
  etpan_contextual_menu_add_item(menu, _("(e-mail adress)"), NULL);
  etpan_contextual_menu_add_item(menu, _("Copy address"), NULL);
  etpan_contextual_menu_add_item(menu, _("New message"), NULL);
  etpan_contextual_menu_add_item(menu, _("Add to Address Book"), NULL);
  
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_CONTEXTUAL_MENU_CLICKED_SIGNAL, menu, msg_view,
      address_popup_menu_item_clicked);
  
  msg_view->address_popup_menu = menu;
}

static void address_popup_menu_unsetup(struct etpan_message_view * msg_view)
{
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_CONTEXTUAL_MENU_CLICKED_SIGNAL, msg_view->address_popup_menu,
      msg_view,
      address_popup_menu_item_clicked);
  
  etpan_contextual_menu_free(msg_view->address_popup_menu);
}

struct etpan_save_data {
  struct etpan_message_view * msg_view;
  struct etpan_part * part;
  char * path;
  int execute;
  struct etpan_thread_op * op;
};

static void save_file_callback(int cancelled,
    struct etpan_part_fetch_result * fetch_result,
    void * cb_data)
{
  struct etpan_message_view * msg_view;
  char path[PATH_MAX];
  struct etpan_part * part;
  struct etpan_part_header * header;
  struct etpan_save_data * save_data;
  char * decoded;
  size_t decoded_len;
  int r;
  int encoding;
  int execute;
  struct etpan_thread_op * save_op;
  unsigned int i;
  struct etpan_error * error;
  
  save_data = cb_data;
  msg_view = save_data->msg_view;
  part = save_data->part;
  strncpy(path, save_data->path, sizeof(path));
  path[sizeof(path) - 1] = 0;
  execute = save_data->execute;
  save_op = save_data->op;
  free(save_data->path);
  free(save_data);
  
  stop_progress_animation(msg_view);
  
  for(i = 0 ; i < carray_count(msg_view->save_op_list) ; i ++) {
    struct etpan_thread_op * op;
    
    op = carray_get(msg_view->save_op_list, i);
    if (op == save_op) {
      carray_delete(msg_view->save_op_list, i);
      break;
    }
  }
  
  if (cancelled) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CANCELLED);
    etpan_error_set_short_description(error, _("Save has been cancelled"));
    etpan_error_strf_long_description(error, _("The user cancelled save of the attachment."));
    etpan_error_log(error);
    ETPAN_ERROR_FREE(error);
    return;
  }
  
  if (fetch_result->error != NULL) {
    etpan_error_log(fetch_result->error);
    return;
  }
  
  header = etpan_part_get_header(part);
  if (header != NULL)
    encoding = etpan_part_header_get_mime_encoding(header);
  else
    encoding = ETPAN_PART_MIME_ENCODING_OTHER;
  
  error = etpan_mime_decode(fetch_result->content, fetch_result->length,
      encoding, &decoded, &decoded_len);
  if (error != NULL) {
    etpan_error_log(error);
    return;
  }
  
  r = etpan_write_file(path, decoded, decoded_len);
  if (r < 0) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FILE);
    etpan_error_set_short_description(error, _("Could not write file"));
    etpan_error_strf_long_description(error,
        _("Attachment could not be saved to %s"), path);
    
    etpan_error_log(error);
    
    ETPAN_ERROR_FREE(error);
    return;
  }
  
  free(decoded);
  
  if (execute) {
    char command[PATH_MAX];
    char quoted[PATH_MAX];
    
    r = etpan_quote_filename(quoted, sizeof(quoted), path);
    if (r < 0) {
      struct etpan_error * error;
      
      error = etpan_error_new();
      etpan_error_set_code(error, ERROR_FILE);
      etpan_error_set_short_description(error, _("Could not open file"));
      etpan_error_strf_long_description(error,
          _("Attachment could not be opened because the path is too long. However, it has been saved to %s."), path);
      
      etpan_error_log(error);
      
      ETPAN_ERROR_FREE(error);
      return;
    }
    
#if defined(__APPLE__) && defined(__MACH__)
    snprintf(command, sizeof(command), "open '%s'", path);
#else
    snprintf(command, sizeof(command), "run-mailcap '%s' &", path);
#endif
    r = system(command);
    if (WEXITSTATUS(r) != 0) {
      struct etpan_error * error;
      
      error = etpan_error_new();
      etpan_error_set_code(error, ERROR_FILE);
      etpan_error_set_short_description(error, _("Could not open file"));
      etpan_error_strf_long_description(error,
          _("Attachment could not be opened because it has an unknown type. However, it has been saved to %s."), path);
      
      etpan_error_log(error);
      
      ETPAN_ERROR_FREE(error);
    }
  }
  
  ETPAN_MSGVIEW_LOG("file written");
}

static void save_file(struct etpan_message_view * msg_view,
    struct etpan_part * part, int execute)
{
  char path_buf[PATH_MAX];
  char * home;
  struct etpan_part_header * header;
  char * path;
  struct etpan_save_data * save_data;
  struct etpan_thread_op * op;
  struct etpan_error * error;
  int r;
  
  home = getenv("HOME");
  if (home == NULL) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_INVAL);
    etpan_error_set_short_description(error, _("Home directory not found"));
    etpan_error_strf_long_description(error, _("Home directory is not defined. It should be defined in environment variable HOME."));
    
    etpan_error_log(error);
    ETPAN_ERROR_FREE(error);
    
    return;
  }
  
  save_data = malloc(sizeof(* save_data));
  if (save_data == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  save_data->msg_view = msg_view;
  save_data->part = part;
  save_data->path = NULL;
  save_data->execute = execute;
  
  path = NULL;
  header = etpan_part_get_header(part);
  if (header != NULL) {
    char * filename;
    
    filename = etpan_part_header_get_filename(header);
    
    if (filename != NULL) {
      char basename_buf[PATH_MAX];
      char * bname;
      char ext[PATH_MAX];
      char * p;
      int count;
      
      strncpy(basename_buf, filename, sizeof(basename_buf));
      basename_buf[sizeof(basename_buf) - 1] = '\0';
      bname = basename(basename_buf);
      
      p = strchr(bname, '.');
      if (p != NULL) {
        strncpy(ext, p, sizeof(ext));
        ext[sizeof(ext) - 1] = '\0';
        * p = '\0';
      }
      else {
        ext[0] = '\0';
      }
      
      count = 0;
      while (1) {
        char * attachment_folder;
        
        attachment_folder = etpan_global_config_get_attachment_folder(etpan_global_config_get_default());
        
        if (count == 0)
          snprintf(path_buf, sizeof(path_buf), "%s/%s%s",
              attachment_folder, bname, ext);
        else
          snprintf(path_buf, sizeof(path_buf), "%s/%s-%i%s",
              attachment_folder, bname, count, ext);
        
        if (!etpan_file_exists(path_buf))
          break;
        
        count ++;
      }
      
      path = path_buf;
    }
  }
  
  if (path == NULL) {
    GtkWidget *dialog;
    int r;
    char * attachment_folder;
    
    dialog = gtk_file_chooser_dialog_new ("Save Attachment",
        NULL,
        GTK_FILE_CHOOSER_ACTION_SAVE,
        GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
        GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT,
        NULL);
#if 0 /* not supported in old versions of GTK */
    gtk_file_chooser_set_do_overwrite_confirmation(GTK_FILE_CHOOSER(dialog),
        TRUE);
#endif
    
    attachment_folder = etpan_global_config_get_attachment_folder(etpan_global_config_get_default());
    
    gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(dialog), attachment_folder);
    gtk_file_chooser_set_current_name(GTK_FILE_CHOOSER(dialog),
        "Untitled attachment");
    
    r = gtk_dialog_run(GTK_DIALOG(dialog));
    if (r == GTK_RESPONSE_ACCEPT) {
      char * filename;
      
      filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog));
      strncpy(path_buf, filename, sizeof(path_buf));
      path_buf[sizeof(path_buf) - 1] = '\0';
      path = path_buf;
      g_free(filename);
    }
    
    gtk_widget_destroy(dialog);
  }
  
  if (path == NULL) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_CANCELLED);
    etpan_error_set_short_description(error, _("Save has been cancelled"));
    etpan_error_strf_long_description(error, _("The user cancelled save of the attachment."));
    etpan_error_log(error);
    ETPAN_ERROR_FREE(error);
    free(save_data->path);
    free(save_data);
    return;
  }
  
  save_data->path = strdup(path);
  if (save_data->path == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  /* do the saving */
  
  start_progress_animation(msg_view);
  
  op = etpan_part_fetch(etpan_thread_manager_app_get_default(),
      part, save_file_callback, save_data);
  save_data->op = op;
  r = carray_add(msg_view->save_op_list, op, NULL);
  if (r < 0) {
    ETPAN_LOG_MEMORY_ERROR;
  }
}


static void part_popup_menu_item_clicked(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_contextual_menu * menu;
  struct etpan_message_view * msg_view;
  int * pindex;
  (void) signal_name;
  (void) signal_data;
  
  msg_view = user_data;
  menu = sender;
  pindex = signal_data;
  
  ETPAN_MSGVIEW_LOG("index : %i", * pindex);
  
  switch (* pindex) {
  case 1:
    /* open */
    {
      struct etpan_part_fetch_info * fetch_info;
      
      fetch_info = msg_view->current_part;
      if (fetch_info != NULL) {
        struct etpan_part * part;
        
        part = etpan_part_fetch_info_get_part(fetch_info);
        if (part != NULL) {
          save_file(msg_view, part, 1);
        }
      }
    }
    break;
  case 2:
    /* save */
    {
      struct etpan_part_fetch_info * fetch_info;
      
      fetch_info = msg_view->current_part;
      if (fetch_info != NULL) {
        struct etpan_part * part;
        
        part = etpan_part_fetch_info_get_part(fetch_info);
        if (part != NULL) {
          save_file(msg_view, part, 0);
        }
      }
    }
    break;
  }
}

static void part_popup_menu_setup(struct etpan_message_view * msg_view)
{
  struct etpan_contextual_menu * menu;
  
  menu = etpan_contextual_menu_new();
  etpan_contextual_menu_add_item(menu, _("(filename)"), NULL);
  etpan_contextual_menu_add_item(menu, _("Open"), NULL);
  etpan_contextual_menu_add_item(menu, _("Save"), NULL);
  
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
    ETPAN_CONTEXTUAL_MENU_CLICKED_SIGNAL, menu, msg_view,
      part_popup_menu_item_clicked);
  
  msg_view->part_popup_menu = menu;
}

static void part_popup_menu_unsetup(struct etpan_message_view * msg_view)
{
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_CONTEXTUAL_MENU_CLICKED_SIGNAL, msg_view->part_popup_menu, msg_view,
      part_popup_menu_item_clicked);
  
  etpan_contextual_menu_free(msg_view->part_popup_menu);
}

static void resize_handler(GtkWidget * widget,
    GtkAllocation * allocation,
    gpointer user_data)
{
  struct etpan_message_view * msg_view;
  GtkRequisition req;
  
  msg_view = user_data;
  
  if (msg_view->separator != NULL) {
    gtk_widget_set_size_request(msg_view->separator,
        allocation->width - 100, 5);
  }
}

void etpan_message_view_setup(struct etpan_message_view * msg_view)
{
  etpan_message_fetcher_setup(msg_view->fetcher);
  
  address_popup_menu_setup(msg_view);
  
  part_popup_menu_setup(msg_view);
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_MESSAGE_FETCHER_PARTFETCHED_SIGNAL,
      msg_view->fetcher, msg_view,
      part_fetched_handler);
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_MESSAGE_FETCHER_FINISHED_SIGNAL,
      msg_view->fetcher, msg_view,
      finished_handler);
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_UI_CONFIG_FONT_CHANGED_SIGNAL,
      etpan_ui_config_get_default(), msg_view,
      font_changed_handler);
  
  update_font(msg_view);
  
  msg_view->motion_signal_id =
    g_signal_connect(msg_view->textview,
        "motion-notify-event", G_CALLBACK(motion_handler),
        (gpointer) msg_view);

  msg_view->button_signal_id =
    g_signal_connect(msg_view->textview,
        "button-press-event", G_CALLBACK(button_press_handler),
        (gpointer) msg_view);
  
  msg_view->keypress_signal_id =
    g_signal_connect(msg_view->textview,
        "key-press-event", G_CALLBACK(keypress_handler),
        (gpointer) msg_view);

  msg_view->resize_signal_id =
    g_signal_connect(msg_view->scrolledwindow,
        "size-allocate", G_CALLBACK(resize_handler),
        (gpointer) msg_view);
}

void etpan_message_view_unsetup(struct etpan_message_view * msg_view)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(msg_view->save_op_list) ; i ++) {
    struct etpan_thread_op * op;
    
    op = carray_get(msg_view->save_op_list, i);
    etpan_thread_op_cancel(op);
  }
  
  g_signal_handler_disconnect(msg_view->scrolledwindow,
      msg_view->resize_signal_id);
  
  g_signal_handler_disconnect(msg_view->textview,
      msg_view->keypress_signal_id);
  
  g_signal_handler_disconnect(msg_view->textview,
      msg_view->button_signal_id);
  
  g_signal_handler_disconnect(msg_view->textview,
      msg_view->motion_signal_id);
  
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_UI_CONFIG_FONT_CHANGED_SIGNAL,
      etpan_ui_config_get_default(), msg_view,
      font_changed_handler);
  
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_MESSAGE_FETCHER_FINISHED_SIGNAL,
      msg_view->fetcher, msg_view,
      finished_handler);
  
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_MESSAGE_FETCHER_PARTFETCHED_SIGNAL,
      msg_view->fetcher, msg_view,
      part_fetched_handler);
  
  part_popup_menu_unsetup(msg_view);
  address_popup_menu_unsetup(msg_view);
  
  etpan_message_view_set_message(msg_view, NULL);
#if 0
  etpan_message_fetcher_unsetup(msg_view->fetcher);
#endif
}

static void start_progress_animation(struct etpan_message_view * msg_view)
{
  if (msg_view->status_bar != NULL)
    etpan_status_bar_start_animation(msg_view->status_bar);
}

static void stop_progress_animation(struct etpan_message_view * msg_view)
{
  if (msg_view->status_bar != NULL)
    etpan_status_bar_stop_animation(msg_view->status_bar);
}

static void message_selected_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_message * msg;
  char * uid;
  struct etpan_message_list * msg_list;
  struct etpan_message_view * msg_view;
  (void) signal_name;
  (void) signal_data;
  
  msg_view = user_data;
  msg_list = sender;
  msg = signal_data;
  
  if (msg == NULL) {
    ETPAN_MSGVIEW_LOG("no selection");
  }
  
  etpan_message_view_set_mode(msg_view, ETPAN_MESSAGE_VIEW_MODE_NORMAL);
  etpan_message_view_set_message(msg_view, msg);
}

static void next_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_message_list * msg_list;
  struct etpan_message_view * msg_view;
  (void) signal_name;
  (void) signal_data;
  
  msg_view = user_data;
  msg_list = sender;
  
  etpan_message_view_scroll_page_down(msg_view);
}

void etpan_message_view_set_message_list(struct etpan_message_view * msg_view,
    struct etpan_message_list * msg_list)
{
  if (msg_view->msg_list != NULL) {
    etpan_signal_remove_handler(etpan_signal_manager_get_default(),
        ETPAN_MESSAGE_LIST_NEXT_SIGNAL,
        msg_view->msg_list, msg_view, next_handler);
    etpan_signal_remove_handler(etpan_signal_manager_get_default(),
        ETPAN_MESSAGE_LIST_SELECTIONCHANGED_SIGNAL,
        msg_view->msg_list, msg_view, message_selected_handler);
  }
  
  if (msg_list != NULL) {
    etpan_signal_add_handler(etpan_signal_manager_get_default(),
        ETPAN_MESSAGE_LIST_SELECTIONCHANGED_SIGNAL,
        msg_list, msg_view, message_selected_handler);
    etpan_signal_add_handler(etpan_signal_manager_get_default(),
        ETPAN_MESSAGE_LIST_NEXT_SIGNAL,
        msg_list, msg_view, next_handler);
  }
  
  msg_view->msg_list = msg_list;
}

void etpan_message_view_scroll_page_down(struct etpan_message_view * msg_view)
{
  gdouble page_size;
  gdouble page_increment;
  gdouble value;
  gdouble upper;
  gdouble lower;
  gdouble previous_value;
  GtkAdjustment * adjustment;
  
  adjustment = gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(msg_view->scrolledwindow));
  
  g_object_get(adjustment,
      "page-increment", &page_increment,
      "page-size", &page_size,
      "upper", &upper,
      "lower", &lower,
      NULL);
  
  value = gtk_adjustment_get_value(adjustment);
  previous_value = value;
  value += page_increment;
  if (value > upper - page_size)
    value = upper - page_size;
  if (value < lower)
    value = lower;
  
  gtk_adjustment_set_value(adjustment, value);
  
  if (fabs(previous_value - value) < 0.1) {
    if (msg_view->msg_list != NULL)
      etpan_message_list_next_unread(msg_view->msg_list);
  }
}

void etpan_message_view_set_mode(struct etpan_message_view * msg_view, int mode)
{
  msg_view->mode = mode;
}

int etpan_message_view_get_mode(struct etpan_message_view * msg_view)
{
  return msg_view->mode;
}
