#include "etpan-search-field.h"

#include <stdlib.h>

#include "etpan-backend.h"
#include "sexy-icon-entry.h"
#include "etpan-icon-manager.h"
#include "etpan-contextual-menu.h"
#include "etpan-progress.h"

#define ICON(name) \
  etpan_icon_manager_new_scaled_image(etpan_icon_manager_get_default(), \
      name, 32)
  
static void set_cancel_icon(struct etpan_search_field * search_field);

static void icon_clicked_handler(SexyIconEntry * entry,
    gint icon_pos,
    gint button,
    gpointer user_data);
static void icon_pressed_handler(SexyIconEntry * entry,
    gint icon_pos,
    gint button,
    gpointer user_data);

static void change_handler(GtkEditable * editable,
    gpointer user_data);

static void popup_menu_setup(struct etpan_search_field * search_field);
static void popup_menu_unsetup(struct etpan_search_field * search_field);

struct etpan_search_field * etpan_search_field_new(void)
{
  struct etpan_search_field * search_field;
  
  search_field = malloc(sizeof(* search_field));
  if (search_field == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  search_field->hbox = NULL;
  search_field->entry = NULL;
  search_field->progress = NULL;
  search_field->cancel_find_enabled = 0;
  search_field->search_type =
    ETPAN_FOLDER_INDEXER_SEARCH_TYPE_FROM_AND_SUBJECT;
  search_field->popup_menu = NULL;
  search_field->menu_callback_disabled = 0;
  search_field->animation_count = 0;
  
  return search_field;
}


void etpan_search_field_free(struct etpan_search_field * search_field)
{
  free(search_field);
}

static gboolean motion_handler(GtkWidget * widget,
    GdkEventMotion * event,
    gpointer user_data)
{
  struct etpan_search_field * search_field;
  (void) widget;
  (void) event;
  
  search_field = user_data;
  
  /*
  if (!gtk_widget_is_focus(search_field->entry))
    gtk_widget_grab_focus(search_field->entry);
  */
  
  return FALSE;
}

void etpan_search_field_setup(struct etpan_search_field * search_field)
{
  GtkWidget * tmp_image;
  
  search_field->hbox = gtk_hbox_new(FALSE, 0);
  
  search_field->progress = etpan_progress_new();
  etpan_progress_setup(search_field->progress);
  gtk_box_pack_start_defaults(GTK_BOX(search_field->hbox),
      etpan_progress_get_main_widget(search_field->progress));
  
  search_field->cancel_find_enabled = 0;
  search_field->entry = sexy_icon_entry_new();
  
  tmp_image = ICON("find");
  if (tmp_image != NULL) {
    gtk_widget_show(tmp_image);
    sexy_icon_entry_set_icon(SEXY_ICON_ENTRY(search_field->entry),
        SEXY_ICON_ENTRY_PRIMARY, GTK_IMAGE(tmp_image));
  }
  
  gtk_widget_show(search_field->entry);
  
  gtk_box_pack_start_defaults(GTK_BOX(search_field->hbox),
      search_field->entry);
  
  gtk_widget_show(search_field->hbox);
  
  search_field->modify_signal_id =
    g_signal_connect(search_field->entry,
        "changed", G_CALLBACK(change_handler),
        (gpointer) search_field);

  search_field->icon_clicked_signal_id =
    g_signal_connect(search_field->entry,
        "icon-released", G_CALLBACK(icon_clicked_handler),
        (gpointer) search_field);

  search_field->icon_pressed_signal_id =
    g_signal_connect(search_field->entry,
        "icon-pressed", G_CALLBACK(icon_pressed_handler),
        (gpointer) search_field);
  
  search_field->motion_signal_id =
    g_signal_connect(search_field->entry,
        "motion-notify-event", G_CALLBACK(motion_handler),
        (gpointer) search_field);
  
  popup_menu_setup(search_field);
}

void etpan_search_field_unsetup(struct etpan_search_field * search_field)
{
  popup_menu_unsetup(search_field);
  
  g_signal_handler_disconnect(search_field->entry,
      search_field->motion_signal_id);
  g_signal_handler_disconnect(search_field->entry,
      search_field->icon_pressed_signal_id);
  g_signal_handler_disconnect(search_field->entry,
      search_field->icon_clicked_signal_id);
  g_signal_handler_disconnect(search_field->entry,
      search_field->modify_signal_id);
  
  gtk_widget_destroy(search_field->entry);
  etpan_progress_unsetup(search_field->progress);
  etpan_progress_free(search_field->progress);
  gtk_widget_destroy(search_field->hbox);
}

static void popup_menu_item_clicked(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_contextual_menu * menu;
  struct etpan_search_field * search_field;
  int * pindex;
  unsigned int i;
  int search_type;
  (void) signal_name;
  
  search_field = user_data;
  menu = sender;
  pindex = signal_data;
  
  if (search_field->menu_callback_disabled)
    return;
  
  search_field->menu_callback_disabled = 1;
  for(i = 0 ; i < 5 ; i ++) {
    if (i != (unsigned int) * pindex) {
      int state;
      
      state = etpan_contextual_menu_get_check_item(menu, i);
      if (state)
        etpan_contextual_menu_set_check_item(menu, i, 0);
    }
  }
  
  search_type = etpan_search_field_get_search_type(search_field);
  switch (* pindex) {
  case 0:
    search_type = ETPAN_FOLDER_INDEXER_SEARCH_TYPE_ALL;
    break;
  case 1:
    search_type =
      ETPAN_FOLDER_INDEXER_SEARCH_TYPE_FROM_AND_SUBJECT;
    break;
  case 2:
    search_type =
      ETPAN_FOLDER_INDEXER_SEARCH_TYPE_RECIPIENT_AND_SUBJECT;
    break;
  case 3:
    search_type = ETPAN_FOLDER_INDEXER_SEARCH_TYPE_FROM;
    break;
  case 4:
    search_type = ETPAN_FOLDER_INDEXER_SEARCH_TYPE_RECIPIENT;
    break;
  case 5:
    search_type = ETPAN_FOLDER_INDEXER_SEARCH_TYPE_SUBJECT;
    break;
  }
  
  etpan_search_field_set_search_type(search_field, search_type);
  
  search_field->menu_callback_disabled = 0;
}

static void set_popup_menu(struct etpan_search_field * search_field)
{
  unsigned int index;
  struct etpan_contextual_menu * menu;
  
  menu = search_field->popup_menu;
  index = 0;
  switch (search_field->search_type) {
  case ETPAN_FOLDER_INDEXER_SEARCH_TYPE_ALL:
    index = 0;
    break;
  case ETPAN_FOLDER_INDEXER_SEARCH_TYPE_FROM_AND_SUBJECT:
    index = 1;
    break;
  case ETPAN_FOLDER_INDEXER_SEARCH_TYPE_RECIPIENT_AND_SUBJECT:
    index = 2;
    break;
  case ETPAN_FOLDER_INDEXER_SEARCH_TYPE_FROM:
    index = 3;
    break;
  case ETPAN_FOLDER_INDEXER_SEARCH_TYPE_RECIPIENT:
    index = 4;
    break;
  case ETPAN_FOLDER_INDEXER_SEARCH_TYPE_SUBJECT:
    index = 5;
    break;
  }
  etpan_contextual_menu_set_check_item(menu, index, 1);
}

static void popup_menu_setup(struct etpan_search_field * search_field)
{
  struct etpan_contextual_menu * menu;
  
  menu = etpan_contextual_menu_new();
  
  etpan_contextual_menu_add_check_item(menu, _("Entire message"), NULL);
  etpan_contextual_menu_add_check_item(menu, _("Sender and subject"), NULL);
  etpan_contextual_menu_add_check_item(menu, _("Recipient and subject"), NULL);
  etpan_contextual_menu_add_check_item(menu, _("Sender"), NULL);
  etpan_contextual_menu_add_check_item(menu, _("Recipient"), NULL);
  etpan_contextual_menu_add_check_item(menu, _("Subject"), NULL);
  
  search_field->popup_menu = menu;
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_CONTEXTUAL_MENU_CLICKED_SIGNAL,
      search_field->popup_menu, search_field,
      popup_menu_item_clicked);
  
  set_popup_menu(search_field);
}

static void popup_menu_unsetup(struct etpan_search_field * search_field)
{
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_CONTEXTUAL_MENU_CLICKED_SIGNAL,
      search_field->popup_menu, search_field,
      popup_menu_item_clicked);
  
  etpan_contextual_menu_free(search_field->popup_menu);
  search_field->popup_menu = NULL;
}

#define MARGIN 2

static void menu_position(GtkMenu * menu,
    gint * px, gint * py,
    gboolean * push_in, gpointer user_data)
{
  GtkRequisition requisition;
  gint x;
  gint y;
  struct etpan_search_field * search_field;
  (void) menu;
  
  search_field = user_data;
  
  gtk_widget_size_request(search_field->entry, &requisition);
  
  gdk_window_get_deskrelative_origin(search_field->entry->window, &x, &y);
  * px = x;
  * py = y + requisition.height - 2 * MARGIN;
  * push_in = TRUE;
}

static void icon_pressed_handler(SexyIconEntry * entry,
    gint icon_pos,
    gint button,
    gpointer user_data)
{
  GtkWidget * menu;
  int event_time;
  struct etpan_search_field * search_field;
  (void) entry;
  
  search_field = user_data;
  
  if (icon_pos != SEXY_ICON_ENTRY_PRIMARY)
    return;
  
  button = 0;
  event_time = gtk_get_current_event_time();
  menu = etpan_contextual_menu_get_main_widget(search_field->popup_menu);
  
  gtk_menu_popup(GTK_MENU(menu), NULL, NULL, menu_position, search_field,
      button, event_time);
}

static void icon_clicked_handler(SexyIconEntry * entry,
    gint icon_pos,
    gint button,
    gpointer user_data)
{
  (void) button;
  (void) user_data;
  
  if (icon_pos == SEXY_ICON_ENTRY_SECONDARY) {
    gtk_entry_set_text(GTK_ENTRY(entry), "");
  }
  else if (icon_pos == SEXY_ICON_ENTRY_PRIMARY) {
  }
}

static void set_cancel_icon(struct etpan_search_field * search_field)
{
  int empty;
  const char * value;
  
  value = etpan_search_field_get_value(search_field);
  empty = (value[0] == '\0');
  
  if (search_field->cancel_find_enabled) {
    if (empty) {
      sexy_icon_entry_set_icon(SEXY_ICON_ENTRY(search_field->entry),
          SEXY_ICON_ENTRY_SECONDARY, NULL);
      search_field->cancel_find_enabled = 0;
    }
  }
  else {
    if (!empty) {
      GtkWidget * tmp_image;
      
      tmp_image = ICON("cancel-find");
      if (tmp_image != NULL) {
        gtk_widget_show(tmp_image);
      }
      sexy_icon_entry_set_icon(SEXY_ICON_ENTRY(search_field->entry),
          SEXY_ICON_ENTRY_SECONDARY, GTK_IMAGE(tmp_image));
      
      sexy_icon_entry_set_icon_highlight(SEXY_ICON_ENTRY(search_field->entry),
          SEXY_ICON_ENTRY_SECONDARY,
          TRUE);
      
      search_field->cancel_find_enabled = 1;
    }
  }
}

static void change_handler(GtkEditable * editable,
    gpointer user_data)
{
  struct etpan_search_field * search_field;
  (void) editable;
  
  search_field = user_data;
  
  set_cancel_icon(search_field);
  
  etpan_signal_send(etpan_signal_manager_get_default(),
    ETPAN_SEARCH_FIELD_MODIFIED_SIGNAL,
      search_field, NULL);
}

GtkWidget *
etpan_search_field_get_main_widget(struct etpan_search_field * search_field)
{
  return search_field->hbox;
}

GtkWidget *
etpan_search_field_get_entry(struct etpan_search_field * search_field)
{
  return search_field->entry;
}

const char * etpan_search_field_get_value(struct etpan_search_field * search_field)
{
  return gtk_entry_get_text(GTK_ENTRY(search_field->entry));
}

void etpan_search_field_set_search_type(struct etpan_search_field * search_field,
    int search_type)
{
  search_field->search_type = search_type;
  set_popup_menu(search_field);
  
  etpan_signal_send(etpan_signal_manager_get_default(),
    ETPAN_SEARCH_FIELD_MODIFIED_SIGNAL,
      search_field, NULL);
}

int etpan_search_field_get_search_type(struct etpan_search_field * search_field)
{
  return search_field->search_type;
}

void etpan_search_field_start_animation(struct etpan_search_field * search_field)
{
  search_field->animation_count ++;
  if (search_field->animation_count == 1) {
#if 0
    ETPAN_LOG("start animation of search field");
#endif
    etpan_progress_start_animation(search_field->progress);
  }
}

void etpan_search_field_stop_animation(struct etpan_search_field * search_field)
{
  search_field->animation_count --;
  if (search_field->animation_count == 0) {
#if 0
    ETPAN_LOG("stop animation of search field");
#endif
    etpan_progress_stop_animation(search_field->progress);
  }
}
