#include "etpan-tabbed-message-list.h"

#include <stdlib.h>

#include "etpan-backend.h"
#include "etpan-message-list.h"
#include "etpan-message-view.h"
#include "etpan-folder-list.h"
#include "etpan-bookmark-view.h"
#include "etpan-status-bar.h"
#include "etpan-ui-config.h"

static struct tabbed_elt * tabbed_elt_new(void);
static void tabbed_elt_free(struct tabbed_elt * elt);
static void tabbed_elt_setup(struct tabbed_elt * elt,
    struct etpan_tabbed_message_list * tabbed);
static void tabbed_elt_unsetup(struct tabbed_elt * elt);
static GtkWidget * tabbed_elt_get_main_widget(struct tabbed_elt * elt);
static GtkWidget * tabbed_elt_get_label_widget(struct tabbed_elt * elt);
static struct etpan_message_list *
tabbed_elt_get_message_list(struct tabbed_elt * elt);
static struct etpan_message_view *
tabbed_elt_get_message_view(struct tabbed_elt * elt);
static int tabbed_elt_get_vpaned_pos(struct tabbed_elt * elt);
static void tabbed_elt_set_vpaned_pos(struct tabbed_elt * elt, int paned_pos);
static void tabbed_elt_copy_properties(struct
    etpan_tabbed_message_list * tabbed,
    unsigned int previous_index,
    unsigned int current_index);

struct etpan_tabbed_message_list * etpan_tabbed_message_list_new(void)
{
  struct etpan_tabbed_message_list * tabbed;
  GtkWidget * notebook;
  
  tabbed = malloc(sizeof(* tabbed));
  if (tabbed == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  tabbed->folder_info_list = carray_new(16);
  if (tabbed->folder_info_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
    
  notebook = gtk_notebook_new();
  gtk_widget_show(notebook);
  
  tabbed->notebook = notebook;
  tabbed->folder_list = NULL;
  tabbed->bookmark_view = NULL;
  tabbed->status_bar = NULL;
  tabbed->current_tab = -1;
  
  return tabbed;
}

void etpan_tabbed_message_list_free(struct etpan_tabbed_message_list *
    tabbed)
{
  while (carray_count(tabbed->folder_info_list) > 0)
    etpan_tabbed_message_list_close_page(tabbed, 0);
  gtk_widget_destroy(tabbed->notebook);
  carray_free(tabbed->folder_info_list);
  free(tabbed);
}

GtkWidget * etpan_tabbed_message_list_get_main_widget(struct
    etpan_tabbed_message_list * tabbed)
{
  return tabbed->notebook;
}

static void tabbed_elt_disconnect(struct etpan_tabbed_message_list * tabbed,
    unsigned int index)
{
  struct tabbed_elt * elt;
  
  elt = carray_get(tabbed->folder_info_list, index);
  
  if (tabbed->bookmark_view != NULL)
    etpan_bookmark_view_set_message_list(tabbed->bookmark_view, NULL);
  etpan_message_list_set_activated(tabbed_elt_get_message_list(elt),
      tabbed, 0);
}

static void tabbed_elt_connect(struct etpan_tabbed_message_list * tabbed,
    unsigned int index)
{
  struct tabbed_elt * elt;
  
  elt = carray_get(tabbed->folder_info_list, index);
  
  etpan_message_list_set_activated(tabbed_elt_get_message_list(elt),
      tabbed, 1);
  if (tabbed->bookmark_view != NULL)
    etpan_bookmark_view_set_message_list(tabbed->bookmark_view,
        tabbed_elt_get_message_list(elt));
}

static void select_new_page(struct etpan_tabbed_message_list * tabbed,
    int new_index)
{
  int last_tab;
  struct tabbed_elt * elt;
  
  ETPAN_TABBED_LOG("%i => %i", tabbed->current_tab, new_index);
  
  last_tab = -1;
  if (tabbed->current_tab >= 0) {
    /* get last vpaned pos */
    elt = carray_get(tabbed->folder_info_list, tabbed->current_tab);
    last_tab = tabbed->current_tab;
    
    ETPAN_TABBED_LOG("disconnect old");
    tabbed_elt_disconnect(tabbed, tabbed->current_tab);
  }
  
  tabbed->current_tab = new_index;
  
  if (new_index < 0)
    return;
  
  ETPAN_TABBED_LOG("connect new");
  tabbed_elt_connect(tabbed, new_index);
  
  /* set last vpaned pos */
  elt = carray_get(tabbed->folder_info_list, new_index);
  if (last_tab != -1)
    tabbed_elt_copy_properties(tabbed, last_tab, new_index);
}

static void selected_signal_handler(GtkNotebook * notebook,
    GtkNotebookPage *page,
    guint page_num,
    gpointer user_data)
{
  struct etpan_tabbed_message_list * tabbed;
  (void) notebook;
  (void) page;
  
  tabbed = (struct etpan_tabbed_message_list *) user_data;
  
  select_new_page(tabbed, page_num);
}

static void folder_selected_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  char * selection_path;
  struct etpan_tabbed_message_list * tabbed;
  struct etpan_folder_list * folder_list;
  (void) signal_name;
  
  tabbed = user_data;
  folder_list = sender;
  selection_path = signal_data;
  
  etpan_signal_send(etpan_signal_manager_get_default(),
      ETPAN_TABBED_MESSAGE_LIST_SELECTIONCHANGED_SIGNAL,
      tabbed, selection_path);
}

void etpan_tabbed_message_list_setup(struct etpan_tabbed_message_list *
    tabbed)
{
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_FOLDER_LIST_SELECTIONCHANGED_SIGNAL,
      tabbed->folder_list, tabbed, folder_selected_handler);
  
  /* listen to signal selected */
  tabbed->selected_signal_id = g_signal_connect(tabbed->notebook,
      "switch-page", G_CALLBACK(selected_signal_handler),
      (gpointer) tabbed);
}

void etpan_tabbed_message_list_unsetup(struct etpan_tabbed_message_list *
    tabbed)
{
  etpan_tabbed_message_list_close_all(tabbed);
  
  /* disconnect signal selected */
  g_signal_handler_disconnect(tabbed->notebook,
      tabbed->selected_signal_id);
  
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_FOLDER_LIST_SELECTIONCHANGED_SIGNAL,
      tabbed->folder_list, tabbed, folder_selected_handler);
}

static void tabbed_setup(struct etpan_tabbed_message_list * tabbed)
{
  if (carray_count(tabbed->folder_info_list) <= 1) {
    gtk_notebook_set_show_tabs(GTK_NOTEBOOK(tabbed->notebook), FALSE);
    gtk_notebook_set_show_border(GTK_NOTEBOOK(tabbed->notebook), FALSE);
  }
  else {
    gtk_notebook_set_show_tabs(GTK_NOTEBOOK(tabbed->notebook), TRUE);
    gtk_notebook_set_show_border(GTK_NOTEBOOK(tabbed->notebook), FALSE);
  }
}

void etpan_tabbed_message_list_add_page(struct etpan_tabbed_message_list *
    tabbed, unsigned int * p_index)
{
  struct tabbed_elt * elt;
  unsigned int index;
  gint notebook_index;
  int r;
  int previous;
  
  previous = etpan_tabbed_message_list_get_current_page(tabbed);
  
  elt = tabbed_elt_new();
  r = carray_add(tabbed->folder_info_list, elt, &index);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  g_object_ref(G_OBJECT(tabbed_elt_get_main_widget(elt)));
  g_object_ref(G_OBJECT(tabbed_elt_get_label_widget(elt)));
  
  notebook_index = gtk_notebook_append_page(GTK_NOTEBOOK(tabbed->notebook),
      tabbed_elt_get_main_widget(elt),
      tabbed_elt_get_label_widget(elt));
  if (notebook_index < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  tabbed_elt_setup(elt, tabbed);
  
  tabbed_setup(tabbed);
  
  * p_index = index;
}

void etpan_tabbed_message_list_close_page(struct etpan_tabbed_message_list *
    tabbed, unsigned int index)
{
  struct tabbed_elt * elt;
  int selected;
  
  if (index >= carray_count(tabbed->folder_info_list))
    return;
  
  elt = carray_get(tabbed->folder_info_list, index);
  
  etpan_message_list_set_folder(tabbed_elt_get_message_list(elt), NULL);
  tabbed_elt_disconnect(tabbed, index);
  tabbed_elt_unsetup(elt);
  
  select_new_page(tabbed, -1);
  
  gtk_notebook_remove_page(GTK_NOTEBOOK(tabbed->notebook), index);
  carray_delete_slow(tabbed->folder_info_list, index);
  tabbed_elt_free(elt);
  
  tabbed_setup(tabbed);
  selected = gtk_notebook_get_current_page(GTK_NOTEBOOK(tabbed->notebook));
  if (selected < 0)
    return;
  
  select_new_page(tabbed, selected);
}

int etpan_tabbed_message_list_get_current_page(struct
    etpan_tabbed_message_list * tabbed)
{
  return tabbed->current_tab;
}

void etpan_tabbed_message_list_set_current_page(struct
    etpan_tabbed_message_list *
    tabbed, unsigned int index)
{
  gtk_notebook_set_current_page(GTK_NOTEBOOK(tabbed->notebook), index);
}

unsigned int
etpan_tabbed_message_list_get_count(struct etpan_tabbed_message_list *
    tabbed)
{
  return carray_count(tabbed->folder_info_list);
}

struct etpan_message_list *
etpan_tabbed_message_list_get_current_message_list(struct
    etpan_tabbed_message_list * tabbed)
{
  int index;
  
  index = etpan_tabbed_message_list_get_current_page(tabbed);
  if (index < 0)
    return NULL;
  
  return etpan_tabbed_message_list_get_message_list(tabbed, index);
}

struct etpan_message_list *
etpan_tabbed_message_list_get_message_list(struct
    etpan_tabbed_message_list * tabbed, unsigned int index)
{
  struct tabbed_elt * elt;
  
  elt = carray_get(tabbed->folder_info_list, index);
  
  return tabbed_elt_get_message_list(elt);
}


struct tabbed_elt {
  int vpaned_last_position;
  GtkWidget * vpaned;
  GtkWidget * label;
  struct etpan_message_list * message_list;
  struct etpan_message_view * message_view;
};

static struct tabbed_elt * tabbed_elt_new(void)
{
  struct tabbed_elt * elt;
  struct etpan_message_list * message_list;
  struct etpan_message_view * message_view;
  GtkWidget * vpaned;
  GtkWidget * label;
  GtkWidget * message_list_widget;
  GtkWidget * message_view_widget;
  
  elt = malloc(sizeof(* elt));
  if (elt == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  /* split view for message list / message view */
  vpaned = gtk_vpaned_new();
  gtk_widget_show(vpaned);
  gtk_paned_set_position(GTK_PANED(vpaned), 150);
  
  label = gtk_label_new("folder");
  gtk_widget_show(label);
  
  message_list = etpan_message_list_new();
  
  message_list_widget = etpan_message_list_get_main_widget(message_list);
  gtk_paned_pack1(GTK_PANED(vpaned), message_list_widget, FALSE, TRUE);
  
  message_view = etpan_message_view_new();
  
  message_view_widget = etpan_message_view_get_main_widget(message_view);
  gtk_paned_pack2(GTK_PANED(vpaned), message_view_widget, TRUE, TRUE);
  
  elt->vpaned_last_position = 0;
  elt->vpaned = vpaned;
  elt->message_list = message_list;
  elt->message_view = message_view;
  elt->label = label;
  
  return elt;
}

static void tabbed_elt_free(struct tabbed_elt * elt)
{
  etpan_message_view_free(elt->message_view);
  etpan_message_list_free(elt->message_list);
  gtk_widget_destroy(elt->label);
  gtk_widget_destroy(elt->vpaned);
  free(elt);
}


static void tabbed_elt_setup(struct tabbed_elt * elt,
    struct etpan_tabbed_message_list * tabbed)
{
  int r;
  int slider;
  
  /* XXX - this should listen to set folder signal
     (that must be created), so that the label can be changed */
  
  etpan_message_list_setup(elt->message_list);
  etpan_message_list_set_folder_list(elt->message_list, tabbed->folder_list);
  etpan_message_list_set_search_field(elt->message_list, tabbed->search_field);
  etpan_message_view_setup(elt->message_view);
  
  etpan_message_view_set_message_list(elt->message_view,
      elt->message_list);
  etpan_message_view_set_status_bar(elt->message_view,
      tabbed->status_bar);
  
  etpan_ui_slider_set(etpan_ui_config_get_default(), "message-list-slider",
    elt->vpaned);
}

static void tabbed_elt_unsetup(struct tabbed_elt * elt)
{
  etpan_ui_set_from_slider(etpan_ui_config_get_default(),
      "message-list-slider",
      elt->vpaned);
  
  etpan_message_view_set_status_bar(elt->message_view, NULL);
  etpan_message_view_set_message_list(elt->message_view, NULL);
  
  etpan_message_view_unsetup(elt->message_view);
  etpan_message_list_unsetup(elt->message_list);
}

static GtkWidget * tabbed_elt_get_main_widget(struct tabbed_elt * elt)
{
  return elt->vpaned;
}

static GtkWidget * tabbed_elt_get_label_widget(struct tabbed_elt * elt)
{
  return elt->label;
}

static struct etpan_message_list *
tabbed_elt_get_message_list(struct tabbed_elt * elt)
{
  return elt->message_list;
}

static struct etpan_message_view *
tabbed_elt_get_message_view(struct tabbed_elt * elt)
{
  return elt->message_view;
}

static int tabbed_elt_get_vpaned_pos(struct tabbed_elt * elt)
{
  return gtk_paned_get_position(GTK_PANED(elt->vpaned));
}

static void tabbed_elt_set_vpaned_pos(struct tabbed_elt * elt, int paned_pos)
{
  gtk_paned_set_position(GTK_PANED(elt->vpaned), paned_pos);
}

static void tabbed_elt_copy_properties(struct
    etpan_tabbed_message_list * tabbed,
    unsigned int previous_index,
    unsigned int current_index)
{
  struct tabbed_elt * previous_elt;
  struct tabbed_elt * current_elt;
  int paned_pos;
  struct etpan_message_list * previous_msg_list;
  struct etpan_message_list * current_msg_list;
  GtkTreeViewColumn * previous_col;
  GtkTreeViewColumn * current_col;
  gint col_size;
  
  previous_elt = carray_get(tabbed->folder_info_list, previous_index);
  current_elt = carray_get(tabbed->folder_info_list, current_index);
  
  paned_pos = gtk_paned_get_position(GTK_PANED(previous_elt->vpaned));
  gtk_paned_set_position(GTK_PANED(current_elt->vpaned), paned_pos);
  
  previous_msg_list = tabbed_elt_get_message_list(previous_elt);
  current_msg_list = tabbed_elt_get_message_list(current_elt);
  
  /* from */
  previous_col = etpan_message_list_get_col_from(previous_msg_list);
  current_col = etpan_message_list_get_col_from(current_msg_list);
  
  col_size = gtk_tree_view_column_get_fixed_width(previous_col);
  gtk_tree_view_column_set_fixed_width(current_col, col_size);
  
  /* subject */
  previous_col = etpan_message_list_get_col_subject(previous_msg_list);
  current_col = etpan_message_list_get_col_subject(current_msg_list);
  
  col_size = gtk_tree_view_column_get_fixed_width(previous_col);
  gtk_tree_view_column_set_fixed_width(current_col, col_size);
  
  /* date */
  previous_col = etpan_message_list_get_col_date(previous_msg_list);
  current_col = etpan_message_list_get_col_date(current_msg_list);
  
  col_size = gtk_tree_view_column_get_fixed_width(previous_col);
  gtk_tree_view_column_set_fixed_width(current_col, col_size);
}

void etpan_tabbed_message_list_set_folder_list(struct
    etpan_tabbed_message_list * tabbed,
    struct etpan_folder_list * folder_list)
{
  tabbed->folder_list = folder_list;
  select_new_page(tabbed, tabbed->current_tab);
}

void etpan_tabbed_message_list_set_bookmark_view(struct
    etpan_tabbed_message_list * tabbed,
    struct etpan_bookmark_view * bookmark_view)
{
  tabbed->bookmark_view = bookmark_view;
  select_new_page(tabbed, tabbed->current_tab);
}

void etpan_tabbed_message_list_set_status_bar(struct
    etpan_tabbed_message_list * tabbed,
    struct etpan_status_bar * status_bar)
{
  tabbed->status_bar = status_bar;
}

void etpan_tabbed_message_list_set_search_field(struct
    etpan_tabbed_message_list * tabbed,
    struct etpan_search_field * search_field)
{
  tabbed->search_field = search_field;
}

void etpan_tabbed_message_list_close_all(struct
    etpan_tabbed_message_list * tabbed)
{
  while (carray_count(tabbed->folder_info_list) > 0)
    etpan_tabbed_message_list_close_page(tabbed, 0);
}
