#include "etpan-toolbar.h"

#include <stdlib.h>
#include <string.h>
#include "etpan-backend.h"
#include "etpan-error.h"
#include "etpan-log.h"
#include "etpan-contextual-menu.h"
#include "gtktoolbuttonprivate.h"

struct etpan_toolbar * etpan_toolbar_new(void)
{
  struct etpan_toolbar * toolbar;
  GtkWidget * toolbar_widget;
  
  toolbar = malloc(sizeof(* toolbar));
  if (toolbar == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  toolbar->button_list = carray_new(8);
  if (toolbar->button_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  toolbar_widget = gtk_toolbar_new();
  gtk_widget_show(toolbar_widget);
  gtk_toolbar_set_style(GTK_TOOLBAR(toolbar_widget), GTK_TOOLBAR_BOTH);
  
  toolbar->toolbar = toolbar_widget;
  
  return toolbar;
}

void etpan_toolbar_free(struct etpan_toolbar * toolbar)
{
  etpan_toolbar_clear(toolbar);
  carray_free(toolbar->button_list);
  gtk_widget_destroy(toolbar->toolbar);
  free(toolbar);
}

GtkWidget * etpan_toolbar_get_main_widget(struct etpan_toolbar *
    toolbar)
{
  return toolbar->toolbar;
}

enum {
  ETPAN_TOOLBAR_ITEM_TYPE_SEPARATOR,
  ETPAN_TOOLBAR_ITEM_TYPE_BUTTON,
  ETPAN_TOOLBAR_ITEM_TYPE_TOGGLE_BUTTON,
  ETPAN_TOOLBAR_ITEM_TYPE_CUSTOM,
};

struct toolbar_item {
  int type;
  GtkWidget * button;
  gulong signal_id;
  int index;
  struct etpan_toolbar * toolbar;
  struct etpan_contextual_menu * menu;
};

static void button_clicked(GtkToolButton * toolbutton, gpointer user_data)
{
  struct toolbar_item * item;
  (void) toolbutton;
  
  item = user_data;
  
  ETPAN_LOG("button %i", item->index);
  etpan_signal_send(etpan_signal_manager_get_default(),
      ETPAN_TOOLBAR_BUTTONCLICKED_SIGNAL, item->toolbar, &item->index);
}

static void set_item_info(struct etpan_toolbar * toolbar,
    unsigned int item_index, GtkWidget * button)
{
  gulong signal_id;
  struct toolbar_item * item;
  
  item = carray_get(toolbar->button_list, item_index);
  
  signal_id = g_signal_connect(button, "clicked",
      G_CALLBACK(button_clicked), item);
  
  item->type = ETPAN_TOOLBAR_ITEM_TYPE_BUTTON;
  item->button = button;
  item->signal_id = signal_id;
  item->index = item_index;
  item->toolbar = toolbar;
  item->menu = NULL;
}

void etpan_toolbar_add_item(struct etpan_toolbar * toolbar,
    char * label, GtkWidget * image)
{
  GtkWidget * button;
  int r;
  struct toolbar_item * item;
  gulong signal_id;
  
  item = malloc(sizeof(* item));
  if (item == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  button = (GtkWidget *) gtk_tool_button_new(image, label);
  gtk_widget_show(button);
  /* TODO tooltip should be added */
  gtk_toolbar_insert(GTK_TOOLBAR(toolbar->toolbar),
      GTK_TOOL_ITEM(button), -1);
  
  r = carray_add(toolbar->button_list, item, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  set_item_info(toolbar, carray_count(toolbar->button_list) - 1, button);
}

void etpan_toolbar_replace_item(struct etpan_toolbar * toolbar,
    int position, char * label, GtkWidget * image)
{
  struct toolbar_item * item;
  GtkWidget * button;
  gulong signal_id;
  
  item = carray_get(toolbar->button_list, position);
  
  if (item->type == ETPAN_TOOLBAR_ITEM_TYPE_BUTTON)
    g_signal_handler_disconnect(item->button, item->signal_id);
  
  gtk_widget_destroy(item->button);
   
  button = (GtkWidget *) gtk_tool_button_new(image, label);
  gtk_widget_show(button);
  /* TODO tooltip should be added */
  gtk_toolbar_insert(GTK_TOOLBAR(toolbar->toolbar),
      GTK_TOOL_ITEM(button), position);
  
  set_item_info(toolbar, position, button);
}

static void add_separator(struct etpan_toolbar * toolbar, gboolean visible)
{
  GtkWidget * separator;
  int r;
  struct toolbar_item * item;
  
  item = malloc(sizeof(* item));
  if (item == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  separator = (GtkWidget *) gtk_separator_tool_item_new();
  gtk_separator_tool_item_set_draw(GTK_SEPARATOR_TOOL_ITEM(separator),
      visible);
  gtk_widget_show(separator);
  gtk_toolbar_insert(GTK_TOOLBAR(toolbar->toolbar),
      GTK_TOOL_ITEM(separator), -1);
  
  item->type = ETPAN_TOOLBAR_ITEM_TYPE_SEPARATOR;
  item->button = separator;
  item->signal_id = 0;
  item->index = carray_count(toolbar->button_list);
  item->toolbar = toolbar;
  item->menu = NULL;
  
  r = carray_add(toolbar->button_list, item, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

void etpan_toolbar_add_separator(struct etpan_toolbar * toolbar)
{
  return add_separator(toolbar, TRUE);
}

void etpan_toolbar_add_widget(struct etpan_toolbar * toolbar,
    GtkWidget * widget)
{
  int r;
  struct toolbar_item * item;
  GtkWidget * tool_item;
  
  item = malloc(sizeof(* item));
  if (item == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  tool_item = (GtkWidget *) gtk_tool_item_new();
  if (widget != NULL)
    gtk_container_add(GTK_CONTAINER(tool_item), widget);
  gtk_widget_show(GTK_WIDGET(tool_item));
  
  gtk_toolbar_insert(GTK_TOOLBAR(toolbar->toolbar),
      GTK_TOOL_ITEM(tool_item), -1);
  
  item->type = ETPAN_TOOLBAR_ITEM_TYPE_CUSTOM;
  item->button = tool_item;
  item->signal_id = 0;
  item->index = carray_count(toolbar->button_list);
  item->toolbar = toolbar;
  item->menu = NULL;
  
  r = carray_add(toolbar->button_list, item, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

void etpan_toolbar_add_expander(struct etpan_toolbar * toolbar)
{
  struct toolbar_item * item;
  
  etpan_toolbar_add_widget(toolbar, NULL);
  
  item = carray_get(toolbar->button_list,
      carray_count(toolbar->button_list) - 1);
  gtk_tool_item_set_expand(GTK_TOOL_ITEM(item->button), TRUE);
}

void etpan_toolbar_add_space(struct etpan_toolbar * toolbar)
{
  add_separator(toolbar, FALSE);
}

void etpan_toolbar_clear(struct etpan_toolbar * toolbar)
{
  unsigned int i;
  int r;
  
  for(i = 0 ; i < carray_count(toolbar->button_list) ; i ++) {
    struct toolbar_item * item;
    
    item = carray_get(toolbar->button_list, i);
    
    if (item->type == ETPAN_TOOLBAR_ITEM_TYPE_BUTTON)
      g_signal_handler_disconnect(item->button, item->signal_id);
    
    gtk_widget_destroy(item->button);
    free(item);
  }
  
  r = carray_set_size(toolbar->button_list, 0);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

GtkWidget * etpan_toolbar_get_item_at_position(struct etpan_toolbar * toolbar,
    int position)
{
  struct toolbar_item * item;
  
  item = carray_get(toolbar->button_list, position);
  return item->button;
}

static void menu_position(GtkMenu * menu, gint * px, gint * py,
    gboolean * ppush_in, gpointer user_data)
{
  gint x;
  gint y;
  gint width;
  gint height;
  gint depth;
  GtkWidget * button;
  struct toolbar_item * item;
  GtkWidget * widget;
  
  item = user_data;
  button = item->button;
  gdk_window_get_deskrelative_origin(button->window, &x, &y);
  x += button->allocation.x;
  y += button->allocation.y;
  width = button->allocation.width;
  height = button->allocation.height;
  
  * px = x;
  * py = y + height;
  
  * ppush_in = 0;
}

static void show_menu(struct toolbar_item * item)
{
  int button;
  int event_time;
  GtkWidget * menu_widget;
  
  ETPAN_LOG("button %i", item->index);
  gtk_toggle_tool_button_set_active(GTK_TOGGLE_TOOL_BUTTON(item->button), TRUE);
  button = 0;
  event_time = gtk_get_current_event_time();
  menu_widget = etpan_contextual_menu_get_main_widget(item->menu);
  gtk_menu_popup(GTK_MENU(menu_widget), NULL, NULL, menu_position, item,
      button, event_time);
}

static gboolean button_pressed(GtkToggleButton * toolbutton,
    GdkEventButton * event,
    gpointer user_data)
{
  struct toolbar_item * item;
  (void) toolbutton;
  
  item = user_data;
  show_menu(item);
  
  return TRUE;
}

static void button_toggled(GtkToggleToolButton * toggle_tool_button,
    gpointer user_data) 
{
  struct toolbar_item * item;
  
  if (!gtk_toggle_tool_button_get_active(toggle_tool_button))
    return;
  
  item = user_data;
  
  show_menu(item);
}

static gboolean menu_deactivate_cb(GtkMenuShell * menu_shell,
    gpointer user_data)
{
  struct toolbar_item * item;
  
  item = user_data;
  gtk_toggle_tool_button_set_active(GTK_TOGGLE_TOOL_BUTTON(item->button), FALSE);
  
  return TRUE;
}

void etpan_toolbar_add_item_menu(struct etpan_toolbar * toolbar,
    char * label, GtkWidget * image,
    struct etpan_contextual_menu * menu)
{
  GtkWidget * button;
  int r;
  struct toolbar_item * item;
  gulong signal_id;
  GtkWidget * toggle_button;
  GtkWidget * menu_widget;
  struct _GtkToolButtonPrivate * priv;
  
  item = malloc(sizeof(* item));
  if (item == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  button = (GtkWidget *) gtk_toggle_tool_button_new();
  gtk_tool_button_set_label(GTK_TOOL_BUTTON(button), label);
  gtk_tool_button_set_icon_widget(GTK_TOOL_BUTTON(button), image);
  
  gtk_widget_show(button);
  gtk_toolbar_insert(GTK_TOOLBAR(toolbar->toolbar),
      GTK_TOOL_ITEM(button), -1);
  
  r = carray_add(toolbar->button_list, item, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  priv = GTK_TOOL_BUTTON(button)->priv;
  toggle_button = priv->button;
  g_signal_connect(toggle_button, "button_press_event",
      G_CALLBACK(button_pressed), item);
  signal_id = g_signal_connect(button, "toggled",
      G_CALLBACK(button_toggled), item);
  menu_widget = etpan_contextual_menu_get_main_widget(menu);
  g_signal_connect(menu_widget, "deactivate",
      G_CALLBACK(menu_deactivate_cb), item);
  
  item->type = ETPAN_TOOLBAR_ITEM_TYPE_TOGGLE_BUTTON;
  item->button = button;
  item->signal_id = signal_id;
  item->index = carray_count(toolbar->button_list) - 1;
  item->toolbar = toolbar;
  item->menu = menu;
}
