#include "etpan-ui-config.h"

#include <stdlib.h>
#include "etpan-backend.h"

struct window_position {
  int left;
  int top;
  int width;
  int height;
};

struct etpan_ui_config * etpan_ui_config_new(void)
{
  struct etpan_ui_config * config;
  
  config = malloc(sizeof(* config));
  if (config == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  config->window_position = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYALL);
  if (config->window_position == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  config->values = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYALL);
  if (config->values == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  config->font_other = NULL;
  config->font_message = NULL;
  config->font_list = NULL;
  
  return config;
}

void etpan_ui_config_free(struct etpan_ui_config * config)
{
  free(config->font_other);
  free(config->font_message);
  free(config->font_list);
  chash_free(config->values);
  chash_free(config->window_position);
  free(config);
}

static struct etpan_ui_config * default_config = NULL;

void etpan_ui_config_set_default(struct etpan_ui_config * config)
{
  default_config = config;
}

struct etpan_ui_config * etpan_ui_config_get_default(void)
{
  return default_config;
}

static int get_value(struct etpan_ui_config * config, char * name,
    int * p_value)
{
  int r;
  chashdatum key;
  chashdatum value;
  
  key.data = name;
  key.len = strlen(name) + 1;
  r = chash_get(config->values, &key, &value);
  if (r < 0)
    return -1;
  
  * p_value = * (int *) value.data;
  
  return 0;
}

static void set_value(struct etpan_ui_config * config, char * name,
    int int_value)
{
  int r;
  chashdatum key;
  chashdatum value;
  
  key.data = name;
  key.len = strlen(name) + 1;
  value.data = &int_value;
  value.len = sizeof(int_value);
  r = chash_set(config->values, &key, &value, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

static inline char * get_entry(tcconf_section_t * conf, char * name)
{
  int r;
  char * value;
  
  r = tcconf_getvalue(conf, name, "%s", &value);
  if (r <= 0)
    return NULL;
  
  return value;
}

int etpan_ui_config_get_slider(struct etpan_ui_config * config, char * name,
    int * p_slider)
{
  return get_value(config, name, p_slider);
}

int etpan_ui_config_get_column(struct etpan_ui_config * config, char * name,
    int * p_column)
{
  return get_value(config, name, p_column);
}

int etpan_ui_config_get_window(struct etpan_ui_config * config, char * name,
    int * p_left, int * p_top, int * p_width, int * p_height)
{
  struct window_position * pos;
  int r;
  chashdatum key;
  chashdatum value;
  
  key.data = name;
  key.len = strlen(name) + 1;
  r = chash_get(config->window_position, &key, &value);
  if (r < 0)
    return -1;
  
  pos = value.data;
  
  * p_left = pos->left;
  * p_top = pos->top;
  * p_width = pos->width;
  * p_height = pos->height;
  
  return 0;
}

static int read_window(struct etpan_ui_config * config,
    tcconf_section_t * configfile, char * name)
{
  char attr_name[256];
  int left;
  int top;
  int width;
  int height;
  int r;
  chashdatum key;
  chashdatum value;
  struct window_position position;
  
  snprintf(attr_name, sizeof(attr_name), "%s/left", name);
  r = tcconf_getvalue(configfile, attr_name, "%i", &left);
  if (r <= 0)
    return -1;
  
  snprintf(attr_name, sizeof(attr_name), "%s/top", name);
  r = tcconf_getvalue(configfile, attr_name, "%i", &top);
  if (r <= 0)
    return -1;
  
  snprintf(attr_name, sizeof(attr_name), "%s/width", name);
  r = tcconf_getvalue(configfile, attr_name, "%i", &width);
  if (r <= 0)
    return -1;
  
  snprintf(attr_name, sizeof(attr_name), "%s/height", name);
  r = tcconf_getvalue(configfile, attr_name, "%i", &height);
  if (r <= 0)
    return -1;
  
  position.left = left;
  position.top = top;
  position.width = width;
  position.height = height;
  
  key.data = name;
  key.len = strlen(name) + 1;
  value.data = &position;
  value.len = sizeof(position);
  r = chash_set(config->window_position, &key, &value, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  return 0;
}

static int read_value(struct etpan_ui_config * config,
    tcconf_section_t * configfile,
    char * name)
{
  int r;
  int int_value;
  
  r = tcconf_getvalue(configfile, name, "%i", &int_value);
  if (r <= 0)
    return -1;
  
  set_value(config, name, int_value);
  
  return 0;
}

struct etpan_error * etpan_ui_config_read(struct etpan_ui_config * config,
    char * filename)
{
  tcconf_section_t * configfile;
  char * message_font;
  char * list_font;
  char * other_font;
  
  etpan_ui_config_set_font_other(etpan_ui_config_get_default(), "Sans 11");
  etpan_ui_config_set_font_message(etpan_ui_config_get_default(), "Monospace 10");
  etpan_ui_config_set_font_list(etpan_ui_config_get_default(), "Sans 11");
  
  configfile = tcconf_load_file(NULL, filename);
  if (configfile == NULL) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_PARSE);
    etpan_error_set_short_description(error, _("Parse error"));
    etpan_error_strf_long_description(error, _("User Interface configuration at %s could not be parsed."), filename);
    
    return error;
  }
  
  read_value(config, configfile, "column-from-size");
  read_value(config, configfile, "column-subject-size");
  read_value(config, configfile, "folder-list-slider");
  read_value(config, configfile, "message-list-slider");
  read_value(config, configfile, "account-preferences-slider");
  read_window(config, configfile, "preferences");
  read_window(config, configfile, "main");
  read_window(config, configfile, "composer");
  
  message_font = get_entry(configfile, "message-font");
  list_font = get_entry(configfile, "list-font");
  other_font = get_entry(configfile, "other-font");
  
  if (message_font != NULL) {
    etpan_ui_config_set_font_message(etpan_ui_config_get_default(),
        message_font);
    free(message_font);
  }
  
  if (list_font != NULL) {
    etpan_ui_config_set_font_list(etpan_ui_config_get_default(),
        list_font);
    free(list_font);
  }
  
  if (other_font != NULL) {
    etpan_ui_config_set_font_other(etpan_ui_config_get_default(),
        other_font);
    free(other_font);
  }
  
  tcfree(configfile);
  
  return NULL;
}

struct etpan_error * etpan_ui_config_read_default(void)
{
  char filename[PATH_MAX];
  char * home;
  
  home = getenv("HOME");
  if (home == NULL) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_INVAL);
    etpan_error_set_short_description(error, _("Home directory not found"));
    etpan_error_strf_long_description(error, _("Home directory is not defined. It should be defined in environment variable HOME."));
    
    return error;
  }
  
  snprintf(filename, sizeof(filename), "%s/%s", home, ETPAN_UI_CONFIG);
  
  return etpan_ui_config_read(etpan_ui_config_get_default(), filename);
}

struct etpan_error * etpan_ui_config_save(struct etpan_ui_config * config,
    char * filename)
{
  char * font_name;
  chashiter * iter;
  FILE * f;
  
  f = fopen(filename, "w");
  if (f == NULL) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FILE);
    etpan_error_set_short_description(error, _("Configuration file could not be written"));
    etpan_error_strf_long_description(error, _("User interface configuration file %s could not be written to disk."), filename);
    
    return error;
  }
  
  for(iter = chash_begin(config->values) ; iter != NULL ;
      iter = chash_next(config->values, iter)) {
    chashdatum key;
    char * keystr;
    chashdatum value;
    int keyvalue;
    
    chash_key(iter, &key);
    keystr = key.data;
    chash_value(iter, &value);
    memcpy(&keyvalue, value.data, sizeof(keyvalue));
    fprintf(f, "%s %i\n", keystr, keyvalue);
  }
  
  for(iter = chash_begin(config->window_position) ; iter != NULL ;
      iter = chash_next(config->window_position, iter)) {
    chashdatum key;
    char * keystr;
    chashdatum value;
    struct window_position position;
    
    chash_key(iter, &key);
    keystr = key.data;
    chash_value(iter, &value);
    memcpy(&position, value.data, sizeof(position));
    fprintf(f, "%s {\n", keystr);
    fprintf(f, "  left %i\n", position.left);
    fprintf(f, "  top %i\n", position.top);
    fprintf(f, "  width %i\n", position.width);
    fprintf(f, "  height %i\n", position.height);
    fprintf(f, "}\n");
  }
  
  font_name = etpan_ui_config_get_font_message(etpan_ui_config_get_default());
  fprintf(f, "message-font \"%s\"\n", font_name);
  font_name = etpan_ui_config_get_font_list(etpan_ui_config_get_default());
  fprintf(f, "list-font \"%s\"\n", font_name);
  font_name = etpan_ui_config_get_font_other(etpan_ui_config_get_default());
  fprintf(f, "other-font \"%s\"\n", font_name);
  
  fclose(f);
  
  return NULL;
}  

struct etpan_error * etpan_ui_config_save_default(void)
{
  char filename[PATH_MAX];
  char * home;
  
  home = getenv("HOME");
  if (home == NULL) {
    struct etpan_error * error;
    
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_INVAL);
    etpan_error_set_short_description(error, _("Home directory not found"));
    etpan_error_strf_long_description(error, _("Home directory is not defined. It should be defined in environment variable HOME."));
    
    return error;
  }
  
  snprintf(filename, sizeof(filename), "%s/%s", home, ETPAN_UI_CONFIG);
  
  return etpan_ui_config_save(etpan_ui_config_get_default(), filename);
}

void etpan_ui_config_set_font_message(struct etpan_ui_config * ui_config,
    char * font_message)
{
  if (font_message != ui_config->font_message) {
    free(ui_config->font_message);
    if (font_message != NULL) {
      ui_config->font_message = strdup(font_message);
      if (ui_config->font_message == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      ui_config->font_message = NULL;
    
    etpan_signal_send(etpan_signal_manager_get_default(),
        ETPAN_UI_CONFIG_FONT_CHANGED_SIGNAL,
        ui_config, NULL);
  }
}

void etpan_ui_config_set_font_list(struct etpan_ui_config * ui_config,
    char * font_list)
{
  if (font_list != ui_config->font_list) {
    free(ui_config->font_list);
    if (font_list != NULL) {
      ui_config->font_list = strdup(font_list);
      if (ui_config->font_list == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      ui_config->font_list = NULL;
    
    etpan_signal_send(etpan_signal_manager_get_default(),
        ETPAN_UI_CONFIG_FONT_CHANGED_SIGNAL,
        ui_config, NULL);
  }
}

void etpan_ui_config_set_font_other(struct etpan_ui_config * ui_config,
    char * font_other)
{
  if (font_other != ui_config->font_other) {
    free(ui_config->font_other);
    if (font_other != NULL) {
      ui_config->font_other = strdup(font_other);
      if (ui_config->font_other == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else
      ui_config->font_other = NULL;
    
    etpan_signal_send(etpan_signal_manager_get_default(),
        ETPAN_UI_CONFIG_FONT_CHANGED_SIGNAL,
        ui_config, NULL);
  }
}

char * etpan_ui_config_get_font_message(struct etpan_ui_config * ui_config)
{
  return ui_config->font_message;
}

char * etpan_ui_config_get_font_list(struct etpan_ui_config * ui_config)
{
  return ui_config->font_list;
}

char * etpan_ui_config_get_font_other(struct etpan_ui_config * ui_config)
{
  return ui_config->font_other;
}

void etpan_ui_set_from_window(struct etpan_ui_config * config,
    char * name,
    GtkWidget * window)
{
  struct window_position position;
  int r;
  gint left;
  gint top;
  gint width;
  gint height;
  chashdatum key;
  chashdatum value;
  
  gtk_window_get_position(GTK_WINDOW(window), &left, &top);
  gtk_window_get_size(GTK_WINDOW(window), &width, &height);
  
  position.left = left;
  position.top = top;
  position.width = width;
  position.height = height;
  key.data = name;
  key.len = strlen(name) + 1;
  value.data = &position;
  value.len = sizeof(position);
  r = chash_set(config->window_position, &key, &value, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

void etpan_ui_window_set(struct etpan_ui_config * config,
    char * name,
    GtkWidget * window)
{
  int r;
  int left;
  int top;
  int width;
  int height;
  
  r = etpan_ui_config_get_window(config, name,
    &left, &top, &width, &height);
  if (r == 0) {
    gtk_window_move(GTK_WINDOW(window), left, top);
    gtk_window_set_default_size(GTK_WINDOW(window), width, height);
  }
}

void etpan_ui_set_from_slider(struct etpan_ui_config * config,
    char * name,
    GtkWidget * paned)
{
  int value;
  
  value = gtk_paned_get_position(GTK_PANED(paned));
  set_value(config, name, value);
}

void etpan_ui_slider_set(struct etpan_ui_config * config,
    char * name,
    GtkWidget * paned)
{
  int r;
  int slider;
  
  r = etpan_ui_config_get_slider(config, name, &slider);
  if (r == 0) {
    gtk_paned_set_position(GTK_PANED(paned), slider);
  }
}

void etpan_ui_set_from_column(struct etpan_ui_config * config,
    char * name, GtkWidget * treeview, int n)
{
  int value;
  GtkTreeViewColumn * column;
  
  column = gtk_tree_view_get_column(GTK_TREE_VIEW(treeview), n);
  value = gtk_tree_view_column_get_width(column);
  set_value(config, name, value);
}

void etpan_ui_column_set(struct etpan_ui_config * config,
    char * name, GtkWidget * treeview, int n)
{
  int r;
  int value;
  GtkTreeViewColumn * column;
  
  column = gtk_tree_view_get_column(GTK_TREE_VIEW(treeview), n);
  r = get_value(config, name, &value);
  if (r == 0) {
    gtk_tree_view_column_set_fixed_width(column, value);
  }
}
