#include "etpan-preferences-abook.h"

#include <sys/stat.h>
#include <sys/types.h>

#include "etpan-backend.h"
#include "etpan-preferences-window.h"
#include "etpan-gtk-tree-model.h"
#include "etpan-abook.h"
#include "etpan-abook-manager.h"
#include "etpan-icon-manager.h"

#define PADDING 5

struct panel_data {
  int modified;
  struct etpan_abook * abook_being_modified;
  struct etpan_gtk_tree_data_source abook_datasource;
  etpan_gtk_tree_model * treemodel;
  struct etpan_abook * pending_abook;
  char * chosen_path;
  int chooser_state;
  int abook_set;
};

enum {
  ABOOK_DRIVER_TYPE_LDAP,
  ABOOK_DRIVER_TYPE_VCARD,
};

enum {
  CHOOSER_STATE_IDLE,
  CHOOSER_STATE_SETTING,
};

static void set_add_remove_state(struct etpan_preferences_panel * panel);
static void stop_add(struct etpan_preferences_panel * panel);
static void add_abook(struct etpan_preferences_panel * panel);
static void remove_abook(struct etpan_preferences_panel * panel);
static void abook_selected_handler(GtkTreeView * treeview,
    gpointer user_data);
static void add_pressed_handler(GtkButton *button, gpointer user_data);
static void remove_pressed_handler(GtkButton *button, gpointer user_data);
static void setup(struct etpan_preferences_panel * panel);
static void create_pressed_handler(GtkButton *button, gpointer user_data);
static void cancel_pressed_handler(GtkButton *button, gpointer user_data);
static void apply_pressed_handler(GtkButton *button, gpointer user_data);
static void revert_pressed_handler(GtkButton *button, gpointer user_data);
static void reset_properties(struct etpan_preferences_panel * panel);
static void reset_specific_properties(struct etpan_preferences_panel * panel);
static void abook_set(struct etpan_preferences_panel * panel,
    struct etpan_abook * abook);
static void vcard_abook_set(struct etpan_preferences_panel * panel,
    struct etpan_abook * abook);
static void ldap_abook_set(struct etpan_preferences_panel * panel,
    struct etpan_abook * abook);
static void file_chooser_changed(GtkFileChooser * chooser,
    gpointer user_data);
static void file_chooser_folder_changed(GtkFileChooser * chooser,
    gpointer user_data);
static void local_type_changed(GtkComboBox * widget, gpointer user_data);
static void entry_changed(GtkEditable *editable,
    gpointer user_data);
static void combo_changed(GtkComboBox * widget, gpointer user_data);
static void checkbox_changed(GtkToggleButton * togglebutton,
    gpointer user_data);
static void abook_type_changed(GtkComboBox * widget, gpointer user_data);
static void ldap_ssl_toggled(GtkToggleButton * togglebutton, gpointer user_data);
static void ldap_auth_type_changed(GtkComboBox * widget, gpointer user_data);

static void create_abook(struct etpan_preferences_panel * panel,
    int keep_selection);
static void create_ldap_abook(struct etpan_preferences_panel * panel);
static void create_local_abook(struct etpan_preferences_panel * panel);
static void create_vcard_abook(struct etpan_preferences_panel * panel);

static void modify_abook(struct etpan_preferences_panel * panel,
    int keep_selection);
static void modify_ldap_abook(struct etpan_preferences_panel * panel,
    struct etpan_abook * abook);
static void modify_local_abook(struct etpan_preferences_panel * panel,
    struct etpan_abook * abook);
static void modify_vcard_abook(struct etpan_preferences_panel * panel,
    struct etpan_abook * abook);
static void modify_abook_common(struct etpan_preferences_panel * panel,
    struct etpan_abook * abook);

static int panel_should_close(struct etpan_preferences_panel * panel);
static void panel_close(struct etpan_preferences_panel * panel);
static void panel_open(struct etpan_preferences_panel * panel);

static int get_abook_type(struct etpan_abook * abook);
static char * get_abook_type_str(int value);

static void abook_selected_handler(GtkTreeView * treeview,
    gpointer user_data);

static void enable_buttons(struct etpan_preferences_panel * panel);
static int check_consistency(struct etpan_preferences_panel * panel);
static int is_empty(struct etpan_preferences_panel * panel);

static void switch_text(struct etpan_preferences_panel * panel,
    char * name, char * value);

/* *********************************** */
/* data source */

enum {
  STORE_INDEX_NAME,
  STORE_INDEX_TYPE,
  STORE_INDEX_ABOOK,
};

static unsigned int get_n_columns(struct etpan_gtk_tree_data_source *
    datasource)
{
  (void) datasource;
  
  return 3;
}

static GType get_column_type(struct etpan_gtk_tree_data_source * datasource,
    unsigned int column_index)
{
  static GType column_types[3] = {
    G_TYPE_STRING, G_TYPE_STRING,
    G_TYPE_POINTER
  };
  (void) datasource;
  
  return column_types[column_index];
}
  
static unsigned int get_children_count(struct etpan_gtk_tree_data_source *
    datasource, void * item)
{
  carray * list;
  struct etpan_preferences_panel * panel;
  struct panel_data * data;
  
  panel = datasource->data;
  data = panel->data;
  list = etpan_abook_manager_get_ordered_list(etpan_abook_manager_get_default());
  
  if (item == NULL) {
    if (data->pending_abook != NULL)
      return carray_count(list) + 1;
    else
      return carray_count(list);
  }
  else
    return 0;
}

static int item_has_child(struct etpan_gtk_tree_data_source *
    datasource, void * item)
{
  carray * list;
  (void) datasource;
  
  list = etpan_abook_manager_get_ordered_list(etpan_abook_manager_get_default());
  
  if (item == NULL)
    return carray_count(list) > 0;
  else
    return 0;
}

static void * get_child_item(struct etpan_gtk_tree_data_source *
    datasource, void * item, unsigned int index)
{
  carray * list;
  struct etpan_preferences_panel * panel;
  struct panel_data * data;
  
  panel = datasource->data;
  data = panel->data;
  
  list = etpan_abook_manager_get_ordered_list(etpan_abook_manager_get_default());
  
  if (item == NULL) {
    if (index >= carray_count(list))
      return data->pending_abook;
    else
      return carray_get(list, index);
  }
  else
    return NULL;
}

static void get_item_value(struct etpan_gtk_tree_data_source *
    datasource, void * item, unsigned int column_index,
    GValue * value)
{
  struct panel_data * data;
  struct etpan_preferences_panel * panel;
  
  panel = datasource->data;
  data = panel->data;
  
  if (item == data->pending_abook) {
    switch (column_index) {
    case STORE_INDEX_NAME:
      {
        char * str;
      
        str = _("(New Address Book)");
      
        g_value_init(value, G_TYPE_STRING);
        g_value_set_string(value, str);
      }
      break;
    case STORE_INDEX_TYPE:
      {
        char * str;
      
        str = "";
      
        g_value_init(value, G_TYPE_STRING);
        g_value_set_string(value, str);
      
      }
      break;
    case STORE_INDEX_ABOOK:
      {
        g_value_init(value, G_TYPE_POINTER);
        g_value_set_pointer(value, NULL);
      }
      break;
    }
  }
  else {
    struct etpan_abook * abook;
    
    abook = item;
    
    switch (column_index) {
    case STORE_INDEX_NAME:
      {
        char * str;
        
        str = etpan_abook_get_id(abook);
        
        g_value_init(value, G_TYPE_STRING);
        g_value_set_string(value, str);
      }
      break;
    case STORE_INDEX_TYPE:
      {
        char * str;
        int type;
        
        type = get_abook_type(abook);
        str = get_abook_type_str(type);
        
        g_value_init(value, G_TYPE_STRING);
        g_value_set_string(value, str);
      
      }
      break;
    case STORE_INDEX_ABOOK:
      {
        g_value_init(value, G_TYPE_POINTER);
        g_value_set_pointer(value, abook);
      }
      break;
    }
  }
}

/* *********************************** */
/* build UI */

static GtkWidget * local_properties(struct etpan_preferences_panel * panel)
{
  GtkWidget * label_local_type;
  GtkWidget * label_local_type_value;
  GtkWidget * combo_local_type;
  GtkWidget * label_local_path;
  GtkWidget * chooser_local_path;
  GtkWidget * chooser_local_path_folder;
  int line;
  GtkWidget * table;
  gint signal_id;
  
  table = gtk_table_new(1, 2, FALSE);
  line = 0;
  
  /* abook type */
  gtk_table_resize(GTK_TABLE(table), line + 2, 3);
  label_local_type = gtk_label_new(_("Type"));
  gtk_widget_show(label_local_type);
  gtk_table_attach(GTK_TABLE(table), label_local_type, 0, 1, line, line + 2,
      0, GTK_SHRINK, /* attach options */
      PADDING, PADDING);
  combo_local_type = gtk_combo_box_new_text();
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_local_type), _("V-Card"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_local_type), _("V-Card Folder"));
#if 0
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_local_type), _("abook"));
#endif
  gtk_combo_box_set_active(GTK_COMBO_BOX(combo_local_type), 0);
  gtk_widget_show(combo_local_type);
  gtk_table_attach(GTK_TABLE(table), combo_local_type, 1, 2, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  label_local_type_value = gtk_label_new(_("Undefined"));
  gtk_misc_set_alignment(GTK_MISC(label_local_type_value), 0, 0.5);
  gtk_widget_show(label_local_type_value);
  gtk_table_attach(GTK_TABLE(table), label_local_type_value, 1, 2, line + 1, line + 2,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line += 2;
  
  /* path */
  gtk_table_resize(GTK_TABLE(table), line + 2, 2);
  label_local_path = gtk_label_new(_("Path"));
  gtk_widget_show(label_local_path);
  gtk_table_attach(GTK_TABLE(table), label_local_path, 0, 1, line, line + 2,
      0, GTK_SHRINK, /* attach options */
      PADDING, PADDING);
  chooser_local_path = gtk_file_chooser_button_new(_("Path"),
      GTK_FILE_CHOOSER_ACTION_OPEN);
  gtk_widget_show(chooser_local_path);
  gtk_table_attach(GTK_TABLE(table), chooser_local_path, 1, 2, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  chooser_local_path_folder = gtk_file_chooser_button_new(_("Path"),
      GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER);
  gtk_widget_show(chooser_local_path_folder);
  gtk_table_attach(GTK_TABLE(table), chooser_local_path_folder, 1, 2, line + 1, line + 2,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  gtk_widget_show(table);
  
  etpan_preferences_panel_set_widget(panel, "local-type", combo_local_type);
  etpan_preferences_panel_set_widget(panel, "local-type-value", label_local_type_value);
  etpan_preferences_panel_set_widget(panel, "label-local-path", label_local_path);
  etpan_preferences_panel_set_widget(panel, "local-path", chooser_local_path);
  etpan_preferences_panel_set_widget(panel, "local-path-folder", chooser_local_path_folder);
  
  signal_id = g_signal_connect(chooser_local_path, "selection-changed",
      G_CALLBACK(file_chooser_changed), panel);
  signal_id = g_signal_connect(chooser_local_path_folder, "selection-changed",
      G_CALLBACK(file_chooser_folder_changed), panel);
  signal_id = g_signal_connect(combo_local_type, "changed",
      G_CALLBACK(local_type_changed), panel);
  signal_id = g_signal_connect(combo_local_type, "changed",
      G_CALLBACK(combo_changed), panel);
  
  return table;
}

static GtkWidget * local_frame(struct etpan_preferences_panel * panel)
{
#if 0
  GtkWidget * frame;
  GtkWidget * alignment;
  GtkWidget * properties;
  
  frame = gtk_frame_new(_("Address Book Information"));
  
  properties = local_properties(panel);
  
  alignment = gtk_alignment_new(0.5, 0.5, 1.0, 1.0);
  gtk_container_add(GTK_CONTAINER(alignment), properties);
  gtk_alignment_set_padding(GTK_ALIGNMENT(alignment), 5, 5, 10, 10);
  gtk_widget_show(alignment);
  
  gtk_container_add(GTK_CONTAINER(frame), alignment);
  
  gtk_widget_show(frame);
  
  return frame;
#else
  return local_properties(panel);
#endif
}

/* LDAP */
/*
  properties :
  - server name
  - port = 389
  - use SSL
  - auth type
  - version = 3
  - LDAP base;
  - username
  - password;
  - sasl auth info
  - filter = "(|(cn=%*)(mail=%*)(givenname=%*)(sn=%*))"
  - sizelimit = 0
  - timeout = 0
*/

static GtkWidget * ldap_optional_properties(struct etpan_preferences_panel * panel)
{
  GtkWidget * table;
  GtkWidget * label_port;
  GtkWidget * entry_port;
  GtkWidget * label_auth_type;
  GtkWidget * combo_auth_type;
  GtkWidget * label_login;
  GtkWidget * entry_login;
  GtkWidget * label_password;
  GtkWidget * entry_password;
  GtkWidget * checkbox_use_ssl;
  int line;
  gint signal_id;
  
  table = gtk_table_new(1, 3, FALSE);
  line = 0;
  
  /* port */
  gtk_table_resize(GTK_TABLE(table), line + 1, 3);
  checkbox_use_ssl = gtk_check_button_new_with_label(_("Use SSL"));
  gtk_widget_show(checkbox_use_ssl);
  gtk_table_attach(GTK_TABLE(table), checkbox_use_ssl, 0, 1, line, line + 1,
      0, GTK_SHRINK, /* attach options */
      PADDING, PADDING);
  
  gtk_table_resize(GTK_TABLE(table), line + 1, 3);
  label_port = gtk_label_new(_("Port"));
  gtk_widget_show(label_port);
  gtk_table_attach(GTK_TABLE(table), label_port, 1, 2, line, line + 1,
      0, GTK_SHRINK, /* attach options */
      PADDING, PADDING);
  entry_port = gtk_entry_new();
  gtk_widget_show(entry_port);
  gtk_table_attach(GTK_TABLE(table), entry_port, 2, 3, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  /* auth type */
  /*
    CRAM-MD5: auth_name, password
    anonymous: login
    Digest-MD5: auth_name, login, password
    GSSAPI (Kerberos V): login
    Kerberos4: login
    login: auth_name, login, password
    NTLM: auth_name, password
    OTP: auth_name, login, password
    PASSDSS-3DES-1: auth_name, login, password
    srp: auth_name, login, password
  */
  gtk_table_resize(GTK_TABLE(table), line + 1, 3);
  label_auth_type = gtk_label_new(_("Authentication Type"));
  gtk_widget_show(label_auth_type);
  gtk_table_attach(GTK_TABLE(table), label_auth_type, 0, 1, line, line + 1,
      0, GTK_SHRINK, /* attach options */
      PADDING, PADDING);
  combo_auth_type = gtk_combo_box_new_text();
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_auth_type), _("None"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_auth_type), _("Password"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_auth_type), _("Plain Login"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_auth_type), _("Challenge-Response MD5 (CRAM-MD5)"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_auth_type), _("Digest MD5"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_auth_type), _("Kerberos V (GSSAPI)"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_auth_type), _("Kerberos 4"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_auth_type), _("One-Time-Password"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_auth_type), _("DSS Secured Password (PASSDSS-3DES-1)"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_auth_type), _("Secure Remote Password (SRP)"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_auth_type), _("NTLM"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_auth_type), _("Anonymous"));
  gtk_combo_box_set_active(GTK_COMBO_BOX(combo_auth_type), 0);
  gtk_widget_show(combo_auth_type);
  gtk_table_attach(GTK_TABLE(table), combo_auth_type, 1, 3, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;

  /* login */
  gtk_table_resize(GTK_TABLE(table), line + 1, 2);
  label_login = gtk_label_new(_("Login"));
  gtk_widget_show(label_login);
  gtk_table_attach(GTK_TABLE(table), label_login, 0, 1, line, line + 1,
      0, GTK_SHRINK, /* attach options */
      PADDING, PADDING);
  entry_login = gtk_entry_new();
  gtk_widget_show(entry_login);
  gtk_table_attach(GTK_TABLE(table), entry_login, 1, 3, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  /* password */
  gtk_table_resize(GTK_TABLE(table), line + 1, 2);
  label_password = gtk_label_new(_("Password"));
  gtk_widget_show(label_password);
  gtk_table_attach(GTK_TABLE(table), label_password, 0, 1, line, line + 1,
      0, GTK_SHRINK, /* attach options */
      PADDING, PADDING);
  entry_password = gtk_entry_new();
  gtk_widget_show(entry_password);
  gtk_table_attach(GTK_TABLE(table), entry_password, 1, 3, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  gtk_widget_show(table);
  
  etpan_preferences_panel_set_widget(panel, "ldap-use-ssl", checkbox_use_ssl);
  etpan_preferences_panel_set_widget(panel, "ldap-port", entry_port);
  etpan_preferences_panel_set_widget(panel, "ldap-auth-type", combo_auth_type);
  etpan_preferences_panel_set_widget(panel, "ldap-login", entry_login);
  etpan_preferences_panel_set_widget(panel, "ldap-password", entry_password);
  
  signal_id = g_signal_connect(checkbox_use_ssl, "toggled",
      G_CALLBACK(ldap_ssl_toggled), panel);
  signal_id = g_signal_connect(combo_auth_type, "changed",
      G_CALLBACK(ldap_auth_type_changed), panel);

  signal_id = g_signal_connect(checkbox_use_ssl, "toggled",
      G_CALLBACK(checkbox_changed), panel);
  signal_id = g_signal_connect(entry_port, "changed",
      G_CALLBACK(entry_changed), panel);
  signal_id = g_signal_connect(entry_login, "changed",
      G_CALLBACK(entry_changed), panel);
  signal_id = g_signal_connect(entry_password, "changed",
      G_CALLBACK(entry_changed), panel);
  signal_id = g_signal_connect(combo_auth_type, "changed",
      G_CALLBACK(combo_changed), panel);
  
  return table;
}

static GtkWidget * ldap_properties(struct etpan_preferences_panel * panel)
{
  GtkWidget * label_server;
  GtkWidget * entry_server;
  GtkWidget * label_base;
  GtkWidget * entry_base;
  GtkWidget * server_expander;
  GtkWidget * ldap_properties;
  int line;
  GtkWidget * table;
  gint signal_id;
  
  table = gtk_table_new(1, 2, FALSE);
  line = 0;
  
  /* (for LDAP, POP) */
  /* receive server, settings ... */
  gtk_table_resize(GTK_TABLE(table), line + 1, 2);
  label_server = gtk_label_new(_("Directory server"));
  gtk_widget_show(label_server);
  gtk_table_attach(GTK_TABLE(table), label_server, 0, 1, line, line + 1,
      0, GTK_SHRINK, /* attach options */
      PADDING, PADDING);
  entry_server = gtk_entry_new();
  gtk_widget_show(entry_server);
  gtk_table_attach(GTK_TABLE(table), entry_server, 1, 2, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  /* search base */
  gtk_table_resize(GTK_TABLE(table), line + 1, 2);
  label_base = gtk_label_new(_("Search base"));
  gtk_widget_show(label_base);
  gtk_table_attach(GTK_TABLE(table), label_base, 0, 1, line, line + 1,
      0, GTK_SHRINK, /* attach options */
      PADDING, PADDING);
  entry_base = gtk_entry_new();
  gtk_widget_show(entry_base);
  gtk_table_attach(GTK_TABLE(table), entry_base, 1, 2, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  /* more settings for receive server */
  gtk_table_resize(GTK_TABLE(table), line + 1, 2);
  server_expander = gtk_expander_new(_("Settings"));
  ldap_properties = ldap_optional_properties(panel);
  
  gtk_container_add(GTK_CONTAINER(server_expander), ldap_properties);
  gtk_widget_show(server_expander);
  gtk_table_attach(GTK_TABLE(table), server_expander, 0, 2, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  gtk_widget_show(table);
  
  etpan_preferences_panel_set_widget(panel, "label-ldap-server", label_server);
  etpan_preferences_panel_set_widget(panel, "ldap-server", entry_server);
  etpan_preferences_panel_set_widget(panel, "label-ldap-base", label_base);
  etpan_preferences_panel_set_widget(panel, "ldap-base", entry_base);
  
  signal_id = g_signal_connect(entry_server, "changed",
      G_CALLBACK(entry_changed), panel);
  signal_id = g_signal_connect(entry_base, "changed",
      G_CALLBACK(entry_changed), panel);
  
  return table;
}

static GtkWidget * ldap_frame(struct etpan_preferences_panel * panel)
{
#if 0
  GtkWidget * frame;
  GtkWidget * alignment;
  GtkWidget * properties;
  
  frame = gtk_frame_new(_("Directory Server"));
  
  properties = ldap_properties(panel);
  
  alignment = gtk_alignment_new(0.5, 0.5, 1.0, 1.0);
  gtk_container_add(GTK_CONTAINER(alignment), properties);
  gtk_alignment_set_padding(GTK_ALIGNMENT(alignment), 5, 5, 10, 10);
  gtk_widget_show(alignment);
  
  gtk_container_add(GTK_CONTAINER(frame), alignment);
  
  gtk_widget_show(frame);
  
  return frame;
#else
  return ldap_properties(panel);
#endif
}

static void specific_abook_properties(struct etpan_preferences_panel * panel, GtkWidget * table)
{
  GtkWidget * frame;
  guint gline;
  int line;
  GtkWidget * separator;
  
  g_object_get(table, "n-rows", &gline, NULL);
  line = gline;
  
  gtk_table_resize(GTK_TABLE(table), line + 1, 2);
  
  separator = gtk_hseparator_new();
  gtk_widget_show(separator);
  gtk_table_attach(GTK_TABLE(table), separator, 0, 2, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  /* Local */
  gtk_table_resize(GTK_TABLE(table), line + 1, 2);
  frame = local_frame(panel);
  gtk_table_attach(GTK_TABLE(table), frame, 0, 2, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  etpan_preferences_panel_set_widget(panel, "local-properties", frame);

  /* LDAP */
  gtk_table_resize(GTK_TABLE(table), line + 1, 2);
  frame = ldap_frame(panel);
  gtk_table_attach(GTK_TABLE(table), frame, 0, 2, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  etpan_preferences_panel_set_widget(panel, "ldap-properties", frame);
}

static void basic_abook_properties(struct etpan_preferences_panel * panel,
    GtkWidget * table)
{
  GtkWidget * label_name;
  GtkWidget * entry_name;
  GtkWidget * label_abook_type;
  GtkWidget * label_abook_type_value;
  GtkWidget * combo_abook_type;
  guint gline;
  int line;
  gint signal_id;
  
  g_object_get(table, "n-rows", &gline, NULL);
  line = gline;
  
  /* abook name */
  gtk_table_resize(GTK_TABLE(table), line + 1, 2);
  label_name = gtk_label_new(_("Name"));
  gtk_widget_show(label_name);
  gtk_table_attach(GTK_TABLE(table), label_name, 0, 1, line, line + 1,
      0, GTK_SHRINK, /* attach options */
      PADDING, PADDING);
  entry_name = gtk_entry_new();
  gtk_widget_show(entry_name);
  gtk_table_attach(GTK_TABLE(table), entry_name, 1, 2, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  /* abook type */
  gtk_table_resize(GTK_TABLE(table), line + 2, 2);
  label_abook_type = gtk_label_new(_("Address Book Type"));
  gtk_widget_show(label_abook_type);
  gtk_table_attach(GTK_TABLE(table), label_abook_type, 0, 1, line, line + 2,
      0, GTK_SHRINK, /* attach options */
      PADDING, PADDING);
  
  label_abook_type_value = gtk_label_new(_("Undefined"));
  gtk_misc_set_alignment(GTK_MISC(label_abook_type_value), 0, 0.5);
  gtk_widget_show(label_abook_type_value);
  gtk_table_attach(GTK_TABLE(table), label_abook_type_value, 1, 2, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  
  combo_abook_type = gtk_combo_box_new_text();
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_abook_type),
      get_abook_type_str(0));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_abook_type),
      get_abook_type_str(1));
  gtk_combo_box_set_active(GTK_COMBO_BOX(combo_abook_type), 0);
  
  gtk_widget_show(combo_abook_type);
  gtk_table_attach(GTK_TABLE(table), combo_abook_type, 1, 2, line + 1, line + 2,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line += 2;
  
  etpan_preferences_panel_set_widget(panel, "name", entry_name);
  etpan_preferences_panel_set_widget(panel, "label-name", label_name);
  etpan_preferences_panel_set_widget(panel, "abook-type", combo_abook_type);
  etpan_preferences_panel_set_widget(panel, "abook-type-label", label_abook_type_value);

  signal_id = g_signal_connect(combo_abook_type, "changed",
      G_CALLBACK(abook_type_changed), panel);
  
  signal_id = g_signal_connect(entry_name, "changed",
      G_CALLBACK(entry_changed), panel);
  signal_id = g_signal_connect(combo_abook_type, "changed",
      G_CALLBACK(combo_changed), panel);
}

static GtkWidget * abook_properties(struct etpan_preferences_panel * panel)
{
  GtkWidget * table;
  GtkWidget * scrolledwindow;
  guint gline;
  int line;
  GtkWidget * button_hbox;
  GtkWidget * create_button;
  GtkWidget * cancel_button;
  GtkWidget * revert_button;
  GtkWidget * apply_button;
  gint create_signal_id;
  gint cancel_signal_id;
  gint revert_signal_id;
  gint apply_signal_id;
  GtkWidget * label_info;
  
  table = gtk_table_new(1, 2, FALSE);
  
  basic_abook_properties(panel, table);
  specific_abook_properties(panel, table);
  
  g_object_get(table, "n-rows", &gline, NULL);
  line = gline;
  
  gtk_table_resize(GTK_TABLE(table), line + 1, 2);
  
  button_hbox = gtk_hbutton_box_new();
  gtk_button_box_set_layout(GTK_BUTTON_BOX(button_hbox), GTK_BUTTONBOX_END);
  gtk_widget_show(button_hbox);
  
  cancel_button = gtk_button_new_with_label(_("Cancel"));
  gtk_widget_show(cancel_button);
  gtk_box_pack_start(GTK_BOX(button_hbox), cancel_button, FALSE, FALSE, 0);
  
  create_button = gtk_button_new_with_label(_("Create"));
  gtk_widget_show(create_button);
  gtk_box_pack_start(GTK_BOX(button_hbox), create_button, FALSE, FALSE, 0);
  
  revert_button = gtk_button_new_with_label(_("Revert"));
  gtk_widget_show(revert_button);
  gtk_box_pack_start(GTK_BOX(button_hbox), revert_button, FALSE, FALSE, 0);
  
  apply_button = gtk_button_new_with_label(_("Apply"));
  gtk_widget_show(apply_button);
  gtk_box_pack_start(GTK_BOX(button_hbox), apply_button, FALSE, FALSE, 0);
  
  gtk_widget_show(button_hbox);
  
  gtk_table_attach(GTK_TABLE(table), button_hbox, 0, 2, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  gtk_table_resize(GTK_TABLE(table), line + 1, 2);
  label_info = gtk_label_new("");
  gtk_widget_show(label_info);
  gtk_table_attach(GTK_TABLE(table), label_info, 0, 2, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  gtk_widget_show(table);
  
  scrolledwindow = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledwindow),
      GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW(scrolledwindow),
      GTK_SHADOW_IN);
  
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolledwindow),
      table);
  gtk_widget_show(scrolledwindow);
  
  etpan_preferences_panel_set_widget(panel, "table", table);
  etpan_preferences_panel_set_widget(panel, "label-info", label_info);
  etpan_preferences_panel_set_widget(panel, "create-button", create_button);
  etpan_preferences_panel_set_widget(panel, "cancel-button", cancel_button);
  etpan_preferences_panel_set_widget(panel, "revert-button", revert_button);
  etpan_preferences_panel_set_widget(panel, "apply-button", apply_button);

  create_signal_id = g_signal_connect(create_button,
      "clicked", G_CALLBACK(create_pressed_handler),
      (gpointer) panel);

  cancel_signal_id = g_signal_connect(cancel_button,
      "clicked", G_CALLBACK(cancel_pressed_handler),
      (gpointer) panel);

  revert_signal_id = g_signal_connect(revert_button,
      "clicked", G_CALLBACK(revert_pressed_handler),
      (gpointer) panel);

  apply_signal_id = g_signal_connect(apply_button,
      "clicked", G_CALLBACK(apply_pressed_handler),
      (gpointer) panel);
  
  return scrolledwindow;
}

void etpan_preferences_abook_init(struct etpan_preferences_window * preferences)
{
  GtkWidget * treeview;
  GtkWidget * scrolledwindow;
  GtkWidget * hpaned;
  GtkWidget * vbox;
  GtkWidget * button_hbox;
  GtkWidget * add_button;
  GtkWidget * remove_button;
  GtkWidget * abook_prop;
  struct etpan_preferences_panel * panel;
  GtkTreeSelection * selection;
  GtkTreeViewColumn * col_name;
  GtkCellRenderer * col_name_renderer;
  GtkTreeViewColumn * col_type;
  GtkCellRenderer * col_type_renderer;
  etpan_gtk_tree_model * treemodel;
  struct panel_data * data;
  gint selected_signal_id;
  gint add_signal_id;
  gint remove_signal_id;
  
  /*
  abook page
  - add abook
  - remove abook
  - modify abook
  */
  
  panel = etpan_preferences_panel_new();
  if (panel == NULL)
    ETPAN_LOG_MEMORY_ERROR;

  panel->open_callback = panel_open;
  panel->should_close_callback = panel_should_close;
  panel->close_callback = panel_close;
  
  data = malloc(sizeof(* data));
  if (data == NULL) {
    free(panel);
    ETPAN_LOG_MEMORY_ERROR;
  }
  data->modified = 0;
  data->treemodel = NULL;
  data->pending_abook = NULL;
  data->abook_being_modified = NULL;
  data->chosen_path = NULL;
  data->chooser_state = CHOOSER_STATE_IDLE;
  data->abook_set = 0;
  
  panel->data = data;
  
  hpaned = gtk_hpaned_new();
  gtk_widget_show(hpaned);
  gtk_paned_set_position(GTK_PANED(hpaned), 200);
  
  vbox = gtk_vbox_new(FALSE, 0);
  gtk_widget_show(vbox);
  gtk_paned_pack1(GTK_PANED(hpaned), vbox, FALSE, TRUE);
  
  scrolledwindow = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrolledwindow),
      GTK_SHADOW_IN);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledwindow),
      GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_widget_show(scrolledwindow);
  gtk_box_pack_start(GTK_BOX(vbox), scrolledwindow, TRUE, TRUE, 0);
  
  treeview = gtk_tree_view_new();
  gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(treeview), TRUE);
  gtk_tree_view_set_enable_search(GTK_TREE_VIEW(treeview), FALSE);

  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
  gtk_tree_selection_set_mode(selection, GTK_SELECTION_BROWSE);
  
  /* column name */
  col_name = gtk_tree_view_column_new();
  gtk_tree_view_column_set_title(col_name, _("Name"));
  col_name_renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_column_set_resizable(col_name, TRUE);
  gtk_tree_view_column_pack_start(col_name, col_name_renderer, TRUE);
  gtk_tree_view_column_set_attributes(col_name, col_name_renderer,
      "text", STORE_INDEX_NAME, NULL);
  gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), col_name);
  
  /* column type */
  col_type = gtk_tree_view_column_new();
  gtk_tree_view_column_set_title(col_type, _("Type"));
  col_type_renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_column_pack_start(col_type, col_type_renderer, TRUE);
  gtk_tree_view_column_set_attributes(col_type, col_type_renderer,
      "text", STORE_INDEX_TYPE, NULL);
  gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), col_type);
  
  treemodel = etpan_gtk_tree_model_new();
  
  data->pending_abook = NULL;
  
  data->abook_datasource.data = panel;
  data->abook_datasource.get_n_columns = get_n_columns;
  data->abook_datasource.get_column_type = get_column_type;
  data->abook_datasource.get_children_count = get_children_count;
  data->abook_datasource.item_has_child = item_has_child;
  data->abook_datasource.get_child_item = get_child_item;
  data->abook_datasource.get_item_value = get_item_value;
  etpan_gtk_tree_model_set_datasource(treemodel,
      &data->abook_datasource);
  
  gtk_tree_view_set_model(GTK_TREE_VIEW(treeview), GTK_TREE_MODEL(treemodel));
  
  gtk_widget_show(treeview);
  gtk_container_add(GTK_CONTAINER(scrolledwindow), treeview);

  button_hbox = gtk_hbutton_box_new();
  gtk_button_box_set_layout(GTK_BUTTON_BOX(button_hbox), GTK_BUTTONBOX_START);
  gtk_widget_show(button_hbox);
  gtk_box_pack_start(GTK_BOX(vbox), button_hbox, FALSE, FALSE, 0);
  
  add_button = gtk_button_new_with_label(_("+"));
  gtk_widget_show(add_button);
  gtk_box_pack_start(GTK_BOX(button_hbox), add_button, FALSE, FALSE, 0);
  remove_button = gtk_button_new_with_label(_("-"));
  gtk_widget_show(remove_button);
  gtk_box_pack_start(GTK_BOX(button_hbox), remove_button, FALSE, FALSE, 0);
  
  abook_prop = abook_properties(panel);
  
  gtk_paned_pack2(GTK_PANED(hpaned), abook_prop, TRUE, TRUE);
  
  panel->main_widget = hpaned;
  
#define ICON(name) \
  etpan_icon_manager_new_scaled_image(etpan_icon_manager_get_default(), \
      name, 32)
  etpan_preferences_add_page(preferences, _("Address Book"), ICON("prefs-abook"), panel);
#undef ICON
  
  data->treemodel = treemodel;
  
  etpan_preferences_panel_set_widget(panel, "abook-list", treeview);
  etpan_preferences_panel_set_widget(panel, "add-button", add_button);
  etpan_preferences_panel_set_widget(panel, "remove-button", remove_button);
  
  selected_signal_id = g_signal_connect(treeview,
      "cursor-changed", G_CALLBACK(abook_selected_handler),
      (gpointer) panel);
  
  add_signal_id = g_signal_connect(add_button,
      "clicked", G_CALLBACK(add_pressed_handler),
      (gpointer) panel);
  remove_signal_id = g_signal_connect(remove_button,
      "clicked", G_CALLBACK(remove_pressed_handler),
      (gpointer) panel);
  
  setup(panel);
}

/* *********************************** */
/* action handler */

static void abook_selected_handler(GtkTreeView * treeview,
    gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  struct panel_data * data;
  struct etpan_abook * abook;
  GtkTreeIter iter;
  GtkTreePath * path;
  GtkTreeViewColumn * column;
  int go_back;
  GtkWidget * table;
  
  panel = user_data;
  data = panel->data;
  
  gtk_tree_view_get_cursor(treeview, &path, &column);
  gtk_tree_model_get_iter(GTK_TREE_MODEL(data->treemodel), &iter, path);
  gtk_tree_model_get(GTK_TREE_MODEL(data->treemodel), &iter, 2, &abook, -1);
  gtk_tree_path_free(path);
  
  if (data->abook_set) {
    if (data->abook_being_modified != NULL) {
      if (data->abook_being_modified == abook) {
        return;
      }
    }
    if ((data->pending_abook != NULL) && (abook == NULL)) {
      return;
    }
  }

  go_back = 0;
  if (data->pending_abook) {
    if (data->modified) {
      if (!check_consistency(panel))
        go_back = 1;
    }
  }
  else if (data->abook_being_modified) {
    if (data->modified) {
      if (!check_consistency(panel))
        go_back = 1;
    }
  }
  
  if (go_back) {
    if (data->abook_being_modified != NULL)
      etpan_gtk_tree_model_get_iter_from_item(data->treemodel, &iter,
          data->abook_being_modified);
    else
      etpan_gtk_tree_model_get_iter_from_item(data->treemodel, &iter,
          data->pending_abook);
    path = gtk_tree_model_get_path(GTK_TREE_MODEL(data->treemodel), &iter);
    gtk_tree_view_set_cursor(GTK_TREE_VIEW(treeview), path, NULL, FALSE);
    gtk_tree_path_free(path);
    
    return;
  }
  
  if (data->pending_abook != NULL) {
    if (data->modified) {
      create_abook(panel, 0);
    }
  }
  else if (data->abook_being_modified != NULL) {
    if (data->modified) {
      modify_abook(panel, 0);
    }
  }
  
  if ((data->pending_abook != NULL) && (abook != NULL))
    stop_add(panel);
  
  if (abook != NULL)
    abook_set(panel, abook);
  else
    reset_properties(panel);
  
  table = etpan_preferences_panel_get_widget(panel, "table");
  gtk_widget_hide(table);
  gtk_widget_show(table);
  
  data->abook_set = 1;
}

static void abook_type_changed(GtkComboBox * widget, gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  GtkWidget * combo;
  gint index;
  GtkWidget * frame;
  (void) widget;
  
  panel = user_data;
  combo = etpan_preferences_panel_get_widget(panel, "abook-type");
  
  index = gtk_combo_box_get_active(GTK_COMBO_BOX(combo));
  
  frame = etpan_preferences_panel_get_widget(panel, "ldap-properties");
  if (index == 0)
    gtk_widget_show(frame);
  else
    gtk_widget_hide(frame);

  frame = etpan_preferences_panel_get_widget(panel, "local-properties");
  if (index == 1)
    gtk_widget_show(frame);
  else
    gtk_widget_hide(frame);
  
  reset_specific_properties(panel);
}

static void ldap_ssl_toggled(GtkToggleButton * togglebutton, gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  GtkWidget * entry;
  GtkWidget * checkbox;
  gboolean value;
  (void) togglebutton;
  
  panel = user_data;
  entry = etpan_preferences_panel_get_widget(panel, "ldap-port");
  
  checkbox = etpan_preferences_panel_get_widget(panel, "ldap-use-ssl");
  value = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkbox));
  
  if (value)
    gtk_entry_set_text(GTK_ENTRY(entry), "636");
  else
    gtk_entry_set_text(GTK_ENTRY(entry), "389");
}

static void ldap_auth_type_changed(GtkComboBox * widget, gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  GtkWidget * combo;
  gint index;
  GtkWidget * login;
  GtkWidget * password;
  (void) widget;
  
  panel = user_data;
  combo = etpan_preferences_panel_get_widget(panel, "ldap-auth-type");
  
  login = etpan_preferences_panel_get_widget(panel, "ldap-login");
  password = etpan_preferences_panel_get_widget(panel, "ldap-password");
  index = gtk_combo_box_get_active(GTK_COMBO_BOX(combo));
  
  switch (index) {
  case 0:
  case 11:
    gtk_entry_set_text(GTK_ENTRY(login), "");
    gtk_entry_set_text(GTK_ENTRY(password), "");
    gtk_widget_set_sensitive(login, FALSE);
    gtk_widget_set_sensitive(password, FALSE);
    break;
    
  case 5:
  case 6:
    gtk_entry_set_text(GTK_ENTRY(password), "");
    gtk_widget_set_sensitive(login, TRUE);
    gtk_widget_set_sensitive(password, FALSE);
    break;
    
  default:
    gtk_widget_set_sensitive(login, TRUE);
    gtk_widget_set_sensitive(password, TRUE);
    break;
  }
}  

static void create_pressed_handler(GtkButton *button, gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  (void) button;
  
  panel = user_data;
  
  create_abook(panel, 1);
}

static void cancel_pressed_handler(GtkButton *button, gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  (void) button;
  
  panel = user_data;
  
  remove_abook(panel);
}

static void apply_pressed_handler(GtkButton *button, gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  (void) button;
  
  panel = user_data;
  
  modify_abook(panel, 1);
}

static void revert_pressed_handler(GtkButton *button, gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  struct panel_data * data;
  (void) button;
  
  panel = user_data;
  data = panel->data;
  
  abook_set(panel, data->abook_being_modified);
}

static void add_pressed_handler(GtkButton *button, gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  (void) button;
  
  panel = user_data;
  
  add_abook(panel);
}

static void remove_pressed_handler(GtkButton *button, gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  (void) button;
  
  panel = user_data;
  
  remove_abook(panel);
}

/* track modifications */

static void file_chooser_set_filename(struct etpan_preferences_panel * panel,
    GtkFileChooser * chooser,
    char * path)
{
  struct panel_data * data;
  
  data = panel->data;
  
  free(data->chosen_path);
  if (path != NULL) {
    data->chosen_path = strdup(path);
    gtk_file_chooser_set_filename(GTK_FILE_CHOOSER(chooser), path);
  }
  else {
    data->chosen_path = NULL;
    gtk_file_chooser_unselect_all(GTK_FILE_CHOOSER(chooser));
  }
#if 0 /* this bug is now fixed in GTK */
    data->chooser_state = CHOOSER_STATE_SETTING;
#endif
}

static void file_chooser_changed(GtkFileChooser * chooser,
    gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  struct panel_data * data;
  char * path;
  
  panel = user_data;
  data = panel->data;
  
  /* workaround problem with file chooser that will notify later that
     it is modified */
  path = gtk_file_chooser_get_filename(chooser);
  if ((path == NULL) && (data->chosen_path == NULL)) {
    data->chooser_state = CHOOSER_STATE_IDLE;
    return;
  }
  
  if ((path != NULL) && (data->chosen_path != NULL)) {
    if (strcmp(data->chosen_path, path) == 0) {
      data->chooser_state = CHOOSER_STATE_IDLE;
      return;
    }
  }
  
  if (data->chooser_state == CHOOSER_STATE_SETTING)
    return;
  
  free(data->chosen_path);
  if (path != NULL)
    data->chosen_path = strdup(path);
  else
    data->chosen_path = NULL;
  data->modified = 1;
  
  if (path != NULL)
    g_free(path);
  
  enable_buttons(panel);
}

static void file_chooser_folder_changed(GtkFileChooser * chooser,
    gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  struct panel_data * data;
  char * path;
  
  panel = user_data;
  data = panel->data;
  
  /* workaround problem with file chooser that will notify later that
     it is modified */
  path = gtk_file_chooser_get_filename(chooser);
  if ((path == NULL) && (data->chosen_path == NULL)) {
    data->chooser_state = CHOOSER_STATE_IDLE;
    return;
  }

  if ((path != NULL) && (data->chosen_path != NULL)) {
    if (strcmp(data->chosen_path, path) == 0) {
      data->chooser_state = CHOOSER_STATE_IDLE;
      return;
    }
  }
  
  if (data->chooser_state == CHOOSER_STATE_SETTING)
    return;
  
  free(data->chosen_path);
  if (path != NULL)
    data->chosen_path = strdup(path);
  else
    data->chosen_path = NULL;
  data->modified = 1;
  
  if (path != NULL)
    g_free(path);
  
  enable_buttons(panel);
}

static void local_type_changed(GtkComboBox * widget, gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  struct panel_data * data;
  GtkWidget * combo;
  gint index;
  GtkWidget * button;
  
  panel = user_data;
  data = panel->data;
  combo = etpan_preferences_panel_get_widget(panel, "local-type");
  index = gtk_combo_box_get_active(GTK_COMBO_BOX(combo));
  
  switch (index) {
  case 0:
    button = etpan_preferences_panel_get_widget(panel, "local-path");
    gtk_widget_show(button);
    button = etpan_preferences_panel_get_widget(panel, "local-path-folder");
    gtk_widget_hide(button);
    break;
  case 1:
    button = etpan_preferences_panel_get_widget(panel, "local-path");
    gtk_widget_hide(button);
    button = etpan_preferences_panel_get_widget(panel, "local-path-folder");
    gtk_widget_show(button);
    break;
  }
}

static void entry_changed(GtkEditable *editable,
    gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  struct panel_data * data;
  (void) editable;
  
  panel = user_data;
  data = panel->data;
  data->modified = 1;
  
  enable_buttons(panel);
}

static void combo_changed(GtkComboBox * widget, gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  struct panel_data * data;
  (void) widget;
  
  panel = user_data;
  data = panel->data;
  data->modified = 1;
  
  enable_buttons(panel);
}

static void checkbox_changed(GtkToggleButton * togglebutton,
    gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  struct panel_data * data;
  (void) togglebutton;
  
  panel = user_data;
  data = panel->data;
  data->modified = 1;
  
  enable_buttons(panel);
}


/* *********************************** */
/* setup */

static void setup(struct etpan_preferences_panel * panel)
{
  carray * list;
  
  list = etpan_abook_manager_get_ordered_list(etpan_abook_manager_get_default());
  if (carray_count(list) == 0) {
    add_abook(panel);
  }
}

/* *********************************** */
/* fill entries */

static void reset_specific_properties(struct etpan_preferences_panel * panel)
{
  GtkWidget * combo_local_type;
  GtkWidget * local_type_value;
  GtkWidget * chooser_local_path;
  GtkWidget * chooser_local_path_folder;
  GtkWidget * checkbox_use_ssl;
  GtkWidget * combo_auth_type;
  GtkWidget * entry_port;
  GtkWidget * entry_server;
  GtkWidget * entry_base;
  GtkWidget * entry_login;
  GtkWidget * entry_password;
  struct panel_data * data;
  
  data = panel->data;
  
  combo_local_type = etpan_preferences_panel_get_widget(panel, "local-type");
  gtk_combo_box_set_active(GTK_COMBO_BOX(combo_local_type), 0);
  gtk_widget_show(combo_local_type);
  
  local_type_value = etpan_preferences_panel_get_widget(panel, "local-type-value");
  gtk_label_set_text(GTK_LABEL(local_type_value), _("Undefined"));
  gtk_widget_hide(local_type_value);
  
  chooser_local_path = etpan_preferences_panel_get_widget(panel, "local-path");
  file_chooser_set_filename(panel, GTK_FILE_CHOOSER(chooser_local_path), NULL);
  chooser_local_path_folder = etpan_preferences_panel_get_widget(panel, "local-path-folder");
  file_chooser_set_filename(panel, GTK_FILE_CHOOSER(chooser_local_path_folder), NULL);
  
  checkbox_use_ssl = etpan_preferences_panel_get_widget(panel, "ldap-use-ssl");
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkbox_use_ssl), FALSE);
  ldap_ssl_toggled(GTK_TOGGLE_BUTTON(checkbox_use_ssl), panel);
  
  entry_port = etpan_preferences_panel_get_widget(panel, "ldap-port");
  gtk_entry_set_text(GTK_ENTRY(entry_port), "389");
  
  combo_auth_type = etpan_preferences_panel_get_widget(panel, "ldap-auth-type");
  gtk_combo_box_set_active(GTK_COMBO_BOX(combo_auth_type), 0);
  ldap_auth_type_changed(GTK_COMBO_BOX(combo_auth_type), panel);
  
  entry_server = etpan_preferences_panel_get_widget(panel, "ldap-server");
  gtk_entry_set_text(GTK_ENTRY(entry_server), "");
  
  entry_base = etpan_preferences_panel_get_widget(panel, "ldap-base");
  gtk_entry_set_text(GTK_ENTRY(entry_base), "");

  entry_login = etpan_preferences_panel_get_widget(panel, "ldap-login");
  gtk_entry_set_text(GTK_ENTRY(entry_login), "");

  entry_password = etpan_preferences_panel_get_widget(panel, "ldap-password");
  gtk_entry_set_text(GTK_ENTRY(entry_password), "");
}

static void reset_properties(struct etpan_preferences_panel * panel)
{
  GtkWidget * entry_name;
  GtkWidget * combo_abook_type;
  GtkWidget * combo_local_type;
  GtkWidget * label_abook_type_value;
  GtkWidget * button;
  struct panel_data * data;
  
  data = panel->data;
  
  entry_name = etpan_preferences_panel_get_widget(panel, "name");
  gtk_entry_set_text(GTK_ENTRY(entry_name), "");
  
  combo_abook_type = etpan_preferences_panel_get_widget(panel, "abook-type");
  gtk_widget_show(combo_abook_type);
  
  gtk_combo_box_set_active(GTK_COMBO_BOX(combo_abook_type), 0);
  abook_type_changed(GTK_COMBO_BOX(combo_abook_type), panel);
  
  label_abook_type_value = etpan_preferences_panel_get_widget(panel, "abook-type-label");
  gtk_widget_hide(label_abook_type_value);
  
  combo_local_type = etpan_preferences_panel_get_widget(panel, "local-type");
  gtk_widget_show(combo_local_type);
  
  gtk_combo_box_set_active(GTK_COMBO_BOX(combo_local_type), 0);
  local_type_changed(GTK_COMBO_BOX(combo_local_type), panel);
  
  button = etpan_preferences_panel_get_widget(panel, "create-button");
  gtk_widget_show(button);
  
  button = etpan_preferences_panel_get_widget(panel, "cancel-button");
  gtk_widget_show(button);
  
  button = etpan_preferences_panel_get_widget(panel, "revert-button");
  gtk_widget_hide(button);
  
  button = etpan_preferences_panel_get_widget(panel, "apply-button");
  gtk_widget_hide(button);
  
  switch_text(panel, "label-info", "");
  
  data->abook_being_modified = NULL;
}

static void abook_set(struct etpan_preferences_panel * panel,
    struct etpan_abook * abook)
{
  char * name;
  GtkWidget * label;
  GtkWidget * combo;
  GtkWidget * button;
  int type;
  GtkWidget * entry;
  struct panel_data * data;
  
  data = panel->data;
  reset_properties(panel);
  
  name = etpan_abook_get_id(abook);
  
  entry = etpan_preferences_panel_get_widget(panel, "name");
  if (name != NULL)
    gtk_entry_set_text(GTK_ENTRY(entry), name);
  
  type = get_abook_type(abook);
  
  label = etpan_preferences_panel_get_widget(panel, "abook-type-label");
  gtk_label_set_text(GTK_LABEL(label), get_abook_type_str(type));
  gtk_widget_show(label);
  
  combo = etpan_preferences_panel_get_widget(panel, "abook-type");
  switch (type) {
  case ABOOK_DRIVER_TYPE_LDAP:
    gtk_combo_box_set_active(GTK_COMBO_BOX(combo), type);
    break;
  case ABOOK_DRIVER_TYPE_VCARD:
  default:
    gtk_combo_box_set_active(GTK_COMBO_BOX(combo), 1);
    break;
  }
  gtk_widget_hide(combo);
  
  button = etpan_preferences_panel_get_widget(panel, "create-button");
  gtk_widget_hide(button);
  
  button = etpan_preferences_panel_get_widget(panel, "cancel-button");
  gtk_widget_hide(button);
  
  button = etpan_preferences_panel_get_widget(panel, "revert-button");
  gtk_widget_show(button);
  
  button = etpan_preferences_panel_get_widget(panel, "apply-button");
  gtk_widget_show(button);
  
  switch (type) {
  case ABOOK_DRIVER_TYPE_LDAP:
    ldap_abook_set(panel, abook);
    break;
  case ABOOK_DRIVER_TYPE_VCARD:
    vcard_abook_set(panel, abook);
    break;
  }
  
  data->abook_being_modified = abook;
  
  data->modified = 0;
  enable_buttons(panel);
  check_consistency(panel);
}

static void vcard_abook_set(struct etpan_preferences_panel * panel,
    struct etpan_abook * abook)
{
  GtkWidget * combo_local_type;
  GtkWidget * local_type_value;
  GtkWidget * chooser_local_path;
  char * path;
  struct panel_data * data;
  int r;
  struct stat stat_buf;
  int is_folder;
  
  data = panel->data;
  
  combo_local_type = etpan_preferences_panel_get_widget(panel, "local-type");
  gtk_widget_hide(combo_local_type);
  path = etpan_abook_vcard_get_filename(abook);
  is_folder = 0;
  r = stat(path, &stat_buf);
  if (r < 0) {
    is_folder = 0;
  }
  else {
    is_folder = ((stat_buf.st_mode & S_IFDIR) != 0);
  }
  if (is_folder)
    gtk_combo_box_set_active(GTK_COMBO_BOX(combo_local_type), 1);
  else
    gtk_combo_box_set_active(GTK_COMBO_BOX(combo_local_type), 0);
  
  local_type_value = etpan_preferences_panel_get_widget(panel, "local-type-value");
  gtk_label_set_text(GTK_LABEL(local_type_value), _("V-Card"));
  gtk_widget_show(local_type_value);
  
  if (is_folder) {
    chooser_local_path = etpan_preferences_panel_get_widget(panel, "local-path-folder");
  }
  else {
    chooser_local_path = etpan_preferences_panel_get_widget(panel, "local-path");
  }
  file_chooser_set_filename(panel, GTK_FILE_CHOOSER(chooser_local_path), path);
}

static void ldap_abook_set(struct etpan_preferences_panel * panel,
    struct etpan_abook * abook)
{
  GtkWidget * entry_server;
  GtkWidget * entry_base;
  GtkWidget * checkbox_use_ssl;
  GtkWidget * entry_port;
  GtkWidget * combo_auth_type;
  GtkWidget * entry_login;
  GtkWidget * entry_password;
  char * server;
  char * base;
  char port_str[20];
  int port;
  int connection_type;
  int use_ssl_value;
  int auth_type;
  int value;
  char * login;
  char * password;
  
  entry_server = etpan_preferences_panel_get_widget(panel, "ldap-server");
  server = etpan_abook_ldap_get_hostname(abook);
  if (server != NULL)
    gtk_entry_set_text(GTK_ENTRY(entry_server), server);
  
  entry_base = etpan_preferences_panel_get_widget(panel, "ldap-base");
  base = etpan_abook_ldap_get_base(abook);
  if (base != NULL)
    gtk_entry_set_text(GTK_ENTRY(entry_base), base);
  
  checkbox_use_ssl = etpan_preferences_panel_get_widget(panel, "ldap-use-ssl");
  connection_type = etpan_abook_ldap_get_connection_type(abook);
  use_ssl_value = (connection_type == ETPAN_CONNECTION_TLS);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkbox_use_ssl), use_ssl_value);
  ldap_ssl_toggled(GTK_TOGGLE_BUTTON(checkbox_use_ssl), panel);
  
  entry_port = etpan_preferences_panel_get_widget(panel, "ldap-port");
  port = etpan_abook_ldap_get_port(abook);
  snprintf(port_str, sizeof(port_str), "%i", port);
  gtk_entry_set_text(GTK_ENTRY(entry_port), port_str);
  
  combo_auth_type = etpan_preferences_panel_get_widget(panel, "ldap-auth-type");
  auth_type = etpan_abook_ldap_get_auth_type(abook);
  switch (auth_type) {
  case ETPAN_AUTH_NONE:
    value = 0;
    break;
  case ETPAN_AUTH_PASSWORD:
    value = 1;
    break;
  case ETPAN_AUTH_PLAIN:
    value = 2;
    break;
  case ETPAN_AUTH_CRAM_MD5:
    value = 3;
    break;
  case ETPAN_AUTH_DIGEST_MD5:
    value = 4;
    break;
  case ETPAN_AUTH_GSSAPI:
    value = 5;
    break;
  case ETPAN_AUTH_KERBEROS4:
    value = 6;
    break;
  case ETPAN_AUTH_OTP:
    value = 7;
    break;
  case ETPAN_AUTH_PASSDSS_3DES_1:
    value = 8;
    break;
  case ETPAN_AUTH_SRP:
    value = 9;
    break;
  case ETPAN_AUTH_NTLM:
    value = 10;
    break;
  case ETPAN_AUTH_ANONYMOUS:
    value = 11;
    break;
    
  default:
    value = 0;
    break;
  }
  gtk_combo_box_set_active(GTK_COMBO_BOX(combo_auth_type), value);
  
  entry_login = etpan_preferences_panel_get_widget(panel, "ldap-login");
  login = etpan_abook_ldap_get_username(abook);
  if (login != NULL)
    gtk_entry_set_text(GTK_ENTRY(entry_login), login);
  
  entry_password = etpan_preferences_panel_get_widget(panel, "ldap-password");
  password = etpan_abook_ldap_get_password(abook);
  if (password != NULL)
    gtk_entry_set_text(GTK_ENTRY(entry_password), password);
}

/* *********************************** */
/* set UI state */

static void enable_buttons(struct etpan_preferences_panel * panel)
{
  GtkWidget * button;
  gboolean enabled;
  struct panel_data * data;
  
  data = panel->data;
  
  if (is_empty(panel) && (data->pending_abook != NULL)) {
    data->modified = 0;
  }
  
  enabled = data->modified;
  
  button = etpan_preferences_panel_get_widget(panel, "create-button");
  gtk_widget_set_sensitive(button, enabled);
  
  button = etpan_preferences_panel_get_widget(panel, "cancel-button");
  gtk_widget_set_sensitive(button, TRUE);
  
  button = etpan_preferences_panel_get_widget(panel, "revert-button");
  gtk_widget_set_sensitive(button, enabled);
  
  button = etpan_preferences_panel_get_widget(panel, "apply-button");
  gtk_widget_set_sensitive(button, enabled);
}

static void set_add_remove_state(struct etpan_preferences_panel * panel)
{
  GtkWidget * button;
  carray * list;
  struct panel_data * data;
  
  data = panel->data;
  
  list = etpan_abook_manager_get_ordered_list(etpan_abook_manager_get_default());
  if (carray_count(list) > 0) {
    button = etpan_preferences_panel_get_widget(panel, "add-button");
    if (data->pending_abook != NULL)
      gtk_widget_set_sensitive(button, FALSE);
    else
      gtk_widget_set_sensitive(button, TRUE);
    
    button = etpan_preferences_panel_get_widget(panel, "remove-button");
    gtk_widget_set_sensitive(button, TRUE);
  }
  else {
    button = etpan_preferences_panel_get_widget(panel, "add-button");
    gtk_widget_set_sensitive(button, FALSE);
    
    button = etpan_preferences_panel_get_widget(panel, "remove-button");
    gtk_widget_set_sensitive(button, FALSE);
  }
}

/* *********************************** */
/* add */

static void add_abook(struct etpan_preferences_panel * panel)
{
  struct etpan_abook * abook;
  struct panel_data * data;
  GtkTreePath * path;
  GtkWidget * treeview;
  GtkTreeIter iter;
  
  data = panel->data;
  
  data->abook_set = 0;
  abook = etpan_abook_new();
  data->pending_abook = abook;
  etpan_gtk_tree_model_reload(data->treemodel);
  set_add_remove_state(panel);
  
  etpan_gtk_tree_model_get_iter_from_item(data->treemodel, &iter, abook);
  
  treeview = etpan_preferences_panel_get_widget(panel, "abook-list");
  path = gtk_tree_model_get_path(GTK_TREE_MODEL(data->treemodel), &iter);
  gtk_tree_view_set_cursor(GTK_TREE_VIEW(treeview), path, NULL, FALSE);
  gtk_tree_path_free(path);
  
  data->modified = 0;
  reset_properties(panel);
  check_consistency(panel);
}

static void stop_add(struct etpan_preferences_panel * panel)
{
  struct panel_data * data;
  
  data = panel->data;
  
  etpan_abook_free(data->pending_abook);
  data->pending_abook = NULL;
  etpan_gtk_tree_model_reload(data->treemodel);
  set_add_remove_state(panel);
}

static void create_abook(struct etpan_preferences_panel * panel,
    int keep_selection)
{
  GtkWidget * combo;
  gint index;
  struct panel_data * data;
  GtkTreeIter iter;
  carray * list;
  struct etpan_abook * abook;
  GtkTreePath * path;
  GtkWidget * treeview;
  struct etpan_abook_manager * manager;
  struct etpan_error * error;
  
  if (!check_consistency(panel))
    return;
  
  data = panel->data;
  
  combo = etpan_preferences_panel_get_widget(panel, "abook-type");
  
  index = gtk_combo_box_get_active(GTK_COMBO_BOX(combo));
  switch (index) {
  case 0: /* LDAP */
    create_ldap_abook(panel);
    break;
  case 1: /* Local */
    create_local_abook(panel);
    break;
  }
  
  etpan_abook_manager_notify_modification(etpan_abook_manager_get_default());
  
  manager = etpan_abook_manager_get_default();
  
  list = etpan_abook_manager_get_ordered_list(manager);
  abook = carray_get(list, carray_count(list) - 1);
  
  if (data->pending_abook != NULL) {
    stop_add(panel);
  }
  else {
    etpan_gtk_tree_model_reload(data->treemodel);
    set_add_remove_state(panel);
  }
  
  etpan_gtk_tree_model_get_iter_from_item(data->treemodel, &iter, abook);
  
  if (keep_selection) {
    treeview = etpan_preferences_panel_get_widget(panel, "abook-list");
    path = gtk_tree_model_get_path(GTK_TREE_MODEL(data->treemodel), &iter);
    gtk_tree_view_set_cursor(GTK_TREE_VIEW(treeview), path, NULL, FALSE);
    gtk_tree_path_free(path);
  }
  
  error = etpan_abook_config_save_default();
  ETPAN_ERROR_IGNORE(error);
}

static void create_ldap_abook(struct etpan_preferences_panel * panel)
{
  struct etpan_abook_manager * manager;
  struct etpan_abook * abook;
  
  manager = etpan_abook_manager_get_default();
  
  abook = etpan_abook_ldap_new();
  
  modify_ldap_abook(panel, abook);
}

static void create_local_abook(struct etpan_preferences_panel * panel)
{
  GtkWidget * combo;
  gint index;
  
  combo = etpan_preferences_panel_get_widget(panel, "local-type");
  
  index = gtk_combo_box_get_active(GTK_COMBO_BOX(combo));
  
  switch (index) {
  case 0: /* V-Card */
    create_vcard_abook(panel);
    break;
  case 1: /* V-Card Folder */
    create_vcard_abook(panel);
    break;
  }
}

static void create_vcard_abook(struct etpan_preferences_panel * panel)
{
  struct etpan_abook_manager * manager;
  struct etpan_abook * abook;
  
  manager = etpan_abook_manager_get_default();
  
  abook = etpan_abook_vcard_new();
  
  modify_vcard_abook(panel, abook);
}


/* *********************************** */
/* modify */

static void disconnect_before_modify_callback(int cancelled, void * dummy,
    void * user_data)
{
  struct etpan_abook * abook;
  (void) cancelled;
  (void) dummy;
  
  abook = user_data;
  etpan_abook_unsetup(abook);
  etpan_abook_free(abook);
}

static void modify_abook(struct etpan_preferences_panel * panel,
    int keep_selection)
{
  GtkWidget * combo;
  gint index;
  struct panel_data * data;
  GtkTreeIter iter;
  GtkTreePath * path;
  struct etpan_abook * abook;
  GtkWidget * treeview;
  carray * list;
  unsigned int abook_index;
  unsigned int i;
  struct etpan_error * error;
  
  if (!check_consistency(panel))
    return;
  
  combo = etpan_preferences_panel_get_widget(panel, "abook-type");
  
  data = panel->data;
  
  abook = data->abook_being_modified;
  
  list = etpan_abook_manager_get_ordered_list(etpan_abook_manager_get_default());
  abook_index = 0;
  for(i = 0 ; i < carray_count(list) ; i ++) {
    if (carray_get(list, i) == abook)
      abook_index = i;
  }
  
  /* disconnect */
  etpan_abook_disconnect(etpan_thread_manager_app_get_default(),
      abook, disconnect_before_modify_callback,
      abook);
  
  /* remove from abook manager */
  etpan_abook_manager_remove_abook(etpan_abook_manager_get_default(),
      abook);
  
  index = gtk_combo_box_get_active(GTK_COMBO_BOX(combo));
  switch (index) {
  case 0: /* LDAP */
    create_ldap_abook(panel);
    break;
  case 1: /* Local */
    create_local_abook(panel);
    break;
  }
  abook = carray_get(list, carray_count(list) - 1);
  
  /* move the abook to the previous position */
  if (carray_count(list) >= 2) {
    for(i = carray_count(list) - 2 ; i >= abook_index ; i --) {
      struct etpan_abook * tmp_abook;
      
      tmp_abook = carray_get(list, i);
      carray_set(list, i + 1, tmp_abook);
      
      if (i == 0)
        break;
    }
    
    carray_set(list, abook_index, abook);
  }
  
  etpan_abook_manager_notify_modification(etpan_abook_manager_get_default());
  
  etpan_gtk_tree_model_reload(data->treemodel);
  set_add_remove_state(panel);
  data->abook_being_modified = NULL;
  
  if (keep_selection) {
    if (carray_count(list) > 0) {
      treeview = etpan_preferences_panel_get_widget(panel, "abook-list");
      etpan_gtk_tree_model_get_iter_from_item(data->treemodel, &iter, abook);
      path = gtk_tree_model_get_path(GTK_TREE_MODEL(data->treemodel), &iter);
      gtk_tree_view_set_cursor(GTK_TREE_VIEW(treeview), path, NULL, FALSE);
      gtk_tree_path_free(path);
    }
  }
  
  error = etpan_abook_config_save_default();
  ETPAN_ERROR_IGNORE(error);
}

static void modify_local_abook(struct etpan_preferences_panel * panel,
    struct etpan_abook * abook)
{
  int type;
  
  type = get_abook_type(abook);
  switch (type) {
  case ABOOK_DRIVER_TYPE_VCARD: /* V-Card */
    modify_vcard_abook(panel, abook);
    break;
  }
}

static void modify_abook_common(struct etpan_preferences_panel * panel,
    struct etpan_abook * abook)
{
  const char * name;
  GtkWidget * entry;
  
  entry = etpan_preferences_panel_get_widget(panel, "name");
  name = gtk_entry_get_text(GTK_ENTRY(entry));
  etpan_abook_set_id(abook, (char *) name);
}

static void modify_vcard_abook(struct etpan_preferences_panel * panel,
    struct etpan_abook * abook)
{
  GtkWidget * chooser_local_path;
  GtkWidget * combo;
  gchar * path;
  struct etpan_abook_manager * manager;
  struct etpan_error * error;
  gint index;
  
  manager = etpan_abook_manager_get_default();
  
  modify_abook_common(panel, abook);
  
  combo = etpan_preferences_panel_get_widget(panel, "local-type");
  index = gtk_combo_box_get_active(GTK_COMBO_BOX(combo));
  if (index == 0) {
    chooser_local_path = etpan_preferences_panel_get_widget(panel, "local-path");
  }
  else {
    chooser_local_path = etpan_preferences_panel_get_widget(panel, "local-path-folder");
  }
  
  path = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(chooser_local_path));
  etpan_abook_vcard_set_filename(abook, path);
  if (path != NULL)
    g_free(path);
  
  error = etpan_abook_setup(abook);
  ETPAN_ERROR_IGNORE(error);
  
  etpan_abook_manager_add_abook(manager, abook);
}

static void modify_ldap_abook(struct etpan_preferences_panel * panel,
    struct etpan_abook * abook)
{
  struct etpan_abook_manager * manager;
  GtkWidget * entry_server;
  GtkWidget * entry_base;
  GtkWidget * checkbox_use_ssl;
  GtkWidget * entry_port;
  GtkWidget * combo_auth_type;
  GtkWidget * entry_login;
  GtkWidget * entry_password;
  const char * server;
  const char * base;
  gboolean use_ssl;
  const char * port_str;
  int port;
  gint value_auth_type;
  int auth_type;
  const char * username;
  const char * password;
  struct etpan_error * error;
  
  manager = etpan_abook_manager_get_default();
  
  modify_abook_common(panel, abook);
  
  entry_server = etpan_preferences_panel_get_widget(panel, "ldap-server");
  server = gtk_entry_get_text(GTK_ENTRY(entry_server));
  etpan_abook_ldap_set_hostname(abook, (char *) server);
  
  entry_base = etpan_preferences_panel_get_widget(panel, "ldap-base");
  base = gtk_entry_get_text(GTK_ENTRY(entry_base));
  etpan_abook_ldap_set_base(abook, (char *) base);
  
  checkbox_use_ssl = etpan_preferences_panel_get_widget(panel, "ldap-use-ssl");
  use_ssl = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkbox_use_ssl));
  if (use_ssl)
    etpan_abook_ldap_set_connection_type(abook, ETPAN_CONNECTION_TLS);
  else
    etpan_abook_ldap_set_connection_type(abook, ETPAN_CONNECTION_PLAIN);
  
  entry_port = etpan_preferences_panel_get_widget(panel, "ldap-port");
  port_str = gtk_entry_get_text(GTK_ENTRY(entry_port));
  port = strtoul(port_str, NULL, 10);
  
  combo_auth_type = etpan_preferences_panel_get_widget(panel, "ldap-auth-type");
  value_auth_type = gtk_combo_box_get_active(GTK_COMBO_BOX(combo_auth_type));
  
  auth_type = ETPAN_AUTH_NONE;
  switch (value_auth_type) {
  case 0:
    auth_type = ETPAN_AUTH_NONE;
    break;
  case 1:
    auth_type = ETPAN_AUTH_PASSWORD;
    break;
  case 2:
    auth_type = ETPAN_AUTH_PLAIN;
    break;
  case 3:
    auth_type = ETPAN_AUTH_CRAM_MD5;
    break;
  case 4:
    auth_type = ETPAN_AUTH_DIGEST_MD5;
    break;
  case 5:
    auth_type = ETPAN_AUTH_GSSAPI;
    break;
  case 6:
    auth_type = ETPAN_AUTH_KERBEROS4;
    break;
  case 7:
    auth_type = ETPAN_AUTH_OTP;
    break;
  case 8:
    auth_type = ETPAN_AUTH_PASSDSS_3DES_1;
    break;
  case 9:
    auth_type = ETPAN_AUTH_SRP;
    break;
  case 10:
    auth_type = ETPAN_AUTH_NTLM;
    break;
  case 11:
    auth_type = ETPAN_AUTH_ANONYMOUS;
    break;
  }
  etpan_abook_ldap_set_auth_type(abook, auth_type);
  
  entry_login = etpan_preferences_panel_get_widget(panel, "ldap-login");
  username = gtk_entry_get_text(GTK_ENTRY(entry_login));
  etpan_abook_ldap_set_username(abook, (char *) username);
  if (auth_type == ETPAN_AUTH_ANONYMOUS)
    etpan_abook_ldap_set_username(abook, "etpan@etpan.org");
  
  entry_password = etpan_preferences_panel_get_widget(panel, "ldap-password");
  password = gtk_entry_get_text(GTK_ENTRY(entry_password));
  etpan_abook_ldap_set_username(abook, (char *) password);
  
  error = etpan_abook_setup(abook);
  ETPAN_ERROR_IGNORE(error);
  
  etpan_abook_manager_add_abook(manager, abook);
}


/* *********************************** */
/* remove */

static void disconnect_before_remove_callback(int cancelled, void * dummy,
    void * user_data)
{
  struct etpan_abook * abook;
  (void) cancelled;
  (void) dummy;
  
  abook = user_data;
  etpan_abook_unsetup(abook);
  etpan_abook_free(abook);
}

static void remove_abook(struct etpan_preferences_panel * panel)
{
  struct panel_data * data;
  struct etpan_abook * abook;
  struct etpan_abook * last_abook;
  GtkTreeIter iter;
  GtkTreePath * path;
  GtkWidget * treeview;
  int is_last_abook;
  carray * list;
  unsigned int current_index;
  unsigned int i;
  struct etpan_error * error;
  
  data = panel->data;
  
  treeview = etpan_preferences_panel_get_widget(panel, "abook-list");
  abook = data->abook_being_modified;
  
  if (data->pending_abook != NULL) {
    
    list = etpan_abook_manager_get_ordered_list(etpan_abook_manager_get_default());
    if (carray_count(list) == 0) {
      reset_properties(panel);
      check_consistency(panel);
      return;
    }
    
    stop_add(panel);
    abook = carray_get(list, carray_count(list) - 1);
    etpan_gtk_tree_model_get_iter_from_item(data->treemodel, &iter, abook);
    
    path = gtk_tree_model_get_path(GTK_TREE_MODEL(data->treemodel), &iter);
    gtk_tree_view_set_cursor(GTK_TREE_VIEW(treeview), path, NULL, FALSE);
    gtk_tree_path_free(path);
    
    return;
  }
  
  data->abook_being_modified = NULL;
  
  is_last_abook = 0;
  list = etpan_abook_manager_get_ordered_list(etpan_abook_manager_get_default());
  if (carray_count(list) > 0) {
    last_abook = carray_get(list, carray_count(list) - 1);
    if (abook == last_abook)
      is_last_abook = 1;
  }
  current_index = 0;
  for(i = 0 ; i < carray_count(list) ; i ++) {
    if (carray_get(list, i) == abook)
      current_index = i;
  }
  
  etpan_abook_manager_remove_abook(etpan_abook_manager_get_default(),
      abook);
  etpan_abook_manager_notify_modification(etpan_abook_manager_get_default());
  
  etpan_gtk_tree_model_reload(data->treemodel);
  set_add_remove_state(panel);
  
  if (carray_count(list) > 0) {
    if (is_last_abook) {
      /* select last abook if needed */
      last_abook = carray_get(list, carray_count(list) - 1);
      etpan_gtk_tree_model_get_iter_from_item(data->treemodel, &iter, last_abook);
      path = gtk_tree_model_get_path(GTK_TREE_MODEL(data->treemodel), &iter);
      gtk_tree_view_set_cursor(GTK_TREE_VIEW(treeview), path, NULL, FALSE);
      gtk_tree_path_free(path);
    }
    else {
      struct etpan_abook * abook_to_select;
    
      abook_to_select = carray_get(list, current_index);
      etpan_gtk_tree_model_get_iter_from_item(data->treemodel, &iter, abook_to_select);
      path = gtk_tree_model_get_path(GTK_TREE_MODEL(data->treemodel), &iter);
      gtk_tree_view_set_cursor(GTK_TREE_VIEW(treeview), path, NULL, FALSE);
      gtk_tree_path_free(path);
    }
  }
  else {
    add_abook(panel);
  }
  
  etpan_abook_disconnect(etpan_thread_manager_app_get_default(),
      abook, disconnect_before_remove_callback,
      abook);
  
  error = etpan_abook_config_save_default();
  ETPAN_ERROR_IGNORE(error);
}

/* *********************************** */
/* check consistency */

static void switch_text(struct etpan_preferences_panel * panel,
    char * name, char * value)
{
  GtkWidget * label;
  
  label = etpan_preferences_panel_get_widget(panel, name);
  gtk_label_set_markup(GTK_LABEL(label), value);
}

static int check_consistency(struct etpan_preferences_panel * panel)
{
  GtkWidget * chooser;
  GtkWidget * entry;
  const char * value;
  gboolean enabled;
  struct panel_data * data;
  GtkWidget * combo;
  gint index;
  
  data = panel->data;
  enabled = data->modified;
  
  entry = etpan_preferences_panel_get_widget(panel, "name");
  value = gtk_entry_get_text(GTK_ENTRY(entry));
  if (data->modified && (value[0] == '\0')) {
    switch_text(panel, "label-name", _("<span foreground=\"#ff0000\">Name</span>"));
    enabled = 0;
  }
  else {
    switch_text(panel, "label-name", _("<span foreground=\"#000000\">Name</span>"));
  }
  
  combo = etpan_preferences_panel_get_widget(panel, "abook-type");
  index = gtk_combo_box_get_active(GTK_COMBO_BOX(combo));
  switch (index) {
  case 0: /* LDAP */
    {
      entry = etpan_preferences_panel_get_widget(panel, "ldap-server");
      value = gtk_entry_get_text(GTK_ENTRY(entry));
      if (data->modified && (value[0] == '\0')) {
        switch_text(panel, "label-ldap-server", _("<span foreground=\"#ff0000\">Directory Server</span>"));
        enabled = 0;
      }
      else {
        switch_text(panel, "label-ldap-server", _("<span foreground=\"#000000\">Directory Server</span>"));
      }
      
      entry = etpan_preferences_panel_get_widget(panel, "ldap-base");
      value = gtk_entry_get_text(GTK_ENTRY(entry));
      if (data->modified && (value[0] == '\0')) {
        switch_text(panel, "label-ldap-base", _("<span foreground=\"#ff0000\">Search Base</span>"));
        enabled = 0;
      }
      else {
        switch_text(panel, "label-ldap-base", _("<span foreground=\"#000000\">Search Base</span>"));
      }
    }
    break;
  case 1: /* V-Card */
    {
      chooser = etpan_preferences_panel_get_widget(panel, "local-path");
      value = data->chosen_path;
      if (data->modified && (value == NULL)) {
        switch_text(panel, "label-local-path", _("<span foreground=\"#ff0000\">Path</span>"));
        enabled = 0;
      }
      else {
        switch_text(panel, "label-local-path", _("<span foreground=\"#000000\">Path</span>"));
      }
    }
    break;
  }
  
  if ((!enabled) && data->modified) {
    switch_text(panel, "label-info",
        _("<span foreground=\"#ff0000\">Fields that appear in red needs to be filled in</span>"));
  }
  else {
    switch_text(panel, "label-info", "");
  }
  
  return enabled;
}

static int is_empty(struct etpan_preferences_panel * panel)
{
  GtkWidget * chooser;
  GtkWidget * entry;
  const char * value;
  struct panel_data * data;
  GtkWidget * combo;
  gint index;
  int empty;
  
  data = panel->data;
  empty = 1;
  
  entry = etpan_preferences_panel_get_widget(panel, "name");
  value = gtk_entry_get_text(GTK_ENTRY(entry));
  if (value[0] != '\0') {
    empty = 0;
  }
  
  combo = etpan_preferences_panel_get_widget(panel, "abook-type");
  index = gtk_combo_box_get_active(GTK_COMBO_BOX(combo));
  switch (index) {
  case 0: /* LDAP */
    {
      entry = etpan_preferences_panel_get_widget(panel, "ldap-server");
      value = gtk_entry_get_text(GTK_ENTRY(entry));
      if (value[0] != '\0') {
        empty = 0;
      }
      
      entry = etpan_preferences_panel_get_widget(panel, "ldap-base");
      value = gtk_entry_get_text(GTK_ENTRY(entry));
      if (value[0] != '\0') {
        empty = 0;
      }
    }
    break;
  case 1: /* V-Card */
    {
      chooser = etpan_preferences_panel_get_widget(panel, "local-path");
      value = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(chooser));
      if (value != NULL) {
        empty = 0;
        g_free((gpointer) value);
      }
    }
    break;
  }
  
  return empty;
}

/* *********************************** */
/* preferences panel handler */

static int panel_should_close(struct etpan_preferences_panel * panel)
{
  struct panel_data * data;
  
  data = panel->data;
  
  if (data->pending_abook) {
    if (data->modified) {
      if (!check_consistency(panel))
        return 0;
    }
  }
  else if (data->abook_being_modified) {
    if (data->modified) {
      if (!check_consistency(panel))
        return 0;
    }
  }
  
  return 1;
}

static void panel_close(struct etpan_preferences_panel * panel)
{
  struct panel_data * data;
  
  data = panel->data;
  
  if (data->pending_abook) {
    if (data->modified) {
      create_abook(panel, 0);
    }
    else {
      remove_abook(panel);
    }
  }
  else if (data->abook_being_modified) {
    if (data->modified) {
      modify_abook(panel, 0);
    }
  }
}

static void panel_open(struct etpan_preferences_panel * panel)
{
  GtkWidget * treeview;
  GtkTreePath * path;
  struct panel_data * data;
  
  data = panel->data;
  data->abook_set = 0;
  data->modified = 0;
  treeview = etpan_preferences_panel_get_widget(panel, "abook-list");
  
  etpan_gtk_tree_model_reload(data->treemodel);
  
  path = gtk_tree_path_new_first();
  gtk_tree_view_set_cursor(GTK_TREE_VIEW(treeview), path, NULL, FALSE);
  gtk_tree_path_free(path);
}


/* *********************************** */
/* abook type */

static int get_abook_type(struct etpan_abook * abook)
{
  struct etpan_abook_driver * driver;
  int type;
  
  driver = etpan_abook_get_driver(abook);
  type = ABOOK_DRIVER_TYPE_VCARD;
  if (strcasecmp(driver->name, "vcard") == 0)
    type = ABOOK_DRIVER_TYPE_VCARD;
  else if (strcasecmp(driver->name, "ldap") == 0)
    type = ABOOK_DRIVER_TYPE_LDAP;
  
  return type;
}

static char * get_abook_type_str(int value)
{
  char * abook_type_str;
  
  abook_type_str = NULL;
  switch (value) {
  case -1:
    abook_type_str = "";
    break;    
  case ABOOK_DRIVER_TYPE_VCARD:
    abook_type_str = _("Local");
    break;
  case ABOOK_DRIVER_TYPE_LDAP:
    abook_type_str = _("LDAP");
    break;
  }
  
  return abook_type_str;
}

