#include "etpan-preferences-account.h"

#include <libgen.h>

#include "etpan-backend.h"

#include "etpan-preferences-window.h"
#include "etpan-folder-list.h"
#include "etpan-gtk-tree-model.h"
#include "etpan-ui-config.h"
#include "etpan-icon-manager.h"

struct panel_data {
  int modified;
  struct etpan_account * account_being_modified;
  struct etpan_gtk_tree_data_source account_datasource;
  etpan_gtk_tree_model * treemodel;
  struct etpan_account * pending_account;
  int chooser_state;
  int account_set;
  char * local_path;
  char * pop_path;
};

#define PADDING 5

enum {
  STORAGE_DRIVER_TYPE_POP,
  STORAGE_DRIVER_TYPE_IMAP,
  STORAGE_DRIVER_TYPE_NEWS,
  STORAGE_DRIVER_TYPE_MAILDIR,
  STORAGE_DRIVER_TYPE_MH,
  STORAGE_DRIVER_TYPE_MBOX,
};

enum {
  CHOOSER_STATE_IDLE,
  CHOOSER_STATE_SETTING,
};

static void set_add_remove_state(struct etpan_preferences_panel * panel);

static void panel_open(struct etpan_preferences_panel * panel);
static int panel_should_close(struct etpan_preferences_panel * panel);
static void panel_close(struct etpan_preferences_panel * panel);

static void add_account(struct etpan_preferences_panel * panel);

static int get_account_type(struct etpan_account * account);
static char * get_account_type_str(int value);
static int is_empty(struct etpan_preferences_panel * panel);

static void switch_text(struct etpan_preferences_panel * panel,
    char * name, char * value);

static void imap_ssl_toggled(GtkToggleButton * togglebutton, gpointer user_data);
static void imap_auth_type_changed(GtkComboBox * widget, gpointer user_data);
static void pop_ssl_toggled(GtkToggleButton * togglebutton, gpointer user_data);
static void pop_auth_type_changed(GtkComboBox * widget, gpointer user_data);
static void pop_path_clicked(GtkButton * button, gpointer user_data);
static void local_path_clicked(GtkButton * button, gpointer user_data);
static void news_ssl_toggled(GtkToggleButton * togglebutton, gpointer user_data);
static void news_auth_type_changed(GtkComboBox * widget, gpointer user_data);
static void smtp_ssl_toggled(GtkToggleButton * togglebutton, gpointer user_data);
static void smtp_auth_type_changed(GtkComboBox * widget, gpointer user_data);
static void local_type_changed(GtkComboBox * widget, gpointer user_data);
static void account_id_changed(GtkEditable *editable,
    gpointer user_data);
static void pop_path_update(struct etpan_preferences_panel * panel,
    char * filename);
static void local_path_update(struct etpan_preferences_panel * panel,
    char * filename);

static void create_account(struct etpan_preferences_panel * panel,
    int keep_selection);

static void modify_account(struct etpan_preferences_panel * panel,
    int keep_selection);

static void account_set(struct etpan_preferences_panel * panel,
    struct etpan_account * account);

static void stop_add(struct etpan_preferences_panel * panel);

static int check_consistency(struct etpan_preferences_panel * panel);

static void add_pressed_handler(GtkButton *button, gpointer user_data);
static void remove_pressed_handler(GtkButton *button, gpointer user_data);
static void account_type_changed(GtkComboBox * widget, gpointer user_data);

static void entry_changed(GtkEditable *editable,
    gpointer user_data);
static void combo_changed(GtkComboBox * widget, gpointer user_data);
static void checkbox_changed(GtkToggleButton * togglebutton,
    gpointer user_data);

static void reset_properties(struct etpan_preferences_panel * panel);

static void setup(struct etpan_preferences_panel * panel);

static void modify_imap_account(struct etpan_preferences_panel * panel,
    struct etpan_account * account);
static void modify_pop_account(struct etpan_preferences_panel * panel,
    struct etpan_account * account);
static void modify_news_account(struct etpan_preferences_panel * panel,
    struct etpan_account * account);
static void modify_mbox_account(struct etpan_preferences_panel * panel,
    struct etpan_account * account);
static void modify_maildir_account(struct etpan_preferences_panel * panel,
    struct etpan_account * account);
static void modify_mh_account(struct etpan_preferences_panel * panel,
    struct etpan_account * account);

static void create_imap_account(struct etpan_preferences_panel * panel);
static void create_pop_account(struct etpan_preferences_panel * panel);
static void create_news_account(struct etpan_preferences_panel * panel);
static void create_mbox_account(struct etpan_preferences_panel * panel);
static void create_maildir_account(struct etpan_preferences_panel * panel);
static void create_mh_account(struct etpan_preferences_panel * panel);
static void create_local_account(struct etpan_preferences_panel * panel);

static void create_pressed_handler(GtkButton *button, gpointer user_data);
static void cancel_pressed_handler(GtkButton *button, gpointer user_data);
static void apply_pressed_handler(GtkButton *button, gpointer user_data);
static void revert_pressed_handler(GtkButton *button, gpointer user_data);
static void account_selected_handler(GtkTreeView * treeview,
    gpointer user_data);
static void reset_specific_properties(struct etpan_preferences_panel * panel);

static void remove_account(struct etpan_preferences_panel * panel);
static void enable_buttons(struct etpan_preferences_panel * panel);

static void switch_text(struct etpan_preferences_panel * panel,
    char * name, char * value);

static void free_data(struct etpan_preferences_panel * panel);

/* *********************************** */
/* data source */

enum {
  STORE_INDEX_NAME,
  STORE_INDEX_TYPE,
  STORE_INDEX_ACCOUNT,
};

static unsigned int get_n_columns(struct etpan_gtk_tree_data_source *
    datasource)
{
  (void) datasource;
  
  return 3;
}

static GType get_column_type(struct etpan_gtk_tree_data_source * datasource,
    unsigned int column_index)
{
  static GType column_types[3] = {
    G_TYPE_STRING, G_TYPE_STRING,
    G_TYPE_POINTER
  };
  (void) datasource;
  
  return column_types[column_index];
}
  
static unsigned int get_children_count(struct etpan_gtk_tree_data_source *
    datasource, void * item)
{
  carray * list;
  struct etpan_preferences_panel * panel;
  struct panel_data * data;
  
  panel = datasource->data;
  data = panel->data;
  list = etpan_account_manager_get_ordered_list(etpan_account_manager_get_default());
  
  if (item == NULL) {
    if (data->pending_account != NULL)
      return carray_count(list) + 1;
    else
      return carray_count(list);
  }
  else
    return 0;
}

static int item_has_child(struct etpan_gtk_tree_data_source *
    datasource, void * item)
{
  carray * list;
  (void) datasource;
  
  list = etpan_account_manager_get_ordered_list(etpan_account_manager_get_default());
  
  if (item == NULL)
    return carray_count(list) > 0;
  else
    return 0;
}

static void * get_child_item(struct etpan_gtk_tree_data_source *
    datasource, void * item, unsigned int index)
{
  carray * list;
  struct etpan_preferences_panel * panel;
  struct panel_data * data;
  
  panel = datasource->data;
  data = panel->data;
  
  list = etpan_account_manager_get_ordered_list(etpan_account_manager_get_default());
  
  if (item == NULL) {
    if (index >= carray_count(list))
      return data->pending_account;
    else
      return carray_get(list, index);
  }
  else
    return NULL;
}

static void get_item_value(struct etpan_gtk_tree_data_source *
    datasource, void * item, unsigned int column_index,
    GValue * value)
{
  struct panel_data * data;
  struct etpan_preferences_panel * panel;
  
  panel = datasource->data;
  data = panel->data;
  
  if (item == data->pending_account) {
    switch (column_index) {
    case STORE_INDEX_NAME:
      {
        char * str;
      
        str = _("(New Account)");
      
        g_value_init(value, G_TYPE_STRING);
        g_value_set_string(value, str);
      }
      break;
    case STORE_INDEX_TYPE:
      {
        char * str;
      
        str = "";
      
        g_value_init(value, G_TYPE_STRING);
        g_value_set_string(value, str);
      
      }
      break;
    case STORE_INDEX_ACCOUNT:
      {
        g_value_init(value, G_TYPE_POINTER);
        g_value_set_pointer(value, NULL);
      }
      break;
    }
  }
  else {
    struct etpan_account * account;
    
    account = item;
    
    switch (column_index) {
    case STORE_INDEX_NAME:
      {
        char * str;
        
        str = etpan_account_get_id(account);
        
        g_value_init(value, G_TYPE_STRING);
        g_value_set_string(value, str);
      }
      break;
    case STORE_INDEX_TYPE:
      {
        char * str;
        int type;
        
        type = get_account_type(account);
        str = get_account_type_str(type);
        
        g_value_init(value, G_TYPE_STRING);
        g_value_set_string(value, str);
      
      }
      break;
    case STORE_INDEX_ACCOUNT:
      {
        g_value_init(value, G_TYPE_POINTER);
        g_value_set_pointer(value, account);
      }
      break;
    }
  }
}

/* *********************************** */
/* build UI */

static void basic_account_properties(struct etpan_preferences_panel * panel,
    GtkWidget * table)
{
  GtkWidget * label_name;
  GtkWidget * entry_name;
  GtkWidget * label_display_name;
  GtkWidget * entry_display_name;
  GtkWidget * label_mail;
  GtkWidget * entry_mail;
  GtkWidget * label_account_type;
  GtkWidget * combo_account_type;
  GtkWidget * label_account_type_value;
  guint gline;
  int line;
  gint signal_id;
  gint combo_account_type_signal_id;
  gint entry_name_signal_id;
  
  g_object_get(table, "n-rows", &gline, NULL);
  line = gline;
  
  /* account name */
  gtk_table_resize(GTK_TABLE(table), line + 1, 2);
  label_name = gtk_label_new(_("Account Name"));
  gtk_widget_show(label_name);
  gtk_table_attach(GTK_TABLE(table), label_name, 0, 1, line, line + 1,
      0, GTK_SHRINK, /* attach options */
      PADDING, PADDING);
  entry_name = gtk_entry_new();
  gtk_widget_show(entry_name);
  gtk_table_attach(GTK_TABLE(table), entry_name, 1, 2, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  /* name */
  gtk_table_resize(GTK_TABLE(table), line + 1, 2);
  label_display_name = gtk_label_new(_("Your Name"));
  gtk_widget_show(label_display_name);
  gtk_table_attach(GTK_TABLE(table), label_display_name, 0, 1, line, line + 1,
      0, GTK_SHRINK, /* attach options */
      PADDING, PADDING);
  entry_display_name = gtk_entry_new();
  gtk_widget_show(entry_display_name);
  gtk_table_attach(GTK_TABLE(table), entry_display_name, 1, 2, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  /* e-mail address */
  gtk_table_resize(GTK_TABLE(table), line + 1, 2);
  label_mail = gtk_label_new(_("E-mail Address"));
  gtk_widget_show(label_mail);
  gtk_table_attach(GTK_TABLE(table), label_mail, 0, 1, line, line + 1,
      0, GTK_SHRINK, /* attach options */
      PADDING, PADDING);
  entry_mail = gtk_entry_new();
  gtk_widget_show(entry_mail);
  gtk_table_attach(GTK_TABLE(table), entry_mail, 1, 2, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  /* account type */
  gtk_table_resize(GTK_TABLE(table), line + 2, 2);
  label_account_type = gtk_label_new(_("Account Type"));
  gtk_widget_show(label_account_type);
  gtk_table_attach(GTK_TABLE(table), label_account_type, 0, 1, line, line + 2,
      0, GTK_SHRINK, /* attach options */
      PADDING, PADDING);
  
  label_account_type_value = gtk_label_new(_("Undefined"));
  gtk_misc_set_alignment(GTK_MISC(label_account_type_value), 0, 0.5);
  gtk_widget_show(label_account_type_value);
  gtk_table_attach(GTK_TABLE(table), label_account_type_value, 1, 2, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  
  combo_account_type = gtk_combo_box_new_text();
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_account_type),
      get_account_type_str(STORAGE_DRIVER_TYPE_POP));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_account_type),
      get_account_type_str(STORAGE_DRIVER_TYPE_IMAP));
#if 0
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_account_type),
      get_account_type_str(STORAGE_DRIVER_TYPE_NEWS));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_account_type),
      get_account_type_str(STORAGE_DRIVER_TYPE_MAILDIR));
#endif
  gtk_combo_box_set_active(GTK_COMBO_BOX(combo_account_type), 0);
  
  gtk_widget_show(combo_account_type);
  gtk_table_attach(GTK_TABLE(table), combo_account_type, 1, 2, line + 1, line + 2,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line += 2;
  
  etpan_preferences_panel_set_widget(panel, "label-account-name", label_name);
  etpan_preferences_panel_set_widget(panel, "account-name", entry_name);
  etpan_preferences_panel_set_widget(panel, "display-name", entry_display_name);
  etpan_preferences_panel_set_widget(panel, "label-address", label_mail);
  etpan_preferences_panel_set_widget(panel, "address", entry_mail);
  etpan_preferences_panel_set_widget(panel, "account-type", combo_account_type);
  etpan_preferences_panel_set_widget(panel, "account-type-label", label_account_type_value);
  
  combo_account_type_signal_id = g_signal_connect(combo_account_type,
      "changed", G_CALLBACK(account_type_changed),
      (gpointer) panel);
  entry_name_signal_id = g_signal_connect(entry_name, "changed",
      G_CALLBACK(account_id_changed), panel);
  
  signal_id = g_signal_connect(entry_name, "changed",
      G_CALLBACK(entry_changed), panel);
  signal_id = g_signal_connect(entry_display_name, "changed",
      G_CALLBACK(entry_changed), panel);
  signal_id = g_signal_connect(entry_mail, "changed",
      G_CALLBACK(entry_changed), panel);
  signal_id = g_signal_connect(combo_account_type, "changed",
      G_CALLBACK(combo_changed), panel);
}

/* IMAP */

static GtkWidget * imap_optional_properties(struct etpan_preferences_panel * panel)
{
  GtkWidget * table;
  GtkWidget * label_port;
  GtkWidget * entry_port;
  GtkWidget * label_auth_type;
  GtkWidget * combo_auth_type;
  GtkWidget * label_prefix;
  GtkWidget * entry_prefix;
  GtkWidget * checkbox_use_ssl;
  int line;
  gint signal_id;
  gint ssl_toggled_signal_id;
  gint combo_auth_type_signal_id;
  
  table = gtk_table_new(1, 3, FALSE);
  line = 0;
  
  /* port */
  gtk_table_resize(GTK_TABLE(table), line + 1, 3);
  checkbox_use_ssl = gtk_check_button_new_with_label(_("Use SSL"));
  gtk_widget_show(checkbox_use_ssl);
  gtk_table_attach(GTK_TABLE(table), checkbox_use_ssl, 0, 1, line, line + 1,
      0, GTK_SHRINK, /* attach options */
      PADDING, PADDING);
  
  gtk_table_resize(GTK_TABLE(table), line + 1, 3);
  label_port = gtk_label_new(_("Port"));
  gtk_widget_show(label_port);
  gtk_table_attach(GTK_TABLE(table), label_port, 1, 2, line, line + 1,
      0, GTK_SHRINK, /* attach options */
      PADDING, PADDING);
  entry_port = gtk_entry_new();
  gtk_widget_show(entry_port);
  gtk_table_attach(GTK_TABLE(table), entry_port, 2, 3, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  /* auth type */
  /*
    CRAM-MD5: auth_name, password
    anonymous: login
    Digest-MD5: auth_name, login, password
    GSSAPI (Kerberos V): login
    Kerberos4: login
    login: auth_name, login, password
    NTLM: auth_name, password
    OTP: auth_name, login, password
    PASSDSS-3DES-1: auth_name, login, password
    srp: auth_name, login, password
  */
  gtk_table_resize(GTK_TABLE(table), line + 1, 3);
  label_auth_type = gtk_label_new(_("Authentication Type"));
  gtk_widget_show(label_auth_type);
  gtk_table_attach(GTK_TABLE(table), label_auth_type, 0, 1, line, line + 1,
      0, GTK_SHRINK, /* attach options */
      PADDING, PADDING);
  combo_auth_type = gtk_combo_box_new_text();
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_auth_type), _("Password"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_auth_type), _("Plain login"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_auth_type), _("Challenge-Response MD5 (CRAM-MD5)"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_auth_type), _("Digest MD5"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_auth_type), _("Kerberos V (GSSAPI)"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_auth_type), _("Kerberos 4"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_auth_type), _("One-Time-Password"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_auth_type), _("DSS Secured Password (PASSDSS-3DES-1)"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_auth_type), _("Secure Remote Password (SRP)"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_auth_type), _("NTLM"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_auth_type), _("Anonymous"));
  gtk_combo_box_set_active(GTK_COMBO_BOX(combo_auth_type), 0);
  gtk_widget_show(combo_auth_type);
  gtk_table_attach(GTK_TABLE(table), combo_auth_type, 1, 3, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  label_prefix = gtk_label_new(_("Mailbox prefix"));
  gtk_widget_show(label_prefix);
  gtk_table_attach(GTK_TABLE(table), label_prefix, 0, 1, line, line + 1,
      0, GTK_SHRINK, /* attach options */
      PADDING, PADDING);
  
  entry_prefix = gtk_entry_new();
  gtk_widget_show(entry_prefix);
  gtk_table_attach(GTK_TABLE(table), entry_prefix, 1, 3, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  gtk_widget_show(table);
  
  ssl_toggled_signal_id = g_signal_connect(checkbox_use_ssl,
      "toggled", G_CALLBACK(imap_ssl_toggled),
      (gpointer) panel);
  
  combo_auth_type_signal_id = g_signal_connect(combo_auth_type,
      "changed", G_CALLBACK(imap_auth_type_changed),
      (gpointer) panel);
  
  etpan_preferences_panel_set_widget(panel, "imap-use-ssl", checkbox_use_ssl);
  etpan_preferences_panel_set_widget(panel, "imap-port", entry_port);
  etpan_preferences_panel_set_widget(panel, "imap-auth-type", combo_auth_type);
  etpan_preferences_panel_set_widget(panel, "imap-prefix", entry_prefix);

  signal_id = g_signal_connect(checkbox_use_ssl, "toggled",
      G_CALLBACK(checkbox_changed), panel);
  signal_id = g_signal_connect(entry_port, "changed",
      G_CALLBACK(entry_changed), panel);
  signal_id = g_signal_connect(combo_auth_type, "changed",
      G_CALLBACK(combo_changed), panel);
  signal_id = g_signal_connect(entry_prefix, "changed",
      G_CALLBACK(entry_changed), panel);
  
  return table;
}

static GtkWidget * imap_properties(struct etpan_preferences_panel * panel)
{
  GtkWidget * label_server;
  GtkWidget * entry_server;
  GtkWidget * label_login;
  GtkWidget * entry_login;
  GtkWidget * label_password;
  GtkWidget * entry_password;
  GtkWidget * server_expander;
  GtkWidget * imap_properties;
  int line;
  GtkWidget * table;
  gint signal_id;
  
  table = gtk_table_new(1, 2, FALSE);
  line = 0;
  
  /* (for IMAP, POP) */
  /* receive server, settings ... */
  gtk_table_resize(GTK_TABLE(table), line + 1, 2);
  label_server = gtk_label_new(_("Receive Server"));
  gtk_widget_show(label_server);
  gtk_table_attach(GTK_TABLE(table), label_server, 0, 1, line, line + 1,
      0, GTK_SHRINK, /* attach options */
      PADDING, PADDING);
  entry_server = gtk_entry_new();
  gtk_widget_show(entry_server);
  gtk_table_attach(GTK_TABLE(table), entry_server, 1, 2, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  /* login */
  gtk_table_resize(GTK_TABLE(table), line + 1, 2);
  label_login = gtk_label_new(_("Login"));
  gtk_widget_show(label_login);
  gtk_table_attach(GTK_TABLE(table), label_login, 0, 1, line, line + 1,
      0, GTK_SHRINK, /* attach options */
      PADDING, PADDING);
  entry_login = gtk_entry_new();
  gtk_widget_show(entry_login);
  gtk_table_attach(GTK_TABLE(table), entry_login, 1, 2, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  /* password */
  gtk_table_resize(GTK_TABLE(table), line + 1, 2);
  label_password = gtk_label_new(_("Password"));
  gtk_widget_show(label_password);
  gtk_table_attach(GTK_TABLE(table), label_password, 0, 1, line, line + 1,
      0, GTK_SHRINK, /* attach options */
      PADDING, PADDING);
  entry_password = gtk_entry_new();
  gtk_entry_set_visibility(GTK_ENTRY(entry_password), FALSE);
  gtk_widget_show(entry_password);
  gtk_table_attach(GTK_TABLE(table), entry_password, 1, 2, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  /* more settings for receive server */
  gtk_table_resize(GTK_TABLE(table), line + 1, 2);
  server_expander = gtk_expander_new(_("Settings"));
  imap_properties = imap_optional_properties(panel);
  
  gtk_container_add(GTK_CONTAINER(server_expander), imap_properties);
  gtk_widget_show(server_expander);
  gtk_table_attach(GTK_TABLE(table), server_expander, 0, 2, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  gtk_widget_show(table);
  
  etpan_preferences_panel_set_widget(panel, "label-imap-server", label_server);
  etpan_preferences_panel_set_widget(panel, "imap-server", entry_server);
  etpan_preferences_panel_set_widget(panel, "imap-login", entry_login);
  etpan_preferences_panel_set_widget(panel, "imap-password", entry_password);
  etpan_preferences_panel_set_widget(panel, "imap-expander", server_expander);

  signal_id = g_signal_connect(entry_server, "changed",
      G_CALLBACK(entry_changed), panel);
  signal_id = g_signal_connect(entry_login, "changed",
      G_CALLBACK(entry_changed), panel);
  signal_id = g_signal_connect(entry_password, "changed",
      G_CALLBACK(entry_changed), panel);
  
  return table;
}

static GtkWidget * imap_frame(struct etpan_preferences_panel * panel)
{
#if 0
  GtkWidget * frame;
  GtkWidget * alignment;
  GtkWidget * properties;
  
  frame = gtk_frame_new(_("Receive server"));
  
  properties = imap_properties(panel);
  
  alignment = gtk_alignment_new(0.5, 0.5, 1.0, 1.0);
  gtk_container_add(GTK_CONTAINER(alignment), properties);
  gtk_alignment_set_padding(GTK_ALIGNMENT(alignment), 5, 5, 10, 10);
  gtk_widget_show(alignment);
  
  gtk_container_add(GTK_CONTAINER(frame), alignment);
  
  gtk_widget_show(frame);
  
  return frame;
#else
  return imap_properties(panel);
#endif
}

/* POP */

static GtkWidget * pop_optional_properties(struct etpan_preferences_panel * panel)
{
  GtkWidget * table;
  GtkWidget * label_port;
  GtkWidget * entry_port;
  GtkWidget * label_auth_type;
  GtkWidget * combo_auth_type;
  GtkWidget * checkbox_use_ssl;
  GtkWidget * label_path;
  GtkWidget * entry_local_path;
  GtkWidget * button_local_path;
  GtkWidget * hbox;
  int line;
  gint signal_id;
  gint ssl_toggled_signal_id;
  gint combo_auth_type_signal_id;
  gint path_signal_id;
  struct panel_data * data;
  
  data = panel->data;
  
  table = gtk_table_new(1, 3, FALSE);
  line = 0;
  
  /* port */
  gtk_table_resize(GTK_TABLE(table), line + 1, 3);
  checkbox_use_ssl = gtk_check_button_new_with_label(_("Use SSL"));
  gtk_widget_show(checkbox_use_ssl);
  gtk_table_attach(GTK_TABLE(table), checkbox_use_ssl, 0, 1, line, line + 1,
      0, GTK_SHRINK, /* attach options */
      PADDING, PADDING);
  
  gtk_table_resize(GTK_TABLE(table), line + 1, 3);
  label_port = gtk_label_new(_("Port"));
  gtk_widget_show(label_port);
  gtk_table_attach(GTK_TABLE(table), label_port, 1, 2, line, line + 1,
      0, GTK_SHRINK, /* attach options */
      PADDING, PADDING);
  entry_port = gtk_entry_new();
  gtk_widget_show(entry_port);
  gtk_table_attach(GTK_TABLE(table), entry_port, 2, 3, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  /* auth type */
  /*
    CRAM-MD5: auth_name, password
    anonymous: login
    Digest-MD5: auth_name, login, password
    GSSAPI (Kerberos V): login
    Kerberos4: login
    login: auth_name, login, password
    NTLM: auth_name, password
    OTP: auth_name, login, password
    PASSDSS-3DES-1: auth_name, login, password
    srp: auth_name, login, password
  */
  gtk_table_resize(GTK_TABLE(table), line + 1, 3);
  label_auth_type = gtk_label_new(_("Authentication Type"));
  gtk_widget_show(label_auth_type);
  gtk_table_attach(GTK_TABLE(table), label_auth_type, 0, 1, line, line + 1,
      0, GTK_SHRINK, /* attach options */
      PADDING, PADDING);
  combo_auth_type = gtk_combo_box_new_text();
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_auth_type), _("Password"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_auth_type), _("Plain Login"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_auth_type), _("Challenge-Response MD5 (CRAM-MD5)"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_auth_type), _("Digest MD5"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_auth_type), _("Kerberos V (GSSAPI)"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_auth_type), _("Kerberos 4"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_auth_type), _("One-Time-Password"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_auth_type), _("DSS Secured Password (PASSDSS-3DES-1)"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_auth_type), _("Secure Remote Password (SRP)"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_auth_type), _("NTLM"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_auth_type), _("Anonymous"));
  gtk_combo_box_set_active(GTK_COMBO_BOX(combo_auth_type), 0);
  gtk_widget_show(combo_auth_type);
  gtk_table_attach(GTK_TABLE(table), combo_auth_type, 1, 3, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  gtk_table_resize(GTK_TABLE(table), line + 1, 3);
  label_path = gtk_label_new(_("Mailbox Path"));
  gtk_widget_show(label_path);
  gtk_table_attach(GTK_TABLE(table), label_path, 0, 1, line, line + 1,
      0, GTK_SHRINK, /* attach options */
      PADDING, PADDING);
  
  hbox = gtk_hbox_new(FALSE, 0);
  entry_local_path = gtk_entry_new();
  gtk_entry_set_editable(GTK_ENTRY(entry_local_path), FALSE);
  gtk_widget_show(entry_local_path);
  gtk_box_pack_start(GTK_BOX(hbox), entry_local_path, TRUE, TRUE, 0);
  button_local_path = gtk_button_new_with_label(_("Change..."));
  gtk_widget_show(button_local_path);
  gtk_box_pack_start(GTK_BOX(hbox), button_local_path, FALSE, FALSE, 0);
  gtk_widget_show(hbox);
  gtk_table_attach(GTK_TABLE(table), hbox, 1, 3, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  
  line ++;
  
  gtk_widget_show(table);
  
  etpan_preferences_panel_set_widget(panel, "pop-use-ssl", checkbox_use_ssl);
  etpan_preferences_panel_set_widget(panel, "pop-port", entry_port);
  etpan_preferences_panel_set_widget(panel, "pop-auth-type", combo_auth_type);
  etpan_preferences_panel_set_widget(panel, "pop-path", entry_local_path);

  ssl_toggled_signal_id = g_signal_connect(checkbox_use_ssl,
      "toggled", G_CALLBACK(pop_ssl_toggled),
      (gpointer) panel);
  
  combo_auth_type_signal_id = g_signal_connect(combo_auth_type,
      "changed", G_CALLBACK(pop_auth_type_changed),
      (gpointer) panel);
  
  path_signal_id = g_signal_connect(button_local_path, "clicked",
      G_CALLBACK(pop_path_clicked), panel);
  
  signal_id = g_signal_connect(checkbox_use_ssl, "toggled",
      G_CALLBACK(checkbox_changed), panel);
  signal_id = g_signal_connect(entry_port, "changed",
      G_CALLBACK(entry_changed), panel);
  signal_id = g_signal_connect(combo_auth_type, "changed",
      G_CALLBACK(combo_changed), panel);
  
  return table;
}

static GtkWidget * pop_properties(struct etpan_preferences_panel * panel)
{
  GtkWidget * label_server;
  GtkWidget * entry_server;
  GtkWidget * label_login;
  GtkWidget * entry_login;
  GtkWidget * label_password;
  GtkWidget * entry_password;
  GtkWidget * server_expander;
  GtkWidget * pop_properties;
  int line;
  GtkWidget * table;
  gint signal_id;
  
  table = gtk_table_new(1, 2, FALSE);
  line = 0;
  
  /* (for POP, POP) */
  /* receive server, settings ... */
  gtk_table_resize(GTK_TABLE(table), line + 1, 2);
  label_server = gtk_label_new(_("Receive server"));
  gtk_widget_show(label_server);
  gtk_table_attach(GTK_TABLE(table), label_server, 0, 1, line, line + 1,
      0, GTK_SHRINK, /* attach options */
      PADDING, PADDING);
  entry_server = gtk_entry_new();
  gtk_widget_show(entry_server);
  gtk_table_attach(GTK_TABLE(table), entry_server, 1, 2, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  /* login */
  gtk_table_resize(GTK_TABLE(table), line + 1, 2);
  label_login = gtk_label_new(_("Login"));
  gtk_widget_show(label_login);
  gtk_table_attach(GTK_TABLE(table), label_login, 0, 1, line, line + 1,
      0, GTK_SHRINK, /* attach options */
      PADDING, PADDING);
  entry_login = gtk_entry_new();
  gtk_widget_show(entry_login);
  gtk_table_attach(GTK_TABLE(table), entry_login, 1, 2, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  /* password */
  gtk_table_resize(GTK_TABLE(table), line + 1, 2);
  label_password = gtk_label_new(_("Password"));
  gtk_widget_show(label_password);
  gtk_table_attach(GTK_TABLE(table), label_password, 0, 1, line, line + 1,
      0, GTK_SHRINK, /* attach options */
      PADDING, PADDING);
  entry_password = gtk_entry_new();
  gtk_entry_set_visibility(GTK_ENTRY(entry_password), FALSE);
  gtk_widget_show(entry_password);
  gtk_table_attach(GTK_TABLE(table), entry_password, 1, 2, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  /* more settings for receive server */
  gtk_table_resize(GTK_TABLE(table), line + 1, 2);
  server_expander = gtk_expander_new(_("Settings"));
  pop_properties = pop_optional_properties(panel);
  
  gtk_container_add(GTK_CONTAINER(server_expander), pop_properties);
  gtk_widget_show(server_expander);
  gtk_table_attach(GTK_TABLE(table), server_expander, 0, 2, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  gtk_widget_show(table);
  
  etpan_preferences_panel_set_widget(panel, "label-pop-server", label_server);
  etpan_preferences_panel_set_widget(panel, "pop-server", entry_server);
  etpan_preferences_panel_set_widget(panel, "pop-login", entry_login);
  etpan_preferences_panel_set_widget(panel, "pop-password", entry_password);
  etpan_preferences_panel_set_widget(panel, "pop-expander", server_expander);
  
  signal_id = g_signal_connect(entry_server, "changed",
      G_CALLBACK(entry_changed), panel);
  signal_id = g_signal_connect(entry_login, "changed",
      G_CALLBACK(entry_changed), panel);
  signal_id = g_signal_connect(entry_password, "changed",
      G_CALLBACK(entry_changed), panel);
  
  return table;
}

static GtkWidget * pop_frame(struct etpan_preferences_panel * panel)
{
#if 0
  GtkWidget * frame;
  GtkWidget * alignment;
  GtkWidget * properties;
  
  frame = gtk_frame_new(_("Receive server"));
  
  properties = pop_properties(panel);
  
  alignment = gtk_alignment_new(0.5, 0.5, 1.0, 1.0);
  gtk_container_add(GTK_CONTAINER(alignment), properties);
  gtk_alignment_set_padding(GTK_ALIGNMENT(alignment), 5, 5, 10, 10);
  gtk_widget_show(alignment);
  
  gtk_container_add(GTK_CONTAINER(frame), alignment);
  
  gtk_widget_show(frame);
  
  return frame;
#else
  return pop_properties(panel);
#endif
}

/* News */

static GtkWidget * news_optional_properties(struct etpan_preferences_panel * panel)
{
  GtkWidget * table;
  GtkWidget * label_port;
  GtkWidget * entry_port;
  GtkWidget * label_auth_type;
  GtkWidget * combo_auth_type;
  GtkWidget * checkbox_use_ssl;
  GtkWidget * label_login;
  GtkWidget * entry_login;
  GtkWidget * label_password;
  GtkWidget * entry_password;
  int line;
  gint signal_id;
  gint ssl_toggled_signal_id;
  gint combo_auth_type_signal_id;
  
  table = gtk_table_new(1, 3, FALSE);
  line = 0;
  
  /* port */
  gtk_table_resize(GTK_TABLE(table), line + 1, 3);
  checkbox_use_ssl = gtk_check_button_new_with_label(_("Use SSL"));
  gtk_widget_show(checkbox_use_ssl);
  gtk_table_attach(GTK_TABLE(table), checkbox_use_ssl, 0, 1, line, line + 1,
      0, GTK_SHRINK, /* attach options */
      PADDING, PADDING);
  
  gtk_table_resize(GTK_TABLE(table), line + 1, 3);
  label_port = gtk_label_new(_("Port"));
  gtk_widget_show(label_port);
  gtk_table_attach(GTK_TABLE(table), label_port, 1, 2, line, line + 1,
      0, GTK_SHRINK, /* attach options */
      PADDING, PADDING);
  entry_port = gtk_entry_new();
  gtk_widget_show(entry_port);
  gtk_table_attach(GTK_TABLE(table), entry_port, 2, 3, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  /* auth type */
  /*
    CRAM-MD5: auth_name, password
    anonymous: login
    Digest-MD5: auth_name, login, password
    GSSAPI (Kerberos V): login
    Kerberos4: login
    login: auth_name, login, password
    NTLM: auth_name, password
    OTP: auth_name, login, password
    PASSDSS-3DES-1: auth_name, login, password
    srp: auth_name, login, password
  */
  gtk_table_resize(GTK_TABLE(table), line + 1, 3);
  label_auth_type = gtk_label_new(_("Authentication Type"));
  gtk_widget_show(label_auth_type);
  gtk_table_attach(GTK_TABLE(table), label_auth_type, 0, 1, line, line + 1,
      0, GTK_SHRINK, /* attach options */
      PADDING, PADDING);
  combo_auth_type = gtk_combo_box_new_text();
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_auth_type), _("None"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_auth_type), _("Password"));
  gtk_combo_box_set_active(GTK_COMBO_BOX(combo_auth_type), 0);
  gtk_widget_show(combo_auth_type);
  gtk_table_attach(GTK_TABLE(table), combo_auth_type, 1, 3, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  /* login */
  gtk_table_resize(GTK_TABLE(table), line + 1, 2);
  label_login = gtk_label_new(_("Login"));
  gtk_widget_show(label_login);
  gtk_table_attach(GTK_TABLE(table), label_login, 0, 1, line, line + 1,
      0, GTK_SHRINK, /* attach options */
      PADDING, PADDING);
  entry_login = gtk_entry_new();
  gtk_widget_show(entry_login);
  gtk_table_attach(GTK_TABLE(table), entry_login, 1, 3, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  /* password */
  gtk_table_resize(GTK_TABLE(table), line + 1, 2);
  label_password = gtk_label_new(_("Password"));
  gtk_widget_show(label_password);
  gtk_table_attach(GTK_TABLE(table), label_password, 0, 1, line, line + 1,
      0, GTK_SHRINK, /* attach options */
      PADDING, PADDING);
  entry_password = gtk_entry_new();
  gtk_entry_set_visibility(GTK_ENTRY(entry_password), FALSE);
  gtk_widget_show(entry_password);
  gtk_table_attach(GTK_TABLE(table), entry_password, 1, 3, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  gtk_widget_show(table);
  
  etpan_preferences_panel_set_widget(panel, "news-use-ssl", checkbox_use_ssl);
  etpan_preferences_panel_set_widget(panel, "news-port", entry_port);
  etpan_preferences_panel_set_widget(panel, "news-auth-type", combo_auth_type);
  etpan_preferences_panel_set_widget(panel, "news-login", entry_login);
  etpan_preferences_panel_set_widget(panel, "news-password", entry_password);

  ssl_toggled_signal_id = g_signal_connect(checkbox_use_ssl,
      "toggled", G_CALLBACK(news_ssl_toggled),
      (gpointer) panel);
  
  combo_auth_type_signal_id = g_signal_connect(combo_auth_type,
      "changed", G_CALLBACK(news_auth_type_changed),
      (gpointer) panel);

  signal_id = g_signal_connect(entry_login, "changed",
      G_CALLBACK(entry_changed), panel);
  signal_id = g_signal_connect(entry_password, "changed",
      G_CALLBACK(entry_changed), panel);
  signal_id = g_signal_connect(checkbox_use_ssl, "toggled",
      G_CALLBACK(entry_changed), panel);
  signal_id = g_signal_connect(entry_port, "changed",
      G_CALLBACK(entry_changed), panel);
  signal_id = g_signal_connect(combo_auth_type, "changed",
      G_CALLBACK(combo_changed), panel);
  
  return table;
}

static GtkWidget * news_properties(struct etpan_preferences_panel * panel)
{
  GtkWidget * label_server;
  GtkWidget * entry_server;
  GtkWidget * server_expander;
  GtkWidget * news_properties;
  int line;
  GtkWidget * table;
  gint signal_id;
  
  table = gtk_table_new(1, 2, FALSE);
  line = 0;
  
  /* (for NEWS, POP) */
  /* receive server, settings ... */
  gtk_table_resize(GTK_TABLE(table), line + 1, 2);
  label_server = gtk_label_new(_("Receive server"));
  gtk_widget_show(label_server);
  gtk_table_attach(GTK_TABLE(table), label_server, 0, 1, line, line + 1,
      0, GTK_SHRINK, /* attach options */
      PADDING, PADDING);
  entry_server = gtk_entry_new();
  gtk_widget_show(entry_server);
  gtk_table_attach(GTK_TABLE(table), entry_server, 1, 2, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  /* more settings for receive server */
  gtk_table_resize(GTK_TABLE(table), line + 1, 2);
  server_expander = gtk_expander_new(_("Settings"));
  news_properties = news_optional_properties(panel);
  
  gtk_container_add(GTK_CONTAINER(server_expander), news_properties);
  gtk_widget_show(server_expander);
  gtk_table_attach(GTK_TABLE(table), server_expander, 0, 2, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  gtk_widget_show(table);
  
  etpan_preferences_panel_set_widget(panel, "label-news-server", label_server);
  etpan_preferences_panel_set_widget(panel, "news-server", entry_server);
  etpan_preferences_panel_set_widget(panel, "news-expander", server_expander);
  
  signal_id = g_signal_connect(entry_server, "changed",
      G_CALLBACK(entry_changed), panel);
  
  return table;
}

static GtkWidget * news_frame(struct etpan_preferences_panel * panel)
{
#if 0
  GtkWidget * frame;
  GtkWidget * alignment;
  GtkWidget * properties;
  
  frame = gtk_frame_new(_("Receive server"));
  
  properties = news_properties(panel);
  
  alignment = gtk_alignment_new(0.5, 0.5, 1.0, 1.0);
  gtk_container_add(GTK_CONTAINER(alignment), properties);
  gtk_alignment_set_padding(GTK_ALIGNMENT(alignment), 5, 5, 10, 10);
  gtk_widget_show(alignment);
  
  gtk_container_add(GTK_CONTAINER(frame), alignment);
  
  gtk_widget_show(frame);
  
  return frame;
#else
  return news_properties(panel);
#endif
}

/* Local */

static GtkWidget * local_properties(struct etpan_preferences_panel * panel)
{
  GtkWidget * label_local_type;
  GtkWidget * combo_local_type;
  GtkWidget * label_local_type_value;
  GtkWidget * label_local_path;
  GtkWidget * entry_local_path;
  GtkWidget * button_local_path;
  GtkWidget * hbox;
  int line;
  GtkWidget * table;
  gint signal_id;
  gint path_signal_id;
  gint combo_local_type_signal_id;
  struct panel_data * data;
  
  data = panel->data;
  
  table = gtk_table_new(1, 2, FALSE);
  line = 0;
  
  /* mailbox type */
  gtk_table_resize(GTK_TABLE(table), line + 2, 2);
  label_local_type = gtk_label_new(_("Mailbox Type"));
  gtk_widget_show(label_local_type);
  gtk_table_attach(GTK_TABLE(table), label_local_type, 0, 1, line, line + 2,
      0, GTK_SHRINK, /* attach options */
      PADDING, PADDING);
  combo_local_type = gtk_combo_box_new_text();
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_local_type), _("Maildir"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_local_type), _("MH"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_local_type), _("mbox"));
  gtk_combo_box_set_active(GTK_COMBO_BOX(combo_local_type), 0);
  gtk_widget_show(combo_local_type);
  gtk_table_attach(GTK_TABLE(table), combo_local_type, 1, 2, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  label_local_type_value = gtk_label_new(_("Undefined"));
  gtk_misc_set_alignment(GTK_MISC(label_local_type_value), 0, 0.5);
  gtk_widget_show(label_local_type_value);
  gtk_table_attach(GTK_TABLE(table), label_local_type_value, 1, 2, line + 1, line + 2,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line += 2;
  
  /* path */
  gtk_table_resize(GTK_TABLE(table), line + 1, 2);
  label_local_path = gtk_label_new(_("Mailbox Path"));
  gtk_widget_show(label_local_path);
  gtk_table_attach(GTK_TABLE(table), label_local_path, 0, 1, line, line + 1,
      0, GTK_SHRINK, /* attach options */
      PADDING, PADDING);

  hbox = gtk_hbox_new(FALSE, 0);
  entry_local_path = gtk_entry_new();
  gtk_entry_set_editable(GTK_ENTRY(entry_local_path), FALSE);
  gtk_widget_show(entry_local_path);
  gtk_box_pack_start(GTK_BOX(hbox), entry_local_path, TRUE, TRUE, 0);
  button_local_path = gtk_button_new_with_label(_("Change..."));
  gtk_widget_show(button_local_path);
  gtk_box_pack_start(GTK_BOX(hbox), button_local_path, FALSE, FALSE, 0);
  gtk_widget_show(hbox);
  gtk_table_attach(GTK_TABLE(table), hbox, 1, 2, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);

  line ++;
  
  gtk_widget_show(table);
  
  etpan_preferences_panel_set_widget(panel, "local-type", combo_local_type);
  etpan_preferences_panel_set_widget(panel, "local-type-value", label_local_type_value);
  etpan_preferences_panel_set_widget(panel, "label-local-path", label_local_path);
  etpan_preferences_panel_set_widget(panel, "local-path", entry_local_path);

  path_signal_id = g_signal_connect(button_local_path, "clicked",
      G_CALLBACK(local_path_clicked), panel);
  combo_local_type_signal_id = g_signal_connect(combo_local_type, "changed",
      G_CALLBACK(local_type_changed), panel);

  signal_id = g_signal_connect(combo_local_type, "changed",
      G_CALLBACK(combo_changed), panel);
  
  return table;
}

static GtkWidget * local_frame(struct etpan_preferences_panel * panel)
{
#if 0
  GtkWidget * frame;
  GtkWidget * alignment;
  GtkWidget * properties;
  
  frame = gtk_frame_new(_("Mailbox Information"));
  
  properties = local_properties(panel);
  
  alignment = gtk_alignment_new(0.5, 0.5, 1.0, 1.0);
  gtk_container_add(GTK_CONTAINER(alignment), properties);
  gtk_alignment_set_padding(GTK_ALIGNMENT(alignment), 5, 5, 10, 10);
  gtk_widget_show(alignment);
  
  gtk_container_add(GTK_CONTAINER(frame), alignment);
  
  gtk_widget_show(frame);
  
  return frame;
#else
  return local_properties(panel);
#endif
}

/* receive info */

static void receive_properties(struct etpan_preferences_panel * panel, GtkWidget * table)
{
  GtkWidget * frame;
  guint gline;
  int line;
  GtkWidget * separator;
  
  g_object_get(table, "n-rows", &gline, NULL);
  line = gline;
  
  gtk_table_resize(GTK_TABLE(table), line + 1, 2);
  
  separator = gtk_hseparator_new();
  gtk_widget_show(separator);
  gtk_table_attach(GTK_TABLE(table), separator, 0, 2, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  /* IMAP */
  gtk_table_resize(GTK_TABLE(table), line + 1, 2);
  frame = imap_frame(panel);
  gtk_table_attach(GTK_TABLE(table), frame, 0, 2, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  etpan_preferences_panel_set_widget(panel, "imap-properties", frame);
  
  /* POP */
  gtk_table_resize(GTK_TABLE(table), line + 1, 2);
  frame = pop_frame(panel);
  gtk_table_attach(GTK_TABLE(table), frame, 0, 2, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  etpan_preferences_panel_set_widget(panel, "pop-properties", frame);
  
  /* News */
  gtk_table_resize(GTK_TABLE(table), line + 1, 2);
  frame = news_frame(panel);
  gtk_table_attach(GTK_TABLE(table), frame, 0, 2, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  etpan_preferences_panel_set_widget(panel, "news-properties", frame);
  
  /* Local */
  gtk_table_resize(GTK_TABLE(table), line + 1, 2);
  frame = local_frame(panel);
  gtk_table_attach(GTK_TABLE(table), frame, 0, 2, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  etpan_preferences_panel_set_widget(panel, "local-properties", frame);
}


static GtkWidget * smtp_server_properties(struct etpan_preferences_panel * panel)
{
  GtkWidget * table;
  GtkWidget * checkbox_smtp_use_ssl;
  GtkWidget * label_smtp_port;
  GtkWidget * entry_smtp_port;
  GtkWidget * label_smtp_auth_type;
  GtkWidget * combo_smtp_auth_type;
  GtkWidget * label_smtp_login;
  GtkWidget * entry_smtp_login;
  GtkWidget * label_smtp_password;
  GtkWidget * entry_smtp_password;
  int line;
  gint signal_id;
  gint ssl_toggled_signal_id;
  gint combo_auth_type_signal_id;
  
  table = gtk_table_new(1, 3, FALSE);
  line = 0;
  
  /* port */
  gtk_table_resize(GTK_TABLE(table), line + 1, 3);
  checkbox_smtp_use_ssl = gtk_check_button_new_with_label(_("Use SSL"));
  gtk_widget_show(checkbox_smtp_use_ssl);
  gtk_table_attach(GTK_TABLE(table), checkbox_smtp_use_ssl, 0, 1, line, line + 1,
      0, GTK_SHRINK, /* attach options */
      PADDING, PADDING);
  
  gtk_table_resize(GTK_TABLE(table), line + 1, 3);
  label_smtp_port = gtk_label_new(_("Port"));
  gtk_widget_show(label_smtp_port);
  gtk_table_attach(GTK_TABLE(table), label_smtp_port, 1, 2, line, line + 1,
      0, GTK_SHRINK, /* attach options */
      PADDING, PADDING);
  entry_smtp_port = gtk_entry_new();
  gtk_widget_show(entry_smtp_port);
  gtk_table_attach(GTK_TABLE(table), entry_smtp_port, 2, 3, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  gtk_table_resize(GTK_TABLE(table), line + 1, 3);
  label_smtp_auth_type = gtk_label_new(_("Authentication Type"));
  gtk_widget_show(label_smtp_auth_type);
  gtk_table_attach(GTK_TABLE(table), label_smtp_auth_type, 0, 1, line, line + 1,
      0, GTK_SHRINK, /* attach options */
      PADDING, PADDING);
  combo_smtp_auth_type = gtk_combo_box_new_text();
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_smtp_auth_type), _("None"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_smtp_auth_type), _("Plain Login"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_smtp_auth_type), _("Challenge-Response MD5 (CRAM-MD5)"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_smtp_auth_type), _("Digest MD5"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_smtp_auth_type), _("Kerberos V (GSSAPI)"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_smtp_auth_type), _("Kerberos 4"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_smtp_auth_type), _("One-Time-Password"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_smtp_auth_type), _("DSS Secured Password (PASSDSS-3DES-1)"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_smtp_auth_type), _("Secure Remote Password (SRP)"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_smtp_auth_type), _("NTLM"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_smtp_auth_type), _("Anonymous"));
  gtk_combo_box_set_active(GTK_COMBO_BOX(combo_smtp_auth_type), 0);
  gtk_widget_show(combo_smtp_auth_type);
  gtk_table_attach(GTK_TABLE(table), combo_smtp_auth_type, 1, 3, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  /* login */
  gtk_table_resize(GTK_TABLE(table), line + 1, 3);
  label_smtp_login = gtk_label_new(_("Login"));
  gtk_widget_show(label_smtp_login);
  gtk_table_attach(GTK_TABLE(table), label_smtp_login, 0, 1, line, line + 1,
      0, GTK_SHRINK, /* attach options */
      PADDING, PADDING);
  entry_smtp_login = gtk_entry_new();
  gtk_widget_show(entry_smtp_login);
  gtk_table_attach(GTK_TABLE(table), entry_smtp_login, 1, 3, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  /* password */
  gtk_table_resize(GTK_TABLE(table), line + 1, 3);
  label_smtp_password = gtk_label_new(_("Password"));
  gtk_widget_show(label_smtp_password);
  gtk_table_attach(GTK_TABLE(table), label_smtp_password, 0, 1, line, line + 1,
      0, GTK_SHRINK, /* attach options */
      PADDING, PADDING);
  entry_smtp_password = gtk_entry_new();
  gtk_entry_set_visibility(GTK_ENTRY(entry_smtp_password), FALSE);
  gtk_widget_show(entry_smtp_password);
  gtk_table_attach(GTK_TABLE(table), entry_smtp_password, 1, 3, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  gtk_widget_show(table);
  
  etpan_preferences_panel_set_widget(panel, "smtp-use-ssl", checkbox_smtp_use_ssl);
  etpan_preferences_panel_set_widget(panel, "smtp-port", entry_smtp_port);
  etpan_preferences_panel_set_widget(panel, "smtp-auth-type", combo_smtp_auth_type);
  etpan_preferences_panel_set_widget(panel, "smtp-login", entry_smtp_login);
  etpan_preferences_panel_set_widget(panel, "smtp-password", entry_smtp_password);

  ssl_toggled_signal_id = g_signal_connect(checkbox_smtp_use_ssl,
      "toggled", G_CALLBACK(smtp_ssl_toggled),
      (gpointer) panel);
  
  combo_auth_type_signal_id = g_signal_connect(combo_smtp_auth_type,
      "changed", G_CALLBACK(smtp_auth_type_changed),
      (gpointer) panel);
  
  signal_id = g_signal_connect(checkbox_smtp_use_ssl, "toggled",
      G_CALLBACK(checkbox_changed), panel);
  signal_id = g_signal_connect(entry_smtp_port, "changed",
      G_CALLBACK(entry_changed), panel);
  signal_id = g_signal_connect(combo_smtp_auth_type, "changed",
      G_CALLBACK(combo_changed), panel);
  signal_id = g_signal_connect(entry_smtp_login, "changed",
      G_CALLBACK(entry_changed), panel);
  signal_id = g_signal_connect(entry_smtp_password, "changed",
      G_CALLBACK(entry_changed), panel);
  
  return table;
}

static GtkWidget * smtp_frame(struct etpan_preferences_panel * panel)
{
#if 0
  GtkWidget * frame;
#endif
  GtkWidget * label_smtp_server;
  GtkWidget * entry_smtp_server;
  GtkWidget * smtp_expander;
  GtkWidget * smtp_properties;
  int line;
  GtkWidget * table;
  GtkWidget * alignment;
  gint signal_id;
  GtkWidget * separator;
  
#if 0
  frame = gtk_frame_new(_("Sending Server"));
#endif
  
  table = gtk_table_new(1, 2, FALSE);
  line = 0;
  
  gtk_table_resize(GTK_TABLE(table), line + 1, 2);
  
  separator = gtk_hseparator_new();
  gtk_widget_show(separator);
  gtk_table_attach(GTK_TABLE(table), separator, 0, 2, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  /* send server (combo box entry), settings ... */
  gtk_table_resize(GTK_TABLE(table), line + 1, 2);
  label_smtp_server = gtk_label_new(_("Sending Server"));
  gtk_widget_show(label_smtp_server);
  gtk_table_attach(GTK_TABLE(table), label_smtp_server, 0, 1, line, line + 1,
      0, GTK_SHRINK, /* attach options */
      PADDING, PADDING);
  entry_smtp_server = gtk_entry_new();
  gtk_widget_show(entry_smtp_server);
  gtk_table_attach(GTK_TABLE(table), entry_smtp_server, 1, 2, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  /* expander */
  gtk_table_resize(GTK_TABLE(table), line + 1, 2);
  smtp_expander = gtk_expander_new(_("Settings"));
  smtp_properties = smtp_server_properties(panel);
  gtk_container_add(GTK_CONTAINER(smtp_expander), smtp_properties);
  gtk_widget_show(smtp_expander);
  gtk_table_attach(GTK_TABLE(table), smtp_expander, 0, 2, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  gtk_widget_show(table);
  
#if 0
  alignment = gtk_alignment_new(0.5, 0.5, 1.0, 1.0);
  gtk_container_add(GTK_CONTAINER(alignment), table);
  gtk_alignment_set_padding(GTK_ALIGNMENT(alignment), 5, 5, 10, 10);
  gtk_widget_show(alignment);
#endif
  
#if 0
  gtk_container_add(GTK_CONTAINER(frame), alignment);
  
  gtk_widget_show(frame);
#endif

  etpan_preferences_panel_set_widget(panel, "label-smtp-server", label_smtp_server);
  etpan_preferences_panel_set_widget(panel, "smtp-server", entry_smtp_server);
  etpan_preferences_panel_set_widget(panel, "smtp-expander", smtp_expander);

  signal_id = g_signal_connect(entry_smtp_server, "changed",
      G_CALLBACK(entry_changed), panel);
  
#if 0
  return frame;
#endif
  return table;
}

static void smtp_properties(struct etpan_preferences_panel * panel, GtkWidget * table)
{
  GtkWidget * frame;
  guint gline;
  int line;
  
  g_object_get(table, "n-rows", &gline, NULL);
  line = gline;
  
  gtk_table_resize(GTK_TABLE(table), line + 1, 2);
  
  frame = smtp_frame(panel);
  
  gtk_table_attach(GTK_TABLE(table), frame, 0, 2, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
}

static GtkWidget * account_properties(struct etpan_preferences_panel * panel)
{
  GtkWidget * table;
  GtkWidget * scrolledwindow;
  guint gline;
  int line;
  GtkWidget * button_hbox;
  GtkWidget * create_button;
  GtkWidget * cancel_button;
  GtkWidget * revert_button;
  GtkWidget * apply_button;
  gint create_signal_id;
  gint cancel_signal_id;
  gint revert_signal_id;
  gint apply_signal_id;
  GtkWidget * label_info;
  
  table = gtk_table_new(1, 2, FALSE);
  
  basic_account_properties(panel, table);
  receive_properties(panel, table);
  smtp_properties(panel, table);
  
  g_object_get(table, "n-rows", &gline, NULL);
  line = gline;
  
  gtk_table_resize(GTK_TABLE(table), line + 1, 2);
  
  button_hbox = gtk_hbutton_box_new();
  gtk_button_box_set_layout(GTK_BUTTON_BOX(button_hbox), GTK_BUTTONBOX_END);
  gtk_widget_show(button_hbox);
  
  cancel_button = gtk_button_new_with_label(_("Cancel"));
  gtk_widget_show(cancel_button);
  gtk_box_pack_start(GTK_BOX(button_hbox), cancel_button, FALSE, FALSE, 0);
  
  create_button = gtk_button_new_with_label(_("Create"));
  gtk_widget_show(create_button);
  gtk_box_pack_start(GTK_BOX(button_hbox), create_button, FALSE, FALSE, 0);
  
  revert_button = gtk_button_new_with_label(_("Revert"));
  gtk_widget_show(revert_button);
  gtk_box_pack_start(GTK_BOX(button_hbox), revert_button, FALSE, FALSE, 0);
  
  apply_button = gtk_button_new_with_label(_("Apply"));
  gtk_widget_show(apply_button);
  gtk_box_pack_start(GTK_BOX(button_hbox), apply_button, FALSE, FALSE, 0);
  
  gtk_widget_show(button_hbox);
  
  gtk_table_attach(GTK_TABLE(table), button_hbox, 0, 2, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  gtk_table_resize(GTK_TABLE(table), line + 1, 2);
  label_info = gtk_label_new("");
  gtk_widget_show(label_info);
  gtk_table_attach(GTK_TABLE(table), label_info, 0, 2, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  gtk_widget_show(table);
  
  scrolledwindow = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledwindow),
      GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW(scrolledwindow),
      GTK_SHADOW_IN);
  
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolledwindow),
      table);
  gtk_widget_show(scrolledwindow);

  etpan_preferences_panel_set_widget(panel, "table", table);
  etpan_preferences_panel_set_widget(panel, "label-info", label_info);
  etpan_preferences_panel_set_widget(panel, "create-button", create_button);
  etpan_preferences_panel_set_widget(panel, "cancel-button", cancel_button);
  etpan_preferences_panel_set_widget(panel, "revert-button", revert_button);
  etpan_preferences_panel_set_widget(panel, "apply-button", apply_button);

  create_signal_id = g_signal_connect(create_button,
      "clicked", G_CALLBACK(create_pressed_handler),
      (gpointer) panel);

  cancel_signal_id = g_signal_connect(cancel_button,
      "clicked", G_CALLBACK(cancel_pressed_handler),
      (gpointer) panel);

  revert_signal_id = g_signal_connect(revert_button,
      "clicked", G_CALLBACK(revert_pressed_handler),
      (gpointer) panel);

  apply_signal_id = g_signal_connect(apply_button,
      "clicked", G_CALLBACK(apply_pressed_handler),
      (gpointer) panel);
  
  return scrolledwindow;
}

void etpan_preferences_account_init(struct etpan_preferences_window * preferences)
{
  GtkWidget * hpaned;
  GtkWidget * treeview;
  GtkWidget * scrolledwindow;
  GtkWidget * vbox;
  GtkWidget * button_hbox;
  GtkWidget * add_button;
  GtkWidget * remove_button;
  GtkWidget * account_prop;
  struct etpan_preferences_panel * panel;
  GtkTreeSelection * selection;
  GtkTreeViewColumn * col_name;
  GtkCellRenderer * col_name_renderer;
  GtkTreeViewColumn * col_type;
  GtkCellRenderer * col_type_renderer;
  etpan_gtk_tree_model * treemodel;
  struct panel_data * data;
  gint selected_signal_id;
  gint add_signal_id;
  gint remove_signal_id;
  
  /*
  account page
  - add account
  - remove account
  - modify account
  */
  
  panel = etpan_preferences_panel_new();
  
  panel->open_callback = panel_open;
  panel->should_close_callback = panel_should_close;
  panel->close_callback = panel_close;
  panel->free_data = free_data;
  
  data = malloc(sizeof(* data));
  if (data == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  data->modified = 0;
  data->treemodel = NULL;
  data->pending_account = NULL;
  data->account_being_modified = NULL;
  data->chooser_state = CHOOSER_STATE_IDLE;
  data->account_set = 0;
  data->pop_path = NULL;
  data->local_path = NULL;
  
  panel->data = data;
  
  hpaned = gtk_hpaned_new();
  gtk_widget_show(hpaned);
  gtk_paned_set_position(GTK_PANED(hpaned), 200);
  
  vbox = gtk_vbox_new(FALSE, 0);
  gtk_widget_show(vbox);
  gtk_paned_pack1(GTK_PANED(hpaned), vbox, FALSE, TRUE);
  
  scrolledwindow = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrolledwindow),
      GTK_SHADOW_IN);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledwindow),
      GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_widget_show(scrolledwindow);
  gtk_box_pack_start(GTK_BOX(vbox), scrolledwindow, TRUE, TRUE, 0);
  
  treeview = gtk_tree_view_new();
  gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(treeview), TRUE);
  gtk_tree_view_set_enable_search(GTK_TREE_VIEW(treeview), FALSE);
  
  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
  gtk_tree_selection_set_mode(selection, GTK_SELECTION_BROWSE);
  
  /* column name */
  col_name = gtk_tree_view_column_new();
  gtk_tree_view_column_set_title(col_name, _("Name"));
  col_name_renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_column_set_resizable(col_name, TRUE);
  gtk_tree_view_column_pack_start(col_name, col_name_renderer, TRUE);
  gtk_tree_view_column_set_attributes(col_name, col_name_renderer,
      "text", STORE_INDEX_NAME, NULL);
  gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), col_name);
  
  /* column type */
  col_type = gtk_tree_view_column_new();
  gtk_tree_view_column_set_title(col_type, _("Type"));
  col_type_renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_column_pack_start(col_type, col_type_renderer, TRUE);
  gtk_tree_view_column_set_attributes(col_type, col_type_renderer,
      "text", STORE_INDEX_TYPE, NULL);
  gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), col_type);
  
  treemodel = etpan_gtk_tree_model_new();
  
  data->pending_account = NULL;
  
  data->account_datasource.data = panel;
  data->account_datasource.get_n_columns = get_n_columns;
  data->account_datasource.get_column_type = get_column_type;
  data->account_datasource.get_children_count = get_children_count;
  data->account_datasource.item_has_child = item_has_child;
  data->account_datasource.get_child_item = get_child_item;
  data->account_datasource.get_item_value = get_item_value;
  etpan_gtk_tree_model_set_datasource(treemodel,
      &data->account_datasource);
  
  gtk_tree_view_set_model(GTK_TREE_VIEW(treeview), GTK_TREE_MODEL(treemodel));
  
  gtk_widget_show(treeview);
  gtk_container_add(GTK_CONTAINER(scrolledwindow), treeview);
  
  button_hbox = gtk_hbutton_box_new();
  gtk_button_box_set_layout(GTK_BUTTON_BOX(button_hbox), GTK_BUTTONBOX_START);
  gtk_widget_show(button_hbox);
  gtk_box_pack_start(GTK_BOX(vbox), button_hbox, FALSE, FALSE, 0);
  
  add_button = gtk_button_new_with_label(_("+"));
  gtk_widget_show(add_button);
  gtk_box_pack_start(GTK_BOX(button_hbox), add_button, FALSE, FALSE, 0);
  remove_button = gtk_button_new_with_label(_("-"));
  gtk_widget_show(remove_button);
  gtk_box_pack_start(GTK_BOX(button_hbox), remove_button, TRUE, FALSE, 0);
  
  account_prop = account_properties(panel);
  
  gtk_paned_pack2(GTK_PANED(hpaned), account_prop, TRUE, TRUE);
  
  panel->main_widget = hpaned;
  
#define ICON(name) \
  etpan_icon_manager_new_scaled_image(etpan_icon_manager_get_default(), \
      name, 32)
  etpan_preferences_add_page(preferences, _("Account"), ICON("prefs-account"), panel);
#undef ICON
  
  data->treemodel = treemodel;
  
  etpan_preferences_panel_set_widget(panel, "account-slider", hpaned);
  etpan_preferences_panel_set_widget(panel, "account-list", treeview);
  etpan_preferences_panel_set_widget(panel, "add-button", add_button);
  etpan_preferences_panel_set_widget(panel, "remove-button", remove_button);
  
  selected_signal_id = g_signal_connect(treeview,
      "cursor-changed", G_CALLBACK(account_selected_handler),
      (gpointer) panel);
  
  add_signal_id = g_signal_connect(add_button,
      "clicked", G_CALLBACK(add_pressed_handler),
      (gpointer) panel);
  remove_signal_id = g_signal_connect(remove_button,
      "clicked", G_CALLBACK(remove_pressed_handler),
      (gpointer) panel);
  
  setup(panel);
}

/* *********************************** */
/* action handler */

static void account_selected_handler(GtkTreeView * treeview,
    gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  struct panel_data * data;
  struct etpan_account * account;
  GtkTreeIter iter;
  GtkTreePath * path;
  GtkTreeViewColumn * column;
  int go_back;
  GtkWidget * table;
  
  panel = user_data;
  data = panel->data;
  
  account = NULL;
  gtk_tree_view_get_cursor(treeview, &path, &column);
  if (path != NULL) {
    gtk_tree_model_get_iter(GTK_TREE_MODEL(data->treemodel), &iter, path);
    gtk_tree_model_get(GTK_TREE_MODEL(data->treemodel), &iter,
        STORE_INDEX_ACCOUNT,
        &account, -1);
    gtk_tree_path_free(path);
  }
  
  if (data->account_set) {
    if (data->account_being_modified != NULL) {
      if (data->account_being_modified == account) {
        return;
      }
    }
    if ((data->pending_account != NULL) && (account == NULL)) {
      return;
    }
  }
  
  go_back = 0;
  if (data->pending_account != NULL) {
    if (data->modified) {
      if (!check_consistency(panel))
        go_back = 1;
    }
  }
  else if (data->account_being_modified) {
    if (data->modified) {
      if (!check_consistency(panel))
        go_back = 1;
    }
  }
  
  if (go_back) {
    if (data->account_being_modified != NULL)
      etpan_gtk_tree_model_get_iter_from_item(data->treemodel, &iter,
          data->account_being_modified);
    else
      etpan_gtk_tree_model_get_iter_from_item(data->treemodel, &iter,
          data->pending_account);
    path = gtk_tree_model_get_path(GTK_TREE_MODEL(data->treemodel), &iter);
    gtk_tree_view_set_cursor(GTK_TREE_VIEW(treeview), path, NULL, FALSE);
    gtk_tree_path_free(path);
    
    return;
  }
  
  if (data->pending_account != NULL) {
    if (data->modified) {
      create_account(panel, 0);
    }
  }
  else if (data->account_being_modified != NULL) {
    if (data->modified) {
      modify_account(panel, 0);
    }
  }
  
  if ((data->pending_account != NULL) && (account != NULL))
    stop_add(panel);
  
  if (account != NULL)
    account_set(panel, account);
  else
    reset_properties(panel);
  
  table = etpan_preferences_panel_get_widget(panel, "table");
  gtk_widget_hide(table);
  gtk_widget_show(table);
  
  data->account_set = 1;
}

static void local_type_changed(GtkComboBox * widget, gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  char * filename;
  struct panel_data * data;
  
  panel = user_data;
  data = panel->data;
  if (data->pop_path != NULL)
    filename = strdup(data->pop_path);
  else
    filename = NULL;
  pop_path_update(panel, filename);
  local_path_update(panel, filename);
  free(filename);
}

static void account_id_changed(GtkEditable *editable,
    gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  char * filename;
  struct panel_data * data;
  
  panel = user_data;
  data = panel->data;
  if (data->pop_path != NULL)
    filename = strdup(data->pop_path);
  else
    filename = NULL;
  pop_path_update(panel, filename);
  local_path_update(panel, filename);
  free(filename);
}

static void account_type_changed(GtkComboBox * widget, gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  GtkWidget * combo;
  gint index;
  GtkWidget * frame;
  (void) widget;
  
  panel = user_data;
  combo = etpan_preferences_panel_get_widget(panel, "account-type");
  
  index = gtk_combo_box_get_active(GTK_COMBO_BOX(combo));
  
  frame = etpan_preferences_panel_get_widget(panel, "pop-properties");
  if (index == 0)
    gtk_widget_show(frame);
  else
    gtk_widget_hide(frame);

  frame = etpan_preferences_panel_get_widget(panel, "imap-properties");
  if (index == 1)
    gtk_widget_show(frame);
  else
    gtk_widget_hide(frame);

  frame = etpan_preferences_panel_get_widget(panel, "news-properties");
  if (index == 2)
    gtk_widget_show(frame);
  else
    gtk_widget_hide(frame);

  frame = etpan_preferences_panel_get_widget(panel, "local-properties");
  if (index == 3)
    gtk_widget_show(frame);
  else
    gtk_widget_hide(frame);
  
  reset_specific_properties(panel);
}

static void imap_ssl_toggled(GtkToggleButton * togglebutton, gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  GtkWidget * entry;
  GtkWidget * checkbox;
  gboolean value;
  (void) togglebutton;
  
  panel = user_data;
  entry = etpan_preferences_panel_get_widget(panel, "imap-port");
  
  checkbox = etpan_preferences_panel_get_widget(panel, "imap-use-ssl");
  value = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkbox));
  
  if (value)
    gtk_entry_set_text(GTK_ENTRY(entry), "993");
  else
    gtk_entry_set_text(GTK_ENTRY(entry), "143");
}

static void imap_auth_type_changed(GtkComboBox * widget, gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  GtkWidget * combo;
  gint index;
  GtkWidget * login;
  GtkWidget * password;
  (void) widget;
  
  panel = user_data;
  combo = etpan_preferences_panel_get_widget(panel, "imap-auth-type");
  
  login = etpan_preferences_panel_get_widget(panel, "imap-login");
  password = etpan_preferences_panel_get_widget(panel, "imap-password");
  index = gtk_combo_box_get_active(GTK_COMBO_BOX(combo));
  
  switch (index) {
  case 10:
    gtk_entry_set_text(GTK_ENTRY(login), "");
    gtk_entry_set_text(GTK_ENTRY(password), "");
    gtk_widget_set_sensitive(login, FALSE);
    gtk_widget_set_sensitive(password, FALSE);
    break;
    
  case 4:
  case 5:
    gtk_entry_set_text(GTK_ENTRY(password), "");
    gtk_widget_set_sensitive(login, TRUE);
    gtk_widget_set_sensitive(password, FALSE);
    break;
    
  default:
    gtk_widget_set_sensitive(login, TRUE);
    gtk_widget_set_sensitive(password, TRUE);
    break;
  }
}  

static void pop_ssl_toggled(GtkToggleButton * togglebutton, gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  GtkWidget * entry;
  GtkWidget * checkbox;
  gboolean value;
  (void) togglebutton;
  
  panel = user_data;
  entry = etpan_preferences_panel_get_widget(panel, "pop-port");
  
  checkbox = etpan_preferences_panel_get_widget(panel, "pop-use-ssl");
  value = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkbox));
  
  if (value)
    gtk_entry_set_text(GTK_ENTRY(entry), "995");
  else
    gtk_entry_set_text(GTK_ENTRY(entry), "110");
}

static void pop_auth_type_changed(GtkComboBox * widget, gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  GtkWidget * combo;
  gint index;
  GtkWidget * login;
  GtkWidget * password;
  (void) widget;
  
  panel = user_data;
  combo = etpan_preferences_panel_get_widget(panel, "pop-auth-type");
  
  login = etpan_preferences_panel_get_widget(panel, "pop-login");
  password = etpan_preferences_panel_get_widget(panel, "pop-password");
  index = gtk_combo_box_get_active(GTK_COMBO_BOX(combo));
  
  switch (index) {
  case 10:
    gtk_entry_set_text(GTK_ENTRY(login), "");
    gtk_entry_set_text(GTK_ENTRY(password), "");
    gtk_widget_set_sensitive(login, FALSE);
    gtk_widget_set_sensitive(password, FALSE);
    break;
    
  case 4:
  case 5:
    gtk_entry_set_text(GTK_ENTRY(password), "");
    gtk_widget_set_sensitive(login, TRUE);
    gtk_widget_set_sensitive(password, FALSE);
    break;
    
  default:
    gtk_widget_set_sensitive(login, TRUE);
    gtk_widget_set_sensitive(password, TRUE);
    break;
  }
}  

static void pop_path_update(struct etpan_preferences_panel * panel,
    char * filename)
{
  struct panel_data * data;
  
  data = panel->data;
  
  free(data->pop_path);
  if (filename != NULL) {
    GtkWidget * entry;
    char * base;
    
    data->pop_path = strdup(filename);
    if (data->pop_path == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  
    entry = etpan_preferences_panel_get_widget(panel, "pop-path");
    filename = strdup(filename);
    if (filename == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    base = basename(filename);
    gtk_entry_set_text(GTK_ENTRY(entry), base);
    free(filename);
  }
  else {
    GtkWidget * entry;
    char path[PATH_MAX];
    const char * name;
    
    data->pop_path = NULL;
    entry = etpan_preferences_panel_get_widget(panel, "account-name");
    name = gtk_entry_get_text(GTK_ENTRY(entry));
    if ((name == NULL) || (* name == '\0')) {
      entry = etpan_preferences_panel_get_widget(panel, "pop-path");
      gtk_entry_set_text(GTK_ENTRY(entry), _("(None)"));
    }
    else {
      entry = etpan_preferences_panel_get_widget(panel, "pop-path");
      snprintf(path, sizeof(path), "Maildir-%s", name);
      gtk_entry_set_text(GTK_ENTRY(entry), path);
    }
  }
}

static void pop_path_clicked(GtkButton * button, gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  struct panel_data * data;
  GtkWidget * dialog;
  int response;
  (void) button;

  panel = user_data;
  data = panel->data;
  
  dialog = gtk_file_chooser_dialog_new(_("Change Mailbox Path"),
      NULL, GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER,
      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
      GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT, NULL);
  if (data->pop_path != NULL) {
    gtk_file_chooser_set_filename(GTK_FILE_CHOOSER(dialog), data->pop_path);
  }
  else {
    char * home;
    
    home = getenv("HOME");
    ETPAN_ASSERT(home != NULL, "home should not be nil");
    gtk_file_chooser_set_filename(GTK_FILE_CHOOSER(dialog), home);
  }
  response = gtk_dialog_run(GTK_DIALOG(dialog));
  
  if (response == GTK_RESPONSE_ACCEPT) {
    char * filename;
    
    data->modified = 1;
    
    enable_buttons(panel);
    
    filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog));
    
    pop_path_update(panel, filename);
    
    g_free(filename);
  }
  
  gtk_widget_destroy (dialog);
}

static void local_path_update(struct etpan_preferences_panel * panel,
    char * filename)
{
  struct panel_data * data;
  gint index;
  GtkWidget * combo;
  int type;
  
  data = panel->data;

  combo = etpan_preferences_panel_get_widget(panel, "local-type");
  index = gtk_combo_box_get_active(GTK_COMBO_BOX(combo));
  
  switch (index) {
  case 0:
    type = STORAGE_DRIVER_TYPE_MAILDIR;
    break;
  case 1:
    type = STORAGE_DRIVER_TYPE_MH;
    break;
  case 2:
    type = STORAGE_DRIVER_TYPE_MBOX;
    break;
  default:
    type = STORAGE_DRIVER_TYPE_MAILDIR;
    ETPAN_CRASH("invalid storage type");
    break;
  }
  
  free(data->local_path);
  if (filename != NULL) {
    GtkWidget * entry;
    char * base;
    
    data->local_path = strdup(filename);
    if (data->local_path == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  
    entry = etpan_preferences_panel_get_widget(panel, "local-path");
    filename = strdup(filename);
    if (filename == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    base = basename(filename);
    gtk_entry_set_text(GTK_ENTRY(entry), base);
    free(filename);
  }
  else {
    GtkWidget * entry;
    char path[PATH_MAX];
    const char * name;
    
    data->local_path = NULL;
    entry = etpan_preferences_panel_get_widget(panel, "account-name");
    name = gtk_entry_get_text(GTK_ENTRY(entry));
    if ((name == NULL) || (* name == '\0')) {
      entry = etpan_preferences_panel_get_widget(panel, "local-path");
      gtk_entry_set_text(GTK_ENTRY(entry), _("(None)"));
    }
    else {
      entry = etpan_preferences_panel_get_widget(panel, "local-path");
      switch (type) {
      case STORAGE_DRIVER_TYPE_MAILDIR:
        snprintf(path, sizeof(path), "Maildir-%s", name);
        break;
      case STORAGE_DRIVER_TYPE_MH:
        snprintf(path, sizeof(path), "MH-%s", name);
        break;
      case STORAGE_DRIVER_TYPE_MBOX:
        snprintf(path, sizeof(path), "mail-%s", name);
        break;
      default:
        ETPAN_CRASH("invalid storage type");
        break;
      }
      gtk_entry_set_text(GTK_ENTRY(entry), path);
    }
  }
}

static void local_path_clicked(GtkButton * button, gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  struct panel_data * data;
  GtkWidget * dialog;
  int response;
  (void) button;
  
  panel = user_data;
  data = panel->data;
  
  dialog = gtk_file_chooser_dialog_new(_("Change Mailbox Path"),
      NULL, GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER,
      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
      GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT, NULL);
  if (data->local_path != NULL) {
    gtk_file_chooser_set_filename(GTK_FILE_CHOOSER(dialog), data->local_path);
  }
  else {
    char * home;
    
    home = getenv("HOME");
    ETPAN_ASSERT(home != NULL, "home should not be nil");
    gtk_file_chooser_set_filename(GTK_FILE_CHOOSER(dialog), home);
  }
  response = gtk_dialog_run(GTK_DIALOG(dialog));
  
  if (response == GTK_RESPONSE_ACCEPT) {
    char * filename;
    
    data->modified = 1;
    
    enable_buttons(panel);
    
    filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog));
    
    local_path_update(panel, filename);
    
    g_free(filename);
  }
  
  gtk_widget_destroy (dialog);
}

static void news_ssl_toggled(GtkToggleButton * togglebutton, gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  GtkWidget * entry;
  GtkWidget * checkbox;
  gboolean value;
  (void) togglebutton;
  
  panel = user_data;
  entry = etpan_preferences_panel_get_widget(panel, "news-port");
  
  checkbox = etpan_preferences_panel_get_widget(panel, "news-use-ssl");
  value = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkbox));
  
  if (value)
    gtk_entry_set_text(GTK_ENTRY(entry), "563");
  else
    gtk_entry_set_text(GTK_ENTRY(entry), "119");
}

static void news_auth_type_changed(GtkComboBox * widget, gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  GtkWidget * combo;
  gint index;
  GtkWidget * login;
  GtkWidget * password;
  (void) widget;
  
  panel = user_data;
  combo = etpan_preferences_panel_get_widget(panel, "news-auth-type");
  
  login = etpan_preferences_panel_get_widget(panel, "news-login");
  password = etpan_preferences_panel_get_widget(panel, "news-password");
  index = gtk_combo_box_get_active(GTK_COMBO_BOX(combo));
  
  switch (index) {
  case 0:
    gtk_entry_set_text(GTK_ENTRY(login), "");
    gtk_entry_set_text(GTK_ENTRY(password), "");
    gtk_widget_set_sensitive(login, FALSE);
    gtk_widget_set_sensitive(password, FALSE);
    break;
    
  default:
    gtk_widget_set_sensitive(login, TRUE);
    gtk_widget_set_sensitive(password, TRUE);
    break;
  }
}  

static void smtp_ssl_toggled(GtkToggleButton * togglebutton, gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  GtkWidget * entry;
  GtkWidget * checkbox;
  gboolean value;
  (void) togglebutton;
  
  panel = user_data;
  entry = etpan_preferences_panel_get_widget(panel, "smtp-port");
  
  checkbox = etpan_preferences_panel_get_widget(panel, "smtp-use-ssl");
  value = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkbox));
  
  if (value)
    gtk_entry_set_text(GTK_ENTRY(entry), "465");
  else
    gtk_entry_set_text(GTK_ENTRY(entry), "25");
}

static void smtp_auth_type_changed(GtkComboBox * widget, gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  GtkWidget * combo;
  gint index;
  GtkWidget * login;
  GtkWidget * password;
  (void) widget;
  
  panel = user_data;
  combo = etpan_preferences_panel_get_widget(panel, "smtp-auth-type");
  
  login = etpan_preferences_panel_get_widget(panel, "smtp-login");
  password = etpan_preferences_panel_get_widget(panel, "smtp-password");
  index = gtk_combo_box_get_active(GTK_COMBO_BOX(combo));
  
  switch (index) {
  case 0:
  case 10:
    gtk_entry_set_text(GTK_ENTRY(login), "");
    gtk_entry_set_text(GTK_ENTRY(password), "");
    gtk_widget_set_sensitive(login, FALSE);
    gtk_widget_set_sensitive(password, FALSE);
    break;
    
  case 4:
  case 5:
    gtk_entry_set_text(GTK_ENTRY(password), "");
    gtk_widget_set_sensitive(login, TRUE);
    gtk_widget_set_sensitive(password, FALSE);
    break;
    
  default:
    gtk_widget_set_sensitive(login, TRUE);
    gtk_widget_set_sensitive(password, TRUE);
    break;
  }
}  

static void create_pressed_handler(GtkButton *button, gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  (void) button;
  
  panel = user_data;
  
  create_account(panel, 1);
}

static void cancel_pressed_handler(GtkButton *button, gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  (void) button;
  
  panel = user_data;
  
  remove_account(panel);
}

static void apply_pressed_handler(GtkButton *button, gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  (void) button;
  
  panel = user_data;
  
  modify_account(panel, 1);
}

static void revert_pressed_handler(GtkButton *button, gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  struct panel_data * data;
  (void) button;
  
  panel = user_data;
  data = panel->data;
  
  account_set(panel, data->account_being_modified);
}

static void add_pressed_handler(GtkButton *button, gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  (void) button;
  
  panel = user_data;
  
  add_account(panel);
}

static void remove_pressed_handler(GtkButton *button, gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  (void) button;
  
  panel = user_data;
  
  remove_account(panel);
}

static void entry_changed(GtkEditable *editable,
    gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  struct panel_data * data;
  (void) editable;
  
  panel = user_data;
  data = panel->data;
  data->modified = 1;
  
  enable_buttons(panel);
}

static void combo_changed(GtkComboBox * widget, gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  struct panel_data * data;
  (void) widget;
  
  panel = user_data;
  data = panel->data;
  data->modified = 1;
  
  enable_buttons(panel);
}

static void checkbox_changed(GtkToggleButton * togglebutton,
    gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  struct panel_data * data;
  (void) togglebutton;
  
  panel = user_data;
  data = panel->data;
  data->modified = 1;
  
  enable_buttons(panel);
}

/* *********************************** */
/* setup */

static void setup(struct etpan_preferences_panel * panel)
{
  carray * list;
  
  list = etpan_account_manager_get_ordered_list(etpan_account_manager_get_default());
  if (carray_count(list) == 0) {
    add_account(panel);
  }
}

/* *********************************** */
/* fill entries */

static void reset_specific_properties(struct etpan_preferences_panel * panel)
{
  GtkWidget * expander;
  GtkWidget * entry;
  GtkWidget * combo;
  GtkWidget * checkbox;
  GtkWidget * label;
  GtkWidget * chooser;
  struct panel_data * data;
  
  /* IMAP */
  
  data = panel->data;
  expander = etpan_preferences_panel_get_widget(panel, "imap-expander");
  gtk_expander_set_expanded(GTK_EXPANDER(expander), FALSE);

  entry = etpan_preferences_panel_get_widget(panel, "imap-server");
  gtk_entry_set_text(GTK_ENTRY(entry), "");
  
  entry = etpan_preferences_panel_get_widget(panel, "imap-login");
  gtk_entry_set_text(GTK_ENTRY(entry), "");
  
  entry = etpan_preferences_panel_get_widget(panel, "imap-password");
  gtk_entry_set_text(GTK_ENTRY(entry), "");
  
  checkbox = etpan_preferences_panel_get_widget(panel, "imap-use-ssl");
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkbox), FALSE);
  imap_ssl_toggled(GTK_TOGGLE_BUTTON(checkbox), panel);
  
  entry = etpan_preferences_panel_get_widget(panel, "imap-port");
  gtk_entry_set_text(GTK_ENTRY(entry), "143");
  
  combo = etpan_preferences_panel_get_widget(panel, "imap-auth-type");
  gtk_combo_box_set_active(GTK_COMBO_BOX(combo), 0);
  imap_auth_type_changed(GTK_COMBO_BOX(combo), panel);
  
  entry = etpan_preferences_panel_get_widget(panel, "imap-prefix");
  gtk_entry_set_text(GTK_ENTRY(entry), "");
  
  /* POP */

  expander = etpan_preferences_panel_get_widget(panel, "pop-expander");
  gtk_expander_set_expanded(GTK_EXPANDER(expander), FALSE);

  entry = etpan_preferences_panel_get_widget(panel, "pop-server");
  gtk_entry_set_text(GTK_ENTRY(entry), "");
  
  entry = etpan_preferences_panel_get_widget(panel, "pop-login");
  gtk_entry_set_text(GTK_ENTRY(entry), "");
  
  entry = etpan_preferences_panel_get_widget(panel, "pop-password");
  gtk_entry_set_text(GTK_ENTRY(entry), "");
  
  checkbox = etpan_preferences_panel_get_widget(panel, "pop-use-ssl");
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkbox), FALSE);
  pop_ssl_toggled(GTK_TOGGLE_BUTTON(checkbox), panel);
  
  entry = etpan_preferences_panel_get_widget(panel, "pop-port");
  gtk_entry_set_text(GTK_ENTRY(entry), "110");
  
  combo = etpan_preferences_panel_get_widget(panel, "pop-auth-type");
  gtk_combo_box_set_active(GTK_COMBO_BOX(combo), 0);
  pop_auth_type_changed(GTK_COMBO_BOX(combo), panel);

  pop_path_update(panel, NULL);
  
  /* News */

  expander = etpan_preferences_panel_get_widget(panel, "news-expander");
  gtk_expander_set_expanded(GTK_EXPANDER(expander), FALSE);

  entry = etpan_preferences_panel_get_widget(panel, "news-server");
  gtk_entry_set_text(GTK_ENTRY(entry), "");
  
  checkbox = etpan_preferences_panel_get_widget(panel, "news-use-ssl");
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkbox), FALSE);
  news_ssl_toggled(GTK_TOGGLE_BUTTON(checkbox), panel);
  
  entry = etpan_preferences_panel_get_widget(panel, "news-port");
  gtk_entry_set_text(GTK_ENTRY(entry), "119");
  
  combo = etpan_preferences_panel_get_widget(panel, "news-auth-type");
  gtk_combo_box_set_active(GTK_COMBO_BOX(combo), 0);
  news_auth_type_changed(GTK_COMBO_BOX(combo), panel);
  
  entry = etpan_preferences_panel_get_widget(panel, "news-login");
  gtk_entry_set_text(GTK_ENTRY(entry), "");
  
  entry = etpan_preferences_panel_get_widget(panel, "news-password");
  gtk_entry_set_text(GTK_ENTRY(entry), "");
  
  /* Local */
  combo = etpan_preferences_panel_get_widget(panel, "local-type");
  gtk_combo_box_set_active(GTK_COMBO_BOX(combo), 0);
  gtk_widget_show(combo);
  
  label = etpan_preferences_panel_get_widget(panel, "local-type-value");
  gtk_label_set_text(GTK_LABEL(label), _("Undefined"));
  gtk_widget_hide(label);
  
  local_path_update(panel, NULL);
}

static void reset_properties(struct etpan_preferences_panel * panel)
{
  GtkWidget * entry_name;
  GtkWidget * entry_display_name;
  GtkWidget * entry_address;
  GtkWidget * combo_account_type;
  GtkWidget * label_account_type_value;
  GtkWidget * expander;
  GtkWidget * entry;
  GtkWidget * checkbox;
  GtkWidget * combo;
  GtkWidget * button;
  struct panel_data * data;
  
  data = panel->data;
  
  switch_text(panel, "label-account-name", _("<span foreground=\"#000000\">Account Name</span>"));
  switch_text(panel, "label-address", _("<span foreground=\"#000000\">E-mail Address</span>"));
  switch_text(panel, "label-pop-server", _("<span foreground=\"#000000\">Receive Server</span>"));
  switch_text(panel, "label-imap-server", _("<span foreground=\"#000000\">Receive Server</span>"));
  switch_text(panel, "label-news-server", _("<span foreground=\"#000000\">Receive Server</span>"));
  switch_text(panel, "label-smtp-server", _("<span foreground=\"#000000\">Sending Server</span>"));
  switch_text(panel, "label-local-path", "<span foreground=\"#000000\">Mailbox Path</span>");
  
  entry_name = etpan_preferences_panel_get_widget(panel, "account-name");
  gtk_entry_set_text(GTK_ENTRY(entry_name), "");

  entry_display_name = etpan_preferences_panel_get_widget(panel, "display-name");
  gtk_entry_set_text(GTK_ENTRY(entry_display_name), "");
  
  entry_address = etpan_preferences_panel_get_widget(panel, "address");
  gtk_entry_set_text(GTK_ENTRY(entry_address), "");
  
  label_account_type_value = etpan_preferences_panel_get_widget(panel, "account-type-label");
  gtk_widget_hide(label_account_type_value);
  
  combo_account_type = etpan_preferences_panel_get_widget(panel, "account-type");
  gtk_widget_show(combo_account_type);
  
  gtk_combo_box_set_active(GTK_COMBO_BOX(combo_account_type), 0);
  account_type_changed(GTK_COMBO_BOX(combo_account_type), panel);
  
  /* SMTP */
  
  expander = etpan_preferences_panel_get_widget(panel, "smtp-expander");
  gtk_expander_set_expanded(GTK_EXPANDER(expander), FALSE);
  
  entry = etpan_preferences_panel_get_widget(panel, "smtp-server");
  gtk_entry_set_text(GTK_ENTRY(entry), "");
  
  checkbox = etpan_preferences_panel_get_widget(panel, "smtp-use-ssl");
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkbox), FALSE);
  smtp_ssl_toggled(GTK_TOGGLE_BUTTON(checkbox), panel);
  
  entry = etpan_preferences_panel_get_widget(panel, "smtp-port");
  gtk_entry_set_text(GTK_ENTRY(entry), "25");
  
  combo = etpan_preferences_panel_get_widget(panel, "smtp-auth-type");
  gtk_combo_box_set_active(GTK_COMBO_BOX(combo), 0);
  smtp_auth_type_changed(GTK_COMBO_BOX(combo), panel);
  
  entry = etpan_preferences_panel_get_widget(panel, "smtp-login");
  gtk_entry_set_text(GTK_ENTRY(entry), "");
  
  entry = etpan_preferences_panel_get_widget(panel, "smtp-password");
  gtk_entry_set_text(GTK_ENTRY(entry), "");

  button = etpan_preferences_panel_get_widget(panel, "create-button");
  gtk_widget_show(button);

  button = etpan_preferences_panel_get_widget(panel, "cancel-button");
  gtk_widget_show(button);
  
  button = etpan_preferences_panel_get_widget(panel, "revert-button");
  gtk_widget_hide(button);
  
  button = etpan_preferences_panel_get_widget(panel, "apply-button");
  gtk_widget_hide(button);
  
  switch_text(panel, "label-info", "");
  
  data->account_being_modified = NULL;
}

static void pop_account_set(struct etpan_preferences_panel * panel,
    struct etpan_account * account)
{
  struct etpan_storage * storage;
  char * hostname;
  int port;
  int connection_type;
  int auth_type;
  char * username;
  char * password;
  char * path;
  GtkWidget * entry;
  GtkWidget * checkbox;
  GtkWidget * combo;
  GtkWidget * chooser;
  gboolean use_ssl_value;
  gint value;
  char port_str[20];
  
  storage = etpan_account_get_storage(account);
  
  entry = etpan_preferences_panel_get_widget(panel, "pop-server");
  hostname = etpan_storage_pop_sync_get_hostname(storage);
  if (hostname != NULL)
    gtk_entry_set_text(GTK_ENTRY(entry), hostname);
  
  entry = etpan_preferences_panel_get_widget(panel, "pop-login");
  username = etpan_storage_pop_sync_get_username(storage);
  if (username != NULL)
    gtk_entry_set_text(GTK_ENTRY(entry), username);
  
  entry = etpan_preferences_panel_get_widget(panel, "pop-password");
  password = etpan_storage_pop_sync_get_password(storage);
  if (password != NULL)
    gtk_entry_set_text(GTK_ENTRY(entry), password);
  
  connection_type = etpan_storage_pop_sync_get_connection_type(storage);
  checkbox = etpan_preferences_panel_get_widget(panel, "pop-use-ssl");
  use_ssl_value = (connection_type == ETPAN_CONNECTION_TLS);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkbox), use_ssl_value);
  pop_ssl_toggled(GTK_TOGGLE_BUTTON(checkbox), panel);

  entry = etpan_preferences_panel_get_widget(panel, "pop-port");
  port = etpan_storage_pop_sync_get_port(storage);
  snprintf(port_str, sizeof(port_str), "%i", port);
  gtk_entry_set_text(GTK_ENTRY(entry), port_str);
  
  combo = etpan_preferences_panel_get_widget(panel, "pop-auth-type");
  auth_type = etpan_storage_pop_sync_get_auth_type(storage);
  switch (auth_type) {
  case ETPAN_AUTH_PLAIN:
    value = 1;
    break;
  case ETPAN_AUTH_CRAM_MD5:
    value = 2;
    break;
  case ETPAN_AUTH_DIGEST_MD5:
    value = 3;
    break;
  case ETPAN_AUTH_GSSAPI:
    value = 4;
    break;
  case ETPAN_AUTH_KERBEROS4:
    value = 5;
    break;
  case ETPAN_AUTH_OTP:
    value = 6;
    break;
  case ETPAN_AUTH_PASSDSS_3DES_1:
    value = 7;
    break;
  case ETPAN_AUTH_SRP:
    value = 8;
    break;
  case ETPAN_AUTH_NTLM:
    value = 9;
    break;
  case ETPAN_AUTH_ANONYMOUS:
    value = 10;
    break;
  default: /* ETPAN_AUTH_PASSWORD */
    value = 0;
    break;
  }
  
  gtk_combo_box_set_active(GTK_COMBO_BOX(combo), value);
  pop_auth_type_changed(GTK_COMBO_BOX(combo), panel);

  path = etpan_storage_pop_sync_get_path(storage);
  pop_path_update(panel, path);
}

static void imap_account_set(struct etpan_preferences_panel * panel,
    struct etpan_account * account)
{
  struct etpan_storage * storage;
  char * hostname;
  int port;
  int connection_type;
  int auth_type;
  char * username;
  char * password;
  char * base_location;
  GtkWidget * entry;
  GtkWidget * checkbox;
  GtkWidget * combo;
  gboolean use_ssl_value;
  gint value;
  char port_str[20];
  
  storage = etpan_account_get_storage(account);
  
  entry = etpan_preferences_panel_get_widget(panel, "imap-server");
  hostname = etpan_storage_imap_sync_get_hostname(storage);
  if (hostname != NULL)
    gtk_entry_set_text(GTK_ENTRY(entry), hostname);
  
  entry = etpan_preferences_panel_get_widget(panel, "imap-login");
  username = etpan_storage_imap_sync_get_username(storage);
  if (username != NULL)
    gtk_entry_set_text(GTK_ENTRY(entry), username);
  
  entry = etpan_preferences_panel_get_widget(panel, "imap-password");
  password = etpan_storage_imap_sync_get_password(storage);
  if (password != NULL)
    gtk_entry_set_text(GTK_ENTRY(entry), password);
  
  entry = etpan_preferences_panel_get_widget(panel, "imap-prefix");
  base_location = etpan_storage_imap_sync_get_base_location(storage);
  if (base_location != NULL)
    gtk_entry_set_text(GTK_ENTRY(entry), base_location);
  
  connection_type = etpan_storage_imap_sync_get_connection_type(storage);
  checkbox = etpan_preferences_panel_get_widget(panel, "imap-use-ssl");
  use_ssl_value = (connection_type == ETPAN_CONNECTION_TLS);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkbox), use_ssl_value);
  imap_ssl_toggled(GTK_TOGGLE_BUTTON(checkbox), panel);

  entry = etpan_preferences_panel_get_widget(panel, "imap-port");
  port = etpan_storage_imap_sync_get_port(storage);
  snprintf(port_str, sizeof(port_str), "%i", port);
  gtk_entry_set_text(GTK_ENTRY(entry), port_str);
  
  combo = etpan_preferences_panel_get_widget(panel, "imap-auth-type");
  auth_type = etpan_storage_imap_sync_get_auth_type(storage);
  switch (auth_type) {
  case ETPAN_AUTH_PLAIN:
    value = 1;
    break;
  case ETPAN_AUTH_CRAM_MD5:
    value = 2;
    break;
  case ETPAN_AUTH_DIGEST_MD5:
    value = 3;
    break;
  case ETPAN_AUTH_GSSAPI:
    value = 4;
    break;
  case ETPAN_AUTH_KERBEROS4:
    value = 5;
    break;
  case ETPAN_AUTH_OTP:
    value = 6;
    break;
  case ETPAN_AUTH_PASSDSS_3DES_1:
    value = 7;
    break;
  case ETPAN_AUTH_SRP:
    value = 8;
    break;
  case ETPAN_AUTH_NTLM:
    value = 9;
    break;
  case ETPAN_AUTH_ANONYMOUS:
    value = 10;
    break;
  default: /* ETPAN_AUTH_PASSWORD */
    value = 0;
    break;
  }
  
  gtk_combo_box_set_active(GTK_COMBO_BOX(combo), value);
  imap_auth_type_changed(GTK_COMBO_BOX(combo), panel);
}

static void news_account_set(struct etpan_preferences_panel * panel,
    struct etpan_account * account)
{
  struct etpan_storage * storage;
  char * hostname;
  int port;
  int connection_type;
  char * username;
  char * password;
  GtkWidget * entry;
  GtkWidget * checkbox;
  GtkWidget * combo;
  gboolean use_ssl_value;
  gint value;
  char port_str[20];
  
  storage = etpan_account_get_storage(account);
  
  entry = etpan_preferences_panel_get_widget(panel, "news-server");
  hostname = etpan_storage_news_get_hostname(storage);
  if (hostname != NULL)
    gtk_entry_set_text(GTK_ENTRY(entry), hostname);
  
  entry = etpan_preferences_panel_get_widget(panel, "news-login");
  username = etpan_storage_news_get_username(storage);
  if (username != NULL)
    gtk_entry_set_text(GTK_ENTRY(entry), username);
  
  entry = etpan_preferences_panel_get_widget(panel, "news-password");
  password = etpan_storage_news_get_password(storage);
  if (password != NULL)
    gtk_entry_set_text(GTK_ENTRY(entry), password);
  
  connection_type = etpan_storage_news_get_connection_type(storage);
  checkbox = etpan_preferences_panel_get_widget(panel, "news-use-ssl");
  use_ssl_value = (connection_type == ETPAN_CONNECTION_TLS);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkbox), use_ssl_value);
  news_ssl_toggled(GTK_TOGGLE_BUTTON(checkbox), panel);
  
  entry = etpan_preferences_panel_get_widget(panel, "news-port");
  port = etpan_storage_news_get_port(storage);
  snprintf(port_str, sizeof(port_str), "%i", port);
  gtk_entry_set_text(GTK_ENTRY(entry), port_str);
  
  combo = etpan_preferences_panel_get_widget(panel, "news-auth-type");
  if ((username != NULL) || (password != NULL))
    value = 1;
  else
    value = 0;
  gtk_combo_box_set_active(GTK_COMBO_BOX(combo), value);
  news_auth_type_changed(GTK_COMBO_BOX(combo), panel);
}

static void maildir_account_set(struct etpan_preferences_panel * panel,
    struct etpan_account * account)
{
  GtkWidget * label;
  GtkWidget * combo;
  GtkWidget * chooser;
  struct etpan_storage * storage;
  char * path;
  struct panel_data * data;
  
  data = panel->data;
  storage = etpan_account_get_storage(account);
  
  combo = etpan_preferences_panel_get_widget(panel, "local-type");
  gtk_combo_box_set_active(GTK_COMBO_BOX(combo), 0);
  gtk_widget_hide(combo);
  
  label = etpan_preferences_panel_get_widget(panel, "local-type-value");
  gtk_label_set_text(GTK_LABEL(label), _("Maildir"));
  gtk_widget_show(label);
  
  path = etpan_storage_maildir_get_path(storage);
  local_path_update(panel, path);
}

static void mh_account_set(struct etpan_preferences_panel * panel,
    struct etpan_account * account)
{
  GtkWidget * label;
  GtkWidget * combo;
  GtkWidget * chooser;
  struct etpan_storage * storage;
  char * path;
  struct panel_data * data;
  
  data = panel->data;
  storage = etpan_account_get_storage(account);
  
  combo = etpan_preferences_panel_get_widget(panel, "local-type");
  gtk_combo_box_set_active(GTK_COMBO_BOX(combo), 1);
  gtk_widget_hide(combo);
  
  label = etpan_preferences_panel_get_widget(panel, "local-type-value");
  gtk_label_set_text(GTK_LABEL(label), _("MH"));
  gtk_widget_show(label);
  
  path = etpan_storage_mh_get_path(storage);
  local_path_update(panel, path);
}

static void mbox_account_set(struct etpan_preferences_panel * panel,
    struct etpan_account * account)
{
  GtkWidget * label;
  GtkWidget * combo;
  GtkWidget * chooser;
  struct etpan_storage * storage;
  char * path;
  struct panel_data * data;
  
  data = panel->data;
  storage = etpan_account_get_storage(account);
  
  combo = etpan_preferences_panel_get_widget(panel, "local-type");
  gtk_combo_box_set_active(GTK_COMBO_BOX(combo), 2);
  gtk_widget_hide(combo);
  
  label = etpan_preferences_panel_get_widget(panel, "local-type-value");
  gtk_label_set_text(GTK_LABEL(label), _("mbox"));
  gtk_widget_show(label);
  
  path = etpan_storage_mbox_get_path(storage);
  local_path_update(panel, path);
}

static void smtp_account_set(struct etpan_preferences_panel * panel,
    struct etpan_account * account)
{
  carray * outbox_list;
  unsigned int i;
  struct etpan_sender * sender;
  struct etpan_outbox * outbox;
  
  outbox_list = etpan_account_get_outbox_list(account);
  ETPAN_LOG("outbox list : %p %p", account, outbox_list);
  for(i = 0 ; i < carray_count(outbox_list) ; i ++) {
    struct etpan_sender_driver * driver;
    GtkWidget * entry;
    const char * hostname;
    char port_str[20];
    int port;
    GtkWidget * checkbox;
    GtkWidget * combo;
    int auth_type;
    const char * login;
    const char * password;
    gboolean use_ssl_value;
    int connection_type;
    gint value;
    
    outbox = carray_get(outbox_list, i);
    
    sender = etpan_outbox_get_sender(outbox);
    if (sender == NULL) {
      ETPAN_LOG("outbox has no sender");
      etpan_crash();
    }
    
    driver = etpan_sender_get_driver(sender);
    if (driver == NULL) {
      ETPAN_LOG("sender has no driver");
      etpan_crash();
    }
    
    if (strcasecmp(driver->name, "smtp") != 0) {
      continue;
    }
    
    entry = etpan_preferences_panel_get_widget(panel, "smtp-server");
    hostname = etpan_sender_smtp_get_hostname(sender);
    if (hostname != NULL)
      gtk_entry_set_text(GTK_ENTRY(entry), hostname);
    
    entry = etpan_preferences_panel_get_widget(panel, "smtp-port");
    port = etpan_sender_smtp_get_port(sender);
    snprintf(port_str, sizeof(port_str), "%i", port);
    gtk_entry_set_text(GTK_ENTRY(entry), port_str);
    
    connection_type = etpan_sender_smtp_get_connection_type(sender);
    checkbox = etpan_preferences_panel_get_widget(panel, "smtp-use-ssl");
    use_ssl_value = (connection_type == ETPAN_CONNECTION_TLS);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkbox), use_ssl_value);
    imap_ssl_toggled(GTK_TOGGLE_BUTTON(checkbox), panel);
    
    combo = etpan_preferences_panel_get_widget(panel, "smtp-auth-type");
    auth_type = etpan_sender_smtp_get_auth_type(sender);
    switch (auth_type) {
    case ETPAN_AUTH_PLAIN:
      value = 1;
      break;
    case ETPAN_AUTH_CRAM_MD5:
      value = 2;
      break;
    case ETPAN_AUTH_DIGEST_MD5:
      value = 3;
      break;
    case ETPAN_AUTH_GSSAPI:
      value = 4;
      break;
    case ETPAN_AUTH_KERBEROS4:
      value = 5;
      break;
    case ETPAN_AUTH_OTP:
      value = 6;
      break;
    case ETPAN_AUTH_PASSDSS_3DES_1:
      value = 7;
      break;
    case ETPAN_AUTH_SRP:
      value = 8;
      break;
    case ETPAN_AUTH_NTLM:
      value = 9;
      break;
    case ETPAN_AUTH_ANONYMOUS:
      value = 10;
      break;
    default: /* ETPAN_AUTH_NONE */
      value = 0;
      break;
    }
    
    gtk_combo_box_set_active(GTK_COMBO_BOX(combo), value);
    smtp_auth_type_changed(GTK_COMBO_BOX(combo), panel);
    
    entry = etpan_preferences_panel_get_widget(panel, "smtp-login");
    login = etpan_sender_smtp_get_username(sender);
    if (login != NULL)
      gtk_entry_set_text(GTK_ENTRY(entry), login);
    
    entry = etpan_preferences_panel_get_widget(panel, "smtp-password");
    password = etpan_sender_smtp_get_password(sender);
    if (password != NULL)
      gtk_entry_set_text(GTK_ENTRY(entry), password);
  }
}


static void account_set(struct etpan_preferences_panel * panel,
    struct etpan_account * account)
{
  char * name;
  GtkWidget * label;
  GtkWidget * combo;
  GtkWidget * button;
  int type;
  GtkWidget * entry;
  char * display_name;
  char * mail;
  struct panel_data * data;
  
  ETPAN_LOG("accout set %p", account);
  
  data = panel->data;
  reset_properties(panel);
  
  name = etpan_account_get_id(account);
  
  entry = etpan_preferences_panel_get_widget(panel, "account-name");
  if (name != NULL)
    gtk_entry_set_text(GTK_ENTRY(entry), name);
  
  entry = etpan_preferences_panel_get_widget(panel, "display-name");
  display_name = etpan_account_get_display_name(account);
  if (display_name != NULL)
    gtk_entry_set_text(GTK_ENTRY(entry), display_name);
  
  entry = etpan_preferences_panel_get_widget(panel, "address");
  mail = etpan_account_get_mail(account);
  if (mail != NULL)
    gtk_entry_set_text(GTK_ENTRY(entry), mail);
  
  smtp_account_set(panel, account);
  
  type = get_account_type(account);
  
  label = etpan_preferences_panel_get_widget(panel, "account-type-label");
  gtk_label_set_text(GTK_LABEL(label), get_account_type_str(type));
  gtk_widget_show(label);
  
  combo = etpan_preferences_panel_get_widget(panel, "account-type");
  switch (type) {
  case STORAGE_DRIVER_TYPE_POP:
  case STORAGE_DRIVER_TYPE_IMAP:
  case STORAGE_DRIVER_TYPE_NEWS:
    gtk_combo_box_set_active(GTK_COMBO_BOX(combo), type);
    break;
  case STORAGE_DRIVER_TYPE_MAILDIR:
  case STORAGE_DRIVER_TYPE_MH:
  case STORAGE_DRIVER_TYPE_MBOX:
  default:
    gtk_combo_box_set_active(GTK_COMBO_BOX(combo), 3);
    break;
  }
  gtk_widget_hide(combo);
  
  button = etpan_preferences_panel_get_widget(panel, "create-button");
  gtk_widget_hide(button);
  
  button = etpan_preferences_panel_get_widget(panel, "cancel-button");
  gtk_widget_hide(button);
  
  button = etpan_preferences_panel_get_widget(panel, "revert-button");
  gtk_widget_show(button);
  
  button = etpan_preferences_panel_get_widget(panel, "apply-button");
  gtk_widget_show(button);
  
  switch (type) {
  case STORAGE_DRIVER_TYPE_POP:
    pop_account_set(panel, account);
    break;
  case STORAGE_DRIVER_TYPE_IMAP:
    imap_account_set(panel, account);
    break;
  case STORAGE_DRIVER_TYPE_NEWS:
    news_account_set(panel, account);
    break;
  case STORAGE_DRIVER_TYPE_MAILDIR:
    maildir_account_set(panel, account);
    break;
  case STORAGE_DRIVER_TYPE_MH:
    mh_account_set(panel, account);
    break;
  case STORAGE_DRIVER_TYPE_MBOX:
    mbox_account_set(panel, account);
    break;
  }
  
  data->account_being_modified = account;
  
  data->modified = 0;
  enable_buttons(panel);
  check_consistency(panel);
}



/* *********************************** */
/* set UI state */

static void enable_buttons(struct etpan_preferences_panel * panel)
{
  GtkWidget * button;
  gboolean enabled;
  struct panel_data * data;
  
  data = panel->data;
  
  if (is_empty(panel) && (data->pending_account != NULL)) {
    data->modified = 0;
  }
  
  enabled = data->modified;
  
  button = etpan_preferences_panel_get_widget(panel, "create-button");
  gtk_widget_set_sensitive(button, enabled);

  button = etpan_preferences_panel_get_widget(panel, "cancel-button");
  gtk_widget_set_sensitive(button, TRUE);
  
  button = etpan_preferences_panel_get_widget(panel, "revert-button");
  gtk_widget_set_sensitive(button, enabled);
  
  button = etpan_preferences_panel_get_widget(panel, "apply-button");
  gtk_widget_set_sensitive(button, enabled);
}

static void set_add_remove_state(struct etpan_preferences_panel * panel)
{
  GtkWidget * button;
  carray * list;
  struct panel_data * data;
  
  data = panel->data;
  
  list = etpan_account_manager_get_ordered_list(etpan_account_manager_get_default());
  if (carray_count(list) > 0) {
    button = etpan_preferences_panel_get_widget(panel, "add-button");
    if (data->pending_account != NULL)
      gtk_widget_set_sensitive(button, FALSE);
    else
      gtk_widget_set_sensitive(button, TRUE);
    
    button = etpan_preferences_panel_get_widget(panel, "remove-button");
    gtk_widget_set_sensitive(button, TRUE);
  }
  else {
    button = etpan_preferences_panel_get_widget(panel, "add-button");
    gtk_widget_set_sensitive(button, FALSE);
    
    button = etpan_preferences_panel_get_widget(panel, "remove-button");
    gtk_widget_set_sensitive(button, FALSE);
  }
}

/* *********************************** */
/* add */

static void add_account(struct etpan_preferences_panel * panel)
{
  struct etpan_account * account;
  struct panel_data * data;
  GtkTreePath * path;
  GtkWidget * treeview;
  GtkTreeIter iter;
  
  data = panel->data;
  
  data->account_set = 0;
  account = etpan_account_new();
  data->pending_account = account;
  etpan_gtk_tree_model_reload(data->treemodel);
  set_add_remove_state(panel);
  
  etpan_gtk_tree_model_get_iter_from_item(data->treemodel, &iter, account);
  
  treeview = etpan_preferences_panel_get_widget(panel, "account-list");
  path = gtk_tree_model_get_path(GTK_TREE_MODEL(data->treemodel), &iter);
  gtk_tree_view_set_cursor(GTK_TREE_VIEW(treeview), path, NULL, FALSE);
  gtk_tree_path_free(path);
  
  data->modified = 0;
  reset_properties(panel);
  check_consistency(panel);
}

static void stop_add(struct etpan_preferences_panel * panel)
{
  struct panel_data * data;
  
  data = panel->data;
  
  etpan_account_free(data->pending_account);
  data->pending_account = NULL;
  etpan_gtk_tree_model_reload(data->treemodel);
  set_add_remove_state(panel);
}

static void create_account(struct etpan_preferences_panel * panel,
    int keep_selection)
{
  GtkWidget * combo;
  gint index;
  struct panel_data * data;
  GtkTreeIter iter;
  carray * list;
  struct etpan_account * account;
  GtkTreePath * path;
  GtkWidget * treeview;
  struct etpan_account_manager * manager;
  struct etpan_error * error;
  
  if (!check_consistency(panel))
    return;
  
  data = panel->data;
  
  combo = etpan_preferences_panel_get_widget(panel, "account-type");
  
  index = gtk_combo_box_get_active(GTK_COMBO_BOX(combo));
  switch (index) {
  case 0: /* POP */
    create_pop_account(panel);
    break;
  case 1: /* IMAP */
    create_imap_account(panel);
    break;
  case 2: /* News */
    create_news_account(panel);
    break;
  case 3: /* Local */
    create_local_account(panel);
    break;
  }

  etpan_account_manager_notify_modification(etpan_account_manager_get_default());
  
  manager = etpan_account_manager_get_default();
  
  list = etpan_account_manager_get_ordered_list(manager);
  account = carray_get(list, carray_count(list) - 1);
  
  if (data->pending_account != NULL) {
    stop_add(panel);
  }
  else {
    etpan_gtk_tree_model_reload(data->treemodel);
    set_add_remove_state(panel);
  }
  
  etpan_gtk_tree_model_get_iter_from_item(data->treemodel, &iter, account);
  
  if (keep_selection) {
    treeview = etpan_preferences_panel_get_widget(panel, "account-list");
    path = gtk_tree_model_get_path(GTK_TREE_MODEL(data->treemodel), &iter);
    gtk_tree_view_set_cursor(GTK_TREE_VIEW(treeview), path, NULL, FALSE);
    gtk_tree_path_free(path);
  }
  
  error = etpan_account_config_save_default();
  ETPAN_ERROR_IGNORE(error);
}

static void create_imap_account(struct etpan_preferences_panel * panel)
{
  struct etpan_account_manager * manager;
  struct etpan_storage * storage;
  struct etpan_account * account;
  
  manager = etpan_account_manager_get_default();
  
  account = etpan_account_new();
  storage = etpan_storage_imap_sync_new();
  etpan_account_set_storage(account, storage);
  
  modify_imap_account(panel, account);
}

static void create_pop_account(struct etpan_preferences_panel * panel)
{
  struct etpan_account_manager * manager;
  struct etpan_storage * storage;
  struct etpan_account * account;
  
  manager = etpan_account_manager_get_default();
  
  account = etpan_account_new();
  storage = etpan_storage_pop_sync_new();
  etpan_account_set_storage(account, storage);
  
  modify_pop_account(panel, account);
}

static void create_news_account(struct etpan_preferences_panel * panel)
{
  struct etpan_account_manager * manager;
  struct etpan_storage * storage;
  struct etpan_account * account;
  
  manager = etpan_account_manager_get_default();
  
  account = etpan_account_new();
  storage = etpan_storage_news_new();
  etpan_account_set_storage(account, storage);
  
  modify_news_account(panel, account);
}

static void create_mbox_account(struct etpan_preferences_panel * panel)
{
  struct etpan_account_manager * manager;
  struct etpan_storage * storage;
  struct etpan_account * account;
  
  manager = etpan_account_manager_get_default();
  
  account = etpan_account_new();
  storage = etpan_storage_mbox_new();
  etpan_account_set_storage(account, storage);
  
  modify_mbox_account(panel, account);
}

static void create_maildir_account(struct etpan_preferences_panel * panel)
{
  struct etpan_account_manager * manager;
  struct etpan_storage * storage;
  struct etpan_account * account;
  
  manager = etpan_account_manager_get_default();
  
  account = etpan_account_new();
  storage = etpan_storage_maildir_new();
  etpan_account_set_storage(account, storage);
  
  modify_maildir_account(panel, account);
}

static void create_mh_account(struct etpan_preferences_panel * panel)
{
  struct etpan_account_manager * manager;
  struct etpan_storage * storage;
  struct etpan_account * account;
  
  manager = etpan_account_manager_get_default();
  
  account = etpan_account_new();
  storage = etpan_storage_mh_new();
  etpan_account_set_storage(account, storage);
  
  modify_mh_account(panel, account);
}

static void create_local_account(struct etpan_preferences_panel * panel)
{
  gint index;
  GtkWidget * combo;
  
  combo = etpan_preferences_panel_get_widget(panel, "local-type");
  index = gtk_combo_box_get_active(GTK_COMBO_BOX(combo));
  
  switch (index) {
  case 0:
    create_maildir_account(panel);
    break;
  case 1:
    create_mh_account(panel);
    break;
  case 2:
    create_mbox_account(panel);
    break;
  }
}

/* *********************************** */
/* modify */

static void modify_smtp_account(struct etpan_preferences_panel * panel,
    struct etpan_account * account)
{
  carray * outbox_list;
  unsigned int i;
  char * outbox_path;
  struct etpan_sender * sender;
  struct etpan_outbox * outbox;
  struct etpan_storage * storage;
  struct etpan_storage_driver * storage_driver;
  
  storage = etpan_account_get_storage(account);
  storage_driver = etpan_storage_get_driver(storage);
  
  /* create basic outbox */
  
  sender = etpan_sender_smtp_new();
  outbox = etpan_outbox_new();
  outbox_path = etpan_get_outbox_dir(etpan_account_manager_get_default(),
      account);
  etpan_outbox_set_path(outbox, outbox_path);
  free(outbox_path);
  
  etpan_outbox_set_sender(outbox, sender);
  
  etpan_account_add_outbox(account, outbox);
  
  /* sent folder outbox */
  
  if (strcasecmp(storage_driver->name, "news") != 0) {
    /* XXX - currently, sent folder can't be added for news account */
    sender = etpan_sender_sent_folder_new();
    etpan_sender_sent_folder_set_storage(sender, storage);
    outbox = etpan_outbox_new();
    outbox_path = etpan_get_outbox_sent_folder_dir(etpan_account_manager_get_default(),
        account);
    etpan_outbox_set_path(outbox, outbox_path);
    free(outbox_path);
    
    etpan_outbox_set_sender(outbox, sender);
    
    etpan_account_add_outbox(account, outbox);
  }
  
  if (strcasecmp(storage_driver->name, "news") == 0) {
    /* news, add a second outbox */
    
    sender = etpan_sender_news_new();
    etpan_sender_news_set_storage(sender, storage);
    
    outbox = etpan_outbox_new();
    outbox_path = etpan_get_outbox_news_dir(etpan_account_manager_get_default(),
        account);
    etpan_outbox_set_path(outbox, outbox_path);
    free(outbox_path);
    
    etpan_outbox_set_sender(outbox, sender);
    
    etpan_account_add_outbox(account, outbox);
  }
  
  outbox_list = etpan_account_get_outbox_list(account);
  for(i = 0 ; i < carray_count(outbox_list) ; i ++) {
    struct etpan_sender_driver * driver;
    GtkWidget * entry;
    const char * hostname;
    const char * port_str;
    int port;
    GtkWidget * checkbox;
    GtkWidget * combo;
    gint value_auth_type;
    int auth_type;
    const char * login;
    const char * password;
    gboolean use_ssl;
    
    outbox = carray_get(outbox_list, i);
    
    sender = etpan_outbox_get_sender(outbox);
    if (sender == NULL) {
      ETPAN_LOG("outbox has no sender");
      etpan_crash();
    }
    
    driver = etpan_sender_get_driver(sender);
    if (driver == NULL) {
      ETPAN_LOG("sender has no driver");
      etpan_crash();
    }
    
    if (strcasecmp(driver->name, "smtp") != 0) {
      continue;
    }
    
    entry = etpan_preferences_panel_get_widget(panel, "smtp-server");
    hostname = gtk_entry_get_text(GTK_ENTRY(entry));
    if (* hostname != 0)
      etpan_sender_smtp_set_hostname(sender, (char *) hostname);
    
    entry = etpan_preferences_panel_get_widget(panel, "smtp-port");
    port_str = gtk_entry_get_text(GTK_ENTRY(entry));
    port = strtol(port_str, NULL, 10);
    etpan_sender_smtp_set_port(sender, port);
    
    checkbox = etpan_preferences_panel_get_widget(panel, "smtp-use-ssl");
    use_ssl = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkbox));
    if (use_ssl)
      etpan_sender_smtp_set_connection_type(sender, ETPAN_CONNECTION_TLS);
    else
      etpan_sender_smtp_set_connection_type(sender, ETPAN_CONNECTION_PLAIN);
    
    combo = etpan_preferences_panel_get_widget(panel, "smtp-auth-type");
    value_auth_type = gtk_combo_box_get_active(GTK_COMBO_BOX(combo));
    
    auth_type = ETPAN_AUTH_NONE;
    switch (value_auth_type) {
    case 0:
      auth_type = ETPAN_AUTH_NONE;
      break;
    case 1:
      auth_type = ETPAN_AUTH_PLAIN;
      break;
    case 2:
      auth_type = ETPAN_AUTH_CRAM_MD5;
      break;
    case 3:
      auth_type = ETPAN_AUTH_DIGEST_MD5;
      break;
    case 4:
      auth_type = ETPAN_AUTH_GSSAPI;
      break;
    case 5:
      auth_type = ETPAN_AUTH_KERBEROS4;
      break;
    case 6:
      auth_type = ETPAN_AUTH_OTP;
      break;
    case 7:
      auth_type = ETPAN_AUTH_PASSDSS_3DES_1;
      break;
    case 8:
      auth_type = ETPAN_AUTH_SRP;
      break;
    case 9:
      auth_type = ETPAN_AUTH_NTLM;
      break;
    case 10:
      auth_type = ETPAN_AUTH_ANONYMOUS;
      break;
    }
    
    etpan_sender_smtp_set_auth_type(sender, auth_type);

    entry = etpan_preferences_panel_get_widget(panel, "smtp-login");
    login = gtk_entry_get_text(GTK_ENTRY(entry));
    if (* login != 0)
      etpan_sender_smtp_set_username(sender, (char *) login);
    
    entry = etpan_preferences_panel_get_widget(panel, "smtp-password");
    password = gtk_entry_get_text(GTK_ENTRY(entry));
    if (* password != 0)
      etpan_sender_smtp_set_password(sender, (char *) password);
  }
}

static void modify_account_common(struct etpan_preferences_panel * panel,
    struct etpan_account * account)
{
  const char * name;
  const char * display_name;
  const char * address;
  GtkWidget * entry;
  
  entry = etpan_preferences_panel_get_widget(panel, "account-name");
  name = gtk_entry_get_text(GTK_ENTRY(entry));
  if (* name != 0)
    etpan_account_set_id(account, (char *) name);
  
  entry = etpan_preferences_panel_get_widget(panel, "display-name");
  display_name = gtk_entry_get_text(GTK_ENTRY(entry));
  if (* display_name != 0)
    etpan_account_set_display_name(account, (char *) display_name);
  
  entry = etpan_preferences_panel_get_widget(panel, "address");
  address = gtk_entry_get_text(GTK_ENTRY(entry));
  if (* address != 0)
    etpan_account_set_mail(account, (char *) address);
  
  modify_smtp_account(panel, account);
}

static void modify_imap_account(struct etpan_preferences_panel * panel,
    struct etpan_account * account)
{
  struct etpan_storage * storage;
  char * cache_path;
  struct etpan_account_manager * manager;
  GtkWidget * entry;
  GtkWidget * combo;
  GtkWidget * checkbox;
  const char * server;
  gboolean use_ssl;
  const char * port_str;
  int port;
  const char * username;
  const char * password;
  gint value_auth_type;
  int auth_type;
  const char * prefix;
  const char * address;
  struct etpan_error * error;
  
  storage = etpan_account_get_storage(account);
  
  manager = etpan_account_manager_get_default();
  
  modify_account_common(panel, account);

  entry = etpan_preferences_panel_get_widget(panel, "imap-server");
  server = gtk_entry_get_text(GTK_ENTRY(entry));
  if (* server != 0)
    etpan_storage_imap_sync_set_hostname(storage, (char *) server);
  
  combo = etpan_preferences_panel_get_widget(panel, "imap-auth-type");
  value_auth_type = gtk_combo_box_get_active(GTK_COMBO_BOX(combo));
  
  auth_type = ETPAN_AUTH_PASSWORD;
  switch (value_auth_type) {
  case 0:
    auth_type = ETPAN_AUTH_PASSWORD;
    break;
  case 1:
    auth_type = ETPAN_AUTH_PLAIN;
    break;
  case 2:
    auth_type = ETPAN_AUTH_CRAM_MD5;
    break;
  case 3:
    auth_type = ETPAN_AUTH_DIGEST_MD5;
    break;
  case 4:
    auth_type = ETPAN_AUTH_GSSAPI;
    break;
  case 5:
    auth_type = ETPAN_AUTH_KERBEROS4;
    break;
  case 6:
    auth_type = ETPAN_AUTH_OTP;
    break;
  case 7:
    auth_type = ETPAN_AUTH_PASSDSS_3DES_1;
    break;
  case 8:
    auth_type = ETPAN_AUTH_SRP;
    break;
  case 9:
    auth_type = ETPAN_AUTH_NTLM;
    break;
  case 10:
    auth_type = ETPAN_AUTH_ANONYMOUS;
    break;
  }
  etpan_storage_imap_sync_set_auth_type(storage, auth_type);
  
  checkbox = etpan_preferences_panel_get_widget(panel, "imap-use-ssl");
  use_ssl = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkbox));
  if (use_ssl)
    etpan_storage_imap_sync_set_connection_type(storage, ETPAN_CONNECTION_TLS);
  else
    etpan_storage_imap_sync_set_connection_type(storage, ETPAN_CONNECTION_PLAIN);
  
  entry = etpan_preferences_panel_get_widget(panel, "imap-port");
  port_str = gtk_entry_get_text(GTK_ENTRY(entry));
  port = strtoul(port_str, NULL, 10);
  etpan_storage_imap_sync_set_port(storage, port);
  
  entry = etpan_preferences_panel_get_widget(panel, "imap-login");
  username = gtk_entry_get_text(GTK_ENTRY(entry));
  if (username != 0)
    etpan_storage_imap_sync_set_username(storage, (char *) username);
  if (auth_type == ETPAN_AUTH_ANONYMOUS) {
    entry = etpan_preferences_panel_get_widget(panel, "address");
    address = gtk_entry_get_text(GTK_ENTRY(entry));
    if (* address != 0)
      etpan_storage_imap_sync_set_username(storage, (char *) address);
  }
  
  entry = etpan_preferences_panel_get_widget(panel, "imap-password");
  password = gtk_entry_get_text(GTK_ENTRY(entry));
  if (* password != 0)
    etpan_storage_imap_sync_set_password(storage, (char *) password);
  
  entry = etpan_preferences_panel_get_widget(panel, "imap-prefix");
  prefix = gtk_entry_get_text(GTK_ENTRY(entry));
  if (* prefix != 0)
    etpan_storage_imap_sync_set_base_location(storage, (char *) prefix);
  
  cache_path = etpan_get_cache_dir(manager, account);
  etpan_make_dir(cache_path);
  etpan_storage_set_cache_path(storage, cache_path);
  free(cache_path);
  
  error = etpan_account_setup(account);
  ETPAN_ERROR_IGNORE(error);
  etpan_account_manager_add_account(manager, account);
}

static void modify_pop_account(struct etpan_preferences_panel * panel,
    struct etpan_account * account)
{
  struct etpan_storage * storage;
  char * cache_path;
  char * flags_path;
  struct etpan_account_manager * manager;
  GtkWidget * entry;
  GtkWidget * combo;
  GtkWidget * checkbox;
  GtkWidget * chooser;
  const char * address;
  const char * server;
  gboolean use_ssl;
  const char * port_str;
  int port;
  const char * username;
  const char * password;
  gint value_auth_type;
  int auth_type;
  struct etpan_error * error;
  char * path;
  struct panel_data * data;
  
  data = panel->data;
  
  storage = etpan_account_get_storage(account);
  
  manager = etpan_account_manager_get_default();
  
  modify_account_common(panel, account);
  
  entry = etpan_preferences_panel_get_widget(panel, "pop-server");
  server = gtk_entry_get_text(GTK_ENTRY(entry));
  if (* server != 0)
    etpan_storage_pop_sync_set_hostname(storage, (char *) server);
  
  combo = etpan_preferences_panel_get_widget(panel, "pop-auth-type");
  value_auth_type = gtk_combo_box_get_active(GTK_COMBO_BOX(combo));
  
  auth_type = ETPAN_AUTH_PASSWORD;
  switch (value_auth_type) {
  case 0:
    auth_type = ETPAN_AUTH_PASSWORD;
    break;
  case 1:
    auth_type = ETPAN_AUTH_PLAIN;
    break;
  case 2:
    auth_type = ETPAN_AUTH_CRAM_MD5;
    break;
  case 3:
    auth_type = ETPAN_AUTH_DIGEST_MD5;
    break;
  case 4:
    auth_type = ETPAN_AUTH_GSSAPI;
    break;
  case 5:
    auth_type = ETPAN_AUTH_KERBEROS4;
    break;
  case 6:
    auth_type = ETPAN_AUTH_OTP;
    break;
  case 7:
    auth_type = ETPAN_AUTH_PASSDSS_3DES_1;
    break;
  case 8:
    auth_type = ETPAN_AUTH_SRP;
    break;
  case 9:
    auth_type = ETPAN_AUTH_NTLM;
    break;
  case 10:
    auth_type = ETPAN_AUTH_ANONYMOUS;
    break;
  }
  etpan_storage_pop_sync_set_auth_type(storage, auth_type);
  
  checkbox = etpan_preferences_panel_get_widget(panel, "pop-use-ssl");
  use_ssl = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkbox));
  if (use_ssl)
    etpan_storage_pop_sync_set_connection_type(storage, ETPAN_CONNECTION_TLS);
  else
    etpan_storage_pop_sync_set_connection_type(storage, ETPAN_CONNECTION_PLAIN);
  
  entry = etpan_preferences_panel_get_widget(panel, "pop-port");
  port_str = gtk_entry_get_text(GTK_ENTRY(entry));
  port = strtoul(port_str, NULL, 10);
  etpan_storage_pop_sync_set_port(storage, port);
  
  entry = etpan_preferences_panel_get_widget(panel, "pop-login");
  username = gtk_entry_get_text(GTK_ENTRY(entry));
  if (* username != 0)
    etpan_storage_pop_sync_set_username(storage, (char *) username);
  if (auth_type == ETPAN_AUTH_ANONYMOUS) {
    entry = etpan_preferences_panel_get_widget(panel, "address");
    address = gtk_entry_get_text(GTK_ENTRY(entry));
    if (* address != 0)
      etpan_storage_pop_sync_set_username(storage, (char *) address);
  }
  
  entry = etpan_preferences_panel_get_widget(panel, "pop-password");
  password = gtk_entry_get_text(GTK_ENTRY(entry));
  if (* password != 0)
    etpan_storage_pop_sync_set_password(storage, (char *) password);

  path = data->pop_path;
  if (path == NULL) {
    char path_buffer[PATH_MAX];
    char * home;
    
    home = getenv("HOME");
    ETPAN_ASSERT(home != NULL, "home not defined");
    snprintf(path_buffer, sizeof(path_buffer), "%s/Maildir-%s", home,
        etpan_account_get_id(account));
    etpan_storage_pop_sync_set_path(storage, path_buffer);
  }
  else {
    etpan_storage_pop_sync_set_path(storage, path);
  }
  
  cache_path = etpan_get_cache_dir(manager, account);
  etpan_make_dir(cache_path);
  etpan_storage_set_cache_path(storage, cache_path);
  free(cache_path);
  
  flags_path = etpan_get_flags_dir(manager, account);
  etpan_make_dir(flags_path);
  etpan_storage_set_flags_path(storage, flags_path);
  free(flags_path);
  
  error = etpan_account_setup(account);
  ETPAN_ERROR_IGNORE(error);
  etpan_account_manager_add_account(manager, account);
}

static void modify_news_account(struct etpan_preferences_panel * panel,
    struct etpan_account * account)
{
  struct etpan_storage * storage;
  char * cache_path;
  char * flags_path;
  struct etpan_account_manager * manager;
  GtkWidget * entry;
  GtkWidget * combo;
  GtkWidget * checkbox;
  const char * server;
  gboolean use_ssl;
  const char * port_str;
  int port;
  const char * username;
  const char * password;
  gint value_auth_type;
  int auth_type;
  struct etpan_error * error;
  
  storage = etpan_account_get_storage(account);
  
  manager = etpan_account_manager_get_default();
  
  modify_account_common(panel, account);
  
  entry = etpan_preferences_panel_get_widget(panel, "news-server");
  server = gtk_entry_get_text(GTK_ENTRY(entry));
  if (* server != 0)
    etpan_storage_news_set_hostname(storage, (char *) server);
  
  combo = etpan_preferences_panel_get_widget(panel, "news-auth-type");
  value_auth_type = gtk_combo_box_get_active(GTK_COMBO_BOX(combo));
  
  auth_type = ETPAN_AUTH_NONE;
  switch (value_auth_type) {
  case 0:
    auth_type = ETPAN_AUTH_NONE;
    break;
  case 1:
    auth_type = ETPAN_AUTH_PASSWORD;
    break;
  }
  
  checkbox = etpan_preferences_panel_get_widget(panel, "news-use-ssl");
  use_ssl = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkbox));
  if (use_ssl)
    etpan_storage_news_set_connection_type(storage, ETPAN_CONNECTION_TLS);
  else
    etpan_storage_news_set_connection_type(storage, ETPAN_CONNECTION_PLAIN);
  
  entry = etpan_preferences_panel_get_widget(panel, "news-port");
  port_str = gtk_entry_get_text(GTK_ENTRY(entry));
  port = strtoul(port_str, NULL, 10);
  etpan_storage_news_set_port(storage, port);
  
  if (auth_type != ETPAN_AUTH_NONE) {
    entry = etpan_preferences_panel_get_widget(panel, "news-login");
    username = gtk_entry_get_text(GTK_ENTRY(entry));
    if (* username != 0)
      etpan_storage_news_set_username(storage, (char *) username);
    
    entry = etpan_preferences_panel_get_widget(panel, "news-password");
    password = gtk_entry_get_text(GTK_ENTRY(entry));
    if (* password != 0)
      etpan_storage_news_set_password(storage, (char *) password);
  }
  
  cache_path = etpan_get_cache_dir(manager, account);
  etpan_make_dir(cache_path);
  etpan_storage_set_cache_path(storage, cache_path);
  free(cache_path);

  flags_path = etpan_get_flags_dir(manager, account);
  etpan_make_dir(flags_path);
  etpan_storage_set_flags_path(storage, flags_path);
  free(flags_path);
  
  error = etpan_account_setup(account);
  ETPAN_ERROR_IGNORE(error);
  etpan_account_manager_add_account(manager, account);
}

static void modify_mbox_account(struct etpan_preferences_panel * panel,
    struct etpan_account * account)
{
  GtkWidget * chooser;
  char * path;
  char * cache_path;
  char * flags_path;
  struct etpan_storage * storage;
  struct etpan_account_manager * manager;
  struct etpan_error * error;
  struct panel_data * data;
  
  data = panel->data;
  storage = etpan_account_get_storage(account);
  
  manager = etpan_account_manager_get_default();
  
  modify_account_common(panel, account);
  
  path = data->local_path;
  if (path == NULL) {
    char path_buffer[PATH_MAX];
    char * home;
    
    home = getenv("HOME");
    ETPAN_ASSERT(home != NULL, "home not defined");
    snprintf(path_buffer, sizeof(path_buffer), "%s/mail-%s", home,
        etpan_account_get_id(account));
    etpan_storage_mbox_set_path(storage, path_buffer);
  }
  else {
    etpan_storage_mbox_set_path(storage, path);
  }

  cache_path = etpan_get_cache_dir(manager, account);
  etpan_make_dir(cache_path);
  etpan_storage_set_cache_path(storage, cache_path);
  free(cache_path);
  
  flags_path = etpan_get_flags_dir(manager, account);
  etpan_make_dir(flags_path);
  etpan_storage_set_flags_path(storage, flags_path);
  free(flags_path);
  
  error = etpan_account_setup(account);
  ETPAN_ERROR_IGNORE(error);
  etpan_account_manager_add_account(manager, account);
}

static void modify_maildir_account(struct etpan_preferences_panel * panel,
    struct etpan_account * account)
{
  GtkWidget * chooser;
  char * path;
  char * cache_path;
  char * flags_path;
  struct etpan_storage * storage;
  struct etpan_account_manager * manager;
  struct etpan_error * error;
  struct panel_data * data;
  
  data = panel->data;
  storage = etpan_account_get_storage(account);
  
  manager = etpan_account_manager_get_default();
  
  modify_account_common(panel, account);
  
  path = data->local_path;
  if (path == NULL) {
    char path_buffer[PATH_MAX];
    char * home;
    
    home = getenv("HOME");
    ETPAN_ASSERT(home != NULL, "home not defined");
    snprintf(path_buffer, sizeof(path_buffer), "%s/Maildir-%s", home,
        etpan_account_get_id(account));
    etpan_storage_maildir_set_path(storage, path_buffer);
  }
  else {
    etpan_storage_maildir_set_path(storage, path);
  }
  
  cache_path = etpan_get_cache_dir(manager, account);
  etpan_make_dir(cache_path);
  etpan_storage_set_cache_path(storage, cache_path);
  free(cache_path);
  
  flags_path = etpan_get_flags_dir(manager, account);
  etpan_make_dir(flags_path);
  etpan_storage_set_flags_path(storage, flags_path);
  free(flags_path);
  
  error = etpan_account_setup(account);
  ETPAN_ERROR_IGNORE(error);
  etpan_account_manager_add_account(manager, account);
}

static void modify_mh_account(struct etpan_preferences_panel * panel,
    struct etpan_account * account)
{
  GtkWidget * chooser;
  char * path;
  char * cache_path;
  char * flags_path;
  struct etpan_storage * storage;
  struct etpan_account_manager * manager;
  struct etpan_error * error;
  struct panel_data * data;
  
  data = panel->data;
  storage = etpan_account_get_storage(account);
  
  manager = etpan_account_manager_get_default();
  
  modify_account_common(panel, account);
  
  path = data->local_path;
  if (path == NULL) {
    char path_buffer[PATH_MAX];
    char * home;
    
    home = getenv("HOME");
    ETPAN_ASSERT(home != NULL, "home not defined");
    snprintf(path_buffer, sizeof(path_buffer), "%s/MH-%s", home,
        etpan_account_get_id(account));
    etpan_storage_mh_set_path(storage, path_buffer);
  }
  else {
    etpan_storage_mh_set_path(storage, path);
  }
  
  cache_path = etpan_get_cache_dir(manager, account);
  etpan_make_dir(cache_path);
  etpan_storage_set_cache_path(storage, cache_path);
  free(cache_path);
  
  flags_path = etpan_get_flags_dir(manager, account);
  etpan_make_dir(flags_path);
  etpan_storage_set_flags_path(storage, flags_path);
  free(flags_path);
  
  error = etpan_account_setup(account);
  ETPAN_ERROR_IGNORE(error);
  etpan_account_manager_add_account(manager, account);
}

#if 0
static void disconnect_before_modify_callback(void * cb_data)
{
  struct etpan_account * account;
  
  account = cb_data;
  etpan_account_unsetup(account);
  etpan_account_free(account);
}
#endif

static void modify_account(struct etpan_preferences_panel * panel,
    int keep_selection)
{
  GtkWidget * combo;
  gint index;
  struct panel_data * data;
  GtkTreeIter iter;
  GtkTreePath * path;
  struct etpan_account * account;
  GtkWidget * treeview;
  carray * list;
  unsigned int account_index;
  unsigned int i;
  struct etpan_error * error;
  
  if (!check_consistency(panel))
    return;
  
  combo = etpan_preferences_panel_get_widget(panel, "account-type");
  
  data = panel->data;
  
  account = data->account_being_modified;
  
  list = etpan_account_manager_get_ordered_list(etpan_account_manager_get_default());
  account_index = 0;
  for(i = 0 ; i < carray_count(list) ; i ++) {
    if (carray_get(list, i) == account)
      account_index = i;
  }
  
  /* disconnect */
#if 0
  etpan_account_stop(account,
      disconnect_before_modify_callback, account);
#endif
  
  /* remove from account manager */
  etpan_account_manager_remove_account(etpan_account_manager_get_default(),
      account);
  
  index = gtk_combo_box_get_active(GTK_COMBO_BOX(combo));
  switch (index) {
  case 0: /* POP */
    create_pop_account(panel);
    break;
  case 1: /* IMAP */
    create_imap_account(panel);
    break;
  case 2: /* News */
    create_news_account(panel);
    break;
  case 3: /* Local */
    create_local_account(panel);
    break;
  }
  account = carray_get(list, carray_count(list) - 1);
  
  /* move the account to the previous position */
  if (carray_count(list) >= 2) {
    for(i = carray_count(list) - 2 ; i >= account_index ; i --) {
      struct etpan_account * tmp_account;
      
      tmp_account = carray_get(list, i);
      carray_set(list, i + 1, tmp_account);
      
      if (i == 0)
        break;
    }
    
    carray_set(list, account_index, account);
  }
  
  etpan_account_manager_notify_modification(etpan_account_manager_get_default());
  
  etpan_gtk_tree_model_reload(data->treemodel);
  set_add_remove_state(panel);
  data->account_being_modified = NULL;
  
  if (keep_selection) {
    if (carray_count(list) > 0) {
      treeview = etpan_preferences_panel_get_widget(panel, "account-list");
      etpan_gtk_tree_model_get_iter_from_item(data->treemodel, &iter, account);
      path = gtk_tree_model_get_path(GTK_TREE_MODEL(data->treemodel), &iter);
      gtk_tree_view_set_cursor(GTK_TREE_VIEW(treeview), path, NULL, FALSE);
      gtk_tree_path_free(path);
    }
  }
  
  error = etpan_account_config_save_default();
  ETPAN_ERROR_IGNORE(error);
}


/* *********************************** */
/* remove */

#if 0
static void disconnect_before_remove_callback(void * cb_data)
{
  struct etpan_account * account;
  
  account = cb_data;
  etpan_account_unsetup(account);
  etpan_account_free(account);
}
#endif

static void remove_account(struct etpan_preferences_panel * panel)
{
  struct panel_data * data;
  struct etpan_account * account;
  struct etpan_account * last_account;
  GtkTreeIter iter;
  GtkTreePath * path;
  GtkWidget * treeview;
  int is_last_account;
  carray * list;
  unsigned int current_index;
  unsigned int i;
  struct etpan_error * error;
  
  data = panel->data;
  
  treeview = etpan_preferences_panel_get_widget(panel, "account-list");
  account = data->account_being_modified;
  
  if (data->pending_account != NULL) {
    
    list = etpan_account_manager_get_ordered_list(etpan_account_manager_get_default());
    if (carray_count(list) == 0) {
      reset_properties(panel);
      check_consistency(panel);
      return;
    }
    
    stop_add(panel);
    account = carray_get(list, carray_count(list) - 1);
    etpan_gtk_tree_model_get_iter_from_item(data->treemodel, &iter, account);
    
    path = gtk_tree_model_get_path(GTK_TREE_MODEL(data->treemodel), &iter);
    gtk_tree_view_set_cursor(GTK_TREE_VIEW(treeview), path, NULL, FALSE);
    gtk_tree_path_free(path);
    
    return;
  }
  
  data->account_being_modified = NULL;
  
  is_last_account = 0;
  list = etpan_account_manager_get_ordered_list(etpan_account_manager_get_default());
  if (carray_count(list) > 0) {
    last_account = carray_get(list, carray_count(list) - 1);
    if (account == last_account)
      is_last_account = 1;
  }
  current_index = 0;
  for(i = 0 ; i < carray_count(list) ; i ++) {
    if (carray_get(list, i) == account)
      current_index = i;
  }
  
  etpan_account_manager_remove_account(etpan_account_manager_get_default(),
      account);
  etpan_account_manager_notify_modification(etpan_account_manager_get_default());
  
  etpan_gtk_tree_model_reload(data->treemodel);
  set_add_remove_state(panel);
  
  if (carray_count(list) > 0) {
    if (is_last_account) {
      /* select last account if needed */
      last_account = carray_get(list, carray_count(list) - 1);
      etpan_gtk_tree_model_get_iter_from_item(data->treemodel, &iter, last_account);
      path = gtk_tree_model_get_path(GTK_TREE_MODEL(data->treemodel), &iter);
      gtk_tree_view_set_cursor(GTK_TREE_VIEW(treeview), path, NULL, FALSE);
      gtk_tree_path_free(path);
    }
    else {
      struct etpan_account * account_to_select;
    
      account_to_select = carray_get(list, current_index);
      etpan_gtk_tree_model_get_iter_from_item(data->treemodel, &iter, account_to_select);
      path = gtk_tree_model_get_path(GTK_TREE_MODEL(data->treemodel), &iter);
      gtk_tree_view_set_cursor(GTK_TREE_VIEW(treeview), path, NULL, FALSE);
      gtk_tree_path_free(path);
    }
  }
  else {
    add_account(panel);
  }
  
#if 0
  etpan_account_stop(account,
      disconnect_before_remove_callback, account);
#endif
  
  error = etpan_account_config_save_default();
  ETPAN_ERROR_IGNORE(error);
}

/* *********************************** */
/* check consistency */

static void switch_text(struct etpan_preferences_panel * panel,
    char * name, char * value)
{
  GtkWidget * label;
  
  label = etpan_preferences_panel_get_widget(panel, name);
  gtk_label_set_markup(GTK_LABEL(label), value);
}

static int check_consistency(struct etpan_preferences_panel * panel)
{
  GtkWidget * chooser;
  GtkWidget * entry;
  const char * value;
  gboolean enabled;
  struct panel_data * data;
  GtkWidget * combo;
  gint index;
  
  data = panel->data;
  enabled = data->modified;
  
  entry = etpan_preferences_panel_get_widget(panel, "account-name");
  value = gtk_entry_get_text(GTK_ENTRY(entry));
  if (data->modified && (value[0] == '\0')) {
    switch_text(panel, "label-account-name", _("<span foreground=\"#ff0000\">Account Name</span>"));
    enabled = 0;
  }
  else {
    switch_text(panel, "label-account-name", _("<span foreground=\"#000000\">Account Name</span>"));
  }
  
  entry = etpan_preferences_panel_get_widget(panel, "address");
  value = gtk_entry_get_text(GTK_ENTRY(entry));
  if (data->modified && (value[0] == '\0')) {
    switch_text(panel, "label-address", _("<span foreground=\"#ff0000\">E-mail Address</span>"));
    enabled = 0;
  }
  else {
    switch_text(panel, "label-address", _("<span foreground=\"#000000\">E-mail Address</span>"));
  }
  
  combo = etpan_preferences_panel_get_widget(panel, "account-type");
  index = gtk_combo_box_get_active(GTK_COMBO_BOX(combo));
  switch (index) {
  case 0: /* POP */
    {
      entry = etpan_preferences_panel_get_widget(panel, "pop-server");
      value = gtk_entry_get_text(GTK_ENTRY(entry));
      if (data->modified && (value[0] == '\0')) {
        switch_text(panel, "label-pop-server", _("<span foreground=\"#ff0000\">Receive Server</span>"));
        enabled = 0;
      }
      else {
        switch_text(panel, "label-pop-server", _("<span foreground=\"#000000\">Receive Server</span>"));
      }
    }
    break;
  case 1: /* IMAP */
    {
      entry = etpan_preferences_panel_get_widget(panel, "imap-server");
      value = gtk_entry_get_text(GTK_ENTRY(entry));
      if (data->modified && (value[0] == '\0')) {
        switch_text(panel, "label-imap-server", _("<span foreground=\"#ff0000\">Receive Server</span>"));
        enabled = 0;
      }
      else {
        switch_text(panel, "label-imap-server", _("<span foreground=\"#000000\">Receive Server</span>"));
      }
    }
    break;
  case 2: /* News */
    {
      entry = etpan_preferences_panel_get_widget(panel, "news-server");
      value = gtk_entry_get_text(GTK_ENTRY(entry));
      if (data->modified && (value[0] == '\0')) {
        switch_text(panel, "label-news-server", _("<span foreground=\"#ff0000\">Receive Server</span>"));
        enabled = 0;
      }
      else {
        switch_text(panel, "label-news-server", _("<span foreground=\"#000000\">Receive Server</span>"));
      }
    }
    break;
  case 3: /* Local */
#if 0
    {
      value = data->local_path;
      if (data->modified && (value[0] == '\0')) {
        switch_text(panel, "label-local-path", _("<span foreground=\"#ff0000\">Mailbox Path</span>"));
        enabled = 0;
      }
      else {
        switch_text(panel, "label-local-path", _("<span foreground=\"#000000\">Mailbox Path</span>"));
      }
    }
#endif
    break;
  }
  
  entry = etpan_preferences_panel_get_widget(panel, "smtp-server");
  value = gtk_entry_get_text(GTK_ENTRY(entry));
  if (data->modified && (value[0] == '\0')) {
    switch_text(panel, "label-smtp-server", _("<span foreground=\"#ff0000\">Sending Server</span>"));
    enabled = 0;
  }
  else {
    switch_text(panel, "label-smtp-server", _("<span foreground=\"#000000\">Sending Server</span>"));
  }
  
  if ((!enabled) && data->modified) {
    switch_text(panel, "label-info",
        _("<span foreground=\"#ff0000\">Fields that appear in red needs to be filled in</span>"));
  }
  else {
    switch_text(panel, "label-info", "");
  }
  
  return enabled;
}


static int is_empty(struct etpan_preferences_panel * panel)
{
  GtkWidget * chooser;
  GtkWidget * entry;
  const char * value;
  struct panel_data * data;
  GtkWidget * combo;
  gint index;
  int empty;
  
  data = panel->data;
  empty = 1;
  
  entry = etpan_preferences_panel_get_widget(panel, "account-name");
  value = gtk_entry_get_text(GTK_ENTRY(entry));
  if (value[0] != '\0') {
    empty = 0;
  }
  
  entry = etpan_preferences_panel_get_widget(panel, "address");
  value = gtk_entry_get_text(GTK_ENTRY(entry));
  if (value[0] != '\0') {
    empty = 0;
  }
  
  combo = etpan_preferences_panel_get_widget(panel, "account-type");
  index = gtk_combo_box_get_active(GTK_COMBO_BOX(combo));
  switch (index) {
  case 0: /* POP */
    {
      entry = etpan_preferences_panel_get_widget(panel, "pop-server");
      value = gtk_entry_get_text(GTK_ENTRY(entry));
      if (value[0] != '\0') {
        empty = 0;
      }
      if (data->pop_path != NULL)
        empty = 0;
    }
    break;
  case 1: /* IMAP */
    {
      entry = etpan_preferences_panel_get_widget(panel, "imap-server");
      value = gtk_entry_get_text(GTK_ENTRY(entry));
      if (value[0] != '\0') {
        empty = 0;
      }
    }
    break;
  case 2: /* News */
    {
      entry = etpan_preferences_panel_get_widget(panel, "news-server");
      value = gtk_entry_get_text(GTK_ENTRY(entry));
      if (value[0] != '\0') {
        empty = 0;
      }
    }
    break;
  case 3: /* Local */
#if 0
    {
      chooser = etpan_preferences_panel_get_widget(panel, "local-path");
      value = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(chooser));
      if (value != NULL) {
        empty = 0;
        g_free((gpointer) value);
      }
    }
#endif
    if (data->local_path != NULL)
      empty = 0;
    break;
  }
  
  entry = etpan_preferences_panel_get_widget(panel, "smtp-server");
  value = gtk_entry_get_text(GTK_ENTRY(entry));
  if (value[0] != '\0') {
    empty = 0;
  }
  
  return empty;
}

/* *********************************** */
/* preferences panel handler */

static int panel_should_close(struct etpan_preferences_panel * panel)
{
  struct panel_data * data;
  
  data = panel->data;
  
  if (data->pending_account) {
    if (data->modified) {
      if (!check_consistency(panel))
        return 0;
    }
  }
  else if (data->account_being_modified != NULL) {
    if (data->modified) {
      if (!check_consistency(panel))
        return 0;
    }
  }
  
  return 1;
}

static void panel_close(struct etpan_preferences_panel * panel)
{
  struct panel_data * data;
  GtkWidget * hpaned;
  
  hpaned = etpan_preferences_panel_get_widget(panel, "account-slider");
  etpan_ui_set_from_slider(etpan_ui_config_get_default(),
      "account-preferences-slider", hpaned);
  
  data = panel->data;
  
  if (data->pending_account) {
    if (data->modified) {
      create_account(panel, 0);
    }
    else {
      remove_account(panel);
    }
  }
  else if (data->account_being_modified) {
    if (data->modified) {
      modify_account(panel, 0);
    }
  }
}

static void panel_open(struct etpan_preferences_panel * panel)
{
  GtkWidget * treeview;
  GtkTreePath * path;
  struct panel_data * data;
  GtkWidget * hpaned;
  
  hpaned = etpan_preferences_panel_get_widget(panel, "account-slider");
  etpan_ui_slider_set(etpan_ui_config_get_default(),
      "account-preferences-slider", hpaned);
  
  data = panel->data;
  data->account_set = 0;
  data->modified = 0;
  treeview = etpan_preferences_panel_get_widget(panel, "account-list");
  
  etpan_gtk_tree_model_reload(data->treemodel);
  
  path = gtk_tree_path_new_first();
  gtk_tree_view_set_cursor(GTK_TREE_VIEW(treeview), path, NULL, FALSE);
  gtk_tree_path_free(path);
}


/* *********************************** */
/* account type */

static int get_account_type(struct etpan_account * account)
{
  struct etpan_storage * storage;
  struct etpan_storage_driver * driver;
  int type;
  
  storage = etpan_account_get_storage(account);
  if (storage == NULL) {
    ETPAN_LOG("account has no storage");
    etpan_crash();
  }
  
  driver = etpan_storage_get_driver(storage);
  type = STORAGE_DRIVER_TYPE_POP;
  if (strcasecmp(driver->name, "pop-sync") == 0)
    type = STORAGE_DRIVER_TYPE_POP;
  else if (strcasecmp(driver->name, "imap-sync") == 0)
    type = STORAGE_DRIVER_TYPE_IMAP;
  else if (strcasecmp(driver->name, "news") == 0)
    type = STORAGE_DRIVER_TYPE_NEWS;
  else if (strcasecmp(driver->name, "maildir") == 0)
    type = STORAGE_DRIVER_TYPE_MAILDIR;
  else if (strcasecmp(driver->name, "mh") == 0)
    type = STORAGE_DRIVER_TYPE_MH;
  else if (strcasecmp(driver->name, "mbox") == 0)
    type = STORAGE_DRIVER_TYPE_MBOX;
  
  return type;
}

static char * get_account_type_str(int value)
{
  char * account_type_str;
  
  account_type_str = NULL;
  switch (value) {
  case -1:
    account_type_str = "";
    break;    
  case STORAGE_DRIVER_TYPE_POP:
    account_type_str = _("POP");
    break;
  case STORAGE_DRIVER_TYPE_IMAP:
    account_type_str = _("IMAP");
    break;
  case STORAGE_DRIVER_TYPE_NEWS:
    account_type_str = _("News");
    break;
  case STORAGE_DRIVER_TYPE_MAILDIR:
  case STORAGE_DRIVER_TYPE_MH:
  case STORAGE_DRIVER_TYPE_MBOX:
    account_type_str = _("Local");
    break;
  }
  
  return account_type_str;
}

static void free_data(struct etpan_preferences_panel * panel)
{
  (void) panel;
}
