#include "etpan-preferences-compose.h"

#include "etpan-backend.h"
#include "etpan-preferences-window.h"
#include "etpan-ui-config.h"
#include "etpan-icon-manager.h"

struct panel_data {
  GtkTextBuffer * buffer;
};

/*
  message composer page
  - edit signature
  - address book -- later
*/

/* TODO - save after 2 seconds if modified by user */

#define PADDING 5

static void panel_close(struct etpan_preferences_panel * panel);
static void panel_open(struct etpan_preferences_panel * panel);
static void free_data(struct etpan_preferences_panel * panel);

static void message_compose_properties_content(struct etpan_preferences_panel * panel,
    GtkWidget * table)
{
  guint gline;
  int line;
  GtkWidget * label_signature;
  GtkWidget * textview;
  GtkWidget * scrolledwindow;
  GtkTextBuffer * buffer;
  struct panel_data * data;
  
  data = panel->data;
  
  g_object_get(table, "n-rows", &gline, NULL);
  line = gline;
  
  /* signature */
  gtk_table_resize(GTK_TABLE(table), line + 1, 2);
  label_signature = gtk_label_new(_("Signature"));
  gtk_misc_set_alignment(GTK_MISC(label_signature), 0.0, 0.5);
  gtk_widget_show(label_signature);
  gtk_table_attach(GTK_TABLE(table), label_signature, 0, 2, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  scrolledwindow = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrolledwindow),
      GTK_SHADOW_IN);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledwindow),
      GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  
  buffer = gtk_text_buffer_new(NULL);
  textview = gtk_text_view_new();
  gtk_text_view_set_buffer(GTK_TEXT_VIEW(textview), buffer);
  gtk_widget_show(textview);
  
  gtk_container_add(GTK_CONTAINER(scrolledwindow), textview);
  
  gtk_widget_show(scrolledwindow);
  
  gtk_table_attach(GTK_TABLE(table), scrolledwindow, 0, 2, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  data->buffer = buffer;
  etpan_preferences_panel_set_widget(panel, "signature-textview", textview);
}

GtkWidget * message_compose_properties(struct etpan_preferences_panel * panel)
{
  GtkWidget * table;
  GtkWidget * scrolledwindow;
  
  table = gtk_table_new(1, 2, FALSE);
  
  message_compose_properties_content(panel, table);
  
  gtk_widget_show(table);
  
  scrolledwindow = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledwindow),
      GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW(scrolledwindow),
      GTK_SHADOW_IN);
  
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolledwindow),
      table);
  gtk_widget_show(scrolledwindow);
  
  return scrolledwindow;
}

static void setup(struct etpan_preferences_panel * panel);

void etpan_preferences_compose_init(struct etpan_preferences_window * preferences)
{
  GtkWidget * prop;
  struct etpan_preferences_panel * panel;
  struct panel_data * data;
  
  panel = etpan_preferences_panel_new();
  panel->open_callback = panel_open;
  panel->close_callback = panel_close;
  panel->free_data = free_data;
  
  data = malloc(sizeof(* data));
  if (data == NULL) {
    free(panel);
    ETPAN_LOG_MEMORY_ERROR;
  }
  
  panel->data = data;
  
  prop = message_compose_properties(panel);
  
  panel->main_widget = prop;
  
#define ICON(name) \
  etpan_icon_manager_new_scaled_image(etpan_icon_manager_get_default(), \
      name, 32)
  etpan_preferences_add_page(preferences, _("Compose"), ICON("prefs-compose"), panel);
#undef ICON

  setup(panel);
}


static void update_font(struct etpan_preferences_panel * panel)
{
  PangoFontDescription * font_desc;
  GtkWidget * textview;
  
  textview = etpan_preferences_panel_get_widget(panel, "signature-textview");
  
  font_desc = pango_font_description_from_string(etpan_ui_config_get_font_message(etpan_ui_config_get_default()));
  gtk_widget_modify_font(textview, font_desc);
  pango_font_description_free(font_desc);
}

static void font_changed_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct etpan_preferences_panel * panel;
  (void) signal_name;
  (void) signal_data;
  (void) sender;
  
  panel = user_data;
  update_font(panel);
}

static void setup(struct etpan_preferences_panel * panel)
{
  etpan_signal_add_handler(etpan_signal_manager_get_default(),
      ETPAN_UI_CONFIG_FONT_CHANGED_SIGNAL,
      etpan_ui_config_get_default(), panel,
      font_changed_handler);
  
  update_font(panel);
}

static void unsetup(struct etpan_preferences_panel * panel)
{
  etpan_signal_remove_handler(etpan_signal_manager_get_default(),
      ETPAN_UI_CONFIG_FONT_CHANGED_SIGNAL,
      etpan_ui_config_get_default(), panel,
      font_changed_handler);
}

static void panel_close(struct etpan_preferences_panel * panel)
{
  struct panel_data * data;
  char * content;
  GtkTextIter start_iter;
  GtkTextIter end_iter;
  FILE * f;
  char * home;
  char path[PATH_MAX];
  struct etpan_error * error;
  
  /* save */
  data = panel->data;
  
  gtk_text_buffer_get_start_iter(data->buffer, &start_iter);
  gtk_text_buffer_get_end_iter(data->buffer, &end_iter);
  content = gtk_text_buffer_get_text(data->buffer,
      &start_iter, &end_iter, FALSE);
  
  home = getenv("HOME");
  if (home == NULL) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_INVAL);
    etpan_error_set_short_description(error, _("Home directory not found"));
    etpan_error_strf_long_description(error, _("Home directory is not defined. It should be defined in environment variable HOME."));
    
    goto free;
  }
  
  snprintf(path, sizeof(path), "%s/.signature", home);
  f = fopen(path, "wb");
  if (f == NULL) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_FILE);
    etpan_error_set_short_description(error, _("File could not be written"));
    etpan_error_strf_long_description(error, _("Signature file %s could not be written. Check permission of the file."), path);
    
    goto free;
  }
  
  fputs(content, f);
  fclose(f);

  g_free(content);
  return;
  
 free:
  g_free(content);
  etpan_error_log(error);
  ETPAN_ERROR_FREE(error);
}

static void panel_open(struct etpan_preferences_panel * panel)
{
  struct panel_data * data;
  GtkTextIter start_iter;
  GtkTextIter end_iter;
  FILE * f;
  char * home;
  GtkTextIter iter;
  char buffer[1024];
  char path[PATH_MAX];
  struct etpan_error * error;
  MMAPString * str;
  int r;
  
  /* load */
  data = panel->data;
  
  gtk_text_buffer_get_start_iter(data->buffer, &start_iter);
  gtk_text_buffer_get_end_iter(data->buffer, &end_iter);
  gtk_text_buffer_delete(data->buffer,
      &start_iter, &end_iter);
  
  home = getenv("HOME");
  if (home == NULL) {
    error = etpan_error_new();
    etpan_error_set_code(error, ERROR_INVAL);
    etpan_error_set_short_description(error, _("Home directory not found"));
    etpan_error_strf_long_description(error, _("Home directory is not defined. It should be defined in environment variable HOME."));
    
    goto err;
  }
  
  snprintf(path, sizeof(path), "%s/.signature", home);

  str = mmap_string_new("");
  if (str == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  f = fopen(path, "r");
  if (f != NULL) {
    char * converted;
    size_t converted_length;
    int is_utf8;
    
    while (fgets(buffer, sizeof(buffer), f)) {
      if (mmap_string_append(str, buffer) == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    
    r = charconv_buffer("utf-8", "utf-8",
        str->str, str->len,
        &converted, &converted_length);
    if (r != MAIL_CHARCONV_NO_ERROR)
      ETPAN_LOG_MEMORY_ERROR;
    
    is_utf8 = (strcmp(converted, str->str) == 0);
    charconv_buffer_free(converted);
    
    if (is_utf8) {
      gtk_text_buffer_get_start_iter(data->buffer, &iter);
      gtk_text_buffer_insert(data->buffer, &iter, str->str, str->len);
    }
    else {
      charconv_buffer("utf-8", "iso-8859-1",
          str->str, str->len,
          &converted, &converted_length);
      if (r != MAIL_CHARCONV_NO_ERROR)
        ETPAN_LOG_MEMORY_ERROR;
      
      gtk_text_buffer_get_start_iter(data->buffer, &iter);
      gtk_text_buffer_insert(data->buffer, &iter, converted, converted_length);
      
      charconv_buffer_free(converted);
    }
    
  }
  mmap_string_free(str);
  
#if 0
  f = fopen(path, "r");
  if (f != NULL) {
    gtk_text_buffer_get_start_iter(data->buffer, &iter);
    while (fgets(buffer, sizeof(buffer), f)) {
      gtk_text_buffer_insert(data->buffer, &iter, buffer, strlen(buffer));
    }
    
    fclose(f);
  }
#endif
  
  return;
  
 err:
  etpan_error_log(error);
  ETPAN_ERROR_FREE(error);
  {}
}

static void free_data(struct etpan_preferences_panel * panel)
{
  struct panel_data * data;
  
  unsetup(panel);
  
  data = panel->data;
  free(data);
}

