#include "etpan-preferences-filter.h"

#include "etpan-preferences-window.h"
#include "etpan-gtk-tree-model.h"
#include "etpan-backend.h"
#include "etpan-icon-manager.h"
#include "etpan-ui-config.h"
#include "etpan-filter-config.h"
#include "etpan-filter.h"

#define PADDING 5

static carray * param_list_dup(carray * param_list);
static struct etpan_filter_condition_ui_description * get_condition_description(char * name);
static struct etpan_filter_action_ui_description * get_action_description(char * name);

struct condition_action_ui {
  int is_new;
  struct etpan_preferences_panel * panel;
  chash * widget_hash;
};

struct panel_data {
  int modified;
  int adding_element;
  int init_done;
  int invalid_filter;
  struct etpan_gtk_tree_data_source filter_datasource;
  etpan_gtk_tree_model * treemodel;
  struct etpan_gtk_tree_data_source account_datasource;
  etpan_gtk_tree_model * account_treemodel;
  char * chosen_path;
  carray * condition_ui_list;
  carray * action_ui_list;
  struct etpan_filter * pending_filter;
  struct etpan_filter * filter_being_modified;
};

enum {
  CONDITION_TYPE_NONE,
  CONDITION_TYPE_OR,
  CONDITION_TYPE_AND,
  CONDITION_TYPE_NOT,
};

struct mapped_filter {
  int valid;
  int condition_type;
  char * name;
  carray * condition_list;
  carray * action_list;
};

struct mapped_condition {
  char * name;
  carray * param_list;
};

struct mapped_action {
  char * name;
  carray * param_list;
};

struct etpan_filter_condition_ui_description {
  char * name;
  char * localization_key;
  void (* condition_action_properties)(struct condition_action_ui * info,
      GtkWidget * hbox);
  void (* condition_unsetup)(struct condition_action_ui * info);
  void (* condition_get)(struct condition_action_ui * info,
      struct mapped_condition * condition);
  void (* condition_set)(struct condition_action_ui * info,
      struct mapped_condition * condition);
  struct etpan_filter_condition * (* condition_generate)(struct mapped_condition * condition);
  int (* check_consistency)(struct condition_action_ui * info);
};

struct etpan_filter_action_ui_description {
  char * name;
  char * localization_key;
  void (* condition_action_properties)(struct condition_action_ui * info,
      GtkWidget * hbox);
  void (* action_unsetup)(struct condition_action_ui * info);
  void (* action_get)(struct condition_action_ui * info,
      struct mapped_action * action);
  void (* action_set)(struct condition_action_ui * info,
      struct mapped_action * action);
  int (* check_consistency)(struct condition_action_ui * info);
};

/* datasource */

enum {
  STORE_INDEX_NAME,
  STORE_INDEX_FILTER,
};

static unsigned int get_n_columns(struct etpan_gtk_tree_data_source *
    datasource);
static GType get_column_type(struct etpan_gtk_tree_data_source * datasource,
    unsigned int column_index);
static unsigned int get_children_count(struct etpan_gtk_tree_data_source *
    datasource, void * item);
static int item_has_child(struct etpan_gtk_tree_data_source *
    datasource, void * item);
static void * get_child_item(struct etpan_gtk_tree_data_source *
    datasource, void * item, unsigned int index);
static void get_item_value(struct etpan_gtk_tree_data_source *
    datasource, void * item, unsigned int column_index,
    GValue * value);

/* account data source */

enum {
  ACCOUNT_STORE_INDEX_NAME,
  ACCOUNT_STORE_INDEX_ACCOUNT,
};

static unsigned int account_get_n_columns(struct etpan_gtk_tree_data_source *
    datasource);
static GType account_get_column_type(struct etpan_gtk_tree_data_source * datasource,
    unsigned int column_index);
static unsigned int account_get_children_count(struct etpan_gtk_tree_data_source *
    datasource, void * item);
static int account_item_has_child(struct etpan_gtk_tree_data_source *
    datasource, void * item);
static void * account_get_child_item(struct etpan_gtk_tree_data_source *
    datasource, void * item, unsigned int index);
static void account_get_item_value(struct etpan_gtk_tree_data_source *
    datasource, void * item, unsigned int column_index,
    GValue * value);

static void panel_open(struct etpan_preferences_panel * panel);
static int panel_should_close(struct etpan_preferences_panel * panel);
static void panel_close(struct etpan_preferences_panel * panel);
static void free_data(struct etpan_preferences_panel * panel);
static void setup(struct etpan_preferences_panel * panel);

static void filter_selected_handler(GtkTreeView * treeview,
    gpointer user_data);
static void add_pressed_handler(GtkButton *button, gpointer user_data);
static void remove_pressed_handler(GtkButton *button, gpointer user_data);
static void create_pressed_handler(GtkButton *button, gpointer user_data);
static void cancel_pressed_handler(GtkButton *button, gpointer user_data);
static void apply_pressed_handler(GtkButton *button, gpointer user_data);
static void revert_pressed_handler(GtkButton *button, gpointer user_data);
static void add_condition_pressed_handler(GtkButton *button, gpointer user_data);
static void remove_condition_pressed_handler(GtkButton *button, gpointer user_data);
static void add_action_pressed_handler(GtkButton *button, gpointer user_data);
static void remove_action_pressed_handler(GtkButton *button, gpointer user_data);

static int check_consistency(struct etpan_preferences_panel * panel);

static GtkWidget * filter_properties(struct etpan_preferences_panel * panel);
static void layout_filter_preferences(struct etpan_preferences_panel * panel);
static void enable_buttons(struct etpan_preferences_panel * panel);

static struct etpan_filter *
filter_get_ui(struct etpan_preferences_panel * panel);

void etpan_preferences_filter_init(struct etpan_preferences_window * preferences)
{
  struct etpan_preferences_panel * panel;
  GtkWidget * hpaned;
  GtkWidget * treeview;
  GtkWidget * scrolledwindow;
  GtkWidget * vbox;
  GtkWidget * button_hbox;
  GtkWidget * add_button;
  GtkWidget * remove_button;
  etpan_gtk_tree_model * treemodel;
  etpan_gtk_tree_model * account_treemodel;
  gint selected_signal_id;
  gint add_signal_id;
  gint remove_signal_id;
  struct panel_data * data;
  GtkTreeSelection * selection;
  GtkTreeViewColumn * col_name;
  GtkCellRenderer * col_name_renderer;
  GtkWidget * filter_prop;
  
  panel = etpan_preferences_panel_new();
  
  panel->open_callback = panel_open;
  panel->should_close_callback = panel_should_close;
  panel->close_callback = panel_close;
  panel->free_data = free_data;
  
  data = malloc(sizeof(* data));
  if (data == NULL) {
    ETPAN_LOG_MEMORY_ERROR;
  }
  data->modified = 0;
  data->adding_element = 0;
  data->init_done = 0;
  data->treemodel = NULL;
  data->chosen_path = NULL;
  data->condition_ui_list = carray_new(4);
  if (data->condition_ui_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  data->action_ui_list = carray_new(4);
  if (data->action_ui_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  data->pending_filter = NULL;
  data->filter_being_modified = NULL;
  
  panel->data = data;
  
  hpaned = gtk_hpaned_new();
  gtk_widget_show(hpaned);
  gtk_paned_set_position(GTK_PANED(hpaned), 200);
  
  vbox = gtk_vbox_new(FALSE, FALSE);
  gtk_widget_show(vbox);
  gtk_paned_pack1(GTK_PANED(hpaned), vbox, FALSE, TRUE);
  
  scrolledwindow = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrolledwindow),
      GTK_SHADOW_IN);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledwindow),
      GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_widget_show(scrolledwindow);
  gtk_box_pack_start(GTK_BOX(vbox), scrolledwindow, TRUE, TRUE, 0);
  
  treeview = gtk_tree_view_new();
  gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(treeview), TRUE);
  gtk_tree_view_set_enable_search(GTK_TREE_VIEW(treeview), FALSE);
  
  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
  gtk_tree_selection_set_mode(selection, GTK_SELECTION_BROWSE);
  
  /* column name */
  col_name = gtk_tree_view_column_new();
  gtk_tree_view_column_set_title(col_name, _("Name"));
  col_name_renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_column_set_resizable(col_name, TRUE);
  gtk_tree_view_column_pack_start(col_name, col_name_renderer, TRUE);
  gtk_tree_view_column_set_attributes(col_name, col_name_renderer,
      "text", STORE_INDEX_NAME, NULL);
  gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), col_name);
  
  treemodel = etpan_gtk_tree_model_new();
  
  data->filter_datasource.data = panel;
  data->filter_datasource.get_n_columns = get_n_columns;
  data->filter_datasource.get_column_type = get_column_type;
  data->filter_datasource.get_children_count = get_children_count;
  data->filter_datasource.item_has_child = item_has_child;
  data->filter_datasource.get_child_item = get_child_item;
  data->filter_datasource.get_item_value = get_item_value;
  etpan_gtk_tree_model_set_datasource(treemodel,
      &data->filter_datasource);
  
  gtk_tree_view_set_model(GTK_TREE_VIEW(treeview), GTK_TREE_MODEL(treemodel));
  
  gtk_widget_show(treeview);
  gtk_container_add(GTK_CONTAINER(scrolledwindow), treeview);

  account_treemodel = etpan_gtk_tree_model_new();
  data->account_treemodel = account_treemodel;
  
  data->account_datasource.data = panel;
  data->account_datasource.get_n_columns = account_get_n_columns;
  data->account_datasource.get_column_type = account_get_column_type;
  data->account_datasource.get_children_count = account_get_children_count;
  data->account_datasource.item_has_child = account_item_has_child;
  data->account_datasource.get_child_item = account_get_child_item;
  data->account_datasource.get_item_value = account_get_item_value;
  etpan_gtk_tree_model_set_datasource(account_treemodel,
      &data->account_datasource);
  
  button_hbox = gtk_hbutton_box_new();
  gtk_button_box_set_layout(GTK_BUTTON_BOX(button_hbox), GTK_BUTTONBOX_START);
  gtk_widget_show(button_hbox);
  gtk_box_pack_start(GTK_BOX(vbox), button_hbox, FALSE, FALSE, 0);
  
  add_button = gtk_button_new_with_label(_("+"));
  gtk_widget_show(add_button);
  gtk_box_pack_start(GTK_BOX(button_hbox), add_button, FALSE, FALSE, 0);
  remove_button = gtk_button_new_with_label(_("-"));
  gtk_widget_show(remove_button);
  gtk_box_pack_start(GTK_BOX(button_hbox), remove_button, TRUE, FALSE, 0);
  
  filter_prop = filter_properties(panel);
  
  gtk_paned_pack2(GTK_PANED(hpaned), filter_prop, TRUE, TRUE);
  
  panel->main_widget = hpaned;
  
#define ICON(name) \
  etpan_icon_manager_new_scaled_image(etpan_icon_manager_get_default(), \
      name, 32)
  etpan_preferences_add_page(preferences, _("Filter"), ICON("prefs-filter"), panel);
#undef ICON
  
  data->treemodel = treemodel;
  
  etpan_preferences_panel_set_widget(panel, "filter-list", treeview);
  etpan_preferences_panel_set_widget(panel, "add-button", add_button);
  etpan_preferences_panel_set_widget(panel, "remove-button", remove_button);
  
  selected_signal_id = g_signal_connect(treeview,
      "cursor-changed", G_CALLBACK(filter_selected_handler),
      (gpointer) panel);
  
  add_signal_id = g_signal_connect(add_button,
      "clicked", G_CALLBACK(add_pressed_handler),
      (gpointer) panel);
  remove_signal_id = g_signal_connect(remove_button,
      "clicked", G_CALLBACK(remove_pressed_handler),
      (gpointer) panel);
  
  setup(panel);
  
  data->init_done = 1;
}

static void show_selected_filter(struct etpan_preferences_panel * panel);
static void add_filter(struct etpan_preferences_panel * panel);

static void panel_open(struct etpan_preferences_panel * panel)
{
  GtkWidget * treeview;
  struct panel_data * data;
  GtkWidget * hpaned;
  GtkTreePath * path;
  struct etpan_filter_config * config;
  carray * list;
  
  hpaned = etpan_preferences_panel_get_widget(panel, "filter-slider");
  etpan_ui_slider_set(etpan_ui_config_get_default(),
      "filter-preferences-slider", hpaned);
  
  data = panel->data;
  treeview = etpan_preferences_panel_get_widget(panel, "filter-list");
  
  etpan_gtk_tree_model_reload(data->treemodel);
  
  config = etpan_filter_config_get_default();
  list = etpan_filter_config_get_rule_list(config);
  if (carray_count(list) > 0) {
    ETPAN_LOG("list : %i", carray_count(list));
    path = gtk_tree_path_new_first();
    gtk_tree_view_set_cursor(GTK_TREE_VIEW(treeview), path, NULL, FALSE);
    gtk_tree_path_free(path);
  }
  else {
    ETPAN_LOG("add filter");
    add_filter(panel);
  }
}

static int panel_should_close(struct etpan_preferences_panel * panel)
{
  struct panel_data * data;
  
  data = panel->data;
  
  if (data->pending_filter) {
    if (data->modified) {
      if (!check_consistency(panel))
        return 0;
    }
  }
  else if (data->filter_being_modified != NULL) {
    if (data->modified) {
      if (!check_consistency(panel))
        return 0;
    }
  }
  
  return 1;
}

static void filter_reset_ui(struct etpan_preferences_panel * panel);

static void create_filter(struct etpan_preferences_panel * panel);
static struct etpan_filter *
apply_filter(struct etpan_preferences_panel * panel);
static void cancel_create_filter(struct etpan_preferences_panel * panel);

static void panel_close(struct etpan_preferences_panel * panel)
{
  struct panel_data * data;
  
  data = panel->data;
  
  if (data->pending_filter) {
    if (data->modified) {
      create_filter(panel);
    }
    else {
      cancel_create_filter(panel);
    }
  }
  else if (data->filter_being_modified) {
    if (data->modified) {
      apply_filter(panel);
    }
  }
}

static void free_data(struct etpan_preferences_panel * panel)
{
}

static void setup(struct etpan_preferences_panel * panel)
{
}

static void filter_set_ui(struct etpan_preferences_panel * panel,
    struct etpan_filter * filter);
static void filter_new_ui(struct etpan_preferences_panel * panel);
static struct etpan_filter * get_selected_filter(struct etpan_preferences_panel * panel);

static void filter_selected_handler(GtkTreeView * treeview,
    gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  struct panel_data * data;
  int go_back;
  struct etpan_filter * filter;
  
  panel = user_data;
  data = panel->data;
  
  filter = get_selected_filter(panel);
  if (data->filter_being_modified == filter) {
    return;
  }
  
  go_back = 0;
  if (data->filter_being_modified != NULL) {
    if (data->modified) {
      if (!check_consistency(panel))
        go_back = 1;
    }
  }
  ETPAN_LOG("go back: %i", go_back);
  
  if (go_back) {
    GtkTreeIter iter;
    GtkTreePath * path;
    GtkWidget * treeview;
    
    treeview = etpan_preferences_panel_get_widget(panel, "filter-list");
    
    etpan_gtk_tree_model_get_iter_from_item(data->treemodel, &iter,
        data->filter_being_modified);
    path = gtk_tree_model_get_path(GTK_TREE_MODEL(data->treemodel), &iter);
    gtk_tree_view_set_cursor(GTK_TREE_VIEW(treeview), path, NULL, FALSE);
    gtk_tree_path_free(path);
    
    return;
  }
  
  if (data->filter_being_modified != NULL) {
    if (data->filter_being_modified == data->pending_filter) {
      if (data->modified) {
        create_filter(panel);
      }
      else {
        cancel_create_filter(panel);
      }
    }
    else {
      if (data->modified) {
        apply_filter(panel);
      }
    }
  }
  
  show_selected_filter(panel);
  check_consistency(panel);
}

static void start_filter_edition(struct etpan_preferences_panel * panel);

static struct etpan_filter * get_selected_filter(struct etpan_preferences_panel * panel)
{
  struct etpan_filter * filter;
  GtkTreePath * path;
  GtkTreeIter iter;
  GtkTreeViewColumn * column;
  struct panel_data * data;
  GtkWidget * treeview;
  
  data = panel->data;
  
  filter = NULL;
  treeview = etpan_preferences_panel_get_widget(panel, "filter-list");
  gtk_tree_view_get_cursor(GTK_TREE_VIEW(treeview), &path, &column);
  if (path != NULL) {
    gtk_tree_model_get_iter(GTK_TREE_MODEL(data->treemodel), &iter, path);
    gtk_tree_model_get(GTK_TREE_MODEL(data->treemodel), &iter,
        STORE_INDEX_FILTER,
        &filter, -1);
    gtk_tree_path_free(path);
  }
  
  return filter;
}

static void show_selected_filter(struct etpan_preferences_panel * panel)
{
  struct etpan_filter * filter;
  struct panel_data * data;
  
  data = panel->data;
  
  filter = get_selected_filter(panel);
  
  ETPAN_LOG("show selected filter %p", filter);
  if (filter == NULL) {
    filter_new_ui(panel);
  }
  else {
    filter_set_ui(panel, filter);
    start_filter_edition(panel);
  }
  data->modified = 0;
  data->filter_being_modified = filter;
}

static void add_pressed_handler(GtkButton *button, gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  
  panel = user_data;
  add_filter(panel);
}

static void add_filter(struct etpan_preferences_panel * panel)
{
  struct etpan_filter_config * config;
  carray * list;
  unsigned int rule_index;
  char filter_name[512];
  struct panel_data * data;
  struct etpan_serialize_data * serialize_data;
  struct etpan_filter_condition * condition;
  struct etpan_filter_action * action;
  struct etpan_filter * filter;
  int r;
  carray * param_list;
  GtkTreePath * path;
  GtkWidget * treeview;
  GtkTreeIter iter;
  
  data = panel->data;
  
  filter = etpan_filter_new();
  data->pending_filter = filter;
  
  config = etpan_filter_config_get_default();
  list = etpan_filter_config_get_rule_list(config);
  rule_index = 1;
  while (1) {
    unsigned int i;
    int has_index;
    
    snprintf(filter_name, sizeof(filter_name), "%s #%i", _("Rule"), rule_index);
    
    has_index = 0;
    for(i = 0 ; i < carray_count(list) ; i ++) {
      struct etpan_filter * filter;
      
      filter = carray_get(list, i);
      if (strcmp(filter_name, etpan_filter_get_name(filter)) == 0) {
        has_index = 1;
      }
    }
    
    if (!has_index) {
      break;
    }
    
    rule_index ++;
  }
  
  etpan_filter_set_name(data->pending_filter, filter_name);
  
  param_list = carray_new(4);
  if (param_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  serialize_data = etpan_serialize_data_new_str("contains");
  r = carray_add(param_list, serialize_data, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  serialize_data = etpan_serialize_data_new_str("");
  r = carray_add(param_list, serialize_data, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  condition = etpan_filter_condition_new("from", param_list);
  etpan_filter_set_condition(filter, condition);
  
  param_list = carray_new(4);
  if (param_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  serialize_data = etpan_serialize_data_new_str("");
  r = carray_add(param_list, serialize_data, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  serialize_data = etpan_serialize_data_new_str("");
  r = carray_add(param_list, serialize_data, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  action = etpan_filter_action_new("move", param_list);
  etpan_filter_add_action(filter, action);
  
  etpan_gtk_tree_model_reload(data->treemodel);
  
  treeview = etpan_preferences_panel_get_widget(panel, "filter-list");
  etpan_gtk_tree_model_get_iter_from_item(data->treemodel, &iter, filter);
  path = gtk_tree_model_get_path(GTK_TREE_MODEL(data->treemodel), &iter);
  gtk_tree_view_set_cursor(GTK_TREE_VIEW(treeview), path, NULL, FALSE);
  gtk_tree_path_free(path);
  
  start_filter_edition(panel);
  enable_buttons(panel);
}

static void remove_pressed_handler(GtkButton *button, gpointer user_data)
{
  struct panel_data * data;
  struct etpan_preferences_panel * panel;
  
  panel = user_data;
  data = panel->data;
  
  if (data->pending_filter != NULL) {
    carray * list;
    struct etpan_filter * filter;
    GtkWidget * treeview;
    GtkTreePath * path;
    GtkTreeIter iter;
    
    cancel_create_filter(panel);
    
    list = etpan_filter_config_get_rule_list(etpan_filter_config_get_default());
    if (carray_count(list) == 0) {
      add_filter(panel);
      return;
    }
    filter = carray_get(list, carray_count(list) - 1);
    data = panel->data;
    treeview = etpan_preferences_panel_get_widget(panel, "filter-list");
    etpan_gtk_tree_model_get_iter_from_item(data->treemodel, &iter, filter);
    path = gtk_tree_model_get_path(GTK_TREE_MODEL(data->treemodel), &iter);
    gtk_tree_view_set_cursor(GTK_TREE_VIEW(treeview), path, NULL, FALSE);
    gtk_tree_path_free(path);
  }
  else {
    carray * list;
    unsigned int i;
    unsigned int index_to_select;
    int found;
    
    found = 0;
    list = etpan_filter_config_get_rule_list(etpan_filter_config_get_default());
    for(i = 0 ; i < carray_count(list) ; i ++) {
      struct etpan_filter * current_filter;
      
      current_filter = carray_get(list, i);
      if (current_filter == data->filter_being_modified) {
        if (i == carray_count(list) - 1) {
          index_to_select = carray_count(list) - 1;
        }
        else {
          index_to_select = i;
        }
        carray_delete_slow(list, i);
        data->filter_being_modified = 0;
        data->modified = 0;
        found = 1;
        break;
      }
    }
    if (found) {
      struct etpan_filter * filter;
      GtkWidget * treeview;
      GtkTreeIter iter;
      GtkTreePath * path;
      
      if (carray_count(list) == 0) {
        add_filter(panel);
        return;
      }
      etpan_gtk_tree_model_reload(data->treemodel);
      filter = carray_get(list, index_to_select);
      treeview = etpan_preferences_panel_get_widget(panel, "filter-list");
      etpan_gtk_tree_model_get_iter_from_item(data->treemodel, &iter, filter);
      path = gtk_tree_model_get_path(GTK_TREE_MODEL(data->treemodel), &iter);
      gtk_tree_view_set_cursor(GTK_TREE_VIEW(treeview), path, NULL, FALSE);
      gtk_tree_path_free(path);
    }
  }
  enable_buttons(panel);
}

/* *********************************** */
/* data source */

static unsigned int get_n_columns(struct etpan_gtk_tree_data_source *
    datasource)
{
  (void) datasource;
  
  return 2;
}

static GType get_column_type(struct etpan_gtk_tree_data_source * datasource,
    unsigned int column_index)
{
  static GType column_types[2] = {
    G_TYPE_STRING,
    G_TYPE_POINTER
  };
  (void) datasource;
  
  return column_types[column_index];
}
  
static unsigned int get_children_count(struct etpan_gtk_tree_data_source *
    datasource, void * item)
{
  carray * list;
  struct etpan_preferences_panel * panel;
  struct panel_data * data;
  
  panel = datasource->data;
  data = panel->data;
  list = etpan_filter_config_get_rule_list(etpan_filter_config_get_default());
  
  if (item == NULL) {
    if (data->pending_filter) {
      return carray_count(list) + 1;
    }
    else {
      return carray_count(list);
    }
  }
  else
    return 0;
}

static int item_has_child(struct etpan_gtk_tree_data_source *
    datasource, void * item)
{
  carray * list;
  struct etpan_preferences_panel * panel;
  struct panel_data * data;
  (void) datasource;
  
  panel = datasource->data;
  data = panel->data;
  list = etpan_filter_config_get_rule_list(etpan_filter_config_get_default());
  
  if (item == NULL) {
    if (data->pending_filter) {
      return 1;
    }
    else {
      return carray_count(list) > 0;
    }
  }
  else
    return 0;
}

static void * get_child_item(struct etpan_gtk_tree_data_source *
    datasource, void * item, unsigned int index)
{
  carray * list;
  struct etpan_preferences_panel * panel;
  struct panel_data * data;
  
  panel = datasource->data;
  data = panel->data;
  
  list = etpan_filter_config_get_rule_list(etpan_filter_config_get_default());
  
  if (item == NULL) {
    if (index == carray_count(list)) {
      return data->pending_filter;
    }
    else {
      return carray_get(list, index);
    }
  }
  else
    return NULL;
}

static void get_item_value(struct etpan_gtk_tree_data_source *
    datasource, void * item, unsigned int column_index,
    GValue * value)
{
  struct panel_data * data;
  struct etpan_preferences_panel * panel;
  struct etpan_filter * filter;
  
  panel = datasource->data;
  data = panel->data;
  
  filter = item;
  
  switch (column_index) {
  case STORE_INDEX_NAME:
    {
      char * str;
      
      if (filter == data->pending_filter) {
        str = _("(New Filter)");
      }
      else {
        str = etpan_filter_get_name(filter);
      }
      
      g_value_init(value, G_TYPE_STRING);
      g_value_set_string(value, str);
    }
    break;
  case STORE_INDEX_FILTER:
    {
      g_value_init(value, G_TYPE_POINTER);
      g_value_set_pointer(value, filter);
    }
    break;
  }
}

/* account data source */

static unsigned int account_get_n_columns(struct etpan_gtk_tree_data_source *
    datasource)
{
  (void) datasource;
  
  return 2;
}

static GType account_get_column_type(struct etpan_gtk_tree_data_source * datasource,
    unsigned int column_index)
{
  static GType column_types[2] = {
    G_TYPE_STRING,
    G_TYPE_POINTER
  };
  (void) datasource;
  
  return column_types[column_index];
}

static unsigned int account_get_children_count(struct etpan_gtk_tree_data_source *
    datasource, void * item)
{
  if (item == NULL) {
    carray * list;
    
    list = etpan_account_manager_get_ordered_list(etpan_account_manager_get_default());
    
    return carray_count(list);
  }
  else {
    return 0;
  }
}

static int account_item_has_child(struct etpan_gtk_tree_data_source *
    datasource, void * item)
{
  if (item == NULL) {
    carray * list;
    
    list = etpan_account_manager_get_ordered_list(etpan_account_manager_get_default());
    
    return carray_count(list) > 0;
  }
  else {
    return 0;
  }
}

static void * account_get_child_item(struct etpan_gtk_tree_data_source *
    datasource, void * item, unsigned int index)
{
  if (item == NULL) {
    carray * list;
    
    list = etpan_account_manager_get_ordered_list(etpan_account_manager_get_default());
    
    return carray_get(list, index);
  }
  else {
    return NULL;
  }
}

static void account_get_item_value(struct etpan_gtk_tree_data_source *
    datasource, void * item, unsigned int column_index,
    GValue * value)
{
  struct etpan_account * account;
  
  account = item;
  
  switch (column_index) {
  case ACCOUNT_STORE_INDEX_NAME:
    {
      char * str;
      
      str = etpan_account_get_id(account);
      
      g_value_init(value, G_TYPE_STRING);
      g_value_set_string(value, str);
    }
    break;
  case ACCOUNT_STORE_INDEX_ACCOUNT:
    {
      g_value_init(value, G_TYPE_POINTER);
      g_value_set_pointer(value, account);
    }
    break;
  }
}

/* **** filter properties *** */

static void basic_filter_properties(struct etpan_preferences_panel * panel,
    GtkWidget * table);

static GtkWidget * filter_properties(struct etpan_preferences_panel * panel)
{
  GtkWidget * table;
  GtkWidget * scrolledwindow;
  guint gline;
  int line;
  GtkWidget * hbox;
  GtkWidget * button_hbox;
  GtkWidget * create_button;
  GtkWidget * cancel_button;
  GtkWidget * revert_button;
  GtkWidget * apply_button;
  gint create_signal_id;
  gint cancel_signal_id;
  gint revert_signal_id;
  gint apply_signal_id;
  GtkWidget * label_info;
  
  table = gtk_table_new(1, 4, FALSE);
  
  basic_filter_properties(panel, table);
  
  g_object_get(table, "n-rows", &gline, NULL);
  line = gline;
  
  gtk_table_resize(GTK_TABLE(table), line + 1, 4);
  hbox = gtk_hbox_new(FALSE, PADDING);
  
  button_hbox = gtk_hbutton_box_new();
  gtk_button_box_set_layout(GTK_BUTTON_BOX(button_hbox), GTK_BUTTONBOX_END);
  
  cancel_button = gtk_button_new_with_label(_("Cancel"));
  gtk_widget_show(cancel_button);
  gtk_box_pack_start(GTK_BOX(button_hbox), cancel_button, FALSE, FALSE, 0);
  
  create_button = gtk_button_new_with_label(_("Create"));
  gtk_widget_show(create_button);
  gtk_box_pack_start(GTK_BOX(button_hbox), create_button, FALSE, FALSE, 0);
  
  revert_button = gtk_button_new_with_label(_("Revert"));
  gtk_widget_show(revert_button);
  gtk_box_pack_start(GTK_BOX(button_hbox), revert_button, FALSE, FALSE, 0);
  
  apply_button = gtk_button_new_with_label(_("Apply"));
  gtk_widget_show(apply_button);
  gtk_box_pack_start(GTK_BOX(button_hbox), apply_button, FALSE, FALSE, 0);
  
  gtk_widget_show(button_hbox);
  
  gtk_box_pack_start(GTK_BOX(hbox), button_hbox, TRUE, TRUE, 0);
  
  gtk_widget_show(hbox);
  
  gtk_table_attach(GTK_TABLE(table), hbox, 0, 4, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  gtk_table_resize(GTK_TABLE(table), line + 1, 4);
  label_info = gtk_label_new("");
  gtk_widget_show(label_info);
  gtk_table_attach(GTK_TABLE(table), label_info, 0, 4, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  gtk_widget_show(table);
  
  scrolledwindow = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledwindow),
      GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW(scrolledwindow),
      GTK_SHADOW_IN);
  
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolledwindow),
      table);
  gtk_widget_show(scrolledwindow);

  etpan_preferences_panel_set_widget(panel, "table", table);
  etpan_preferences_panel_set_widget(panel, "label-info", label_info);
  etpan_preferences_panel_set_widget(panel, "create-button", create_button);
  etpan_preferences_panel_set_widget(panel, "cancel-button", cancel_button);
  etpan_preferences_panel_set_widget(panel, "revert-button", revert_button);
  etpan_preferences_panel_set_widget(panel, "apply-button", apply_button);

  create_signal_id = g_signal_connect(create_button,
      "clicked", G_CALLBACK(create_pressed_handler),
      (gpointer) panel);

  cancel_signal_id = g_signal_connect(cancel_button,
      "clicked", G_CALLBACK(cancel_pressed_handler),
      (gpointer) panel);

  revert_signal_id = g_signal_connect(revert_button,
      "clicked", G_CALLBACK(revert_pressed_handler),
      (gpointer) panel);

  apply_signal_id = g_signal_connect(apply_button,
      "clicked", G_CALLBACK(apply_pressed_handler),
      (gpointer) panel);
  
  return scrolledwindow;
}

static void create_pressed_handler(GtkButton *button, gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  GtkWidget * treeview;
  GtkTreeIter iter;
  GtkTreePath * path;
  struct panel_data * data;
  carray * list;
  struct etpan_filter * filter;
  
  panel = user_data;
  if (!check_consistency(panel)) {
    return;
  }
  
  create_filter(panel);
  
  list = etpan_filter_config_get_rule_list(etpan_filter_config_get_default());
  filter = carray_get(list, carray_count(list) - 1);
  data = panel->data;
  treeview = etpan_preferences_panel_get_widget(panel, "filter-list");
  etpan_gtk_tree_model_get_iter_from_item(data->treemodel, &iter, filter);
  path = gtk_tree_model_get_path(GTK_TREE_MODEL(data->treemodel), &iter);
  gtk_tree_view_set_cursor(GTK_TREE_VIEW(treeview), path, NULL, FALSE);
  gtk_tree_path_free(path);
}

static void create_filter(struct etpan_preferences_panel * panel)
{
  struct panel_data * data;
  struct etpan_filter * filter;
  struct etpan_error * error;
  
  data = panel->data;
  if (data->pending_filter != NULL) {
    etpan_filter_free(data->pending_filter);
    data->pending_filter = NULL;
  }
  
  filter = filter_get_ui(panel);
  etpan_filter_config_add_rule(etpan_filter_config_get_default(), filter);
  etpan_gtk_tree_model_reload(data->treemodel);
  
  data->filter_being_modified = NULL;
  data->modified = 0;
  
  error = etpan_filter_config_write_default();
  ETPAN_ERROR_IGNORE(error);
}

static void cancel_create_filter(struct etpan_preferences_panel * panel)
{
  struct panel_data * data;
  struct etpan_filter * filter;
  
  data = panel->data;
  etpan_filter_free(data->pending_filter);
  data->pending_filter = NULL;
  data->filter_being_modified = NULL;
  data->modified = 0;
  etpan_gtk_tree_model_reload(data->treemodel);
}

static void cancel_pressed_handler(GtkButton *button, gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  carray * list;
  struct etpan_filter * filter;
  GtkWidget * treeview;
  GtkTreePath * path;
  GtkTreeIter iter;
  struct panel_data * data;
  
  panel = user_data;
  data = panel->data;
  cancel_create_filter(panel);
  
  list = etpan_filter_config_get_rule_list(etpan_filter_config_get_default());
  if (carray_count(list) == 0) {
    add_filter(panel);
    return;
  }
  filter = carray_get(list, carray_count(list) - 1);
  data = panel->data;
  treeview = etpan_preferences_panel_get_widget(panel, "filter-list");
  etpan_gtk_tree_model_get_iter_from_item(data->treemodel, &iter, filter);
  path = gtk_tree_model_get_path(GTK_TREE_MODEL(data->treemodel), &iter);
  gtk_tree_view_set_cursor(GTK_TREE_VIEW(treeview), path, NULL, FALSE);
  gtk_tree_path_free(path);
}

static struct etpan_filter *
apply_filter(struct etpan_preferences_panel * panel)
{
  struct etpan_filter * filter;
  carray * list;
  struct etpan_filter * old_filter;
  unsigned int i;
  unsigned int old_index;
  struct panel_data * data;
  struct etpan_error * error;
  
  data = panel->data;
  list = etpan_filter_config_get_rule_list(etpan_filter_config_get_default());
  old_filter = data->filter_being_modified;
  if (old_filter == NULL) {
    ETPAN_CRASH("filter should be in the list");
  }
  
  old_index = 0;
  for(i = 0 ; i < carray_count(list) ; i ++) {
    struct etpan_filter * current_filter;
    
    current_filter = carray_get(list, i);
    if (current_filter == old_filter) {
      old_index = i;
      break;
    }
  }
  
  filter = filter_get_ui(panel);
  carray_set(list, old_index, filter);
  etpan_filter_free(old_filter);
  data->filter_being_modified = NULL;
  data->modified = 0;
  
  etpan_gtk_tree_model_reload(data->treemodel);
  
  error = etpan_filter_config_write_default();
  ETPAN_ERROR_IGNORE(error);
  
  return filter;
}

static void apply_pressed_handler(GtkButton *button, gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  GtkWidget * treeview;
  GtkTreePath * path;
  GtkTreeIter iter;
  struct panel_data * data;
  struct etpan_filter * filter;
  
  panel = user_data;
  data = panel->data;
  if (!check_consistency(panel)) {
    return;
  }
  
  filter = apply_filter(panel);
  
  treeview = etpan_preferences_panel_get_widget(panel, "filter-list");
  etpan_gtk_tree_model_get_iter_from_item(data->treemodel, &iter, filter);
  path = gtk_tree_model_get_path(GTK_TREE_MODEL(data->treemodel), &iter);
  gtk_tree_view_set_cursor(GTK_TREE_VIEW(treeview), path, NULL, FALSE);
  gtk_tree_path_free(path);
}

static void revert_pressed_handler(GtkButton *button, gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  
  panel = user_data;
  show_selected_filter(panel);
}

static void multiple_condition_filter_properties(struct etpan_preferences_panel * panel,
    GtkWidget * table);
static void filter_condition_properties(struct condition_action_ui * info,
    GtkWidget * table);
static void multiple_action_filter_properties(struct etpan_preferences_panel * panel,
    GtkWidget * table);

static struct condition_action_ui * condition_action_ui_new(void);
static void condition_action_ui_free(struct condition_action_ui * info);

static void add_new_action_ui(struct etpan_preferences_panel * panel,
    GtkWidget * table);
static void add_new_condition_ui(struct etpan_preferences_panel * panel,
    GtkWidget * table);

static void entry_changed(GtkEditable *editable, gpointer user_data);
static void combo_changed(GtkComboBox * combo, gpointer user_data);
static void condition_action_entry_changed(GtkEditable *editable, gpointer user_data);
static void condition_action_textbuffer_changed(GtkTextBuffer * textbuffer,
    gpointer user_data);
static void condition_action_combo_changed(GtkComboBox * combo, gpointer user_data);
static void condition_action_color_changed(GtkColorButton * widget, gpointer user_data);

static void basic_filter_properties(struct etpan_preferences_panel * panel,
    GtkWidget * table)
{
  GtkWidget * label_name;
  GtkWidget * entry_name;
  guint gline;
  int line;
  GtkWidget * hbox;
  GtkWidget * condition_table;
  GtkWidget * action_table;
  GtkWidget * add_button;
  GtkWidget * remove_button;
  GtkWidget * button_hbox;
  GtkWidget * separator;
  GtkWidget * label_unsupported;
  GtkWidget * label_info;
  
  g_object_get(table, "n-rows", &gline, NULL);
  line = gline;
  
  hbox = gtk_hbox_new(FALSE, PADDING);
  
  /* account name */
  gtk_table_resize(GTK_TABLE(table), line + 1, 4);
  
  label_name = gtk_label_new(_("Filter Name"));
  gtk_widget_show(label_name);
  gtk_box_pack_start(GTK_BOX(hbox), label_name, FALSE, FALSE, 0);
  entry_name = gtk_entry_new();
  gtk_widget_show(entry_name);
  gtk_box_pack_start(GTK_BOX(hbox), entry_name, TRUE, TRUE, 0);
  
  gtk_widget_show(hbox);
  gtk_table_attach(GTK_TABLE(table), hbox, 0, 4, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  etpan_preferences_panel_set_widget(panel, "label-name", label_name);
  etpan_preferences_panel_set_widget(panel, "filter-name", entry_name);
  g_signal_connect(entry_name, "changed",
      G_CALLBACK(entry_changed), panel);
  gtk_table_resize(GTK_TABLE(table), line + 1, 4);
  separator = gtk_hseparator_new();
  gtk_widget_show(separator);
  gtk_table_attach(GTK_TABLE(table), separator, 0, 4, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  gtk_table_resize(GTK_TABLE(table), line + 1, 4);
  label_unsupported = gtk_label_new(_("This filter cannot be modified."));
  gtk_widget_show(label_unsupported);
  gtk_table_attach(GTK_TABLE(table), label_unsupported, 0, 4, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  etpan_preferences_panel_set_widget(panel, "label-unsupported", label_unsupported);
  
  multiple_condition_filter_properties(panel, table);
  
  g_object_get(table, "n-rows", &gline, NULL);
  line = gline;
  
  gtk_table_resize(GTK_TABLE(table), line + 1, 4);
  
  condition_table = gtk_table_new(1, 4, FALSE);
  add_new_condition_ui(panel, condition_table);
  gtk_widget_show(condition_table);
  etpan_preferences_panel_set_widget(panel, "condition-table", condition_table);
  
  gtk_table_attach(GTK_TABLE(table), condition_table, 0, 4, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;

  gtk_table_resize(GTK_TABLE(table), line + 1, 4);
  
  hbox = gtk_hbox_new(FALSE, PADDING);
  
  button_hbox = gtk_hbutton_box_new();
  gtk_button_box_set_layout(GTK_BUTTON_BOX(button_hbox), GTK_BUTTONBOX_START);
  
  add_button = gtk_button_new_with_label(_("+"));
  gtk_widget_show(add_button);
  gtk_box_pack_start(GTK_BOX(button_hbox), add_button, FALSE, FALSE, 0);
  
  remove_button = gtk_button_new_with_label(_("-"));
  gtk_widget_show(remove_button);
  gtk_box_pack_start(GTK_BOX(button_hbox), remove_button, FALSE, FALSE, 0);
  
  etpan_preferences_panel_set_widget(panel, "add-condition-button", add_button);
  etpan_preferences_panel_set_widget(panel, "remove-condition-button", remove_button);
  
  gtk_widget_show(button_hbox);
  
  gtk_box_pack_start(GTK_BOX(hbox), button_hbox, TRUE, TRUE, 0);
  etpan_preferences_panel_set_widget(panel, "condition-button-hbox", button_hbox);
  
  gtk_widget_show(hbox);
  
  gtk_table_attach(GTK_TABLE(table), hbox, 0, 4, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;

  g_signal_connect(add_button,
      "clicked", G_CALLBACK(add_condition_pressed_handler),
      (gpointer) panel);
  g_signal_connect(remove_button,
      "clicked", G_CALLBACK(remove_condition_pressed_handler),
      (gpointer) panel);
  
  gtk_table_resize(GTK_TABLE(table), line + 1, 4);
  separator = gtk_hseparator_new();
  gtk_widget_show(separator);
  gtk_table_attach(GTK_TABLE(table), separator, 0, 4, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  etpan_preferences_panel_set_widget(panel, "condition-action-separator", separator);
  
  multiple_action_filter_properties(panel, table);

  g_object_get(table, "n-rows", &gline, NULL);
  line = gline;
  
  gtk_table_resize(GTK_TABLE(table), line + 1, 4);
  
  action_table = gtk_table_new(1, 4, FALSE);
  add_new_action_ui(panel, action_table);
  gtk_widget_show(action_table);
  etpan_preferences_panel_set_widget(panel, "action-table", action_table);
  
  gtk_table_attach(GTK_TABLE(table), action_table, 0, 4, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;

  gtk_table_resize(GTK_TABLE(table), line + 1, 4);
  
  hbox = gtk_hbox_new(FALSE, PADDING);
  
  button_hbox = gtk_hbutton_box_new();
  gtk_button_box_set_layout(GTK_BUTTON_BOX(button_hbox), GTK_BUTTONBOX_START);
  
  add_button = gtk_button_new_with_label(_("+"));
  gtk_widget_show(add_button);
  gtk_box_pack_start(GTK_BOX(button_hbox), add_button, FALSE, FALSE, 0);
  
  remove_button = gtk_button_new_with_label(_("-"));
  gtk_widget_show(remove_button);
  gtk_box_pack_start(GTK_BOX(button_hbox), remove_button, FALSE, FALSE, 0);
  
  etpan_preferences_panel_set_widget(panel, "add-action-button", add_button);
  etpan_preferences_panel_set_widget(panel, "remove-action-button", remove_button);
  
  gtk_widget_show(button_hbox);
  
  gtk_box_pack_start(GTK_BOX(hbox), button_hbox, TRUE, TRUE, 0);
  etpan_preferences_panel_set_widget(panel, "action-button-hbox", button_hbox);
  
  gtk_widget_show(hbox);
  
  gtk_table_attach(GTK_TABLE(table), hbox, 0, 4, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;

  g_signal_connect(add_button,
      "clicked", G_CALLBACK(add_action_pressed_handler),
      (gpointer) panel);
  g_signal_connect(remove_button,
      "clicked", G_CALLBACK(remove_action_pressed_handler),
      (gpointer) panel);
  
  gtk_table_resize(GTK_TABLE(table), line + 1, 2);
  label_info = gtk_label_new("");
  gtk_widget_show(label_info);
  gtk_table_attach(GTK_TABLE(table), label_info, 0, 4, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
}

static void add_new_condition_ui(struct etpan_preferences_panel * panel,
    GtkWidget * table)
{
  struct condition_action_ui * info;
  int r;
  struct panel_data * data;
  
  data = panel->data;
  data->adding_element = 1;
  info = condition_action_ui_new();
  info->panel = panel;
  filter_condition_properties(info, table);
  data = panel->data;
  r = carray_add(data->condition_ui_list, info, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  layout_filter_preferences(panel);
  data->adding_element = 0;
}

static void multiple_condition_filter_properties(struct etpan_preferences_panel * panel,
    GtkWidget * table)
{
  GtkWidget * combo_and_or;
  guint gline;
  int line;
  GtkWidget * hbox;
  GtkWidget * label;
  GtkRequisition req;
  
  g_object_get(table, "n-rows", &gline, NULL);
  line = gline;
  
  /* or/and */
  gtk_table_resize(GTK_TABLE(table), line + 1, 4);
  
  hbox = gtk_hbox_new(FALSE, PADDING);
  
  label = gtk_label_new(_("When"));
  gtk_widget_show(label);
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
  combo_and_or = gtk_combo_box_new_text();
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_and_or),
      _("Any of"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_and_or),
      _("All"));
  gtk_widget_show(combo_and_or);
  gtk_box_pack_start(GTK_BOX(hbox), combo_and_or, FALSE, FALSE, 0);
  label = gtk_label_new(_("the following conditions match:"));
  gtk_widget_show(label);
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
  gtk_widget_show(hbox);
  
  gtk_table_attach(GTK_TABLE(table), hbox, 0, 4, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  etpan_preferences_panel_set_widget(panel, "combo-and-or", combo_and_or);
  etpan_preferences_panel_set_widget(panel, "hbox-multiple", hbox);
  gtk_widget_size_request(hbox, &req);
  
  gtk_table_resize(GTK_TABLE(table), line + 1, 4);
  hbox = gtk_hbox_new(FALSE, PADDING);
  label = gtk_label_new(_("When the following condition matches:"));
  gtk_widget_show(label);
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
  gtk_widget_set_size_request(hbox, -1, req.height + 2);
  gtk_widget_show(hbox);
  gtk_table_attach(GTK_TABLE(table), hbox, 0, 4, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  etpan_preferences_panel_set_widget(panel, "hbox-one", hbox);
  
  gtk_combo_box_set_active(GTK_COMBO_BOX(combo_and_or), 0);
}

static void multiple_action_filter_properties(struct etpan_preferences_panel * panel,
    GtkWidget * table)
{
  GtkWidget * combo_and_or;
  guint gline;
  int line;
  GtkWidget * hbox;
  GtkWidget * label;
  GtkRequisition req;
  
  g_object_get(table, "n-rows", &gline, NULL);
  line = gline;
  
  /* or/and */
  gtk_table_resize(GTK_TABLE(table), line + 1, 4);
  
  hbox = gtk_hbox_new(FALSE, PADDING);
  
  label = gtk_label_new(_("Do the following actions:"));
  gtk_widget_show(label);
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
  etpan_preferences_panel_set_widget(panel, "label-action", label);
  
  gtk_widget_show(hbox);
  
  gtk_table_attach(GTK_TABLE(table), hbox, 0, 4, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
}

static void filter_action_properties(struct condition_action_ui * info,
    GtkWidget * table);

static void add_new_action_ui(struct etpan_preferences_panel * panel,
    GtkWidget * table)
{
  struct condition_action_ui * info;
  int r;
  struct panel_data * data;
  
  data = panel->data;
  data->adding_element = 1;
  info = condition_action_ui_new();
  info->panel = panel;
  filter_action_properties(info, table);
  data = panel->data;
  r = carray_add(data->action_ui_list, info, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  data->adding_element = 0;
}

static void condition_from_properties(struct condition_action_ui * info,
    GtkWidget * hbox);
static void condition_from_set(struct condition_action_ui * info,
    struct mapped_condition * condition);
static void condition_from_get(struct condition_action_ui * info,
    struct mapped_condition * condition);
static int condition_from_check_consistency(struct condition_action_ui * info);

static void condition_content_properties(struct condition_action_ui * info,
    GtkWidget * hbox);
static void condition_content_set(struct condition_action_ui * info,
    struct mapped_condition * condition);
static void condition_content_get(struct condition_action_ui * info,
    struct mapped_condition * condition);
static int condition_content_check_consistency(struct condition_action_ui * info);

static void condition_recipient_properties(struct condition_action_ui * info,
    GtkWidget * hbox);
static void condition_recipient_set(struct condition_action_ui * info,
    struct mapped_condition * condition);
static void condition_recipient_get(struct condition_action_ui * info,
    struct mapped_condition * condition);
static int condition_recipient_check_consistency(struct condition_action_ui * info);

static void condition_header_properties(struct condition_action_ui * info,
    GtkWidget * hbox);
static void condition_header_set(struct condition_action_ui * info,
    struct mapped_condition * condition);
static void condition_header_get(struct condition_action_ui * info,
    struct mapped_condition * condition);
static int condition_header_check_consistency(struct condition_action_ui * info);

static void condition_mailing_list_properties(struct condition_action_ui * info,
    GtkWidget * hbox);
static void condition_mailing_list_set(struct condition_action_ui * info,
    struct mapped_condition * condition);
static void condition_mailing_list_get(struct condition_action_ui * info,
    struct mapped_condition * condition);
static int condition_mailing_list_check_consistency(struct condition_action_ui * info);

static void condition_subject_properties(struct condition_action_ui * info,
    GtkWidget * hbox);
static void condition_subject_set(struct condition_action_ui * info,
    struct mapped_condition * condition);
static void condition_subject_get(struct condition_action_ui * info,
    struct mapped_condition * condition);
static int condition_subject_check_consistency(struct condition_action_ui * info);

static struct etpan_filter_condition * condition_no_abook_generate(struct mapped_condition * condition);

static struct etpan_filter_condition * condition_has_no_attachment_generate(struct mapped_condition * condition);

static void condition_attachment_name_properties(struct condition_action_ui * info,
    GtkWidget * hbox);
static void condition_attachment_name_set(struct condition_action_ui * info,
    struct mapped_condition * condition);
static void condition_attachment_name_get(struct condition_action_ui * info,
    struct mapped_condition * condition);
static int condition_attachment_name_check_consistency(struct condition_action_ui * info);

static void condition_account_properties(struct condition_action_ui * info,
    GtkWidget * hbox);
static void condition_account_unsetup(struct condition_action_ui * info);
static void condition_account_set(struct condition_action_ui * info,
    struct mapped_condition * condition);
static void condition_account_get(struct condition_action_ui * info,
    struct mapped_condition * condition);

struct etpan_filter_condition_ui_description condition_ui_description_list[] = {
  {"from", "Sender", condition_from_properties, NULL, condition_from_get, condition_from_set, NULL, condition_from_check_consistency},
  {"recipient", "Recipient", condition_recipient_properties, NULL, condition_recipient_get, condition_recipient_set, NULL, condition_recipient_check_consistency},
  {"subject", "Subject", condition_subject_properties, NULL, condition_subject_get, condition_subject_set, NULL, condition_subject_check_consistency},
  {"abook", "Sender is in Address Book", NULL, NULL, NULL, NULL, NULL, NULL},
  {"no_abook", "Sender is Not in Address Book", NULL, NULL, NULL, NULL, condition_no_abook_generate, NULL},
  {"mailing_list", "Mailing-List", condition_mailing_list_properties, NULL, condition_mailing_list_get, condition_mailing_list_set, NULL, condition_mailing_list_check_consistency},
  {NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL}, 
  {"content", "Message Content", condition_content_properties, NULL, condition_content_get, condition_content_set, NULL, condition_content_check_consistency},
  {"header", "Header", condition_header_properties, NULL, condition_header_get, condition_header_set, NULL, condition_header_check_consistency},
  {NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL},
  {"true", "All Messages", NULL, NULL, NULL, NULL, NULL, NULL},
  {"account", "Account", condition_account_properties, condition_account_unsetup, condition_account_get, condition_account_set, NULL, NULL},
  {NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL},
  {"has_attachment", "Has Attachment", NULL, NULL, NULL, NULL, NULL, NULL},
  {"has_no_attachment", "Has No Attachment", NULL, NULL, NULL, NULL, condition_has_no_attachment_generate, NULL},
  {"attachment_name", "Attachment Name", condition_attachment_name_properties, NULL, condition_attachment_name_get, condition_attachment_name_set, NULL, condition_attachment_name_check_consistency},
};

static struct etpan_filter_condition * condition_no_abook_generate(struct mapped_condition * mapped)
{
  struct etpan_filter_condition * condition;
  struct etpan_filter_condition * child;
  carray * param_list;
  
  param_list = carray_new(4);
  if (param_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  condition = etpan_filter_condition_new("not", param_list);
  param_list = carray_new(4);
  if (param_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  child = etpan_filter_condition_new("abook", param_list);
  etpan_filter_condition_add(condition, child);
  
  return condition;
}

static struct etpan_filter_condition * condition_has_no_attachment_generate(struct mapped_condition * mapped)
{
  struct etpan_filter_condition * condition;
  struct etpan_filter_condition * child;
  carray * param_list;
  
  param_list = carray_new(4);
  if (param_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  condition = etpan_filter_condition_new("not", param_list);
  param_list = carray_new(4);
  if (param_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  child = etpan_filter_condition_new("has_attachment", param_list);
  etpan_filter_condition_add(condition, child);
  
  return condition;
}

static struct etpan_filter_condition * condition_generate(struct mapped_condition * mapped)
{
  char * description_name;
  struct etpan_filter_condition * condition;
  int description_index;
  struct etpan_filter_condition_ui_description * description;
  carray * param_list;
  
  description_name = mapped->name;
  description = get_condition_description(description_name);
  if (description == NULL) {
    ETPAN_CRASH("should not reach");
  }
  
  if (description->condition_generate != NULL) {
    condition = description->condition_generate(mapped);
    
    return condition;
  }
  else {
    param_list = mapped->param_list;
    param_list = param_list_dup(param_list);
    condition = etpan_filter_condition_new(description_name, param_list);
    
    return condition;
  }
}

static gboolean combo_text_item_is_separator(GtkTreeModel * model,
    GtkTreeIter * iter,
    struct etpan_preferences_panel * panel)
{
  GValue value;
  const gchar * str;
  
  memset(&value, 0, sizeof(value));
  gtk_tree_model_get_value(model, iter, 0, &value);
  str = g_value_get_string(&value);
  if (str == NULL)
    return FALSE;
  
  if (strcmp(str, "--") == 0) {
    return TRUE;
  }
  
  return FALSE;
}

static struct condition_action_ui * condition_action_ui_new(void)
{
  struct condition_action_ui * info;
  
  info = malloc(sizeof(* info));
  if (info == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  info->is_new = 1;
  info->widget_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (info->widget_hash == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  info->panel = NULL;
  
  return info;
}

static void condition_action_ui_free(struct condition_action_ui * info)
{
  chash_free(info->widget_hash);
  free(info);
}

static void condition_action_ui_set(struct condition_action_ui * info, char * name,
    GtkWidget * widget)
{
  chashdatum key;
  chashdatum value;
  int r;
  
  key.data = name;
  key.len = strlen(name) + 1;
  value.data = widget;
  value.len = 0;
  r = chash_set(info->widget_hash, &key, &value, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

static GtkWidget * condition_action_ui_get(struct condition_action_ui * info, char * name)
{
  chashdatum key;
  chashdatum value;
  int r;
  
  key.data = name;
  key.len = strlen(name) + 1;
  r = chash_get(info->widget_hash, &key, &value);
  if (r < 0)
    return NULL;
  
  return value.data;
}

static void match_kind_handler(GtkComboBox * widget, gpointer user_data);

static void filter_condition_properties(struct condition_action_ui * info,
    GtkWidget * table)
{
  struct etpan_preferences_panel * panel;
  guint gline;
  int line;
  char * name;
  GtkWidget * combo_match_type;
  unsigned int i;
  GtkWidget * radio_button;
  GtkWidget * hbox;
  GtkWidget * vbox;
  GtkWidget * radio_button_group;
  GtkTreeModel * combo_model;
  GtkWidget * alert_image;
  
  panel = info->panel;
  
  g_object_get(table, "n-rows", &gline, NULL);
  line = gline;
  
  gtk_table_resize(GTK_TABLE(table), line + 1, 4);
  
#define ICON(name) \
  etpan_icon_manager_new_scaled_image(etpan_icon_manager_get_default(), \
      name, 24)
  alert_image = ICON("alert");
  gtk_widget_hide(alert_image);
  condition_action_ui_set(info, "alert-image", alert_image);
#undef ICON
  gtk_table_attach(GTK_TABLE(table), alert_image, 0, 1, line, line + 1,
      GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  
  hbox = gtk_hbox_new(FALSE, PADDING);
  
  radio_button_group = etpan_preferences_panel_get_widget(panel, "selection-radio-button");
  radio_button = gtk_radio_button_new(NULL);
  condition_action_ui_set(info, "radio-button", radio_button);
  if (radio_button_group != NULL) {
    GSList * group;
    
    group = gtk_radio_button_get_group(GTK_RADIO_BUTTON(radio_button_group));
    gtk_radio_button_set_group(GTK_RADIO_BUTTON(radio_button), group);
  }
  else {
    etpan_preferences_panel_set_widget(panel, "selection-radio-button",
        radio_button);
  }
  
  gtk_widget_show(radio_button);
  gtk_box_pack_start(GTK_BOX(hbox), radio_button, FALSE, FALSE, 0);
  
  combo_match_type = gtk_combo_box_new_text();
  combo_model = gtk_combo_box_get_model(GTK_COMBO_BOX(combo_match_type));
  gtk_combo_box_set_row_separator_func(GTK_COMBO_BOX(combo_match_type),
      (GtkTreeViewRowSeparatorFunc) combo_text_item_is_separator, panel, NULL);
  
  for(i = 0 ; i < sizeof(condition_ui_description_list) / sizeof(condition_ui_description_list[0]) ; i ++) {
    if (condition_ui_description_list[i].name == NULL) {
      gtk_combo_box_append_text(GTK_COMBO_BOX(combo_match_type), "--");
    }
    else {
      gtk_combo_box_append_text(GTK_COMBO_BOX(combo_match_type),
          _(condition_ui_description_list[i].localization_key));
    }
  }
  gtk_widget_show(combo_match_type);
  gtk_box_pack_start(GTK_BOX(hbox), combo_match_type, FALSE, FALSE, 0);
  
  vbox = gtk_vbox_new(FALSE, PADDING);
  for(i = 0 ; i < sizeof(condition_ui_description_list) / sizeof(condition_ui_description_list[0]) ; i ++) {
    GtkWidget * sub_hbox;
    
    if (condition_ui_description_list[i].name == NULL)
      continue;
    
    sub_hbox = gtk_hbox_new(FALSE, PADDING);
    
    if (condition_ui_description_list[i].condition_action_properties != NULL) {
      condition_ui_description_list[i].condition_action_properties(info, sub_hbox);
    }
    
    gtk_widget_show(sub_hbox);
    
    gtk_box_pack_start(GTK_BOX(vbox), sub_hbox, FALSE, FALSE, 0);
    
    condition_action_ui_set(info, condition_ui_description_list[i].name,
        sub_hbox);
  }
  gtk_widget_show(vbox);
  gtk_box_pack_start(GTK_BOX(hbox), vbox, TRUE, TRUE, 0);
  
  gtk_widget_show(hbox);
  
  condition_action_ui_set(info, "main", hbox);
  condition_action_ui_set(info, "combo-type", combo_match_type);
  
  gtk_table_attach(GTK_TABLE(table), hbox, 1, 4, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  g_signal_connect(combo_match_type,
      "changed", G_CALLBACK(condition_action_combo_changed),
      (gpointer) info);
  g_signal_connect(combo_match_type,
      "changed", G_CALLBACK(match_kind_handler),
      (gpointer) info);
  gtk_combo_box_set_active(GTK_COMBO_BOX(combo_match_type), 0);
  match_kind_handler(GTK_COMBO_BOX(combo_match_type), info);
}

static void filter_condition_unsetup(struct condition_action_ui * info)
{
  unsigned int i;
  
  for(i = 0 ; i < sizeof(condition_ui_description_list) / sizeof(condition_ui_description_list[0]) ; i ++) {
    if (condition_ui_description_list[i].condition_unsetup != NULL) {
      condition_ui_description_list[i].condition_unsetup(info);
    }
  }
}

static void match_kind_handler(GtkComboBox * combo, gpointer user_data)
{
  struct condition_action_ui * info;
  unsigned int i;
  GtkWidget * widget;
  GtkWidget * tmp_combo;
  GtkRequisition req;
  unsigned int selected_index;
  char * name;
  
  info = user_data;
  
  for(i = 0 ; i < sizeof(condition_ui_description_list) / sizeof(condition_ui_description_list[0]) ; i ++) {
    if (condition_ui_description_list[i].name == NULL)
      continue;
    
    widget = condition_action_ui_get(info, condition_ui_description_list[i].name);
    gtk_widget_hide(widget);
  }
  
  selected_index = gtk_combo_box_get_active(combo);
  name = condition_ui_description_list[selected_index].name;
  if (name != NULL) {
    widget = condition_action_ui_get(info, name);
    gtk_widget_show(widget);
  }
  
  tmp_combo = gtk_combo_box_new_text();
  gtk_combo_box_append_text(GTK_COMBO_BOX(tmp_combo), gtk_combo_box_get_active_text(combo));
  gtk_widget_show(tmp_combo);
  gtk_widget_size_request(tmp_combo, &req);
  gtk_widget_destroy(tmp_combo);
  
  gtk_widget_set_size_request(GTK_WIDGET(combo), req.width, -1);
}

static void condition_match_free_text_properties(struct condition_action_ui * info,
    GtkWidget * hbox, char * combo_name, char * entry_name)
{
  GtkWidget * combo_match_type;
  GtkWidget * entry_value;
  
  /* match type */
  combo_match_type = gtk_combo_box_new_text();
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_match_type),
      _("Contains"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_match_type),
      _("Does not contain"));
  gtk_widget_show(combo_match_type);
  gtk_box_pack_start(GTK_BOX(hbox), combo_match_type, FALSE, FALSE, 0);
  condition_action_ui_set(info, combo_name, combo_match_type);
  gtk_combo_box_set_active(GTK_COMBO_BOX(combo_match_type), 0);
  g_signal_connect(combo_match_type, "changed",
      G_CALLBACK(condition_action_combo_changed), info);
  
  entry_value = gtk_entry_new();
  gtk_widget_show(entry_value);
  gtk_box_pack_start(GTK_BOX(hbox), entry_value, TRUE, TRUE, 0);
  condition_action_ui_set(info, entry_name, entry_value);
  
  g_signal_connect(entry_value, "changed",
      G_CALLBACK(condition_action_entry_changed), info);
}

static void condition_match_free_text_set(struct condition_action_ui * info,
    struct mapped_condition * condition,
    unsigned int combo_index, char * combo_name,
    unsigned int entry_index, char * entry_name)
{
  char * str;
  struct etpan_serialize_data * data;
  GtkWidget * combo_match_type;
  carray * param_list;
  GtkWidget * entry_value;
  gint match_type_index;
  
  param_list = condition->param_list;
  data = carray_get(param_list, 0);
  str = etpan_serialize_data_get_str(data);
  
  match_type_index = 0;
  if (strcasecmp(str, "contains") == 0)
    match_type_index = 0;
  else if (strcasecmp(str, "doesnotcontain") == 0)
    match_type_index = 1;
  
  combo_match_type = condition_action_ui_get(info, combo_name);
  gtk_combo_box_set_active(GTK_COMBO_BOX(combo_match_type), match_type_index);
  
  data = carray_get(param_list, 1);
  str = etpan_serialize_data_get_str(data);
  
  entry_value = condition_action_ui_get(info, entry_name);
  gtk_entry_set_text(GTK_ENTRY(entry_value), str);
}

static void condition_match_free_text_get(struct condition_action_ui * info,
    struct mapped_condition * condition,
    unsigned int combo_index, char * combo_name,
    unsigned int entry_index, char * entry_name)
{
  GtkWidget * combo_match_type;
  GtkWidget * entry_value;
  gint match_type_index;
  char * str;
  int r;
  struct etpan_serialize_data * data;
  carray * param_list;
  
  param_list = condition->param_list;
  combo_match_type = condition_action_ui_get(info, combo_name);
  
  match_type_index = gtk_combo_box_get_active(GTK_COMBO_BOX(combo_match_type));
  switch (match_type_index) {
  case 0:
    str = "contains";
    break;
  case 1:
    str = "doesnotcontain";
    break;
  default:
    str = "contains";
    break;
  }
  
  data = etpan_serialize_data_new_str(str);
  r = carray_add(param_list, data, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  entry_value = condition_action_ui_get(info, entry_name);
  str = (char *) gtk_entry_get_text(GTK_ENTRY(entry_value));
  data = etpan_serialize_data_new_str(str);
  r = carray_add(param_list, data, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}


static void condition_match_text_properties(struct condition_action_ui * info,
    GtkWidget * hbox, char * combo_name, char * entry_name)
{
  GtkWidget * combo_match_type;
  GtkWidget * entry_value;
  
  /* match type */
  combo_match_type = gtk_combo_box_new_text();
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_match_type),
      _("Contains"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_match_type),
      _("Does not contain"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_match_type),
      _("Equals"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_match_type),
      _("Is not equal to"));
  gtk_widget_show(combo_match_type);
  gtk_box_pack_start(GTK_BOX(hbox), combo_match_type, FALSE, FALSE, 0);
  condition_action_ui_set(info, combo_name, combo_match_type);
  gtk_combo_box_set_active(GTK_COMBO_BOX(combo_match_type), 0);
  g_signal_connect(combo_match_type, "changed",
      G_CALLBACK(condition_action_combo_changed), info);
  
  entry_value = gtk_entry_new();
  gtk_widget_show(entry_value);
  gtk_box_pack_start(GTK_BOX(hbox), entry_value, TRUE, TRUE, 0);
  condition_action_ui_set(info, entry_name, entry_value);
  
  g_signal_connect(entry_value, "changed",
      G_CALLBACK(condition_action_entry_changed), info);
}

static void condition_match_text_set(struct condition_action_ui * info,
    struct mapped_condition * condition,
    unsigned int combo_index, char * combo_name,
    unsigned int entry_index, char * entry_name)
{
  char * str;
  struct etpan_serialize_data * data;
  GtkWidget * combo_match_type;
  carray * param_list;
  GtkWidget * entry_value;
  gint match_type_index;
  
  param_list = condition->param_list;
  data = carray_get(param_list, combo_index);
  str = etpan_serialize_data_get_str(data);
  
  match_type_index = 0;
  if (strcasecmp(str, "contains") == 0)
    match_type_index = 0;
  else if (strcasecmp(str, "doesnotcontain") == 0)
    match_type_index = 1;
  else if (strcasecmp(str, "equals") == 0)
    match_type_index = 2;
  else if (strcasecmp(str, "isnotequalto") == 0)
    match_type_index = 3;
  
  combo_match_type = condition_action_ui_get(info, combo_name);
  gtk_combo_box_set_active(GTK_COMBO_BOX(combo_match_type), match_type_index);
  
  data = carray_get(param_list, entry_index);
  str = etpan_serialize_data_get_str(data);
  
  entry_value = condition_action_ui_get(info, entry_name);
  gtk_entry_set_text(GTK_ENTRY(entry_value), str);
}

static void condition_match_text_get(struct condition_action_ui * info,
    struct mapped_condition * condition,
    unsigned int combo_index, char * combo_name,
    unsigned int entry_index, char * entry_name)
{
  GtkWidget * combo_match_type;
  GtkWidget * entry_value;
  gint match_type_index;
  char * str;
  int r;
  struct etpan_serialize_data * data;
  carray * param_list;
  
  param_list = condition->param_list;
  combo_match_type = condition_action_ui_get(info, combo_name);
  
  match_type_index = gtk_combo_box_get_active(GTK_COMBO_BOX(combo_match_type));
  switch (match_type_index) {
  case 0:
    str = "contains";
    break;
  case 1:
    str = "doesnotcontain";
    break;
  case 2:
    str = "equals";
    break;
  case 3:
    str = "isnotequalto";
    break;
  default:
    str = "contains";
    break;
  }
  
  data = etpan_serialize_data_new_str(str);
  carray_set(param_list, combo_index, data);
  
  entry_value = condition_action_ui_get(info, entry_name);
  str = (char *) gtk_entry_get_text(GTK_ENTRY(entry_value));
  data = etpan_serialize_data_new_str(str);
  carray_set(param_list, entry_index, data);
}

static void condition_from_properties(struct condition_action_ui * info,
    GtkWidget * hbox)
{
  condition_match_text_properties(info, hbox, "from-match-type", "from-entry");
}

static void condition_from_set(struct condition_action_ui * info,
    struct mapped_condition * condition)
{
  condition_match_text_set(info, condition, 0, "from-match-type", 1, "from-entry");
}

static void condition_from_get(struct condition_action_ui * info,
    struct mapped_condition * condition)
{
  carray * param_list;
  int r;
  
  param_list = condition->param_list;
  r = carray_add(param_list, NULL, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  r = carray_add(param_list, NULL, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  condition_match_text_get(info, condition, 0, "from-match-type", 1, "from-entry");
}

static int condition_entry_check_consistency(struct condition_action_ui * info, char * name)
{
  GtkWidget * from_entry;
  char * str;
  
  from_entry = condition_action_ui_get(info, name);
  str = (char *) gtk_entry_get_text(GTK_ENTRY(from_entry));
  if (str[0] == '\0') {
    return 0;
  }
  
  return 1;
}

static int condition_from_check_consistency(struct condition_action_ui * info)
{
  return condition_entry_check_consistency(info, "from-entry");
}

static void condition_content_properties(struct condition_action_ui * info,
    GtkWidget * hbox)
{
  condition_match_free_text_properties(info, hbox, "content-match-type", "content-entry");
}

static void condition_content_set(struct condition_action_ui * info,
    struct mapped_condition * condition)
{
  condition_match_free_text_set(info, condition, 0, "content-match-type", 1, "content-entry");
}

static void condition_content_get(struct condition_action_ui * info,
    struct mapped_condition * condition)
{
  carray * param_list;
  int r;
  
  param_list = condition->param_list;
  r = carray_add(param_list, NULL, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  r = carray_add(param_list, NULL, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  condition_match_free_text_get(info, condition, 0, "content-match-type", 1, "content-entry");
}

static int condition_content_check_consistency(struct condition_action_ui * info)
{
  return condition_entry_check_consistency(info, "content-entry");
}

static void condition_recipient_properties(struct condition_action_ui * info,
    GtkWidget * hbox)
{
  condition_match_text_properties(info, hbox, "recipient-match-type", "recipient-entry");
}

static void condition_recipient_set(struct condition_action_ui * info,
    struct mapped_condition * condition)
{
  condition_match_text_set(info, condition, 0, "recipient-match-type", 1, "recipient-entry");
}

static void condition_recipient_get(struct condition_action_ui * info,
    struct mapped_condition * condition)
{
  carray * param_list;
  int r;
  
  param_list = condition->param_list;
  r = carray_add(param_list, NULL, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  r = carray_add(param_list, NULL, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  condition_match_text_get(info, condition, 0, "recipient-match-type", 1, "recipient-entry");
}

static int condition_recipient_check_consistency(struct condition_action_ui * info)
{
  return condition_entry_check_consistency(info, "recipient-entry");
}

static void condition_header_properties(struct condition_action_ui * info,
    GtkWidget * hbox)
{
  GtkWidget * entry_header;
  GtkWidget * combo_match_type;
  GtkWidget * entry_value;
  
  entry_header = gtk_entry_new();
  gtk_entry_set_width_chars(GTK_ENTRY(entry_header), 10);
  gtk_widget_show(entry_header);
  gtk_box_pack_start(GTK_BOX(hbox), entry_header, FALSE, FALSE, 0);
  condition_action_ui_set(info, "header-name", entry_header);
  
  g_signal_connect(entry_header, "changed",
      G_CALLBACK(condition_action_entry_changed), info);
  
  condition_match_text_properties(info, hbox, "header-match-type", "header-value");
}

static void condition_header_set(struct condition_action_ui * info,
    struct mapped_condition * condition)
{
  char * str;
  struct etpan_serialize_data * data;
  GtkWidget * combo_match_type;
  carray * param_list;
  GtkWidget * entry_value;
  gint match_type_index;
  
  param_list = condition->param_list;
  
  data = carray_get(param_list, 1);
  str = etpan_serialize_data_get_str(data);
  
  entry_value = condition_action_ui_get(info, "header-name");
  gtk_entry_set_text(GTK_ENTRY(entry_value), str);
  
  condition_match_text_set(info, condition, 0, "header-match-type", 2, "header-value");
}

static void condition_header_get(struct condition_action_ui * info,
    struct mapped_condition * condition)
{
  GtkWidget * combo_match_type;
  GtkWidget * entry_value;
  gint match_type_index;
  char * str;
  int r;
  struct etpan_serialize_data * data;
  carray * param_list;
  
  param_list = condition->param_list;
  r = carray_add(param_list, NULL, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  r = carray_add(param_list, NULL, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  r = carray_add(param_list, NULL, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  entry_value = condition_action_ui_get(info, "header-name");
  str = (char *) gtk_entry_get_text(GTK_ENTRY(entry_value));
  data = etpan_serialize_data_new_str(str);
  carray_set(param_list, 1, data);
  
  condition_match_text_get(info, condition, 0, "header-match-type", 2, "header-value");
}

static int condition_header_check_consistency(struct condition_action_ui * info)
{
  return condition_entry_check_consistency(info, "header-value");
}

static void condition_mailing_list_properties(struct condition_action_ui * info,
    GtkWidget * hbox)
{
  condition_match_text_properties(info, hbox, "mailing-list-match-type", "mailing-list-entry");
}

static void condition_mailing_list_set(struct condition_action_ui * info,
    struct mapped_condition * condition)
{
  condition_match_text_set(info, condition, 0, "mailing-list-match-type", 1, "mailing-list-entry");
}

static void condition_mailing_list_get(struct condition_action_ui * info,
    struct mapped_condition * condition)
{
  carray * param_list;
  int r;
  
  param_list = condition->param_list;
  r = carray_add(param_list, NULL, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  r = carray_add(param_list, NULL, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  condition_match_text_get(info, condition, 0, "recipient-match-type", 1, "recipient-entry");
}

static int condition_mailing_list_check_consistency(struct condition_action_ui * info)
{
  return condition_entry_check_consistency(info, "recipient-entry");
}

static void condition_subject_properties(struct condition_action_ui * info,
    GtkWidget * hbox)
{
  condition_match_free_text_properties(info, hbox, "subject-match-type", "subject-entry");
}

static void condition_subject_set(struct condition_action_ui * info,
    struct mapped_condition * condition)
{
  carray * param_list;
  int r;
  
  param_list = condition->param_list;
  r = carray_add(param_list, NULL, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  r = carray_add(param_list, NULL, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  condition_match_free_text_set(info, condition, 0, "subject-match-type", 1, "subject-entry");
}

static void condition_subject_get(struct condition_action_ui * info,
    struct mapped_condition * condition)
{
  carray * param_list;
  int r;
  
  param_list = condition->param_list;
  r = carray_add(param_list, NULL, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  r = carray_add(param_list, NULL, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  condition_match_free_text_get(info, condition, 0, "subject-match-type", 1, "subject-entry");
}

static int condition_subject_check_consistency(struct condition_action_ui * info)
{
  return condition_entry_check_consistency(info, "subject-entry");
}

static void condition_attachment_name_properties(struct condition_action_ui * info,
    GtkWidget * hbox)
{
  condition_match_text_properties(info, hbox, "attachment-name-match-type", "attachment-name-entry");
}

static void condition_attachment_name_set(struct condition_action_ui * info,
    struct mapped_condition * condition)
{
  condition_match_text_set(info, condition, 0, "attachment-name-match-type", 1, "attachment-name-entry");
}

static void condition_attachment_name_get(struct condition_action_ui * info,
    struct mapped_condition * condition)
{
  carray * param_list;
  int r;
  
  param_list = condition->param_list;
  r = carray_add(param_list, NULL, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  r = carray_add(param_list, NULL, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  condition_match_text_get(info, condition, 0, "attachment-match-type", 1, "attachment-name-entry");
}

static int condition_attachment_name_check_consistency(struct condition_action_ui * info)
{
  return condition_entry_check_consistency(info, "attachment-name-entry");
}

static void account_list_updated(char * signal_name,
    void * sender, void * signal_data, void * user_data);

static void condition_account_properties(struct condition_action_ui * info,
    GtkWidget * hbox)
{
  GtkWidget * combo_account;
  carray * account_list;
  unsigned int i;
  struct panel_data * data;
  GtkCellRenderer * cell;
  struct etpan_preferences_panel * panel;
  
  ETPAN_SIGNAL_ADD_HANDLER(etpan_account_manager_get_default(),
      ETPAN_ACCOUNT_MANAGER_MODIFICATION_SIGNAL, account_list_updated,
      info);
  
  panel = info->panel;
  data = panel->data;
  g_object_ref(data->account_treemodel);
  combo_account = gtk_combo_box_new_with_model(GTK_TREE_MODEL(data->account_treemodel));
  cell = gtk_cell_renderer_text_new();
  gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(combo_account), cell, TRUE);
  gtk_cell_layout_set_attributes(GTK_CELL_LAYOUT(combo_account), cell,
      "text", 0, NULL);
  etpan_gtk_tree_model_reload(data->account_treemodel);
  
  gtk_widget_show(combo_account);
  gtk_box_pack_start(GTK_BOX(hbox), combo_account, FALSE, FALSE, 0);
  
  gtk_combo_box_set_active(GTK_COMBO_BOX(combo_account), 0);
  condition_action_ui_set(info, "account-combo", combo_account);
}

static void condition_account_unsetup(struct condition_action_ui * info)
{
  ETPAN_SIGNAL_REMOVE_HANDLER(etpan_account_manager_get_default(),
      ETPAN_ACCOUNT_MANAGER_MODIFICATION_SIGNAL, account_list_updated,
      info);
}

static void account_list_updated(char * signal_name,
    void * sender, void * signal_data, void * user_data)
{
  struct condition_action_ui * info;
  struct etpan_preferences_panel * panel;
  struct panel_data * data;
  gint selected_index;
  GtkWidget * combo_account;
  
  info = user_data;
  
  panel = info->panel;
  data = panel->data;
  etpan_gtk_tree_model_reload(data->account_treemodel);
  
  combo_account = condition_action_ui_get(info, "account-combo");
  selected_index = gtk_combo_box_get_active(GTK_COMBO_BOX(combo_account));
  if (selected_index == -1)
    gtk_combo_box_set_active(GTK_COMBO_BOX(combo_account), 0);
}

static void condition_account_set(struct condition_action_ui * info,
    struct mapped_condition * condition)
{
  carray * account_list;
  unsigned int i;
  char * str;
  struct etpan_serialize_data * data;
  carray * param_list;
  
  param_list = condition->param_list;
  
  data = carray_get(param_list, 0);
  str = etpan_serialize_data_get_str(data);
  
  account_list = etpan_account_manager_get_ordered_list(etpan_account_manager_get_default());
  for(i = 0 ; i < carray_count(account_list) ; i ++) {
    struct etpan_account * account;
    char * id;
    
    account = carray_get(account_list, i);
    id = etpan_account_get_id(account);
    if (id == NULL)
      continue;
    
    if (strcmp(id, str) == 0) {
      GtkWidget * combo_account;
      
      combo_account = condition_action_ui_get(info, "account-combo");
      gtk_combo_box_set_active(GTK_COMBO_BOX(combo_account), i);
      break;
    }
  }
}

static void condition_account_get(struct condition_action_ui * info,
    struct mapped_condition * condition)
{
  gint selected_index;
  struct etpan_serialize_data * data;
  char * str;
  carray * account_list;
  GtkWidget * combo_account;
  struct etpan_account * account;
  int r;
  carray * param_list;
  
  param_list = condition->param_list;
  combo_account = condition_action_ui_get(info, "account-combo");
  
  selected_index = gtk_combo_box_get_active(GTK_COMBO_BOX(combo_account));
  account_list = etpan_account_manager_get_ordered_list(etpan_account_manager_get_default());
  account = carray_get(account_list, selected_index);
  str = etpan_account_get_id(account);
  
  data = etpan_serialize_data_new_str(str);
  r = carray_add(param_list, data, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

static void layout_filter_preferences(struct etpan_preferences_panel * panel)
{
  GtkWidget * hbox_one;
  GtkWidget * hbox_multiple;
  struct panel_data * data;
  
  data = panel->data;
  hbox_one = etpan_preferences_panel_get_widget(panel, "hbox-one");
  hbox_multiple = etpan_preferences_panel_get_widget(panel, "hbox-multiple");
  
  if (carray_count(data->condition_ui_list) == 1) {
    gtk_widget_show(hbox_one);
    gtk_widget_hide(hbox_multiple);
  }
  else {
    gtk_widget_hide(hbox_one);
    gtk_widget_show(hbox_multiple);
  }
}

static void add_condition_pressed_handler(GtkButton *button, gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  GtkWidget * table;
  struct panel_data * data;
  
  panel = user_data;
  data = panel->data;
  data->modified = 1;
  table = etpan_preferences_panel_get_widget(panel, "condition-table");
  add_new_condition_ui(panel, table);
  enable_buttons(panel);
}

static void remove_condition_pressed_handler(GtkButton *button, gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  struct panel_data * data;
  unsigned int i;
  
  panel = user_data;
  data = panel->data;
  for(i = 0 ; i < carray_count(data->condition_ui_list) ; i ++) {
    struct condition_action_ui * info;
    GtkWidget * radio_button;
    
    info = carray_get(data->condition_ui_list, i);
    radio_button = condition_action_ui_get(info, "radio-button");
    if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(radio_button))) {
      GtkWidget * hbox;
      GtkWidget * alert_image;
      
      data->modified = 1;
      filter_condition_unsetup(info);
      hbox = condition_action_ui_get(info, "main");
      gtk_widget_destroy(hbox);
      alert_image = condition_action_ui_get(info, "alert-image");
      gtk_widget_destroy(alert_image);
      condition_action_ui_free(info);
      carray_delete_slow(data->condition_ui_list, i);
      
      if (i < carray_count(data->condition_ui_list)) {
        info = carray_get(data->condition_ui_list, i);
        radio_button = condition_action_ui_get(info, "radio-button");
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radio_button), TRUE);
        etpan_preferences_panel_set_widget(panel, "selection-radio-button",
            radio_button);
      }
      else if (i > 0) {
        info = carray_get(data->condition_ui_list, i - 1);
        radio_button = condition_action_ui_get(info, "radio-button");
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radio_button), TRUE);
        etpan_preferences_panel_set_widget(panel, "selection-radio-button",
            radio_button);
      }
      else {
        etpan_preferences_panel_set_widget(panel, "selection-radio-button",
            NULL);
      }
      
      layout_filter_preferences(panel);
      break;
    }
  }
  enable_buttons(panel);
}

static void add_action_pressed_handler(GtkButton *button, gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  GtkWidget * table;
  struct panel_data * data;
  
  panel = user_data;
  data = panel->data;
  data->modified = 1;
  table = etpan_preferences_panel_get_widget(panel, "action-table");
  add_new_action_ui(panel, table);
  enable_buttons(panel);
}

static void filter_action_unsetup(struct condition_action_ui * info);

static void remove_action_pressed_handler(GtkButton *button, gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  struct panel_data * data;
  unsigned int i;
  
  panel = user_data;
  data = panel->data;
  for(i = 0 ; i < carray_count(data->action_ui_list) ; i ++) {
    struct condition_action_ui * info;
    GtkWidget * radio_button;
    
    info = carray_get(data->action_ui_list, i);
    radio_button = condition_action_ui_get(info, "radio-button");
    if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(radio_button))) {
      GtkWidget * hbox;
      GtkWidget * alert_image;
      
      data->modified = 1;
      filter_action_unsetup(info);
      hbox = condition_action_ui_get(info, "main");
      gtk_widget_destroy(hbox);
      alert_image = condition_action_ui_get(info, "alert-image");
      gtk_widget_destroy(alert_image);
      condition_action_ui_free(info);
      carray_delete_slow(data->action_ui_list, i);
      
      if (i < carray_count(data->action_ui_list)) {
        info = carray_get(data->action_ui_list, i);
        radio_button = condition_action_ui_get(info, "radio-button");
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radio_button), TRUE);
        etpan_preferences_panel_set_widget(panel, "selection-radio-action",
            radio_button);
      }
      else if (i > 0) {
        info = carray_get(data->action_ui_list, i - 1);
        radio_button = condition_action_ui_get(info, "radio-button");
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radio_button), TRUE);
        etpan_preferences_panel_set_widget(panel, "selection-radio-action",
            radio_button);
      }
      else {
        etpan_preferences_panel_set_widget(panel, "selection-radio-action",
            NULL);
      }
      break;
    }
  }
  enable_buttons(panel);
}

static void action_kind_handler(GtkComboBox * combo, gpointer user_data);

static void action_move_properties(struct condition_action_ui * info,
    GtkWidget * hbox);
static void action_move_get(struct condition_action_ui * info, struct mapped_action * action);
static void action_move_set(struct condition_action_ui * info, struct mapped_action * action);
static void action_move_unsetup(struct condition_action_ui * info);
static int action_move_check_consistency(struct condition_action_ui * info);

static void action_copy_properties(struct condition_action_ui * info,
    GtkWidget * hbox);
static void action_copy_unsetup(struct condition_action_ui * info);
static void action_copy_get(struct condition_action_ui * info, struct mapped_action * action);
static void action_copy_set(struct condition_action_ui * info, struct mapped_action * action);
static int action_copy_check_consistency(struct condition_action_ui * info);

static void action_reply_properties(struct condition_action_ui * info,
    GtkWidget * hbox);
static void action_reply_get(struct condition_action_ui * info, struct mapped_action * action);
static void action_reply_set(struct condition_action_ui * info, struct mapped_action * action);

static void action_forward_properties(struct condition_action_ui * info,
    GtkWidget * hbox);
static void action_forward_get(struct condition_action_ui * info, struct mapped_action * action);
static void action_forward_set(struct condition_action_ui * info, struct mapped_action * action);
static int action_forward_check_consistency(struct condition_action_ui * info);

static void action_color_properties(struct condition_action_ui * info,
    GtkWidget * hbox);
static void action_color_get(struct condition_action_ui * info, struct mapped_action * action);
static void action_color_set(struct condition_action_ui * info, struct mapped_action * action);

struct etpan_filter_action_ui_description action_ui_description_list[] = {
  {"move", "Move", action_move_properties, action_move_unsetup, action_move_get, action_move_set, action_move_check_consistency},
  {"copy", "Copy", action_copy_properties, action_copy_unsetup, action_copy_get, action_copy_set, action_copy_check_consistency},
  {"archive", "Archive", NULL, NULL, NULL, NULL, NULL},
  {NULL, NULL, NULL, NULL, NULL, NULL, NULL},
  {"mark_as_seen", "Mark As Seen", NULL, NULL, NULL, NULL, NULL},
  {"mark_as_flagged", "Mark As Flagged", NULL, NULL, NULL, NULL, NULL},
  {"color", "Set Message Color", action_color_properties, NULL, action_color_get, action_color_set, NULL},
  {NULL, NULL, NULL, NULL, NULL, NULL, NULL},
  {"reply", "Reply", action_reply_properties, NULL, action_reply_get, action_reply_set, NULL},
  {"forward", "Forward", action_forward_properties, NULL, action_forward_get, action_forward_set, action_forward_check_consistency},
  {NULL, NULL, NULL, NULL, NULL, NULL, NULL},
  {"stop", "Stop Evaluating Rules", NULL, NULL, NULL, NULL, NULL},
};

static void filter_action_properties(struct condition_action_ui * info,
    GtkWidget * table)
{
  guint gline;
  int line;
  char * name;
  GtkWidget * combo_action_type;
  unsigned int i;
  GtkWidget * radio_button;
  GtkWidget * hbox;
  GtkWidget * sub_hbox;
  GtkWidget * vbox;
  GtkWidget * radio_button_group;
  GtkTreeModel * combo_model;
  struct etpan_preferences_panel * panel;
  GtkWidget * alert_image;
  
  panel = info->panel;
  
  g_object_get(table, "n-rows", &gline, NULL);
  line = gline;
  
  gtk_table_resize(GTK_TABLE(table), line + 1, 4);
  
#define ICON(name) \
  etpan_icon_manager_new_scaled_image(etpan_icon_manager_get_default(), \
      name, 24)
  alert_image = ICON("alert");
  gtk_widget_hide(alert_image);
  condition_action_ui_set(info, "alert-image", alert_image);
#undef ICON
  gtk_table_attach(GTK_TABLE(table), alert_image, 0, 1, line, line + 1,
      GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  
  hbox = gtk_hbox_new(FALSE, PADDING);
  
  vbox = gtk_vbox_new(FALSE, PADDING);
  
  sub_hbox = gtk_hbox_new(FALSE, PADDING);
  
  radio_button_group = etpan_preferences_panel_get_widget(panel, "selection-radio-action");
  radio_button = gtk_radio_button_new(NULL);
  condition_action_ui_set(info, "radio-button", radio_button);
  if (radio_button_group != NULL) {
    GSList * group;
    
    group = gtk_radio_button_get_group(GTK_RADIO_BUTTON(radio_button_group));
    gtk_radio_button_set_group(GTK_RADIO_BUTTON(radio_button), group);
  }
  else {
    etpan_preferences_panel_set_widget(panel, "selection-radio-action",
        radio_button);
  }
  
  gtk_widget_show(radio_button);
  gtk_box_pack_start(GTK_BOX(sub_hbox), radio_button, FALSE, FALSE, 0);
  
  combo_action_type = gtk_combo_box_new_text();
  combo_model = gtk_combo_box_get_model(GTK_COMBO_BOX(combo_action_type));
  gtk_combo_box_set_row_separator_func(GTK_COMBO_BOX(combo_action_type),
      (GtkTreeViewRowSeparatorFunc) combo_text_item_is_separator, panel, NULL);
  
  for(i = 0 ; i < sizeof(action_ui_description_list) / sizeof(action_ui_description_list[0]) ; i ++) {
    if (action_ui_description_list[i].name == NULL) {
      gtk_combo_box_append_text(GTK_COMBO_BOX(combo_action_type), "--");
    }
    else {
      gtk_combo_box_append_text(GTK_COMBO_BOX(combo_action_type),
          _(action_ui_description_list[i].localization_key));
    }
  }
  gtk_widget_show(combo_action_type);
  gtk_box_pack_start(GTK_BOX(sub_hbox), combo_action_type, FALSE, FALSE, 0);
  gtk_widget_show(sub_hbox);
  gtk_box_pack_start(GTK_BOX(vbox), sub_hbox, FALSE, FALSE, 0);
  gtk_widget_show(vbox);
  gtk_box_pack_start(GTK_BOX(hbox), vbox, FALSE, FALSE, 0);
  
  vbox = gtk_vbox_new(FALSE, PADDING);
  for(i = 0 ; i < sizeof(action_ui_description_list) / sizeof(action_ui_description_list[0]) ; i ++) {
    if (action_ui_description_list[i].name == NULL)
      continue;
    
    sub_hbox = gtk_hbox_new(FALSE, PADDING);
    
    if (action_ui_description_list[i].condition_action_properties != NULL) {
      action_ui_description_list[i].condition_action_properties(info, sub_hbox);
    }
    
    gtk_widget_show(sub_hbox);
    
    gtk_box_pack_start(GTK_BOX(vbox), sub_hbox, FALSE, FALSE, 0);
    
    condition_action_ui_set(info, action_ui_description_list[i].name,
        sub_hbox);
  }
  gtk_widget_show(vbox);
  gtk_box_pack_start(GTK_BOX(hbox), vbox, TRUE, TRUE, 0);
  
  gtk_widget_show(hbox);
  
  condition_action_ui_set(info, "main", hbox);
  condition_action_ui_set(info, "combo-type", combo_action_type);
  
  gtk_table_attach(GTK_TABLE(table), hbox, 1, 4, line, line + 1,
      GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, /* attach options */
      PADDING, PADDING);
  line ++;
  
  g_signal_connect(combo_action_type,
      "changed", G_CALLBACK(condition_action_combo_changed),
      (gpointer) info);
  g_signal_connect(combo_action_type,
      "changed", G_CALLBACK(action_kind_handler),
      (gpointer) info);
  gtk_combo_box_set_active(GTK_COMBO_BOX(combo_action_type), 0);
  action_kind_handler(GTK_COMBO_BOX(combo_action_type), info);
}

static void filter_action_unsetup(struct condition_action_ui * info)
{
  unsigned int i;
  
  for(i = 0 ; i < sizeof(action_ui_description_list) / sizeof(action_ui_description_list[0]) ; i ++) {
    if (action_ui_description_list[i].action_unsetup != NULL) {
      action_ui_description_list[i].action_unsetup(info);
    }
  }
}

static void update_folder_list(struct condition_action_ui * info,
    GtkComboBox * combo_folder)
{
  carray * account_list;
  unsigned int i;
  GtkTreeModel * model;
  unsigned int count;
  
  model = gtk_combo_box_get_model(combo_folder);
  
  count = gtk_tree_model_iter_n_children(model, NULL);
  while (1) {
    if (count == 0)
      break;
    
    count --;
    gtk_combo_box_remove_text(combo_folder, count);
  }
  
  gtk_combo_box_append_text(GTK_COMBO_BOX(combo_folder), _("No mailbox selected"));
  account_list = etpan_account_manager_get_ordered_list(etpan_account_manager_get_default());
  for(i = 0 ; i < carray_count(account_list) ; i ++) {
    struct etpan_account * account;
    struct etpan_storage * storage;
    struct etpan_storage_folder_order * folder_order;
    carray * list;
    unsigned int k;
    
    account = carray_get(account_list, i);
    storage = etpan_account_get_storage(account);
    
    folder_order = etpan_mail_manager_storage_get_order(etpan_mail_manager_get_default(), storage);
    
    list = etpan_storage_folder_order_get_folder_order(folder_order);
    
    for(k = 0 ; k < carray_count(list) ; k ++) {
      char * folder_name;
      struct etpan_folder * folder;
      char * ui_path;
      
      folder_name = carray_get(list, k);
      folder = etpan_storage_folder_order_get_folder(folder_order, folder_name);
      if (folder != NULL) {
        ui_path = etpan_folder_get_ui_path(folder);
        gtk_combo_box_append_text(GTK_COMBO_BOX(combo_folder), ui_path);
      }
    }
  }
  gtk_combo_box_set_active(GTK_COMBO_BOX(combo_folder), 0);
}

static void action_copy_move_properties(struct condition_action_ui * info,
    GtkWidget * hbox, char * combo_name)
{
  GtkWidget * combo_folder;
  GtkWidget * label;
  
  label = gtk_label_new(_("to mailbox:"));
  gtk_widget_show(label);
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
  
  combo_folder = gtk_combo_box_new_text();
  gtk_widget_show(combo_folder);
  gtk_box_pack_start(GTK_BOX(hbox), combo_folder, FALSE, FALSE, 0);
  g_signal_connect(combo_folder,
      "changed", G_CALLBACK(condition_action_combo_changed),
      (gpointer) info);
  
  condition_action_ui_set(info, combo_name, combo_folder);
  
  update_folder_list(info, GTK_COMBO_BOX(combo_folder));
}

static struct etpan_folder * get_folder_for_index(unsigned int selected_index)
{
  carray * account_list;
  unsigned int i;
  unsigned int count;
  
  if (selected_index == 0)
    return NULL;
  
  count = 0;
  account_list = etpan_account_manager_get_ordered_list(etpan_account_manager_get_default());
  for(i = 0 ; i < carray_count(account_list) ; i ++) {
    struct etpan_account * account;
    struct etpan_storage * storage;
    struct etpan_storage_folder_order * folder_order;
    carray * list;
    unsigned int k;
    
    account = carray_get(account_list, i);
    storage = etpan_account_get_storage(account);
    
    folder_order = etpan_mail_manager_storage_get_order(etpan_mail_manager_get_default(), storage);
    
    list = etpan_storage_folder_order_get_folder_order(folder_order);
    
    for(k = 0 ; k < carray_count(list) ; k ++) {
      char * folder_name;
      struct etpan_folder * folder;
      
      folder_name = carray_get(list, k);
      folder = etpan_storage_folder_order_get_folder(folder_order, folder_name);
      if (selected_index - 1 == count)
        return folder;
      count ++;
    }
  }
  
  return NULL;
}

static int get_index_for_folder(struct etpan_folder * folder)
{
  carray * account_list;
  unsigned int i;
  unsigned int count;
  
  count = 1;
  account_list = etpan_account_manager_get_ordered_list(etpan_account_manager_get_default());
  for(i = 0 ; i < carray_count(account_list) ; i ++) {
    struct etpan_account * account;
    struct etpan_storage * storage;
    struct etpan_storage_folder_order * folder_order;
    carray * list;
    unsigned int k;
    
    account = carray_get(account_list, i);
    storage = etpan_account_get_storage(account);
    
    folder_order = etpan_mail_manager_storage_get_order(etpan_mail_manager_get_default(), storage);
    
    list = etpan_storage_folder_order_get_folder_order(folder_order);
    
    for(k = 0 ; k < carray_count(list) ; k ++) {
      char * folder_name;
      struct etpan_folder * current_folder;
      
      folder_name = carray_get(list, k);
      current_folder = etpan_storage_folder_order_get_folder(folder_order, folder_name);
      if (current_folder == folder)
        return count;
      count ++;
    }
  }
  
  return 0;
}

static void action_copy_move_get(struct condition_action_ui * info, struct mapped_action * action, char * combo_name)
{
  struct etpan_storage * storage;
  struct etpan_account * account;
  struct etpan_folder * folder;
  carray * param_list;
  char * location;
  char * account_name;
  gint selected_index;
  GtkWidget * combo_folder;
  struct etpan_serialize_data * data;
  int r;
  
  param_list = action->param_list;
  
  combo_folder = condition_action_ui_get(info, combo_name);
  selected_index = gtk_combo_box_get_active(GTK_COMBO_BOX(combo_folder));
  if (selected_index == -1) {
    ETPAN_LOG("state error");
    return;
  }
  
  folder = get_folder_for_index(selected_index);
  if (folder == NULL) {
    ETPAN_LOG("state error");
    return;
  }
  
  location = etpan_folder_get_location(folder);
  storage = etpan_folder_get_storage(folder);
  account = etpan_storage_get_account(storage);
  
  account_name = etpan_account_get_id(account);
  
  data = etpan_serialize_data_new_str(account_name);
  r = carray_add(param_list, data, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  data = etpan_serialize_data_new_str(location);
  r = carray_add(param_list, data, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

static void action_copy_move_set(struct condition_action_ui * info, struct mapped_action * action, char * combo_name)
{
  carray * param_list;
  char * location;
  char * account_name;
  struct etpan_account * account;
  struct etpan_storage * storage;
  struct etpan_folder * folder;
  struct etpan_serialize_data * data;
  gint selected_index;
  GtkWidget * combo_folder;
  
  combo_folder = condition_action_ui_get(info, combo_name);
  
  param_list = action->param_list;
  data = carray_get(param_list, 0);
  account_name = etpan_serialize_data_get_str(data);
  account = etpan_account_manager_get_account(etpan_account_manager_get_default(), account_name);
  if (account == NULL)
    return;
  
  storage = etpan_account_get_storage(account);
  
  data = carray_get(param_list, 1);
  location = etpan_serialize_data_get_str(data);
  folder = etpan_storage_get_folder(storage, location);
  if (folder == NULL) {
    return;
  }
  
  selected_index = get_index_for_folder(folder);
  if (selected_index == -1) {
    gtk_combo_box_set_active(GTK_COMBO_BOX(combo_folder), 0);
    return;
  }
  
  gtk_combo_box_set_active(GTK_COMBO_BOX(combo_folder), selected_index);
}


static int action_copy_move_check_consistency(struct condition_action_ui * info,
    char * combo_name)
{
  GtkWidget * combo_folder;
  gint selected_index;
  
  combo_folder = condition_action_ui_get(info, combo_name);
  selected_index = gtk_combo_box_get_active(GTK_COMBO_BOX(combo_folder));
  if ((selected_index == -1) || (selected_index == 0))
    return 0;
  
  return 1;
}

static void move_folder_list_updated_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct condition_action_ui * info;
  GtkComboBox * combo_folder;
  (void) signal_name;
  (void) signal_data;
  
  info = user_data;
  combo_folder = GTK_COMBO_BOX(condition_action_ui_get(info, "move-folder-combo"));
  update_folder_list(info, combo_folder);
}

static void action_move_properties(struct condition_action_ui * info,
    GtkWidget * hbox)
{
  ETPAN_SIGNAL_ADD_HANDLER(etpan_mail_manager_get_default(),
      ETPAN_MAIL_MANAGER_FOLDER_LIST_UPDATED_SIGNAL,
      move_folder_list_updated_handler,
      info);
  
  action_copy_move_properties(info, hbox, "move-folder-combo");
}

static void action_move_unsetup(struct condition_action_ui * info)
{
  ETPAN_SIGNAL_REMOVE_HANDLER(etpan_mail_manager_get_default(),
      ETPAN_MAIL_MANAGER_FOLDER_LIST_UPDATED_SIGNAL,
      move_folder_list_updated_handler,
      info);
}

static void action_move_get(struct condition_action_ui * info, struct mapped_action * action)
{
  action_copy_move_get(info, action, "move-folder-combo");
}

static void action_move_set(struct condition_action_ui * info, struct mapped_action * action)
{
  action_copy_move_set(info, action, "move-folder-combo");
}

static int action_move_check_consistency(struct condition_action_ui * info)
{
  return action_copy_move_check_consistency(info, "move-folder-combo");
}

static void copy_folder_list_updated_handler(char * signal_name, void * sender,
    void * signal_data, void * user_data)
{
  struct condition_action_ui * info;
  GtkComboBox * combo_folder;
  (void) signal_name;
  (void) signal_data;
  
  info = user_data;
  combo_folder = GTK_COMBO_BOX(condition_action_ui_get(info, "copy-folder-combo"));
  update_folder_list(info, combo_folder);
}

static void action_copy_properties(struct condition_action_ui * info,
    GtkWidget * hbox)
{
  ETPAN_SIGNAL_ADD_HANDLER(etpan_mail_manager_get_default(),
      ETPAN_MAIL_MANAGER_FOLDER_LIST_UPDATED_SIGNAL,
      copy_folder_list_updated_handler,
      info);
  
  action_copy_move_properties(info, hbox, "copy-folder-combo");
}

static void action_copy_unsetup(struct condition_action_ui * info)
{
  ETPAN_SIGNAL_REMOVE_HANDLER(etpan_mail_manager_get_default(),
      ETPAN_MAIL_MANAGER_FOLDER_LIST_UPDATED_SIGNAL,
      copy_folder_list_updated_handler,
      info);
}

static void action_copy_get(struct condition_action_ui * info, struct mapped_action * action)
{
  action_copy_move_get(info, action, "copy-folder-combo");
}

static void action_copy_set(struct condition_action_ui * info, struct mapped_action * action)
{
  action_copy_move_set(info, action, "copy-folder-combo");
}

static int action_copy_check_consistency(struct condition_action_ui * info)
{
  return action_copy_move_check_consistency(info, "copy-folder-combo");
}

static void action_reply_properties(struct condition_action_ui * info,
    GtkWidget * hbox)
{
  GtkWidget * text;
  GtkRequisition req;
  GtkWidget * tmp_combo;
  GtkWidget * scrolled;
  GtkWidget * vbox;
  GtkWidget * label;
  
  vbox = gtk_vbox_new(FALSE, PADDING);
  
  label = gtk_label_new(_("Include the following text:"));
  gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
  gtk_widget_show(label);
  gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
  
  scrolled = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
      GTK_POLICY_AUTOMATIC,
      GTK_POLICY_AUTOMATIC);  
  text = gtk_text_view_new();
  gtk_widget_show(text);
  g_signal_connect(gtk_text_view_get_buffer(GTK_TEXT_VIEW(text)), "changed",
      G_CALLBACK(condition_action_textbuffer_changed), info);
  
  gtk_container_add(GTK_CONTAINER(scrolled), text);  
  
  gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrolled),
      GTK_SHADOW_IN);
  gtk_widget_show(scrolled);
  
  gtk_box_pack_start(GTK_BOX(vbox), scrolled, TRUE, TRUE, 0);
  condition_action_ui_set(info, "reply-textview", text);
  
  tmp_combo = gtk_combo_box_new_text();
  gtk_widget_show(tmp_combo);
  gtk_widget_size_request(tmp_combo, &req);
  gtk_widget_destroy(tmp_combo);
  
  req.height *= 3;
  gtk_widget_set_size_request(GTK_WIDGET(text), -1, req.height);
  
  gtk_widget_show(vbox);
  gtk_box_pack_start(GTK_BOX(hbox), vbox, TRUE, TRUE, 0);
}

static void get_text_view(struct condition_action_ui * info, struct mapped_action * action, unsigned int text_index, char * name)
{
  GtkWidget * text;
  GtkTextBuffer * buffer;
  struct etpan_serialize_data * data;
  carray * param_list;
  gchar * str;
  GtkTextIter start;
  GtkTextIter end;
  
  text = condition_action_ui_get(info, name);
  buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(text));
  gtk_text_buffer_get_bounds(buffer, &start, &end);
  
  str = gtk_text_buffer_get_text(buffer, &start, &end, FALSE);
  data = etpan_serialize_data_new_str(str);
  g_free(str);
  
  param_list = action->param_list;
  carray_set(param_list, text_index, data);
}

static void set_text_view(struct condition_action_ui * info, struct mapped_action * action, unsigned int text_index, char * name)
{
  GtkWidget * text;
  GtkTextBuffer * buffer;
  carray * param_list;
  struct etpan_serialize_data * data;
  char * contents;
  
  text = condition_action_ui_get(info, name);
  buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(text));
  
  param_list = action->param_list;
  data = carray_get(param_list, text_index);
  contents = etpan_serialize_data_get_str(data);
  gtk_text_buffer_set_text(buffer, contents, strlen(contents));
}

static void action_reply_get(struct condition_action_ui * info, struct mapped_action * action)
{
  int r;
  carray * param_list;
  
  param_list = action->param_list;
  r = carray_add(param_list, NULL, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  get_text_view(info, action, 0, "reply-textview");
}

static void action_reply_set(struct condition_action_ui * info, struct mapped_action * action)
{
  set_text_view(info, action, 0, "reply-textview");
}

static void action_forward_properties(struct condition_action_ui * info,
    GtkWidget * hbox)
{
  GtkWidget * text;
  GtkRequisition req;
  GtkWidget * tmp_combo;
  GtkWidget * scrolled;
  GtkWidget * entry;
  GtkWidget * vbox;
  GtkWidget * sub_hbox;
  GtkWidget * label;
  
  vbox = gtk_vbox_new(FALSE, PADDING);
  
  label = gtk_label_new(_("To the following address:"));
  gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
  gtk_widget_show(label);
  gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
  
  entry = gtk_entry_new();
  gtk_widget_show(entry);
  gtk_box_pack_start(GTK_BOX(vbox), entry, FALSE, FALSE, 0);
  g_signal_connect(entry, "changed",
      G_CALLBACK(condition_action_entry_changed), info);
  
  label = gtk_label_new(_("Include the following text:"));
  gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
  gtk_widget_show(label);
  gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
  
  scrolled = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
      GTK_POLICY_AUTOMATIC,
      GTK_POLICY_AUTOMATIC);  
  
  text = gtk_text_view_new();
  gtk_widget_show(text);
  g_signal_connect(gtk_text_view_get_buffer(GTK_TEXT_VIEW(text)), "changed",
      G_CALLBACK(condition_action_textbuffer_changed), info);
  
  gtk_container_add(GTK_CONTAINER(scrolled), text);  
  
  gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrolled),
      GTK_SHADOW_IN);
  gtk_widget_show(scrolled);
  
  gtk_box_pack_start(GTK_BOX(vbox), scrolled, TRUE, TRUE, 0);
  condition_action_ui_set(info, "forward-entry", entry);
  condition_action_ui_set(info, "forward-textview", text);
  
  tmp_combo = gtk_combo_box_new_text();
  gtk_widget_show(tmp_combo);
  gtk_widget_size_request(tmp_combo, &req);
  gtk_widget_destroy(tmp_combo);
  
  req.height *= 3;
  gtk_widget_set_size_request(GTK_WIDGET(text), -1, req.height);

  gtk_widget_show(vbox);
  gtk_box_pack_start(GTK_BOX(hbox), vbox, TRUE, TRUE, 0);
}

static void action_forward_get(struct condition_action_ui * info, struct mapped_action * action)
{
  int r;
  carray * param_list;
  GtkWidget * entry;
  const gchar * str;
  struct etpan_serialize_data * data;
  
  param_list = action->param_list;
  r = carray_add(param_list, NULL, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  get_text_view(info, action, 0, "forward-textview");
  
  entry = condition_action_ui_get(info, "forward-entry");
  str = gtk_entry_get_text(GTK_ENTRY(entry));
  data = etpan_serialize_data_new_str((char *) str);
  
  r = carray_add(param_list, data, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

static void action_forward_set(struct condition_action_ui * info, struct mapped_action * action)
{
  GtkWidget * entry;
  char * str;
  struct etpan_serialize_data * data;
  carray * param_list;
  
  set_text_view(info, action, 0, "forward-textview");
  
  param_list = action->param_list;
  data = carray_get(param_list, 1);
  str = etpan_serialize_data_get_str(data);
  
  entry = condition_action_ui_get(info, "forward-entry");
  gtk_entry_set_text(GTK_ENTRY(entry), str);
}

static int action_forward_check_consistency(struct condition_action_ui * info)
{
  GtkWidget * entry;
  char * str;
  
  entry = condition_action_ui_get(info, "forward-entry");
  str = (char *) gtk_entry_get_text(GTK_ENTRY(entry));
  if (str[0] == '\0')
    return 0;
  
  return 1;
}

static void action_color_properties(struct condition_action_ui * info,
    GtkWidget * hbox)
{
  GtkWidget * color_button;
  
  color_button = gtk_color_button_new();
  gtk_widget_show(color_button);
  gtk_box_pack_start(GTK_BOX(hbox), color_button, FALSE, FALSE, 0);
  condition_action_ui_set(info, "color-button", color_button);
  
  g_signal_connect(color_button,
      "color-set", G_CALLBACK(condition_action_color_changed),
      (gpointer) info);
}

static void action_color_get(struct condition_action_ui * info, struct mapped_action * action)
{
  GtkWidget * color_button;
  GdkColor color;
  gchar * color_str;
  struct etpan_serialize_data * data;
  carray * param_list;
  int r;
  
  color_button = condition_action_ui_get(info, "color-button");
  gtk_color_button_get_color(GTK_COLOR_BUTTON(color_button), &color);
  color_str = gdk_color_to_string(&color);
  param_list = action->param_list;
  data = etpan_serialize_data_new_str((char *) color_str);
  r = carray_add(param_list, data, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  g_free(color_str);
}

static void action_color_set(struct condition_action_ui * info, struct mapped_action * action)
{
  GtkWidget * color_button;
  struct etpan_serialize_data * data;
  carray * param_list;
  char * color_str;
  GdkColor color;
  
  color_button = condition_action_ui_get(info, "color-button");
  param_list = action->param_list;
  data = carray_get(param_list, 0);
  color_str = etpan_serialize_data_get_str(data);
  gdk_color_parse(color_str, &color);
  gtk_color_button_set_color(GTK_COLOR_BUTTON(color_button), &color);
}

static void action_kind_handler(GtkComboBox * combo, gpointer user_data)
{
  struct condition_action_ui * info;
  unsigned int i;
  GtkWidget * widget;
  GtkWidget * tmp_combo;
  GtkRequisition req;
  unsigned int selected_index;
  char * name;
  
  info = user_data;
  
  for(i = 0 ; i < sizeof(action_ui_description_list) / sizeof(action_ui_description_list[0]) ; i ++) {
    if (action_ui_description_list[i].name == NULL)
      continue;
    
    widget = condition_action_ui_get(info, action_ui_description_list[i].name);
    gtk_widget_hide(widget);
  }
  
  selected_index = gtk_combo_box_get_active(combo);
  name = action_ui_description_list[selected_index].name;
  if (name != NULL) {
    widget = condition_action_ui_get(info, name);
    gtk_widget_show(widget);
  }
  
  tmp_combo = gtk_combo_box_new_text();
  gtk_combo_box_append_text(GTK_COMBO_BOX(tmp_combo), gtk_combo_box_get_active_text(combo));
  gtk_widget_show(tmp_combo);
  gtk_widget_size_request(tmp_combo, &req);
  gtk_widget_destroy(tmp_combo);
  
  gtk_widget_set_size_request(GTK_WIDGET(combo), req.width, -1 /*req.height */);
}

static void map_condition(struct mapped_filter * filter, struct etpan_filter_condition * condition, int level);
static void map_action_list(struct mapped_filter * filter, carray * action_list);

static void mapped_condition_free(struct mapped_condition * mapped);
static void mapped_action_free(struct mapped_action * mapped);

static struct mapped_filter * mapped_filter_new(void)
{
  struct mapped_filter * mapped;
  
  mapped = malloc(sizeof(* mapped));
  if (mapped == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  mapped->valid = 1;
  mapped->condition_type = CONDITION_TYPE_NONE;
  mapped->name = NULL;
  mapped->condition_list = carray_new(4);
  if (mapped->condition_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  mapped->action_list = carray_new(4);
  if (mapped->action_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  return mapped;
}

static void mapped_filter_free(struct mapped_filter * mapped)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(mapped->condition_list) ; i ++) {
    struct mapped_condition * cond;
    
    cond = carray_get(mapped->condition_list, i);
    mapped_condition_free(cond);
  }

  for(i = 0 ; i < carray_count(mapped->action_list) ; i ++) {
    struct mapped_action * action;
    
    action = carray_get(mapped->action_list, i);
    mapped_action_free(action);
  }
  
  free(mapped->name);
  
  free(mapped);
}

static void mapped_filter_set_name(struct mapped_filter * filter,
    char * name)
{
  if (filter->name != name) {
    free(filter->name);
    if (name != NULL) {
      filter->name = strdup(name);
      if (filter->name == NULL)
        ETPAN_LOG_MEMORY_ERROR;
    }
    else {
      filter->name = NULL;
    }
  }
}

static void mapped_filter_add_condition(struct mapped_filter * filter,
    struct mapped_condition * condition)
{
  int r;
  
  r = carray_add(filter->condition_list, condition, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

static void mapped_filter_add_action(struct mapped_filter * filter,
    struct mapped_action * action)
{
  int r;
  
  r = carray_add(filter->action_list, action, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
}

static struct mapped_condition * mapped_condition_new(char * name,
    carray * param_list)
{
  struct mapped_condition * mapped;
  
  mapped = malloc(sizeof(* mapped));
  if (mapped == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  if (name == NULL) {
    mapped->name = NULL;
  }
  else {
    mapped->name = strdup(name);
    if (mapped->name == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  mapped->param_list = param_list;
  
  return mapped;
}

static void mapped_condition_free(struct mapped_condition * mapped)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(mapped->param_list) ; i ++) {
    struct etpan_serialize_data * param;
    
    param = carray_get(mapped->param_list, i);
    etpan_serialize_data_free(param);
  }
  
  free(mapped->name);
  free(mapped);
}

static struct mapped_action * mapped_action_new(char * name,
    carray * param_list)
{
  struct mapped_action * mapped;
  
  mapped = malloc(sizeof(* mapped));
  if (mapped == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  if (name == NULL) {
    mapped->name = NULL;
  }
  else {
    mapped->name = strdup(name);
    if (mapped->name == NULL)
      ETPAN_LOG_MEMORY_ERROR;
  }
  mapped->param_list = param_list;
  
  return mapped;
}

static void mapped_action_free(struct mapped_action * mapped)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(mapped->param_list) ; i ++) {
    struct etpan_serialize_data * param;
    
    param = carray_get(mapped->param_list, i);
    etpan_serialize_data_free(param);
  }
  
  free(mapped->name);
  free(mapped);
}

static struct mapped_filter * map_filter(struct etpan_filter * filter)
{
  struct mapped_filter * mapped;
  
  mapped = mapped_filter_new();
  
  mapped_filter_set_name(mapped, etpan_filter_get_name(filter));
  map_condition(mapped, etpan_filter_get_condition(filter), 0);
  map_action_list(mapped, etpan_filter_get_action_list(filter));
  
  return mapped;
}

static carray * param_list_dup(carray * param_list)
{
  carray * dup_list;
  unsigned int i;
  int r;
  
  dup_list = carray_new(carray_count(param_list));
  if (dup_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  for(i = 0 ; i < carray_count(param_list) ; i ++) {
    struct etpan_serialize_data * data;
    struct etpan_serialize_data * dup_data;
    
    data = carray_get(param_list, i);
    dup_data = etpan_serialize_data_dup(data);
    r = carray_add(dup_list, dup_data, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  return dup_list;
}

static void reverse_match_type(carray * param_list, int param_index)
{
  struct etpan_serialize_data * data;
  char * str;
  
  data = carray_get(param_list, param_index);
  str = etpan_serialize_data_get_str(data);
  
  if (strcasecmp(str, "contains") == 0)
    str = "doesnotcontain";
  else if (strcasecmp(str, "doesnotcontain") == 0)
    str = "contains";
  else if (strcasecmp(str, "begin") == 0)
    str = "doesnotbegin";
  else if (strcasecmp(str, "end") == 0)
    str = "doesnotend";
  else if (strcasecmp(str, "equals") == 0)
    str = "isnotequalto";
  else if (strcasecmp(str, "isnotequalto") == 0)
    str = "equals";
  else
    str = "doesnotcontain";
  
  etpan_serialize_data_free(data);
  data = etpan_serialize_data_new_str(str);
  carray_set(param_list, param_index, data);
}

static void map_one_condition(struct mapped_filter * filter, struct etpan_filter_condition * condition, int reverse)
{
  char * description;
  struct mapped_condition * mapped;
  
  description = etpan_filter_condition_get_description_name(condition);
  
  if (reverse) {
    if (strcmp(description, "abook") == 0) {
      carray * param_list;
      
      param_list = etpan_filter_condition_get_param_list(condition);
      param_list = param_list_dup(param_list);
      mapped = mapped_condition_new("no-abook", param_list);
      mapped_filter_add_condition(filter, mapped);
    }
    else if (strcmp(description, "has_attachment") == 0) {
      carray * param_list;
      
      param_list = etpan_filter_condition_get_param_list(condition);
      param_list = param_list_dup(param_list);
      mapped = mapped_condition_new("has_no_attachment", param_list);
      mapped_filter_add_condition(filter, mapped);
    }
    else if (strcmp(description, "from") == 0) {
      carray * param_list;
      
      param_list = etpan_filter_condition_get_param_list(condition);
      param_list = param_list_dup(param_list);
      reverse_match_type(param_list, 0);
      mapped = mapped_condition_new("from", param_list);
      mapped_filter_add_condition(filter, mapped);
    }
    else if (strcmp(description, "recipient") == 0) {
      carray * param_list;
      
      param_list = etpan_filter_condition_get_param_list(condition);
      param_list = param_list_dup(param_list);
      reverse_match_type(param_list, 0);
      mapped = mapped_condition_new("recipient", param_list);
      mapped_filter_add_condition(filter, mapped);
    }
    else if (strcmp(description, "subject") == 0) {
      carray * param_list;
      
      param_list = etpan_filter_condition_get_param_list(condition);
      param_list = param_list_dup(param_list);
      reverse_match_type(param_list, 0);
      mapped = mapped_condition_new("subject", param_list);
      mapped_filter_add_condition(filter, mapped);
    }
    else if (strcmp(description, "mailing_list") == 0) {
      carray * param_list;
      
      param_list = etpan_filter_condition_get_param_list(condition);
      param_list = param_list_dup(param_list);
      reverse_match_type(param_list, 0);
      mapped = mapped_condition_new("mailing_list", param_list);
      mapped_filter_add_condition(filter, mapped);
    }
    else if (strcmp(description, "content") == 0) {
      carray * param_list;
      
      param_list = etpan_filter_condition_get_param_list(condition);
      param_list = param_list_dup(param_list);
      reverse_match_type(param_list, 0);
      mapped = mapped_condition_new("content", param_list);
      mapped_filter_add_condition(filter, mapped);
    }
    else if (strcmp(description, "header") == 0) {
      carray * param_list;
      
      param_list = etpan_filter_condition_get_param_list(condition);
      param_list = param_list_dup(param_list);
      reverse_match_type(param_list, 0);
      mapped = mapped_condition_new("header", param_list);
      mapped_filter_add_condition(filter, mapped);
    }
    else if (strcmp(description, "attachment_name") == 0) {
      carray * param_list;
      
      param_list = etpan_filter_condition_get_param_list(condition);
      param_list = param_list_dup(param_list);
      reverse_match_type(param_list, 0);
      mapped = mapped_condition_new("attachment_name", param_list);
      mapped_filter_add_condition(filter, mapped);
    }
    else {
      filter->valid = 0;
    }
  }
  else {
    if (strcmp(description, "from") == 0) {
      carray * param_list;
    
      param_list = etpan_filter_condition_get_param_list(condition);
      param_list = param_list_dup(param_list);
      mapped = mapped_condition_new("from", param_list);
      mapped_filter_add_condition(filter, mapped);
    }
    else if (strcmp(description, "recipient") == 0) {
      carray * param_list;
    
      param_list = etpan_filter_condition_get_param_list(condition);
      param_list = param_list_dup(param_list);
      mapped = mapped_condition_new("recipient", param_list);
      mapped_filter_add_condition(filter, mapped);
    }
    else if (strcmp(description, "subject") == 0) {
      carray * param_list;
    
      param_list = etpan_filter_condition_get_param_list(condition);
      param_list = param_list_dup(param_list);
      mapped = mapped_condition_new("subject", param_list);
      mapped_filter_add_condition(filter, mapped);
    }
    else if (strcmp(description, "abook") == 0) {
      carray * param_list;
    
      param_list = etpan_filter_condition_get_param_list(condition);
      param_list = param_list_dup(param_list);
      mapped = mapped_condition_new("abook", param_list);
      mapped_filter_add_condition(filter, mapped);
    }
    else if (strcmp(description, "mailing_list") == 0) {
      carray * param_list;
    
      param_list = etpan_filter_condition_get_param_list(condition);
      param_list = param_list_dup(param_list);
      mapped = mapped_condition_new("mailing_list", param_list);
      mapped_filter_add_condition(filter, mapped);
    }
    else if (strcmp(description, "content") == 0) {
      carray * param_list;
    
      param_list = etpan_filter_condition_get_param_list(condition);
      param_list = param_list_dup(param_list);
      mapped = mapped_condition_new("content", param_list);
      mapped_filter_add_condition(filter, mapped);
    }
    else if (strcmp(description, "header") == 0) {
      carray * param_list;
    
      param_list = etpan_filter_condition_get_param_list(condition);
      param_list = param_list_dup(param_list);
      mapped = mapped_condition_new("header", param_list);
      mapped_filter_add_condition(filter, mapped);
    }
    else if (strcmp(description, "true") == 0) {
      carray * param_list;
    
      param_list = etpan_filter_condition_get_param_list(condition);
      param_list = param_list_dup(param_list);
      mapped = mapped_condition_new("true", param_list);
      mapped_filter_add_condition(filter, mapped);
    }
    else if (strcmp(description, "account") == 0) {
      carray * param_list;
    
      param_list = etpan_filter_condition_get_param_list(condition);
      param_list = param_list_dup(param_list);
      mapped = mapped_condition_new("account", param_list);
      mapped_filter_add_condition(filter, mapped);
    }
    else if (strcmp(description, "has_attachment") == 0) {
      carray * param_list;
    
      param_list = etpan_filter_condition_get_param_list(condition);
      param_list = param_list_dup(param_list);
      mapped = mapped_condition_new("has_attachment", param_list);
      mapped_filter_add_condition(filter, mapped);
    }
    else if (strcmp(description, "attachment_name") == 0) {
      carray * param_list;
    
      param_list = etpan_filter_condition_get_param_list(condition);
      param_list = param_list_dup(param_list);
      mapped = mapped_condition_new("attachment_name", param_list);
      mapped_filter_add_condition(filter, mapped);
    }
    else {
      filter->valid = 0;
    }
  }
}

static void map_condition(struct mapped_filter * filter, struct etpan_filter_condition * condition, int level)
{
  char * description;
  
  /*
    level 0: root
    level 1: and/or/not
    level 2: and/or, then, not
  */
  
  description = etpan_filter_condition_get_description_name(condition);
  
  switch (level) {
  case 0:
    if (strcmp(description, "or") == 0) {
      unsigned int i;
      carray * children;
      
      filter->condition_type = CONDITION_TYPE_OR;
      
      children = etpan_filter_condition_get_children(condition);
      for(i = 0 ; i < carray_count(children) ; i ++) {
        struct etpan_filter_condition * sub_condition;
        
        sub_condition = carray_get(children, i);
        map_condition(filter, sub_condition, 1);
      }
      return;
    }
    else if (strcmp(description, "and") == 0) {
      unsigned int i;
      carray * children;
      
      filter->condition_type = CONDITION_TYPE_AND;
      
      children = etpan_filter_condition_get_children(condition);
      for(i = 0 ; i < carray_count(children) ; i ++) {
        struct etpan_filter_condition * sub_condition;
        
        sub_condition = carray_get(children, i);
        map_condition(filter, sub_condition, 1);
      }
      return;
    }
    else if (strcmp(description, "not") == 0) {
      unsigned int i;
      carray * children;
      
      filter->condition_type = CONDITION_TYPE_NOT;
      
      children = etpan_filter_condition_get_children(condition);
      for(i = 0 ; i < carray_count(children) ; i ++) {
        struct etpan_filter_condition * sub_condition;
        
        sub_condition = carray_get(children, i);
        map_condition(filter, sub_condition, 1);
      }
      return;
    }
    else {
      map_one_condition(filter, condition, 0);
    }
    break;
    
  case 1:
    if (strcmp(description, "or") == 0) {
      unsigned int i;
      carray * children;
      
      if (filter->condition_type != CONDITION_TYPE_OR) {
        filter->valid = 0;
        return;
      }
      
      children = etpan_filter_condition_get_children(condition);
      for(i = 0 ; i < carray_count(children) ; i ++) {
        struct etpan_filter_condition * sub_condition;
        
        sub_condition = carray_get(children, i);
        map_condition(filter, sub_condition, 1);
      }
      return;
    }
    else if (strcmp(description, "and") == 0) {
      unsigned int i;
      carray * children;
      
      if (filter->condition_type != CONDITION_TYPE_AND) {
        filter->valid = 0;
        return;
      }
      
      children = etpan_filter_condition_get_children(condition);
      for(i = 0 ; i < carray_count(children) ; i ++) {
        struct etpan_filter_condition * sub_condition;
        
        sub_condition = carray_get(children, i);
        map_condition(filter, sub_condition, 1);
      }
      return;
    }
    else if (strcmp(description, "not") == 0) {
      unsigned int i;
      carray * children;
      
      children = etpan_filter_condition_get_children(condition);
      for(i = 0 ; i < carray_count(children) ; i ++) {
        struct etpan_filter_condition * sub_condition;
        
        sub_condition = carray_get(children, i);
        map_condition(filter, sub_condition, 2);
      }
      return;
    }
    
    if (filter->condition_type == CONDITION_TYPE_NOT) {
      map_one_condition(filter, condition, 1);
    }
    else {
      map_one_condition(filter, condition, 0);
    }
    break;
    
  case 2:
    map_one_condition(filter, condition, 1);
    break;
  }
}

static void map_one_action(struct mapped_filter * filter, struct etpan_filter_action * action);

static void map_action_list(struct mapped_filter * filter, carray * action_list)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(action_list) ; i ++) {
    struct etpan_filter_action * action;
    
    action = carray_get(action_list, i);
    map_one_action(filter, action);
  }
}

static void map_one_action(struct mapped_filter * filter, struct etpan_filter_action * action)
{
  char * description_name;
  struct etpan_filter_action_ui_description * description;
  struct mapped_action * mapped;
  carray * param_list;
  
  description_name = etpan_filter_action_get_description_name(action);
  description = get_action_description(description_name);
  if (description == NULL) {
    ETPAN_LOG("unsupported action");
    filter->valid = 0;
    return;
  }
  
  param_list = etpan_filter_action_get_param_list(action);
  param_list = param_list_dup(param_list);
  mapped = mapped_action_new(description_name, param_list);
  mapped_filter_add_action(filter, mapped);
}

static void mapped_filter_set_ui(struct etpan_preferences_panel * panel,
    struct mapped_filter * filter);

static void filter_set_ui(struct etpan_preferences_panel * panel,
    struct etpan_filter * filter)
{
  struct mapped_filter * mapped;
  struct panel_data * data;
  
  mapped = map_filter(filter);
  mapped_filter_set_ui(panel, mapped);
  mapped_filter_free(mapped);
  
  data = panel->data;
  data->modified = 0;
  enable_buttons(panel);
}

static void filter_condition_reset_ui(struct etpan_preferences_panel * panel);
static void filter_action_reset_ui(struct etpan_preferences_panel * panel);
static void filter_reset_ui(struct etpan_preferences_panel * panel);

static void filter_new_ui(struct etpan_preferences_panel * panel)
{
  GtkWidget * table;
  
  filter_reset_ui(panel);
  
  table = etpan_preferences_panel_get_widget(panel, "condition-table");
  add_new_condition_ui(panel, table);
  table = etpan_preferences_panel_get_widget(panel, "action-table");
  add_new_action_ui(panel, table);
}

static void filter_reset_ui(struct etpan_preferences_panel * panel)
{
  filter_condition_reset_ui(panel);
  filter_action_reset_ui(panel);
}

static void filter_condition_reset_ui(struct etpan_preferences_panel * panel)
{
  struct panel_data * data;
  unsigned int i;
  
  data = panel->data;
  for(i = 0 ; i < carray_count(data->condition_ui_list) ; i ++) {
    struct condition_action_ui * info;
    GtkWidget * hbox;
    GtkWidget * alert_image;
    
    info = carray_get(data->condition_ui_list, i);
    filter_condition_unsetup(info);
    hbox = condition_action_ui_get(info, "main");
    gtk_widget_destroy(hbox);
    alert_image = condition_action_ui_get(info, "alert-image");
    gtk_widget_destroy(alert_image);
    condition_action_ui_free(info);
  }
  carray_set_size(data->condition_ui_list, 0);
  etpan_preferences_panel_set_widget(panel, "selection-radio-button", NULL);
  layout_filter_preferences(panel);
}

static void filter_action_reset_ui(struct etpan_preferences_panel * panel)
{
  struct panel_data * data;
  unsigned int i;
  
  data = panel->data;
  for(i = 0 ; i < carray_count(data->action_ui_list) ; i ++) {
    struct condition_action_ui * info;
    GtkWidget * hbox;
    GtkWidget * alert_image;
    
    info = carray_get(data->action_ui_list, i);
    filter_action_unsetup(info);
    hbox = condition_action_ui_get(info, "main");
    gtk_widget_destroy(hbox);
    alert_image = condition_action_ui_get(info, "alert-image");
    gtk_widget_destroy(alert_image);
    condition_action_ui_free(info);
  }
  carray_set_size(data->action_ui_list, 0);
  etpan_preferences_panel_set_widget(panel, "selection-radio-action", NULL);
}

static int get_condition_description_index(char * name)
{
  unsigned int i;
  
  for(i = 0 ; i < sizeof(condition_ui_description_list) / sizeof(condition_ui_description_list[0]) ; i ++) {
    if (condition_ui_description_list[i].name != NULL) {
      if (strcmp(condition_ui_description_list[i].name, name) == 0) {
        return i;
      }
    }
  }
  
  return -1;
}

static struct etpan_filter_condition_ui_description * get_condition_description(char * name)
{
  int i;
  
  i = get_condition_description_index(name);
  if (i == -1)
    return NULL;
  
  return &condition_ui_description_list[i];
}

static int get_action_description_index(char * name)
{
  unsigned int i;
  
  for(i = 0 ; i < sizeof(action_ui_description_list) / sizeof(action_ui_description_list[0]) ; i ++) {
    if (action_ui_description_list[i].name != NULL) {
      if (strcmp(action_ui_description_list[i].name, name) == 0) {
        return i;
      }
    }
  }
  
  return -1;
}

static struct etpan_filter_action_ui_description * get_action_description(char * name)
{
  int i;
  
  i = get_action_description_index(name);
  if (i == -1)
    return NULL;
  
  return &action_ui_description_list[i];
}

static void mapped_condition_set_ui(struct condition_action_ui * info,
    struct mapped_condition * condition)
{
  GtkWidget * combo_type;
  int description_index;
  struct etpan_filter_condition_ui_description * description;
  
  combo_type = condition_action_ui_get(info, "combo-type");
  description_index = get_condition_description_index(condition->name);
  if (description_index == -1)
    return;
  
  description = get_condition_description(condition->name);
  gtk_combo_box_set_active(GTK_COMBO_BOX(combo_type), description_index);
  if (description->condition_set != NULL) {
    description->condition_set(info, condition);
  }
  info->is_new = 0;
}

static void mapped_action_set_ui(struct condition_action_ui * info,
    struct mapped_action * action)
{
  GtkWidget * combo_type;
  int description_index;
  struct etpan_filter_action_ui_description * description;
  
  combo_type = condition_action_ui_get(info, "combo-type");
  description_index = get_action_description_index(action->name);
  if (description_index == -1)
    return;
  
  description = get_action_description(action->name);
  gtk_combo_box_set_active(GTK_COMBO_BOX(combo_type), description_index);
  if (description->action_set != NULL) {
    description->action_set(info, action);
  }
  info->is_new = 0;
}

static void mapped_filter_set_ui(struct etpan_preferences_panel * panel,
    struct mapped_filter * filter)
{
  unsigned int i;
  GtkWidget * table;
  struct panel_data * data;
  GtkWidget * label_unsupported;
  GtkWidget * hbox;
  GtkWidget * separator;
  GtkWidget * label;
  char * name;
  GtkWidget * entry_name;
  
  data = panel->data;
  
  filter_reset_ui(panel);
  
  name = filter->name;
  entry_name = etpan_preferences_panel_get_widget(panel, "filter-name");
  gtk_entry_set_text(GTK_ENTRY(entry_name), name);
  
  label_unsupported = etpan_preferences_panel_get_widget(panel, "label-unsupported");
  data->invalid_filter = 0;
  if (!filter->valid) {
    data->invalid_filter = 1;
    gtk_widget_show(label_unsupported);
    table = etpan_preferences_panel_get_widget(panel, "condition-table");
    gtk_widget_hide(table);
    table = etpan_preferences_panel_get_widget(panel, "action-table");
    gtk_widget_hide(table);
    hbox = etpan_preferences_panel_get_widget(panel, "hbox-multiple");
    gtk_widget_hide(hbox);
    hbox = etpan_preferences_panel_get_widget(panel, "hbox-one");
    gtk_widget_hide(hbox);
    hbox = etpan_preferences_panel_get_widget(panel, "condition-button-hbox");
    gtk_widget_hide(hbox);
    hbox = etpan_preferences_panel_get_widget(panel, "action-button-hbox");
    gtk_widget_hide(hbox);
    separator = etpan_preferences_panel_get_widget(panel, "condition-action-separator");
    gtk_widget_hide(separator);
    label = etpan_preferences_panel_get_widget(panel, "label-action");
    gtk_widget_hide(label);
    gtk_widget_queue_draw(panel->main_widget);
    table = etpan_preferences_panel_get_widget(panel, "table");
    gtk_container_check_resize(GTK_CONTAINER(table));
    
    ETPAN_LOG("invalid filter");
    return;
  }
  gtk_widget_hide(label_unsupported);
  hbox = etpan_preferences_panel_get_widget(panel, "condition-button-hbox");
  gtk_widget_show(hbox);
  hbox = etpan_preferences_panel_get_widget(panel, "action-button-hbox");
  gtk_widget_show(hbox);
  separator = etpan_preferences_panel_get_widget(panel, "condition-action-separator");
  gtk_widget_show(separator);
  label = etpan_preferences_panel_get_widget(panel, "label-action");
  gtk_widget_show(label);
  table = etpan_preferences_panel_get_widget(panel, "table");
  gtk_container_check_resize(GTK_CONTAINER(table));
  
  table = etpan_preferences_panel_get_widget(panel, "condition-table");
  gtk_widget_show(table);
  for(i = 0 ; i < carray_count(filter->condition_list) ; i ++) {
    struct mapped_condition * condition;
    struct condition_action_ui * info;
    
    condition = carray_get(filter->condition_list, i);
    add_new_condition_ui(panel, table);
    info = carray_get(data->condition_ui_list,
        carray_count(data->condition_ui_list) - 1);
    mapped_condition_set_ui(info, condition);
    ETPAN_LOG("add condition");
  }
  
  table = etpan_preferences_panel_get_widget(panel, "action-table");
  gtk_widget_show(table);
  for(i = 0 ; i < carray_count(filter->action_list) ; i ++) {
    struct mapped_action * action;
    struct condition_action_ui * info;
    
    action = carray_get(filter->action_list, i);
    add_new_action_ui(panel, table);
    info = carray_get(data->action_ui_list,
        carray_count(data->action_ui_list) - 1);
    mapped_action_set_ui(info, action);
    ETPAN_LOG("add action");
  }
  gtk_widget_queue_draw(panel->main_widget);
}

static void mapped_condition_get_ui(struct condition_action_ui * info,
    struct mapped_condition * condition)
{
  GtkWidget * combo_type;
  int description_index;
  struct etpan_filter_condition_ui_description * description;
  
  combo_type = condition_action_ui_get(info, "combo-type");
  description_index = gtk_combo_box_get_active(GTK_COMBO_BOX(combo_type));
  if (description_index == -1)
    return;
  
  description = &condition_ui_description_list[description_index];
  free(condition->name);
  condition->name = strdup(description->name);
  if (condition->name == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  if (description->condition_get != NULL) {
    description->condition_get(info, condition);
  }
}

static void mapped_action_get_ui(struct condition_action_ui * info,
    struct mapped_action * action)
{
  GtkWidget * combo_type;
  int description_index;
  struct etpan_filter_action_ui_description * description;
  
  combo_type = condition_action_ui_get(info, "combo-type");
  description_index = gtk_combo_box_get_active(GTK_COMBO_BOX(combo_type));
  if (description_index == -1)
    return;
  
  description = &action_ui_description_list[description_index];
  free(action->name);
  action->name = strdup(description->name);
  if (action->name == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  if (description->action_get != NULL) {
    description->action_get(info, action);
  }
}

static void mapped_filter_get_ui(struct etpan_preferences_panel * panel,
    struct mapped_filter * filter)
{
  int r;
  struct panel_data * data;
  unsigned int i;
  const char * name;
  GtkWidget * entry_name;
  
  data = panel->data;
  
  entry_name = etpan_preferences_panel_get_widget(panel, "filter-name");
  name = gtk_entry_get_text(GTK_ENTRY(entry_name));
  filter->name = strdup(name);
  if (filter->name == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  for(i =  0 ; i < carray_count(data->condition_ui_list) ; i ++) {
    struct condition_action_ui * info;
    struct mapped_condition * condition;
    carray * param_list;
    
    param_list = carray_new(4);
    if (param_list == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    condition = mapped_condition_new(NULL, param_list);
    info = carray_get(data->condition_ui_list, i);
    mapped_condition_get_ui(info, condition);
    r = carray_add(filter->condition_list, condition, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
  
  for(i =  0 ; i < carray_count(data->action_ui_list) ; i ++) {
    struct condition_action_ui * info;
    struct mapped_action * action;
    carray * param_list;
    
    param_list = carray_new(4);
    if (param_list == NULL)
      ETPAN_LOG_MEMORY_ERROR;
    action = mapped_action_new(NULL, param_list);
    info = carray_get(data->action_ui_list, i);
    mapped_action_get_ui(info, action);
    r = carray_add(filter->action_list, action, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
}

static struct etpan_filter *
unmap_filter(struct mapped_filter * mapped);
static struct etpan_filter_condition *
unmap_condition(struct mapped_condition * mapped);
static struct etpan_filter_action *
unmap_action(struct mapped_action * mapped);

static struct etpan_filter *
filter_get_ui(struct etpan_preferences_panel * panel)
{
  struct mapped_filter * mapped;
  struct etpan_filter * filter;
  struct panel_data * data;
  
  data = panel->data;
  if (data->invalid_filter) {
    struct etpan_filter * old_filter;
    const char * name;
    GtkWidget * entry_name;
    
    entry_name = etpan_preferences_panel_get_widget(panel, "filter-name");
    name = gtk_entry_get_text(GTK_ENTRY(entry_name));
    
    old_filter = get_selected_filter(panel);
    filter = etpan_filter_dup(old_filter);
    etpan_filter_set_name(filter, (char *) name);
  }
  else {
    mapped = mapped_filter_new();
    mapped_filter_get_ui(panel, mapped);
    filter = unmap_filter(mapped);
  }
  
  return filter;
}

static struct etpan_filter *
unmap_filter(struct mapped_filter * mapped)
{
  unsigned int i;
  struct etpan_filter_condition * main_condition;
  struct etpan_filter * filter;
  carray * param_list;
  
  filter = etpan_filter_new();
  param_list = carray_new(4);
  if (param_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  main_condition = NULL;
  switch (mapped->condition_type) {
  case CONDITION_TYPE_NONE:
  case CONDITION_TYPE_OR:
    main_condition = etpan_filter_condition_new("or", param_list);
    break;
  case CONDITION_TYPE_AND:
    main_condition = etpan_filter_condition_new("and", param_list);
    break;
  case CONDITION_TYPE_NOT:
    main_condition = etpan_filter_condition_new("not", param_list);
    break;
  default:
    ETPAN_CRASH("invalid");
    break;
  }
  etpan_filter_set_condition(filter, main_condition);
  etpan_filter_set_name(filter, mapped->name);
  
  for(i = 0 ; i < carray_count(mapped->condition_list) ; i ++) {
    struct mapped_condition * condition;
    struct etpan_filter_condition * unmapped_condition;
    
    condition = carray_get(mapped->condition_list, i);
    unmapped_condition = unmap_condition(condition);
    etpan_filter_condition_add(main_condition, unmapped_condition);
  }
  
  for(i = 0 ; i < carray_count(mapped->action_list) ; i ++) {
    struct mapped_action * action;
    struct etpan_filter_action * unmapped_action;
    
    action = carray_get(mapped->action_list, i);
    unmapped_action = unmap_action(action);
    etpan_filter_add_action(filter, unmapped_action);
  }
  
  return filter;
}

static struct etpan_filter_condition *
unmap_condition(struct mapped_condition * mapped)
{
  char * description_name;
  struct etpan_filter_condition_ui_description * description;
  
  description_name = mapped->name;
  description = get_condition_description(description_name);
  if (description == NULL) {
    ETPAN_CRASH("invalid action");
  }
  
  return condition_generate(mapped);
}

static struct etpan_filter_action *
unmap_action(struct mapped_action * mapped)
{
  char * description_name;
  struct etpan_filter_action_ui_description * description;
  struct etpan_filter_action * action;
  carray * param_list;
  
  description_name = mapped->name;
  description = get_action_description(description_name);
  if (description == NULL) {
    ETPAN_CRASH("invalid action");
  }
  
  param_list = param_list_dup(mapped->param_list);
  action = etpan_filter_action_new(description_name, param_list);
  
  return action;
}

static void modify_data(struct etpan_preferences_panel * panel)
{
  struct panel_data * data;
  
  data = panel->data;
  if (!data->adding_element) {
    data->modified = 1;
  }
  
  enable_buttons(panel);
}

static void condition_action_modify_data(struct condition_action_ui * info)
{
  struct panel_data * data;
  struct etpan_preferences_panel * panel;
  
  panel = info->panel;
  data = panel->data;
  if (!data->adding_element) {
    info->is_new = 0;
  }
  
  modify_data(panel);
}

static void enable_buttons(struct etpan_preferences_panel * panel)
{
  GtkWidget * button;
  gboolean enabled;
  struct panel_data * data;
  carray * list;
  
  data = panel->data;
  if (!data->init_done)
    return;
  
  enabled = data->modified;
  
  button = etpan_preferences_panel_get_widget(panel, "create-button");
  gtk_widget_set_sensitive(button, enabled);

  button = etpan_preferences_panel_get_widget(panel, "cancel-button");
  gtk_widget_set_sensitive(button, TRUE);
  
  button = etpan_preferences_panel_get_widget(panel, "revert-button");
  gtk_widget_set_sensitive(button, enabled);
  
  button = etpan_preferences_panel_get_widget(panel, "apply-button");
  gtk_widget_set_sensitive(button, enabled);
  
  button = etpan_preferences_panel_get_widget(panel, "remove-condition-button");
  if (carray_count(data->condition_ui_list) > 1) {
    gtk_widget_set_sensitive(button, TRUE);
  }
  else {
    gtk_widget_set_sensitive(button, FALSE);
  }

  button = etpan_preferences_panel_get_widget(panel, "remove-action-button");
  if (carray_count(data->action_ui_list) > 1) {
    gtk_widget_set_sensitive(button, TRUE);
  }
  else {
    gtk_widget_set_sensitive(button, FALSE);
  }
  
  list = etpan_filter_config_get_rule_list(etpan_filter_config_get_default());
  if (carray_count(list) > 0) {
    button = etpan_preferences_panel_get_widget(panel, "add-button");
    if (data->pending_filter == NULL) {
      gtk_widget_set_sensitive(button, TRUE);
    }
    else {
      gtk_widget_set_sensitive(button, FALSE);
    }
    
    button = etpan_preferences_panel_get_widget(panel, "remove-button");
    gtk_widget_set_sensitive(button, TRUE);
  }
  else {
    button = etpan_preferences_panel_get_widget(panel, "add-button");
    gtk_widget_set_sensitive(button, FALSE);
    
    button = etpan_preferences_panel_get_widget(panel, "remove-button");
    gtk_widget_set_sensitive(button, FALSE);
  }
}

static void entry_changed(GtkEditable *editable,
    gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  
  panel = user_data;
  modify_data(panel);
}

static void combo_changed(GtkComboBox * combo, gpointer user_data)
{
  struct etpan_preferences_panel * panel;
  
  panel = user_data;
  modify_data(panel);
}

static void condition_action_entry_changed(GtkEditable * editable,
    gpointer user_data)
{
  struct condition_action_ui * info;
  
  info = user_data;
  condition_action_modify_data(info);
}

static void condition_action_textbuffer_changed(GtkTextBuffer * textbuffer,
    gpointer user_data)
{
  struct condition_action_ui * info;
  
  info = user_data;
  condition_action_modify_data(info);
}

static void condition_action_combo_changed(GtkComboBox * combo, gpointer user_data)
{
  struct condition_action_ui * info;
  
  info = user_data;
  condition_action_modify_data(info);
}

static void condition_action_color_changed(GtkColorButton * widget, gpointer user_data)
{
  struct condition_action_ui * info;
  
  info = user_data;
  condition_action_modify_data(info);
}

static void start_filter_edition(struct etpan_preferences_panel * panel)
{
  GtkWidget * button;
  struct panel_data * data;
  
  data = panel->data;
  
  if (data->pending_filter != NULL) {
    button = etpan_preferences_panel_get_widget(panel, "create-button");
    gtk_widget_show(button);
    
    button = etpan_preferences_panel_get_widget(panel, "cancel-button");
    gtk_widget_show(button);
    
    button = etpan_preferences_panel_get_widget(panel, "revert-button");
    gtk_widget_hide(button);
    
    button = etpan_preferences_panel_get_widget(panel, "apply-button");
    gtk_widget_hide(button);
  }
  else {
    button = etpan_preferences_panel_get_widget(panel, "create-button");
    gtk_widget_hide(button);
    
    button = etpan_preferences_panel_get_widget(panel, "cancel-button");
    gtk_widget_hide(button);
    
    button = etpan_preferences_panel_get_widget(panel, "revert-button");
    gtk_widget_show(button);
    
    button = etpan_preferences_panel_get_widget(panel, "apply-button");
    gtk_widget_show(button);
  }
}

/* *********************************** */
/* check consistency */

static void switch_text(struct etpan_preferences_panel * panel,
    char * name, char * value)
{
  GtkWidget * label;
  
  label = etpan_preferences_panel_get_widget(panel, name);
  gtk_label_set_markup(GTK_LABEL(label), value);
}

static int condition_check_consistency(struct condition_action_ui * info);
static int action_check_consistency(struct condition_action_ui * info);

static int check_consistency(struct etpan_preferences_panel * panel)
{
  struct panel_data * data;
  unsigned int i;
  const char * name;
  GtkWidget * entry_name;
  int enabled;
  
  data = panel->data;
  enabled = data->modified;
  
  entry_name = etpan_preferences_panel_get_widget(panel, "filter-name");
  name = gtk_entry_get_text(GTK_ENTRY(entry_name));
  if (data->modified && (name[0] == '\0')) {
    switch_text(panel, "label-name", _("<span foreground=\"#ff0000\">Filter Name</span>"));
    enabled = 0;
  }
  else {
    switch_text(panel, "label-name", _("<span foreground=\"#000000\">Filter Name</span>"));
  }
  
  if (data->modified) {
    for(i =  0 ; i < carray_count(data->condition_ui_list) ; i ++) {
      struct condition_action_ui * info;
      
      info = carray_get(data->condition_ui_list, i);
      if (!condition_check_consistency(info)) {
        enabled = 0;
      }
    }
  }
  
  if (data->modified) {
    for(i =  0 ; i < carray_count(data->action_ui_list) ; i ++) {
      struct condition_action_ui * info;
      
      info = carray_get(data->action_ui_list, i);
      if (!action_check_consistency(info)) {
        enabled = 0;
      }
    }
  }
  
  if ((!enabled) && data->modified) {
    switch_text(panel, "label-info",
        _("<span foreground=\"#ff0000\">Some information needs to be filled in</span>"));
  }
  else {
    switch_text(panel, "label-info", "");
  }
  
  return enabled;
}

static int condition_check_consistency(struct condition_action_ui * info)
{
  GtkWidget * combo_type;
  int description_index;
  struct etpan_filter_condition_ui_description * description;
  GtkWidget * alert_image;
  
  combo_type = condition_action_ui_get(info, "combo-type");
  description_index = gtk_combo_box_get_active(GTK_COMBO_BOX(combo_type));
  if (description_index == -1)
    return 1;
  
  description = &condition_ui_description_list[description_index];
  if (description->check_consistency == NULL)
    return 1;
  
  alert_image = condition_action_ui_get(info, "alert-image");
  if (!description->check_consistency(info)) {
    gtk_widget_show(alert_image);
    return 0;
  }
  gtk_widget_hide(alert_image);
  
  return 1;
}

static int action_check_consistency(struct condition_action_ui * info)
{
  GtkWidget * combo_type;
  int description_index;
  struct etpan_filter_action_ui_description * description;
  GtkWidget * alert_image;
  
  combo_type = condition_action_ui_get(info, "combo-type");
  description_index = gtk_combo_box_get_active(GTK_COMBO_BOX(combo_type));
  if (description_index == -1)
    return 1;
  
  description = &action_ui_description_list[description_index];
  if (description->check_consistency == NULL)
    return 1;
  
  alert_image = condition_action_ui_get(info, "alert-image");
  if (!description->check_consistency(info)) {
    gtk_widget_show(alert_image);
    return 0;
  }
  gtk_widget_hide(alert_image);
  
  return 1;
}
