#include "etpan-preferences-window.h"

#include "etpan-backend.h"
#include <stdlib.h>

#include "etpan-preferences-account.h"
#include "etpan-preferences-view.h"
#include "etpan-preferences-compose.h"
#include "etpan-preferences-abook.h"
#include "etpan-preferences-filter.h"
#include "etpan-ui-config.h"
#include "etpan-icon-manager.h"

struct etpan_preferences_window * etpan_preferences_window_new(void)
{
  /* add window */
  /* add notebook */
  GtkWidget * window;
  GtkWidget * notebook;
  GtkWidget * vbox;
  GtkWidget * close_button;
  struct etpan_preferences_window * preferences;
  GdkPixbuf * icon;
  GValue value;
  
  preferences = malloc(sizeof(* preferences));
  if (preferences == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  preferences->panel_data_list = carray_new(4);
  if (preferences->panel_data_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  window = gtk_dialog_new();
  gtk_window_set_title(GTK_WINDOW(window), _("Preferences"));
  
  gtk_window_set_default_size(GTK_WINDOW(window), 800, 600);
  
  vbox = GTK_DIALOG(window)->vbox;
  
  notebook = gtk_notebook_new();
  memset(&value, 0, sizeof(value));
  g_value_init(&value, G_TYPE_BOOLEAN);
  g_value_set_boolean(&value, TRUE);
  g_object_set_property(G_OBJECT(notebook), "homogeneous", &value);
  gtk_widget_show(notebook);
  gtk_box_pack_start(GTK_BOX(vbox), notebook, TRUE, TRUE, 0);
  
  close_button =  gtk_dialog_add_button(GTK_DIALOG(window), _("Close"), 0);
  
  preferences->window = window;
  preferences->notebook = notebook;
  preferences->vbox = vbox;
  preferences->hbox = vbox;
  preferences->close_button = close_button;
  
  icon = etpan_icon_manager_get_pixbuf(etpan_icon_manager_get_default(),
      "etpanX");
  gtk_window_set_icon(GTK_WINDOW(window), icon);
  
  return preferences;
}

void etpan_preferences_window_free(struct etpan_preferences_window * preferences)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(preferences->panel_data_list) ; i ++) {
    struct etpan_preferences_panel * panel;
    
    panel = carray_get(preferences->panel_data_list, i);
    etpan_preferences_panel_free(panel);
  }
  while (1) {
    int count;
    
    count = gtk_notebook_get_n_pages(GTK_NOTEBOOK(preferences->notebook));
    if (count == 0)
      break;
    
    gtk_notebook_remove_page(GTK_NOTEBOOK(preferences->notebook), count - 1);
  }
  
  gtk_widget_destroy(preferences->close_button);
  gtk_widget_destroy(preferences->notebook);
  gtk_widget_destroy(preferences->window);
  carray_free(preferences->panel_data_list);
  free(preferences);
}

/*
  account page
  - add account
  - remove account
  - modify account
  
  message view page
  - font
  - message colors
  
  message composer page
  - edit signature
  - address book -- later
  
  junk -- later
  - enable junk mail
  
  filters -- later
  - add filter
  - remove filter
  - modify filter
  
  address book

  advanced
    - other ?
  
*/

void etpan_preferences_add_page(struct etpan_preferences_window * preferences,
    char * text, GtkWidget * image, struct etpan_preferences_panel * panel)
{
  GtkWidget * label;
  int r;
  GtkWidget * widget;
  
  r = carray_add(preferences->panel_data_list, panel, NULL);
  if (r < 0)
    ETPAN_LOG_MEMORY_ERROR;
  
  widget = gtk_vbox_new(FALSE, 0);
  if (image != NULL) {
    gtk_widget_show(image);
    gtk_box_pack_start(GTK_BOX(widget), image, TRUE, FALSE, 0);
  }
  label = gtk_label_new(text);
  gtk_widget_show(label);
  gtk_box_pack_start(GTK_BOX(widget), label, TRUE, FALSE, 0);
  
  gtk_notebook_append_page(GTK_NOTEBOOK(preferences->notebook),
      panel->main_widget, widget);
}

struct etpan_preferences_panel * etpan_preferences_panel_new(void)
{
  struct etpan_preferences_panel * panel;
  
  panel = malloc(sizeof(* panel));
  if (panel == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  panel->main_widget = NULL;
  panel->widget_list = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (panel->widget_list == NULL)
    ETPAN_LOG_MEMORY_ERROR;
  
  panel->data = NULL;
  panel->open_callback = NULL;
  panel->should_close_callback = NULL;
  panel->close_callback = NULL;
  panel->free_data = NULL;
  
  return panel;
}

void etpan_preferences_panel_free(struct etpan_preferences_panel * panel)
{
  if (panel->free_data != NULL)
    panel->free_data(panel);
  
  chash_free(panel->widget_list);
  free(panel);
}

void etpan_preferences_panel_set_widget(struct etpan_preferences_panel * panel,
    char * name, GtkWidget * widget)
{
  chashdatum key;
  chashdatum value;
  int r;
  
  key.data = name;
  key.len = strlen(name) + 1;
  value.data = widget;
  value.len = 0;
  
  if (widget == NULL) {
    chash_delete(panel->widget_list, &key, NULL);
  }
  else {
    r = chash_set(panel->widget_list, &key, &value, NULL);
    if (r < 0)
      ETPAN_LOG_MEMORY_ERROR;
  }
}

GtkWidget * etpan_preferences_panel_get_widget(struct etpan_preferences_panel * panel, char * name)
{
  chashdatum key;
  chashdatum value;
  int r;
  
  key.data = name;
  key.len = strlen(name) + 1;
  
  r = chash_get(panel->widget_list, &key, &value);
  if (r < 0)
    return NULL;
  
  return value.data;
}


GtkWidget * etpan_preferences_panel_get_main_widget(struct etpan_preferences_window * preferences)
{
  return preferences->window;
}

static struct etpan_preferences_window * default_prefs = NULL;

struct etpan_preferences_window * etpan_preferences_window_get_default(void)
{
  return default_prefs;
}

void etpan_preferences_window_set_default(struct etpan_preferences_window * preferences)
{
  default_prefs = preferences;
}

static void switch_page(struct etpan_preferences_window * preferences,
    int page_num);

static void close_window(struct etpan_preferences_window * preferences)
{
  GtkWidget * window;
  
  switch_page(preferences, -1);
  if (preferences->current_page != -1) {
    /* cancel close */
    return;
  }
  
  window = etpan_preferences_panel_get_main_widget(preferences);
  
  gtk_widget_hide(window);
}

static gboolean close_handler(GtkWidget * widget,
    GdkEvent * event,
    gpointer user_data)
{
  struct etpan_preferences_window * preferences;
  (void) widget;
  (void) event;
  
  preferences = user_data;
  
  close_window(preferences);
  
  return TRUE;
}

static void show_handler(GtkWidget *widget,
    gpointer user_data)
{
  struct etpan_preferences_window * preferences;
  (void) widget;
  
  preferences = user_data;
  gtk_notebook_set_current_page(GTK_NOTEBOOK(preferences->notebook), 0);
  switch_page(preferences, 0);
}

static gboolean switch_back(gpointer data)
{
  struct etpan_preferences_window * preferences;
  
  preferences = data;
  
  gtk_notebook_set_current_page(GTK_NOTEBOOK(preferences->notebook),
      preferences->current_page);

  return FALSE;
}

static void switch_page(struct etpan_preferences_window * preferences,
    int page_num)
{
  struct etpan_preferences_panel * previous_panel;
  struct etpan_preferences_panel * next_panel;
  int can_close;
  
  if (page_num == preferences->current_page) {
    return;
  }
  
  previous_panel = NULL;
  if (preferences->current_page != -1) {
    previous_panel = carray_get(preferences->panel_data_list, preferences->current_page);
  }
  
  next_panel = NULL;
  if (page_num != -1)
    next_panel = carray_get(preferences->panel_data_list, page_num);
  
  can_close = 1;
  if (previous_panel != NULL) {
    if (previous_panel->should_close_callback != NULL) {
      can_close = previous_panel->should_close_callback(previous_panel);
    }
  }
  
  if (can_close) {
    if (previous_panel != NULL) {
      if (previous_panel->close_callback != NULL) {
        previous_panel->close_callback(previous_panel);
      }
    }
    
    if (next_panel != NULL) {
      if (next_panel->open_callback != NULL) {
        next_panel->open_callback(next_panel);
      }
    }
    
    preferences->current_page = page_num;
  }
  else {
    g_timeout_add(0, switch_back, preferences);
  }
}

static void page_switched(GtkNotebook * notebook,
    GtkNotebookPage * page,
    guint page_num,
    gpointer user_data)
{
  struct etpan_preferences_window * preferences;
  (void) notebook;
  (void) page;
  
  preferences = user_data;
  switch_page(preferences, (unsigned int) page_num);
}

static void close_clicked(GtkButton * button,
    gpointer user_data)
{
  struct etpan_preferences_window * preferences;
  (void) button;
  
  ETPAN_LOG("close handler %p", button);
  preferences = user_data;
  close_window(preferences);
}

void etpan_preferences_window_setup(struct etpan_preferences_window * preferences)
{
  GtkWidget * window;
  
  window = etpan_preferences_panel_get_main_widget(preferences);
  
  preferences->current_page = -1;
  etpan_preferences_account_init(preferences);
  etpan_preferences_view_init(preferences);
  etpan_preferences_compose_init(preferences);
  etpan_preferences_filter_init(preferences);
  etpan_preferences_abook_init(preferences);

  etpan_ui_window_set(etpan_ui_config_get_default(), "preferences",
      window);
  
  preferences->delete_signal_id = g_signal_connect(window, "delete-event",
      G_CALLBACK(close_handler), (gpointer) preferences);
  preferences->show_signal_id = g_signal_connect(window, "show",
      G_CALLBACK(show_handler), (gpointer) preferences);
  preferences->switch_signal_id = g_signal_connect(preferences->notebook, "switch-page",
      G_CALLBACK(page_switched), (gpointer) preferences);
  preferences->close_clicked_signal_id = g_signal_connect(preferences->close_button, "clicked",
      G_CALLBACK(close_clicked), (gpointer) preferences);
}

void etpan_preferences_window_unsetup(struct etpan_preferences_window * preferences)
{
  GtkWidget * window;
  
  window = etpan_preferences_panel_get_main_widget(preferences);
  
  etpan_ui_set_from_window(etpan_ui_config_get_default(), "preferences",
      window);
  
  close_window(preferences);
  
  g_signal_handler_disconnect(preferences->close_button,
      preferences->close_clicked_signal_id);
  g_signal_handler_disconnect(preferences->notebook,
      preferences->switch_signal_id);
  g_signal_handler_disconnect(window,
      preferences->show_signal_id);
  g_signal_handler_disconnect(window,
      preferences->delete_signal_id);
}
